<?php
/**
 * Rest api class for booksmark.
 *
 * @package BuddyBossApp\Api\Bookmark\V1
 */

namespace BuddyBossApp\Api\Bookmark\V1;

// Contain functionality for required additional rest api endpoints for Bookmarks.
use BuddyBossApp\Api\Bookmark\Main;
use BuddyBossApp\BB_Bookmarks;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * Class RestAPI
 *
 * @package BuddyBossApp\Api\Bookmark\V1
 */
class RestAPI extends WP_REST_Controller {

	/**
	 * Rest namespace
	 *
	 * @var string $namespace
	 */
	protected $namespace = 'buddyboss/v1';

	/**
	 * Rest base.
	 *
	 * @var string $rest_base
	 */
	protected $rest_base = 'bookmarks';

	/**
	 * The single instance of the class.
	 *
	 * @var null $instance
	 */
	private static $instance;

	/**
	 * Class construct.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of class
	 *
	 * @return RestAPI
	 */
	public static function instance() {

		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Load all rest hooks.
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register Quick & Lite Rest Endpoints
	 *
	 * @since 1.7.4
	 * @return void
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::CREATABLE ),
				),
				array(
					'methods'             => WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item_by_type' ),
					'permission_callback' => array( $this, 'delete_item_by_type_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::DELETABLE ),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				'args'   => array(
					'id' => array(
						'description' => __( 'A unique numeric ID for the bookmark.', 'buddyboss-app' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param(
							array(
								'default' => 'view',
							)
						),
					),
				),
				array(
					'methods'             => WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( WP_REST_Server::DELETABLE ),
				),
				'schema' => array( $this, 'get_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/bookmark-types',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_type_items' ),
					'permission_callback' => array( $this, 'get_type_items_permissions_check' ),
				),
				'schema' => array( $this, 'get_type_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/bookmark-types/(?P<type>[\w-]+)',
			array(
				'args'   => array(
					'type' => array(
						'description' => __( 'A unique type for the bookmark.', 'buddyboss-app' ),
						'type'        => 'string',
					),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_type_item' ),
					'permission_callback' => array( $this, 'get_type_item_permissions_check' ),
				),
				'schema' => array( $this, 'get_type_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/batch',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'batch_items' ),
					'permission_callback' => array( $this, 'batch_items_permissions_check' ),
					'args'                => array(
						'bookmarks' => array(
							'description' => __( 'The list of batch bookmark items.', 'buddyboss-app' ),
							'type'        => 'array',
							'required'    => true,
							'items'       => array(
								'type'       => array(
									'description' => __( 'The type of the bookmark.', 'buddyboss-app' ),
									'type'        => 'array',
									'required'    => true,
								),
								'item_id'    => array(
									'description' => __( 'The ID of bookmark item.', 'buddyboss-app' ),
									'type'        => 'array',
									'required'    => true,
								),
								'is_deleted' => array(
									'description' => __( 'Is delete bookmark batch item.', 'buddyboss-app' ),
									'type'        => 'array',
									'required'    => false,
								),
							),
						),
					),
				),
				'schema' => array( $this, 'batch_items_schema' ),
			)
		);
	}

	/**
	 * Retrieve bookmarks.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error List of bookmarks object data.
	 *
	 * @api            {GET} /wp-json/buddyboss/v1/bookmark Get Bookmarks
	 * @apiName        GetBBBookmarks
	 * @apiGroup       Bookmarks
	 * @apiDescription Retrieve bookmarks
	 * @apiVersion     1.0.0
	 * @apiParam {Number} [page=1] Current page of the collection.
	 * @apiParam {Number} [per_page=10] Maximum number of items to be returned in result set.
	 * @apiParam {String=post,page} [type] Limit results based on bookmark type.
	 * @apiParam {String=asc,desc} [order=desc] Order sort attribute ascending or descending.
	 * @apiParam {String=id,type,item_id,date_recorded} [orderby=date_recorded] Order Bookmarks by which attribute.
	 * @apiParam {Number} [user_id] Pass a user_id to limit to only Bookmarks that this user is a bookmarked.
	 * @apiParam {Number} [blog_id] Get bookmark site wise. Default current site ID.
	 * @apiParam {Number} [item_id] Get Bookmarks that are user bookmarked items.
	 * @apiParam {Number} [status=1] Active Bookmarks. 1 = Active, 0 = Inactive.
	 * @apiParam {Array} [include] Ensure result set includes Bookmarks with specific IDs.
	 * @apiParam {Array} [exclude] Ensure result set excludes Bookmarks with specific IDs.
	 */
	public function get_items( $request ) {

		$args = array(
			'type'     => $request->get_param( 'type' ),
			'blog_id'  => $request->get_param( 'blog_id' ),
			'item_id'  => $request->get_param( 'item_id' ),
			'per_page' => $request->get_param( 'per_page' ),
			'page'     => $request->get_param( 'page' ),
			'user_id'  => get_current_user_id(),
			'order_by' => $request->get_param( 'order_by' ),
			'order'    => $request->get_param( 'order' ),
			'include'  => $request->get_param( 'include' ),
			'exclude'  => $request->get_param( 'exclude' ),
			'status'   => $request->get_param( 'status' ),
			'count'    => true,
		);

		$bookmarks = bb_get_bookmarks( $args );
		$retval    = array();
		foreach ( $bookmarks['bookmarks'] as $bookmark ) {
			$retval[] = $this->prepare_response_for_collection(
				$this->prepare_item_for_response( $bookmark, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		$response = Main::instance()->bb_rest_response_add_total_headers( $response, $bookmarks['total'], $args['per_page'] );

		/**
		 * Fires after a list of bookmarks is fetched via the REST API.
		 *
		 * @param array            $bookmarks Fetched bookmarks.
		 * @param WP_REST_Response $response  The response data.
		 * @param WP_REST_Request  $request   The request sent to the API.
		 */
		do_action( 'bb_rest_bookmarks_get_items', $bookmarks, $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to bookmark items.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function get_items_permissions_check( $request ) {
		$retval = new WP_Error(
			'bb_rest_authorization_required',
			__( 'Sorry, you are not allowed to view bookmark.', 'buddyboss-app' ),
			array(
				'status' => rest_authorization_required_code(),
			)
		);

		if ( is_user_logged_in() ) {
			$retval = true;
		}

		/**
		 * Filter the bookmarks `get_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bb_rest_bookmarks_get_items_permissions_check', $retval, $request );
	}

	/**
	 * Create a bookmark.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_REST_Response | WP_Error
	 *
	 * @api            {POST} /wp-json/buddyboss/v1/bookmark Create Bookmark
	 * @apiName        CreateBBBookmark
	 * @apiGroup       Bookmark
	 * @apiDescription Create bookmark
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {String=forum,topic} type The type bookmark.
	 * @apiParam {Number} type The Type of post/page etc.
	 * @apiParam {Number} item_id The ID of post/page etc.
	 * @apiParam {Number} [user_id] The ID of the user who created the Bookmark. default logged-in user id.
	 * @apiParam {Number} [blog_id] The ID of site. default current site id.
	 */
	public function create_item( $request ) {

		// Setting context.
		$request->set_param( 'context', 'edit' );

		$bookmark_id = bb_add_bookmark( $this->prepare_item_for_database( $request ) );

		if ( is_wp_error( $bookmark_id ) ) {
			return $bookmark_id;
		} elseif ( ! is_numeric( $bookmark_id ) ) {
			return new WP_Error(
				'bb_rest_user_cannot_create_bookmark',
				__( 'There is an error while adding the bookmark.', 'buddyboss-app' ),
				array(
					'status' => 500,
				)
			);
		}

		$bookmark = $this->get_bookmark_object( $bookmark_id );

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $bookmark, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a bookmark is created via the REST API.
		 *
		 * @param object           $bookmark The created bookmark.
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 */
		do_action( 'bb_rest_bookmarks_create_item', $bookmark, $response, $request );

		return $response;
	}

	/**
	 * Checks if a given request has access to create a bookmark.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function create_item_permissions_check( $request ) {
		if ( ! is_user_logged_in() ) {
			$retval = new WP_Error(
				'bb_rest_authorization_required',
				__( 'Sorry, you are not allowed to create bookmark.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		} else {
			$retval = $this->validate_request_item( $request );
		}

		/**
		 * Filter the bookmarks `create_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bb_rest_bookmarks_create_item_permissions_check', $retval, $request );
	}

	/**
	 * Delete a bookmark.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error
	 *
	 * @api            {DELETE} /wp-json/buddyboss/v1/bookmark/:id Delete Bookmark
	 * @apiName        DeleteBBBookmark
	 * @apiGroup       Bookmarks
	 * @apiDescription Delete a bookmark.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Bookmark.
	 */
	public function delete_item( $request ) {
		// Setting context.
		$request->set_param( 'context', 'edit' );

		// Get the bookmark before it's deleted.
		$bookmark = $this->get_bookmark_object( $request );
		$previous = $this->prepare_item_for_response( $bookmark, $request );
		if ( ! bb_delete_bookmark( $bookmark->id ) ) {
			$error = __( 'There was a problem removing bookmark.', 'buddyboss-app' );
			if ( isset( $previous->data, $previous->data['item_title'] ) && ! empty( $previous->data['item_title'] ) ) {
				$error = sprintf(
				/* translators: %s is forum/discussion title */
					__( 'There was a problem removing bookmark from %s', 'buddyboss-app' ),
					$previous->data['item_title']
				);
			}

			return new WP_Error(
				'bb_rest_bookmark_cannot_delete',
				$error,
				array(
					'status' => 500,
				)
			);
		}

		$response_array = array(
			'deleted'  => true,
			'previous' => $previous->get_data(),
		);

		$type        = $request->get_param( 'type' );
		$total_pages = (int) $request->get_param( 'total_pages' );
		if ( ! empty( $type ) && 1 < $total_pages ) {
			$args = array(
				'type'     => $type,
				'per_page' => (int) $request->get_param( 'per_page' ),
				'page'     => (int) $request->get_param( 'page' ),
				'user_id'  => get_current_user_id(),
				'count'    => true,
			);

			$bookmarks   = bb_get_bookmarks( $args );
			$total_items = (int) $bookmarks['total'];
			$max_pages   = ceil( $total_items / (int) $args['per_page'] );

			$new_page = $args['page'];
			if ( 1 === (int) $max_pages ) {
				$new_page = 1;
			} elseif ( empty( $bookmarks['bookmarks'] ) && $args['page'] === $max_pages ) {
				$new_page = $args['page'] - 1;
			}

			$response_array = array_merge(
				$response_array,
				array(
					'type'        => $type,
					'total_pages' => (int) $max_pages,
					'per_page'    => (int) $request->get_param( 'per_page' ),
					'page'        => (int) $new_page,
				)
			);
		}

		// Build the response.
		$response = new WP_REST_Response();
		$response->set_data( $response_array );

		/**
		 * Fires after a bookmark is deleted via the REST API.
		 *
		 * @param object           $bookmark The deleted bookmark.
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 */
		do_action( 'bb_rest_bookmarks_delete_item', $bookmark, $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to delete a bookmark.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function delete_item_permissions_check( $request ) {
		$retval = new WP_Error(
			'bb_rest_authorization_required',
			__( 'Sorry, you need to be logged in to delete this bookmark.', 'buddyboss-app' ),
			array(
				'status' => rest_authorization_required_code(),
			)
		);

		if ( is_user_logged_in() ) {
			$retval = new WP_Error(
				'bb_rest_authorization_required',
				__( 'Sorry, you are not allowed to delete this bookmark.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);

			$bookmark = $this->get_bookmark_object( $request );

			if ( empty( $bookmark->id ) ) {
				$retval = new WP_Error(
					'bb_rest_bookmark_invalid_id',
					__( 'Invalid bookmark ID.', 'buddyboss-app' ),
					array(
						'status' => 404,
					)
				);
			} elseif ( $this->can_user_see_or_delete( $bookmark ) ) {
				$retval = true;
			}
		}

		/**
		 * Filter the bookmark `delete_item` permissions check.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 */
		return apply_filters( 'bb_rest_bookmarks_delete_item_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve a bookmark.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error
	 *
	 * @api            {GET} /wp-json/buddyboss/v1/bookmarks/:id Get Bookmark
	 * @apiName        GetBBBookmark
	 * @apiGroup       Bookmarks
	 * @apiDescription Retrieve single bookmark
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser if the site is in Private Network.
	 * @apiParam {Number} id A unique numeric ID for the Bookmarks.
	 */
	public function get_item( $request ) {
		$bookmark = $this->get_bookmark_object( $request );

		$retval = $this->prepare_response_for_collection(
			$this->prepare_item_for_response( $bookmark, $request )
		);

		$response = rest_ensure_response( $retval );

		/**
		 * Fires after a bookmark is fetched via the REST API.
		 *
		 * @param object           $bookmark Fetched bookmark.
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 */
		do_action( 'bb_rest_bookmars_get_item', $bookmark, $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to get information about a specific bookmark.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function get_item_permissions_check( $request ) {
		$retval = new WP_Error(
			'bb_rest_authorization_required',
			__( 'Sorry, you are not allowed to view bookmark.', 'buddyboss-app' ),
			array(
				'status' => rest_authorization_required_code(),
			)
		);

		if ( is_user_logged_in() ) {
			$retval = new WP_Error(
				'bb_rest_authorization_required',
				__( 'Sorry, you cannot view the bookmark.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);

			$bookmark = $this->get_bookmark_object( $request );
			if ( empty( $bookmark->id ) ) {
				$retval = new WP_Error(
					'bb_rest_bookmark_invalid_id',
					__( 'Invalid bookmark ID.', 'buddyboss-app' ),
					array(
						'status' => 404,
					)
				);
			} elseif ( $this->can_user_see_or_delete( $bookmark ) ) {
				$retval = true;
			}
		}

		/**
		 * Filter the bookmark `get_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bb_rest_bookmarks_get_item_permissions_check', $retval, $request );
	}

	/**
	 * Delete a bookmark by type and item id.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error
	 *
	 * @api            {DELETE} /wp-json/buddyboss/v1/bookmark/?type={{type}}&item_ids[]={{item_id}} Delete Bookmark
	 * @apiName        DeleteBBBookmark
	 * @apiGroup       Bookmarks
	 * @apiDescription Delete a bookmark.
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {Number} id A unique numeric ID for the Bookmark.
	 */
	public function delete_item_by_type( $request ) {
		// Setting context.
		$request->set_param( 'context', 'edit' );

		// Get the bookmark before it's deleted.
		$bookmarks      = $this->get_bookmarks_by_item( $request );
		$response_array = array();
		foreach ( $bookmarks as $bookmark ) {
			if ( ! $this->can_user_see_or_delete( $bookmark ) ) {
				$previous_data = new WP_Error(
					'bb_rest_bookmark_invalid_id',
					__( 'Invalid bookmark ID.', 'buddyboss-app' ),
					array(
						'status' => 404,
					)
				);
			} else {
				$bookmark = bb_bookmarks_get_bookmark( $bookmark->id );
				$previous = $this->prepare_item_for_response( $bookmark, $request );
				if ( ! bb_delete_bookmark( $bookmark->id ) ) {
					$error = __( 'There was a problem removing bookmark.', 'buddyboss-app' );
					if ( isset( $previous->data, $previous->data['item_title'] ) && ! empty( $previous->data['item_title'] ) ) {
						$error = sprintf(
						/* translators: %s is forum/discussion title */
							__( 'There was a problem removing bookmark from %s', 'buddyboss-app' ),
							$previous->data['item_title']
						);
					}

					$previous_data = new WP_Error(
						'bb_rest_bookmark_cannot_delete',
						$error,
						array(
							'status' => 500,
						)
					);
				} else {
					$previous_data = $previous->get_data();
				}
			}

			$response_array[] = array(
				'deleted'  => true,
				'previous' => $previous_data,
			);

		}

		$type        = $request->get_param( 'type' );
		$total_pages = (int) $request->get_param( 'total_pages' );
		if ( ! empty( $type ) && 1 < $total_pages ) {
			$args = array(
				'type'     => $type,
				'per_page' => (int) $request->get_param( 'per_page' ),
				'page'     => (int) $request->get_param( 'page' ),
				'user_id'  => get_current_user_id(),
				'count'    => true,
			);

			$bookmarks   = bb_get_bookmarks( $args );
			$total_items = (int) $bookmarks['total'];
			$max_pages   = ceil( $total_items / (int) $args['per_page'] );

			$new_page = $args['page'];
			if ( 1 === (int) $max_pages ) {
				$new_page = 1;
			} elseif ( empty( $bookmarks['bookmarks'] ) && $args['page'] === $max_pages ) {
				$new_page = $args['page'] - 1;
			}

			$response_array = array_merge(
				$response_array,
				array(
					'type'        => $type,
					'total_pages' => (int) $max_pages,
					'per_page'    => (int) $request->get_param( 'per_page' ),
					'page'        => (int) $new_page,
				)
			);
		}

		// Build the response.
		$response = new WP_REST_Response();
		$response->set_data( $response_array );

		/**
		 * Fires after a bookmark is deleted via the REST API.
		 *
		 * @param object           $bookmark The deleted bookmark.
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  The request sent to the API.
		 */
		do_action( 'bb_rest_bookmarks_delete_item', $bookmark, $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to delete a bookmark.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function delete_item_by_type_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = new WP_Error(
				'bb_rest_authorization_required',
				__( 'Sorry, you are not allowed to delete this bookmark.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		$bookmarks = $this->get_bookmarks_by_item( $request );

		if ( empty( $bookmarks ) ) {
			$retval = new WP_Error(
				'bb_rest_bookmarks_invalid',
				__( 'Invalid bookmark ids.', 'buddyboss-app' ),
				array(
					'status' => 404,
				)
			);
		}

		/**
		 * Filter the bookmark `delete_item_by_type` permissions check.
		 *
		 * @param WP_REST_Request $request The request sent to the API.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 */
		return apply_filters( 'bb_rest_bookmarks_delete_item_by_type_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve bookmarks types.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error List of bookmark types object data.
	 *
	 * @api            {GET} /wp-json/buddyboss/v1/bookmark-types Get Bookmark types
	 * @apiName        GetBBBookmarkTypes
	 * @apiGroup       Bookmarks
	 * @apiDescription Retrieve bookmark Types
	 * @apiVersion     1.0.0
	 */
	public function get_type_items( $request ) {
		$bookmark_types = bb_get_bookmark_types();

		if ( empty( $bookmark_types ) ) {
			return new WP_Error(
				'bb_rest_types_not_exists',
				__( 'There is an error while fetching the bookmark types.', 'buddyboss-app' ),
				array(
					'status' => 404,
				)
			);
		}

		$retval = array();
		foreach ( $bookmark_types as $key => $type ) {
			$data        = new \stdClass();
			$data->type  = $key;
			$data->label = $type['label'];
			$retval[]    = $this->prepare_response_for_collection(
				$this->prepare_type_item_for_response( $data, $request )
			);
		}

		$response = rest_ensure_response( $retval );
		$response = Main::instance()->bb_rest_response_add_total_headers( $response, count( $bookmark_types ), count( $bookmark_types ) );

		/**
		 * Fires after a list of bookmark type is fetched via the REST API.
		 *
		 * @param array            $bookmark_types Fetched bookmark types.
		 * @param WP_REST_Response $response       The response data.
		 * @param WP_REST_Request  $request        The request sent to the API.
		 */
		do_action( 'bb_rest_bookmarks_get_type_items', $bookmark_types, $response, $request );

		return $response;
	}

	/**
	 * Get bookmark object.
	 *
	 * @param WP_REST_Request|int|array $request Full details about the request.
	 *
	 * @return bool|object|BB_Bookmarks
	 */
	public function get_bookmark_object( $request ) {
		if ( is_numeric( $request ) ) {
			$bookmark_id = $request;
		} else {
			$bookmark_id = (int) $request['id'];
		}

		if ( empty( $bookmark_id ) ) {
			return false;
		}

		$bookmark = bb_bookmarks_get_bookmark( $bookmark_id );

		if ( empty( $bookmark ) || empty( $bookmark->id ) ) {
			return false;
		}

		return $bookmark;
	}

	/**
	 * Get bookmarks object by type and item id.
	 *
	 * @param WP_REST_Request|int|array $request Full details about the request.
	 *
	 * @return bool|object|BB_Bookmarks
	 */
	public function get_bookmarks_by_item( $request ) {

		$r = bb_parse_args(
			$request->get_params(),
			array(
				'type'     => false,
				'item_ids' => array(),
				'user_id'  => get_current_user_id(),
			),
			'bookmarks_by_item'
		);

		$bookmark_type     = $r['type'];
		$bookmark_item_ids = ! is_array( $r['item_ids'] ) ? array( $r['item_ids'] ) : $r['item_ids'];
		$bookmark_user_id  = $r['user_id'];

		if ( empty( $bookmark_type ) && empty( $bookmark_item_ids ) ) {
			return false;
		}

		$bookmarks = bb_bookmarks_get_bookmark_by_item( $bookmark_type, $bookmark_item_ids, $bookmark_user_id );

		if ( empty( $bookmarks ) ) {
			return false;
		}

		return $bookmarks;
	}

	/**
	 * Check if a given request has access to bookmark types.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function get_type_items_permissions_check( $request ) {
		$retval = new WP_Error(
			'bb_rest_authorization_required',
			__( 'Sorry, you are not allowed to view bookmark types.', 'buddyboss-app' ),
			array(
				'status' => rest_authorization_required_code(),
			)
		);

		if ( is_user_logged_in() ) {
			$retval = true;
		}

		/**
		 * Filter the bookmarks `get_type_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bb_rest_bookmarks_get_type_items_permissions_check', $retval, $request );
	}

	/**
	 * Retrieve single bookmark based on type.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response | WP_Error List of bookmark types object data.
	 *
	 * @api            {GET} /wp-json/buddyboss/v1/bookmark-types/{type} Get Bookmark type
	 * @apiName        GetBBBookmarkType
	 * @apiGroup       Bookmarks
	 * @apiDescription Retrieve bookmark type
	 * @apiVersion     1.0.0
	 */
	public function get_type_item( $request ) {
		$type           = ! empty( $request['type'] ) ? $request['type'] : '';
		$bookmark_types = bb_get_bookmark_types();
		$bookmark_type  = array();

		if ( ! empty( $type ) && isset( $bookmark_types[ $type ] ) ) {
			$bookmark_type = $bookmark_types[ $type ];
		}

		if ( empty( $bookmark_type ) ) {
			return new WP_Error(
				'bb_rest_type_not_exists',
				__( 'There is an error while fetching the bookmark type.', 'buddyboss-app' ),
				array(
					'status' => 404,
				)
			);
		}

		$data        = new \stdClass();
		$data->type  = $type;
		$data->label = $bookmark_type['label'];
		$retval      = $this->prepare_response_for_collection(
			$this->prepare_type_item_for_response( $data, $request )
		);

		$response = rest_ensure_response( $retval );
		/**
		 * Fires after a list of bookmark type is fetched via the REST API.
		 *
		 * @param array            $bookmark_types Fetched bookmark types.
		 * @param WP_REST_Response $response       The response data.
		 * @param WP_REST_Request  $request        The request sent to the API.
		 */
		do_action( 'bb_rest_bookmarks_get_type_item', $bookmark_type, $response, $request );

		return $response;
	}

	/**
	 * Check if a given request has access to bookmark types.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function get_type_item_permissions_check( $request ) {
		$retval = new WP_Error(
			'bb_rest_authorization_required',
			__( 'Sorry, you are not allowed to view bookmark type.', 'buddyboss-app' ),
			array(
				'status' => rest_authorization_required_code(),
			)
		);

		if ( is_user_logged_in() ) {
			$retval = true;
		}

		/**
		 * Filter the bookmarks `get_type_items` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bb_rest_bookmarks_get_type_item_permissions_check', $retval, $request );
	}

	/**
	 * Bookmark batch.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_REST_Response | WP_Error | array
	 *
	 * @api            {POST} /wp-json/buddyboss/v1/bookmark Create Bookmark
	 * @apiName        CreateBBBookmark
	 * @apiGroup       Bookmark
	 * @apiDescription Create bookmark
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiParam {String=forum,topic} type The type bookmark.
	 * @apiParam {Number} type The Type of post/page etc.
	 * @apiParam {Number} item_id The ID of post/page etc.
	 * @apiParam {Number} [user_id] The ID of the user who created the Bookmark. default logged-in user id.
	 * @apiParam {Number} [blog_id] The ID of site. default current site id.
	 */
	public function batch_items( $request ) {

		/**
		 * REST Server
		 *
		 * @var WP_REST_Server $wp_rest_server
		 */
		global $wp_rest_server;

		// Get the request params.
		$items    = array_filter( $request->get_params() );
		$query    = $request->get_query_params();
		$response = array();

		$user_id = $request->get_param( 'user_id' );
		if ( empty( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		// Check batch limit.
		$limit = $this->check_batch_limit( $items['bookmarks'] );

		if ( is_wp_error( $limit ) ) {
			return $limit;
		}

		if ( ! empty( $items['bookmarks'] ) ) {
			foreach ( $items['bookmarks'] as $item ) {
				if ( true === (bool) $item['is_deleted'] ) {
					$bookmark_items = bb_bookmarks_get_bookmark_by_item( $item['type'], $item['item_id'], $user_id );

					foreach ( $bookmark_items as $bookmark_item ) {
						$id = ! empty( $bookmark_item->id ) ? $bookmark_item->id : 0;
						if ( 0 === $id ) {
							continue;
						}

						$delete_item = new WP_REST_Request( 'DELETE', $request->get_route() );
						$delete_item->set_query_params(
							array(
								'id'    => $id,
								'force' => true,
							)
						);
						$_response = $this->delete_item( $delete_item );

						if ( is_wp_error( $_response ) ) {
							$response[] = array(
								'id'    => $id,
								'error' => array(
									'code'    => $_response->get_error_code(),
									'message' => $_response->get_error_message(),
									'data'    => $_response->get_error_data(),
								),
							);
						} else {
							$response[] = $wp_rest_server->response_to_data( $_response, '' );
						}
					}
				} else {
					$create_item = new WP_REST_Request( 'POST', $request->get_route() );

					// Default parameters.
					$defaults = array();
					$schema   = $this->get_item_schema();
					foreach ( $schema['properties'] as $arg => $options ) {
						if ( isset( $options['default'] ) ) {
							$defaults[ $arg ] = $options['default'];
						}
					}
					$create_item->set_default_params( $defaults );

					// Set request parameters.
					$create_item->set_body_params( $item );

					// Set query (GET) parameters.
					$create_item->set_query_params( $query );

					$_response = $this->create_item( $create_item );

					if ( is_wp_error( $_response ) ) {
						$response[] = array(
							'id'    => 0,
							'error' => array(
								'code'    => $_response->get_error_code(),
								'message' => $_response->get_error_message(),
								'data'    => $_response->get_error_data(),
							),
						);
					} else {
						$response[] = $wp_rest_server->response_to_data( $_response, '' );
					}
				}
			}
		}

		return $response;
	}

	/**
	 * Checks if a given request has access to batch a bookmark.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function batch_items_permissions_check( $request ) {
		$retval = true;

		if ( ! is_user_logged_in() ) {
			$retval = new WP_Error(
				'bb_rest_authorization_required',
				__( 'Sorry, you are not allowed to run batch bookmark items.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		/**
		 * Filter the bookmarks `batch_item` permissions check.
		 *
		 * @param bool|WP_Error   $retval  Returned value.
		 * @param WP_REST_Request $request The request sent to the API.
		 */
		return apply_filters( 'bb_rest_bookmarks_batch_items_permissions_check', $retval, $request );
	}

	/**
	 * Prepare a bookmark for create or update.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return \stdClass|WP_Error Object or WP_Error.
	 */
	protected function prepare_item_for_database( $request ) {
		$prepared_bookmark = new \stdClass();
		$schema            = $this->get_item_schema();
		$bookmark          = $this->get_bookmark_object( $request );

		// Bookmark ID.
		if ( ! empty( $schema['properties']['id'] ) && ! empty( $bookmark->id ) ) {
			$prepared_bookmark->id = $bookmark->id;
		}

		// Bookmark blog ID.
		if ( ! empty( $schema['properties']['blog_id'] ) && isset( $request['blog_id'] ) ) {
			$prepared_bookmark->blog_id = (int) $request['blog_id'];

			// Fallback on the existing blog ID in case of an update.
		} elseif ( isset( $bookmark->blog_id ) && $bookmark->blog_id ) {
			$prepared_bookmark->blog_id = (int) $bookmark->blog_id;
		}

		// Bookmark user ID.
		if ( ! empty( $schema['properties']['user_id'] ) && isset( $request['user_id'] ) ) {
			$prepared_bookmark->user_id = (int) $request['user_id'];

			// Fallback on the existing user id in case of an update.
		} elseif ( isset( $bookmark->user_id ) && $bookmark->user_id ) {
			$prepared_bookmark->user_id = (int) $bookmark->user_id;

			// Fallback on the current user otherwise.
		} else {
			$prepared_bookmark->user_id = get_current_user_id();
		}

		// Bookmark type.
		if ( ! empty( $schema['properties']['type'] ) && isset( $request['type'] ) ) {
			$prepared_bookmark->type = $request['type'];

			// Fallback on the existing Bookmark type in case of an update.
		} elseif ( isset( $bookmark->type ) && $bookmark->type ) {
			$prepared_bookmark->type = $bookmark->type;
		}

		// Bookmark item ID.
		if ( ! empty( $schema['properties']['item_id'] ) && isset( $request['item_id'] ) ) {
			$prepared_bookmark->item_id = (int) $request['item_id'];

			// Fallback on the existing item id in case of an update.
		} elseif ( isset( $bookmark->item_id ) && $bookmark->item_id ) {
			$prepared_bookmark->item_id = (int) $bookmark->item_id;
		}

		// Bookmark status.
		if ( ! empty( $schema['properties']['status'] ) && isset( $request['status'] ) ) {
			$prepared_bookmark->status = (int) $request['status'];

			// Fallback on the existing status in case of an update.
		} elseif ( isset( $bookmark->status ) && $bookmark->status ) {
			$prepared_bookmark->status = (int) $bookmark->status;
		}

		/**
		 * Filters a bookmark before it is inserted or updated via the REST API.
		 *
		 * @param \stdClass       $prepared_bookmark An object prepared for inserting or updating the database.
		 * @param WP_REST_Request $request           Request object.
		 */
		return apply_filters( 'bb_rest_bookmarks_pre_insert_value', $prepared_bookmark, $request );
	}

	/**
	 * Prepares bookmark data for return as an object.
	 *
	 * @param object          $item    BB_Bookmarks object.
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 */
	public function prepare_item_for_response( $item, $request ) {

		$data = array(
			'id'                  => (int) $item->id,
			'user_id'             => (int) $item->user_id,
			'type'                => $item->type,
			'item_id'             => (int) $item->item_id,
			'blog_id'             => (int) $item->blog_id,
			'date'                => bb_rest_prepare_date_response( get_date_from_gmt( $item->date_recorded ) ),
			'date_gmt'            => bb_rest_prepare_date_response( $item->date_recorded ),
			'status'              => (bool) $item->status,
			'item_title'          => $item->item_title,
			'item_featured_image' => $item->item_featured_image,
			'author'              => $item->author,
			'link'                => $item->link,
		);

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';

		$data     = $this->add_additional_fields_to_object( $data, $request );
		$data     = $this->filter_response_by_context( $data, $context );
		$response = rest_ensure_response( $data );
		$response->add_links( $this->prepare_links( $item, $request ) );

		/**
		 * Filter a bookmark value returned from the API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  Request used to generate the response.
		 * @param object           $item     BB_Bookmarks object.
		 */
		return apply_filters( 'bb_rest_bookmarks_prepare_value', $response, $request, $item );
	}

	/**
	 * Prepares bookmark data for return as an object.
	 *
	 * @param array|object    $item    Bookmark type object.
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response
	 */
	public function prepare_type_item_for_response( $item, $request ) {
		$bookmarks = bb_get_bookmarks(
			array(
				'type'     => array( $item->type ),
				'per_page' => 1,
				'page'     => 1,
				'user_id'  => get_current_user_id(),
				'count'    => true,
			),
			true
		);

		$data = array(
			'type'  => $item->type,
			'label' => $item->label,
			'count' => isset( $bookmarks['total'] ) ? $bookmarks['total'] : 0,
		);

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';

		$data     = $this->add_additional_fields_to_object( $data, $request );
		$data     = $this->filter_response_by_context( $data, $context );
		$response = rest_ensure_response( $data );
		$response->add_links( $this->prepare_type_links( $item, $request ) );

		/**
		 * Filter a bookmark type value returned from the API.
		 *
		 * @param WP_REST_Response $response The response data.
		 * @param WP_REST_Request  $request  Request used to generate the response.
		 * @param BB_Bookmarks     $item     Bookmark object.
		 */
		return apply_filters( 'bb_rest_bookmarks_type_prepare_value', $response, $request, $item );
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param object          $item    Bookmark object.
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return array
	 */
	protected function prepare_links( $item, $request ) {
		$links = array(
			'user' => array(
				'href'       => rest_url( $this->bb_rest_get_user_url( $item->user_id ) ),
				'embeddable' => true,
			),
		);

		/**
		 * Filter links prepared for the REST response.
		 *
		 * @param array           $links   The prepared links of the REST response.
		 * @param BB_Bookmarks    $item    Bookmark object.
		 * @param WP_REST_Request $request Full details about the request.
		 */
		return apply_filters( 'bb_rest_bookmarks_prepare_links', $links, $item, $request );
	}

	/**
	 * Prepare links for the request.
	 *
	 * @param object          $item    Bookmark type object.
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return array
	 */
	protected function prepare_type_links( $item, $request ) {
		$base = '/' . $this->namespace . '/' . $this->rest_base;

		$links = array(
			'options' => array(
				'embeddable' => true,
				'href'       => add_query_arg(
					'type',
					$item->type,
					rest_url( untrailingslashit( $base ) )
				),
			),
		);

		/**
		 * Filter links prepared for the REST response.
		 *
		 * @param array           $links   The prepared links of the REST response.
		 * @param object          $item    Bookmark type object.
		 * @param WP_REST_Request $request Full details about the request.
		 */
		return apply_filters( 'bb_rest_bookmarks_type_prepare_links', $links, $item, $request );
	}

	/**
	 * Get the query params for collections of plugins.
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params                       = parent::get_collection_params();
		$params['context']['default'] = 'view';

		unset( $params['search'] );

		$params['exclude'] = array(
			'description'       => __( 'Ensure result set excludes specific IDs.', 'buddyboss-app' ),
			'default'           => array(),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['include'] = array(
			'description'       => __( 'Ensure result set includes specific IDs.', 'buddyboss-app' ),
			'default'           => array(),
			'type'              => 'array',
			'items'             => array( 'type' => 'integer' ),
			'sanitize_callback' => 'wp_parse_id_list',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['order'] = array(
			'description'       => __( 'Order sort attribute ascending or descending.', 'buddyboss-app' ),
			'default'           => 'desc',
			'type'              => 'string',
			'enum'              => array( 'asc', 'desc' ),
			'sanitize_callback' => 'sanitize_key',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['order_by'] = array(
			'description'       => __( 'Order by a specific parameter.', 'buddyboss-app' ),
			'default'           => 'date_recorded',
			'type'              => 'string',
			'enum'              => array( 'id', 'type', 'item_id', 'date_recorded' ),
			'sanitize_callback' => 'sanitize_key',
		);

		$params['blog_id'] = array(
			'description'       => __( 'The ID of the current blog site.', 'buddyboss-app' ),
			'default'           => get_current_blog_id(),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['user_id'] = array(
			'description'       => __( 'Limit result set to items created by a specific user (ID).', 'buddyboss-app' ),
			'default'           => get_current_user_id(),
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['type'] = array(
			'description'       => __( 'Limit result set to items with a specific bookmark type.', 'buddyboss-app' ),
			'type'              => 'string',
			'enum'              => array_keys( bb_get_bookmark_types() ),
			'sanitize_callback' => 'sanitize_key',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['item_id'] = array(
			'description'       => __( 'Limit result set to items with a specific prime association ID.', 'buddyboss-app' ),
			'default'           => 0,
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		$params['status'] = array(
			'description'       => __( 'Limit result set to active bookmarks or not.', 'buddyboss-app' ),
			'default'           => 1,
			'type'              => 'integer',
			'enum'              => array( 0, 1 ),
			'sanitize_callback' => 'absint',
			'validate_callback' => 'rest_validate_request_arg',
		);

		/**
		 * Filters the collection query params.
		 *
		 * @param array $params Query params.
		 */
		return apply_filters( 'bb_rest_bookmarks_collection_params', $params );
	}

	/**
	 * Edit the type of the some properties for the CREATABLE & EDITABLE methods.
	 *
	 * @param string $method Optional. HTTP method of the request.
	 *
	 * @return array Endpoint arguments.
	 */
	public function get_endpoint_args_for_item_schema( $method = WP_REST_Server::CREATABLE ) {
		$args = WP_REST_Controller::get_endpoint_args_for_item_schema( $method );
		$key  = 'create_item';

		if ( WP_REST_Server::CREATABLE === $method ) {
			$args['type']['required']            = true;
			$args['item_id']['required']         = true;
			$args['status']['required']          = true;
			$args['status']['default']           = 1;
			$args['status']['enum']              = array( 0, 1 );
			$args['status']['sanitize_callback'] = 'absint';
			$args['status']['validate_callback'] = 'rest_validate_request_arg';
		}

		if ( WP_REST_Server::DELETABLE === $method ) {
			$key = 'delete_item';
			if ( 'bb_bookmark_types' === WP_REST_Controller::get_object_type() ) {
				$key = 'delete_item_type';
				unset( $args['item_id'] );
				unset( $args['blog_id'] );
				unset( $args['status'] );
				$args['item_ids']         = array(
					'description' => __( 'The IDs of bookmark item.', 'buddyboss-app' ),
					'type'        => 'array',
					'required'    => true,
				);
				$args['type']['required'] = true;
			}
		}

		/**
		 * Filters the method query arguments.
		 *
		 * @param array  $args   Query arguments.
		 * @param string $method HTTP method of the request.
		 */
		return apply_filters( "bb_rest_bookmarks_{$key}_query_arguments", $args, $method );
	}

	/**
	 * Get the bookmark schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'bb_bookmark',
			'type'       => 'object',
			'properties' => array(
				'id'                  => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'A unique numeric ID for the bookmark.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'integer',
				),
				'user_id'             => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The ID of the user who created the bookmark.', 'buddyboss-app' ),
					'type'        => 'integer',
					'default'     => get_current_user_id(),
				),
				'type'                => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The type of the bookmark.', 'buddyboss-app' ),
					'enum'        => array_keys( bb_get_bookmark_types() ),
					'type'        => 'string',
				),
				'item_id'             => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The ID of bookmark item.', 'buddyboss-app' ),
					'type'        => 'integer',
				),
				'blog_id'             => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The ID of the current blog site.', 'buddyboss-app' ),
					'type'        => 'integer',
					'default'     => get_current_blog_id(),
				),
				'date'                => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The date the Bookmark was created, in the site\'s timezone.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'string',
					'format'      => 'date-time',
				),
				'date_gmt'            => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The date the Bookmark was created, in the GMT timezone.', 'buddyboss-app' ),
					'readonly'    => true,
					'type'        => 'string',
					'format'      => 'date-time',
				),
				'status'              => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Whether to check the bookmark is active or not.', 'buddyboss-app' ),
					'type'        => 'integer',
				),
				'item_title'          => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Bookmark item title.', 'buddyboss-app' ),
					'type'        => 'string',
					'readonly'    => true,
				),
				'item_featured_image' => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Feature image URLs of the item.', 'buddyboss-app' ),
					'type'        => 'object',
					'readonly'    => true,
					'properties'  => array(
						'full'  => array(
							'context'     => array( 'embed', 'view', 'edit' ),
							'description' => __( 'Feature image URLs with full image size.', 'buddyboss-app' ),
							'type'        => 'string',
							'format'      => 'uri',
						),
						'thumb' => array(
							'context'     => array( 'embed', 'view', 'edit' ),
							'description' => __( 'Feature image URLs with thumb image size.', 'buddyboss-app' ),
							'type'        => 'string',
							'format'      => 'uri',
						),
					),
				),
				'author'              => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Author name of item.', 'buddyboss-app' ),
					'type'        => 'string',
					'readonly'    => true,
				),
				'link'                => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Bookmark item link.', 'buddyboss-app' ),
					'type'        => 'string',
					'readonly'    => true,
				),
			),
		);

		/**
		 * Filters the bookmark schema.
		 *
		 * @param array $schema The endpoint schema.
		 */
		return apply_filters( 'bb_rest_bookmarks_schema', $this->add_additional_fields_schema( $schema ) );
	}

	/**
	 * Get the bookmark type schema, conforming to JSON Schema.
	 *
	 * @return mixed|null
	 */
	public function get_type_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'bb_bookmark_types',
			'type'       => 'object',
			'properties' => array(
				'type'  => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'The type of the bookmark.', 'buddyboss-app' ),
					'type'        => 'string',
					'readonly'    => true,
				),
				'label' => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Label of bookmark type.', 'buddyboss-app' ),
					'type'        => 'object',
					'readonly'    => true,
					'properties'  => array(
						'singular' => array(
							'context'     => array( 'embed', 'view', 'edit' ),
							'description' => __( 'Singular label of bookmark type.', 'buddyboss-app' ),
							'type'        => 'string',
							'readonly'    => true,
						),
						'plural'   => array(
							'context'     => array( 'embed', 'view', 'edit' ),
							'description' => __( 'Plural label of bookmark type.', 'buddyboss-app' ),
							'type'        => 'string',
							'readonly'    => true,
						),
					),
				),
				'count' => array(
					'context'     => array( 'embed', 'view', 'edit' ),
					'description' => __( 'Count of the bookmark items.', 'buddyboss-app' ),
					'type'        => 'integer',
					'readonly'    => true,
				),
			),
		);

		/**
		 * Filters the bookmark types schema.
		 *
		 * @param array $schema The endpoint schema.
		 */
		return apply_filters( 'bb_rest_bookmark_types_schema', $this->add_additional_fields_schema( $schema ) );
	}


	/**
	 * Get the bookmark batch items schema, conforming to JSON Schema.
	 *
	 * @return mixed|null
	 */
	public function batch_items_schema() {
		$item_schema = $this->get_item_schema();

		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'bb_bookmark_batch',
			'type'       => 'object',
			'properties' => $item_schema['properties'],
		);

		/**
		 * Filters the bookmark batch schema.
		 *
		 * @param array $schema The endpoint schema.
		 */
		return apply_filters( 'bb_rest_batch_items_schema', $this->add_additional_fields_schema( $schema ) );
	}

	/**
	 * Validate bookmark user ID/item ID.
	 *
	 * @param WP_REST_Request|int|array $request Full details about the request.
	 *
	 * @return bool|WP_Error
	 */
	public function validate_request_item( $request ) {
		$user_id = $request->get_param( 'user_id' );
		$type    = $request->get_param( 'type' );
		$item_id = $request->get_param( 'item_id' );

		$retval = new WP_Error(
			'bb_rest_authorization_required',
			__( 'Sorry, you are not allowed to create bookmark.', 'buddyboss-app' ),
			array(
				'status' => rest_authorization_required_code(),
			)
		);

		if ( empty( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		// Checked loggedin member.
		if ( get_current_user_id() !== $user_id ) {
			$retval = new WP_Error(
				'bb_rest_authorization_required',
				__( 'Sorry, you are not allowed to create bookmark.', 'buddyboss-app' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		} elseif ( empty( $item_id ) ) {
			$retval = new WP_Error(
				'bb_rest_bookmark_required_item_id',
				__( 'The item ID is required.', 'buddyboss-app' ),
				array(
					'status' => 400,
				)
			);
		} elseif ( empty( $type ) ) {
			$retval = new WP_Error(
				'bb_rest_bookmark_required_item_type',
				__( 'The item type is required.', 'buddyboss-app' ),
				array(
					'status' => 400,
				)
			);
		} else {
			$retval = true;
		}

		/**
		 * Filters the validate request bookmark item data.
		 *
		 * @param boolean|WP_Error          $retval  The validate response.
		 * @param WP_REST_Request|int|array $request Full details about the request.
		 */
		return apply_filters( 'bb_rest_bookmarks_validate_request_item', $retval, $request );
	}

	/**
	 * Update Members endpoint url.
	 *
	 * @param integer $user_id User's ID.
	 *
	 * @return string
	 */
	public function bb_rest_get_user_url( $user_id ) {
		if ( function_exists( 'bp_rest_get_user_url' ) ) {
			return bp_rest_get_user_url( $user_id );
		} elseif ( function_exists( 'boss_bp_rest_get_user_url' ) ) {
			return boss_bp_rest_get_user_url( $user_id );
		}

		return '/wp/v2/users/' . $user_id;
	}

	/**
	 * Whether user can able to access the bookmark or not.
	 *
	 * @param object $bookmark BB_Bookmarks object.
	 *
	 * @return bool
	 */
	public function can_user_see_or_delete( $bookmark ) {
		// Check for admin or if user is bookmarked to this item.
		if ( current_user_can( 'administrator' ) ) {
			return true;
		}

		return ( get_current_user_id() === $bookmark->user_id );
	}

	/**
	 * Check batch limit.
	 *
	 * @param array $items Request items.
	 *
	 * @return bool|WP_Error
	 */
	protected function check_batch_limit( $items ) {
		$limit = apply_filters( 'bb_rest_batch_items_limit', 100, $this->get_normalized_rest_base() );
		$total = 0;

		if ( ! empty( $items ) ) {
			$total += count( $items );
		}

		if ( $total > $limit ) {
			/* translators: %s: items limit */
			return new WP_Error( 'bb_rest_request_entity_too_large', sprintf( __( 'Unable to accept more than %s items for this request.', 'buddyboss-app' ), $limit ), array( 'status' => 413 ) );
		}

		return true;
	}

	/**
	 * Get normalized rest base.
	 *
	 * @return string
	 */
	protected function get_normalized_rest_base() {
		return preg_replace( '/\(.*\)\//i', '', $this->rest_base );
	}

}
