<?php
/**
 * Class Bookmark Rest
 *
 * @package BuddyBossApp\Api\Core
 */

namespace BuddyBossApp\Api\Bookmark;

use BuddyBossApp\Api\Bookmark\V1\RestAPI;
use BuddyBossApp\Bookmark;
use WP_REST_Response;

/**
 * Class Main
 *
 * @package BuddyBossApp\Api\Bookmark
 */
class Main {

	/**
	 * Class instance.
	 *
	 * @var object
	 */
	private static $instance;

	/**
	 * Main constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return Main
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Load Rest API.
	 */
	public function load() {
		// Setting rest.
		add_filter( 'bbapp_settings_rest_response', array( $this, 'bb_settings_rest_response' ), 10, 2 );

		if ( Bookmark::instance()->bb_is_post_bookmark_enable() ) {
			RestAPI::instance();
			add_action( 'rest_api_init', array( $this, 'bb_load_rest_field' ) );
		}

	}

	/**
	 * Function to pass bookmark on BuddyBoss App setting api.
	 *
	 * @param object|array $response Rest response.
	 * @param object|array $request  Rest request.
	 *
	 * @since 1.7.4
	 *
	 * @return mixed
	 */
	public function bb_settings_rest_response( $response, $request ) {
		$response['bookmark']['posts'] = Bookmark::instance()->bb_is_post_bookmark_enable();

		return $response;
	}

	/**
	 * Set headers to let the Client Script be aware of the pagination.
	 *
	 * @param WP_REST_Response $response The response data.
	 * @param integer          $total    The total number of found items.
	 * @param integer          $per_page The number of items per page of results.
	 *
	 * @since 0.1.0
	 *
	 * @return WP_REST_Response $response The response data.
	 */
	public function bb_rest_response_add_total_headers( WP_REST_Response $response, $total = 0, $per_page = 0 ) {
		if ( ! $total || ! $per_page ) {
			return $response;
		}

		$total_items = (int) $total;
		$max_pages   = ceil( $total_items / (int) $per_page );

		$response->header( 'X-WP-Total', $total_items );
		$response->header( 'X-WP-TotalPages', (int) $max_pages );

		return $response;
	}

	/**
	 * Load rest field.
	 *
	 * @since 1.7.4
	 */
	public function bb_load_rest_field() {
		// Register bookmarked field on post api.
		register_rest_field(
			'post',
			'bb_bookmark',
			array(
				'get_callback' => array( $this, 'bb_get_bb_bookmark' ),
				'schema'       => array(
					'description' => __( 'The post bookmarked object.', 'buddyboss-app' ),
					'type'        => 'object',
					'context'     => array( 'embed', 'view', 'single', 'edit' ),
					'properties'  => array(
						'bookmark_id'   => array(
							'context'     => array( 'embed', 'view', 'single', 'edit' ),
							'description' => __( 'A unique numeric ID for the bookmark.', 'buddyboss-app' ),
							'readonly'    => true,
							'type'        => 'integer',
						),
						'is_bookmarked' => array(
							'context'     => array( 'embed', 'view', 'single', 'edit' ),
							'description' => __( 'The post is bookmarked or not.', 'buddyboss-app' ),
							'type'        => 'boolean',
							'default'     => get_current_user_id(),
						),
						'bookmark_date' => array(
							'context'     => array( 'embed', 'view', 'single', 'edit' ),
							'description' => __( 'The date the post was bookmarked, in the site\'s timezone.', 'buddyboss-app' ),
							'type'        => array( 'string', 'null' ),
							'format'      => 'date-time',
							'default'     => get_current_user_id(),
						),
					),
				),
			)
		);
	}

	/**
	 * Added new object into the post rest api.
	 *
	 * @param \WP_Post $post      Post object.
	 * @param string   $attribute The REST Field key used into the REST response.
	 *
	 * @since 1.7.4
	 *
	 * @return array
	 */
	public function bb_get_bb_bookmark( $post, $attribute ) {
		$post_id = $post['id'];

		$bookmark_obj = array(
			'bookmark_id'   => 0,
			'is_bookmarked' => false,
			'bookmark_date' => 0,
		);

		if ( is_user_logged_in() ) {
			$bookmarks = bb_bookmarks_get_bookmark_by_item( 'post', $post_id );
			if ( ! is_wp_error( $bookmarks ) && ! empty( $bookmarks[0] ) ) {
				$bookmark_obj['bookmark_id']   = $bookmarks[0]->id;
				$bookmark_obj['is_bookmarked'] = true;
				$bookmark_obj['bookmark_date'] = mysql_to_rfc3339( $bookmarks[0]->date_recorded ); // phpcs:ignore WordPress.DB.RestrictedFunctions.mysql_to_rfc3339, PHPCompatibility.Extensions.RemovedExtensions.mysql_DeprecatedRemoved
			}
		}

		return $bookmark_obj;
	}

}
