<?php
/**
 * Holds user device related APIs.
 *
 * @package BuddyBossApp\Api\Auth\V2
 */

namespace BuddyBossApp\Api\Auth\V2;

use WP_Error;
use WP_HTTP_Response;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

/**
 * User device class.
 */
class UserDevice {
	/**
	 * API namespace.
	 *
	 * @var string $namespace
	 */
	protected $namespace = 'buddyboss-app/auth/v2';

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * User device table name.
	 *
	 * @var string $table_name
	 */
	protected $table_name = 'bbapp_user_devices';

	/**
	 * UserDevice constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get class instance.
	 *
	 * @since 1.7.7
	 * @return UserDevice
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Hooks/Filters here.
	 *
	 * @since 1.7.7
	 * @return void
	 */
	public function hooks() {
		$this->table_name = bbapp_get_network_table( $this->table_name );

		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
	}

	/**
	 * Register API Routes.
	 *
	 * @since 1.7.7
	 * @return void
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/app-device/',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'add_device' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'device_token' => array(
						'type'        => 'string',
						'required'    => false,
						'description' => __( 'Firebase app device token.', 'buddyboss-app' ),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/app-device/(?P<uuid>[a-zA-Z0-9-]+)',
			array(
				'args' => array(
					'uuid' => array(
						'description' => __( 'A unique device ID for the app.', 'buddyboss-app' ),
						'type'        => 'string',
					),
				),
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'update_device_by_uuid' ),
					'permission_callback' => '__return_true',
					'args'                => array(
						'secret_key'   => array(
							'type'        => 'string',
							'required'    => true,
							'description' => __( 'Secret key for device', 'buddyboss-app' ),
						),
						'device_token' => array(
							'type'        => 'string',
							'required'    => false,
							'description' => __( 'Firebase app device token.', 'buddyboss-app' ),
						),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/delete-app-device/',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'delete_app_device' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'uuid'       => array(
						'type'        => 'string',
						'required'    => true,
						'description' => __( 'Device UUID.', 'buddyboss-app' ),
					),
					'secret_key' => array(
						'type'        => 'string',
						'required'    => true,
						'description' => __( 'Secret key for device', 'buddyboss-app' ),
					),
				),
			)
		);
	}

	/**
	 * Add app device.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @since          1.7.7
	 *
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/auth/v2/app-device
	 * @apiPrivate     false
	 * @apiName        Add app device
	 * @apiGroup       Authentication
	 * @apiVersion     2.0.0
	 * @apiPermission  Public
	 * @apiParam {String=device_token} Device token.
	 * @apiParam {String=device_id} Device id.
	 * @apiParam {String=device_model} Device model.
	 * @apiDescription Add app device.
	 * @apiUse         apidocForAppDevice.
	 */
	public function add_device( $request ) {
		global $wpdb;

		$device_token = $request->get_param( 'device_token' );
		$device_id    = $request->get_param( 'device_id' );
		$device_model = $request->get_param( 'device_model' );
		$uuid         = \BuddyBossApp\Library\Composer::instance()->uuid_instance()->uuid4()->toString();

		/**
		 * Filters user token algorithm
		 *
		 * @param string $hash Hash algorithm.
		 *
		 * @since 1.7.7
		 */
		$hash = apply_filters( 'bbapp_get_user_token_algorithm', 'sha256' );

		/**
		 * Filters user token secret.
		 *
		 * @param string $secret User token secret.
		 *
		 * @since 1.7.7
		 */
		$secret = apply_filters( 'bbapp_user_token_secret', hash( $hash, wp_salt( $uuid ) ) );

		if ( ! empty( $device_token ) ) {
			$device = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $this->table_name WHERE device_token=%s", $device_token ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

			if ( ! empty( $device ) ) {
				$update_data['date_active']  = current_time( 'mysql', 1 );
				$update_data['date_updated'] = current_time( 'mysql', 1 );
				$device_data                 = ( ! empty( $device->data ) ) ? maybe_unserialize( $device->data ) : array();

				if ( empty( $device->uuid ) ) {
					$update_data['uuid'] = $uuid;
				}

				if ( empty( $device_data['device_token_date'] ) ) {
					$update_device_data['device_token_date'] = current_time( 'mysql', 1 );
				}

				if ( empty( $device_data['secret_key'] ) ) {
					$update_device_data['secret_key'] = $secret;
				}

				$update_data = \BuddyBossApp\Auth\UserDevice::instance()->prepare_args( $update_data );

				if ( ! empty( $update_device_data ) ) {
					$update_data['data'] = maybe_serialize( $update_device_data );
				}

				if ( ! empty( $device_id ) ) {
					$update_data['device_id'] = $device_id;
				}

				if ( ! empty( $device_model ) ) {
					$update_data['device_model'] = $device_model;
				}

				$update = $wpdb->update( $this->table_name, $update_data, array( 'id' => $device->id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

				if ( $update ) {
					$get_device_row           = $wpdb->get_row( $wpdb->prepare( "SELECT device_token,data,uuid FROM $this->table_name WHERE id=%d", $device->id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
					$response['uuid']         = ( ! empty( $get_device_row->uuid ) ) ? $get_device_row->uuid : '';
					$get_device_data          = ( ! empty( $get_device_row->data ) ) ? maybe_unserialize( $get_device_row->data ) : array();
					$response['secret_key']   = ( ! empty( $get_device_data ) && ! empty( $get_device_data['secret_key'] ) ) ? $get_device_data['secret_key'] : '';
					$response['device_token'] = ( ! empty( $get_device_row->device_token ) ) ? $get_device_row->device_token : '';
				} else {
					$response = new WP_Error(
						'bbapp_add_app_device',
						__( 'There is an error while adding the device', 'buddyboss-app' ),
						array(
							'status' => 404,
						)
					);
				}
			} else {
				$response = \BuddyBossApp\Auth\UserDevice::instance()->add_new_device( $secret, '', $device_token, $device_id, $device_model );
			}
		} else {
			$response = \BuddyBossApp\Auth\UserDevice::instance()->add_new_device( $secret, '', $device_token, $device_id, $device_model );
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Update app device by uuid.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @since          1.7.7
	 *
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/auth/v2/app-device/:uuid
	 * @apiPrivate     false
	 * @apiName        Update app device by uuid
	 * @apiGroup       Authentication
	 * @apiVersion     2.0.0
	 * @apiPermission  Public
	 * @apiParam {String=uuid} Device uuid.
	 * @apiParam {String=secret_key} Secret key.
	 * @apiParam {String=device_token} Device token.
	 * @apiParam {String=device_id} Device id.
	 * @apiParam {String=device_model} Device model.
	 * @apiDescription Update app device by uuid.
	 * @apiUse         apidocForUpdateAppDeviceByUuid.
	 */
	public function update_device_by_uuid( $request ) {
		global $wpdb;

		$uuid         = $request->get_param( 'uuid' );
		$secret_key   = $request->get_param( 'secret_key' );
		$device_token = $request->get_param( 'device_token' );
		$device_id    = $request->get_param( 'device_id' );
		$device_model = $request->get_param( 'device_model' );

		if ( empty( $secret_key ) ) {
			return rest_ensure_response( new WP_Error( 'rest_bbapp_secret_key', __( 'Secret key is missing.', 'buddyboss-app' ), array( 'status' => 400 ) ) );
		}

		$device_data         = array();
		$device_data_by_uuid = $wpdb->get_var( $wpdb->prepare( "SELECT data from {$this->table_name} WHERE uuid=%s", $uuid ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! $device_data_by_uuid ) {
			return rest_ensure_response( \BuddyBossApp\Auth\UserDevice::instance()->add_new_device( $secret_key, $uuid, $device_token, $device_id, $device_model ) );
		}

		if ( ! empty( $device_data_by_uuid ) ) {
			$device_data = maybe_unserialize( $device_data_by_uuid );

			if ( empty( $device_data['secret_key'] ) ) {
				return rest_ensure_response( new WP_Error( 'bbapp_delete_app_device', __( 'No secret key found with provided UUID.', 'buddyboss-app' ), array( 'status' => 404 ) ) );
			}

			if ( $secret_key !== $device_data['secret_key'] ) {
				return rest_ensure_response( new WP_Error( 'rest_bbapp_secret_key', __( 'There is an error verifying the secret key.', 'buddyboss-app' ), array( 'status' => 400 ) ) );
			}
		}

		$update_args = array();

		if ( ! empty( $device_token ) ) {
			$device_token_hash                = hash( 'sha256', $device_token );
			$update_args['date_active']       = current_time( 'mysql', 1 );
			$device_data['date_updated']      = current_time( 'mysql', 1 );
			$device_data['device_token_date'] = current_time( 'mysql', 1 );
			$update_args['data']              = maybe_serialize( $device_data );
			$update_args['device_token']      = $device_token;
			$update_args['device_token_hash'] = $device_token_hash;

			\BuddyBossApp\Auth\UserDevice::instance()->make_device_token_empty( $device_token, $uuid );
		}

		$update_args = \BuddyBossApp\Auth\UserDevice::instance()->prepare_args( $update_args );

		if ( ! empty( $device_id ) ) {
			$update_args['device_id'] = $device_id;
		}

		if ( ! empty( $device_model ) ) {
			$update_args['device_model'] = $device_model;
		}

		$updated = $wpdb->update( //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$this->table_name,
			$update_args,
			array(
				'uuid' => $uuid,
			)
		);

		if ( ! empty( $updated ) ) {
			$response['uuid']         = $uuid;
			$response['secret_key']   = $secret_key;
			$response['device_token'] = ( ! empty( $device_token ) ) ? $device_token : '';
		} elseif ( 0 === $updated ) {
			$response = new WP_Error(
				'bbapp_update_app_device_by_uuid',
				__( 'The device is already up to date with information.', 'buddyboss-app' ),
				array(
					'status' => 404,
				)
			);
		} else {
			$response = new WP_Error(
				'bbapp_update_app_device_by_uuid',
				__( 'There is an error while updating the device', 'buddyboss-app' ),
				array(
					'status' => 404,
				)
			);
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Delete app device.
	 *
	 * @param WP_REST_Request $request Rest request.
	 *
	 * @since          1.7.7
	 *
	 * @return WP_Error|WP_HTTP_Response|WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/auth/v2/delete-app-device
	 * @apiPrivate     false
	 * @apiName        Delete app device
	 * @apiGroup       Authentication
	 * @apiVersion     2.0.0
	 * @apiPermission  Public
	 * @apiDescription Delete app device.
	 * @apiUse         apidocForAppDevice
	 */
	public function delete_app_device( $request ) {
		$uuid       = $request->get_param( 'uuid' );
		$secret_key = $request->get_param( 'secret_key' );

		return rest_ensure_response( \BuddyBossApp\Auth\UserDevice::instance()->delete_app_device_by_uuid( $uuid, $secret_key ) );
	}
}
