<?php
/**
 * Holds mobile app V2 API related functionality.
 *
 * @package BuddyBossApp\Api\Auth\V2
 */

namespace BuddyBossApp\Api\Auth\V2;

use BuddyBossApp\Auth\Jwt;
use BuddyBossApp\Library\Composer;
use Exception;
use ReflectionClass;
use Tutor\Helpers\QueryHelper;
use TUTOR\RestAuth;
use WP_Error as WP_Error;
use WP_REST_Controller as WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

// Contain functionality for required additional rest api endpoints for Authentication.

/**
 * Class RestAPI
 *
 * @package BuddyBossApp\Api\Auth\V2
 */
class RestAPI extends WP_REST_Controller {

	/**
	 * API namespace.
	 *
	 * @var string $namespace
	 */
	protected $namespace = 'buddyboss-app/auth/v2';

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * RestAPI constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get class instance
	 *
	 * @return RestAPI
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Hooks.
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ), 99 );
		add_filter( 'rest_post_dispatch', array( $this, 'handle_user_suspension_message' ), 10, 3 );
	}

	/**
	 * Register API Routes.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/jwt/login',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'rest_login' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'username'     => array(
						'type'              => 'string',
						'required'          => true,
						'description'       => __( 'Username of user wants to authenticate, Email is also valid.', 'buddyboss-app' ),
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_user( $param );
						},
					),
					'password'     => array(
						'type'        => 'string',
						'required'    => true,
						'description' => __( 'Password of user wants to authenticate.', 'buddyboss-app' ),
					),
					'device_token' => array(
						'type'        => 'string',
						'required'    => false,
						'description' => __( 'Firebase app device token.', 'buddyboss-app' ),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/access-token',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'rest_access_token' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'refresh_token' => array(
						'type'        => 'string',
						'required'    => true,
						'description' => __( 'User Refresh Token.', 'buddyboss-app' ),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/logout',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'rest_logout' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'refresh_token' => array(
						'type'        => 'string',
						'required'    => true,
						'description' => __( 'User Refresh Token.', 'buddyboss-app' ),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/validate',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'validate_token' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'token' => array(
						'type'        => 'string',
						'required'    => true,
						'description' => __( 'Require access token or refresh token.', 'buddyboss-app' ),
					),
				),
			)
		);

		// Added verify account API for V2.
		register_rest_route(
			$this->namespace,
			'/verify',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'verify_account' ),
				'permission_callback' => '__return_true',
				'args'                => array(),
			)
		);
	}

	/**
	 * Handle user suspension message.
	 *
	 * @param WP_REST_Response $response Current response being served.
	 * @param WP_REST_Server   $server   ResponseHandler instance (usually WP_REST_Server).
	 * @param WP_REST_Request  $request  The request that was used to make current response.
	 *
	 * @since 2.1.00
	 * @return WP_REST_Response Response to be served, with "Allow" header if route has allowed methods.
	 */
	public function handle_user_suspension_message( $response, $server, $request ) {
		$data      = $response->get_data();
		$status    = $response->get_status();
		$header    = $request->get_headers();
		$jwt_token = false;

		foreach ( $header as $k => $v ) {
			if ( 'accesstoken' === strtolower( $k ) ) {
				$jwt_token = current( $v );
				break;
			}
		}

		if ( $jwt_token && in_array( $status, array( 403, 401 ), true ) ) {
			// Extract values from the array
			$code    = isset( $data['code'] ) ? $data['code'] : '';
			$message = isset( $data['message'] ) ? $data['message'] : '';
			$status  = isset( $data['data']['status'] ) ? $data['data']['status'] : 0;

			// Create a WP_Error instance
			$error = new WP_Error( $code, $message, array( 'status' => $status ) );

			if ( ! is_wp_error( $error ) ) {
				return $response;
			}

			try {
				$jwt      = Jwt::instance();
				$validate = $jwt->verify_token( $jwt_token );

				if ( is_wp_error( $validate ) && 'bbapp_rest_user_suspended' === $validate->get_error_code() ) {
					$data['code']           = $validate->get_error_code();
					$data['message']        = $validate->get_error_message();
					$data['data']['status'] = 401;

					$response->set_data( $data );
				}
			} catch ( Exception $e ) {
				return $response;
			}
		}

		return $response;
	}

	/**
	 * Rest login.
	 *
	 * @param WP_REST_Request $request Rest request data.
	 *
	 * @return WP_Error
	 * @api            {POST} /wp-json/buddyboss-app/auth/v2/jwt/login Request token
	 * @apiPrivate     true
	 * @apiName        RequestToken
	 * @apiGroup       Authentication
	 * @apiVersion     2.0.0
	 * @apiPermission  Public
	 * @apiDescription Get token on authentication
	 * @apiUse         apidocForRequestTokenV2
	 */
	public function rest_login( $request ) {

		global $bbapp_var;

		$username     = $request->get_param( 'username' );
		$password     = $request->get_param( 'password' );
		$device_token = $request->get_param( 'device_token' );

		if ( empty( $username ) ) {
			return new WP_Error( 'rest_bbapp_jwt_username_req', __( 'A valid username param is required.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( empty( $password ) ) {
			return new WP_Error( 'rest_bbapp_jwt_password_req', __( 'A valid password param is required.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		/**
		 * Hook can be used to trigger a custom message before user login.
		 *
		 * @param bool $flag Validate token.
		 * @param WP_REST_Request $request Rest request.
		 */
		$flag = apply_filters( 'bbapp_validate_before_generate_token', true, $request );

		if ( true !== $flag ) {
			return new WP_Error( 'rest_bbapp_jwt_error', $flag, array( 'status' => 404 ) );
		}

		$accounts = \BuddyBossApp\Auth\Account::instance();

		// Get user details based on username.
		$user = $accounts->get_user( $username );

		/**
		 * User detail check exits or not.
		 */
		if ( ! $user ) {
			return new WP_Error( 'rest_bbapp_jwt_invalid_username', __( 'Unknown username. Check again or try your email address.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}
		if ( ! wp_check_password( $password, $user->user_pass, $user->ID ) ) {
			return new WP_Error(
				'rest_bbapp_jwt_incorrect_password',
				sprintf(
				/* translators: %s: User name. */
					__( 'The password you entered for the username %s is incorrect.', 'buddyboss-app' ),
					$username
				),
				array( 'status' => 400 )
			);
		}
		/**
		 * Check existing user is active or not.
		 */
		$signup = ( isset( $user->data ) ) ? $accounts->get_signup_user( $user->data->user_email ) : false;
		if ( $user && ( ! empty( $signup ) && ! $signup->active ) ) {
			return new Wp_Error( 'bbapp_auth_require_activation', __( 'User account activation is pending.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		/**
		 * Generate the token for user.
		 */
		$jwt = Jwt::instance();

		$token_args = array(
			'expire_at_days' => 1, // allow only 1 day expire for access token. we have refresh token on service for renew.
		);

		$generate_token = $jwt->generate_user_token( $username, $password, $token_args, true );

		return $this->generate_user_token_response( $generate_token, $device_token );

	}


	/**
	 * Rest access token.
	 *
	 * @param WP_REST_Request $request Rest request data.
	 *
	 * @return WP_Error
	 * @apiPrivate     true
	 * @api            {POST} /wp-json/buddyboss-app/auth/v2/jwt/access-token Regenerate token
	 * @apiName        RegenerateAccessToken
	 * @apiGroup       Authentication
	 * @apiVersion     2.0.0
	 * @apiPermission  Public
	 * @apiDescription Regenerate user access token
	 * @apiParam       device_token Firebase app device token.
	 * @apiParam       refresh_token User Refresh Token.
	 */
	public function rest_access_token( $request ) {

		global $bbapp_var;

		$device_token  = $request->get_param( 'device_token' );
		$refresh_token = $request->get_param( 'refresh_token' );

		$jwt = Jwt::instance();

		$validate = $jwt->verify_token( $refresh_token );

		if ( is_wp_error( $validate ) ) {
			return new WP_Error( 'jwt_invalid_token', __( 'Your token is invalid or is expired.', 'buddyboss-app' ), array( 'status' => 401 ) );
		}

		$user = $validate['user'];

		if ( true !== $validate['refresh_token'] ) {
			return new WP_Error( 'jwt_invalid_token', __( 'Given token is not a valid refresh token.', 'buddyboss-app' ), array( 'status' => 401 ) );
		}

		// Everything is validated. lets generate a access token.
		$jwt            = Jwt::instance();
		$generate_token = $jwt->generate_jwt_base( $user, false, $refresh_token );

		return $this->generate_user_token_response( $generate_token, $device_token );

	}

	/**
	 * Handle & Reponse for Successfully Login Situations.
	 *
	 * @param array  $generate_token Generate token.
	 * @param string $device_token   Device token.
	 *
	 * @return WP_Error
	 */
	public function generate_user_token_response( $generate_token, $device_token ) {

		if ( is_wp_error( $generate_token ) ) {
			return $generate_token;
		}

		if ( ! $generate_token ) {
			return new WP_Error( 'jwt_token_error', __( 'Error while generating jwt token.', 'buddyboss-app' ), array( 'status' => 401 ) );
		}

		if ( ! empty( $device_token ) && isset( $generate_token['user_id'] ) && ! empty( $generate_token['user_id'] ) ) {
			if ( function_exists( 'bbapp_notifications' ) ) {
				bbapp_notifications()->register_device_for_user( $generate_token['user_id'], $device_token, $generate_token['access_token'] );
			}
		}

		$response = array(
			'access_token'      => $generate_token['token'], // access token.
			'refresh_token'     => $generate_token['refresh_token'], // refresh token.
			'user_display_name' => $generate_token['user_display_name'], // user display name.
			'user_nicename'     => $generate_token['user_nicename'], // user nicename.
			'user_email'        => $generate_token['user_email'], // user email.
			'user_id'           => $generate_token['user_id'], // user id.
		);

		if ( function_exists( 'tutor_lms' ) ) {
			$response['tutor_lms_auth'] = $this->bbapp_get_tutor_lms_rest_permission();
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Get tutor lms rest permission.
	 *
	 * @since 2.2.80
	 * @return array|mixed
	 */
	public function bbapp_get_tutor_lms_rest_permission() {
		global $wpdb;

		$api_key_secrets = QueryHelper::get_all(
			$wpdb->usermeta,
			array(
				'meta_key' => RestAuth::KEYS_USER_META_KEY, //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
			),
			'umeta_id'
		);

		if ( ! empty( $api_key_secrets ) ) {
			foreach ( $api_key_secrets as $value ) {
				return ! empty( $value->meta_value ) ? json_decode( $value->meta_value ) : array();
			}
		}

		return array();
	}

	/**
	 * Rest logout.
	 *
	 * @param WP_REST_Request $request Rest request data.
	 *
	 * @return array|bool|WP_Error
	 * @apiPrivate     true
	 * @api            {POST} /wp-json/buddyboss-app/auth/v2/jwt/logout Revoke refresh token
	 * @apiName        RevokeRefreshToken
	 * @apiGroup       Authentication
	 * @apiVersion     2.0.0
	 * @apiPermission  Public
	 * @apiDescription Revoke refresh token
	 * @apiParam {String} refresh_token User Refresh Token.
	 */
	public function rest_logout( $request ) {

		$refresh_token = $request->get_param( 'refresh_token' );
		$refresh_token = isset( $refresh_token ) ? $refresh_token : false;

		$jwt = Jwt::instance();

		$revoke = $jwt->revoke_jwt_token( $refresh_token );

		do_action( 'bbapp_auth_delete_jwt_token_request', $request );

		return rest_ensure_response(
			array(
				'revoked' => is_wp_error( $revoke ) ? false : true,
			)
		);
	}

	/**
	 * Validate token.
	 *
	 * @param WP_REST_Request $request Rest request data.
	 *
	 * @return WP_Error
	 * @apiPrivate     true
	 * @api            {POST} /wp-json/buddyboss-app/auth/v2/jwt/validate Validate token
	 * @apiName        ValidateToken
	 * @apiGroup       Authentication
	 * @apiVersion     2.0.0
	 * @apiPermission  Public
	 * @apiDescription Validate if token is valid or not
	 * @apiParam {String} token Access token
	 */
	public function validate_token( $request ) {

		$token = $request->get_param( 'token' );

		if ( empty( $token ) ) {
			return new WP_Error( 'rest_bbapp_jwt_token_req', __( 'A valid token param is required.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$jwt = Jwt::instance();

		$validate = $jwt->verify_token( $token );

		if ( is_wp_error( $validate ) ) {
			return new WP_Error( 'jwt_invalid_token', __( 'Your token is invalid or is expired.', 'buddyboss-app' ), array( 'status' => 401 ) );
		}

		$data = array(
			'user_id' => $validate['user_id'],
		);

		return rest_ensure_response( $data );

	}

	/**
	 * Verify account.
	 *
	 * @param WP_REST_Request $request Rest request data.
	 *
	 * @since          1.5.1
	 * @return WP_Error
	 * @api            {POST} /wp-json/buddyboss-app/auth/v2/verify Verify account
	 * @apiName        VerifyAccount
	 * @apiGroup       Authentication
	 * @apiVersion     2.0.0
	 * @apiPermission  Public
	 * @apiDescription Verify if the account is valid based on associated/attached email
	 * @apiParam {String} username Valid username of user
	 * @apiParam {String} email Valid email of user
	 * @apiParam {Number} code Activation code(_bbapp_activation_code) sent as email
	 */
	public function verify_account( $request ) {

		$username = $request->get_param( 'username' );
		$email    = $request->get_param( 'email' );
		$code     = $request->get_param( 'code' );

		if ( ( empty( $email ) && empty( $username ) ) || empty( $code ) ) {
			return new Wp_Error( 'rest_bbapp_missing_require_fields', __( 'Missing one of required field.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( ! is_numeric( $code ) ) {
			return new Wp_Error( 'rest_bbapp_code_invalid', __( 'Provided code param is containing invalid value.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}
		if ( ! is_email( $email ) && empty( $username ) ) {
			return new Wp_Error( 'rest_bbapp_email_invalid', __( 'Provided email is not containing a valid email format.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}
		if ( empty( $email ) && ! empty( $username ) && ! username_exists( $username ) ) {
			return new Wp_Error( 'rest_bbapp_username_invalid', __( 'Provided username is invalid.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		// If username exists than fetch the email from username.
		if ( ! empty( $username ) && username_exists( $username ) ) {
			$user  = get_user_by( 'login', $username );
			$email = $user->user_email;
		}

		$user_id = \BuddyBossApp\Auth\Account::instance()->activate_account( $email, $code );

		if ( is_wp_error( $user_id ) ) {
			if ( empty( $user_id ) ) {
				return new Wp_Error( 'rest_bbapp_error_activating', __( 'There is an error while activating your account.', 'buddyboss-app' ), array( 'status' => 500 ) );
			} else {
				return $user_id;
			}
		}

		$user = get_userdata( $user_id );

		/**
		 * Validate before generate token.
		 *
		 * @param bool $flag Validate token.
		 * @param WP_REST_Request $request Rest request.
		 */
		$flag = apply_filters( 'bbapp_validate_before_generate_token', true, $request );

		if ( true !== $flag ) {
			return new Wp_Error( 'rest_bbapp_jwt_error', $flag, array( 'status' => 500 ) );
		}

		// Looks everything is working let's send back the token.
		$jwt        = Jwt::instance();
		$token_args = array(
			'expire_at_days' => 1,
			// allow only 1 day expire for access token. we have refresh token on service for renew.
		);

		$generate_token = $jwt->generate_user_token( false, false, $token_args, true, $user );

		if ( is_wp_error( $generate_token ) ) {
			return $generate_token;
		}

		if ( ! $generate_token ) {
			return new Wp_Error( 'jwt_token_error', __( 'Error while generating jwt token.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		return rest_ensure_response( $generate_token );

	}
}
