<?php
/**
 * Contains functionality for required additional rest api endpoints for Authentication.
 *
 * @package BuddyBossApp\Api\Auth\V1
 */

namespace BuddyBossApp\Api\Auth\V1;

use BP_Signup;
use BuddyBossApp\Auth\Jwt;
use BuddyBossApp\Auth\SwitchUser;
use BuddyBossApp\RestErrors;
use WP_Error as WP_Error;
use WP_REST_Controller as WP_REST_Controller;
use WP_REST_Request as WP_REST_Request;
use WP_User;

/**
 * Class RestAPI
 *
 * @package BuddyBossApp\Api\Auth\V1
 */
class RestAPI extends WP_REST_Controller {

	/**
	 * API namespace.
	 *
	 * @var string $namespace
	 */
	protected $namespace = 'buddyboss-app/auth/v1';

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * RestAPI constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get class instance
	 *
	 * @return RestAPI
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
			self::$instance->hooks();
		}

		return self::$instance;
	}

	/**
	 * Filters/hooks here.
	 */
	public function hooks() {
		add_action( 'rest_api_init', array( $this, 'registerRoutes' ), 99 );
	}

	/**
	 * Register Quick & Lite Rest Endpoints
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function registerRoutes() {
		register_rest_route(
			$this->namespace,
			'/register',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'user_register_response_v1' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'username'    => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_user( $param );
						},
					),
					'email'       => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return is_email( $param );
						},
					),
					'description' => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/forgot',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'user_forgot_pass_v1' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'email' => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return is_email( $param );
						},
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/token',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'generate_jwt_token' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'username' => array(
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_user( $param );
						},
					),
					'email'    => array(),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/token',
			array(
				'methods'             => 'DELETE',
				'callback'            => array( $this, 'delete_jwt_token' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'accesstoken' => array(),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/token/validate',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'validate_jwt_token' ),
				'permission_callback' => '__return_true',
				'args'                => array(),
			)
		);

		register_rest_route(
			$this->namespace,
			'/verify',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'verify_account' ),
				'permission_callback' => '__return_true',
				'args'                => array(),
			)
		);

		register_rest_route(
			$this->namespace,
			'/jwt/switch',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'app_switch_user' ),
				'permission_callback' => '__return_true',
				'args'                => array(
					'app_id'         => array(
						'required'    => true,
						'type'        => 'string',
						'description' => __( 'App ID', 'buddyboss-app' ),
					),
					'action'         => array(
						'required'    => true,
						'type'        => 'string',
						'description' => __( 'Switch user action', 'buddyboss-app' ),
						'enum'        => array( 'switch', 'restore' ),
					),
					'switch_user_id' => array(
						'required'    => false,
						'type'        => 'string',
						'description' => __( 'User id', 'buddyboss-app' ),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/reset-password',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'reset_password_callback' ),
				'permission_callback' => function ( $request ) {
					if ( is_wp_error( check_password_reset_key( $request->get_param( 'key' ), $request->get_param( 'login' ) ) ) ) {
						return new WP_Error( 'invalid_password_reset_key', __( 'Password reset key is invalid .', 'buddyboss-app' ), array( 'status' => 404 ) );
					}

					return true;
				},
				'args'                => array(
					'login'            => array(
						'required'          => true,
						'description'       => __( 'Username for which the password needs to be reset.', 'buddyboss-app' ),
						'type'              => 'string',
						'validate_callback' => function ( $param, $request, $key ) {
							return absint( $param );
						},
					),
					'password'         => array(
						'required'          => true,
						'description'       => __( 'New password for the user.', 'buddyboss-app' ),
						'type'              => 'string',
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
					'confirm_password' => array(
						'required'          => true,
						'description'       => __( 'Confirm password for the user.', 'buddyboss-app' ),
						'type'              => 'string',
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
					'key'              => array(
						'required'          => true,
						'description'       => __( 'Password reset key.', 'buddyboss-app' ),
						'type'              => 'string',
						'validate_callback' => function ( $param, $request, $key ) {
							return sanitize_text_field( $param );
						},
					),
				),
			)
		);
	}

	/**
	 * Generate JWT token.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_Error
	 * @api            {POST} /wp-json/buddyboss-app/auth/v1/jwt/token Request token
	 * @apiName        RequestToken
	 * @apiGroup       Authentication
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Get token on authentication
	 * @apiUse         apidocForRequestTokenV1
	 */
	public function generate_jwt_token( $request ) {
		$username = $request->get_param( 'username' );
		$password = $request->get_param( 'password' );

		if ( ! $request->get_param( 'username' ) || '' === $request->get_param( 'username' ) ) {
			return new WP_Error( 'rest_bbapp_jwt_username_req', __( 'a valid username param is required.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		if ( ! $request->get_param( 'password' ) || '' === $request->get_param( 'password' ) ) {
			return new WP_Error( 'rest_bbapp_jwt_password_req', __( 'a valid password param is required.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		/**
		 * Filters before generate token.
		 *
		 * @param bool            $flag    Token generate.
		 * @param WP_REST_Request $request Rest request.
		 */
		$flag = apply_filters( 'bbapp_validate_before_generate_token', true, $request );

		if ( true !== $flag ) {
			return new Wp_Error( 'rest_bbapp_jwt_error', $flag, array( 'status' => 500 ) );
		}

		$accounts = \BuddyBossApp\Auth\Account::instance();
		// Get user details based on username.
		$user = $accounts->get_user( $username );

		/**
		 * User detail check exits or not.
		 */
		if ( ! $user ) {
			return new WP_Error( 'rest_bbapp_jwt_invalid_username', __( 'Unknown username. Check again or try your email address.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		if ( ! wp_check_password( $password, $user->user_pass, $user->ID ) ) {
			return new WP_Error(
				'rest_bbapp_jwt_incorrect_password',
				sprintf(
				/* translators: %s: User name. */
					__( 'The password you entered for the username %s is incorrect.', 'buddyboss-app' ),
					$username
				),
				array( 'status' => 400 )
			);
		}

		/**
		 * Check existing user is active or not.
		 */
		$signup = ( isset( $user->data ) ) ? $accounts->get_signup_user( $user->data->user_email ) : false;

		if ( $user && ( ! empty( $signup ) && ! $signup->active ) ) {
			return new Wp_Error( 'bbapp_auth_require_activation', __( 'User account activation is pending.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$jwt = Jwt::instance();

		$token_args = array(
			'expire_at_days' => 90, // allow 90 days as v1 don't have refresh token feature.
		);

		$generate_token = $jwt->generate_user_token( $username, $password, $token_args );

		if ( is_wp_error( $generate_token ) ) {
			return $generate_token;
		}

		if ( ! $generate_token ) {
			return new Wp_Error( 'jwt_token_error', __( 'Error while generating jwt token.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$device_token = $request->get_param( 'deviceToken' );

		if ( '' !== $device_token && isset( $generate_token['user_id'] ) && ! empty( $generate_token['user_id'] ) ) {
			if ( function_exists( 'bbapp_notifications' ) ) {
				bbapp_notifications()->register_device_for_user( $generate_token['user_id'], $device_token, $generate_token['token'] );
			}
		}

		return rest_ensure_response( $generate_token );
	}

	/**
	 * Switch user rest callback function.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return array|bool|mixed|WP_Error|\WP_HTTP_Response|\WP_REST_Response
	 * @api            {POST} /wp-json/buddyboss-app/auth/v1/jwt/switch Request token
	 * @apiName        RequestToken
	 * @apiGroup       Authentication
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Get token on authentication     *
	 */
	public function app_switch_user( $request ) {

		/**
		 * Login is required on this endpoint.
		 */
		if ( ! is_user_logged_in() ) {
			return RestErrors::instance()->user_not_logged_in();
		}

		$user_id     = isset( $request['switch_user_id'] ) ? $request['switch_user_id'] : false;
		$user_action = isset( $request['action'] ) ? $request['action'] : false;

		if ( ! $user_action || '' === $user_action ) {
			return new WP_Error( 'rest_bbapp_jwt_action_req', __( 'a valid action param is required.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$switch_user = SwitchUser::instance();

		if ( 'restore' === $user_action && empty( $switch_user->get_original_user() ) ) {
			return new WP_Error( 'rest_bbapp_not_switched', __( "You don't have any user switched.", 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		// Returns the switched user.
		$switched_user = $switch_user->switch_to_user( ( 'switch' !== $user_action ), $user_id );

		if ( $switched_user ) {
			return array(
				'switched_user_id' => $switched_user,
			);
		} else {
			return new Wp_Error( 'bbapp_auth_user_not_permission', __( 'Could not switch users.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}
	}

	/**
	 * Reset user password.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_Error
	 * @api            {POST} /wp-json/buddyboss-app/auth/v1/reset-password
	 * @apiName        ResetPassword
	 * @apiGroup       Authentication
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Reset User password
	 */
	public function reset_password_callback( $request ) {
		$new_password     = $request->get_param( 'password' );
		$confirm_password = $request->get_param( 'confirm_password' );

		if ( $new_password !== $confirm_password ) {
			return new WP_Error( 'password_mismatch', __( 'New password and confirm password do not match.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		$user = get_user_by( 'login', $request->get_param( 'login' ) );

		if ( ! $user->ID ) {
			return new WP_Error( 'invalid_user', __( 'User not found.', 'buddyboss-app' ), array( 'status' => 404 ) );
		}

		reset_password( $user, $new_password );

		return rest_ensure_response(
			array(
				'reset' => true,
			)
		);
	}

	/**
	 * Delete JWT token.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return array|bool|WP_Error
	 * @api            {DELETE} /wp-json/buddyboss-app/auth/v1/jwt/token Delete token
	 * @apiName        DeleteToken
	 * @apiGroup       Authentication
	 * @apiVersion     1.0.0
	 * @apiPermission  LoggedInUser
	 * @apiDescription Delete token on authentication
	 * @apiParam {String} accesstoken Token
	 */
	public function delete_jwt_token( $request ) {
		if ( ! is_user_logged_in() ) {
			return new Wp_Error( 'rest_not_logged_in', esc_html__( 'You are not currently logged in.', 'buddyboss-app' ), array( 'status' => rest_authorization_required_code() ) );
		}

		$jwt    = Jwt::instance();
		$revoke = $jwt->revoke_jwt_token();

		if ( is_wp_error( $revoke ) ) {
			return $revoke;
		}

		/**
		 * Fires after JWT token deleted.
		 *
		 * @type WP_REST_Request $request Request.
		 */
		do_action( 'bbapp_auth_delete_jwt_token_request', $request );

		return rest_ensure_response(
			array(
				'revoked' => true,
			)
		);
	}

	/**
	 * Validate JWT token.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_Error
	 * @api            {POST} /wp-json/buddyboss-app/auth/v1/jwt/token/validate Validate token
	 * @apiName        ValidateToken
	 * @apiGroup       Authentication
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Validate existing token. Can be invalid or expired.
	 * @apiHeader {String} accessToken Auth token
	 */
	public function validate_jwt_token( $request ) {
		$jwt      = Jwt::instance();
		$validate = $jwt->verify_token();

		if ( is_wp_error( $validate ) ) {
			return new Wp_Error( 'jwt_invalid_token', __( 'Your token is invalid or is expired.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		$data = array(
			'user_id' => $validate['user_id'],
		);

		return rest_ensure_response( $data );
	}

	/**
	 * Forgot password API.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return array
	 * @api            {POST} /wp-json/buddyboss-app/auth/v1/forgot Reset password
	 * @apiName        ResetPassword
	 * @apiGroup       Authentication
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Reset password
	 * @apiParam {String} user_login User login
	 */
	public function user_forgot_pass_v1( $request ) {
		$user_login = $request['user_login'];
		$response   = rest_ensure_response( \BuddyBossApp\Auth\Common::api_retrieve_password( $user_login ) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		return array( 'success' => true );
	}

	/**
	 * Get user register response.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_Error
	 * @api            {POST} /wp-json/buddyboss-app/auth/v1/register Register account
	 * @apiName        RegisterAccount
	 * @apiGroup       Authentication
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription This endpoint allow user to register a account on WordPress, note on multisite registration are only allowed on main network site.
	 * @apiParam {String} username Username
	 * @apiParam {String} password Password
	 * @apiParam {String} email Email-address
	 * @apiParam {String} [first_name=null] First name
	 * @apiParam {String} [last_name=null] Last name
	 * @apiParam {String} [display_name=null] Display name
	 * @apiParam {String} [description=null] This is user description same like WordPress core has.
	 */
	public function user_register_response_v1( WP_REST_Request $request ) {
		/**
		 * Check if current WordPress is multisite & endpoint is not being accessed from mainsite.
		 * On that case we will throw the error because we don't support authentication on subsites.
		 */
		if ( is_multisite() && ! is_main_site() ) {
			return new Wp_Error( 'rest_bbapp_register_subsite_not_supported', esc_html__( 'Registration is only available on main network site.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		/**
		 * Throw the error if user registration are disabled on admin settings.
		 */
		if ( ! get_option( 'users_can_register' ) ) {
			return new Wp_Error( 'rest_bbapp_register_is_not_allowed', esc_html__( 'User registration is currently not allowed.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		/**
		 * Filter user registration inputs.
		 *
		 * @param WP_REST_Request $request Rest request.
		 */
		$request = apply_filters( 'bbapp_auth_rest_user_registration_inputs', $request );

		/**
		 * Validate Incoming Inputs.
		 */
		if ( ! $request->get_param( 'username' ) || '' === $request->get_param( 'username' ) ) {
			return new Wp_Error( 'rest_bbapp_register_username_req', esc_html__( 'A valid username param is required.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		if ( ! $request->get_param( 'email' ) || '' === $request->get_param( 'email' ) ) {
			return new Wp_Error( 'rest_bbapp_register_email_req', esc_html__( 'A valid email param is required.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		if ( ! $request->get_param( 'password' ) || '' === $request->get_param( 'password' ) ) {
			return new Wp_Error( 'rest_bbapp_register_password_req', esc_html__( 'A valid password param is required.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		/**
		 * Validate if UserName or Email Exists or not.
		 */
		if ( username_exists( $request->get_param( 'username' ) ) ) {
			return new Wp_Error( 'rest_bbapp_register_username_exists', esc_html__( 'Given username already exists.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		if ( email_exists( $request->get_param( 'email' ) ) ) {
			return new Wp_Error( 'rest_bbapp_register_email_exists', esc_html__( 'Given email already exists.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		/**
		 * Pre Registration Check Hook.
		 * This filter can used to extra custom validation or for aborting the registration process.
		 *
		 * @param bool $pre_check Pre check user inputs.
		 * @param WP_REST_Request $request Rest request.
		 */
		$pre_check = apply_filters( 'bbapp_auth_rest_user_registration_pre_check', true, $request );

		if ( is_wp_error( $pre_check ) ) {
			return $pre_check;
		}

		/**
		 * Prepare the User Data for Registration.
		 */
		$data = array();

		if ( $request->get_param( 'first_name' ) ) {
			$data['first_name'] = $request->get_param( 'first_name' );
		}

		if ( $request->get_param( 'last_name' ) ) {
			$data['last_name'] = $request->get_param( 'last_name' );
		}

		if ( $request->get_param( 'display_name' ) ) {
			$data['display_name'] = $request->get_param( 'display_name' );
		}

		if ( $request->get_param( 'description' ) ) {
			$data['description'] = $request->get_param( 'description' );
		}

		// Registration form type.
		$settings = \BuddyBossApp\Auth\Auth::instance()->is_app_registration_form();

		/**
		 * If BuddyPress is Enabled than do Registration BuddyPress Way.
		 * If platform plugin activate and select registration type is buddyboss_registration then user will create from here.
		 */
		if ( function_exists( 'bp_core_signup_user' ) && 'buddyboss_registration' === $settings ) {
			// Make use of buddypress function.
			if ( is_multisite() ) {
				$data['user_pass'] = wp_hash_password( $request->get_param( 'password' ) );
			}

			$user_id = bp_core_signup_user( $request->get_param( 'username' ), $request->get_param( 'password' ), $request->get_param( 'email' ), $data );

			// Bp Signup.
			$is_in_activation = BP_Signup::get(
				array(
					'user_login' => $request->get_param( 'username' ),
				)
			);

			$bp_activation_pending = false;

			if ( isset( $is_in_activation['signups'][0] ) ) {
				$bp_activation_pending = true;
			}
		} else {
			/**
			 * If it's a Multi Site than do the Multi Site way.
			 */
			if ( is_multisite() ) {
				$user_id           = false;
				$data['user_pass'] = wp_hash_password( $request->get_param( 'password' ) );

				wpmu_signup_user( $request->get_param( 'username' ), $request->get_param( 'email' ), $data );

				$bp_activation_pending = true;
				/**
				 * Use the core registration feature.
				 */
			} else {
				// If platform plugin activate and select registration type is wordpress_registration then user will create from here.
				$user_id = wp_create_user( $request->get_param( 'username' ), $request->get_param( 'password' ), $request->get_param( 'email' ) );

				if ( is_wp_error( $user_id ) ) {
					return new Wp_Error( 'rest_bbapp_register_error_' . $user_id->get_error_code(), $user_id->get_error_message(), array( 'status' => 500 ) );
				} else {
					$data['ID'] = $user_id;

					wp_update_user( $data );
				}
			}
		}

		/**
		 * Post Registration Action
		 * This action hook will fire once the user is created.
		 * Can be useful to inject custom code to perform on particular User.
		 *
		 * @param int $user_id User id.
		 * @param WP_REST_Request $request Rest request.
		 */
		do_action( 'bbapp_auth_rest_user_registration_after', $user_id, $request );

		$data = array(
			'user_id'               => $user_id,
			'email'                 => $request->get_param( 'email' ),
			'username'              => $request->get_param( 'username' ),
			'first_name'            => $request->get_param( 'first_name' ),
			'last_name'             => $request->get_param( 'last_name' ),
			'display_name'          => $request->get_param( 'display_name' ),
			'description'           => $request->get_param( 'description' ),
			'bp_activation_pending' => false,
			'activation_pending'    => false,
		);

		if ( isset( $bp_activation_pending ) ) {
			$data['bp_activation_pending'] = $bp_activation_pending;
			$data['activation_pending']    = $bp_activation_pending;
		}

		$response = rest_ensure_response( $data );

		return $response;
	}

	/**
	 * Verify user account.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_Error
	 * @api            {POST} /wp-json/buddyboss-app/auth/v1/verify Verify account
	 * @apiName        VerifyAccount
	 * @apiGroup       Authentication
	 * @apiVersion     1.0.0
	 * @apiPermission  Public
	 * @apiDescription Verify if the account is valid based on associated/attached email
	 * @apiParam {String} username Valid username of user
	 * @apiParam {String} email Valid email of user
	 * @apiParam {Number} code Activation code(_bbapp_activation_code) sent as email
	 */
	public function verify_account( WP_REST_Request $request ) {
		$username = $request->get_param( 'username' );
		$email    = $request->get_param( 'email' );
		$code     = $request->get_param( 'code' );

		if ( ( empty( $email ) && empty( $username ) ) || empty( $code ) ) {
			return new Wp_Error( 'rest_bbapp_missing_require_fields', __( 'Missing one of required field.', 'buddyboss-app' ), array( 'status' => 400 ) );
		}

		if ( ! is_numeric( $code ) ) {
			return new Wp_Error( 'rest_bbapp_code_invalid', __( 'Provided code param is containing invalid value.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}
		if ( ! is_email( $email ) && empty( $username ) ) {
			return new Wp_Error( 'rest_bbapp_email_invalid', __( 'Provided email is not containing a valid email format.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}
		if ( empty( $email ) && ! empty( $username ) && ! username_exists( $username ) ) {
			return new Wp_Error( 'rest_bbapp_username_invalid', __( 'Provided username is invalid.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		// If username exists than fetch the email from username.
		if ( ! empty( $username ) && username_exists( $username ) ) {
			$user  = get_user_by( 'login', $username );
			$email = $user->user_email;
		}

		$user_id = \BuddyBossApp\Auth\Account::instance()->activate_account( $email, $code );

		if ( is_wp_error( $user_id ) ) {
			if ( empty( $user_id ) ) {
				return new Wp_Error( 'rest_bbapp_error_activating', __( 'There is an error while activating your account.', 'buddyboss-app' ), array( 'status' => 500 ) );
			} else {
				return $user_id;
			}
		}

		$user = get_userdata( $user_id );
		$flag = apply_filters( 'bbapp_validate_before_generate_token', true, $request );

		if ( true !== $flag ) {
			return new Wp_Error( 'rest_bbapp_jwt_error', $flag, array( 'status' => 500 ) );
		}

		// Looks everything is working let's send back the token.
		$jwt        = Jwt::instance();
		$token_args = array(
			'expire_at_days' => 90, // allow 90 days as v1 don't have refresh token feature.
		);

		$generate_token = $jwt->generate_user_token( false, false, $token_args, false, $user );

		if ( is_wp_error( $generate_token ) ) {
			return $generate_token;
		}

		if ( ! $generate_token ) {
			return new Wp_Error( 'jwt_token_error', __( 'Error while generating jwt token.', 'buddyboss-app' ), array( 'status' => 500 ) );
		}

		return rest_ensure_response( $generate_token );
	}
}
