<?php
/**
 * Class Access Rules Rest
 *
 * @package BuddyBossApp\Api\Core
 */

namespace BuddyBossApp\Api\AccessControls;

use BuddyBossApp\AccessControls\AccessRule;
use BuddyBossApp\AccessControls\Core\Settings\AppPages;
use BuddyBossApp\AccessControls\Core\Settings\Pages;
use BuddyBossApp\AccessControls\Core\Settings\Posts;

/**
 * Class Main
 *
 * @package BuddyBossApp\Api\AccessControls
 */
class Main {

	/**
	 * Class instance.
	 *
	 * @var object
	 */
	private static $instance;

	/**
	 * Main constructor.
	 */
	public function __construct() {
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return mixed|object Class object
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Load Rest API.
	 */
	public function load() {
		// Setting rest.
		add_filter( 'bbapp_settings_rest_response', array( $this, 'bbapp_settings_rest_response' ), 10, 2 );

		// App support cpt (post, page, app page).
		add_filter( 'rest_prepare_' . AppPages::instance()->post_type, array( $this, 'get_bb_access' ), 10, 3 );
		add_filter( 'rest_prepare_' . Pages::instance()->post_type, array( $this, 'get_bb_access' ), 10, 3 );
		add_filter( 'rest_prepare_' . Posts::instance()->post_type, array( $this, 'get_bb_access' ), 10, 3 );

		// Menus rest.
		add_filter( 'bbapp_menu_filter_menu_data', array( $this, 'get_menu_bb_access' ), 10, 1 );

		// Learndash.
		add_filter( 'bbapp_ld_rest_prepare_course', array( $this, 'get_bb_access' ), 10, 3 );
		add_filter( 'bbapp_ld_rest_prepare_lesson', array( $this, 'get_bb_access' ), 10, 3 );
		add_filter( 'bbapp_ld_rest_prepare_topic', array( $this, 'get_bb_access' ), 10, 3 );
		add_filter( 'bbapp_ld_rest_prepare_quiz', array( $this, 'get_bb_access' ), 10, 3 );
		add_filter( 'bbapp_deeplinking_details', array( $this, 'get_bb_access_for_deeplinking' ), 999, 3 );

		// Tutor LMS.
		add_filter( 'bbapp_tutor_rest_prepare_course', array( $this, 'get_bb_access' ), 10, 3 );
		add_filter( 'bbapp_tutor_rest_prepare_lesson', array( $this, 'get_bb_access' ), 10, 3 );
		add_filter( 'bbapp_tutor_rest_prepare_quiz', array( $this, 'get_bb_access' ), 10, 3 );
		add_filter( 'bbapp_tutor_rest_prepare_assignment', array( $this, 'get_bb_access' ), 10, 3 );
	}

	/**
	 * Function to pass access controls on BuddyBoss App setting api.
	 *
	 * @param object|array $response Rest response.
	 * @param object|array $request  Rest request.
	 *
	 * @since 1.5.2.1
	 *
	 * @return mixed
	 */
	public function bbapp_settings_rest_response( $response, $request ) {

		/**
		 * Get bb access field on rest.
		 *
		 * @param array             $access_controls Default settings.
		 * @param \WP_REST_Response $request    The response object.
		 *
		 * @since 1.5.2.1
		 */

		$access_controls             = apply_filters( 'bbapp_settings_rest_response_access_controls', array(), $request );
		$response['access_controls'] = $access_controls;

		return $response;
	}

	/**
	 * Get bb access field on rest.
	 *
	 * @param \WP_REST_Response $response The response object.
	 * @param \WP_Post          $post     Post object.
	 * @param \WP_REST_Request  $request  Request object.
	 *
	 * @since 1.5.2.1
	 *
	 * @return \WP_REST_Response
	 */
	public function get_bb_access( $response, $post, $request ) {

		$default = array(
			'can_access'       => true,
			'restrict_message' => null,
		);

		/**
		 * Get bb access field on rest.
		 *
		 * @param array             $default  Default settings.
		 * @param \WP_REST_Response $response The response object.
		 * @param \WP_Post          $post     Post object.
		 * @param \WP_REST_Request  $request  Request object.
		 *
		 * @since 1.5.2.1
		 */
		$response->data['bb_access'] = $this->bb_access_controls_data( $default, $response, $post, $request );

		return $response;
	}

	/**
	 * Get bb access for deeplinking.
	 *
	 * @param \WP_REST_Response $response The response object.
	 * @param string            $url      The url.
	 * @param string            $version  The version.
	 *
	 * @since 1.8.30
	 *
	 * @return mixed
	 */
	public function get_bb_access_for_deeplinking( $response, $url, $version ) {

		if ( isset( $response['bb_access'] ) ) {
			return $response;
		}

		$scree_item_id = isset( $response['item_id'] ) ? $response['item_id'] : 0;

		if ( ! empty( $scree_item_id ) ) {
			$scree_namespace = isset( $response['namespace'] ) ? $response['namespace'] : '';

			if ( 'core' === $scree_namespace ) {
				$page = get_page_by_path( $scree_item_id );

				if ( $page ) {
					$response['bb_access'] = $this->get_bb_access_by_post_id( $page->ID );
				}
			}
		} else {

			$url_to_post_id = url_to_postid( $url );

			if ( ! empty( $url_to_post_id ) ) {
				$response['bb_access'] = $this->get_bb_access_by_post_id( $url_to_post_id );
			} else {
				$page_id = $this->get_page_id_from_url( $url );
				if ( $page_id ) {
					$response['bb_access'] = $this->get_bb_access_by_post_id( $page_id );
				} else {
					$page_for_posts = get_option( 'page_for_posts' );
					$link           = get_permalink( $page_for_posts );

					if ( untrailingslashit( $url ) === untrailingslashit( $link ) ) {
						$response['bb_access'] = $this->get_bb_access_by_post_id( $page_for_posts );
					}
				}
			}
		}

		return $response;
	}

	/**
	 * Get post id from url.
	 *
	 * @param string|int $page_url The page url.
	 *
	 * @since 1.8.30
	 *
	 * @return false|mixed
	 */
	public function get_page_id_from_url( $page_url ) {
		$page = get_page_by_path( basename( $page_url ) );

		return ! empty( $page ) ? $page->ID : 0;
	}

	/**
	 * Get bb access by post id.
	 *
	 * @param int|string $post_id The post id.
	 *
	 * @since 1.8.30
	 *
	 * @return true[]
	 */
	public function get_bb_access_by_post_id( $post_id ) {
		$bb_access = array(
			'can_access' => true,
		);

		$post      = get_post( $post_id );
		$rule_data = AccessRule::instance()->get_access_rule( $post->ID, $post->post_type );

		// Check if stored access group conditions are still exist in code.
		if ( ! empty( $rule_data['rule_data'] ) ) {
			$bb_access['can_access'] = bb_access_controls_user_can_access_rule( $rule_data['rule_data'] );
		}

		$bb_access['restrict_message'] = ( true !== $bb_access['can_access'] && isset( $rule_data['rule_data']['restricted_message'] ) ) ? bb_access_controls_get_restrict_message( $rule_data['rule_data']['restricted_message'] ) : null;

		return $bb_access;
	}

	/**
	 * Function to get bb access.
	 *
	 * @param array             $bb_access Access rule format for rest.
	 * @param \WP_REST_Response $response  The response object.
	 * @param \WP_Post          $post      Post object.
	 * @param \WP_REST_Request  $request   Request object.
	 *
	 * @since 1.5.2.1
	 * @return array|mixed
	 */
	public function bb_access_controls_data( $bb_access, $response, $post, $request ) {
		$rule_data  = AccessRule::instance()->get_access_rule( $post->ID, $post->post_type );
		$can_access = true;
		// Check if stored access group conditions are still exist in code.
		if ( ! empty( $rule_data['rule_data'] ) ) {
			$can_access = bb_access_controls_user_can_access_rule( $rule_data['rule_data'] );
		}
		$bb_access['can_access']       = $can_access;
		$bb_access['restrict_message'] = ( true !== $can_access && isset( $rule_data['rule_data']['restricted_message'] ) ) ? bb_access_controls_get_restrict_message( $rule_data['rule_data']['restricted_message'] ) : null;

		return apply_filters( 'bb_access_controls_data', $bb_access, $response, $post, $request );
	}

	/**
	 * Get bb access menu field on rest.
	 *
	 * @param array $menu The menu object.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function get_menu_bb_access( $menu ) {
		$default = array(
			'can_access'       => true,
			'restrict_message' => null,
		);

		/**
		 * Get bb access field on rest.
		 *
		 * @param array $default Default settings.
		 * @param array $menu    The menu object.
		 *
		 * @since 1.5.2.1
		 */
		$menu['bb_access'] = apply_filters( 'bb_access_controls_menu_data', $default, $menu );

		return $menu;
	}
}
