<?php
/**
 * Holds user profile functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

defined( 'ABSPATH' ) || exit;

/**
 * Class UserProfile to use add custom tab in user profile section.
 *
 * @package BuddyBossApp\Admin
 */
class UserProfile {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Variable for users profile.
	 *
	 * @var $capability
	 */
	public $capability;

	/**
	 * User profile.
	 *
	 * @var $user_profile
	 */
	public $user_profile;

	/**
	 * Current user id.
	 *
	 * @var $current_user_id
	 */
	public $current_user_id;

	/**
	 * User id.
	 *
	 * @var $user_id
	 */
	public $user_id;

	/**
	 * If users own profile.
	 *
	 * @var $is_self_profile
	 */
	public $is_self_profile;

	/**
	 * Screen id.
	 *
	 * @var $screen_id
	 */
	public $screen_id;

	/**
	 * Stats metabox.
	 *
	 * @var $stats_metabox
	 */
	public $stats_metabox;

	/**
	 * Edit url.
	 *
	 * @var $edit_url
	 */
	public $edit_url;

	/**
	 * Users url.
	 *
	 * @var $users_url
	 */
	public $users_url;

	/**
	 * Subsite activated.
	 *
	 * @var $subsite_activated
	 */
	public $subsite_activated;

	/**
	 * Users page.
	 *
	 * @var $users_page
	 */
	public $users_page;

	/**
	 * Users screen.
	 *
	 * @var $users_screen
	 */
	public $users_screen;

	/**
	 * Redirect.
	 *
	 * @var $redirect
	 */
	public $redirect;

	/**
	 * Push notifications for screen id for edit user's profile page.
	 *
	 * @var $user_push_page
	 */
	public $user_push_page = '';

	/**
	 * User device page.
	 *
	 * @var string $user_devices_page
	 */
	public $user_devices_page = '';

	/**
	 * Push notification.
	 *
	 * @var $edit_push_notification
	 */
	public $edit_push_notification;

	/**
	 * User notification list.
	 *
	 * @var $user_notification_list
	 */
	public $user_notification_list;

	/**
	 * Edit user devices.
	 *
	 * @var $edit_user_devices
	 */
	public $edit_user_devices;

	/**
	 * User device list.
	 *
	 * @var $user_devices_list
	 */
	public $user_devices_list;

	/**
	 * Access groups for screen id for edit user's profile page.
	 *
	 * @since 1.5.2.1
	 *
	 * @var $user_access_group_page
	 */
	public $user_access_group_page = '';

	/**
	 * Edit access groups.
	 *
	 * @var $edit_access_groups
	 */
	public $edit_access_groups;

	/**
	 * User access group list.
	 *
	 * @var $user_access_groups_list
	 */
	public $user_access_groups_list;

	/**
	 * Get the instance of class.
	 *
	 * @return BuddyBossApp\Admin\UserProfile
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * UserProfile constructor.
	 */
	public function __construct() {
	}

	/**
	 * Load users hooks.
	 */
	public function load() {
		$this->capability = is_main_network() ? 'manage_network_users' : 'edit_users';

		// The Edit Push and Devices Screen id.
		$this->user_push_page         = '';
		$this->edit_push_notification = '';
		$this->user_devices_page      = '';
		$this->edit_user_devices      = '';
		// The Edit Access Group Screen id.
		$this->user_access_group_page = '';
		$this->edit_access_groups     = '';
		// The Show Profile Screen id.
		$this->user_profile = is_network_admin() ? 'users' : 'profile';
		// The current user id.
		$this->current_user_id = get_current_user_id();
		// The user id being edited.
		$this->user_id = 0;
		// Is a member editing their own profile.
		$this->is_self_profile = false;
		// The screen ids to load specific css for.
		$this->screen_id = array();
		// The stats metabox default position.
		$this->stats_metabox = new \StdClass();
		$this->edit_url      = '';
		// Data specific to sign ups.
		$this->users_page   = '';
		$this->users_url    = bbapp_get_admin_url( 'users.php' );
		$this->users_screen = is_main_network() ? 'users-network' : 'users';
		// Specific config: BuddBossApp is not network activated.
		$this->subsite_activated = (bool) is_multisite() && ! is_main_site();
		// When BuddBossApp is not network activated, only Super Admin can moderate signups.
		if ( ! empty( $this->subsite_activated ) ) {
			$this->capability = 'manage_network_users';
		}

		// Add some page specific output to the <head>.
		add_action( 'admin_head', array( $this, 'admin_head' ), 999 );

		// Add menu item to all users menu.
		add_action( 'admin_menu', array( $this, 'admin_menus' ), 5 );
		add_action( 'network_admin_menu', array( $this, 'admin_menus' ), 5 );
		add_action( 'user_admin_menu', array( $this, 'user_profile_menu' ), 5 );

		// Editing users of a specific site.
		add_action( 'admin_head-site-users.php', array( $this, 'profile_admin_head' ) );

		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			add_action( 'bb_members_admin_user_profile_nav', array( $this, 'bbapp_admin_user_profile_nav' ), 10, 2 );
		} else {
			// Create the Profile Navigation (Profile/Extended Profile).
			add_action( 'edit_user_profile', array( $this, 'bbapp_admin_user_profile_nav' ), 99, 1 );
			add_action( 'show_user_profile', array( $this, 'bbapp_admin_user_profile_nav' ), 99, 1 );
		}
	}

	/**
	 * Get the user ID.
	 *
	 * Look for $_GET['user_id']. If anything else, force the user ID to the
	 * current user's ID so they aren't left without a user to edit.
	 *
	 * @return int
	 */
	private function get_user_id() {
		if ( ! empty( $this->user_id ) ) {
			return $this->user_id;
		}

		$this->user_id = (int) get_current_user_id();
		$user_id_get   = ( ! empty( $_GET['user_id'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['user_id'] ) ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		// We'll need a user ID when not on self profile.
		if ( ! empty( $user_id_get ) ) {
			$this->user_id = (int) $user_id_get;
		}

		return $this->user_id;
	}

	/**
	 * Can the current user edit the one displayed.
	 *
	 * This might be replaced by more granular capabilities
	 * in the future.
	 *
	 * @param int $user_id ID of the user being checked for edit ability.
	 *
	 * @return bool
	 */
	private function member_can_edit( $user_id = 0 ) {
		$retval = false;

		// Bail if no user ID was passed.
		if ( empty( $user_id ) ) {
			return $retval;
		}

		// Member can edit if they are viewing their own profile.
		if ( $this->current_user_id === $user_id ) {
			$retval = true;
		} else {
			$retval = current_user_can( 'edit_user', $user_id );
		}

		return $retval;
	}

	/**
	 * Get admin notice when saving a user or member profile.
	 *
	 * @return array
	 */
	private function get_user_notice() {
		// Setup empty notice for return value.
		$notice          = array();
		$updated_request = ( ! empty( $_REQUEST['updated'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['updated'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		// Updates.
		if ( ! empty( $updated_request ) ) {
			switch ( $updated_request ) {
				case 'device':
					$notice = array(
						'class'   => 'updated',
						'message' => __( 'Logged out of selected device.', 'buddyboss-app' ),
					);
					break;
				case 'devices':
					$notice = array(
						'class'   => 'updated',
						'message' => __( 'Logged out from all devices.', 'buddyboss-app' ),
					);
					break;
			}
		}

		$error_request = ( ! empty( $_REQUEST['error'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['error'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		// Errors.
		if ( ! empty( $error_request ) ) {
			switch ( $error_request ) {
				case 'device':
					$notice = array(
						'class'   => 'error',
						'message' => __( 'Unable to log out from the selected device.', 'buddyboss-app' ),
					);
					break;
				case 'devices':
					$notice = array(
						'class'   => 'error',
						'message' => __( 'Unable to log out from all devices.', 'buddyboss-app' ),
					);
					break;
			}
		}

		return $notice;
	}

	/**
	 * Create the /user/ admin Profile submenus for all members.
	 */
	public function user_profile_menu() {
		// Setup the hooks array.
		$hooks = array();

		if ( current_user_can( 'administrator' ) ) {
			if ( bbapp_is_active( 'push_notification' ) ) {
				// Add the faux "User Push Notifications" submenu page.
				$hooks['user_push']   = add_submenu_page(
					'profile.php',
					__( 'User Push Notifications', 'buddyboss-app' ),
					__( 'User Push Notifications', 'buddyboss-app' ),
					'exist',
					'bbapp-user-push',
					array( $this, 'user_admin' )
				);
				$this->user_push_page = $hooks['user_push'];
			}
			// Add the faux "User Devices" submenu page.
			$hooks['user_devices']   = add_submenu_page(
				'profile.php',
				__( 'User Devices', 'buddyboss-app' ),
				__( 'User Devices', 'buddyboss-app' ),
				'exist',
				'bbapp-user-devices',
				array( $this, 'user_admin' )
			);
			$this->user_devices_page = $hooks['user_devices'];

			if ( bbapp_is_active( 'access_controls' ) ) {
				// Add the faux "User Group Access" submenu page.
				$hooks['user_access_group']   = add_submenu_page(
					'profile.php',
					__( 'User Access Groups', 'buddyboss-app' ),
					__( 'User Access Groups', 'buddyboss-app' ),
					'exist',
					'bbapp-user-access-groups',
					array( $this, 'user_admin' )
				);
				$this->user_access_group_page = $hooks['user_access_group'];
			}
		}

		// Setup the screen ID's.
		$this->screen_id = array(
			$this->user_push_page . '-user',
			$this->user_devices_page . '-user',
			$this->user_access_group_page . '-user',
			$this->user_profile . '-user',
		);

		// Loop through new hooks and add method actions.
		foreach ( $hooks as $key => $hook ) {
			add_action( "load-{$hook}", array( $this, $key . '_admin_load' ) );
		}

		// Add the profile_admin_head method to proper admin_head actions.
		add_action( "admin_head-{$this->user_push_page}", array( $this, 'profile_admin_head' ) );
		add_action( "admin_head-{$this->user_devices_page}", array( $this, 'profile_admin_head' ) );
		add_action( "admin_head-{$this->user_access_group_page}", array( $this, 'profile_admin_head' ) );
		add_action( 'admin_head-profile.php', array( $this, 'profile_admin_head' ) );
	}

	/**
	 * Create the All Users / Profile > Edit Profile and All Users submenus.
	 */
	public function admin_menus() {
		// Setup the hooks array.
		$hooks = array();

		if ( current_user_can( 'administrator' ) ) {
			if ( bbapp_is_active( 'push_notification' ) ) {
				// Manage user's push notification.
				$hooks['user_push']   = add_submenu_page(
					$this->user_profile . '.php',
					__( 'Edit Push Notifications', 'buddyboss-app' ),
					__( 'Edit Push Notifications', 'buddyboss-app' ),
					'read',
					'bbapp-user-push',
					array( $this, 'user_admin' )
				);
				$this->user_push_page = $hooks['user_push'];
			}
			// Manage user's profile.
			$hooks['user_devices']   = add_submenu_page(
				'profile.php',
				__( 'Edit Devices', 'buddyboss-app' ),
				__( 'Edit Devices', 'buddyboss-app' ),
				'exist',
				'bbapp-user-devices',
				array( $this, 'user_admin' )
			);
			$this->user_devices_page = $hooks['user_devices'];


			if ( bbapp_is_active( 'access_controls' ) ) {
				$hooks['user_access_group']   = add_submenu_page(
					$this->user_profile . '.php',
					__( 'Edit Access Groups', 'buddyboss-app' ),
					__( 'Edit Access Groups', 'buddyboss-app' ),
					'read',
					'bbapp-user-access-groups',
					array( $this, 'user_admin' )
				);
				$this->user_access_group_page = $hooks['user_access_group'];
			}
		}

		$edit_page        = 'user-edit';
		$profile_page     = 'profile';
		$this->users_page = 'users';

		// Self profile check is needed for this pages.
		$page_head = array(
			$edit_page . '.php',
			$profile_page . '.php',
			$this->user_push_page,
			$this->user_devices_page,
			$this->user_access_group_page,
			$this->users_page . '.php',
		);

		// Append '-network' to each array item if in network admin.
		if ( is_network_admin() ) {
			$edit_page                    .= '-network';
			$profile_page                 .= '-network';
			$this->user_push_page         .= '-network';
			$this->user_devices_page      .= '-network';
			$this->user_access_group_page .= '-network';
			$this->users_page             .= '-network';
		}

		// Setup the screen ID's.
		$this->screen_id = array(
			$edit_page,
			$this->user_push_page,
			$this->user_devices_page,
			$this->user_access_group_page,
			$profile_page,
		);

		// Loop through new hooks and add method actions.
		foreach ( $hooks as $key => $hook ) {
			add_action( "load-{$hook}", array( $this, $key . '_admin_load' ) );
		}

		// Add the profile_admin_head method to proper admin_head actions.
		foreach ( $page_head as $head ) {
			add_action( "admin_head-{$head}", array( $this, 'profile_admin_head' ) );
		}
	}

	/**
	 * Highlight the Users menu if on Edit Profile and check if on the user's admin profile.
	 */
	public function profile_admin_head() {
		global $submenu_file, $parent_file;

		// Is the user editing their own profile?
		if ( is_user_admin() || ( defined( 'IS_PROFILE_PAGE' ) && IS_PROFILE_PAGE ) ) {
			$this->is_self_profile = true;

			// Is the user attempting to edit their own profile.
		} elseif ( isset( $_GET['user_id'] ) || ( $this->get_current_user_screen() ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$this->is_self_profile = (bool) ( $this->get_user_id() === $this->current_user_id );
		}

		// Force the parent file to users.php to open the correct top level menu
		// but only if not editing a site via the network site editing page.
		if ( 'sites.php' !== $parent_file ) {
			$parent_file  = 'users.php'; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
			$submenu_file = 'users.php'; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		}

		// Editing your own profile, so recheck some vars.
		if ( true === $this->is_self_profile ) {
			// Use profile.php as the edit page.
			$edit_page = 'profile.php';

			// Set profile.php as the parent & sub files to correct the menu nav.
			if ( is_blog_admin() || is_user_admin() ) {
				$parent_file  = 'profile.php'; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
				$submenu_file = 'profile.php'; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
			}

			// Not editing yourself, so use user-edit.php.
		} else {
			$edit_page = 'user-edit.php';
		}

		$tab_admin_url  = '';
		$edit_admin_url = '';

		if ( is_user_admin() ) {
			$tab_admin_url  = user_admin_url( 'profile.php' );
			$edit_admin_url = user_admin_url( 'profile.php' );
		} elseif ( is_blog_admin() ) {
			$tab_admin_url  = admin_url( 'users.php' );
			$edit_admin_url = admin_url( $edit_page );
		} elseif ( is_network_admin() ) {
			$tab_admin_url  = network_admin_url( 'users.php' );
			$edit_admin_url = network_admin_url( $edit_page );
		}

		if ( bbapp_is_active( 'push_notification' ) ) {
			$this->edit_push_notification = add_query_arg( array( 'page' => 'bbapp-user-push' ), $tab_admin_url );
		}
		$this->edit_user_devices      = add_query_arg( array( 'page' => 'bbapp-user-devices' ), $tab_admin_url );

		if ( bbapp_is_active( 'access_controls' ) ) {
			$this->edit_access_groups = add_query_arg( array( 'page' => 'bbapp-user-access-groups' ), $tab_admin_url );
		}

		$this->edit_url = $edit_admin_url;
	}

	/**
	 * Remove the Edit Profile page.
	 *
	 * We add these pages in order to integrate with WP's Users panel, but
	 * we want them to show up as a row action of the WP panel, not as separate
	 * subnav items under the Users menu.
	 */
	public function admin_head() {
		if ( bbapp_is_active( 'push_notification' ) ) {
			remove_submenu_page( 'users.php', 'bbapp-user-push' );
			remove_submenu_page( 'profile.php', 'bbapp-user-push' );
		}

		remove_submenu_page( 'users.php', 'bbapp-user-devices' );
		remove_submenu_page( 'profile.php', 'bbapp-user-devices' );

		if ( bbapp_is_active( 'access_controls' ) ) {
			remove_submenu_page( 'users.php', 'bbapp-user-access-groups' );
			remove_submenu_page( 'profile.php', 'bbapp-user-access-groups' );
		}
	}

	/**
	 * Create the Profile navigation in Edit User & Edit Profile pages.
	 *
	 * @param object|null $user   User to create profile navigation for.
	 * @param string      $active Which profile to highlight.
	 *
	 * @return string|null
	 */
	public function bbapp_admin_user_profile_nav( $user = null, $active = 'WordPress' ) {
		// Bail if no user ID exists here.
		if ( empty( $user->ID ) ) {
			return;
		}

		// Add the user ID to query arguments when not editing yourself.
		if ( false === $this->is_self_profile ) {
			$query_args = array( 'user_id' => $user->ID );
		} else {
			$query_args = array();
		}

		// Setup the two distinct "edit" URL's.
		$bbapp_push_active          = false;
		$bbapp_device_active        = false;
		$bbapp_access_groups_active = false;
		$wp_active                  = ' nav-tab-active';

		if ( bbapp_is_active( 'push_notification' ) ) {
			if ( 'bbapp-user-push' === $active ) {
				$bbapp_push_active = ' nav-tab-active';
				$wp_active         = false;
			}
		}

		if ( 'bbapp-user-devices' === $active ) {
			$bbapp_device_active = ' nav-tab-active';
			$wp_active           = false;
		}

		if ( bbapp_is_active( 'access_controls' ) && 'bbapp-user-access-groups' === $active ) {
			$bbapp_access_groups_active = ' nav-tab-active';
			$wp_active                  = false;
		}

		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			$this->profile_admin_head();

			if ( bbapp_is_active( 'push_notification' ) ) {
				$this->push_notification_tab_html( $bbapp_push_active, $query_args );
			}

			$this->device_tab_html( $bbapp_device_active, $query_args );

			if ( bbapp_is_active( 'access_controls' ) ) {
				$this->access_groups_tab_html( $bbapp_access_groups_active, $query_args );
			}
		} else {
			?>
			<h2 id="profile-nav" class="bbapp-profile-nav nav-tab-wrapper">
				<?php
				/**
				 * In configs where BuddBossApp is not network activated, as regular
				 * admins do not have the capacity to edit other users, we must add
				 * this check.
				 */
				if ( current_user_can( 'edit_user', $user->ID ) ) {
					?>
					<a class="nav-tab<?php echo esc_attr( $wp_active ); ?>" href="<?php echo esc_url( add_query_arg( $query_args, $this->edit_url ) ); ?>">
						<?php esc_html_e( 'Profile', 'buddyboss-app' ); ?>
					</a>

				<?php } ?>
				<?php
				if ( bbapp_is_active( 'push_notification' ) ) {
					$this->push_notification_tab_html( $bbapp_push_active, $query_args );
				}

				$this->device_tab_html( $bbapp_device_active, $query_args );

				if ( bbapp_is_active( 'access_controls' ) ) {
					$this->access_groups_tab_html( $bbapp_access_groups_active, $query_args );
				}
				?>
			</h2>
			<?php
		}
	}

	/**
	 * Push notification tab in profile.
	 *
	 * @param string $bbapp_push_active Is tabb active.
	 * @param array  $query_args        argument of query param.
	 */
	public function push_notification_tab_html( $bbapp_push_active, $query_args ) {
		if ( current_user_can( 'administrator' ) ) {
			?>
			<a class="nav-tab<?php echo esc_attr( $bbapp_push_active ); ?>" href="<?php echo esc_url( add_query_arg( $query_args, $this->edit_push_notification ) ); ?>">
				<?php esc_html_e( 'Push Notifications', 'buddyboss-app' ); ?>
			</a>
			<?php
		}
	}

	/**
	 * Set up the user's profile admin page for Push notification.
	 *
	 * Loaded before the page is rendered, this function does all initial
	 * setup, including: processing form requests, registering contextual
	 * help, and setting up screen options.
	 */
	public function user_push_admin_load() {
		// Get the user ID.
		$user_id = $this->get_user_id();

		// Can current user edit this profile?
		if ( ! $this->member_can_edit( $user_id ) ) {
			wp_die( esc_html__( 'You cannot edit the requested user.', 'buddyboss-app' ) );
		}

		$request_uri = ! empty( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
		// User devices.
		$redirect_to = remove_query_arg(
			array(
				'action',
				'error',
				'updated',
				'delete_device',
			),
			$request_uri
		);
		$doaction    = ! empty( $_REQUEST['action'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['action'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		/**
		 * Filters the allowed actions for use in the user admin page.
		 *
		 * @param array $value Array of allowed actions to use.
		 */
		$allowed_actions = apply_filters( 'bbapp_admin_push_allowed_actions', array( 'update', 'spam', 'ham' ) );

		// User push screen.
		if ( bbapp_is_active( 'push_notification' ) ) {
			$this->user_notification_list = new \BuddyBossApp\Notification\UserNotificationList();
		}

		// Prepare the display of the Community Profile screen.
		if ( ! in_array( $doaction, $allowed_actions, true ) ) {
			get_current_screen()->add_help_tab(
				array(
					'id'      => 'bbapp-user-device-overview',
					'title'   => __( 'Overview', 'buddyboss-app' ),
					'content' =>
						'<p>' . __( 'This is the admin view of a user\'s devices.', 'buddyboss-app' ) . '</p>',
				)
			);

			// Help panel - sidebar links.
			get_current_screen()->set_help_sidebar(
				'<p><strong>' . __( 'For more information:', 'buddyboss-app' ) . '</strong></p>' .
				'<p>' . __( '<a href="https://www.buddyboss.com/resources/docs/app/">Documentation</a>', 'buddyboss-app' ) . '</p>'
			);

		} else {
			$this->redirect = $redirect_to;

			/**
			 * Fires at end of user profile admin load if doaction does not match any available actions.
			 *
			 * @param string $doaction Current bulk action being processed.
			 * @param int    $user_id  Current user ID.
			 * @param array  $_REQUEST Current $_REQUEST global.
			 * @param string $redirect Determined redirect url to send user to.
			 */
			do_action_ref_array(
				'bbapp_admin_update_user_push_notification',
				array(
					$doaction,
					$user_id,
					map_deep( $_REQUEST, 'sanitize_text_field' ), // phpcs:ignore WordPress.Security.NonceVerification.Recommended
					$this->redirect,
				)
			);

			wp_safe_redirect( $this->redirect );
			exit();
		}
	}

	/**
	 * Device tab in profile.
	 *
	 * @since 1.5.2.1
	 *
	 * @param string $bbapp_active Is tabb active.
	 * @param array  $query_args   argument of query param.
	 */
	public function device_tab_html( $bbapp_active, $query_args ) {
		if ( current_user_can( 'administrator' ) ) {
			?>
			<a class="nav-tab<?php echo esc_attr( $bbapp_active ); ?>" href="<?php echo esc_url( add_query_arg( $query_args, $this->edit_user_devices ) ); ?>">
				<?php esc_html_e( 'Devices', 'buddyboss-app' ); ?>
			</a>
			<?php
		}
	}

	/**
	 * Set up the user's profile admin page for Devices.
	 *
	 * Loaded before the page is rendered, this function does all initial
	 * setup, including: processing form requests, registering contextual
	 * help, and setting up screen options.
	 */
	public function user_devices_admin_load() {
		// Get the user ID.
		$user_id = $this->get_user_id();

		// Can current user edit this profile?
		if ( ! $this->member_can_edit( $user_id ) ) {
			wp_die( esc_html__( 'You cannot edit the requested user.', 'buddyboss-app' ) );
		}

		$request_uri = ! empty( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
		// User devices.
		$redirect_to = remove_query_arg(
			array(
				'action',
				'error',
				'updated',
				'delete_device',
			),
			$request_uri
		);
		$doaction    = ! empty( $_REQUEST['action'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['action'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		/**
		 * Filters the allowed actions for use in the user admin page.
		 *
		 * @param array $value Array of allowed actions to use.
		 */
		$allowed_actions = apply_filters(
			'bbapp_admin_device_allowed_actions',
			array(
				'update',
				'delete_device',
				'delete_all_device',
				'spam',
				'ham',
			)
		);

		$this->user_devices_list = new \BuddyBossApp\Auth\UserDevice\UserDevicesList();

		// Prepare the display of the Community Profile screen.
		if ( ! in_array( $doaction, $allowed_actions, true ) ) {
			get_current_screen()->add_help_tab(
				array(
					'id'      => 'bbapp-user-device-overview',
					'title'   => __( 'Overview', 'buddyboss-app' ),
					'content' =>
						'<p>' . __( 'This is the admin view of a user\'s devices.', 'buddyboss-app' ) . '</p>',
				)
			);

			// Help panel - sidebar links.
			get_current_screen()->set_help_sidebar(
				'<p><strong>' . __( 'For more information:', 'buddyboss-app' ) . '</strong></p>' .
				'<p>' . __( '<a href="https://www.buddyboss.com/resources/docs/app/">Documentation</a>', 'buddyboss-app' ) . '</p>'
			);

		} else {
			$this->redirect = $redirect_to;

			/**
			 * Fires at end of user profile admin load if doaction does not match any available actions.
			 *
			 * @param string $doaction Current bulk action being processed.
			 * @param int    $user_id  Current user ID.
			 * @param array  $_REQUEST Current $_REQUEST global.
			 * @param string $redirect Determined redirect url to send user to.
			 */
			do_action_ref_array(
				'bbapp_admin_update_user_device',
				array(
					$doaction,
					$user_id,
					map_deep( $_REQUEST, 'sanitize_text_field' ), // phpcs:ignore WordPress.Security.NonceVerification.Recommended
					$this->redirect,
				)
			);

			wp_safe_redirect( $this->redirect );
			exit();
		}
	}

	/**
	 * Access group tab in profile.
	 *
	 * @since 1.5.2.1
	 *
	 * @param string $bbapp_active Is tabb active.
	 * @param array  $query_args   argument of query param.
	 */
	public function access_groups_tab_html( $bbapp_active, $query_args ) {
		if ( current_user_can( 'administrator' ) ) {
			?>
			<a class="nav-tab<?php echo esc_attr( $bbapp_active ); ?>" href="<?php echo esc_url( add_query_arg( $query_args, $this->edit_access_groups ) ); ?>">
				<?php esc_html_e( 'Access Groups', 'buddyboss-app' ); ?>
			</a>
			<?php
		}
	}

	/**
	 * Set up the user's profile admin page for Access group.
	 *
	 * Loaded before the page is rendered, this function does all initial
	 * setup, including: processing form requests, registering contextual
	 * help, and setting up screen options.
	 *
	 * @since 1.5.2.1
	 */
	public function user_access_group_admin_load() {
		// Get the user ID.
		$user_id = $this->get_user_id();

		// Can current user edit this profile?
		if ( ! $this->member_can_edit( $user_id ) ) {
			wp_die( esc_html__( 'You cannot edit the requested user.', 'buddyboss-app' ) );
		}

		// User access group screen.
		if ( bbapp_is_active( 'access_controls' ) ) {
			$this->user_access_groups_list = new \BuddyBossApp\Admin\AccessControls\AccessGroups\GroupsList();
		}
	}

	/**
	 * Display the user's profile.
	 */
	public function user_admin() {
		// Get the user ID.
		$user_id = $this->get_user_id();
		$user    = get_user_to_edit( $user_id );

		// Construct title.
		if ( true === $this->is_self_profile ) {
			$title = esc_Html__( 'Profile', 'buddyboss-app' );
		} else {
			/* translators: %s: User's display name. */
			$title = sprintf( 'Edit User %s', $user->display_name );
		}
		$request_uri = ! empty( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
		// Construct URL for form.
		$request_url     = remove_query_arg(
			array(
				'action',
				'error',
				'updated',
				'spam',
				'ham',
			),
			$request_uri
		);
		$form_action_url = add_query_arg( 'action', 'update', $request_url );
		$wp_http_referer = false;

		if ( ! empty( $_REQUEST['wp_http_referer'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$wp_http_referer = wp_unslash( $_REQUEST['wp_http_referer'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$wp_http_referer = remove_query_arg( array( 'action', 'updated' ), $wp_http_referer );
			$wp_http_referer = wp_validate_redirect( esc_url_raw( $wp_http_referer ) );
		}

		// Prepare notice for admin.
		$notice = $this->get_user_notice();

		if ( ! empty( $notice ) ) :
			?>
			<div
				<?php
				if ( 'updated' === $notice['class'] ) :
					?>
					id="message" <?php endif; ?>class="<?php echo esc_attr( $notice['class'] ); ?>">
				<p><?php echo esc_html( $notice['message'] ); ?></p>

				<?php if ( ! empty( $wp_http_referer ) && ( 'updated' === $notice['class'] ) ) : ?>
					<p>
						<a href="<?php echo esc_url( $wp_http_referer ); ?>"><?php esc_html_e( '&larr; Back to Users', 'buddyboss-app' ); ?></a>
					</p>
				<?php endif; ?>
			</div>
		<?php endif; ?>

		<div class="wrap" id="community-profile-page">
			<?php if ( version_compare( $GLOBALS['wp_version'], '4.8', '>=' ) ) : ?>
				<h1 class="wp-heading-inline"><?php echo esc_html( $title ); ?></h1>

				<?php if ( empty( $this->is_self_profile ) ) : ?>
					<?php if ( current_user_can( 'create_users' ) ) : ?>
						<a href="user-new.php" class="page-title-action">
							<?php esc_html_e( 'Add New', 'buddyboss-app' ); ?>
						</a>
					<?php elseif ( is_multisite() && current_user_can( 'promote_users' ) ) : ?>
						<a href="user-new.php" class="page-title-action">
							<?php esc_html_e( 'Add Existing', 'buddyboss-app' ); ?>
						</a>
					<?php endif; ?>
				<?php endif; ?>
				<hr class="wp-header-end">
			<?php else : ?>
				<h1><?php echo esc_html( $title ); ?>

					<?php if ( empty( $this->is_self_profile ) ) : ?>
						<?php if ( current_user_can( 'create_users' ) ) : ?>
							<a href="user-new.php" class="add-new-h2">
								<?php esc_html_e( 'Add New', 'buddyboss-app' ); ?>
							</a>
						<?php elseif ( is_multisite() && current_user_can( 'promote_users' ) ) : ?>
							<a href="user-new.php" class="add-new-h2">
								<?php esc_html_e( 'Add Existing', 'buddyboss-app' ); ?>
							</a>
						<?php endif; ?>
					<?php endif; ?>
				</h1>
			<?php endif; ?>

			<?php if ( ! empty( $user ) ) : ?>
				<?php
				if ( class_exists( 'BP_Members_Admin', false ) ) {
					$bp_member_admin = new \BP_Members_Admin();
					$bp_member_admin->profile_admin_head();
					$bp_member_admin->profile_nav( $user, $this->get_current_user_screen() );
				} else {
					$this->bbapp_admin_user_profile_nav( $user, $this->get_current_user_screen() );
				}
				?>

				<form action="<?php echo esc_url( $form_action_url ); ?>" id="bbapp-your-profile" method="post">
					<div id="poststuff">
						<div id="post-body" class="metabox-holder">
							<?php
							if ( 'bbapp-user-push' === $this->get_current_user_screen() && bbapp_is_active( 'push_notification' ) ) {
								$this->user_notification_list->prepare_items();
								$this->user_notification_list->display();
							} elseif ( 'bbapp-user-devices' === $this->get_current_user_screen() ) {
								$this->user_devices_list->prepare_items();
								/**
								 * Fires before user device list.
								 *
								 * @param object $user User object.
								 */
								do_action( 'before_bbapp_user_devices_list', $user );

								$this->user_devices_list->display();

								/**
								 * Fires after user device list.
								 *
								 * @param object $user              User object.
								 * @param object $user_devices_list User device list.
								 */
								do_action( 'after_bbapp_user_devices_list', $user, $this->user_devices_list );
							}

							if ( bbapp_is_active( 'access_controls' ) && 'bbapp-user-access-groups' === $this->get_current_user_screen() ) {
								$this->user_access_groups_list->prepare_items();
								/**
								 * Fires before access group list.
								 *
								 * @param object $user User object.
								 */
								do_action( 'before_bbapp_user_access_groups_list', $user );

								$this->user_access_groups_list->display();

								/**
								 * Fires after access group list.
								 *
								 * @param object $user                    User object.
								 * @param object $user_access_groups_list Access group list..
								 */
								do_action( 'after_bbapp_user_access_groups_list', $user, $this->user_access_groups_list );
							}
							?>
						</div><!-- #post-body -->
					</div><!-- #poststuff -->
				</form>
			<?php else : ?>
				<p>
					<?php
					printf( '%1$s <a href="%2$s">%3$s</a>', esc_html__( 'No user found with this ID.', 'buddyboss-app' ), esc_url( bbapp_get_admin_url( 'users.php' ) ), esc_html__( 'Go back and try again.', 'buddyboss-app' ) );
					?>
				</p>
			<?php endif; ?>
		</div><!-- .wrap -->
		<?php
	}

	/**
	 * Current user screen.
	 *
	 * @return mixed|string
	 */
	protected function get_current_user_screen() {
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! empty( $page_get ) ) {
			return $page_get;
		}

		return '';
	}
}
