<?php
/**
 * Telemetry class.
 *
 * @package BuddyBossApp\Tools
 */

namespace BuddyBossApp\Admin\Tools;

use BuddyBossApp\ManageApp;
use WP_Error;
use WP_Theme;

/**
 * Telemetry class.
 */
class Telemetry {
	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return Telemetry
	 * @since 2.2.70
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
			self::$instance->setup_actions();
		}

		return self::$instance;
	}

	/**
	 * Settings constructor.
	 *
	 * @since 2.2.70
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Filters/hooks here.
	 *
	 * @since 2.2.70
	 */
	public function load() {
		if ( ! wp_next_scheduled( 'bbapp_telemetry_report_cron_event' ) ) {
			wp_schedule_event(
				strtotime( 'next Sunday midnight' ),
				'weekly',
				'bbapp_telemetry_report_cron_event'
			);
		}

		// Schedule the single event in next 10 minute.
		if ( ! bbapp_get_option( 'bbapp_telemetry_report_single_cron_event_scheduled' ) && ! wp_next_scheduled( 'bbapp_telemetry_report_single_cron_event' ) ) {
			wp_schedule_single_event( time() + ( 10 * MINUTE_IN_SECONDS ), 'bbapp_telemetry_report_single_cron_event' );
			update_option( 'bbapp_telemetry_report_single_cron_event_scheduled', 1 );
		}

		add_action( 'bbapp_telemetry_report_cron_event', array( $this, 'send_telemetry_data' ) );
	}

	/**
	 * Setup actions for telemetry reporting.
	 *
	 * @since BuddyBoss 2.2.70
	 */
	public function setup_actions() {
		add_action( 'bbapp_telemetry_report_single_cron_event', array( $this, 'send_telemetry_data' ) );
		add_action( 'admin_notices', array( $this, 'bbapp_telemetry_admin_notice' ) );
		add_action( 'wp_ajax_dismiss_bbapp_telemetry_notice', array( $this, 'bbapp_telemetry_notice_dismissed' ) );
	}

	/**
	 * Send telemetry data.
	 *
	 * @return WP_Error | void
	 * @since 2.2.70
	 */
	public function send_telemetry_data() {
		$bbapp_telemetry_option = ManageApp::instance()->get_app_setting( 'app_telemetry' );

        if (
			'disable' === $bbapp_telemetry_option ||
			! $this->bbapp_whitelist_domain_for_telemetry()
		) {
			return;
		}

		$telemetry_data = $this->bbapp_collect_site_data();

		$api_url =
			'aHR0cHM6Ly9h' .
			'bmFseXRpY3Mu' .
			'YnVkZHlib3Nz' .
			'LmNvbS93cC1q' .
			'c29uL3dwL3Yx' .
			'L2JiLXRlbGVt' .
			'ZXRyeQ==';

		$args = array(
			'headers'   => array(
				'Accept'       => 'application/json;ver=1.0',
				'Content-Type' => 'application/json; charset=UTF-8',
				'Site-URL'     => get_site_url(),
			),
			'timeout'   => 10,
			'blocking'  => true,
			'body'      => wp_json_encode( $telemetry_data ),
			'sslverify' => apply_filters( 'https_local_ssl_verify', false ), // Local requests.
		);

		$raw_response = bbapp_remote_post( base64_decode( $api_url ), $args );

		if ( ! empty( $raw_response ) && is_wp_error( $raw_response ) ) {
			unset( $telemetry_data, $auth_key, $api_url, $args );

			return $raw_response;
		} elseif ( ! empty( $raw_response ) && 200 !== wp_remote_retrieve_response_code( $raw_response ) ) {
			unset( $telemetry_data, $auth_key, $api_url, $args );

			return new WP_Error( 'server_error', wp_remote_retrieve_response_message( $raw_response ) );
		} else {
			unset( $telemetry_data, $auth_key, $api_url, $args, $raw_response );

			return new WP_Error( 'server_error', __( 'An error occurred while sending the telemetry report.', 'buddyboss-app' ) );
		}
	}

	/**
	 * Generate unique id.
	 *
	 * @return string
	 * @since 2.2.70
	 */
	public function uuid() {
		$uuid_key = 'bb-telemetry-uuid';
		$uuid     = bbapp_get_option( $uuid_key );

		if ( empty( $uuid ) ) {
			// Definitely not cryptographically secure but
			// close enough to provide a unique id.
			$uuid = md5( uniqid() . site_url() );
			update_site_option( $uuid_key, $uuid );
		}

		return $uuid;
	}

	/**
	 * Collect site data for telemetry reporting.
	 *
	 * @return array An array of collected site data.
	 * @since BuddyBoss 2.2.70
	 */
	public function bbapp_collect_site_data() {
		global $wpdb;

		$bbapp_telemetry_data = array(
			'site_url'      => site_url(),
			'admin_url'     => admin_url(),
			'php_version'   => phpversion(),
			'mysql_version' => $wpdb->db_version(),
			'db_provider'   => $wpdb->db_server_info(),
			'os'            => php_uname( 's' ),
			'webserver'     => $_SERVER['SERVER_SOFTWARE'],
			'plugins'       => $this->bbapp_get_plugins_data(),
			'themes'        => $this->bbapp_get_themes_data(),
			'is_multisite'  => is_multisite(),
		);

		if ( function_exists( 'bbapp_telemetry_data' ) ) {
			$bbapp_telemetry_data = bbapp_telemetry_data( $bbapp_telemetry_data );
		}

		$result = array(
			'uuid' => $this->uuid(),
			'data' => $bbapp_telemetry_data,
		);

		unset( $bbapp_telemetry_data );

		return $result;
	}

	/**
	 * Retrieves the list of installed plugins along with their name, slug, version, and activation status.
	 *
	 * @return array List of plugins with 'name', 'slug', 'version', and 'active' keys.
	 * @since BuddyBoss 2.2.70
	 */
	public function bbapp_get_plugins_data() {
		$plugin_list = function_exists( 'get_plugins' ) ? get_plugins() : array();

		wp_cache_delete( 'plugins', 'plugins' );

		$plugins = array();

		if ( ! empty( $plugin_list ) ) {
			foreach ( $plugin_list as $slug => $info ) {
				$plugins[] = array(
					'name'    => $info['Name'],
					'slug'    => $slug,
					'version' => $info['Version'],
					'active'  => function_exists( 'is_plugin_active' ) && is_plugin_active( $slug ),
				);
			}
		}

		unset( $plugin_list );

		return $plugins;
	}

	/**
	 * Retrieves data for the active theme, including the parent theme if a child theme is active.
	 *
	 * @return array List of themes with 'name', 'stylesheet', 'version', and 'template' keys.
	 * @since BuddyBoss 2.2.70
	 */
	public function bbapp_get_themes_data() {
		$theme  = function_exists( 'wp_get_theme' ) ? wp_get_theme() : null;
		$themes = $theme ? $this->get_theme_data( $theme ) : array();

		// Check if the active theme is a child theme and retrieve the parent theme data.
		if ( function_exists( 'is_child_theme' ) && is_child_theme() && $theme ) {
			$themes[] = $this->get_theme_data( wp_get_theme( $theme->get( 'Template' ) ) );
		}

		unset( $theme );

		return $themes;
	}

	/**
	 * Retrieves specific data from the provided theme object.
	 *
	 * @param WP_Theme $theme Theme object.
	 *
	 * @return array Array containing the 'name', 'stylesheet', 'version', and 'template' of the theme.
	 * @since 2.2.70
	 */
	public function get_theme_data( $theme ) {
		if ( ! $theme ) {
			return array();
		}

		return array(
			'name'       => $theme->get( 'Name' ),
			'stylesheet' => $theme->get_stylesheet(),
			'version'    => $theme->get( 'Version' ),
			'template'   => $theme->get( 'Template' ),
		);
	}

	/**
	 * Check if the domain is allowlisted for telemetry data.
	 *
	 * @since BuddyBoss 2.2.70
	 *
	 * @return bool True if the domain is not allowlisted, false otherwise.
	 */
	public function bbapp_whitelist_domain_for_telemetry() {
		$server_name = ! empty( $_SERVER['SERVER_NAME'] ) ? wp_unslash( $_SERVER['SERVER_NAME'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		$whitelist_domain = array(
			'.test',
			'.dev',
			'staging.',
			'localhost',
			'.local',
			'.rapydapps.cloud',
			'ddev.site',
		);

		// Check for the test domain.
		if ( defined( 'WP_TESTS_DOMAIN' ) && WP_TESTS_DOMAIN === $server_name ) {
			return false;
		}

		// Check if the server name matches any whitelisted domain
		foreach ( $whitelist_domain as $domain ) {
			if ( false !== strpos( $server_name, $domain ) ) {
				return false; // Exclude allowlisted domains
			}
		}

		return true; // Allow telemetry data to be sent for non-allowlisted domains
	}

	/**
	 * Telemetry notice.
	 *
	 * @since BuddyBoss 2.2.70
	 */
	public function bbapp_telemetry_admin_notice() {

		// Check if the notice has already been dismissed.
		if ( true === (bool) bbapp_get_option( 'bbapp_telemetry_notice_dismissed' ) ) {
			return; // Do not display the notice if it's been dismissed.
		}
		// URL for the telemetry settings page.
		$settings_url  = admin_url( 'admin.php?page=bbapp-settings&setting=advanced' );
		$telemetry_url = 'https://www.buddyboss.com/usage-tracking/?utm_source=product&utm_medium=app-plugin&utm_campaign=telemetry';
		?>
		<div class="notice notice-info is-dismissible bbapp-telemetry-notice" data-nonce="<?php echo esc_attr( wp_create_nonce( 'bbapp-telemetry-notice-nonce' ) ); ?>">
			<div class="bbapp-telemetry-notice_logo"><i class="bb-icon-brand-buddyboss bb-icon-rf"></i></div>
			<div class="bbapp-telemetry-notice_content">
				<p class="bbapp-telemetry-notice_heading"><strong><?php esc_html_e( 'Help us improve BuddyBoss', 'buddyboss-app' ); ?></strong></p>
				<p>
					<?php
					// Message with link to telemetry settings.
					printf(
						wp_kses(
							/* translators: %1$s and %2$s are links. */
							__( 'We gather statistics about how our users use the product. We aggregate this information to help us improve the product and provide you with a better service. If you\'re happy with that you can dismiss this message, otherwise you can <a href="%1$s">adjust your telemetry settings</a>. To read more about what statistics we collect and why, click below.', 'buddyboss-app' ),
							array(
								'a' => array(
									'href' => array(),
								),
							)
						),
						esc_url( $settings_url )
					);
					?>
				</p>
				<p>
					<a href="<?php echo esc_url( $telemetry_url ); ?>" class="button button-primary" target="_blank" >
						<?php esc_html_e( 'About Telemetry', 'buddyboss-app' ); ?>
					</a>
				</p>
			</div>		
		</div>
		<?php
	}

	/**
	 * Store the dismissal of the notice.
	 *
	 * @since BuddyBoss 2.2.70
	 */
	public function bbapp_telemetry_notice_dismissed() {

		$bb_telemetry_nonce = sanitize_text_field( $_POST['nonce'] );

		// Nonce check.
		if ( empty( $bb_telemetry_nonce ) || ! wp_verify_nonce( $bb_telemetry_nonce, 'bbapp-telemetry-notice-nonce' ) ) {
			wp_send_json_error( array( 'error' => __( 'Sorry, something goes wrong please try again.', 'buddyboss-app' ) ) );
			unset( $bb_telemetry_nonce );
		}

		update_option( 'bbapp_telemetry_notice_dismissed', 1 );
		wp_send_json_success();
		unset( $bb_telemetry_nonce );
	}
}
