<?php
/**
 * Holds import/export functionality for Typography.
 *
 * @package BuddyBossApp\Admin\Tools\ImportExport
 */

namespace BuddyBossApp\Admin\Tools\ImportExport;

use BuddyBossApp\Admin\Tools\ImportExport;
use BuddyBossApp\Helpers\BBAPP_File;

/**
 * Class Typography.
 */
class Typography {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Module name.
	 *
	 * @var $module_name
	 */
	private $module_name;

	/**
	 * Get the instance of the class.
	 *
	 * @return Typography
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Typography constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Instance load method.
	 */
	public function load() {
		$this->module_name = 'typography';

		// Register The Module.
		ImportExport::instance()->module_register( $this->module_name, __( 'Typography', 'buddyboss-app' ) );

		// Register the hooks for import and export data.
		ImportExport::instance()->hook_register(
			$this->module_name,
			array( $this, 'import' ),
			array( $this, 'export' )
		);
	}

	/**
	 * Import data function.
	 *
	 * @param array $data Import data.
	 *
	 * @return array
	 */
	public function import( $data ) {

		$typography = \BuddyBossApp\Typography::instance();

		if ( isset( $data['data']['typography'] ) ) {
			$typography->set_options( $data['data']['typography'] );
		}
		if ( isset( $data['data']['font_scales'] ) ) {
			$typography->set_fonts_scales( $data['data']['font_scales'] );
		}

		// Copy the fonts.
		$font_dir = $typography::instance()->get_fonts_upload_dir();

		// Clear fonts directory first.
		BBAPP_File::delete_dir( $font_dir );
		BBAPP_File::create_dir( $font_dir );

		// Create font json.
		if ( isset( $data['data']['fonts_json'] ) ) {
			BBAPP_File::write_file( trailingslashit( $font_dir ) . 'fonts.json', $data['data']['fonts_json'] );
		}

		// Copy Fonts Files.
		foreach ( $data['assets'] as $file_key => $file_path ) {
			BBAPP_File::copy_file( $file_path, trailingslashit( $font_dir ) . $file_key );
		}

		return $data;
	}

	/**
	 * Export data function.
	 *
	 * @param array $data Export data.
	 *
	 * @return array
	 */
	public function export( $data ) {
		$typography                 = \BuddyBossApp\Typography::instance();
		$typography_option          = $typography->get_options();
		$data['data']['typography'] = $typography_option;

		// Fonts.
		$font_dir = $typography::instance()->get_fonts_upload_dir();

		$typography_scales          = $typography->get_fonts_scales();
		$data['data']['fonts_json'] = '';

		if ( file_exists( trailingslashit( $font_dir ) . 'fonts.json' ) ) {
			$info = BBAPP_File::read_file( trailingslashit( $font_dir ) . 'fonts.json' );
			$info = json_decode( $info, true );
			foreach ( $info as $index => $fonts ) {
				$unset_font = false;
				foreach ( $fonts as $font_key => $font ) {
					foreach ( $font['fonts_family'] as $key => $font_item ) {
						$file = trailingslashit( $font_dir ) . $font_item['file'];
						if ( file_exists( $file ) ) {
							$data['assets'][ $font_item['file'] ] = $file;
						} else {
							$unset_font = true; // if the one of the style file not found unset the font while export.
						}
					}
				}
				if ( $unset_font ) {
					unset( $info[ $index ] ); // unset the corrupted fonts.
				}
			}

			// Refresh list indexes.
			$_info = array();
			foreach ( $info as $index => $fonts ) {
				$_info[] = $fonts;
			}
			$info = $_info;
			unset( $_info );

			$data['data']['fonts_json'] = wp_json_encode( $info, JSON_PRETTY_PRINT );
		}

		if ( ! empty( $typography_scales ) ) {
			$data['data']['font_scales'] = $typography_scales;
		}

		return $data;
	}


}
