<?php
/**
 * Holds import/export functionality for Translations.
 *
 * @package BuddyBossApp\Admin\Tools\ImportExport
 */

namespace BuddyBossApp\Admin\Tools\ImportExport;

use BuddyBossApp\Admin\Tools\ImportExport;
use BuddyBossApp\AppLanguages;
use BuddyBossApp\Admin\AppLanguages as AdminAppLanguages;

/**
 * Class Translations.
 */
class Translations {

	/**
	 * Class instance.
	 *
	 * @var self $instance
	 */
	private static $instance;

	/**
	 * Module name.
	 *
	 * @var string $module_name
	 */
	private $module_name;

	/**
	 * Get the instance of the class.
	 *
	 * @return self
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Translations constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Instance load method.
	 *
	 * This method is called to register the module and its hooks.
	 *
	 * @since 2.4.10
	 */
	public function load() {
		$this->module_name = 'translations';

		// Register The Module.
		ImportExport::instance()->module_register( $this->module_name, __( 'Translations', 'buddyboss-app' ) );

		// Register the hooks for import and export data.
		ImportExport::instance()->hook_register(
			$this->module_name,
			array( $this, 'import' ),
			array( $this, 'export' )
		);
	}

	/**
	 * Import data function.
	 *
	 * @param array $data Import data.
	 *
	 * @since 2.4.10
	 * @return array
	 */
	public function import( $data ) {
		// Import active languages.
		if ( isset( $data['data']['active_languages'] ) ) {
			$active_languages = maybe_unserialize( $data['data']['active_languages'] );
			if ( is_array( $active_languages ) ) {
				update_option( 'bbapp_active_languages', $active_languages );
			}
		}

		// Import string translations from newer format.
		if ( isset( $data['data']['string_translations'] ) ) {
			$string_translations = maybe_unserialize( $data['data']['string_translations'] );
			if ( is_array( $string_translations ) ) {
				$this->import_string_translations( $string_translations );
			}
		}

		return $data;
	}

	/**
	 * Export data function.
	 *
	 * @param array $data Export data.
	 *
	 * @since 2.4.10
	 * @return array
	 */
	public function export( $data ) {
		// Export active languages.
		$active_languages                 = get_option( 'bbapp_active_languages', array() );
		$data['data']['active_languages'] = $active_languages;

		// Export string translations from the database.
		$string_translations                 = $this->export_string_translations();
		$data['data']['string_translations'] = $string_translations;

		return $data;
	}

	/**
	 * Import string translations into the database.
	 *
	 * @param array $translations Array of translation records.
	 *
	 * @since 2.4.10
	 *
	 * @return int Number of imported translations.
	 */
	private function import_string_translations( $translations ) {
		global $wpdb;
		$imported_count = 0;

		// Get the AppLanguages instance.
		$app_languages = \BuddyBossApp\AppLanguages::instance();

		// Begin transaction.
		$wpdb->query( 'START TRANSACTION' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery

		try {
			foreach ( $translations as $translation ) {
				// Ensure required fields are set.
				if ( empty( $translation['string_handle'] ) || empty( $translation['language_code'] ) ) {
					continue;
				}

				// Check if translation exists.
				$existing = $app_languages->get_translation(
					array(
						'language_code'   => $translation['language_code'],
						'string_handle'   => $translation['string_handle'],
						'include_deleted' => true,
					)
				);

				if ( $existing ) {
					// Update existing translation.
					$update_data = array(
						'updated_string' => $translation['updated_string'],
						'default_string' => $translation['default_string'],
					);

					// If is_build_string is set in the import data.
					if ( isset( $translation['is_build_string'] ) ) {
						$update_data['is_build_string'] = $translation['is_build_string'];
					}

					// If is_custom_string is set in the import data.
					if ( isset( $translation['is_custom_string'] ) ) {
						$update_data['is_custom_string'] = $translation['is_custom_string'];
					}

					// Handle deleted status.
					if ( isset( $translation['deleted_at'] ) ) {
						$update_data['deleted_at'] = $translation['deleted_at'] ? true : null;
					}

					$result = $app_languages->update_translation( $existing->id, $update_data );

					if ( $result ) {
						++$imported_count;
					}
				} else {
					// Insert new translation.
					$insert_data = array(
						'language_code'  => $translation['language_code'],
						'string_handle'  => $translation['string_handle'],
						'default_string' => $translation['default_string'],
						'updated_string' => $translation['updated_string'],
					);

					// If is_build_string is set in the import data.
					if ( isset( $translation['is_build_string'] ) ) {
						$insert_data['is_build_string'] = $translation['is_build_string'];
					}

					// If is_custom_string is set in the import data.
					if ( isset( $translation['is_custom_string'] ) ) {
						$insert_data['is_custom_string'] = $translation['is_custom_string'];
					}

					$insert_id = $app_languages->insert_translation( $insert_data );

					if ( $insert_id ) {
						++$imported_count;
					}
				}
			}

			// Commit transaction.
			$wpdb->query( 'COMMIT' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery

			return $imported_count;
		} catch ( \Exception $e ) {
			// Rollback on error.
			$wpdb->query( 'ROLLBACK' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery

			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				// Log the error message.
				error_log( 'BuddyBoss App: Error importing string translations: ' . $e->getMessage() ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			}

			return 0;
		}
	}

	/**
	 * Export string translations from the database.
	 *
	 * @since 2.4.10
	 * @return array Array of translation records
	 */
	private function export_string_translations() {
		// Get the AppLanguages instance.
		$app_languages = \BuddyBossApp\AppLanguages::instance();

		// Get all translations including deleted ones.
		$translations = $app_languages->get_translations(
			array(
				'include_deleted' => true,
				'only_updated'    => true,
				'include_custom'  => true,
				'limit'           => 0, // No limit.
			)
		);

		// Convert objects to arrays for better serialization.
		$export_data = array();
		foreach ( $translations as $translation ) {
			$export_data[] = (array) $translation;
		}

		return $export_data;
	}
}
