<?php
/**
 * Holds import/export functionality for TabBar.
 *
 * @package BuddyBossApp\Admin\Tools\ImportExport
 */

namespace BuddyBossApp\Admin\Tools\ImportExport;

use BuddyBossApp\Admin\Tools\ImportExport;
use BuddyBossApp\ManageApp;

/**
 * Class TabBar.
 */
class TabBar {

	/**
	 * Class instance.
	 *
	 * @var TabBar $instance
	 */
	private static $instance;

	/**
	 * Module name.
	 *
	 * @var string $module_name
	 */
	private $module_name;

	/**
	 * Get the instance of the class.
	 *
	 * @return TabBar Instance of the class.
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * TabBar constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Instance load method.
	 */
	public function load() {
		$this->module_name = 'tabbar';

		// Register The Module.
		ImportExport::instance()->module_register( $this->module_name, __( 'Tab Bar', 'buddyboss-app' ) );

		// Register the hooks for import and export data.
		ImportExport::instance()->hook_register(
			$this->module_name,
			array( $this, 'import' ),
			array( $this, 'export' )
		);
	}

	/**
	 * Import data function.
	 *
	 * @param array $data Import data.
	 *
	 * @return array
	 */
	public function import( $data ) {
		global $wpdb;

		// Import app menu settings if available in the data.
		if ( isset( $data['data']['app_menus'] ) ) {
			$app_settings                                = ManageApp::instance()->get_app_settings();
			$app_settings['app_menu.tab_bar_visibility'] = ! empty( $data['data']['app_menus']['tab_bar_visibility'] ) ? $data['data']['app_menus']['tab_bar_visibility'] : 'show_on_tab_bar_menu';
			$app_settings['app_menu.appmenu_labels']     = ! empty( $data['data']['app_menus']['appmenu_labels'] ) ? $data['data']['app_menus']['appmenu_labels'] : '';
			$app_settings['app_menu.tab_bar_icon_style'] = ! empty( $data['data']['app_menus']['tab_bar_icon_style'] ) ? $data['data']['app_menus']['tab_bar_icon_style'] : 'outlined';
			$app_settings['app_menu.more_icon_style']    = ! empty( $data['data']['app_menus']['more_icon_style'] ) ? $data['data']['app_menus']['more_icon_style'] : 'filled';

			ManageApp::instance()->update_app_settings( $app_settings );
		}

		// Import menu data if available.
		if ( isset( $data['data']['menus'] ) ) {
			$menus_data = $data['data']['menus'];
			$now        = current_time( 'mysql' );
			$site_id    = get_current_blog_id();

			// Store mapping between original IDs and new IDs.
			$menu_id_mapping         = array();
			$item_id_mapping         = array();
			$menu_name_to_id_mapping = array();

			// First pass: create all menus and store their IDs.
			if ( isset( $menus_data['bbapp_menus']['menus'] ) && is_array( $menus_data['bbapp_menus']['menus'] ) ) {
				foreach ( $menus_data['bbapp_menus']['menus'] as $menu ) {
					// Extract menu data from the array.
					$menu_type       = isset( $menu['menu_type'] ) ? $menu['menu_type'] : '';
					$menu_name       = isset( $menu['menu_name'] ) ? $menu['menu_name'] : '';
					$login_state     = isset( $menu['login_state'] ) ? $menu['login_state'] : 1;
					$access_groups   = isset( $menu['access_groups'] ) ? $menu['access_groups'] : '';
					$language_code   = isset( $menu['language_code'] ) ? $menu['language_code'] : 'en';
					$priority        = isset( $menu['priority'] ) ? $menu['priority'] : 0;
					$data_serialized = isset( $menu['data'] ) ? $menu['data'] : '';
					$original_id     = isset( $menu['id'] ) ? $menu['id'] : '';

					// Skip if essential data is missing or invalid.
					if ( empty( $menu_type ) || empty( $menu_name ) ) {
						continue;
					}

					// Prepare menu data for database insertion.
					$menu_db_data = array(
						'site_id'       => $site_id,
						'menu_type'     => $menu_type,
						'menu_name'     => $menu_name,
						'login_state'   => $login_state,
						'access_groups' => $access_groups,
						'language_code' => $language_code,
						'priority'      => $priority,
						'data'          => $data_serialized,
						'created_at'    => $now,
						'updated_at'    => $now,
					);

					// Insert the menu data into the database.
					$wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
						$wpdb->prefix . 'bbapp_menus',
						$menu_db_data
					);

					$new_menu_id = $wpdb->insert_id;

					// Store mapping of original ID to new ID if available.
					if ( $original_id ) {
						$menu_id_mapping[ $original_id ] = $new_menu_id;
					}

					// Also store menu name to ID mapping for later use.
					$menu_name_to_id_mapping[ $menu_name ] = $new_menu_id;

					// Extract related menu IDs from data if available and store them.
					$menu_data = maybe_unserialize( $data_serialized );
					if ( is_array( $menu_data ) ) {
						if ( isset( $menu_data['related_headerbar_id'] ) && 'tabbar' === $menu_type ) {
							$menu_id_mapping[ 'tabbar_headerbar_' . $new_menu_id ] = $menu_data['related_headerbar_id'];
						}
						if ( isset( $menu_data['related_tabbar_id'] ) && 'headerbar' === $menu_type ) {
							$menu_id_mapping[ 'headerbar_tabbar_' . $new_menu_id ] = $menu_data['related_tabbar_id'];
						}
					}
				}
			}

			// Update menu relationships if available.
			if ( ! empty( $menu_id_mapping ) ) {
				foreach ( $menu_id_mapping as $original_id => $related_id ) {
					if ( strpos( $original_id, 'tabbar_headerbar_' ) === 0 ) {
						$tabbar_id    = str_replace( 'tabbar_headerbar_', '', $original_id );
						$headerbar_id = isset( $menu_id_mapping[ $related_id ] ) ? $menu_id_mapping[ $related_id ] : $related_id;

						// Update the tabbar menu data to include the headerbar ID.
						$tabbar_data = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
							$wpdb->prepare(
								"SELECT data FROM {$wpdb->prefix}bbapp_menus WHERE id = %d",
								$tabbar_id
							)
						);

						if ( $tabbar_data ) {
							$tabbar_data = maybe_unserialize( $tabbar_data );
							if ( is_array( $tabbar_data ) ) {
								$tabbar_data['related_headerbar_id'] = $headerbar_id;
								$wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
									$wpdb->prefix . 'bbapp_menus',
									array( 'data' => maybe_serialize( $tabbar_data ) ),
									array( 'id' => $tabbar_id )
								);
							}
						}
					} elseif ( strpos( $original_id, 'headerbar_tabbar_' ) === 0 ) {
						$headerbar_id = str_replace( 'headerbar_tabbar_', '', $original_id );
						$tabbar_id    = isset( $menu_id_mapping[ $related_id ] ) ? $menu_id_mapping[ $related_id ] : $related_id;

						// Update the headerbar menu data to include the tabbar ID.
						$headerbar_data = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
							$wpdb->prepare(
								"SELECT data FROM {$wpdb->prefix}bbapp_menus WHERE id = %d",
								$headerbar_id
							)
						);

						if ( $headerbar_data ) {
							$headerbar_data = maybe_unserialize( $headerbar_data );
							if ( is_array( $headerbar_data ) ) {
								$headerbar_data['related_tabbar_id'] = $tabbar_id;
								$wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
									$wpdb->prefix . 'bbapp_menus',
									array( 'data' => maybe_serialize( $headerbar_data ) ),
									array( 'id' => $headerbar_id )
								);
							}
						}
					}
				}
			}

			// Second pass: create all menu items and store their IDs.
			if ( isset( $menus_data['menu_items'] ) && is_array( $menus_data['menu_items'] ) ) {
				foreach ( $menus_data['menu_items'] as $menu_id => $menu_items ) {
					// Find the new menu ID if this is using the old ID or a related ID.
					$new_menu_id = isset( $menu_id_mapping[ $menu_id ] ) ? $menu_id_mapping[ $menu_id ] : $menu_id;

					// Check if menu exists
					$menu_exists = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
						$wpdb->prepare(
							"SELECT COUNT(*) FROM {$wpdb->prefix}bbapp_menus WHERE id = %d",
							$new_menu_id
						)
					);

					if ( ! $menu_exists ) {
						continue; // Skip if menu not found.
					}

					// First create all parent items and store their IDs.
					foreach ( $menu_items as $item ) {
						// Skip child items - they'll be processed later.
						if ( isset( $item['parent_id'] ) && intval( $item['parent_id'] ) > 0 ) {
							continue;
						}

						// Insert the menu item data into the database.
						$item_data = array(
							'menu_id'        => $new_menu_id,
							'label'          => isset( $item['label'] ) ? $item['label'] : '',
							'item_id'        => isset( $item['item_id'] ) ? $item['item_id'] : 0,
							'item_slug'      => isset( $item['item_slug'] ) ? $item['item_slug'] : '',
							'item_type'      => isset( $item['item_type'] ) ? $item['item_type'] : '',
							'item_icon_data' => isset( $item['item_icon_data'] ) ? $item['item_icon_data'] : null,
							'item_link'      => isset( $item['item_link'] ) ? $item['item_link'] : '',
							'parent_id'      => 0,
							'menu_order'     => isset( $item['menu_order'] ) ? $item['menu_order'] : 0,
							'created_at'     => $now,
							'updated_at'     => $now,
						);

						$wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
							$wpdb->prefix . 'bbapp_menu_items',
							$item_data
						);

						$new_item_id = $wpdb->insert_id;
						$original_id = isset( $item['id'] ) ? $item['id'] : '';

						if ( $original_id ) {
							$item_id_mapping[ $original_id ] = $new_item_id;
						}
					}

					// Process child items and insert them with the new parent IDs.
					foreach ( $menu_items as $item ) {
						if ( isset( $item['parent_id'] ) && intval( $item['parent_id'] ) > 0 ) {
							$parent_id = intval( $item['parent_id'] );

							// Find new parent ID if mapped or use the original ID.
							$new_parent_id = isset( $item_id_mapping[ $parent_id ] ) ? $item_id_mapping[ $parent_id ] : $parent_id;

							// Check if parent exists in the new menu.
							$parent_exists = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
								$wpdb->prepare(
									"SELECT COUNT(*) FROM {$wpdb->prefix}bbapp_menu_items WHERE id = %d AND menu_id = %d",
									$new_parent_id,
									$new_menu_id
								)
							);

							if ( ! $parent_exists ) {
								continue; // Skip if parent not found in the new menu.
							}

							// Insert child item data into the database.
							$item_data = array(
								'menu_id'        => $new_menu_id,
								'label'          => isset( $item['label'] ) ? $item['label'] : '',
								'item_id'        => isset( $item['item_id'] ) ? $item['item_id'] : 0,
								'item_slug'      => isset( $item['item_slug'] ) ? $item['item_slug'] : '',
								'item_type'      => isset( $item['item_type'] ) ? $item['item_type'] : '',
								'item_icon_data' => isset( $item['item_icon_data'] ) ? $item['item_icon_data'] : null,
								'item_link'      => isset( $item['item_link'] ) ? $item['item_link'] : '',
								'parent_id'      => $new_parent_id,
								'menu_order'     => isset( $item['menu_order'] ) ? $item['menu_order'] : 0,
								'created_at'     => $now,
								'updated_at'     => $now,
							);

							$wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
								$wpdb->prefix . 'bbapp_menu_items',
								$item_data
							);
						}
					}
				}
			}
		}

		return $data;
	}

	/**
	 * Export data function.
	 *
	 * @param array $data Export data.
	 *
	 * @return array
	 */
	public function export( $data ) {
		global $wpdb;

		// Export app settings if available.
		$bbapp_settings = ManageApp::instance()->get_app_settings();

		if ( ! empty( $bbapp_settings ) ) {
			$data['data']['app_menus']['tab_bar_visibility'] = ! empty( $bbapp_settings['app_menu.tab_bar_visibility'] ) ? $bbapp_settings['app_menu.tab_bar_visibility'] : 'show_on_tab_bar_menu';
			$data['data']['app_menus']['appmenu_labels']     = ! empty( $bbapp_settings['app_menu.appmenu_labels'] ) ? $bbapp_settings['app_menu.appmenu_labels'] : '';
			$data['data']['app_menus']['tab_bar_icon_style'] = ! empty( $bbapp_settings['app_menu.tab_bar_icon_style'] ) ? $bbapp_settings['app_menu.tab_bar_icon_style'] : 'outlined';
			$data['data']['app_menus']['more_icon_style']    = ! empty( $bbapp_settings['app_menu.more_icon_style'] ) ? $data['data']['app_menus']['more_icon_style'] : 'filled';
		}

		// Get all menus directly from database to avoid any caching issues.
		$menus = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
			"SELECT * FROM {$wpdb->prefix}bbapp_menus ORDER BY menu_type, priority",
			ARRAY_A
		);

		$bbapp_menus = array();
		// Process menus and organize them by type.
		if ( ! empty( $menus ) ) {
			$bbapp_menus['menus'] = $menus;
		}

		// Get menu items for each menu and organize them.
		$menu_items_export = array();

		foreach ( $menus as $menu ) {
			$menu_id = $menu['id'];

			// Get all items for this menu from the database.
			$items = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
				$wpdb->prepare(
					"SELECT * FROM {$wpdb->prefix}bbapp_menu_items WHERE menu_id = %d ORDER BY parent_id ASC, menu_order ASC",
					$menu_id
				),
				ARRAY_A
			);

			if ( ! empty( $items ) ) {
				$menu_items_export[ $menu_id ] = $items;
			}
		}

		// Build the final menu export data structure.
		$menu_export = array(
			'bbapp_menus' => $bbapp_menus,
			'menu_items'  => $menu_items_export,
		);

		$data['data']['menus'] = $menu_export;

		return $data;
	}
}
