<?php
/**
 * Import Export main file.
 *
 * @package BuddyBossApp
 */

namespace BuddyBossApp\Admin\Tools;

use BuddyBossApp\Admin\Tools\ImportExport\AndroidKeyStoreInformation;
use BuddyBossApp\Admin\Tools\ImportExport\Colors;
use BuddyBossApp\Admin\Tools\ImportExport\CustomIcons;
use BuddyBossApp\Admin\Tools\ImportExport\Images;
use BuddyBossApp\Admin\Tools\ImportExport\Integrations;
use BuddyBossApp\Admin\Tools\ImportExport\iOSProvisioningProfiles;
use BuddyBossApp\Admin\Tools\ImportExport\iOSSigningCertificates;
use BuddyBossApp\Admin\Tools\ImportExport\Settings;
use BuddyBossApp\Admin\Tools\ImportExport\TabBar;
use BuddyBossApp\Admin\Tools\ImportExport\Translations;
use BuddyBossApp\Admin\Tools\ImportExport\Typography;
use BuddyBossApp\Helpers\BBAPP_File;
use BuddyBossApp\Permissions;

/**
 * Class ImportExport.
 */
class ImportExport {

	/**
	 * Class instance.
	 *
	 * @var self $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return ImportExport
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Class constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Instance load method.
	 */
	public function load() {
		$this->admin_post_hooks( 'bbapp_export', array( $this, 'bbapp_export' ) ); // for export use admin post.
		$this->ajax_hooks( 'bbapp_import', array( $this, 'bbapp_import' ) );
		$this->ajax_hooks( 'bbapp_import_file_upload', array( $this, 'bbapp_import_file_upload' ) );
	}

	/**
	 * Helper function to load all export import module classes.
	 */
	public function load_all_import_export_modules() {
		Images::instance(); // Images Data Support.
		CustomIcons::instance(); // Custom Icons Data Support.
		TabBar::instance(); // Tabbar menus Data Support.
		Colors::instance(); // Colors Data Support.
		Typography::instance(); // Typography Data Support.
		Translations::instance(); // Translations Data Support.
		// Build::instance(); // Build Data Support.
		Integrations::instance(); // Integration Data Support.
		Settings::instance(); // Settings Data Support.
		iOSSigningCertificates::instance(); // iOS Signing Certificates Data Support.
		iOSProvisioningProfiles::instance(); // iOS Provisioning Profiles Data Support.
		AndroidKeyStoreInformation::instance(); // Android KeyStore Information Data Support.
	}

	/**
	 * Helper function tio register module hook.
	 *
	 * @param string $module      Module name.
	 * @param array  $import_func Class method.
	 * @param array  $export_func Class method.
	 *
	 * @return true|void
	 */
	public function hook_register( $module, $import_func, $export_func ) {
		add_filter( "bbapp_import_export_{$module}_export", $export_func, 10, 1 );
		add_filter( "bbapp_import_export_{$module}_import", $import_func, 10, 1 );

		return true;
	}

	/**
	 * Helper function to register module name.
	 *
	 * @param string $name Name.
	 * @param string $label Label.
	 */
	public function module_register( $name, $label = false ) {

		if ( ! $label ) {
			$label = ucfirst( $name );
		}

		add_filter(
			'bbapp_import_export_modules',
			function ( $modules ) use ( $name, $label ) {
				$modules[ $name ] = $label;

				return $modules;
			}
		);
	}

	/**
	 * Returns the Import & Export Modules.
	 *
	 * @return mixed|void
	 */
	public function get_modules() {
		$export_modules = apply_filters( 'bbapp_import_export_modules', array() );
		$modules        = array();

		foreach ( $export_modules as $module_name => $module_label ) {
			$modules[] = $module_name;
		}

		return $modules;
	}

	/**
	 * Return the module label.
	 *
	 * @param string $module_name Module name.
	 *
	 * @return bool
	 */
	public function get_module_label( $module_name ) {
		$export_modules = apply_filters( 'bbapp_import_export_modules', array() );

		if ( isset( $export_modules[ $module_name ] ) ) {
			return $export_modules[ $module_name ];
		}

		return ucfirst( $module_name );
	}

	/**
	 * Renders the Import Export Screen.
	 */
	public function render_screen() {
		include bbapp()->plugin_dir . 'views/tools/view-import-export.php';
	}

	/**
	 * Ajax Hooks Helper.
	 *
	 * @param string    $action action name of ajax hook.
	 * @param int|array $method method to be call.
	 */
	public function ajax_hooks( $action, $method ) {
		add_action( "wp_ajax_nopriv_$action", $method );
		add_action( "wp_ajax_$action", $method );
	}

	/**
	 * Admin post hooks.
	 *
	 * @param string $action Action.
	 * @param array  $method Class method.
	 */
	public function admin_post_hooks( $action, $method ) {
		add_action( "admin_post_$action", $method );
	}

	/**
	 * Open zip file and read module.
	 */
	public function bbapp_import_file_upload() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			bbapp_ajax_error_response( 'invalid_nonce', __( 'You don\'t have permission to import.', 'buddyboss-app' ) );
		}
		if ( ! class_exists( '\ZipArchive' ) ) {
			bbapp_ajax_success_response( 'php_module', __( 'ZipArchive PHP module is not installed. ZipArchive is required in order to perform the export.', 'buddyboss-app' ) );
		}

		if ( ! isset( $_FILES['bbapp_import_file'] ) || empty( $_FILES['bbapp_import_file']['tmp_name'] ) ) {
			bbapp_ajax_error_response( 'invalid_nonce', __( 'No file is selected. Please try again.', 'buddyboss-app' ) );
		}

		if ( ! empty( $_FILES['bbapp_import_file']['name'] ) ) {
			$extension = pathinfo( bbapp_input_clean( wp_unslash( $_FILES['bbapp_import_file']['name'] ) ), PATHINFO_EXTENSION ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( 'zip' !== $extension ) {
				bbapp_ajax_error_response( 'invalid_nonce', __( 'Only zip file is allowed to upload.', 'buddyboss-app' ) );
			}
		}

		$import_dir = $this->get_import_folder( true );

		$zip        = new \ZipArchive();
		$zip_result = $zip->open( bbapp_input_clean( wp_unslash( $_FILES['bbapp_import_file']['tmp_name'] ) ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		ob_start();
		if ( true === $zip_result ) {
			// Load all export import module into runtime.
			$this->load_all_import_export_modules();

			$export_modules = $this->get_modules();

			$available_modules = array();

			$zip->extractTo( $import_dir );
			$zip->close();

			// Scan all available modules.
			$objects = scandir( $import_dir );
			foreach ( $objects as $object ) {
				if ( '.' !== $object && '..' !== $object ) {
					if ( in_array( $object, $export_modules, true ) ) {
						$available_modules[] = $object;
					}
				}
			}

			if ( ! empty( $available_modules ) ) {
				foreach ( $available_modules as $module ) {
					?>
					<label>
						<input checked="checked" type="checkbox" name="modules[]" value="<?php echo esc_attr( $module ); ?>"/>
						<?php echo wp_kses_post( $this->get_module_label( $module ) ); ?>
					</label> <br/>
				<?php } ?>
				<div class="bbapp-import-actions">
					<a href="#" class="button bbapp-import-btn-final">
						<span class="dashicons dashicons-arrow-right-alt"></span> <?php esc_attr_e( "Let's Import", 'buddyboss-app' ); ?>
						<div class="bbapp_loading" style="display:none;vertical-align: middle;"></div>
					</a>
					<a href="#" class="bbapp-import-back"><?php echo esc_attr__( 'Cancel', 'buddyboss-app' ); ?></a>
				</div>
				<?php
			}
			$html = ob_get_clean();
			wp_send_json_success( array( 'html' => $html ) );
		}
		bbapp_ajax_error_response( 'invalid_zip', __( 'Unable to open zip.', 'buddyboss-app' ) );
	}

	/**
	 * Import BuddyBossApp Settings Into WordPress.
	 */
	public function bbapp_import() {

		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			bbapp_ajax_error_response( 'invalid_nonce', __( "You don't have permission to import.", 'buddyboss-app' ) );
		}

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( bbapp_input_clean( wp_unslash( $_POST['nonce'] ) ), 'bbapp_import' ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			bbapp_ajax_error_response( 'invalid_nonce', __( 'Unable to verify security nonce, Please refresh the page and try again.', 'buddyboss-app' ) );
		}

		if ( ! class_exists( '\ZipArchive' ) ) {
			bbapp_ajax_success_response( 'php_module', __( 'ZipArchive PHP module is not installed. ZipArchive is required in order to perform the export.', 'buddyboss-app' ) );
		}

		if ( ! isset( $_FILES['bbapp_import'] ) || ! isset( $_FILES['bbapp_import']['tmp_name'] ) || empty( $_FILES['bbapp_import']['tmp_name'] ) ) {
			bbapp_ajax_error_response( 'invalid_nonce', __( 'No file is selected. Please try again.', 'buddyboss-app' ) );
		}

		if ( ! empty( $_FILES['bbapp_import']['name'] ) ) {
			$extension = pathinfo( bbapp_input_clean( wp_unslash( $_FILES['bbapp_import']['name'] ) ), PATHINFO_EXTENSION ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( 'zip' !== $extension ) {
				bbapp_ajax_error_response( 'invalid_nonce', __( 'Only zip file is allowed to upload.', 'buddyboss-app' ) );
			}
		}

		$import_dir = $this->get_import_folder( true );

		$zip = new \ZipArchive();
		$res = $zip->open( bbapp_input_clean( wp_unslash( $_FILES['bbapp_import']['tmp_name'] ) ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( true === $res ) {

			// Load all export import module into runtime.
			$this->load_all_import_export_modules();

			$export_modules = $this->get_modules();

			$available_modules = array();

			$zip->extractTo( $import_dir );
			$zip->close();

			// Scan all available modules.
			$objects = scandir( $import_dir );
			foreach ( $objects as $object ) {
				if ( '.' !== $object && '..' !== $object ) {
					if ( in_array( $object, $export_modules, true ) ) {
						$available_modules[] = $object;
					}
				}
			}

			/**
			 * Only keep module which user has selected.
			 */
			if ( ! empty( $_POST['modules'] ) ) {
				foreach ( $available_modules as $k => $module ) {
					if ( ! in_array( $module, $_POST['modules'], true ) ) {
						unset( $available_modules[ $k ] );
					}
				}
			}

			/**
			 * Check if there is anything to install or not.
			 */
			if ( empty( $available_modules ) ) {
				bbapp_ajax_error_response( 'invalid_nonce', __( 'Sorry, there is no data to import in the Zip.', 'buddyboss-app' ) );
			}

			// Import the data from all available modules.
			foreach ( $available_modules as $module_name ) {

				$module_dir  = trailingslashit( $import_dir ) . $module_name;
				$module_data = array(
					'data'   => array(),
					'assets' => array(),
				);

				/**
				 * Module Data.
				 */

				$data = glob( $module_dir . '/*.txt' );
				foreach ( $data as $data_file ) {
					$data_key                         = str_replace( '.txt', '', basename( $data_file ) );
					$module_data['data'][ $data_key ] = BBAPP_File::read_file( $data_file );
					if ( is_serialized( $module_data['data'][ $data_key ] ) ) {
						$module_data['data'][ $data_key ] = maybe_unserialize( $module_data['data'][ $data_key ] );
					}
				}

				/**
				 * Module Files.
				 */
				if ( file_exists( $module_dir . '/files.json' ) ) {

					$module_files = BBAPP_File::read_file( $module_dir . '/files.json' );
					$module_files = json_decode( $module_files, true );

					if ( ! is_array( $module_files ) ) {
						$module_files = array();
					}

					foreach ( $module_files as $file_key => $file_name ) {
						$module_file = $module_dir . "/{$file_name}";
						if ( file_exists( $module_file ) ) {
							$module_data['assets'][ $file_key ] = $module_file;
						}
					}
				}

				$status = apply_filters( "bbapp_import_export_{$module_name}_import", $module_data );

			}

			bbapp_ajax_success_response( 'success', __( 'Data has been successfully imported.', 'buddyboss-app' ) );

		} else {
			bbapp_ajax_error_response( 'invalid_nonce', __( 'Error while extracting Zip file. Please try again.', 'buddyboss-app' ) );
		}
	}

	/**
	 * Export BuddyBossApp Settings Function.
	 */
	public function bbapp_export() {

		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			wp_die( esc_html__( "You don't have permission to download export file.", 'buddyboss-app' ) );
		}

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( bbapp_input_clean( wp_unslash( $_POST['nonce'] ) ), 'bbapp_export' ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			wp_die( esc_html__( 'The download link to export settings has expired.', 'buddyboss-app' ) );
		}

		if ( ! class_exists( '\ZipArchive' ) ) {
			wp_die( esc_html__( 'ZipArchive PHP module is not installed. ZipArchive is required in order to perform the export.', 'buddyboss-app' ) );
		}
		$selected_export_modules = array();
		if ( isset( $_POST['export_modules'] ) && ! empty( $_POST['export_modules'] ) ) {
			$selected_export_modules = bbapp_input_clean( wp_unslash( $_POST['export_modules'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}

		if ( empty( $selected_export_modules ) ) {
			wp_die( esc_html__( 'No selected modules found.', 'buddyboss-app' ) );
		}

		// Load all export import module into runtime.
		$this->load_all_import_export_modules();

		$export_modules = $this->get_modules();

		// Selected module export only.
		if ( ! empty( $selected_export_modules ) ) {
			$export_modules = array_intersect( $export_modules, $selected_export_modules );
		}

		$modules_data = array();

		foreach ( $export_modules as $module ) {

			$data = apply_filters(
				"bbapp_import_export_{$module}_export",
				array(
					'data'   => array(),
					'assets' => array(),
				)
			);

			if ( ! is_array( $data['data'] ) || ! is_array( $data['assets'] ) ) {
				continue;
			}

			$modules_data[ $module ] = $data;
		}

		$export_dir = $this->get_export_folder( true );

		foreach ( $modules_data as $module_name => $module_data ) {

			// Create Module Dir.
			$module_dir = $export_dir . '/' . $module_name;
			BBAPP_File::create_dir( $module_dir );
			foreach ( $module_data['data'] as $data_key => $data_value ) {
				if ( ! is_string( $data_value ) ) {
					$data_value = maybe_serialize( $data_value );
				}
				BBAPP_File::write_file( "{$module_dir}/{$data_key}.txt", $data_value );
			}

			$files = array();
			$i     = 0;
			foreach ( $module_data['assets'] as $file_key => $file_path ) {
				$filename = $i . '-' . basename( $file_path );
				BBAPP_File::copy_file( $file_path, "{$module_dir}/{$filename}" );
				$files[ $file_key ] = $filename;
				++$i;
			}

			BBAPP_File::write_file( "{$module_dir}/files.json", wp_json_encode( $files, JSON_PRETTY_PRINT ) );

		}

		/**
		 * Zip the folder.
		 */
		// Get real path for our folder.
		$root_path       = realpath( $export_dir );
		$export_zip_path = trailingslashit( dirname( $export_dir ) ) . 'buddyboss-app-export.zip';
		$export_zip_url  = trailingslashit( dirname( $this->get_export_folder_url() ) ) . 'buddyboss-app-export.zip';

		// Initialize archive object.
		$zip = new \ZipArchive();
		$zip->open( $export_zip_path, \ZipArchive::CREATE | \ZipArchive::OVERWRITE );

		// Create recursive directory iterator.
		$files = new \RecursiveIteratorIterator(
			new \RecursiveDirectoryIterator( $root_path ),
			\RecursiveIteratorIterator::LEAVES_ONLY
		);

		foreach ( $files as $name => $file ) {
			// Skip directories (they would be added automatically).
			if ( ! $file->isDir() ) {
				// Get real and relative path for current file.
				$file_path     = $file->getRealPath();
				$relative_path = substr( $file_path, strlen( $root_path ) + 1 );

				// Add current file to archive.
				$zip->addFile( $file_path, $relative_path );
			}
		}

		// Zip archive will be created only after closing object.
		$zip->close();

		BBAPP_File::delete_dir( $export_dir );

		if ( ! file_exists( $export_zip_path ) ) {
			wp_die( 'Unknown error while downloading export file, please try again.', 'buddyboss-app' );
		}

		wp_safe_redirect( $export_zip_url, 302, 'BuddyBossApp>ImportExport' );
		exit;
	}

	/**
	 * Return the import folder.
	 *
	 * @param bool $create_empty Create empty flag.
	 *
	 * @return string
	 */
	public function get_import_folder( $create_empty = false ) {
		$upload_dir = wp_upload_dir();
		$dir        = $upload_dir['basedir'] . '/bbapp-import';
		if ( ! file_exists( $dir ) ) {
			BBAPP_File::create_dir( $dir );
		} elseif ( $create_empty ) {
			BBAPP_File::delete_dir( $dir );
			BBAPP_File::create_dir( $dir );
		}

		return $dir;
	}

	/**
	 * Return the export folder.
	 *
	 * @param bool $create_empty Create empty flag.
	 *
	 * @return string
	 */
	public function get_export_folder( $create_empty = false ) {
		$upload_dir = wp_upload_dir();
		$dir        = $upload_dir['basedir'] . '/bbapp-export';
		if ( ! file_exists( $dir ) ) {
			BBAPP_File::create_dir( $dir );
		} elseif ( $create_empty ) {
				BBAPP_File::delete_dir( $dir );
				BBAPP_File::create_dir( $dir );
		}

		return $dir;
	}

	/**
	 * Return the export folder url.
	 *
	 * @return string
	 */
	public function get_export_folder_url() {
		$upload_dir = wp_upload_dir();
		$url        = $upload_dir['baseurl'] . '/bbapp-export';

		return $url;
	}

	/**
	 * Buddyboss app plugin exportable keys list.
	 *
	 * @param string $module Module.
	 *
	 * @return array|mixed|string[]
	 */
	public function bbapp_db_key_lists( $module = '' ) {
		$db_key_list                             = array();
		$db_key_list['settings']                 = array(
			'browser_auth.enabled',
			'browser_auth.username',
			'browser_auth.password',
			'browser_auth.message',
			'logger.enabled',
			'logger.info_log.enabled',
			'logger.api_log.enabled',
			'logger.iap_log.enabled',
			'app_smartbanner.ios.enabled',
			'app_smartbanner.android.enabled',
			'app_smartbanner.logged_in_only',
			'app_smartbanner.ios',
			'app_smartbanner.android',
			'cdn.enabled',
			'cdn.url',
			'private_app.enabled',
			'app_auth.email_activation_body',
			'app_auth.enable_signup',
			'app_subscription.enabled',
			'app_menu.tabbar_more_menu_icon',
			'offline_support_course_downloading',
			'config.support_email',
			'config.bugs_email',
			'app_rating.ios.enabled',
			'app_rating.android.enabled',
			'app_menu.appmenu_labels',
			'app_menu.tab_bar_visibility',
			'support_email.enabled',
			'support_email.email',
			'support_email.subject',
			'delete_account.admin_email',
			'delete_account.subject',
			'bugs_email.enabled',
			'bugs_email.email',
			'bugs_email.subject',
		);
		$db_key_list['configurations']           = array(
			'publish.ios.appname',
			'publish.ios.namespace',
			'publish.ios.permissions.play_music',
			'publish.android.appname',
			'publish.android.namespace',
			'publish.youtube_api_key',
			'publish.android.dev.keystore',
			'publish.android.dev.keystore_pwd',
			'publish.android.dev.keystore_alias',
			'publish.android.dev.keystore_key_pwd',
			'min_version.ios',
			'ios.store_app_id',
			'ios.store_app_locale',
			'android.application_id',
			'min_version_code.ios',
			'min_version_code.android',
			'ios.bundle_id',
			'publish.ios.dev.namespace',
			'publish.ios.store_app_id',
			'publish.ios.shared_secret',
			'publish.android.namespace.registered',
			'publish.android.import_keystore',
		);
		$db_key_list['integrations']             = array(
			'push.firebase_server_key',
			'publish.ios.firebase_plist',
			'publish.ios.dev.firebase_plist',
			'publish.android.firebase_json',
			'publish.android.dev.firebase_json',
			'google.font_service_key',
			'learndash_course_downloading',
			'learndash_author_visible',
			'learndash_date_visible',
			'learndash_reader_app_compatibility',
			'vimeo.auth_code',
			'vimeo.referer',
		);
		$db_key_list['ios_signing_certificates'] = array(
			'publish.ios.signing_certificates_automatic',
			'publish.ios.signing_certificate',
			'publish.ios.signing_certificate_id',
			'publish.ios.signing_certificate_password',
			'publish.ios.dev.signing_certificate',
			'publish.ios.dev.signing_certificate_password',
		);

		$db_key_list['ios_provisioning_profiles'] = array(
			'publish.ios.provisioning_profile_automatic',
			'publish.ios.provisioning_profile',
			'publish.ios.dev.provisioning_profile',
		);

		$db_key_list['android_keystore_information'] = array(
			'publish.android.keystore',
			'publish.android.keystore_pwd',
			'publish.android.keystore_alias',
			'publish.android.keystore_key_pwd',
		);

		return ! empty( $module ) && isset( $db_key_list[ $module ] ) ? $db_key_list[ $module ] : $db_key_list;
	}

	/**
	 * Integration dir path.
	 *
	 * @return string|string[]
	 */
	public function get_bbapp_upload_dir() {
		$upload_dir = wp_upload_dir();
		$dir        = $upload_dir['basedir'] . '/bbapp/uploads';
		BBAPP_File::create_dir( $dir );
		$dir = str_replace( '/', DIRECTORY_SEPARATOR, $dir );

		return $dir;
	}

	/**
	 * Exportable fields.
	 *
	 * @param string $module_name Module name.
	 *
	 * @return array
	 */
	public function get_exportable_settings( $module_name = '' ) {
		$global_settings = \BuddyBossApp\Admin\Settings::instance()->get_settings();

		$export_settings     = array();
		$exportable_settings = $this->bbapp_db_key_lists( $module_name );
		foreach ( $exportable_settings as $exportable_setting ) {
			if ( isset( $global_settings[ $exportable_setting ] ) ) {
				$export_settings[ $exportable_setting ] = $global_settings[ $exportable_setting ];
			}
		}

		return $export_settings;
	}
}
