<?php
/**
 * Holds abstract functionality for rest test.
 *
 * @package BuddyBossApp\HealthCheck\Tests
 */

namespace BuddyBossApp\HealthCheck\Tests;

use BuddyBossApp\Admin\SetupAdmin;
use BuddyBossApp\HealthCheck\TestAbstract;

/**
 * Rest test class.
 */
abstract class RestTest extends TestAbstract {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	protected static $instance;

	/**
	 * Token transient key.
	 *
	 * @var string $token_transient_key
	 */
	private $token_transient_key = 'bbapp_healthcheck_user_token';

	/**
	 * Get Request
	 *
	 * @param string $url     Rest endpoint url.
	 * @param array  $data    Query parameter.
	 * @param bool   $is_auth authentication required for this endpoint or not. default true.
	 *
	 * @return array|bool
	 */
	public function get( $url, $data = array(), $is_auth = true ) {
		$args = array(
			'timeout'     => 45,
			'redirection' => 5,
			'httpversion' => '1.0',
			'blocking'    => true,
			'headers'     => array(),
			'body'        => $data,
			'cookies'     => array(),
		);

		if ( $is_auth ) {
			$token                          = $this->get_token();
			$args['headers']['accessToken'] = $token;
		}

		$response = wp_remote_get( $url, $args );
		if ( is_wp_error( $response ) ) {
			return false;
		}

		$server_headers = wp_remote_retrieve_headers( $response )->getAll();

		return array(
			'status'       => wp_remote_retrieve_response_code( $response ),
			'headers'      => ( $server_headers instanceof \Requests_Utility_CaseInsensitiveDictionary ? $server_headers->getAll() : $server_headers ),
			'body'         => wp_remote_retrieve_body( $response ),
			'content_type' => wp_remote_retrieve_header( $response, 'content-type' ),
		);
	}

	/**
	 * Post Request
	 *
	 * @param string $url     Rest endpoint url.
	 * @param array  $data    Query parameter.
	 * @param bool   $is_auth authentication required for this endpoint or not. default true.
	 *
	 * @return array|bool
	 */
	public function post( $url, $data = array(), $is_auth = true ) {
		$args = array(
			'timeout'     => 45,
			'redirection' => 5,
			'httpversion' => '1.0',
			'blocking'    => true,
			'headers'     => array(),
			'body'        => $data,
			'cookies'     => array(),
		);

		if ( $is_auth ) {
			$token                          = $this->get_token();
			$args['headers']['accessToken'] = $token;
		}

		$response = wp_remote_post( $url, $args );
		if ( is_wp_error( $response ) ) {
			return false;
		}

		$server_headers = wp_remote_retrieve_headers( $response )->getAll();

		return array(
			'status'       => wp_remote_retrieve_response_code( $response ),
			'headers'      => ( $server_headers instanceof \Requests_Utility_CaseInsensitiveDictionary ? $server_headers->getAll() : $server_headers ),
			'body'         => wp_remote_retrieve_body( $response ),
			'content_type' => wp_remote_retrieve_header( $response, 'content-type' ),
		);
	}

	/**
	 * Function to get expected status.
	 *
	 * @param array $response      Response status.
	 * @param int   $expected_code Expected status.
	 *
	 * @return array
	 */
	public function expect_status( $response, $expected_code ) {
		if ( $this->check_status_code( $response['status'], $expected_code ) ) {
			return $this->expect_return( true );
		}

		/* translators: %s: Expected code. */
		return $this->expect_return( false, sprintf( __( 'Response status code is not correct. Expected status code: (%d)', 'buddyboss-app' ), $expected_code ) );
	}

	/**
	 * Function to get expected content type.
	 *
	 * @param array  $response              Response.
	 * @param string $expected_content_type Expected.
	 *
	 * @return array
	 */
	public function expect_content_type( $response, $expected_content_type ) {
		$content_type = explode( ';', $response['content_type'] );
		if ( $content_type[0] === $expected_content_type ) {
			return $this->expect_return( true );
		}

		/* translators: %s: Content type. */
		return $this->expect_return( false, sprintf( __( 'Response content type is not correct. Expected content type: (%s)', 'buddyboss-app' ), $expected_content_type ) );
	}

	/**
	 * Function to get expected header.
	 *
	 * @param array $response        Response header.
	 * @param array $expected_header Expected header.
	 *
	 * @return array
	 */
	public function expect_header( $response, $expected_header ) {
		if ( is_array( $response['headers'] ) && is_array( $expected_header ) ) {
			$wrong_header = array_diff( $expected_header, $response['headers'] );
			if ( empty( $wrong_header ) ) {
				return $this->expect_return( true );
			}

			/* translators: %s: header. */
			return $this->expect_return( false, sprintf( __( 'Response header is not correct. Incorrect header: (%s)', 'buddyboss-app' ), http_build_query( $wrong_header, '', ', ' ) ) );
		}

		return $this->expect_return( false, __( 'Response/Expected header type is not correct', 'buddyboss-app' ) );
	}

	/**
	 * Function to get expected property.
	 *
	 * @param array $response          Response body.
	 * @param array $expected_property Expected property.
	 *
	 * @return array
	 */
	public function expect_property( $response, $expected_property ) {
		if ( ! is_array( $response ) ) {
			return $this->expect_return( false, __( 'Response content type is not supported for property check', 'buddyboss-app' ) );
		}

		$missing_property = array_diff_key( $expected_property, $response );
		if ( empty( $missing_property ) ) {
			return $this->expect_return( true );
		}

		/* translators: %s: Missing property. */
		return $this->expect_return( false, sprintf( __( 'Response property is not correct. Missing expected property: (%s)', 'buddyboss-app' ), implode( ',', array_keys( $missing_property ) ) ) );
	}

	/**
	 * Get test user token if exist in transient otherwise request for new token.
	 */
	private function get_token() {
		$data = get_transient( $this->token_transient_key );
		if ( false !== $data && isset( $data['token'] ) && ( ! empty( $data['token'] ) ) ) {
			return $data['token'];
		}

		return $this->request_auth_token();
	}

	/**
	 * Check Response status code
	 *
	 * @param int $code          Response status.
	 * @param int $expected_code Status.
	 *
	 * @return bool
	 */
	private function check_status_code( $code, $expected_code ) {
		return $code === $expected_code;
	}

	/**
	 * Check Response content type
	 *
	 * @param string $content_type Content type.
	 * @param string $expected_content_type Expected content type.
	 *
	 * @return bool
	 */
	private function check_content_type( $content_type, $expected_content_type ) {
		$content_type = explode( ';', $content_type );

		return $content_type[0] === $expected_content_type;
	}

	/**
	 * Generate test use token & store it in transient.
	 *
	 * @return bool|mixed
	 */
	private function request_auth_token() {
		$test_user = $this->get_test_user();
		if ( empty( $test_user ) ) {
			return false;
		}

		$response = $this->post(
			rest_url( 'buddyboss-app/auth/v1/jwt/token' ),
			array(
				'username'    => $test_user['username'],
				'password'    => $test_user['password'],
				'unique_id'   => $this->get_app_id(),
				'deviceToken' => '123',
			)
		);

		if ( $this->check_status_code( $response['status'], 200 ) && $this->check_content_type( $response['content_type'], 'application/json' ) ) {
			return $this->setup_test_user_token( $response['body'] );
		}

		return false;
	}

	/**
	 * Store test user auth data in transient & return token.
	 *
	 * @param object $response_body Response body.
	 *
	 * @return bool|mixed
	 */
	protected function setup_test_user_token( $response_body ) {
		$data = (array) json_decode( $response_body );
		if ( isset( $data['token'] ) ) {
			set_transient( $this->token_transient_key, $data, 10 * MINUTE_IN_SECONDS );

			return $data['token'];
		}

		return false;
	}

	/**
	 * Get test user for auth data
	 *
	 * @return bool|mixed|void
	 */
	protected function get_test_user() {
		return get_option( '_bbapp_healthcheck_user', false );
	}

	/**
	 * Get current app id.
	 *
	 * @return mixed|string
	 */
	protected function get_app_id() {
		$app = bbapp_get_app();

		return isset( $app['bbapp_app_id'] ) ? $app['bbapp_app_id'] : false;
	}

	/**
	 * Combine multiple error in one error.
	 *
	 * @param array  $errors          Array of errors.
	 * @param string $success_message Success mesage.
	 *
	 * @return array
	 */
	protected function combine_error( $errors, $success_message = '' ) {
		$is_passed = true;
		$error_str = '';
		foreach ( $errors as $error ) {
			if ( is_array( $error ) && true !== $error['status'] ) {
				$error_str .= $error['message'] . '<br/>';
				$is_passed  = false;
			}
		}

		if ( $is_passed ) {
			$error_str = $success_message;
		}

		return $this->expect_return( $is_passed, $error_str );
	}
}
