<?php
/**
 * Holds site health test abstract functionality.
 *
 * @package BuddyBossApp\HealthCheck
 */

namespace BuddyBossApp\HealthCheck;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * This is abstract Health Test Case class.
 * Extending this class will provide all logical feature to register test case for health check.
 *
 * Class HealthTestCaseAbstract
 *
 * @package BuddyBossApp\HealthCheck
 */
abstract class TestAbstract {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instances = array();

	/**
	 * Group
	 *
	 * @var array $group
	 */
	private $group = array();

	/**
	 * TestAbstract constructor.
	 */
	public function __construct() {
		/** Nothing here */
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return TestAbstract
	 */
	public static function instance() {
		$class = get_called_class();
		if ( ! isset( self::$instances[ $class ] ) ) {
			self::$instances[ $class ] = new $class();
			self::$instances[ $class ]->instance_load();
		}

		return self::$instances[ $class ];
	}

	/**
	 * TestAbstract load method.
	 *
	 * @return void
	 */
	public function instance_load() {
		add_action( 'init', array( $this, 'load' ), 99 );
		add_filter( 'bbapp_health_test_groups', array( $this, 'get_groups' ) );
	}

	/**
	 * Health test groups.
	 *
	 * @param array $groups Groups of health check list.
	 *
	 * @return mixed
	 */
	public function get_groups( $groups ) {
		$groups[ $this->group['name'] ] = $this->group;

		return $groups;
	}

	/**
	 * Get group data.
	 *
	 * @return array
	 */
	public function get_group_data() {
		return $this->group;
	}

	/**
	 * This function need to call to register test group
	 *
	 * @return mixed
	 */
	abstract public function load();

	/**
	 * This function helps to register test group.
	 *
	 * @param string $group_name  : Groups unique name.
	 * @param string $group_label : Group label.
	 * @param string $group_desc  : Group description.
	 *
	 * @return void
	 */
	public function register( $group_name, $group_label, $group_desc ) {
		$this->group = array(
			'name'  => $group_name,
			'label' => $group_label,
			'desc'  => $group_desc,
			'tests' => array(),
		);
	}

	/**
	 * This function helps to add test case for test groups.
	 *
	 * @param string $test_name  : Unique name for test.
	 * @param string $test_label : Test case label.
	 * @param string $test_desc  : Test case description.
	 * @param array  $callback   : Callback function for test case.
	 * @param bool   $async      : Asynce.
	 */
	public function add_test( $test_name, $test_label, $test_desc, $callback, $async = false ) {
		$test_name                          = str_replace( '_', '-', $test_name );
		$this->group['tests'][ $test_name ] = array(
			'label'    => $test_label,
			'desc'     => $test_desc,
			'callback' => $callback,
			'group'    => ( true === $async ) ? 'async' : 'direct',
		);
	}

	/**
	 * Function to expect return.
	 *
	 * @param string       $status  Status.
	 * @param false|string $message Message.
	 *
	 * @return array
	 */
	public function expect_return( $status, $message = false ) {
		return array(
			'status'  => $status,
			'message' => $message,
		);
	}
}

