<?php
/**
 * Health Check functionality.
 *
 * @package BuddyBossApp\HealthCheck
 */

namespace BuddyBossApp\Admin\Tools;

use BuddyBossApp\HealthCheck\Tests\ABCoreTest;
use BuddyBossApp\HealthCheck\Tests\CacheTest;
use BuddyBossApp\HealthCheck\Tests\PHPVersionTest;

/**
 * This class is used to load helth-check functionality.
 *
 * @package BuddyBossApp\HealthCheck
 */
class HealthCheck {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Test groups
	 *
	 * @var $test_groups
	 */
	private static $test_groups;

	/**
	 * Using Singleton, see instance()
	 */
	public function __construct() {
	}

	/**
	 * Class instance.
	 *
	 * @return HealthCheck
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->hooks(); // run the hooks.
		}

		return self::$instance;
	}

	/**
	 * Instance hooks.
	 */
	public function hooks() {
		add_action( 'init', array( $this, 'load' ) );
		add_action( 'init', array( $this, 'ajax_load' ), 100 );
		add_action( 'wp_ajax_health-check-get-sizes', array( 'BuddyBossApp\Helpers\WpDebugData', 'ajax_get_sizes' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_assets' ) );
	}

	/**
	 * Load Test Groups and it testcases.
	 */
	public function load() {
		PHPVersionTest::instance();
		CacheTest::instance();
		ABCoreTest::instance();
	}

	/**
	 * Health Check ajax load.
	 */
	public function ajax_load() {

		self::$test_groups = $this->get_tests();

		foreach ( self::$test_groups as $group ) {
			if ( ! empty( $group['tests'] ) ) {
				foreach ( $group['tests'] as $test_name => $test ) {
					if ( is_callable( $test['callback'] ) ) {
						if ( 'async' === $test['group'] ) {
							add_action( 'wp_ajax_health-check-' . $test_name, array( $this, 'ajax_test' ) );
						}
					}
				}
			}
		}
	}

	/**
	 * Function Health check ajax function.
	 */
	public function ajax_test() {
		$action    = ( ! empty( $_POST['action'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['action'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$test_name = str_replace( 'health-check-', '', $action );

		foreach ( self::$test_groups as $group ) {
			if ( ! empty( $group['tests'][ $test_name ] ) ) {
				$test = $group['tests'][ $test_name ];

				$result = $this->perform_test( $test['callback'] );

				$response = array(
					'label'       => $test['label'],
					'status'      => $group['name'],
					'badge'       => array(
						'label' => ( isset( $result['status'] ) && ! empty( $result['status'] ) ) ? __( 'Passed', 'buddyboss-app' ) : __( 'Failed', 'buddyboss-app' ),
						'color' => ( isset( $result['status'] ) && ! empty( $result['status'] ) ) ? __( 'green', 'buddyboss-app' ) : __( 'red', 'buddyboss-app' ),
					),
					'description' => $test['desc'],
					'actions'     => isset( $result['message'] ) ? $result['message'] : array(),
					'test'        => $test_name,
				);
				wp_send_json_success( $response );
			}
		}
	}

	/**
	 * Register admin assets.
	 */
	public function admin_enqueue_assets() {
		global $pagenow;

		$_get_var = function ( $get_key ) {
			return ( isset( $_GET[ $get_key ] ) ) ? sanitize_text_field( wp_unslash( $_GET[ $get_key ] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		};

		if ( 'admin.php' === $pagenow && $_get_var( 'page' ) === 'bbapp-tools' && ( empty( $_get_var( 'setting' ) ) || $_get_var( 'setting' ) === 'info' ) ) {

			wp_enqueue_style( 'site-health' );
			wp_enqueue_script( 'site-health' );

			$health_check_js_variables = array(
				'nonce'       => array(
					'site_status'        => wp_create_nonce( 'health-check-site-status' ),
					'site_status_result' => wp_create_nonce( 'health-check-site-status-result' ),
				),
				'site_status' => array(
					'direct' => array(),
					'async'  => array(),
					'issues' => array(
						'good'        => 0,
						'recommended' => 0,
						'critical'    => 0,
					),
				),
			);

			foreach ( self::$test_groups as $group ) {
				$health_check_js_variables['site_status']['issues'][ $group['name'] ] = count( $group['tests'] );
				if ( ! empty( $group['tests'] ) ) {
					foreach ( $group['tests'] as $test_name => $test ) {
						if ( is_callable( $test['callback'] ) ) {
							$result = array();
							if ( 'direct' === $test['group'] ) {
								$result = $this->perform_test( $test['callback'] );

								// `bbapp-helth- . $test_name` identifier should be a unique identifier for the test property.
								// Plugins and themes are encouraged to prefix test identifiers with their slug to avoid collisions between tests properties.
								// Referance https://developer.wordpress.org/reference/hooks/site_status_tests/
								$health_check_js_variables['site_status'][ $test['group'] ][ 'bbapp-helth-' . $test_name ] = array(
									'label'       => $test['label'],
									'status'      => $group['name'],
									'badge'       => array(
										'label' => ( isset( $result['status'] ) && ! empty( $result['status'] ) ) ? __( 'Passed', 'buddyboss-app' ) : __( 'Failed', 'buddyboss-app' ),
										'color' => ( isset( $result['status'] ) && ! empty( $result['status'] ) ) ? __( 'green', 'buddyboss-app' ) : __( 'red', 'buddyboss-app' ),
									),
									'description' => $test['desc'],
									'actions'     => isset( $result['message'] ) ? $result['message'] : array(),
									'test'        => $test_name,
								);
							} else {
								// `bbapp-helth- . $test_name` identifier should be a unique for the test property.
								// Plugins and themes are encouraged to prefix test identifiers with their slug to avoid collisions between test properties.
								// Referance https://developer.wordpress.org/reference/hooks/site_status_tests/
								$health_check_js_variables['site_status'][ $test['group'] ][ 'bbapp-helth-' . $test_name ] = array(
									'test'      => $test_name,
									'completed' => false,
								);
							}
						}
					}
				}
			}

			wp_localize_script( 'site-health', 'SiteHealth', $health_check_js_variables );
			wp_localize_script( 'bbapp-script', 'SiteHealth', $health_check_js_variables );
		}
	}

	/**
	 * Run a Site Health test directly.
	 *
	 * @param array|string|object $callback Callback method.
	 *
	 * @return mixed|void
	 * @since 5.4.0
	 */
	private function perform_test( $callback ) {
		/**
		 * Filter the output of a finished Site Health test.
		 *
		 * @param array $test_result {
		 *     An associated array of test result data.
		 *
		 * @param string $status The status of the test, which can be a value of `good`, `recommended` or `critical`.
		 * @param string $message An action to direct the user to where they can resolve the issue, if one exists.
		 * }
		 */
		return apply_filters( 'bbapp_health_test_case_result', call_user_func( $callback ) );
	}

	/**
	 * Get tests groups.
	 *
	 * @return mixed|void
	 */
	public function get_tests() {
		return apply_filters( 'bbapp_health_test_groups', array() );
	}
}
