<?php
/**
 * Holds site health and other tools functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin\Tools;

use BuddyBossApp\Admin\FormProvider;
use BuddyBossApp\Admin\SettingProvider;
use BuddyBossApp\Admin\SetupAdmin;
use BuddyBossApp\Admin\Settings;
use BuddyBossApp\Tools\Logger;
use BuddyBossApp\Permissions;

/**
 * Debug class.
 */
class Debug {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Error/Notice messages.
	 *
	 * @var $messages
	 */
	public $messages;

	/**
	 * Form settings.
	 *
	 * @var $settings_form
	 */
	private $settings_form;

	/**
	 * Get the instance of the class.
	 *
	 * @return Debug
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Debug constructor.
	 */
	public function __construct() {
	}

	/**
	 * Debug load method.
	 *
	 * @return void
	 */
	public function load() {
		// load the active plugin information as early as possible.
		if ( is_admin() && isset( $_GET['page'] ) && 'bbapp-tools' === $_GET['page'] ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			// Avoid option_active_plugins to get altered by 3rd party plugins.
			add_filter( 'option_active_plugins', array( $this, 'option_active_plugins' ), - 1 );
			add_filter( 'option_active_plugins', array( $this, 'option_active_plugins' ), 9999 );
			add_filter( 'site_option_active_sitewide_plugins', array( $this, 'option_active_plugins' ), 9999 );

			$this->settings_form = new FormProvider( 'settings' );

			// holds global settings storage handle.
			$app_settings_provider = new SettingProvider( 'app_settings' );

			// register app settings storing & receiving.
			$app_settings_provider->save_setting_hook( array( Settings::instance(), 'save_settings' ) );
			$app_settings_provider->get_setting_hook( array( Settings::instance(), 'get_settings' ) );
			$this->settings_form->link_setting_provider( $app_settings_provider );

			// Logs Settings.
			if ( isset( $_GET['setting'] ) && 'log' === $_GET['setting'] ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended

				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'logger_heading',
						array(
							'type'  => 'heading',
							'value' => __( 'Log Settings', 'buddyboss-app' ),
							'desc'  => '',
							'icon'  => 'bb-icon-bf bb-icon-clipboard',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'logger.info_log.enabled',
						array(
							'label'          => __( 'General Logs', 'buddyboss-app' ),
							'label_checkbox' => __( 'Enable General Logs', 'buddyboss-app' ),
							'desc'           => __( 'For development and de-bugging of general actions.', 'buddyboss-app' ),
							'type'           => 'checkbox',
							'value'          => true,
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'logger.api_log.enabled',
						array(
							'label'          => __( 'API Logs', 'buddyboss-app' ),
							'label_checkbox' => __( 'Enable API Logs', 'buddyboss-app' ),
							'desc'           => __( 'For development and de-bugging of recent API requests.', 'buddyboss-app' ),
							'type'           => 'checkbox',
							'value'          => true,
						)
					)
				);

				if ( bbapp_is_active( 'iap' ) ) {
					$this->settings_form->add_field(
						$app_settings_provider->field(
							'logger.iap_log.enabled',
							array(
								'label'          => __( 'IAP Logs', 'buddyboss-app' ),
								'label_checkbox' => __( 'Enable In-App Purchase Logs', 'buddyboss-app' ),
								'desc'           => __( 'For development and de-bugging of recent IAP actions.', 'buddyboss-app' ),
								'type'           => 'checkbox',
								'value'          => true,
							)
						)
					);
				}
			}

			add_action( 'admin_init', array( $this->settings_form, 'save' ) );
		}

		add_action( 'admin_init', array( $this, 'process_clear_log_button' ) );
		add_action( 'admin_init', array( $this, 'process_debug_tool_submit' ) );
	}

	/**
	 * Handles The Clear Log action button.
	 *
	 * @return false|void
	 */
	public function process_clear_log_button() {

		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			return false;
		}

		// Only load on bbapp-tools page.
		if ( 'admin.php' !== SetupAdmin::instance()->get_page_now() || empty( $_GET['page'] ) || 'bbapp-tools' !== $_GET['page'] ) {
			return false;
		}

		// Permission check.
		if ( ! Permissions::instance()->can_manage_app() ) {
			wp_die( "You don't have permission to access this page.", 'buddyboss-app' );
		}

		if ( ! isset( $_GET['do_action'] ) ) {
			return;
		}

		if ( ! check_admin_referer( 'bbapp-log-clear' ) ) {
			$this->messages[] = array(
				'type'    => 'error',
				'message' => __( 'Something unexpected happened. Please try again later.', 'buddyboss-app' ),
			);
		}

		// Clear logs.
		if ( 'clear_log' === $_GET['do_action'] && check_admin_referer( 'bbapp-log-clear' ) ) {
			// delete code.
			$supported_types = Logger::get_logger_types();
			$supported_types = array_keys( $supported_types );
			$supported_types = array_filter( $supported_types );
			$log_filter_get  = ! empty( $_GET['log_filter'] ) ? bbapp_input_clean( wp_unslash( $_GET['log_filter'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$log_type        = ( in_array( $log_filter_get, $supported_types, true ) ) ? $log_filter_get : '';

			Logger\Helpers::clear_logs( $log_type );
			$this->messages[] = array(
				'type'    => 'updated',
				'message' => esc_html__( 'Log cleared successfully!', 'buddyboss-app' ),
			);
		}
	}

	/**
	 * We are preserving the original db values on which are returned on most first time.
	 *
	 * @param array $plugins Array of plugins.
	 *
	 * @return mixed
	 */
	public function option_active_plugins( $plugins ) {
		static $cache_plugins;

		// avoid infinite loop.
		if ( isset( $cache_plugins ) && isset( $cache_plugins[ current_action() ] ) ) {
			return $cache_plugins[ current_action() ];
		}

		$cache_plugins[ current_action() ] = $plugins;

		return $plugins;
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {
		$tabs = array(
			'import'            => esc_html__( 'Import', 'buddyboss-app' ),
			'export'            => esc_html__( 'Export', 'buddyboss-app' ),
			'info'              => esc_html__( 'Site Health', 'buddyboss-app' ),
			'debug'             => esc_html__( 'Transients', 'buddyboss-app' ),
			'log'               => esc_html__( 'Logs', 'buddyboss-app' ),
			'generate-keystore' => esc_html__( 'Generate KeyStore', 'buddyboss-app' ),
		);

		if ( bbapp_is_active( 'access_controls' ) ) {
			$tabs['calculate-members'] = esc_html__( 'Calculate Members', 'buddyboss-app' );
		}

		return $tabs;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {
		$default      = 'import';
		$setting      = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Function to current view.
	 *
	 * @param string $default Default view.
	 *
	 * @return string
	 */
	public function get_current_view( $default = 'logs' ) {
		return ( isset( $_GET['view'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['view'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	}

	/**
	 * Return Current tabs for logs.
	 */
	public function get_logs_tabs() {
		$tabs         = array(
			'logs'     => __( 'Logs', 'buddyboss-app' ),
			'settings' => __( 'Log Settings', 'buddyboss-app' ),
		);
		$current_view = $this->get_current_view();
		?>

		<h2 class="nav-tab-wrapper">
			<?php foreach ( $tabs as $k => $v ) : ?>
				<a href="<?php echo esc_url( bbapp_get_super_admin_url( 'admin.php?page=bbapp-tools&setting=log&view=' . $k ) ); ?>" class="nav-tab <?php echo ( $current_view === $k ) ? 'nav-tab-active' : ''; ?>">
					<?php echo esc_html( $v ); ?>
				</a>
			<?php endforeach; ?>
		</h2>
		<?php
	}

	/**
	 * Renders the setting screen.
	 *
	 * @return false|void
	 */
	public function render_screen() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			printf( '<p>%1$s</p>', esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) );

			return false;
		}

		\BuddyBossApp\Admin\SetupAdmin::instance()->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab() );

		if ( 'log' === $this->get_current_sub_tab() ) {
			$this->get_logs_tabs();
			if ( 'settings' === $this->get_current_view() ) {
				$this->settings_form->render_form();
			} else {
				include bbapp()->plugin_dir . 'views/tools/view-log.php';
			}
		} elseif ( 'info' === $this->get_current_sub_tab() ) {
			include bbapp()->plugin_dir . 'views/tools/view-site-health-info.php';
		} elseif ( 'import' === $this->get_current_sub_tab() ) {
			include bbapp()->plugin_dir . 'views/tools/view-import-export.php';
		} elseif ( 'export' === $this->get_current_sub_tab() ) {
			include bbapp()->plugin_dir . 'views/tools/view-import-export.php';
		} elseif ( 'generate-keystore' === $this->get_current_sub_tab() ) {
			include bbapp()->plugin_dir . 'views/tools/view-generate-keystore.php';
		} elseif ( bbapp_is_active( 'access_controls' ) && 'calculate-members' === $this->get_current_sub_tab() ) {
			include bbapp()->plugin_dir . 'views/access-controls/tools/calculate-members.php';
		} else {
			include bbapp()->plugin_dir . 'views/tools/view-debug.php';
		}
	}

	/**
	 * Show the errors & updates
	 */
	public function show_messages() {
		if ( isset( $_GET['success'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$this->messages[] = array(
				'type'    => 'updated',
				'message' => __( 'Settings saved.', 'buddyboss-app' ),
			);
		}

		if ( ! empty( $this->messages ) ) {
			foreach ( $this->messages as $message ) {
				printf( '<div id="message" class="%1$s"><p>%2$s</p></div>', esc_attr( $message['type'] ), esc_html( $message['message'] ) );
			}
		}
	}

	/**
	 * Handles The Debug Process Submit
	 *
	 * @return false|void
	 */
	public function process_debug_tool_submit() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			return false;
		}

		if ( ! isset( $_POST['bbapp_update_settings_debug_tools'] ) ) {
			return;
		}

		if ( ! check_admin_referer( 'bbapp_update_settings_debug_tools', 'bbapp_update_settings_debug_tools' ) ) {
			$this->messages[] = array(
				'type'    => 'error',
				'message' => __( 'Something unexpected happened. Please try again later.', 'buddyboss-app' ),
			);
		}

		// Clear Transients.
		if ( isset( $_POST['clear_transients'] ) ) {
			$delete_transients = array(
				'_bbapp_app_signed_details',
				bbapp_get_app_info_transient_key(),
				'bbapp_preview_members',
				'bbapp_preview_forums',
				'bbapp_preview_single_lesson',
				'bbapp_preview_topics',
				'bbapp_preview_single_profile',
				'bbapp_preview_registration',
				'bbapp_preview_profile_tabs',
				'_bbapp_fetch_last_versions',
				'bbapp_last_build_status_interval_check',
				'bbapp_is_under_maintenance',
				'bbapp_apple_get_local_devices',
				'bbapp_apple_client_apps_bundle_ids',
				'bbapp_apple_client_apps_bundle_id',
				'bbapp_apple_client_apps',
				'bbapp_apple_client_users',
				'bbapp_apple_team_info',
				'bbapp_get_app_core',
				'bbapp_apple_team_info',
				'bbapp_google_app_details',
				'bbapp_google_release_data',
				'bbapp_google_all_production_release_data',
				'bbapp_appcenter_settings',
				'bbapp_publisher_email_address',
				'_bbapp_deeplink_wp_url_data',
				'_bbapp_deeplink_bp_url_data',
				'bbapp_cron_job_health_flag',
				'bbapp_healthcheck_user_token',
				'_bbapp_google_typography_fonts',
				'bbapp_apple_team_id',
				'bbapp_publish_term_condition',
				'bbapp_apple_client_invited_users',
				'bbapp_apple_client_app_id',
				'bbapp_license_type',
				'update_plugins',
				'bbapp_build_lock_ios',
				'bbapp_build_lock_android',
			);

			foreach ( $delete_transients as $tr ) {
				delete_transient( $tr );
				delete_site_transient( $tr );
			}

			\BuddyBossApp\Build::instance()->clear_all_build_transients();
			\BuddyBossApp\Admin\Publish\Publish::instance()->delete_all_publish_transients();

			$this->clear_all_transients();

			$this->messages[] = array(
				'type'    => 'updated',
				'message' => __( 'Transients are cleared!', 'buddyboss-app' ),
			);
		}

		// Clear Duplicates Devices Tokens.
		if ( isset( $_POST['clear_duplicates_device_tokens'] ) ) {
			bbapp_table_fix_remove_dublicates_device_tokens();

			$this->messages[] = array(
				'type'    => 'updated',
				'message' => __( 'Duplicate entries cleared.', 'buddyboss-app' ),
			);
		}

		// Clear all Devices Tokens.
		if ( isset( $_POST['clear_all_device_tokens'] ) ) {
			bbapp_table_remove_all_device_tokens();

			$this->messages[] = array(
				'type'    => 'updated',
				'message' => __( 'All entries cleared!', 'buddyboss-app' ),
			);
		}
	}

	/**
	 * Function to get active plugins.
	 *
	 * @return array|mixed
	 */
	public function get_active_plugins() {
		static $cache_active_plugins;

		if ( isset( $cache_active_plugins ) ) {
			return $cache_active_plugins;
		}

		require_once ABSPATH . 'wp-admin/includes/plugin.php';
		require_once ABSPATH . 'wp-admin/includes/update.php';

		if ( ! function_exists( 'get_plugin_updates' ) ) {
			return array();
		}

		$active_plugins = (array) get_option( 'active_plugins', array() );

		if ( is_multisite() ) {
			$network_activated_plugins = array_keys( get_site_option( 'active_sitewide_plugins', array() ) );
			$active_plugins            = array_merge( $active_plugins, $network_activated_plugins );
		}

		$active_plugins      = array_unique( $active_plugins );
		$active_plugins_data = array();

		foreach ( $active_plugins as $plugin ) {
			$data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin );

			// convert plugin data to json response format.
			$active_plugins_data[] = array(
				'plugin'            => $plugin,
				'name'              => $data['Name'],
				'version'           => $data['Version'],
				'url'               => $data['PluginURI'],
				'author_name'       => $data['AuthorName'],
				'author_url'        => esc_url_raw( $data['AuthorURI'] ),
				'network_activated' => $data['Network'],
			);
		}

		$cache_active_plugins = $active_plugins_data;

		return $active_plugins_data;
	}

	/**
	 * Function will clear all transients from DB.
	 */
	public function clear_all_transients() {
		$option_name_arr = array(
			'_buddyboss_app_ld_course_enrolled_users_',
			'_transient_bbapp-',
			'_get_build_status_cache_',
			'_bbapp_segment_data_',
			'bbapp_vimeo_vid_cache',
			'bbapp_apple_app_fetched_data_',
			'bbapp_android_is_google_client_connected_cache_',
			'bbapp_android_is_connected_cache_',
			'bbapp_build_detail_',
			'bbapp_apple_bundle_ids_cache_',
			'bbapp_apple_app_info_',
			'bbapp_apple_app_details_',
		);
		bbapp_delete_transients( $option_name_arr );
	}
}
