<?php
/**
 * Holds app admin menu related functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\Admin\AccessControls\Settings;
use BuddyBossApp\Admin\Appearance\Branding;
use BuddyBossApp\Admin\Tools\Debug;
use BuddyBossApp\AppSettings;
use BuddyBossApp\InAppPurchases\SubscriptionGroupTaxonomy;
use BuddyBossApp\Menus\MenuManager;
use BuddyBossApp\Permissions;

/**
 * Setup admin class.
 */
class SetupAdmin {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return SetupAdmin
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * SetupAdmin constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Load hooks here.
	 */
	public function load() {
		// Initiate hooks.
		$this->hooks();

		// Load the required classes.
		Components::instance();
		Settings::instance();
		Integrations::instance();
		Appearance::instance();
		AppLanguages::instance();

		if ( bbapp_is_active( 'iap' ) ) {
			InAppPurchases::instance();
		}

		if ( bbapp_is_active( 'push_notification' ) ) {
			Notification::instance();
		}

		Menus::instance();

		if ( bbapp_is_active( 'access_controls' ) ) {
			AccessControls::instance();
		}

		Build::instance();
		BuildUpload::instance();
		Configure::instance();
		Publish::instance();
		ManageApp::instance();
		Debug::instance();
		UserProfile::instance();
	}

	/**
	 * Hooks.
	 */
	public function hooks() {
		if ( is_admin() ) {
			add_action( 'admin_menu', array( $this, 'setup_menus' ), 21 );

			// Only render the BuddyBossApp Nav on Network Dashboard when it's a Network Enabled.
			if ( bbapp()->is_network_activated() ) {
				add_action( 'network_admin_menu', array( $this, 'setup_menus' ), 21 );
			}

			add_action( 'admin_init', array( $this, 'adjust_the_menus' ) );
			add_action( 'admin_notices', array( $this, 'admin_notices' ) );
			add_action( 'all_admin_notices', array( $this, 'render_tabs' ) );
			add_filter( 'parent_file', array( $this, 'bbapp_set_current_menu' ) );
            // About Screens.
			add_action( 'admin_footer', array( $this, 'release_note_modal_content' ) );

			// Add link to settings page.
			add_filter( 'plugin_action_links', array( $this, 'actions_links' ), 10, 2 );
			add_filter( 'network_admin_plugin_action_links', array( $this, 'actions_links' ), 10, 2 );

			// Admin notice dismiss js.
			add_action( 'admin_footer', array( $this, 'admin_notices_dismiss_js' ) );
			add_filter( 'submenu_file', array( $this, 'bbapp_app_page_sub_menu' ) );
		}
	}

	/**
	 * Register new cpt,taxonomy in buddyboss-app menu.
	 *
	 * @param string $parent_file Parent file.
	 *
	 * @return string
	 */
	public function bbapp_set_current_menu( $parent_file ) {
		global $current_screen;

		if ( 'app_page' === $current_screen->post_type ) {
			$parent_file = 'bbapp-settings';
		}

		return $parent_file;
	}

	/**
	 * Show admin notice for connect app
	 */
	public function admin_notices() {
		global $pagenow;

		$page_get      = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$post_type_get = ( ! empty( $_GET['post_type'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['post_type'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ( 'edit.php' === $pagenow && 'app_page' === $post_type_get ) || ( 'admin.php' === $pagenow && 'bbapp-' === substr( $page_get, 0, strlen( 'bbapp-' ) ) ) ) {
			// Notice for all connect.
			if ( ! $this->is_current_app_connected() ) {
				$class   = 'notice notice-info';
				$linkurl = bbapp_get_admin_url( 'admin.php?page=bbapp-connect' );
				$app_get = ( ! empty( $_GET['app'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['app'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

				if ( ! empty( $app_get ) ) {
					$linkurl = add_query_arg( 'app', $app_get, $linkurl );
				}

				/* translators: %s: Link. */
				$message = sprintf( __( 'Your app is not yet connected. <a href="%s">Manage your app connection.</a>', 'buddyboss-app' ), $linkurl );

				printf( '<div id="bbapp-site-connect-notice" class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), wp_kses_post( $message ) );
			}

			/**
			 * Fires on admin notice.
			 */
			do_action( 'bbapp_admin_notice' );
		}
	}

	/**
	 * Setup all admin menus found available in plugin.
	 */
	public function setup_menus() {
		$bbapp_sub_menus = $this->sub_menus();

		foreach ( $bbapp_sub_menus as $k => $menu ) {
			if ( ! isset( $menu[5] ) ) {
				add_submenu_page(
					$menu[0],
					$menu[1],
					$menu[2],
					$menu[3],
					$menu[4]
				);
			} else {
				add_submenu_page(
					$menu[0],
					$menu[1],
					$menu[2],
					$menu[3],
					$menu[4],
					$menu[5]
				);
			}
		}
	}

	/**
	 * Adjust bb app menus.
	 */
	public function adjust_the_menus() {
		global $menu;

		/**
		 * Remove any sap before bbapp menu.
		 */
		$bbapp_found = false;
		$saps        = array();

		if ( is_array( $menu ) ) {
			foreach ( $menu as $index => $m ) {
				// remove sap after buddyboss menu.
				if ( 'separator1' === $m[2] ) {
					$saps[] = $index;
				}

				if ( 'bbapp-settings' === $m[2] ) {
					$bbapp_found = $index;
					break;
				}
			}
		}

		if ( $bbapp_found ) {
			foreach ( $saps as $sap_index ) {
				unset( $menu[ $sap_index ] );
			}
		}

		/**
		 * Fix the index. Solving index contains decimals.
		 */
		$tmp = array();

		if ( is_array( $menu ) ) {
			foreach ( $menu as $m ) {
				$tmp[] = $m;
			}
		}

		$menu = $tmp; //phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited

		/**
		 * Here we will forcefully add separator after bbapp menu.
		 */
		$bbapp_found = false;

		if ( is_array( $menu ) ) {
			// Add Separator after BuddyBossApp Menu.
			foreach ( $menu as $index => $m ) {
				if ( $bbapp_found ) {
					array_splice( $menu, $index, 0, array( 'sap' ) );
					break;
				}

				if ( 'bbapp-settings' === $m[2] ) {
					$bbapp_found = $index;
				}
			}
		}

		// convert all sap into separator.
		foreach ( $menu as $index => $m ) {
			if ( 'sap' === $m ) {
				$menu[ $index ] = array( //phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
					'',
					'read',
					'separator1',
					'',
					'wp-menu-separator',
				);
			}
		}
	}

	/**
	 * Override WP core menu.
	 */
	public function bp_core_modify_admin_menu_highlight() {
		global $plugin_page, $submenu_file;

		// This tweaks the Settings subnav menu to show only one BuddyPress menu item.
		if ( ! in_array( $plugin_page, array( 'bp-activity', 'bp-general-settings' ), true ) ) {
			$submenu_file = 'bp-components'; //phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		}

		// Network Admin > Tools.
		if ( in_array( $plugin_page, array( 'bp-tools', 'available-tools' ), true ) ) {
			$submenu_file = $plugin_page; //phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		}
	}

	/**
	 * Render the bbapp tabs on the pages which WordPress controls.
	 */
	public function render_tabs() {
		global $pagenow;

		if ( ! in_array( $pagenow, array( 'edit.php', 'admin.php', 'edit-tags.php' ), true ) ) {
			return false;
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		$render_tabs = false;

		/**
		 * Native Pages Tab.
		 */
		$post_type_get = ( ! empty( $_GET['post_type'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['post_type'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'edit.php' === $pagenow && 'app_page' === $post_type_get ) {
			$render_tabs      = true;
			$current_page_var = 'bbapp-pages';
		}

		if ( ! $render_tabs ) {
			return false;
		}

		$old_page_var = isset( $_GET['page'] ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$_GET['page'] = $current_page_var;
		?>
		<div class="wrap">
			<?php $this->primary_tabs(); ?>
		</div>
		<?php
		$_GET['page'] = $old_page_var;
		?>
		<style>
			.bbapp-subsubsub {
				width: 100%;
				border-bottom: 1px solid #e6e6e6;
				padding-bottom: 10px;
				margin-bottom: 5px;
			}
		</style>
		<?php
		$old_page_var = isset( $_GET['page'] ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$_GET['page'] = $current_page_var;
		$_GET['page'] = $old_page_var;
	}

	/**
	 * Render sub tabs.
	 *
	 * @param array  $sub_settings Sub settings array.
	 * @param string $current_tab Current tab.
	 * @param string $param_key   Primary key.
	 *
	 * @return bool
	 */
	public function render_subtabs( $sub_settings, $current_tab, $param_key = 'setting' ) {
		// only render something when there is more than 1 sub menu.
		if ( empty( $sub_settings ) || count( $sub_settings ) < 2 ) {
			return false;
		}

		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'bbapp-menus' === $page_get ) {
			$tabbar_item_exist = MenuManager::instance()->is_tabbar_item_exist();
		}

		echo '<ul class="subsubsub bbapp-subsubsub">';

		$i = 1;

		foreach ( $sub_settings as $sub_name => $sub_label ) {
			$sap = '|';

			if ( count( $sub_settings ) === $i ) {
				$sap = '';
			}

			$current = '';

			if ( $current_tab === $sub_name ) {
				$current = ' class=current';
			}

			$page = isset( $_GET['page'] ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$link = bbapp_get_admin_url( 'admin.php?page=' . $page . '&' . $param_key . '=' . $sub_name );

			if ( bbapp_is_valid_url( $sub_name ) ) {
				$link = $sub_name;
			}

			// Redirect type sub setting..
			if ( bbapp_is_valid_url( $sub_name ) ) {
				$link = $sub_name;
			}

			if ( 'bbapp-menus' === $page_get && ! $tabbar_item_exist && 'headerbar' === $sub_name ) {
				printf( '<li><a href="javascript:void(0);" class="bbapp-no-header">%1$s</a> %2$s</li>', wp_kses_post( $sub_label ), esc_html( $sap ) );
			} else {
				echo '<li><a href="' . esc_url( $link ) . '"' . esc_attr( $current ) . '>' . wp_kses_post( $sub_label ) . '</a> ' . esc_html( $sap ) . '</li>';
			}

			++$i;
		}
		echo '</ul>';
		echo '<div class="clear"></div>';
	}

	/**
	 * Renders Components Screen
	 */
	public function components_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php Components::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * Renders Setting Screen
	 */
	public function setting_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php \BuddyBossApp\Admin\Settings::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * Renders Setting Screen
	 */
	public function integrations_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php Integrations::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * Renders Build Screen
	 */
	public function build_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php Build::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * Renders Upload Screen
	 */
	public function upload_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php BuildUpload::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * Renders Piblish Screen
	 */
	public function publish_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php Publish::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * Branding Setting Screen
	 */
	public function app_layout_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php
			$this->primary_tabs();

			$menu_page    = ( ! empty( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : 'branding'; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$current_page = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( 'bbapp-appearance' === $current_page && 'branding' === $menu_page ) {
				$branding_opt = \BuddyBossApp\Branding::instance()->get_options();
				$filename     = 'home_screen_logo_v1.png';

				if ( ! isset( $branding_opt['uploads_hash'][ $filename ] ) ) {
					?>
					<div class="bbapp-notice bbapp-notice-warning">
						<p>
							<?php
							printf(
							/* translators: %s: Link. */
								__( 'Please update your Home Screen logo. %s', 'buddyboss-app' ),
								'<a href="#bbapp-widgets">' . esc_html__( 'Update now', 'buddyboss-app' ) . '</a>'
							);
							?>
						</p>
					</div>
					<?php
				}
			}

			Appearance::instance()->render_screen();
			?>
		</div>
		<?php
	}

	/**
	 * Menus Screen
	 *
	 * @since 1.5.2.1
	 */
	public function app_menus_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php
			$this->primary_tabs();

			$current_page = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$menu_page    = ( ! empty( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( 'bbapp-menus' === $current_page && in_array( $menu_page, array( 'tabbar', 'headerbar' ), true ) ) {
				$current_menu_id    = ( ! empty( $_GET['menu'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['menu'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				$headebar_id        = ( 'tabbar' === $menu_page ) ? MenuManager::instance()->get_headerbar_menu_from_tabbar_id( $current_menu_id ) : $current_menu_id;
				$tabbar_id          = ( 'headerbar' === $menu_page ) ? MenuManager::instance()->get_tabbar_menu_from_headerbar_id( $current_menu_id ) : $current_menu_id;
				$more_menu_position = MenuManager::instance()->get_more_menu_position( $headebar_id );
				$menu_count         = MenuManager::instance()->get_menu_items_count( $tabbar_id );

				if ( 'headerbar' === $more_menu_position && 5 > $menu_count ) {
					?>
					<div class="bbapp-notice bbapp-notice-info">
						<p>
							<?php esc_html_e( 'More menu has now moved to the header. You can use your tab-bar\'s 5th slot for other menu items.', 'buddyboss-app' ); ?>
						</p>
					</div>
					<?php
				}
			}

			if ( 'bbapp-menus' === $current_page ) {
				if ( ! MenuManager::instance()->get_default_menu_id( 'tabbar' ) ) {
					?>
					<div class="bbapp-notice bbapp-notice-info">
						<p>
							<?php esc_html_e( 'Please create and save your first Tab Bar menu to activate the Header Bar menu.', 'buddyboss-app' ); ?>
						</p>
					</div>
					<?php
				}
			}

			Menus::instance()->render_screen();
			?>
		</div>
		<?php
	}

	/**
	 * Access Control Screen
	 *
	 * @since 1.5.2.1
	 */
	public function access_control_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php AccessControls::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * Appearance Setting Screen
	 */
	public function translations_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php AppLanguages::instance()->render(); ?>
		</div>
		<?php
	}

	/**
	 * Help Setting Screen
	 */
	public function help_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php Help::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * NotificationScreen Setting Screen
	 */
	public function notification_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php Notification::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * In-App Purchase Screens
	 */
	public function iap_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php InAppPurchases::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * BuddyBossApp Debug Screen.
	 */
	public function bbapp_import_export_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php Debug::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * App Configuration Screen.
	 */
	public function manage_configuration() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php Configure::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * App Manage Screen.
	 */
	public function manage_screen() {
		?>
		<div class="wrap buddyboss-app-screen">
			<h1 style="display: none;">
				<div id="icon-options-general" class="icon32"><br></div>
			</h1>
			<?php $this->primary_tabs(); ?>
			<?php ManageApp::instance()->render_screen(); ?>
		</div>
		<?php
	}

	/**
	 * App primary tabs.
	 */
	public function primary_tabs() {
		$sub_menus      = $this->sub_menus();
		$tabs_groups    = array();
		$settings_group = array(
			'bbapp-components',
			'bbapp-settings',
			'bbapp-integrations',
			'bbapp-appearance',
			'bbapp-menus',
			'bbapp-translations',
			'bbapp-tools',
			'bbapp-help',
		);
		$page_get       = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( in_array( $page_get, $settings_group, true ) ) {
			$tabs_groups = $settings_group;
		}

		$content_group = array(
			'app_page', // for in_array logic.
			SubscriptionGroupTaxonomy::$taxonomy,
			'bbapp-pages',
			'bbapp-notification',
			'bbapp-iap',
			'bbapp-reporting',
			'bbapp-access-controls',
		);

		if ( in_array( $page_get, $content_group, true ) ) {
			$tabs_groups = $content_group;
		}

		$post_type_get = ( ! empty( $_GET['post_type'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['post_type'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( in_array( $post_type_get, $content_group, true ) ) {
			$tabs_groups = $content_group;
		}

		$taxonomy_get = ( ! empty( $_GET['taxonomy'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['taxonomy'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( in_array( $taxonomy_get, $content_group, true ) ) {
			$tabs_groups = $content_group;
		}

		$manage_group = array(
			'bbapp-build',
			'bbapp-build-upload',
			'bbapp-connect',
			'bbapp-configure',
		);

		if ( in_array( $page_get, $manage_group, true ) ) {
			$tabs_groups = $manage_group;
		}

		/**
		 * Super admin OR regular admin Dashboard.
		 */
		if ( bbapp_is_super_admin_page() ) {
			$tabs['components'] = array(
				'label'        => __( 'Components', 'buddyboss-app' ),
				'setting_page' => 'bbapp-components',
			);

			$tabs['settings'] = array(
				'label'        => __( 'Settings', 'buddyboss-app' ),
				'setting_page' => 'bbapp-settings',
			);

			$tabs['integrations'] = array(
				'label'        => __( 'Integrations', 'buddyboss-app' ),
				'setting_page' => 'bbapp-integrations',
			);
		}

		/**
		 * Only Regular Admin Dashboard
		 */
		if ( bbapp_is_admin_page() ) {
			$tabs['settings'] = array(
				'label'        => __( 'Settings', 'buddyboss-app' ),
				'setting_page' => 'bbapp-settings',
			);

			$tabs['integrations'] = array(
				'label'        => __( 'Integrations', 'buddyboss-app' ),
				'setting_page' => 'bbapp-integrations',
			);

			$tabs['appearance'] = array(
				'label'        => __( 'Branding', 'buddyboss-app' ),
				'setting_page' => 'bbapp-appearance',
			);

			$tabs['menus'] = array(
				'label'        => __( 'Menus', 'buddyboss-app' ),
				'setting_page' => 'bbapp-menus',
			);

			$tabs['translations'] = array(
				'label'        => __( 'Translations', 'buddyboss-app' ),
				'setting_page' => 'bbapp-translations',
			);

			$app_page_tab_link = admin_url( 'edit.php?post_type=app_page' );
			$tabs['app_pages'] = array(
				'label'        => __( 'App Pages', 'buddyboss-app' ),
				'setting_page' => 'bbapp-pages',
				'link'         => $app_page_tab_link,
			);

			if ( bbapp_is_active( 'push_notification' ) ) {
				$tabs['notification'] = array(
					'label'        => __( 'Push Notifications', 'buddyboss-app' ),
					'setting_page' => 'bbapp-notification',
				);
			}

			if ( bbapp_is_active( 'iap' ) ) {
				$tabs['iap'] = array(
					'label'        => __( 'In-App Purchases', 'buddyboss-app' ),
					'setting_page' => 'bbapp-iap',
				);
			}

			if ( bbapp_is_active( 'access_controls' ) ) {
				$tabs['access_controls'] = array(
					'label'        => __( 'Access Controls', 'buddyboss-app' ),
					'setting_page' => 'bbapp-access-controls',
				);
			}
		}

		/**
		 * Super admin OR regular admin Dashboard.
		 */
		if ( bbapp_is_super_admin_page() ) {
			if ( Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
				$tabs['import-export'] = array(
					'label'        => __( 'Tools', 'buddyboss-app' ),
					'setting_page' => 'bbapp-tools',
					'class'        => 'status',
				);
			}
		}

		/**
		 * Only Regular Admin Dashboard
		 */
		if ( bbapp_is_admin_page() ) {
			if ( Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
				$tabs['help'] = array(
					'label'        => __( 'Help', 'buddyboss-app' ),
					'setting_page' => 'bbapp-help',
					'class'        => 'help',
				);
			}
		}

		if ( Permissions::instance()->can_manage_app( get_current_user_id() ) && bbapp_is_super_admin_page() ) {
			$tabs['manage']       = array(
				'label'        => __( 'Connect', 'buddyboss-app' ),
				'setting_page' => 'bbapp-connect',
				'class'        => 'manage',
			);
			$tabs['configure']    = array(
				'label'        => __( 'Configure', 'buddyboss-app' ),
				'setting_page' => 'bbapp-configure',
				'class'        => 'manage',
			);
			$tabs['build']        = array(
				'label'        => __( 'Build', 'buddyboss-app' ),
				'setting_page' => 'bbapp-build',
				'class'        => 'manage',
			);
			$tabs['build-upload'] = array(
				'label'        => __( 'Upload', 'buddyboss-app' ),
				'setting_page' => 'bbapp-build-upload',
				'class'        => 'manage',
			);
		}

		/**
		 * Super admin dashboard on Network Activated or Wp Admin dashboard on Normal.
		 */
		if ( bbapp_is_admin_page() && ! bbapp_is_super_admin_page() ) {
			if ( isset( $tabs['help'] ) ) {
				unset( $tabs['help'] );
			}

			if ( isset( $tabs['integrations'] ) ) {
				unset( $tabs['integrations'] );
			}
		}

		if ( ! bbapp_is_admin_page() && bbapp_is_super_admin_page() ) {
			if ( isset( $tabs['appearance'] ) ) {
				unset( $tabs['appearance'] );
			}

			if ( isset( $tabs['translations'] ) ) {
				unset( $tabs['translations'] );
			}

			if ( isset( $tabs['notification'] ) ) {
				unset( $tabs['notification'] );
			}

			if ( isset( $tabs['app_pages'] ) ) {
				unset( $tabs['app_pages'] );
			}
		}

		if ( ! array_key_exists( 'bbapp-build-upload', $sub_menus ) || bbapp_is_sandbox_enabled() ) {
			unset( $tabs['build-upload'] );
		}

		?>
		<h2 class="nav-tab-wrapper bbapp-tabs">
			<?php
			foreach ( $tabs as $tab_key => $tab ) {
				if ( ! in_array( $tab['setting_page'], $tabs_groups, true ) ) {
					continue;
				}

				$active_class = '';

				if ( $tab['setting_page'] === $page_get && ! isset( $tab['can_active'] ) ) {
					$active_class = 'nav-tab-active';
				}

				$class    = ( isset( $tab['class'] ) ) ? $tab['class'] : '';
				$tab_link = bbapp_get_admin_url( 'admin.php?page=' . $tab['setting_page'] );

				if ( isset( $tab['link'] ) ) {
					$tab_link = $tab['link'];
				}

				$dashicon = ( isset( $tab['dashicon'] ) ) ? "<span class=\"dashicons {$tab['dashicon']}\"></span> " : '';
				?>

				<a href="<?php echo esc_attr( $tab_link ); ?>" class="nav-tab <?php echo esc_attr( $active_class ); ?> <?php echo esc_attr( $class ); ?>">
					<?php
					if ( ! empty( $dashicon ) ) {
						echo wp_kses_post( $dashicon );
					}
					echo esc_html( $tab['label'] );
					?>
				</a>
			<?php } ?>
		</h2>
		<?php
	}

	/**
	 * Tells if app is connected or not.
	 *
	 * @return bool
	 */
	public function is_current_app_connected() {
		$app = \BuddyBossApp\ManageApp::instance()->get_app();

		return ! empty( $app['verified'] );
	}

	/**
	 * Returns which page is currently in.
	 *
	 * @return string
	 */
	public function get_page_now() {
		global $pagenow;

		return $pagenow;
	}

	/**
	 * Release note BuddyBossApp Popup Screen.
	 */
	public function release_note_modal_content() {
		if ( 0 !== strpos( get_current_screen()->id, 'plugins' ) ) {
			return;
		}

		if ( $this->is_current_app_connected() ) {
			include bbapp()->plugin_dir . 'views/admin/release-note.php';
		}

		delete_option( '_bbapp_is_update' );
	}

	/**
	 * Add Settings link to plugins area.
	 *
	 * @param array  $links Array of links.
	 * @param string $file  Plugin file.
	 *
	 * @return array
	 */
	public function actions_links( $links, $file ) {
		// Return normal links if not BuddyPress.
		if ( plugin_basename( bbapp()->root_file ) !== $file ) {
			return $links;
		}

		$links['settings'] = '<a href="' . esc_url( bbapp_get_super_admin_url( 'admin.php?page=bbapp-settings' ) ) . '">' . __( 'Settings', 'buddyboss-app' ) . '</a>';

		if ( $this->is_current_app_connected() ) {
			$links['release_notes'] = '<a href="javascript:void(0);" id="bbapp-plugin-release-link">' . esc_html__( 'Release Notes', 'buddyboss-app' ) . '</a>';
		}

		return $links;
	}

	/**
	 * Admin notice dismiss.
	 */
	public function admin_notices_dismiss_js() {
		?>
		<script type="text/javascript">
			jQuery( document ).on( 'click', '.bbapp-is-dismissible .notice-dismiss', function( ) {
				const $notice = jQuery( this ).closest( '.notice' );
				const noticeId = $notice.data( 'noticeid' );

				jQuery.post(
					{
						url: ajaxurl,
						data: {
							action: 'bbapp_dismiss_notice',
							nonce: jQuery( '#bbapp-dismissible-nonce-' + noticeId ).val(),
							notice_id: noticeId,
						},
					},
				);
			} );
		</script>
		<?php
	}

	/**
	 * Make submenu highlight when on App page edit screen.
	 *
	 * @param string $submenu_file Submenu menu file.
	 *
	 * @since 1.7.1
	 * @return mixed|string
	 */
	public function bbapp_app_page_sub_menu( $submenu_file ) {
		global $pagenow;

		$post_type = ! empty( $_GET['post_type'] ) ? sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( 'post-new.php' === $pagenow && 'app_page' === $post_type ) {
			return 'edit.php?post_type=app_page';
		}

		return $submenu_file;
	}

	/**
	 * APP menu lists.
	 *
	 * @since 1.4.3
	 *
	 * @return array
	 */
	public function sub_menus() {
		$bbapp_sub_menus = array();

		// Settings Menu.
		$menu_label                          = __( 'Components', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-components'] = array(
			'bbapp-settings',
			$menu_label,
			$menu_label,
			'manage_options',
			'bbapp-components',
			array( $this, 'components_screen' ),
		);

		// Settings Menu.
		$menu_label                        = __( 'Settings', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-settings'] = array(
			'bbapp-settings',
			$menu_label,
			$menu_label,
			'manage_options',
			'bbapp-settings',
			array( $this, 'setting_screen' ),
		);

		// Integrations Menu.
		$menu_label                            = __( 'Integrations', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-integrations'] = array(
			'bbapp-settings',
			$menu_label,
			$menu_label,
			'manage_options',
			'bbapp-integrations',
			array( $this, 'integrations_screen' ),
		);

		// Appearance Menu.
		$menu_label                          = __( 'Branding', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-appearance'] = array(
			'bbapp-settings',
			$menu_label,
			$menu_label,
			'manage_options',
			'bbapp-appearance',
			array( $this, 'app_layout_screen' ),
		);

		// Menus Menu.
		$menu_label                     = __( 'Menus', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-menus'] = array(
			'bbapp-settings',
			$menu_label,
			$menu_label,
			'manage_options',
			'bbapp-menus',
			array( $this, 'app_menus_screen' ),
		);

		// Translations Menu.
		$menu_label                            = __( 'Translations', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-translations'] = array(
			'bbapp-settings',
			$menu_label,
			$menu_label,
			'manage_options',
			'bbapp-translations',
			array( $this, 'translations_screen' ),
		);

		// Tools Menu.
		$menu_label                     = __( 'Tools', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-tools'] = array(
			'bbapp-settings',
			$menu_label,
			$menu_label,
			'manage_options',
			'bbapp-tools',
			array( $this, 'bbapp_import_export_screen' ),
		);

		// Help menu.
		$menu_label                    = __( 'Help', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-help'] = array(
			'bbapp-settings',
			$menu_label,
			$menu_label,
			'manage_options',
			'bbapp-help',
			array( $this, 'help_screen' ),
		);

		// Separator.
		$bbapp_sub_menus['bbapp-first-sap'] = array(
			'bbapp-settings',
			'',
			'',
			'manage_options',
			'bbapp-plugin-separator-notice',
			'',
		);

		// App Pages Menu.
		$native_pages                   = 'edit.php?post_type=app_page';
		$menu_label                     = __( 'App Pages', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-pages'] = array(
			'bbapp-settings',
			$menu_label,
			$menu_label,
			'manage_options',
			$native_pages,
		);

		// Notifications Menu.
		if ( bbapp_is_active( 'push_notification' ) ) {
			$menu_label                            = __( 'Push Notifications', 'buddyboss-app' );
			$bbapp_sub_menus['bbapp-notification'] = array(
				'bbapp-settings',
				$menu_label,
				$menu_label,
				'manage_options',
				'bbapp-notification',
				array( $this, 'notification_screen' ),
			);
		}

		// InApp Purchases Menu.
		if ( bbapp_is_active( 'iap' ) ) {
			$menu_label                   = __( 'In-App Purchases', 'buddyboss-app' );
			$bbapp_sub_menus['bbapp-iap'] = array(
				'bbapp-settings',
				$menu_label,
				$menu_label,
				'manage_options',
				'bbapp-iap',
				array( $this, 'iap_screen' ),
			);
		}

		if ( bbapp_is_active( 'access_controls' ) ) {
			// Access Controls Menu.
			$menu_label                                = __( 'Access Controls', 'buddyboss-app' );
			$bbapp_sub_menus['bbapp-access-controls"'] = array(
				'bbapp-settings',
				$menu_label,
				$menu_label,
				'manage_options',
				'bbapp-access-controls',
				array( $this, 'access_control_screen' ),
			);
		}

		// Separator.
		$bbapp_sub_menus['bbapp-second-sap'] = array(
			'bbapp-settings',
			'',
			'',
			'manage_options',
			'bbapp-plugin-separator-notice',
			'',
		);

		// Connect Menu.
		$manage_app_label                 = __( 'Connect', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-connect'] = array(
			'bbapp-settings',
			$manage_app_label,
			$manage_app_label,
			'manage_options',
			'bbapp-connect',
			array( $this, 'manage_screen' ),
		);

		// Configure Menu.
		$manage_app_label                   = __( 'Configure', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-configure'] = array(
			'bbapp-settings',
			$manage_app_label,
			$manage_app_label,
			'manage_options',
			'bbapp-configure',
			array( $this, 'manage_configuration' ),
		);

		// Build Menu.
		$menu_label                     = __( 'Build', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-build'] = array(
			'bbapp-settings',
			$menu_label,
			$menu_label,
			'manage_options',
			'bbapp-build',
			array( $this, 'build_screen' ),
		);

		$menu_label                            = __( 'Upload', 'buddyboss-app' );
		$menu_title                            = __( 'Build Upload', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-build-upload'] = array(
			'bbapp-settings',
			$menu_title,
			$menu_label,
			'manage_options',
			'bbapp-build-upload',
			array( $this, 'upload_screen' ),
		);

		// Separator.
		$bbapp_sub_menus['bbapp-third-sap'] = array(
			'bbapp-settings',
			'',
			'',
			'manage_options',
			'bbapp-plugin-separator-notice',
			'',
		);

		// Publish menu.
		$menu_label                            = __( 'Publishing Service', 'buddyboss-app' );
		$bbapp_sub_menus['bbapp-publish-page'] = array(
			'bbapp-settings',
			$menu_label,
			$menu_label,
			'manage_options',
			'bbapp-publish',
			array( $this, 'publish_screen' ),
		);

		/**
		 * Only regular wp admin dashboard
		 */
		if ( ! bbapp_is_admin_page() ) {
			// Remove Branding & Translation tab in network admin.
			unset( $bbapp_sub_menus['bbapp-appearance'] );
			unset( $bbapp_sub_menus['bbapp-menus'] );
			unset( $bbapp_sub_menus['bbapp-translations'] );
			unset( $bbapp_sub_menus['bbapp-notification'] );
			unset( $bbapp_sub_menus['bbapp-pages'] );
			if ( ! bbapp_is_active( 'iap' ) && ! bbapp_is_active( 'access_controls' ) ) {
				unset( $bbapp_sub_menus['bbapp-second-sap'] );
			}
		} else {
			// Only render the BuddyBossApp Nav on Network Dashboard when it's a Network Enabled.
			if ( bbapp()->is_network_activated() ) {
				unset( $bbapp_sub_menus['bbapp-connect'] );
				unset( $bbapp_sub_menus['bbapp-configure'] );
				unset( $bbapp_sub_menus['bbapp-build'] );
				unset( $bbapp_sub_menus['bbapp-build-upload'] );
				unset( $bbapp_sub_menus['bbapp-publish-page'] );
				unset( $bbapp_sub_menus['bbapp-third-sap'] );
			}
		}

		/**
		 * Super admin dashboard. or Wp Admin dashboard.
		 */
		if ( bbapp_is_admin_page() && ! bbapp_is_super_admin_page() ) {
			unset( $bbapp_sub_menus['bbapp-integrations'] );

			if ( isset( $bbapp_sub_menus['bbapp-help'] ) ) {
				unset( $bbapp_sub_menus['bbapp-help'] );
			}

			unset( $bbapp_sub_menus['bbapp-second-sap'] );
			unset( $bbapp_sub_menus['bbapp-build'] );
			unset( $bbapp_sub_menus['bbapp-tools'] );
		}

		/**
		 * APP is sandbox mode.
		 */
		if ( bbapp_is_sandbox_enabled() ) {
			unset( $bbapp_sub_menus['bbapp-publish-page'] );
			unset( $bbapp_sub_menus['bbapp-build-upload'] );
			unset( $bbapp_sub_menus['bbapp-third-sap'] );
		}

		return $bbapp_sub_menus;
	}
}
