<?php
/**
 * Holds settings include functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\AppSettings;
use BuddyBossApp\AppStores\Apple;
use BuddyBossApp\Auth\Auth;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Performance\BuddyBossApp_Performance;

/**
 * Settings class.
 */
class Settings {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Settings form.
	 *
	 * @var $settings_form
	 */
	public $settings_form;

	/**
	 * Get the instance of the class.
	 *
	 * @return Settings
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Settings constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		// Update performance components setting on active / deactivate component.
		add_action( 'bbapp_components_activate', array( $this, 'update_performance_component_setting' ) );
		add_action( 'bbapp_components_deactivate', array( $this, 'update_performance_component_setting' ) );

		add_action( 'init', array( $this, 'setup_app_share' ) );
		add_action( 'admin_init', array( $this, 'load_init' ), 0 );
	}

	/**
	 * Set up the virtual link for the app share.
	 *
	 * @since 2.0.70
	 * @return void
	 */
	public function setup_app_share() {
		if ( ! AppSettings::instance()->get_setting_value( 'app_sharing.enabled' ) ) {
			return;
		}

		$request_uri = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( get_option( 'permalink_structure' ) ) {
			$page_slug = trim( wp_parse_url( $request_uri, PHP_URL_PATH ), '/' );
		} else {
			parse_str( wp_parse_url( $request_uri, PHP_URL_QUERY ), $params );
			$page_slug = ( isset( $params['page_id'] ) ? $params['page_id'] : false );
		}

		if ( 'dl' === $page_slug ) {
			// Enqueue any page specific styles and scripts here.
			include bbapp()->plugin_dir . 'views/app-installer/share.php';
            exit;
		}
	}

	/**
	 * Admin init method.
	 *
	 * @return false|void
	 */
	public function load_init() {
		/**
		 * Only load on bbapp-settings page.
		 */
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'admin.php' !== SetupAdmin::instance()->get_page_now() || 'bbapp-settings' !== $page_get ) {
			return false;
		}

		// If the current subtab is link redirect it.
		if ( bbapp_is_valid_url( $this->get_current_sub_tab() ) && ! isset( $_GET['setting'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			wp_safe_redirect( $this->get_current_sub_tab() );
			exit;
		}

		$this->settings_form = new FormProvider( 'settings' );

		// Holds app settings storage handle.
		$app_settings_provider = new SettingProvider( 'app_settings' );

		// Holds global settings storage handle.
		$app_global_settings_provider = new SettingProvider( 'app_global_settings' );

		// Register app settings storing & receiving.
		$app_global_settings_provider->save_setting_hook( array( $this, 'save_global_settings' ) );
		$app_global_settings_provider->get_setting_hook( array( $this, 'get_global_settings' ) );

		// Register app settings storing & receiving.
		$app_settings_provider->save_setting_hook( array( $this, 'save_settings' ) );
		$app_settings_provider->get_setting_hook( array( $this, 'get_settings' ) );

		// Link Provider In Form Provider.
		$this->settings_form->link_setting_provider( $app_global_settings_provider );
		$this->settings_form->link_setting_provider( $app_settings_provider );

		// Handle cache purge event if cache code exist.
		if ( bbapp_is_active( 'performance' ) && class_exists( '\BuddyBoss\Performance\Settings' ) ) {
			// Performance Settings Provider.
			$performance_settings_provider = new SettingProvider( 'performance_settings' );

			// Register Performance storing & receiving.
			$performance_settings_provider->save_setting_hook( array( $this, 'performance_save_settings' ) );
			$performance_settings_provider->get_setting_hook( array( $this, 'performance_get_settings' ) );
			$this->settings_form->link_setting_provider( $performance_settings_provider );

			add_action( 'admin_init', array( $this, 'handle_purge_cache' ) );
		}

		add_action( 'admin_init', array( $this->settings_form, 'save' ) );

		/**
		 * General.
		 */
		if ( 'general' === $this->get_current_sub_tab() ) {
			/**
			 * Registration
			 */
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-registration',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'app_registration_heading',
					array(
						'type'         => 'heading',
						'value'        => __( 'Registration', 'buddyboss-app' ),
						'desc'         => '',
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=121791' ),
						'icon'         => 'bb-icon-bf bb-icon-clipboard',
					)
				)
			);

			if ( bbapp()->is_network_activated() ) {
				/* translators: %s: url. */
				$signup_desc = sprintf( __( 'This feature requires enabling WordPress registration at <a href="%s">Network Settings</a>.', 'buddyboss-app' ), bbapp_get_super_admin_url( 'settings.php' ) );
			} else {
				/* translators: %s: url. */
				$signup_desc = sprintf( __( 'This feature requires enabling WordPress registration at <a href="%s">General Settings</a>.', 'buddyboss-app' ), bbapp_get_super_admin_url( 'options-general.php' ) );
			}

			$args = array(
				'label'          => __( 'Enable Registration', 'buddyboss-app' ),
				'label_checkbox' => __( 'Allow registering a new account from within the app', 'buddyboss-app' ),
				'desc'           => $signup_desc,
				'type'           => 'checkbox',
				'value'          => false,
			);

			if ( ! Auth::instance()->is_registration_available() ) {
				$args['disabled']       = true;
				$args['override_value'] = false;
			}

			$this->settings_form->add_field( $app_global_settings_provider->field( 'app_auth.enable_signup', $args ) );

			if ( defined( 'BP_PLATFORM_VERSION' ) ) {
				// Added new option for the registration section.
				$this->settings_form->add_field(
					$app_global_settings_provider->field(
						'app_auth.signup_form',
						array(
							'label'            => __( 'Registration Form', 'buddyboss-app' ),
							'type'             => 'select',
							'options'          => array(
								'buddyboss_registration' => __( 'BuddyBoss Registration', 'buddyboss-app' ),
								'wordpress_registration' => __( 'WordPress Registration', 'buddyboss-app' ),
							),
							'desc'             => function () {
								$desc = '';
								$desc .= sprintf( '<div id="buddyboss_registration" class="registration_description">%s</div>', __( 'Use the BuddyBoss Platform registration process in the app.', 'buddyboss-app' ) );
								$desc .= sprintf( '<div id="wordpress_registration" class="registration_description">%s</div>', __( 'Use the default WordPress registration process in the app.', 'buddyboss-app' ) );

								return $desc;
							},
							'disabled_options' => function () {
								return bbapp_is_lite_app() ? array( 'buddyboss_registration' ) : array();
							},
						)
					)
				);
			}

			if ( function_exists( 'bp_enable_site_registration' ) && Auth::instance()->is_registration_available() ) {
				$this->settings_form->add_field(
					$app_global_settings_provider->field(
						'app_auth.email_activation_body',
						array(
							'label'  => __( 'Activation Email Body', 'buddyboss-app' ),
							/* translators: %s: Key code. */
							'desc'   => sprintf( __( 'This text will be appended to the activation email sent from BuddyBoss Platform. Make sure to include the following tags: %s', 'buddyboss-app' ), '<code>{{KEY_CODE}}</code> <code>{{KEY_CODE_LINK}}</code>' ),
							'type'   => 'textarea',
							'value'  => '',
							'filter' => function ( $value ) {
								return $value;
							},
						)
					)
				);
			}

			/**
			 * Privacy
			 */
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-privacy',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'privacy_heading',
					array(
						'type'         => 'heading',
						'value'        => __( 'Privacy', 'buddyboss-app' ),
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=121745' ),
						'icon'         => 'bb-icon-bf bb-icon-lock-alt',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'private_app.enabled',
					array(
						'label'          => __( 'Private App', 'buddyboss-app' ),
						'label_checkbox' => __( 'Restrict app access to only logged-in members', 'buddyboss-app' ),
						'desc'           => function () {
							$desc = __( 'Require users to log into the app to view all content. When disabled, public app content will be visible while logged out, and users will be prompted to log in or register when trying to interact with content.', 'buddyboss-app' );
							if ( defined( 'BP_PLATFORM_VERSION' ) ) {
								$desc .= '<p class="bbapp-notice-box bbapp-notice-box--info">';
								if ( true === bbapp_is_private_app_enabled() ) {
									$desc .= sprintf(
										wp_kses_post(
										/* translators: Settings link. */
											__( 'To restrict access to your REST APIs for members who are not logged in, enable Private REST APIs in the %s in BuddyBoss Platform.', 'buddyboss-app' )
										),
										sprintf(
											'<a href="%s">' . esc_html__( 'Privacy settings', 'buddyboss-app' ) . '</a>',
											esc_url(
												add_query_arg(
													array( 'page' => 'bp-settings&tab=bp-general' ),
													admin_url( 'admin.php' )
												)
											)
										)
									);
								} else {
									$desc .= sprintf(
										wp_kses_post(
										/* translators: Settings link. */
											__( 'Once your app is private, you can restrict access to your REST APIs for logged-out members in the %s in BuddyBoss Platform.', 'buddyboss-app' )
										),
										sprintf(
											'<a href="%s">' . esc_html__( 'Privacy settings', 'buddyboss-app' ) . '</a>',
											esc_url(
												add_query_arg(
													array( 'page' => 'bp-settings&tab=bp-general' ),
													admin_url( 'admin.php' )
												)
											)
										)
									);
								}
								$desc .= '</p>';
							}

							return $desc;
						},
						'type'           => 'checkbox',
						'value'          => false,
					)
				)
			);

			if ( bbapp_is_active( 'access_controls' ) ) {
				ob_start();
				include bbapp()->plugin_dir . 'views/settings/private-app-modal.php';
				$html = ob_get_clean();

				$private_app_html = array(
					'label' => '',
					'value' => $html,
					'type'  => 'html',
				);

				$this->settings_form->add_field( $app_settings_provider->field( 'private_app.html', $private_app_html ) );
			}

			/**
			 * Website Password
			 */
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'browser_auth_heading',
					array(
						'type'         => 'heading',
						'value'        => __( 'Website Authentication', 'buddyboss-app' ),
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=121749' ),
						'icon'         => 'bb-icon-bf bb-icon-key',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'browser_auth.enabled',
					array(
						'label'          => __( 'Require Password', 'buddyboss-app' ),
						'label_checkbox' => __( 'Require password to view website', 'buddyboss-app' ),
						'desc'           => __( 'Website content will be blocked from visitors until entering a password. Only the Login page will be publicly accessible. Other web/HTTP authentication systems may block the app from receiving API data, so please disable them and use this option instead to test your app on a restricted website.', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'value'          => false,
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'browser_auth.password',
					array(
						'label'  => __( 'Website Password', 'buddyboss-app' ),
						'desc'   => __( 'The required password to view the website.', 'buddyboss-app' ),
						'type'   => 'password',
						'filter' => function ( $value ) {
							return $value;
						},
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'browser_auth.message',
					array(
						'label'  => __( 'Website Notice', 'buddyboss-app' ),
						'desc'   => __( 'Display this notice to site visitors who did not enter the password yet.', 'buddyboss-app' ),
						'type'   => 'textarea',
						'value'  => __( 'This website requires a password. Please enter your password below.', 'buddyboss-app' ),
						'filter' => function ( $value ) {
							return $value;
						},
					)
				)
			);
		}

		/**
		 * Feedback.
		 */
		if ( 'feedback' === $this->get_current_sub_tab() ) {

			/**
			 * Send us Feedback
			 */
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-feedback',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'app_feedback_heading',
					array(
						'type'         => 'heading',
						'value'        => __( 'Send us Feedback', 'buddyboss-app' ),
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=121753' ),
						'icon'         => 'bb-icon-bf bb-icon-thumbs-up',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'support_email.enabled',
					array(
						'label'          => __( 'Send us Feedback', 'buddyboss-app' ),
						'label_checkbox' => __( 'Enable "Send us Feedback" menu in app', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'desc'           => __( 'In the app interface, users can go into "Settings" and tap "Send us Feedback" to submit feedback.', 'buddyboss-app' ),
						'value'          => true,
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'support_email.email',
					array(
						'label'  => __( 'Feedback Email', 'buddyboss-app' ),
						'type'   => 'text',
						'desc'   => __( 'Enter the email address to send feedback to. If left empty the <strong>Administration Email Address</strong> from your WordPress settings will be used instead.', 'buddyboss-app' ),
						'value'  => get_option( 'admin_email' ),
						'filter' => function ( $value ) {
							if ( ! is_email( $value ) ) {
								return '';
							}

							return $value;
						},
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'support_email.subject',
					array(
						'label' => __( 'Email Subject', 'buddyboss-app' ),
						'type'  => 'text',
						'desc'  => __( 'Enter the subject line for all of the feedback emails.', 'buddyboss-app' ),
						/* translators: %s: Blog name. */
						'value' => sprintf( __( '[%s] Send us Feedback', 'buddyboss-app' ), get_option( 'blogname' ) ),
					)
				)
			);

			if ( ! defined( 'BP_PLATFORM_VERSION' ) ) {
				/**
				 * Delete account
				 */
				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-delete-account',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_delete_account_heading',
						array(
							'type'         => 'heading',
							'value'        => __( 'Delete Account', 'buddyboss-app' ),
							'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=127419' ),
							'icon'         => 'bb-icon-bf bb-icon-user-cancel',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'delete_account.admin_email',
						array(
							'label'  => __( 'Delete Request Email', 'buddyboss-app' ),
							'type'   => 'text',
							'desc'   => __( 'Enter the email address to send delete account requests to. If left empty the <strong>Administration Email Address</strong> from your WordPress settings will be used instead.',
								'buddyboss-app' ),
							'value'  => get_option( 'admin_email' ),
							'filter' => function ( $value ) {
								if ( ! is_email( $value ) ) {
									return '';
								}

								return $value;
							},
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'delete_account.subject',
						array(
							'label' => __( 'Email Subject', 'buddyboss-app' ),
							'type'  => 'text',
							'desc'  => __( 'Enter the subject line for all of the account deletion request emails.', 'buddyboss-app' ),
							/* translators: %s: Blog name. */
							'value' => sprintf( __( '[%s] Account Deletion Request', 'buddyboss-app' ), get_option( 'blogname' ) ),
						)
					)
				);
			}

			/**
			 * Report a Bug
			 */
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-report_bug',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'app_report_bug_heading',
					array(
						'type'         => 'heading',
						'value'        => __( 'Report a Bug', 'buddyboss-app' ),
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=121754' ),
						'icon'         => 'bb-icon-bf bb-icon-bug',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'bugs_email.enabled',
					array(
						'label'          => __( 'Report a Bug', 'buddyboss-app' ),
						'label_checkbox' => __( 'Enable "Report a Bug" menu in app', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'desc'           => __( 'In the app interface, users can go into "Settings" and tap "Report a Bug" to submit a bug report.', 'buddyboss-app' ),
						'value'          => false,
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'bugs_email.email',
					array(
						'label'  => __( 'Bug Report Email', 'buddyboss-app' ),
						'type'   => 'text',
						'desc'   => __( 'Enter the email address to send bug reports to. If left empty the <strong>Administration Email Address</strong> from your WordPress settings will be used instead.', 'buddyboss-app' ),
						'value'  => get_option( 'admin_email' ),
						'filter' => function ( $value ) {
							if ( ! is_email( $value ) ) {
								return '';
							}

							return $value;
						},
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'bugs_email.subject',
					array(
						'label' => __( 'Email Subject', 'buddyboss-app' ),
						'type'  => 'text',
						'desc'  => __( 'Enter the subject line for all of the bug report emails.', 'buddyboss-app' ),
						/* translators: %s: Blog name. */
						'value' => sprintf( esc_html__( '[%s] Report a Bug', 'buddyboss-app' ), get_option( 'blogname' ) ),
					)
				)
			);

			/**
			 * App Rating
			 */
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-rating',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'app_rating_heading',
					array(
						'type'         => 'heading',
						'value'        => __( 'Rate this App', 'buddyboss-app' ),
						'desc'         => __( 'When activated, members can go into "Settings" and tap "Rate this App" to be sent to the Apple App Store or Google Play Store to leave a rating. Disabling this option will <strong><em>not</em></strong> turn off ratings in these stores. It is recommended you only activate Rate this App after your app is published as the option will link to your store listings.', 'buddyboss-app' ),
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=122283' ),
						'icon'         => 'bb-icon-bf bb-icon-star-half',
					)
				)
			);

			$ios_app_id = Apple::instance()->get_ios_store_app_id();

			if ( ! empty( $ios_app_id ) ) {
				$app_rating_ios_enabled = array(
					'label'          => __( 'Enable on iOS', 'buddyboss-app' ),
					'label_checkbox' => __( 'Enable "Rate this App" on iOS apps', 'buddyboss-app' ),
					'type'           => 'checkbox',
					'value'          => false,
				);
			} else {
				$app_rating_ios_enabled = array(
					'label' => __( 'Enable on iOS', 'buddyboss-app' ),
					/* translators: %s: IOS settings link. */
					'value' => sprintf( __( 'Enter an App ID in the <a href="%s">iOS settings</a> to enable on your iOS app', 'buddyboss-app' ), bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=ios' ) ),
					'type'  => 'html',
				);
			}

			$this->settings_form->add_field( $app_settings_provider->field( 'app_rating.ios.enabled', $app_rating_ios_enabled ) );

			$android_application_id = Configure::instance()->option( 'publish.android.namespace', false );

			if ( ! empty( $android_application_id ) ) {
				$app_rating_android_enabled = array(
					'label'          => __( 'Enable on Android', 'buddyboss-app' ),
					'label_checkbox' => __( 'Enable "Rate this App" on Android apps', 'buddyboss-app' ),
					'type'           => 'checkbox',
					'value'          => false,
				);
			} else {
				$app_rating_android_enabled = array(
					'label' => __( 'Enable on Android', 'buddyboss-app' ),
					/* translators: %s: Android settings link. */
					'value' => sprintf( __( 'Enter an Application ID in the <a href="%s">Android settings</a> to enable on your Android app', 'buddyboss-app' ), bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=android' ) ),
					'type'  => 'html',
				);
			}

			$this->settings_form->add_field( $app_settings_provider->field( 'app_rating.android.enabled', $app_rating_android_enabled ) );

			/**
			 * App Sharing.
			 */
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-share',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'app_sharing_heading',
					array(
						'type'         => 'heading',
						'value'        => __( 'Share App Link', 'buddyboss-app' ),
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=127279' ),
						'icon'         => 'bb-icon-bf bb-icon-share-square',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'app_sharing.enabled',
					array(
						'label'          => __( 'App Sharing', 'buddyboss-app' ),
						'label_checkbox' => __( 'Enable app sharing button', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'desc'           => __( 'App sharing button can be found under invites and as a link to use in your menus and an app page quick link.', 'buddyboss-app' ),
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'app_sharing.sharing_link',
					array(
						'label' => __( 'App Sharing Link', 'buddyboss-app' ),
						'type'  => 'html',
						'desc'  => __( 'Share this link with your friends to download the App.', 'buddyboss-app' ),
						'value' => function () {
							?>
                            <span>
                                <label>
                                    <div class="bbapp-server-notifications-main-wrp">
                                        <span class="bbapp-server-notifcation-inr-wrp">
                                            <span class="bbapp-sub-elements"><a id="bbapp_page_app_share_url"><?php echo esc_html( home_url( 'dl' ) ) ?></a></span>
                                            <span class="bbapp-sub-tooltip">
                                                <button type="button" onclick="window.bbappCore.core.bbapp_copy_paste_url('bbapp_page_app_share_url')">
                                                    <span class="bbapp-tooltip-text" id="bbapp-span-tooltip-bbapp_page_app_share_url"></span>
                                                    <i class="bb-icon-f bb-icon-clipboard"></i>
                                                </button>
                                            </span>
                                        </span>
                                    </div>
                                </label>
                            </span>
							<?php
						},
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'app_sharing.title',
					array(
						'label' => __( 'Preview Title', 'buddyboss-app' ),
						'type'  => 'text',
					)
				)
			);

			$this->settings_form->add_field(
				$app_global_settings_provider->field(
					'app_sharing.description',
					array(
						'label' => __( 'Preview Description', 'buddyboss-app' ),
						'type'  => 'textarea',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'app_sharing.image',
					array(
						'label'        => __( 'Preview Image', 'buddyboss-app' ),
						'desc'         => __( 'Upload an image to be default preview image shown in the link preview when shared. The recommended size is 300px by 200px.', 'buddyboss-app' ),
						'type'         => 'image-picker',
						'show_preview' => true,
						'button_label' => __( 'Upload', 'buddyboss-app' ),
					)
				)
			);

			if ( ! empty( $ios_app_id ) ) {
				$ios_app_sharing = array(
					'label'          => __( 'Enable on iOS', 'buddyboss-app' ),
					'label_checkbox' => __( 'Enable "App Sharing" on iOS devices', 'buddyboss-app' ),
					'type'           => 'checkbox',
					'value'          => false,
				);
			} else {
				$ios_app_sharing = array(
					'label' => __( 'Enable on iOS', 'buddyboss-app' ),
					/* translators: %s: IOS settings link. */
					'value' => sprintf( __( 'Enter an App ID in the <a href="%s">iOS settings</a> to enable on iOS app sharing', 'buddyboss-app' ), bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=ios' ) ),
					'type'  => 'html',
				);
			}

			$this->settings_form->add_field( $app_settings_provider->field( 'app_sharing.ios.enabled', $ios_app_sharing ) );

			if ( ! empty( $android_application_id ) ) {
				$android_app_sharing = array(
					'label'          => __( 'Enable on Android', 'buddyboss-app' ),
					'label_checkbox' => __( 'Enable "App Sharing" on Android devices', 'buddyboss-app' ),
					'type'           => 'checkbox',
					'value'          => false,
				);
			} else {
				$android_app_sharing = array(
					'label' => __( 'Enable on Android', 'buddyboss-app' ),
					/* translators: %s: Android settings link. */
					'value' => sprintf( __( 'Enter an Application ID in the <a href="%s">Android settings</a> to enable on Android app sharing', 'buddyboss-app' ), bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=android' ) ),
					'type'  => 'html',
				);
			}

			$this->settings_form->add_field( $app_settings_provider->field( 'app_sharing.android.enabled', $android_app_sharing ) );

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'app_sharing.web_fallback',
					array(
						'label' => __( 'Web Fallback URL', 'buddyboss-app' ),
						'type'  => 'text',
						'desc'  => __( 'This will be the url user will be redirected to if they click on your shared app link with a non iOS or Android device.', 'buddyboss-app' ),
					)
				)
			);
		}

		/**
		 * Smart Banner
		 */
		if ( 'smart_banner' === $this->get_current_sub_tab() ) {
			/**
			 * Smart Banner
			 */
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-banner smart-banner-settings_form',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'smart_banner_heading',
					array(
						'type'         => 'heading',
						'value'        => __( 'Smart Banner', 'buddyboss-app' ),
						'desc'         => __( 'When viewing this website from a mobile browser, a <a href="https://developer.apple.com/library/archive/documentation/AppleApplications/Reference/SafariWebContent/PromotingAppswithAppBanners/PromotingAppswithAppBanners.html" target="_blank">smart banner</a> can be displayed to promote your app to your members. It is recommended you only activate Smart Banner after your app is published as the banner will link to your store listings.', 'buddyboss-app' ),
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=122282' ),
						'icon'         => 'bb-icon-bf bb-icon-maximize',
					)
				)
			);

			$ios_app_id = Apple::instance()->get_ios_store_app_id();

			if ( ! empty( $ios_app_id ) ) {
				$app_smartbanner_ios_enabled = array(
					'label'          => __( 'Enable on iOS', 'buddyboss-app' ),
					'label_checkbox' => __( 'Enable smart banner on iOS web browsers.', 'buddyboss-app' ),
					'desc'           => sprintf( __( 'When <a href="%s">Universal Links</a> feature is enabled, Safari browsers will always display the smart banner regardless of this settings.' ), bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=ios' ) ),
					'type'           => 'checkbox',
					'value'          => false,
				);
			} else {
				$app_smartbanner_ios_enabled = array(
					'label' => __( 'Enable on iOS', 'buddyboss-app' ),
					/* translators: %s: IOS settings link. */
					'value' => sprintf( __( 'Enter an App ID in the <a href="%s">iOS settings</a> to enable on iOS mobile browsers', 'buddyboss-app' ), bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=ios' ) ),
					'type'  => 'html',
				);
			}

			$this->settings_form->add_field( $app_settings_provider->field( 'app_smartbanner.ios.enabled', $app_smartbanner_ios_enabled ) );

			$android_application_id = Configure::instance()->option( 'publish.android.namespace', false );

			if ( ! empty( $android_application_id ) ) {
				$app_smartbanner_android_enabled = array(
					'label'          => __( 'Enable on Android', 'buddyboss-app' ),
					'label_checkbox' => __( 'Enable smart banner on Android web browsers', 'buddyboss-app' ),
					'type'           => 'checkbox',
					'value'          => false,
				);
			} else {
				$app_smartbanner_android_enabled = array(
					'label' => __( 'Enable on Android', 'buddyboss-app' ),
					/* translators: %s: Android settings link. */
					'value' => sprintf( __( 'Enter an Application ID in the <a href="%s">Android settings</a> to enable on Android mobile browsers', 'buddyboss-app' ), bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=android' ) ),
					'type'  => 'html',
				);
			}

			$this->settings_form->add_field( $app_settings_provider->field( 'app_smartbanner.android.enabled', $app_smartbanner_android_enabled ) );

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'app_smartbanner.logged_in_only',
					array(
						'label'          => __( 'Only to logged in users', 'buddyboss-app' ),
						'label_checkbox' => __( 'Require users to be logged into your website to see the smart banner', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'value'          => false,
					)
				)
			);
		}

		/**
		 * Bookmarking
		 */
		if ( 'book_posts' === $this->get_current_sub_tab() ) {
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-banner bookmarking-settings_form',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'bookmarking_heading',
					array(
						'type'         => 'heading',
						'value'        => __( 'Bookmarking', 'buddyboss-app' ),
						'desc'         => '',
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=126192' ),
						'icon'         => 'bb-icon-bf bb-icon-bookmark',
					)
				)
			);

			$desc_html  = __( 'Once enabled, members will be able to save blog posts to their bookmarks. You will need to add the', 'buddyboss-app' );
			$desc_html .= sprintf( ' <strong>%1$s</strong> ', __( 'Bookmarks', 'buddyboss-app' ) );
			$desc_html .= __( 'core app screen to your app\'s', 'buddyboss-app' );
			$desc_html .= sprintf( ' <a href="%1$s">%2$s</a> ', esc_url( bbapp_get_super_admin_url( 'admin.php?page=bbapp-menus' ) ), __( 'Tab Bar', 'buddyboss-app' ) );
			$desc_html .= __( 'or', 'buddyboss-app' );
			$desc_html .= sprintf( ' <a href="%1$s">%2$s</a>, ', esc_url( bbapp_get_super_admin_url( 'admin.php?page=bbapp-menus&setting=more' ) ), __( 'More Menu', 'buddyboss-app' ) );
			$desc_html .= __( 'so that your members can access their bookmarked posts.', 'buddyboss-app' );
			$this->settings_form->add_field(
				$app_settings_provider->field(
					'post_bookmark_enable',
					array(
						'label'          => __( 'Enable Bookmarks', 'buddyboss-app' ),
						'label_checkbox' => __( 'Allow members to bookmark blog posts', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'desc'           => $desc_html,
						'value'          => false,
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-banner post-sharing-settings-form',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'post_sharing_heading',
					array(
						'type'         => 'heading',
						'value'        => __( 'Sharing', 'buddyboss-app' ),
						'desc'         => '',
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=127183' ),
						'icon'         => 'bb-icon-bf bb-icon-share-dots',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'post_share_enable',
					array(
						'label'          => __( 'Enable Sharing', 'buddyboss-app' ),
						'label_checkbox' => __( 'Allow members to share posts', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'desc'           => __( 'Allow members to share posts.', 'buddyboss-app' ),
						'value'          => false,
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-banner post-related-post-settings-form',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'related_post_heading',
					array(
						'type'         => 'heading',
						'value'        => __( 'Related Posts', 'buddyboss-app' ),
						'desc'         => '',
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=127183' ),
						'icon'         => 'bb-icon-link bb-icon-bf',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'related_post_enable',
					array(
						'label'          => __( 'Enable Related Posts', 'buddyboss-app' ),
						'label_checkbox' => __( 'Show Related Posts', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'desc'           => __( 'Enabling this option will show three related posts at the bottom of each blog either within the same categories or if categories are not used then the latest five posts will be displayed.', 'buddyboss-app' ),
						'value'          => false,
					)
				)
			);
		}

		/**
		 * Push Notification
		 */
		if ( 'push_notifications' === $this->get_current_sub_tab() ) {
			Notification::instance()->load_settings();
		}

		/**
		 * InApp Purchase
		 */
		if ( 'inapp_purchases' === $this->get_current_sub_tab() ) {
			InAppPurchases::instance()->load_settings();
		}

		/**
		 * Advanced.
		 */
		if ( 'advanced' === $this->get_current_sub_tab() && ! defined( 'BP_PLATFORM_VERSION' ) ) {
			Advanced::instance()->load_settings();
		}

		if ( 'compatibility' === $this->get_current_sub_tab() ) {
			/**
			 * Compatibility  Settings
			 */
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-compatibility',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'compatibility.custom_tab.header',
					array(
						'type'         => 'heading',
						'value'        => __( 'Custom Tabs', 'buddyboss-app' ),
						'desc'         => '',
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=124848' ),
						'icon'         => 'bb-icon-bf bb-icon-server',
					)
				)
			);

			// Lock App if user has not purchased a product.
			$this->settings_form->add_field(
				$app_settings_provider->field(
					'compatibility.custom_tab.profile',
					array(
						'label'          => __( 'Custom Profile Tabs', 'buddyboss-app' ),
						'label_checkbox' => __( 'Enable custom profile tabs in BuddyBoss App', 'buddyboss-app' ),
						'desc'           => __( 'Show profile tabs added by unsupported third-party plugins or code in your app. If a tab doesn\'t have an associated native screen, it will load in a web fallback.', 'buddyboss-app' ),
						'type'           => 'checkbox',
					)
				)
			);

			// Purchase before registration.
			$this->settings_form->add_field(
				$app_settings_provider->field(
					'compatibility.custom_tab.group',
					array(
						'label'          => __( 'Custom Group Tabs', 'buddyboss-app' ),
						'label_checkbox' => __( 'Enable custom group tabs in BuddyBoss App.', 'buddyboss-app' ),
						'desc'           => __( 'Show groups tabs added by unsupported third-party plugins or code in your app. If a tab doesn\'t have an associated native screen, it will load in a web fallback.', 'buddyboss-app' ),
						'type'           => 'checkbox',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-admin-link-compatibility',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'compatibility.link.header',
					array(
						'type'         => 'heading',
						'value'        => __( 'Link Handling', 'buddyboss-app' ),
						'desc'         => '',
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=124902' ),
						'icon'         => 'bb-icon-bf bb-icon-link',
					)
				)
			);

			// Open links in.
			$this->settings_form->add_field(
				$app_settings_provider->field(
					'compatibility.open_link_in',
					array(
						'label'   => __( 'Open Links in', 'buddyboss-app' ),
						'desc'    => __( 'Select whether external links should open in the in-app-browser, an external browser, or left to the member to decide.', 'buddyboss-app' ),
						'type'    => 'select',
						'options' => array(
							'in_app_browser'   => __( 'In-App Browser', 'buddyboss-app' ),
							'external_browser' => __( 'External Browser', 'buddyboss-app' ),
							'ask_member'       => __( 'Ask Member', 'buddyboss-app' ),
						),
					)
				)
			);

			/**
			 * Cleartext Traffic HTTP requests.
			 */
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'security_heading',
					array(
						'type'  => 'heading',
						'value' => __( 'Security', 'buddyboss-app' ),
						'icon'  => 'bb-icon-bf bb-icon-shield-half',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'cleartext_traffic.enabled',
					array(
						'label'          => __( 'HTTP Requests', 'buddyboss-app' ),
						'label_checkbox' => __( 'Allow HTTP requests from the App', 'buddyboss-app' ),
						'desc'           => __( 'By default, the app will only allow secure (HTTPS) requests through the API and in web fallbacks. Enable this option to allow insecure (HTTP) requests as well. You should only enable if you have a specific reason to do so.', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'value'          => false,
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'cleartext_traffic.information',
					array(
						'label' => '&nbsp;',
						'type'  => 'html',
						/* translators: %s: Request build link. */
						'value' => '<div class="bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( __( "<a href='%s'>Changes will require new iOS and Android builds.</a>", 'buddyboss-app' ), bbapp_get_super_admin_url( 'admin.php?page=bbapp-build&setting=request-build' ) ) . '<p></div>',
					)
				)
			);
		}

		/**
		 * Cache Support - Settings
		 */
		if ( bbapp_is_active( 'performance' ) && 'cache_support' === $this->get_current_sub_tab() && class_exists( '\BuddyBoss\Performance\Settings' ) ) {

			$purge_nonce = \BuddyBoss\Performance\Settings::get_purge_nonce();

			// Introduction.
			$this->settings_form->add_field(
				$performance_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-caching-card',
					)
				)
			);

			$this->settings_form->add_field(
				$performance_settings_provider->field(
					'cache_support_heading_introduction',
					array(
						'type'         => 'heading',
						'value'        => __( 'API Caching', 'buddyboss-app' ),
						'desc'         => __( 'API Caching improves the response time for API requests. Third party plugins or custom code that interacts with API data  may conflict with the cache. This can be resolved by disabling the cache for that component, or by adding your own <a href="https://www.buddyboss.com/resources/dev-docs/app-development/extending-the-buddyboss-app-plugin/extending-api-caching-to-support-new-api-endpoints" target="_blank">custom code</a> for supporting additional API data in the cache.', 'buddyboss-app' ),
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=121851' ),
						'icon'         => 'bb-icon-bf bb-icon-database',
					)
				)
			);

			// BuddyBoss App.
			$this->settings_form->add_field(
				$performance_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-caching-card',
					)
				)
			);

			$this->settings_form->add_field(
				$performance_settings_provider->field(
					'cache_support_heading_bbapp',
					array(
						'type'  => 'heading',
						'value' => __( 'BuddyBoss App', 'buddyboss-app' ),
						'desc'  => '<a class="button button-primary purge-all-btn" href="' . bbapp_get_super_admin_url( 'admin.php?page=bbapp-settings&setting=cache_support&group=bbapp&component=all&nonce=' . $purge_nonce ) . '">' . __( 'Purge All', 'buddyboss-app' ) . '</a>',
						'icon'  => 'bb-icon-bf bb-icon-database',
					)
				)
			);

			$this->settings_form->add_field(
				$performance_settings_provider->field(
					'cache_app_core',
					array(
						'label'          => __( 'App Core', 'buddyboss-app' ),
						'label_checkbox' => __( 'Cache App Core', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'value'          => true,
						'purge_url'      => admin_url( 'admin.php?page=bbapp-settings&setting=cache_support&group=bbapp&component=app_core&nonce=' . $purge_nonce ),
					)
				)
			);

			// WordPress.
			$this->settings_form->add_field(
				$performance_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card bbapp-caching-card',
					)
				)
			);

			$this->settings_form->add_field(
				$performance_settings_provider->field(
					'cache_support_heading_wordpress',
					array(
						'type'  => 'heading',
						'value' => __( 'Pages and Posts', 'buddyboss-app' ),
						'desc'  => '<a class="button button-primary purge-all-btn" href="' . bbapp_get_super_admin_url( 'admin.php?page=bbapp-settings&setting=cache_support&group=wordpress&component=all&nonce=' . $purge_nonce ) . '">' . __( 'Purge All', 'buddyboss-app' ) . '</a>',
						'icon'  => 'bb-icon-bf bb-icon-database',
					)
				)
			);

			$this->settings_form->add_field(
				$performance_settings_provider->field(
					'cache_app_page',
					array(
						'label'          => __( 'App Pages', 'buddyboss-app' ),
						'label_checkbox' => __( 'Cache App Pages', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'value'          => true,
						'purge_url'      => admin_url( 'admin.php?page=bbapp-settings&setting=cache_support&group=wordpress&component=app_page&nonce=' . $purge_nonce ),
					)
				)
			);

			$this->settings_form->add_field(
				$performance_settings_provider->field(
					'cache_blog_post',
					array(
						'label'          => __( 'Blog Posts', 'buddyboss-app' ),
						'label_checkbox' => __( 'Cache Blog Posts', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'value'          => true,
						'purge_url'      => admin_url( 'admin.php?page=bbapp-settings&setting=cache_support&group=wordpress&component=blog_post&nonce=' . $purge_nonce ),
					)
				)
			);

			$this->settings_form->add_field(
				$performance_settings_provider->field(
					'cache_post_comment',
					array(
						'label'          => __( 'Blog Comments', 'buddyboss-app' ),
						'label_checkbox' => __( 'Cache Blog Comments', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'value'          => true,
						'purge_url'      => admin_url( 'admin.php?page=bbapp-settings&setting=cache_support&group=wordpress&component=post_comment&nonce=' . $purge_nonce ),
					)
				)
			);

			$this->settings_form->add_field(
				$performance_settings_provider->field(
					'cache_categories',
					array(
						'label'          => __( 'Blog Categories', 'buddyboss-app' ),
						'label_checkbox' => __( 'Cache Blog Categories', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'value'          => true,
						'purge_url'      => admin_url( 'admin.php?page=bbapp-settings&setting=cache_support&group=wordpress&component=categories&nonce=' . $purge_nonce ),
					)
				)
			);

			$this->settings_form->add_field(
				$performance_settings_provider->field(
					'cache_post_bookmarks',
					array(
						'label'          => __( 'Post Bookmarks', 'buddyboss-app' ),
						'label_checkbox' => __( 'Cache Post Bookmarks', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'value'          => true,
						'purge_url'      => admin_url( 'admin.php?page=bbapp-settings&setting=cache_support&group=wordpress&component=bb-bookmarks&nonce=' . $purge_nonce ),
					)
				)
			);

			if ( class_exists( 'BuddyBossApp\Performance\BuddyBossApp_Performance' ) ) {
				$custom_supports = BuddyBossApp_Performance::get_custom_setting();

				if ( ! empty( $custom_supports ) ) {
					foreach ( $custom_supports as $custom_support ) {
						$this->settings_form->add_field(
							$performance_settings_provider->field(
								'cache_' . $custom_support['slug'],
								array(
									'label'          => $custom_support['label'],
									'label_checkbox' => $custom_support['label_checkbox'],
									'type'           => 'checkbox',
									'value'          => isset( $custom_support['default_enabled'] ) ? $custom_support['default_enabled'] : true,
									'purge_url'      => admin_url( 'admin.php?page=bbapp-settings&setting=cache_support&group=wordpress&component=' . $custom_support['slug'] . '&nonce=' . $purge_nonce ),
								)
							)
						);
					}
				}
			}

			// BuddyBoss Platform.
			if ( function_exists( 'buddypress' ) && isset( buddypress()->buddyboss ) && buddypress()->buddyboss ) {
				$bp_platform_api_setting = \BuddyBoss\Performance\Settings::get_performance_components();

				if ( ! empty( $bp_platform_api_setting ) ) {
					$this->settings_form->add_field(
						$performance_settings_provider->field(
							false,
							array(
								'type'    => 'table-open',
								'classes' => 'buddyboss-app-admin-card bbapp-caching-card',
							)
						)
					);

					foreach ( $bp_platform_api_setting as $group_setting_key => $group_setting ) {
						$this->settings_form->add_field(
							$performance_settings_provider->field(
								'cache_support_heading_' . $group_setting_key,
								array(
									'type'  => 'heading',
									'value' => $group_setting['title'],
									'desc'  => '<a class="button button-primary purge-all-btn" href="' . $group_setting['purge_url'] . '">' . __( 'Purge All', 'buddyboss-app' ) . '</a>',
									'icon'  => 'bb-icon-bf bb-icon-database',
								)
							)
						);

						if ( ! empty( $group_setting['settings'] ) && is_array( $group_setting['settings'] ) ) {
							foreach ( $group_setting['settings'] as $performance_setting_key => $performance_setting ) {
								$field_setting = array();

								if ( isset( $performance_setting['label'] ) ) {
									$field_setting['label'] = $performance_setting['label'];
								}

								if ( isset( $performance_setting['label_checkbox'] ) ) {
									$field_setting['label_checkbox'] = $performance_setting['label_checkbox'];
								}

								if ( isset( $performance_setting['desc'] ) ) {
									$field_setting['desc'] = $performance_setting['desc'];
								}

								if ( isset( $performance_setting['type'] ) ) {
									$field_setting['type'] = $performance_setting['type'];
								}

								if ( isset( $performance_setting['value'] ) ) {
									$field_setting['value'] = $performance_setting['value'];
								}

								if ( isset( $performance_setting['purge_url'] ) ) {
									$field_setting['purge_url'] = $performance_setting['purge_url'];
								}

								$this->settings_form->add_field( $performance_settings_provider->field( $performance_setting_key, $field_setting ) );
							}
						}
					}
				}
			}

			// LearnDash.
			if ( bbapp_is_learndash_enabled() ) {

				$this->settings_form->add_field(
					$performance_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-caching-card',
						)
					)
				);

				$this->settings_form->add_field(
					$performance_settings_provider->field(
						'cache_support_heading_learndash',
						array(
							'type'  => 'heading',
							'value' => __( 'LearnDash', 'buddyboss-app' ),
							'icon'  => 'bb-icon-bf bb-icon-database',
						)
					)
				);

				$this->settings_form->add_field(
					$performance_settings_provider->field(
						'cache_ld',
						array(
							'label'          => __( 'LearnDash', 'buddyboss-app' ),
							'label_checkbox' => __( 'Cache LearnDash Courses', 'buddyboss-app' ),
							'type'           => 'checkbox',
							'value'          => true,
							'purge_url'      => admin_url( 'admin.php?page=bbapp-settings&setting=cache_support&group=learndash&component=sfwd-courses,sfwd-lessons,sfwd-topic,sfwd-quiz&nonce=' . $purge_nonce ),
						)
					)
				);
			}
		} // API Cache Settings End.

		if ( 'api_cdn' === $this->get_current_sub_tab() ) {
			// Rest API CDN.
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'cdn_support_heading_rest_api',
					array(
						'type'         => 'heading',
						'value'        => __( 'API &ndash; Content Delivery Network', 'buddyboss-app' ),
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=121834' ),
						'icon'         => 'bb-icon-bf bb-icon-globe-layers',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'cdn.enabled',
					array(
						'label'          => __( 'CDN', 'buddyboss-app' ),
						'label_checkbox' => __( 'Enable API CDN', 'buddyboss-app' ),
						'desc'           => __( 'Using a CDN (Content Delivery Network) improves the response time for downloading static assets, such as images, files and videos. Most WordPress CDN plugins only work on the website, and do not work within API requests. Enter a CDN URL below to be used for pulling CDN assets within API requests.', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'value'          => false,
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'cdn.error',
					array(
						'label' => '&nbsp;',
						'type'  => 'html',
						'value' => '<div class="bbapp-notice-box bbapp-notice-box--error">' . __( 'Failed to load assets using CDN URL provided. Please enter a valid CDN URL and try again.', 'buddyboss-app' ) . '</div>',
						'class' => function () {
							global $bbapp_cdn_error;

							if ( empty( $bbapp_cdn_error ) ) {
								return 'hidden';
							}

							return '';
						},
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'cdn.url',
					array(
						'label' => __( 'CDN URL', 'buddyboss-app' ),
						'desc'  => __( 'Enter the URL to your CDN, with https:// and without a trailing slash at the end.', 'buddyboss-app' ),
						'type'  => 'text',
						'value' => get_home_url(),
						'class' => function () {
							$global_settings = $this->get_global_settings();
							$class           = 'api_cdn_url';

							if ( empty( $global_settings ) || empty( $global_settings['cdn.enabled'] ) ) {
								$class .= ' hidden';
							}

							return $class;
						},
					)
				)
			);
		}

		do_action( 'bbapp_admin_settings', $this );
	}

	/**
	 * Saves the settings which are not app id specific.
	 *
	 * @param array $settings Settings.
	 *
	 * @return bool
	 */
	public function save_global_settings( $settings ) {
		ManageApp::instance()->update_settings( $settings );

		return true;
	}

	/**
	 * Gets the settings which are not app id specific.
	 */
	public function get_global_settings() {
		return ManageApp::instance()->get_settings();
	}

	/**
	 * Saves the settings which are app id specific.
	 *
	 * @param array $settings Settings.
	 *
	 * @return bool
	 */
	public function save_settings( $settings ) {
		if ( 'api_cdn' === $this->get_current_sub_tab() ) {
			global $bbapp_cdn_error;

			$bbapp_cdn_error = false;
			$cdn             = isset( $settings['cdn.enabled'] ) ? $settings['cdn.enabled'] : '';
			$cdn_url         = isset( $settings['cdn.url'] ) ? $settings['cdn.url'] : '';

			if ( ! empty( $cdn_url ) && ! empty( $cdn ) ) {
				$parsed = wp_parse_url( $cdn_url );

				if ( ! empty( $parsed ) ) {
					if ( isset( $parsed['scheme'] ) ) {
						unset( $parsed['scheme'] );
					}

					$cdn_url = untrailingslashit( 'https://' . implode( '', $parsed ) );
				}

				$settings['cdn.url'] = $cdn_url;

				/**
				 * Validate the CDN by fetching the media in same site.
				 */
				$media = get_posts(
					array(
						'post_type'      => 'attachment',
						'posts_per_page' => 1,
						'order'          => 'ASC',
						'orderby'        => 'rand',
						'meta_query'     => array( //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
							'relation' => 'AND',
							array(
								'key'     => 'bp_media_saved',
								'compare' => 'NOT EXISTS',
							),
							array(
								'key'     => 'bp_document_saved',
								'compare' => 'NOT EXISTS',
							),
						),
					)
				);

				if ( ! empty( $media ) ) {
					$media_id         = current( $media )->ID;
					$media_url        = wp_get_attachment_url( $media_id );
					$cdn_media_url    = str_replace( home_url(), $cdn_url, $media_url );
					$request          = bbapp_remote_get( $cdn_media_url );
					$response_code    = wp_remote_retrieve_response_code( $request );
					$response_message = wp_remote_retrieve_response_message( $request );

					if ( 200 !== $response_code || 'OK' !== strtoupper( $response_message ) || is_wp_error( $request ) ) {
						$settings['cdn.enabled'] = '';
						$settings['cdn.url']     = '';
						$bbapp_cdn_error         = true;
					}
				}
			} else {
				$settings['cdn.enabled'] = '';
				$settings['cdn.url']     = '';
			}
		}

		ManageApp::instance()->update_app_settings( $settings );

		return true;
	}

	/**
	 * Gets the settings which are app id specific.
	 *
	 * @param bool $network If plugin is network active.
	 *
	 * @return array|bool
	 */
	public function get_settings( $network = false ) {
		$settings = ManageApp::instance()->get_app_settings( $network );

		return $settings;
	}

	/**
	 * Saves the settings for Performance
	 *
	 * @param array $settings Settings.
	 *
	 * @return bool
	 */
	public function performance_save_settings( $settings ) {
		if ( ! class_exists( '\BuddyBoss\Performance\Settings' ) ) {
			require_once dirname( __FILE__, 2 ) . '/Performance/class-settings.php';
		}

		return \BuddyBoss\Performance\Settings::save_settings( $settings, 'buddyboss-app' );
	}

	/**
	 * Gets the settings for Performance
	 *
	 * @return array
	 */
	public function performance_get_settings() {
		if ( ! class_exists( '\BuddyBoss\Performance\Settings' ) ) {
			require_once dirname( __FILE__, 2 ) . '/Performance/class-settings.php';
		}

		return \BuddyBoss\Performance\Settings::get_settings( 'buddyboss-app' );
	}

	/**
	 * Update `cache_component` setting.
	 *
	 * @param array $active_components Active components.
	 */
	public function update_performance_component_setting( $active_components ) {
		$settings = $this->performance_get_settings();

		if ( empty( $settings ) ) {
			$settings = array();
		}

		// components setting save to check components is active or not on MU level while components update.
        $is_active_performance = ! empty( $active_components['performance'] ) ? $active_components['performance'] : 0;

        $settings['cache_component'] = $is_active_performance;

        if ( $is_active_performance ) {
            // Load Performance Class and call on_activation.
            if ( ! class_exists( 'BuddyBoss\Performance\Performance' ) ) {
                // The class doesn't exist, so we need to load it and then call on_activation.
                require_once trailingslashit( bbapp()->plugin_dir ) . 'include/Performance/class-performance.php';
            }
            \BuddyBoss\Performance\Performance::instance()->on_activation();
        }

		$this->performance_save_settings( $settings );
	}

	/**
	 * Saves the settings for auth
	 *
	 * @param array $settings Setings.
	 *
	 * @return bool
	 */
	public function auth_save_settings( $settings ) {
		return bbapp_set_network_option(
			'bbapp_auth_settings',
			$settings
		);
	}

	/**
	 * Gets the settings for auth
	 */
	public function auth_get_settings() {
		$settings = bbapp_get_network_option( 'bbapp_auth_settings' );

		if ( ! is_array( $settings ) ) {
			$settings = array();
		}

		return $settings;
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {
		$tabs = array(
			'general'            => __( 'General', 'buddyboss-app' ),
			'feedback'           => __( 'Feedback', 'buddyboss-app' ),
			'smart_banner'       => __( 'Smart Banner', 'buddyboss-app' ),
			'book_posts'         => __( 'Blog Posts', 'buddyboss-app' ),
			'push_notifications' => __( 'Push Notifications', 'buddyboss-app' ),
			'inapp_purchases'    => __( 'In-App Purchases', 'buddyboss-app' ),
			'access_controls'    => __( 'Access Controls', 'buddyboss-app' ),
			'compatibility'      => __( 'Compatibility', 'buddyboss-app' ),
			'cache_support'      => __( 'API Caching', 'buddyboss-app' ),
			'api_cdn'            => __( 'API CDN', 'buddyboss-app' ),
			'advanced'           => __( 'Advanced', 'buddyboss-app' ),
		);

		if ( ! bbapp_is_active( 'push_notification' ) || is_network_admin() ) {
			unset( $tabs['push_notifications'] );
		}

		if ( ! bbapp_is_active( 'iap' ) ) {
			unset( $tabs['inapp_purchases'] );
		}

		if ( ! bbapp_is_active( 'access_controls' ) ) {
			unset( $tabs['access_controls'] );
		}

		if ( ! bbapp_is_active( 'performance' ) ) {
			unset( $tabs['cache_support'] );
			unset( $tabs['api_cdn'] );
		}

		if ( defined( 'BP_PLATFORM_VERSION' ) ) {
			unset( $tabs['advanced'] );
		}

		return apply_filters( 'bbapp_admin_setting_tabs', $tabs );
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {
		$default = '';

		// Capture first tab which is default.
		foreach ( $this->sub_settings() as $k => $v ) {
			$default = $k;
			break;
		}

		$setting      = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Renders the setting screen.
	 */
	public function render_screen() {
		\BuddyBossApp\Admin\SetupAdmin::instance()->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab() );

		if ( bbapp_is_active( 'push_notification' ) && 'push_notifications' === $this->get_current_sub_tab() ) {
			Notification::instance()->render_settings();
		} elseif ( bbapp_is_active( 'iap' ) && 'inapp_purchases' === $this->get_current_sub_tab() ) {
			InAppPurchases::instance()->render_settings();
		} elseif ( bbapp_is_active( 'access_controls' ) && 'access_controls' === $this->get_current_sub_tab() ) {
			\BuddyBossApp\Admin\AccessControls\Settings::instance()->render_settings();
		} elseif ( 'advanced' === $this->get_current_sub_tab() && ! defined( 'BP_PLATFORM_VERSION' ) ) {
			Advanced::instance()->render_settings();
        } else {
			if ( isset( $this->settings_form ) && $this->settings_form ) {
				$this->settings_form->render_form();
			}
		}

		if ( 'api_cdn' === $this->get_current_sub_tab() ) {
			?>
			<script>
                jQuery( document ).ready( function() {
                    jQuery( document ).on( 'change', 'input[name="app_settings[cdn.enabled]"]', function() {
                        if( jQuery( this ).prop( 'checked' ) ) {
                            jQuery( '.api_cdn_url' ).removeClass( 'hidden' );
                        } else {
                            jQuery( '.api_cdn_url' ).addClass( 'hidden' );
                        }
                    } );
                } );
			</script>
			<?php
		}
		if ( 'general' === $this->get_current_sub_tab() ) {
			?>
			<script>
				jQuery( document ).ready( function () {
					<?php if( bbapp_is_active( 'access_controls' ) ){ ?>
                        jQuery(document).on('change', 'input[name="app_settings[private_app.enabled]"]', function () {
                            const modal = bbappCore.modal.open('bb_private_app_confirm_modal');

                            if (modal) {
                                modal.getBody().show();
                                if (jQuery(this).prop('checked')) {
                                    jQuery('.private-app-enabled').show();
                                    jQuery('.private-app-disabled').hide();
                                } else {
                                    jQuery('.private-app-disabled').show();
                                    jQuery('.private-app-enabled').hide();
                                }
                            }
                        });
					<?php } ?>

                    jQuery( document ).on( 'click', '.bbapp_privat_app_continue', function() {
                        bbappCore.modal.close( 'bb_private_app_confirm_modal' );
					} );


                    jQuery( document ).on( 'click', '.bbapp_privat_app_cancel', function() {
                        const privateAppSelector = jQuery( 'input[name="app_settings[private_app.enabled]"]' );

                        if ( privateAppSelector.prop( 'checked' ) ) {
                            privateAppSelector.prop( 'checked', false );
                        } else {
                            privateAppSelector.prop( 'checked', true );}


                        bbappCore.modal.close( 'bb_private_app_confirm_modal' );
                    } );
                } );
			</script>
			<?php
		}

		if ( 'feedback' === $this->get_current_sub_tab() ) {
			?>
            <script>
                jQuery(document).ready(function () {
                    shareToggle();

                    jQuery(document).on('change', '#bbapp_field_app_settings_app_sharing__enabled', function () {
                        shareToggle();
                    });

                    function shareToggle() {
                        const sharingFields = [
                            '.bbapp_field_wrapper_app_settings_app_sharing__sharing_link',
                            '.bbapp_field_wrapper_app_settings_app_sharing__title',
                            '.bbapp_field_wrapper_app_global_settings_app_sharing__description',
                            '.bbapp_field_wrapper_app_settings_app_sharing__image',
                            '.bbapp_field_wrapper_app_settings_app_sharing__ios__enabled',
                            '.bbapp_field_wrapper_app_settings_app_sharing__android__enabled',
                            '.bbapp_field_wrapper_app_settings_app_sharing__web_fallback'
                        ].join(', ');

                        const enableSharingCheckbox = jQuery('#bbapp_field_app_settings_app_sharing__enabled');
                        const shareFields = jQuery(sharingFields);

                        if (enableSharingCheckbox.is(':checked')) {
                            shareFields.show();
                        } else {
                            shareFields.hide();
                        }
                    }

                    // Add event listener for input change event
                    jQuery( '.bbapp_field_app_settings_app_sharing__web_fallback' ).on( 'input', function () {
                        const curObj = jQuery( this );
                        const inputURL = curObj.val();
                        curObj.nextAll('.bbapp-valid-url').remove();

                        if( '' !== inputURL && !isValidURL( inputURL ) ) {
                            jQuery( '#save_settings' ).attr( 'disabled', 'disabled' );
                            curObj.after( '<div class="bbapp-valid-url"><?php esc_html_e( 'Please enter valid url containing https OR http.', 'buddyboss-app' ); ?></div>' );
                            return;
                        } else {
                            jQuery( '#save_settings' ).attr( 'disabled', false );
                        }
                    } );
                });

                /**
                 * Check if the URL is valid.
                 * @param url
                 * @returns {boolean}
                 */
                function isValidURL( url ) {
                    const urlRegex = /^(https?):\/\/[^\s/$.?#].[^\s]*\.[^\s]{2,}$/;
                    return urlRegex.test( url );
                }
            </script>
			<?php
		}
	}

	/**
	 * Handle Purge Cache actions
	 */
	public function handle_purge_cache() {
		\BuddyBoss\Performance\Settings::handle_purge_cache();
	}
}
