<?php
/**
 * Holds settings form provider functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

/**
 * Settings provider class.
 */
class SettingProvider {

	/**
	 * Get settings function.
	 *
	 * @var $get_setting_func
	 */
	private $get_setting_func;

	/**
	 * Save settings function.
	 *
	 * @var $save_setting_func
	 */
	private $save_setting_func;

	/**
	 * Settings provider name.
	 *
	 * @var string $provider_name
	 */
	private $provider_name = '';

	/**
	 * SettingProvider constructor.
	 *
	 * @param string $provider_name Settings provider name.
	 */
	public function __construct( $provider_name ) {
		$this->provider_name = $provider_name;
	}

	/**
	 * Return the provider name.
	 *
	 * @return string
	 */
	public function provider_name() {
		return $this->provider_name;
	}

	/**
	 * Overrides the hooks for save & get settings
	 *
	 * @param string|array $hook Hook name.
	 */
	public function get_setting_hook( $hook ) {
		$this->get_setting_func = $hook;
	}

	/**
	 * Override the hook for save and get settings.
	 *
	 * @param string|array $hook Hook name.
	 */
	public function save_setting_hook( $hook ) {
		$this->save_setting_func = $hook;
	}

	/**
	 * Parse the field args.
	 *
	 * @param string $name Field name.
	 * @param array  $args Arguments.
	 *
	 * @return array
	 */
	public function field( $name, $args ) {
		$defaults = array(
			'type'     => false,
			'label'    => false,
			'value'    => false,
			'name'     => $name,
			'settings' => array(),
			'provider' => $this->provider_name,
		);

		return wp_parse_args( $args, $defaults );
	}

	/**
	 * Returns the previous saved settings from provider
	 *
	 * @return array|mixed
	 */
	public function get_settings() {
		if ( is_callable( $this->get_setting_func, true ) ) {
			return call_user_func( $this->get_setting_func );
		} else {
			return array();
		}
	}

	/**
	 * Saves the final array of new settings on provider
	 *
	 * @param array $settings Settings to be saved.
	 *
	 * @return mixed
	 */
	public function save_settings( $settings ) {
		if ( is_callable( $this->get_setting_func, true ) ) {
			return call_user_func( $this->save_setting_func, $settings );
		} else {
			return $settings;
		}
	}

	/**
	 * Returns the settings value.
	 *
	 * @param string $field_name Field name.
	 * @param false  $default    Default value.
	 * @param false  $network    Default value.
	 *
	 * @return mixed
	 */
	public function get_value( $field_name, $default = false, $network = false ) {
		$settings = $this->get_settings();

		if ( true === $network && bbapp()->is_network_activated() ) {
			switch_to_blog( 1 );

			// Get main sites setting for iOS store app id and android id, And pass in to the subsites setting.
			$settings = $this->get_settings();

			restore_current_blog();
		}

		if ( isset( $settings[ $field_name ] ) ) {
			return $settings[ $field_name ];
		} else {
			return $default; // Default value.
		}
	}
}
