<?php
/**
 * Holds admin functionality for publishing.
 *
 * @package BuddyBossApp\Admin\Publish
 */

namespace BuddyBossApp\Admin\Publish;

use BuddyBossApp\Admin\ManageApp;
use BuddyBossApp\ClientCommon;
use BuddyBossApp\Permissions;
use BuddyBossApp\Publish as OuterPublish;
use MingYuanYun\AppStore\Exceptions\ConfigException;
use WP_Error;

/**
 * Admin class for Publish.
 */
class Publish {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * If it's current page to load screen.
	 *
	 * @var bool $is_current_page
	 */
	private $is_current_page = false;

	/**
	 * Build notices/warning.
	 *
	 * @var $messages
	 */
	private $messages;

	/**
	 * Runtime cache for publish details to prevent duplicate API calls in same request.
	 *
	 * @since [BBAPPVESION]
	 * @var array $runtime_cache
	 */
	private static $runtime_cache = array();

	/**
	 * Get the instance of the class.
	 *
	 * @return Publish
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Publish constructor.
	 */
	public function __construct() {
	}

	/**
	 * Filter/hooks here.
	 */
	public function load() {
		add_action( 'admin_init', array( $this, 'screen_blocks' ) );
		add_action( 'wp_ajax_bbapp_publish_request_submit', array( $this, 'process_submit' ) );
		add_action( 'wp_ajax_nopriv_bbapp_publish_request_submit', array( $this, 'process_submit' ) );
		add_action( 'bbapp_queue_task_check_publish_status', array( $this, 'queue_task_check_publish_status' ) );
		add_action( 'wp_ajax_bbapp_publish_status_actions', array( $this, 'publish_status_actions' ) );
		add_action( 'wp_ajax_nopriv_bbapp_publish_status_actions', array( $this, 'publish_status_actions' ) );
		add_action( 'wp_ajax_bbapp_refresh_bundle_id', array( $this, 'refresh_bundle_id' ) );
		add_action( 'wp_ajax_bbapp_refresh_apple_checklist', array( $this, 'refresh_apple_checklist' ) );
		add_action( 'wp_ajax_bbapp_refresh_android_checklist', array( $this, 'refresh_android_checklist' ) );
		add_action( 'wp_ajax_bbapp_refresh_app_data', array( $this, 'refresh_app_data' ) );
		add_action( 'wp_ajax_bbapp_publish_app', array( $this, 'create_publish_app' ) );
		add_action( 'wp_ajax_bbapp_load_build_data', array( $this, 'load_build_data' ) );
	}

	/**
	 * Show to screen as per status of build.
	 */
	public function screen_blocks() {
		if ( $this->will_render() ) {
			if ( 'review' === $this->get_current_step() ) {
				$this->interval_publish_status_check(); // Refresh build check automatic.

				$build_option = OuterPublish::instance()->get_publish_options();
				$app_platform = isset( $_GET['app_platform'] ) ? bbapp_input_clean( wp_unslash( $_GET['app_platform'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				$app_env      = 'publish';

				// if the build is in pending redirect to status page.
				if ( isset( $build_option[ $app_platform ][ $app_env ]['status'] ) && 'pending' === $build_option[ $app_platform ][ $app_env ]['status'] ) {
					$link = $this->get_link( "step=status&app_platform={$app_platform}" );
					wp_safe_redirect( $link );
					exit;
				}
			}
		}
	}

	/**
	 * Functions tells & sets that's if current page is one where it's will render.
	 *
	 * @param bool $set Need to render screen.
	 *
	 * @return bool
	 */
	public function will_render( $set = false ) {
		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Return the each statuses details
	 *
	 * @param string $primary Build status.
	 * @param string $sub     Message.
	 *
	 * @return array|bool
	 */
	public function get_statuses_details( $primary, $sub ) {
		$pending_statuses   = array(
			'cancelled'         => array(
				'icon'  => 'icon-pending',
				'title' => __( 'Publish Cancelled', 'buddyboss-app' ),
				'desc'  => __( 'The publish request is cancelled.', 'buddyboss-app' ),
			),
			'new'               => array(
				'icon'  => 'icon-pending',
				'title' => __( 'Publish Submitted', 'buddyboss-app' ),
				'desc'  => __( 'The publish request is submitted and awaiting review from our staff. You can click the button below to refresh the status.', 'buddyboss-app' ),
			),
			'updating_metadata' => array(
				'icon'  => 'icon-pending',
				'title' => __( 'Updating Metadata', 'buddyboss-app' ),
				'desc'  => __( 'The publish request is currently updating the metadata. You can click the button below to refresh the status.', 'buddyboss-app' ),
			),
			'uploading_package' => array(
				'icon'  => 'icon-pending',
				'title' => __( 'Uploading Package', 'buddyboss-app' ),
				'desc'  => __( 'The publish request is currently updating the package. You can click the button below to refresh the status.', 'buddyboss-app' ),
			),
			'reviewing'         => array(
				'icon'  => 'icon-paused',
				'title' => __( 'Reviewing', 'buddyboss-app' ),
				'desc'  => __( 'The publish request is currently being reviewed by our staff. You can click the button below to refresh the status.', 'buddyboss-app' ),
			),
			'pending'           => array(
				'icon'  => 'icon-cancelled',
				'title' => __( 'Publish Processing', 'buddyboss-app' ),
				'desc'  => __( 'The publish request is processing. You can click the button below to refresh the status.', 'buddyboss-app' ),
			),
			'failed'            => array(
				'icon'  => 'icon-pending',
				'title' => __( 'Publish Failed', 'buddyboss-app' ),
				'desc'  => __( 'The publish is processing on our publish server. This can take up to one hour. You can click the button below to refresh the status.', 'buddyboss-app' ),
			),
		);
		$completed_statuses = array(
			'published' => array(
				'icon'  => 'icon-completed',
				'title' => __( 'Your publish is ready', 'buddyboss-app' ),
				'desc'  => __( 'Congrats, your publish is ready!.', 'buddyboss-app' ),
			),
		);
		$cancelled_statuses = array(
			'default' => array(
				'icon'  => 'icon-cancelled',
				'title' => __( 'Publish Cancelled', 'buddyboss-app' ),
				'desc'  => __( 'The publish review was cancelled. You can contact BuddyBoss customer support to get help, or try requesting a new publish.', 'buddyboss-app' ),
			),
		);

		if ( 'pending' === $primary ) {
			$statuses = $pending_statuses;
		} elseif ( 'completed' === $primary ) {
			$statuses = $completed_statuses;
		} elseif ( 'cancelled' === $primary ) {
			$statuses = $cancelled_statuses;
		} else {
			return false;
		}

		if ( isset( $statuses[ $sub ] ) ) {
			return $statuses[ $sub ];
		} else {
			return $statuses['default'];
		}
	}

	/**
	 * Return the review information which will be sent.
	 *
	 * @param bool $platform App platform.
	 * @param bool $env      App environment.
	 *
	 * @return array
	 */
	public function get_review_information( $platform = false, $env = false ) {
		$platform_get = ( ! empty( $_GET['app_platform'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['app_platform'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! $platform && isset( $platform_get ) ) {
			$platform = $platform_get;
		}

		$info = array();

		if ( 'android' === $platform ) {
			$info = array_merge( $info, AndroidHelper::instance()->get_review_information( $env ) );
		}

		if ( 'ios' === $platform ) {
			$info = array_merge( $info, iOSHelper::instance()->get_review_information( $env ) );
		}

		return $info;
	}

	/**
	 * Return summary of valid & invalid review info.
	 *
	 * @param array $review_info Build review information.
	 *
	 * @return array
	 */
	public function get_review_info_summary( $review_info ) {
		$return = array(
			'valid'   => array(),
			'invalid' => array(),
		);

		foreach ( $review_info as $k => $info ) {
			if ( 'files' === $k ) {
				continue;
			}

			if ( isset( $info['validated'] ) && $info['validated'] ) {
				$return['valid'][] = $k;
			} else {
				$return['invalid'][] = $k;
			}
		}

		return $return;
	}

	/**
	 * Renders the branding screen.
	 *
	 * @return false|void
	 */
	public function render() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			echo '<p>' . esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) . '</p>';

			return false;
		}

		include bbapp()->plugin_dir . 'views/settings/publish/publish.php';
	}

	/**
	 * Output the error or update messages to render.
	 */
	public function show_messages() {
		if ( ! empty( $this->messages ) ) {
			foreach ( $this->messages as $message ) {
				printf( '<div id="message" class="%1$s"><p>%2$s</p></div>', esc_attr( $message['type'] ), wp_kses_post( $message['message'] ) );
			}
		}
	}

	/**
	 * Output the nonce input for publish screen.
	 */
	public function publish_nonce_field() {
		wp_nonce_field( 'bbapp_publish_form_nonce', 'bbapp_publish_form_nonce' );
	}

	/**
	 * Publish Submit Process
	 */
	public function process_submit() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			wp_send_json_error( __( "You don't have permission to access this page.", 'buddyboss-app' ) );

			return false;
		}

		$nonce_post = ( ! empty( $_POST['bbapp_publish_form_nonce'] ) ) ? sanitize_text_field( wp_unslash( $_POST['bbapp_publish_form_nonce'] ) ) : '';

		if ( wp_verify_nonce( $nonce_post, 'bbapp_publish_form_nonce' ) ) {

			if ( ! isset( $_POST['app_platform'] ) || ! isset( $_POST['app_env'] ) ) {
				wp_send_json_error( __( 'App Platform, and App Environment are not specified. Please try again.', 'buddyboss-app' ) );
			}

			$app          = \BuddyBossApp\ManageApp::instance()->get_app();
			$app_platform = ( ! empty( $_POST['app_platform'] ) ) ? sanitize_text_field( wp_unslash( $_POST['app_platform'] ) ) : '';
			$app_env      = ( ! empty( $_POST['app_env'] ) ) ? sanitize_text_field( wp_unslash( $_POST['app_env'] ) ) : 'publish';

			if ( empty( $app ) ) {
				wp_send_json_error( __( 'No connected app found.', 'buddyboss-app' ) );
			}

			$review_information = $this->get_review_information( $app_platform, $app_env );
			$review_summary     = $this->get_review_info_summary( $review_information );

			if ( ! empty( $review_summary['invalid'] ) ) {
				wp_send_json_error( __( 'Please provide the required fields in order to send request.', 'buddyboss-app' ) );
			}

			$buildjson = array(
				'platform' => $app_platform,
				'site_url' => get_home_url(),
			);

			// Append all fields into json variable.
			foreach ( $review_information as $info_name => $info ) {
				if ( 'files' === $info_name ) {
					continue;
				}

				if ( isset( $info['is_multi_image'] ) ) {
					foreach ( $info['value'] as $id ) {
						$buildjson[ $info_name ][] = wp_get_attachment_url( $id );
					}
				} else {
					$buildjson[ $info_name ] = $info['value'];
				}
			}

			$buildjson    = wp_json_encode( $buildjson, JSON_PRETTY_PRINT );
			$build_option = OuterPublish::instance()->get_publish_options();
			$status       = $build_option[ $app_platform ][ $app_env ]['status'];

			// check if any request was already sent.
			if ( 'pending' === $status ) {
				wp_send_json_error( __( 'Publish request is already in progress. Please wait for a response from BuddyBoss Support.', 'buddyboss-app' ) );
				exit; // extra sure.
			}

			$response = OuterPublish::instance()->send_publish_request( $buildjson, $review_information['files'] );

			if ( ! isset( $response['publish_id'] ) && isset( $response['message'] ) ) {
				/**
				 * If return error is related to invalid version let's update.
				 * latest version locally.
				 */
				if ( 'invalid_app_version' === $response['code'] ) {
					OuterPublish::instance()->update_last_versions( true );
				}

				wp_send_json_error( sprintf( '%1$s %2$s', esc_html__( "Error while sending build request. \nReason : ", 'buddyboss-app' ), esc_html( $response['message'] ) ) );
				exit; // extra sure.
			} elseif ( ! isset( $response['publish_id'] ) ) {
				wp_send_json_error( __( 'Invalid error while sending build request. Please try again later or contact BuddyBoss Support.', 'buddyboss-app' ) );
				exit; // extra sure.
			}

			$build_option[ $app_platform ][ $app_env ]['status']       = 'pending';
			$build_option[ $app_platform ][ $app_env ]['sub_status']   = 'new';
			$build_option[ $app_platform ][ $app_env ]['publish_id']   = $response['publish_id'];
			$build_option[ $app_platform ][ $app_env ]['request_time'] = current_time( 'mysql', 1 );

			// If url redirect is enabled update that.
			if ( isset( $review_information['url_redirect'] ) && bbapp_is_valid_url( $review_information['url_redirect']['value'] ) ) {
				// We are storing the url redirect and keeping status false.
				// status will be changed to true once we receive success status of this build.
				$build_option[ $app_platform ][ $app_env ]['has_url_redirect'] = true;
			}

			OuterPublish::instance()->update_publish_options( $build_option );

			wp_send_json_success( array( 'redirect' => $this->get_link( "step=status&app_platform={$app_platform}&app_env={$app_env}" ) ) );
		}

		wp_send_json_error( __( 'Error while verifying security nonce. Reload the page and try again.', 'buddyboss-app' ) );

		return false;
	}

	/**
	 * Check Publish Status Based on Interval.
	 * Main intention is to call this on build status page. this will check the in every 10 min visit.
	 */
	public function interval_publish_status_check() {
		$transient_key = 'bbapp_last_build_status_interval_check';
		set_transient( $transient_key, '1', MINUTE_IN_SECONDS * 10 );
		\BuddyBossApp\Publish::instance()->check_publish_status();
	}

	/**
	 * Background queue task for checking build status.
	 * 1. BuddyBossApp Server Pings WordPress to do build status check.
	 * 2. Plugin adds a background job to attempt it at background.
	 * 3. This function get triggered once job gets triggered.
	 *
	 * @param object $task BG task data.
	 *
	 * @return bool
	 * @throws ConfigException Exception message.
	 */
	public function queue_task_check_publish_status( $task ) {
		$task_data = maybe_unserialize( $task->data );

		\BuddyBossApp\Publish::instance()->check_publish_status();
		\BuddyBossApp\AppStores\Apple::instance()->display_app_data( true );
		\BuddyBossApp\AppStores\Android::instance()->get_release_data( true );
		$this->delete_all_publish_transients();

		return true;
	}

	/**
	 * **Check Action**
	 * Force check the build status on bbapp server.
	 * Sometime client server can miss the update notify
	 * From bbapp server on that case this can be useful.
	 * **Cancel Action**
	 * This will allow to cancel build if it's in new status.
	 *
	 * @return bool
	 * @throws ConfigException Exception message.
	 */
	public function publish_status_actions() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			wp_send_json_error( __( 'You don\'t have permission to access this page.', 'buddyboss-app' ) );

			return false;
		}

		$nonce_post = ( ! empty( $_POST['bbapp_publish_actions'] ) ) ? sanitize_text_field( wp_unslash( $_POST['bbapp_publish_actions'] ) ) : '';

		if ( wp_verify_nonce( $nonce_post, 'bbapp_publish_actions' ) ) {
			$sub_action = isset( $_POST['sub_action'] ) ? sanitize_text_field( wp_unslash( $_POST['sub_action'] ) ) : false;
			$publish_id = isset( $_POST['publish_id'] ) ? sanitize_text_field( wp_unslash( $_POST['publish_id'] ) ) : false;
			$platform   = isset( $_POST['platform'] ) ? sanitize_text_field( wp_unslash( $_POST['platform'] ) ) : false;

			if ( 'cancel' === $sub_action ) {
				if ( empty( $publish_id ) ) {
					wp_send_json_error( __( 'Invalid publish ID. Please try again later.', 'buddyboss-app' ) );
				}

				$this->cancel_publish( $publish_id, $platform );
				$this->delete_all_publish_transients();

				delete_transient( 'get_publish_status_' . $publish_id );
				delete_site_transient( 'get_publish_status_' . $publish_id );

				wp_send_json_success( array( 'reload' => true ) );
			}

			if ( 'status_check' === $sub_action ) {
				$apple_id = '';

				\BuddyBossApp\Publish::instance()->check_publish_status();

				$data = $this->get_publish_details( $publish_id );

				if ( 'ios' === $platform ) {
					$apple_id = ! empty( $data ) && ! empty( $data['apple_id'] ) ? $data['apple_id'] : '';
					\BuddyBossApp\AppStores\Apple::instance()->display_app_data( true, $apple_id );
				} else {
					\BuddyBossApp\AppStores\Android::instance()->get_release_data( true );
				}

				$this->update_data_into_appcenter( $publish_id, $platform, $apple_id );

				$this->delete_all_publish_transients();

				wp_send_json_success( array( 'reload' => true ) );

			}
		} else {
			wp_send_json_error( __( 'Error while validating security. Please refresh the page and try again.', 'buddyboss-app' ) );
		}
	}

	/**
	 * Return the current step.
	 *
	 * @return string
	 */
	public function get_current_step() {
		$step     = 'scope';
		$step_get = ( ! empty( $_GET['step'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['step'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( isset( $step_get ) && in_array( $step_get, array( 'review', 'status' ), true ) ) {
			$step = $step_get;
		}

		return $step;
	}

	/**
	 * Get publish link.
	 *
	 * @param string $extra_param Extra parameter string.
	 *
	 * @return string
	 */
	public function get_link( $extra_param ) {

		return bbapp_get_super_admin_url( "admin.php?page=bbapp-publish&setting=publish&{$extra_param}" );
	}

	/**
	 * Refresh Bundle Ids.
	 */
	public function refresh_bundle_id() {
		\BuddyBossApp\AppStores\Apple::instance()->get_client_apps_bundle_id( true );
		wp_send_json_success();
	}

	/**
	 * Refresh Apple Checklist.
	 */
	public function refresh_apple_checklist() {
		\BuddyBossApp\AppStores\Apple::instance()->validate_app( true );
		wp_send_json_success();
	}

	/**
	 * Refresh Android Checklist.
	 */
	public function refresh_android_checklist() {
		\BuddyBossApp\AppStores\Android::instance()->validate_app( true );
		wp_send_json_success();
	}

	/**
	 * Refresh Apple App Data.
	 */
	public function refresh_app_data() {
		\BuddyBossApp\AppStores\Apple::instance()->fetch_app_data( '', true );
		wp_send_json_success();
	}

	/**
	 * Create new App Publish.
	 */
	public function create_publish_app() {
		$response = array(
			'feedback' => __( 'There was a problem when trying to publish app. Please try again.', 'buddyboss-app' ),
		);

		// Use default nonce.
		$nonce = ( ! empty( $_POST['create-publish-nonce'] ) ) ? sanitize_text_field( wp_unslash( $_POST['create-publish-nonce'] ) ) : '';
		$check = 'create-publish-nonce';

		// Nonce check!
		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, $check ) ) {
			wp_send_json_error( $response, 400 );
		}

		$data       = $_POST;
		$platform   = isset( $data['platform'] ) ? $data['platform'] : '';
		$build_id   = isset( $data['build-id'] ) ? $data['build-id'] : '';
		$publish_id = isset( $data['publish_id'] ) ? $data['publish_id'] : '';
		$args       = array(
			'bbapp_id'  => \BuddyBossApp\ManageApp::instance()->get_app_id(),
			'bbapp_key' => \BuddyBossApp\ManageApp::instance()->get_auth_app_key(),
			'build_id'  => $build_id,
			'status'    => ( isset( $data['status'] ) ? $data['status'] : 'new' ),
			'platform'  => $platform,
			'env'       => 'publish',
		);

		if ( 'ios' === $platform ) {
			if ( ! empty( $publish_id ) ) {
				$made_changes                          = isset( $data['made-changes'] ) ? $data['made-changes'] : '';
				$build_change                          = isset( $data['build-change'] ) ? $data['build-change'] : '';
				$build_updated                         = isset( $data['build-id-updated'] ) ? $data['build-id-updated'] : '';
				$args['is_client_skipped_the_changes'] = ( ! empty( $made_changes ) ? $made_changes : 'no' );
				$args['exclude_meta']                  = true;
				$args['publish_id']                    = $publish_id;

				if ( ! empty( $build_change ) && 'yes' === $build_change && ! empty( $build_updated ) ) {
					$args['build_id'] = $build_updated;
				}
			} else {
				$display_app_data = \BuddyBossApp\AppStores\Apple::instance()->display_app_data();
				$app_data         = \BuddyBossApp\AppStores\Apple::instance()->fetch_app_data();

				if ( ! empty( $app_data['apple_app_id'] ) ) {
					$args['apple_id'] = $app_data['apple_app_id'];
				}

				$app_screenshot_keys     = \BuddyBossApp\AppStores\Apple::instance()->app_screenshot_list();
				$args['app_screenshots'] = array();

				foreach ( $app_screenshot_keys as $key => $app_screenshot_key ) {
					if ( isset( $app_screenshot_key['display_key'] ) ) {
						$args['app_screenshots'][ $app_screenshot_key['display_key'] ] = isset( $display_app_data[ $key ]['value'] ) ? $display_app_data[ $key ]['value'] : array( 'default_value' );
					}
				}

				$args['app_information'] = array(
					'promotional_text' => ( isset( $display_app_data['promotional_text']['value'] ) ? $display_app_data['promotional_text']['value'] : '' ),
					'description'      => ( isset( $display_app_data['description']['value'] ) ? $display_app_data['description']['value'] : '' ),
					'keywords'         => ( isset( $display_app_data['keywords']['value'] ) ? $display_app_data['keywords']['value'] : '' ),
					'support_url'      => ( isset( $display_app_data['support_url']['value'] ) ? $display_app_data['support_url']['value'] : '' ),
					'marketing_url'    => ( isset( $display_app_data['marketing_url']['value'] ) ? $display_app_data['marketing_url']['value'] : '' ),
					'copyright'        => ( isset( $display_app_data['copyright']['value'] ) ? $display_app_data['copyright']['value'] : '' ),
					'version'          => ( isset( $display_app_data['version']['value'] ) ? $display_app_data['version']['value'] : '' ),
					'version_string'   => ( isset( $app_data['versionString'] ) ? $app_data['versionString'] : '' ),
					'app_store_state'  => ( ! empty( $app_data['appStoreState'] ) ? $app_data['appStoreState'] : '' ),
					'date_created'     => ( ! empty( $app_data['createdDate'] ) ? $app_data['createdDate'] : '' ),
					'version_id'       => ( isset( $app_data['version_id'] ) ? $app_data['version_id'] : '' ),
				);

				if ( isset( $display_app_data['whatsNew']['value'] ) ) {
					$args['app_information']['whats_new'] = $display_app_data['whatsNew']['value'];
				}

				$schedule_type            = ( isset( $display_app_data['release_schedule']['value'] ) ? $display_app_data['release_schedule']['value'] : '' );
				$schedule_day             = ( ! empty( $app_data['earliestReleaseDate'] ) && 'SCHEDULED' === $app_data['releaseType'] ? ' ' . $app_data['earliestReleaseDate'] : '' );
				$args['release_schedule'] = array(
					'schedule_type' => $schedule_type,
					'schedule_day'  => $schedule_day,
				);

				$args['review_information'] = array(
					'account_signin_required' => ( isset( $display_app_data['account_signin_required']['value'] ) ? $display_app_data['account_signin_required']['value'] : '' ),
					'apple_username'          => ( ! empty( $app_data['demoAccountName'] ) ? $app_data['demoAccountName'] : '' ),
					'apple_password'          => ( ! empty( $app_data['demoAccountPassword'] ) ? $app_data['demoAccountPassword'] : '' ),
					'skip_bb_review'          => ( isset( $data['skip-bb-review'] ) ? $data['skip-bb-review'] : '0' ),
					'respond_to_request'      =>  isset( $data['respond-to-request'] ) ? $data['respond-to-request'] : '0',
					'first_name'              => ( ! empty( $app_data['contactFirstName'] ) ? $app_data['contactFirstName'] : '' ),
					'last_name'               => ( ! empty( $app_data['contactLastName'] ) ? $app_data['contactLastName'] : '' ),
					'phone'                   => ( ! empty( $app_data['contactPhone'] ) ? $app_data['contactPhone'] : '' ),
					'email'                   => ( ! empty( $app_data['contactEmail'] ) ? $app_data['contactEmail'] : '' ),
					'review_notes'            => ( ! empty( $display_app_data['notes'] ) ? $app_data['notes'] : '' ),
				);
			}
		} elseif ( 'android' === $platform ) {
			if ( ! empty( $publish_id ) ) {
				$made_changes                          = isset( $data['made-changes'] ) ? $data['made-changes'] : '';
				$build_change                          = isset( $data['build-change'] ) ? $data['build-change'] : '';
				$build_updated                         = isset( $data['build-id-updated'] ) ? $data['build-id-updated'] : '';
				$args['is_client_skipped_the_changes'] = ( ! empty( $made_changes ) ? $made_changes : 'no' );
				$args['exclude_meta']                  = true;
				$args['publish_id']                    = $publish_id;
				if ( ! empty( $build_change ) && 'yes' === $build_change && ! empty( $build_updated ) ) {
					$args['build_id'] = $build_updated;
				}
			} else {
				$release_data = \BuddyBossApp\AppStores\Android::instance()->get_release_data();

				$args['developer_id'] = isset( $data['developer_id'] ) ? $data['developer_id'] : '';

				$args['release_details'] = array(
					'release_name'   => ( isset( $release_data['name'] ) ? $release_data['name'] : '' ),
					'release_notes'  => ( isset( $release_data['release_notes'] ) ? $release_data['release_notes'] : array() ),
					'release_status' => ( isset( $release_data['release_status'] ) ? $release_data['release_status'] : '' ),
				);

				$args['review_information'] = array(
					'skip_bb_review'     => ( isset( $data['skip-bb-review'] ) ? $data['skip-bb-review'] : '0' ),
					'respond_to_request' => ( isset( $data['respond-to-request'] ) ? $data['respond-to-request'] : '0' ),
				);
			}
		}

		if ( ! empty( $publish_id ) ) {
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'update-publish' );
		} else {
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'create-publish' );
		}

		$publish_option = OuterPublish::instance()->get_publish_options();

		if ( ! isset( $publish_option['publish_requested'] ) ) {
			$publish_option['publish_requested'] = array();
		}

		if ( ! empty( $publish_option['publish_requested'] ) ) {
			if ( isset( $publish_option['publish_requested'][ $platform ] ) && empty( $publish_option['publish_requested'][ $platform ]['publish_id'] ) ) {
				unset( $publish_option['publish_requested'][ $platform ] );
			}
		}

		$response = $this->send_publish_request( $api_url, $args );

		if ( ! isset( $response['publish_id'] ) && isset( $response['message'] ) ) {
			if ( 'maintenance_mode_enabled' === $response['code'] ) {
				\BuddyBossApp\ManageApp::instance()->is_appcenter_under_maintenance( true );
				wp_send_json_error(
					sprintf(
					/* translators: Platform name */
						__( 'The publish request cannot be sent because the BuddyBoss server is under maintenance mode. Please check back later.', 'buddyboss-app' ),
						( 'ios' === $platform ? 'iOS' : 'Android' )
					)
				);
			}

			$publish_option['publish_requested'][ $platform ]['success']       = false;
			$publish_option['publish_requested'][ $platform ]['publish_id']    = null;
			$publish_option['publish_requested'][ $platform ]['error_code']    = ( isset( $response['code'] ) ? $response['code'] : '' );
			$publish_option['publish_requested'][ $platform ]['error_message'] = ( isset( $response['message'] ) ? $response['message'] : '' );

		} elseif ( ! isset( $response['publish_id'] ) ) {
			$publish_option['publish_requested'][ $platform ]['success']       = false;
			$publish_option['publish_requested'][ $platform ]['publish_id']    = null;
			$publish_option['publish_requested'][ $platform ]['error_code']    = 'unknown';
			$publish_option['publish_requested'][ $platform ]['error_message'] = __( 'Invalid error while sending publish request.', 'buddyboss-app' );
		} else {
			$publish_option['publish_requested'][ $platform ]['success']        = true;
			$publish_option['publish_requested'][ $platform ]['publish_id']     = ( isset( $response['publish_id'] ) ? $response['publish_id'] : null );
			$publish_option['publish_requested'][ $platform ]['error_code']     = null;
			$publish_option['publish_requested'][ $platform ]['error_message']  = null;
			$publish_option['publish_requested'][ $platform ]['publish_status'] = ( isset( $response['publish_id'] ) ? $this->get_publish_status( $response['publish_id'] ) : '' );

			if ( ! empty( $publish_option['publish_requested'][ $platform ]['publish_id'] ) ) {
				bbapp_set_network_option( "bbapp_latest_{$platform}_publish_id", $publish_option['publish_requested'][ $platform ]['publish_id'] );
			}
		}

		$publish_option['publish_requested'][ $platform ]['request_time'] = current_time( 'mysql', 1 );
		$publish_option['publish_requested'][ $platform ]['key']          = ( isset( $response['data']['key'] ) ? $response['data']['key'] : null );

		OuterPublish::instance()->update_publish_options( $publish_option );

		$this->delete_all_publish_transients();

		if ( empty( $publish_option['publish_requested'][ $platform ]['publish_id'] ) ) {
			$redirect_url = bbapp_get_admin_url( "admin.php?page=bbapp-publish&action=create&step=status&setting={$platform}" );
		} else {
			$redirect_url = bbapp_get_admin_url( "admin.php?page=bbapp-publish&setting={$platform}" );
		}

		wp_send_json_success( array( 'redirect' => $redirect_url ) );
		die();
	}

	/**
	 * Send publish.
	 *
	 * @param string $api_url API endpoint URL.
	 * @param array  $args    Array of Arguments.
	 *
	 * @return null|false|mixed
	 */
	public function send_publish_request( $api_url, $args = array() ) {
		$response = bbapp_remote_post(
			$api_url,
			array(
				'method'  => 'POST',
				'timeout' => 15,
				'body'    => $args,
			)
		);

		if ( is_wp_error( $response ) ) {
			return false;
		}

		$body     = wp_remote_retrieve_body( $response );
		$response = json_decode( $body, true );

		return $response;
	}

	/**
	 * Cancel publish.
	 *
	 * @param int    $publish_id Publish ID.
	 * @param string $platform   Platform type.
	 *
	 * @return bool
	 */
	public function cancel_publish( $publish_id, $platform ) {
		$args                 = array();
		$api_url              = ClientCommon::instance()->get_center_api_url( 'v1', 'update-publish' );
		$args['bbapp_id']     = \BuddyBossApp\ManageApp::instance()->get_app_id();
		$args['bbapp_key']    = \BuddyBossApp\ManageApp::instance()->get_auth_app_key();
		$args['publish_id']   = $publish_id;
		$args['platform']     = $platform;
		$args['status']       = 'cancelled';
		$args['exclude_meta'] = true;

		$this->send_publish_request( $api_url, $args );

		return true;
	}

	/**
	 * Get Published APP details from API.
	 *
	 * @param int     $publish_id     Publish review id.
	 * @param boolean $override_cache Overwrite cache.
	 *
	 * @return array | WP_Error
	 */
	public function get_publish_details( $publish_id, $override_cache = false ) {
		if ( ! \BuddyBossApp\ManageApp::instance()->is_app_center_connected() ) {
			return new WP_Error( 'app_not_connected', __( 'The app is not connected to the app center.', 'buddyboss-app' ) );
		}

		// Check runtime cache first (prevents duplicate API calls in same request).
		$runtime_key = 'publish_details_' . $publish_id;

		if ( ! $override_cache && isset( self::$runtime_cache[ $runtime_key ] ) ) {
			return self::$runtime_cache[ $runtime_key ];
		}

		$args = array();

		if ( ! empty( $publish_id ) ) {
			$args['publish_id'] = $publish_id;
		}

		$args['bbapp_id']  = \BuddyBossApp\ManageApp::instance()->get_app_id();
		$args['bbapp_key'] = \BuddyBossApp\ManageApp::instance()->get_auth_app_key();
		$args['_embed']    = true;
		$cache_key         = 'bbapp_publish_detail_' . md5( maybe_serialize( $args ) );
		$cached            = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$retval = $cached;
		} else {
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-publish-details' );
			$request = bbapp_remote_get(
				$api_url,
				array(
					'timeout' => 15,
					'body'    => $args,
				)
			);

			$response         = wp_remote_retrieve_body( $request );
			$response         = json_decode( $response, true );
			$response_code    = wp_remote_retrieve_response_code( $request );
			$response_message = wp_remote_retrieve_response_message( $request );

			if ( 'OK' === $response_message && 200 === $response_code ) {
				$retval = $response;
			} elseif ( 400 !== $response_code ) {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response_message ) ),
					$response_message,
					array( 'status' => $response_code )
				);
			} else {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response['code'] ) ),
					$response['message'],
					array( 'status' => $response_code )
				);
			}

			// Get dynamic cache duration based on status.
			$cache_duration = $this->get_cache_duration_for_status( $retval );

			set_site_transient( $cache_key, $retval, $cache_duration );
		}

		// Store in runtime cache for this request.
		self::$runtime_cache[ $runtime_key ] = $retval;

		return $retval;
	}

	/**
	 * Check status forcefully.
	 *
	 * @param int $publish_id Publish ID.
	 *
	 * @return mixed|string
	 */
	public function get_publish_status( $publish_id ) {
		$status = '';
		$data   = $this->get_publish_details( $publish_id, true );

		if ( ! is_wp_error( $data ) && isset( $data['status'] ) ) {
			$status = $data['status'];
		}

		return $status;
	}

	/**
	 * Get cache duration based on publish status.
	 *
	 * Returns dynamic cache duration based on the status of the publish request.
	 * Completed/canceled statuses are cached longer as they don't change.
	 * Active statuses are cached shorter for more frequent updates.
	 *
	 * @param array|WP_Error $data Publish details data.
	 *
	 * @since [BBAPPVESION]
	 *
	 * @return int Cache duration in seconds.
	 */
	private function get_cache_duration_for_status( $data ) {
		// Default cache duration for errors or missing data.
		if ( is_wp_error( $data ) || empty( $data ) ) {
			return 5 * MINUTE_IN_SECONDS; // 5 minutes for errors.
		}

		$status = isset( $data['status'] ) ? $data['status'] : '';

		switch ( $status ) {
			case 'completed':
			case 'customer_completed':
			case 'cancelled':
			case 'connection_lost':
				// Final states that don't change - cache for 4 hours.
				return 4 * HOUR_IN_SECONDS;
			case 'changes_required':
				// Waiting on client action - cache for 2 hours.
				return 2 * HOUR_IN_SECONDS;
			case 'new':
			case 'pending':
			case 'ready_for_submission':
			case 'submitting_for_publishing':
				// Early stages - cache for 15 minutes.
				return 15 * MINUTE_IN_SECONDS;
			case 'in_review':
			case 'in_reject':
			case 're_submitted':
				// Active review process - cache for 20 minutes.
				return 20 * MINUTE_IN_SECONDS;
			default:
				// Unknown status - use default 30 minutes.
				return 30 * MINUTE_IN_SECONDS;
		}
	}

	/**
	 * Load Build Data.
	 *
	 * @return mixed|string
	 */
	public function load_build_data() {
		$build_id = isset( $_POST['build_version'] ) ? bbapp_input_clean( wp_unslash( $_POST['build_version'] ) ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $build_id ) ) {
			return '';
		}

		include bbapp()->plugin_dir . 'views/settings/publish/publish-build-info.php';
		die();
	}

	/**
	 * Get Published APPs from API.
	 *
	 * @param string  $platform       Platform of build.
	 * @param int     $page           Number of page.
	 * @param int     $per_page       Number of item per page.
	 * @param boolean $override_cache Overwrite cache.
	 *
	 * @return array
	 */
	public function get_publish_history( $platform = '', $page = 1, $per_page = 3, $override_cache = true ) {
		if ( empty( $per_page ) ) {
			$per_page = 3;
		}

		$args = array();

		if ( ! empty( $platform ) ) {
			$args['platform'] = $platform;
		}

		if ( ! empty( $page ) ) {
			$args['page'] = $page;
		}

		if ( ! empty( $per_page ) ) {
			$args['per_page'] = $per_page;
		}

		$args['bbapp_id']  = \BuddyBossApp\ManageApp::instance()->get_app_id();
		$args['bbapp_key'] = \BuddyBossApp\ManageApp::instance()->get_auth_app_key();
		$cache_key         = 'bbapp_publish_history_' . md5( maybe_serialize( $args ) );
		$cached            = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$retval = $cached;
		} else {
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-publish' );

			$request = bbapp_remote_get(
				$api_url,
				array(
					'timeout' => 15,
					'body'    => $args,
				)
			);

			$response         = wp_remote_retrieve_body( $request );
			$response         = json_decode( $response, true );
			$response_code    = wp_remote_retrieve_response_code( $request );
			$response_message = wp_remote_retrieve_response_message( $request );

			if ( 'OK' === $response_message && 200 === $response_code ) {
				$data['data']                  = $response;
				$data['headers']['total']      = wp_remote_retrieve_header( $request, 'Total' );
				$data['headers']['totalpages'] = wp_remote_retrieve_header( $request, 'TotalPages' );
				$retval                        = $data;

				set_site_transient( $cache_key, $retval, 10 * 60 ); // cache it for 10 min.
			} elseif ( 400 !== $response_code ) {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response_message ) ),
					$response_message,
					array( 'status' => $response_code )
				);
			} else {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response['code'] ) ),
					$response['message'],
					array( 'status' => $response_code )
				);
			}
		}

		return $retval;
	}

	/**
	 * Get publish status label.
	 *
	 * @param string $status Status text.
	 *
	 * @return string
	 */
	public function publish_status_text( $status ) {
		if ( empty( $status ) ) {
			return '<span class="bbapp-status bbapp-status--publish failed">' . esc_html__( 'Failed', 'buddyboss-app' ) . '</span>';
		}

		switch ( $status ) {
			case 'new':
			case 'ready_for_submission':
			case 'submitting_for_publishing':
			case 'in_review':
			case 'in_reject':
			case 're_submitted':
			case 'pending':
				$status_text = '<span class="bbapp-status bbapp-status--publish running">' . esc_html__( 'In Progress', 'buddyboss-app' ) . '</span>';
				break;
			case 'changes_required':
				$status_text = '<span class="bbapp-status bbapp-status--publish changes-required">' . esc_html__( 'Changes Recommended', 'buddyboss-app' ) . '</span>';
				break;
			case 'completed':
				$status_text = '<span class="bbapp-status bbapp-status--publish completed">' . esc_html__( 'Approved', 'buddyboss-app' ) . '</span>';
				break;
			case 'customer_completed':
				$status_text = '<span class="bbapp-status bbapp-status--publish completed">' . esc_html__( 'Completed', 'buddyboss-app' ) . '</span>';
				break;
			case 'cancelled':
			case 'connection_lost':
			default:
				$status_text = '<span class="bbapp-status bbapp-status--publish cancelled">' . esc_html__( 'Cancelled', 'buddyboss-app' ) . '</span>';
				break;
		}

		return $status_text;
	}

	/**
	 * Get publish list actions.
	 *
	 * @param string $status     Status Text.
	 * @param int    $publish_id Publish ID.
	 */
	public function publish_action_url( $status, $publish_id = '' ) {
		switch ( $status ) {
			case 'new':
			case 'ready_for_submission':
			case 'submitting_for_publishing':
			case 'in_reject':
			case 're_submitted':
			case 'pending':
				$status_text = 'pending';
				break;
			case 'in_review':
				$status_text = 'in-review';
				break;
			case 'changes_required':
				$status_text = 'change-requested';
				break;
			case 'completed':
				$status_text = 'completed';
				break;
			case 'customer_completed':
				$status_text = 'customer-completed';
				break;
			case 'cancelled':
			case 'connection_lost':
			default:
				$status_text = 'cancelled';
				break;
		}

		$action_endpoint = '';

		if ( 'pending' === $status_text || 'change-requested' === $status_text ) {
			$action_endpoint = '<a href="javascript:;" class="button button-primary cancle-request" data-id="' . esc_attr( $publish_id ) . '"><div class="bbapp_loading dark" style="display:none"></div>' . esc_html__( 'Cancel Request', 'buddyboss-app' ) . '</a>';
		}

		return $action_endpoint;
	}

	/**
	 * Clear all transients for publish lists.
	 */
	public function delete_all_publish_transients() {
		$option_name_arr = array(
			'_transient_bbapp_publish_history_',
			'_transient_bbapp_publish_detail_',
			'_transient_bbapp_apple_client_apps_bundle_ids',
		);

		bbapp_delete_transients( $option_name_arr );

		// Clear runtime cache as well.
		self::$runtime_cache = array();
	}

	/**
	 * Get error text based on the error key.
	 *
	 * @param string $key Key text.
	 */
	public function publish_error_key_label( $key ) {
		$data = array(
			'apple_id'                => __( 'Missing Apple App ID.', 'buddyboss-app' ),
			'app_screenshots'         => __( 'Missing app screenshots.', 'buddyboss-app' ),
			'app_information'         => __( 'Missing app information.', 'buddyboss-app' ),
			'promotional_text'        => __( 'Missing promotional text.', 'buddyboss-app' ),
			'description'             => __( 'Missing description.', 'buddyboss-app' ),
			'keywords'                => __( 'Missing keywords.', 'buddyboss-app' ),
			'support_url'             => __( 'Missing support URL.', 'buddyboss-app' ),
			'marketing_url'           => __( 'Missing marketing URL.', 'buddyboss-app' ),
			'copyright'               => __( 'Missing copyright information.', 'buddyboss-app' ),
			'version'                 => __( 'Missing version information.', 'buddyboss-app' ),
			'age_rating'              => __( 'Missing age rating configuration.', 'buddyboss-app' ),
			'whats_new'               => __( 'Missing what\'s new in this version.', 'buddyboss-app' ),
			'release_schedule'        => __( 'Missing release schedule information.', 'buddyboss-app' ),
			'schedule_type'           => __( 'Missing release schedule type.', 'buddyboss-app' ),
			'schedule_day'            => __( 'Missing release schedule date.', 'buddyboss-app' ),
			'review_information'      => __( 'Missing app review information.', 'buddyboss-app' ),
			'account_signin_required' => __( 'Missing sign-in configuration.', 'buddyboss-app' ),
			'apple_username'          => __( 'Missing sign-in username.', 'buddyboss-app' ),
			'apple_password'          => __( 'Missing sign-in password.', 'buddyboss-app' ),
			'first_name'              => __( 'Missing contact first name.', 'buddyboss-app' ),
			'last_name'               => __( 'Missing contact last name.', 'buddyboss-app' ),
			'phone'                   => __( 'Missing contact phone number.', 'buddyboss-app' ),
			'email'                   => __( 'Missing contact email address.', 'buddyboss-app' ),
			'review_notes'            => __( 'Missing review notes.', 'buddyboss-app' ),
			'developer_id'            => __( 'Missing Developer ID.', 'buddyboss-app' ),
			'release_name'            => __( 'Missing Release Name.', 'buddyboss-app' ),
			'release_status'          => __( 'Missing Release Status.', 'buddyboss-app' ),
		);

		// Dynamically add error labels for screenshots using app_screenshot_list()
		$app_screenshot_keys     = \BuddyBossApp\AppStores\Apple::instance()->app_screenshot_list();
		foreach ( $app_screenshot_keys as $screenshot_key => $screenshot_data ) {
			if ( isset( $screenshot_data['display_key'] ) ) {
				$data[ $screenshot_data['display_key'] ] = sprintf( __( 'Missing %s screenshots.', 'buddyboss-app' ), $screenshot_data['label'] );
			}
		}


		return ( ! empty( $key ) && isset( $data[ $key ] ) ? $data[ $key ] : $data );
	}

	/**
	 * Post back data to app-center when someone click on the sync data from app plugin.
	 *
	 * @param int    $publish_id Publish ID.
	 * @param string $platform   Platform Type.
	 * @param string $apple_id   Apple  ID.
	 *
	 * @throws ConfigException Exception error message.
	 */
	public function update_data_into_appcenter( $publish_id, $platform, $apple_id = '' ) {
		if ( empty( $publish_id ) ) {
			return;
		}

		$args = array(
			'bbapp_id'   => \BuddyBossApp\ManageApp::instance()->get_app_id(),
			'bbapp_key'  => \BuddyBossApp\ManageApp::instance()->get_auth_app_key(),
			'platform'   => $platform,
			'publish_id' => $publish_id,
		);

		if ( 'ios' === $platform ) {
			$display_app_data = \BuddyBossApp\AppStores\Apple::instance()->display_app_data( false, $apple_id );
			$app_data         = \BuddyBossApp\AppStores\Apple::instance()->fetch_app_data();

			if ( ! empty( $app_data['apple_app_id'] ) ) {
				$args['apple_id'] = $app_data['apple_app_id'];
			}

			$app_screenshot_keys     = \BuddyBossApp\AppStores\Apple::instance()->app_screenshot_list();
			$args['app_screenshots'] = array();

			foreach ( $app_screenshot_keys as $key => $data ) {
				if ( isset( $data['display_key'] ) ) {
					$args['app_screenshots'][ $data['display_key'] ] = isset( $display_app_data[ $key ]['value'] ) ? $display_app_data[ $key ]['value'] : array( 'default_value' );
				}
			}

			$args['app_information'] = array(
				'promotional_text' => ( isset( $display_app_data['promotional_text']['value'] ) ? $display_app_data['promotional_text']['value'] : '' ),
				'description'      => ( isset( $display_app_data['description']['value'] ) ? $display_app_data['description']['value'] : '' ),
				'keywords'         => ( isset( $display_app_data['keywords']['value'] ) ? $display_app_data['keywords']['value'] : '' ),
				'support_url'      => ( isset( $display_app_data['support_url']['value'] ) ? $display_app_data['support_url']['value'] : '' ),
				'marketing_url'    => ( isset( $display_app_data['marketing_url']['value'] ) ? $display_app_data['marketing_url']['value'] : '' ),
				'copyright'        => ( isset( $display_app_data['copyright']['value'] ) ? $display_app_data['copyright']['value'] : '' ),
				'version'          => ( isset( $display_app_data['version']['value'] ) ? $display_app_data['version']['value'] : '' ),
				'version_string'   => ( isset( $app_data['versionString'] ) ? $app_data['versionString'] : '' ),
				'app_store_state'  => ( ! empty( $app_data['appStoreState'] ) ? $app_data['appStoreState'] : '' ),
				'date_created'     => ( ! empty( $app_data['createdDate'] ) ? $app_data['createdDate'] : '' ),
				'version_id'       => ( isset( $app_data['version_id'] ) ? $app_data['version_id'] : '' ),
			);

			if ( isset( $display_app_data['whatsNew']['value'] ) ) {
				$args['app_information']['whats_new'] = $display_app_data['whatsNew']['value'];
			}

			$schedule_type = ( isset( $display_app_data['release_schedule']['value'] ) ? $display_app_data['release_schedule']['value'] : '' );
			$schedule_day  = ( ! empty( $app_data['earliestReleaseDate'] ) && 'SCHEDULED' === $app_data['releaseType'] ? ' ' . $app_data['earliestReleaseDate'] : '' );

			$args['release_schedule'] = array(
				'schedule_type' => $schedule_type,
				'schedule_day'  => $schedule_day,
			);

			$args['review_information'] = array(
				'account_signin_required' => ( isset( $display_app_data['account_signin_required']['value'] ) ? $display_app_data['account_signin_required']['value'] : '' ),
				'apple_username'          => ( ! empty( $app_data['demoAccountName'] ) ? $app_data['demoAccountName'] : '' ),
				'apple_password'          => ( ! empty( $app_data['demoAccountPassword'] ) ? $app_data['demoAccountPassword'] : '' ),
				'first_name'              => ( ! empty( $app_data['contactFirstName'] ) ? $app_data['contactFirstName'] : '' ),
				'last_name'               => ( ! empty( $app_data['contactLastName'] ) ? $app_data['contactLastName'] : '' ),
				'phone'                   => ( ! empty( $app_data['contactPhone'] ) ? $app_data['contactPhone'] : '' ),
				'email'                   => ( ! empty( $app_data['contactEmail'] ) ? $app_data['contactEmail'] : '' ),
				'review_notes'            => ( ! empty( $display_app_data['notes'] ) ? $app_data['notes'] : '' ),
			);
		} elseif ( 'android' === $platform ) {
			$release_data            = \BuddyBossApp\AppStores\Android::instance()->get_release_data();
			$args['release_details'] = array(
				'release_name'   => ( isset( $release_data['name'] ) ? $release_data['name'] : '' ),
				'release_notes'  => ( isset( $release_data['release_notes'] ) ? $release_data['release_notes'] : array() ),
				'release_status' => ( isset( $release_data['release_status'] ) ? $release_data['release_status'] : '' ),
			);
		}

		$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'update-publish' );

		$this->send_publish_request( $api_url, $args );
	}

	/**
	 * Get Publishing status from AppCenter.
	 *
	 * @param false $override_cache Over-ride cache.
	 *
	 * @return mixed|WP_Error
	 */
	public function get_publishing_status( $override_cache = false ) {
		$args              = array();
		$args['bbapp_id']  = \BuddyBossApp\ManageApp::instance()->get_app_id();
		$args['bbapp_key'] = \BuddyBossApp\ManageApp::instance()->get_auth_app_key();
		$cache_key         = 'bbapp_publish_status_' . md5( maybe_serialize( $args ) );
		$cached            = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$retval = $cached;
		} else {
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-publish-status' );

			$request = bbapp_remote_get(
				$api_url,
				array(
					'timeout' => 15,
					'body'    => $args,
				)
			);

			$response         = wp_remote_retrieve_body( $request );
			$response         = json_decode( $response, true );
			$response_code    = wp_remote_retrieve_response_code( $request );
			$response_message = wp_remote_retrieve_response_message( $request );

			if ( 'OK' === $response_message && 200 === $response_code ) {
				$retval = $response;
			} elseif ( 400 !== $response_code ) {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response_message ) ),
					$response_message,
					array( 'status' => $response_code )
				);
			} else {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response['code'] ) ),
					$response['message'],
					array( 'status' => $response_code )
				);
			}

			set_site_transient( $cache_key, $retval, 10 * 60 ); // cache it for 10 min.
		}

		return $retval;
	}

	/**
	 * BuddyBoss Publishing service
	 * return true or false based on if user has already requested publishing services for specific platform or not (excluding - CANCELLED).
	 *
	 * @param string $platform       Platform.
	 * @param false  $override_cache Over-ride cache or not.
	 *
	 * @return bool
	 */
	public function is_publishing_services_already_requested( $platform, $override_cache = false ) {
		$publishing_status = $this->get_publishing_status( $override_cache );

		if ( is_wp_error( $publishing_status ) ) {
			return true;
		}

		if ( isset( $publishing_status[ $platform ] ) && isset( $publishing_status[ $platform ]['status'] ) && 'cancelled' !== $publishing_status[ $platform ]['status'] ) {
			return true;
		}

		return false;
	}
}
