<?php
/**
 * Holds publish functionality for admin.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\Admin\Publish\Publish as InnerPublish;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Permissions;
use BuddyBossApp\Publish as OuterPublish;

/**
 * Publish class for admin.
 */
class Publish {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * SSettings form.
	 *
	 * @var $settings_form
	 */
	private $settings_form;

	/**
	 * Get the instance of the class.
	 *
	 * @return Publish
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Publish constructor.
	 */
	public function __construct() {
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		add_action( 'init', array( $this, 'load_init' ) );
		add_action( 'bbapp_every_30min', array( $this, 'sync_data_with_appcenter' ), 99 );
	}

	/**
	 * Init method.
	 */
	public function load_init() {
		InnerPublish::instance(); // Initiate the instance.

		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( SetupAdmin::instance()->get_page_now() === 'admin.php' && 'bbapp-publish' === $page_get ) {
			$this->settings_form = new FormProvider( 'settings' );

			// holds app settings storage handle.
			$app_settings_provider = new SettingProvider( 'app_settings' );

			// register app settings storing & receiving.
			$app_settings_provider->save_setting_hook( array( $this, 'save_settings' ) );
			$app_settings_provider->get_setting_hook( array( $this, 'get_settings' ) );

			// Link Provider In Form Provider.
			$this->settings_form->link_setting_provider( $app_settings_provider );

			add_action( 'admin_init', array( $this->settings_form, 'save' ) );

			// General.
			if ( 'ios' === $this->get_current_sub_tab() ) {
				InnerPublish::instance()->will_render( true );
			}

			// IOS Related Settings.
			if ( 'ios' === $this->get_current_sub_tab() ) {
				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_publish_ios_gen_settings',
						array(
							'type'  => 'heading',
							'value' => __( 'GROUP', 'buddyboss-app' ),
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.api_private_key',
						array(
							'label'       => __( 'Private Key', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.api_issuer_id',
						array(
							'label'       => __( 'Issuer ID', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.api_key_id',
						array(
							'label'       => __( 'Key ID', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_publish_ios_new_app_settings',
						array(
							'type'  => 'heading',
							'value' => __( 'Create New App', 'buddyboss-app' ),
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_platforms',
						array(
							'label'       => __( 'Platforms', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.app_name',
						array(
							'label'       => __( 'App Name', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.app_lang',
						array(
							'label'       => __( 'Primary Language', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'select',
							'options'     => array(
								'English' => 'English',
								'Greek'   => 'Greek',
							),
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.bundle_id',
						array(
							'label'       => __( 'Bundle ID', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.sku',
						array(
							'label'       => __( 'SKU', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_publish_ios_app_settings',
						array(
							'type'  => 'heading',
							'value' => __( 'App Info', 'buddyboss-app' ),
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_app_name',
						array(
							'label'       => __( 'Name', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.app_privacy_policy_url',
						array(
							'label'       => __( 'Privacy Policy URL', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_app_subtitle',
						array(
							'label'       => __( 'Subtitle', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_app_bundle_id',
						array(
							'label'       => __( 'Bundle ID', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_primary_category',
						array(
							'label'   => __( 'Category / Primary', 'buddyboss-app' ),
							'desc'    => '',
							'type'    => 'select',
							'options' => array(
								'Art & Design'            => 'Art & Design',
								'Auto & Vehicles'         => 'Auto & Vehicles',
								'Beauty'                  => 'Beauty',
								'Books & Reference'       => 'Books & Reference',
								'Business'                => 'Business',
								'Comics'                  => 'Comics',
								'Communication'           => 'Communication',
								'Dating'                  => 'Dating',
								'Education'               => 'Education',
								'Entertainment'           => 'Entertainment',
								'Events'                  => 'Events',
								'Finance'                 => 'Finance',
								'Food & Drink'            => 'Food & Drink',
								'Health & Fitness'        => 'Health & Fitness',
								'House & Home'            => 'House & Home',
								'Libraries & Demo'        => 'Libraries & Demo',
								'Lifestyle'               => 'Lifestyle',
								'Maps & Navigation'       => 'Maps & Navigation',
								'Medical'                 => 'Medical',
								'Music & Audio'           => 'Music & Audio',
								'News & Magazines'        => 'News & Magazines',
								'Parenting'               => 'Parenting',
								'Personalization'         => 'Personalization',
								'Photography'             => 'Photography',
								'Productivity'            => 'Productivity',
								'Shopping'                => 'Shopping',
								'Social'                  => 'Social',
								'Sports'                  => 'Sports',
								'Tools'                   => 'Tools',
								'Travel & Local'          => 'Travel & Local',
								'Video Players & Editors' => 'Video Players & Editors',
								'Weather'                 => 'Weather',
							),
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_secondary_category',
						array(
							'label'   => __( 'Category / Secondary', 'buddyboss-app' ),
							'desc'    => '',
							'type'    => 'select',
							'options' => array(
								'Art & Design'            => 'Art & Design',
								'Auto & Vehicles'         => 'Auto & Vehicles',
								'Beauty'                  => 'Beauty',
								'Books & Reference'       => 'Books & Reference',
								'Business'                => 'Business',
								'Comics'                  => 'Comics',
								'Communication'           => 'Communication',
								'Dating'                  => 'Dating',
								'Education'               => 'Education',
								'Entertainment'           => 'Entertainment',
								'Events'                  => 'Events',
								'Finance'                 => 'Finance',
								'Food & Drink'            => 'Food & Drink',
								'Health & Fitness'        => 'Health & Fitness',
								'House & Home'            => 'House & Home',
								'Libraries & Demo'        => 'Libraries & Demo',
								'Lifestyle'               => 'Lifestyle',
								'Maps & Navigation'       => 'Maps & Navigation',
								'Medical'                 => 'Medical',
								'Music & Audio'           => 'Music & Audio',
								'News & Magazines'        => 'News & Magazines',
								'Parenting'               => 'Parenting',
								'Personalization'         => 'Personalization',
								'Photography'             => 'Photography',
								'Productivity'            => 'Productivity',
								'Shopping'                => 'Shopping',
								'Social'                  => 'Social',
								'Sports'                  => 'Sports',
								'Tools'                   => 'Tools',
								'Travel & Local'          => 'Travel & Local',
								'Video Players & Editors' => 'Video Players & Editors',
								'Weather'                 => 'Weather',
							),
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_appmeta_info',
						array(
							'label'       => __( 'App Meta Info', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_promotional',
						array(
							'label'       => __( 'Promotional Text', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_keywords',
						array(
							'label'       => __( 'Keywords', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => __( 'Please add comma separated Keywords', 'buddyboss-app' ),
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_description',
						array(
							'label'       => __( 'Description', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_support_url',
						array(
							'label'       => __( 'Support URL', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_territories',
						array(
							'label'       => __( 'Territories', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => __( 'Please add comma separated Territories', 'buddyboss-app' ),
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_publish_ios_dev_settings',
						array(
							'type'  => 'heading',
							'value' => __( 'Pricing', 'buddyboss-app' ),
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_price',
						array(
							'label'       => __( 'Price', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_app_multiple_screenshots',
						array(
							'label'       => __( 'Screenshot ', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'multiple-image-picker',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_display',
						array(
							'label'       => __( 'Display Type', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_app_screenshots',
						array(
							'label'       => __( 'App Screenshoot', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'image-picker',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_price_build_id',
						array(
							'label'       => __( 'Build ID', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.ios_changelog',
						array(
							'label'       => __( 'Changelog', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);
			}

			// Android Settings.
			if ( 'android' === $this->get_current_sub_tab() ) {
				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_publish_android_gen_settings',
						array(
							'type'  => 'heading',
							'value' => __( 'General Settings', 'buddyboss-app' ),
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_name',
						array(
							'label'       => __( 'App Name', 'buddyboss-app' ),
							'desc'        => __(
								'For Android, the app name will be displayed in the home screen below the app icon. A maximum of 30 characters can be used.',
								'buddyboss-app'
							),
							'type'        => 'text',
							'value'       => '',
							'placeholder' => 'eg. Wonderful Yoga Training',
							'maxlength'   => 30,
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_lang',
						array(
							'label'       => __( 'App Language', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'select',
							'options'     => array(
								'English' => 'English',
								'Greek'   => 'Greek',
							),
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_publish_android_store_listening',
						array(
							'type'  => 'heading',
							'value' => __( 'Store listening', 'buddyboss-app' ),
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_short_desc',
						array(
							'label'       => __( 'Short description', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'textarea',
							'value'       => '',
							'placeholder' => '',
							'maxlength'   => 80,
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_full_desc',
						array(
							'label'       => __( 'Full description', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'textarea',
							'value'       => '',
							'placeholder' => '',
							'maxlength'   => 4000,
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_screenshots',
						array(
							'label'       => __( 'Screenshot ', 'buddyboss-app' ),
							'desc'        => '',
							'type'        => 'multiple-image-picker',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_promo_url',
						array(
							'label'       => __( 'Promo Video URL ', 'buddyboss-app' ),
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_type',
						array(
							'label'   => __( 'App Type', 'buddyboss-app' ),
							'type'    => 'select',
							'value'   => '',
							'options' => array(
								'App'  => 'App',
								'Game' => 'Game',
							),
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_category',
						array(
							'label'   => __( 'Category', 'buddyboss-app' ),
							'type'    => 'select',
							'value'   => '',
							'options' => array(
								'Art & Design'            => 'Art & Design',
								'Auto & Vehicles'         => 'Auto & Vehicles',
								'Beauty'                  => 'Beauty',
								'Books & Reference'       => 'Books & Reference',
								'Business'                => 'Business',
								'Comics'                  => 'Comics',
								'Communication'           => 'Communication',
								'Dating'                  => 'Dating',
								'Education'               => 'Education',
								'Entertainment'           => 'Entertainment',
								'Events'                  => 'Events',
								'Finance'                 => 'Finance',
								'Food & Drink'            => 'Food & Drink',
								'Health & Fitness'        => 'Health & Fitness',
								'House & Home'            => 'House & Home',
								'Libraries & Demo'        => 'Libraries & Demo',
								'Lifestyle'               => 'Lifestyle',
								'Maps & Navigation'       => 'Maps & Navigation',
								'Medical'                 => 'Medical',
								'Music & Audio'           => 'Music & Audio',
								'News & Magazines'        => 'News & Magazines',
								'Parenting'               => 'Parenting',
								'Personalization'         => 'Personalization',
								'Photography'             => 'Photography',
								'Productivity'            => 'Productivity',
								'Shopping'                => 'Shopping',
								'Social'                  => 'Social',
								'Sports'                  => 'Sports',
								'Tools'                   => 'Tools',
								'Travel & Local'          => 'Travel & Local',
								'Video Players & Editors' => 'Video Players & Editors',
								'Weather'                 => 'Weather',
							),
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_content_rating',
						array(
							'label'   => __( 'Content Rating', 'buddyboss-app' ),
							'type'    => 'select',
							'value'   => '',
							'options' => array(
								'rate1' => 'rate1',
								'rate2' => 'rate2',
								'rate3' => 'rate3',
							),
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_publish_android_contact_details',
						array(
							'type'  => 'heading',
							'value' => __( 'Contact details', 'buddyboss-app' ),
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_web',
						array(
							'label'       => __( 'Web', 'buddyboss-app' ),
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_email',
						array(
							'label'       => __( 'Email', 'buddyboss-app' ),
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_privacy_policy_url',
						array(
							'label'       => __( 'Privacy Policy URL', 'buddyboss-app' ),
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_apk',
						array(
							'label'       => __( 'APK (build)', 'buddyboss-app' ),
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_publish_android_pricing_distribution',
						array(
							'type'  => 'heading',
							'value' => __( 'Pricing & Distribution', 'buddyboss-app' ),
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_price',
						array(
							'label'   => __( 'Paid/Free', 'buddyboss-app' ),
							'type'    => 'select',
							'value'   => '',
							'options' => array(
								'paid' => 'paid',
								'free' => 'free',
							),
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_countries',
						array(
							'label'   => __( 'Countries', 'buddyboss-app' ),
							'type'    => 'select',
							'value'   => '',
							'options' => array(
								'Afghanistan'              => 'Afghanistan',
								'Albania'                  => 'Albania',
								'Algeria'                  => 'Algeria',
								'American Samoa'           => 'American Samoa',
								'Andorra'                  => 'Andorra',
								'Angola'                   => 'Angola',
								'Anguilla'                 => 'Anguilla',
								'Antarctica'               => 'Antarctica',
								'Antigua and Barbuda'      => 'Antigua and Barbuda',
								'Argentina'                => 'Argentina',
								'Armenia'                  => 'Armenia',
								'Aruba'                    => 'Aruba',
								'Australia'                => 'Australia',
								'Austria'                  => 'Austria',
								'Azerbaijan'               => 'Azerbaijan',
								'Bahamas'                  => 'Bahamas',
								'Bahrain'                  => 'Bahrain',
								'Bangladesh'               => 'Bangladesh',
								'Barbados'                 => 'Barbados',
								'Belarus'                  => 'Belarus',
								'Belgium'                  => 'Belgium',
								'Belize'                   => 'Belize',
								'Benin'                    => 'Benin',
								'Bermuda'                  => 'Bermuda',
								'Bhutan'                   => 'Bhutan',
								'Bolivia'                  => 'Bolivia',
								'Bosnia and Herzegowina'   => 'Bosnia and Herzegowina',
								'Botswana'                 => 'Botswana',
								'Bouvet Island'            => 'Bouvet Island',
								'Brazil'                   => 'Brazil',
								'British Indian Ocean Territory' => 'British Indian Ocean Territory',
								'Brunei Darussalam'        => 'Brunei Darussalam',
								'Bulgaria'                 => 'Bulgaria',
								'Burkina Faso'             => 'Burkina Faso',
								'Burundi'                  => 'Burundi',
								'Cambodia'                 => 'Cambodia',
								'Cameroon'                 => 'Cameroon',
								'Canada'                   => 'Canada',
								'Cape Verde'               => 'Cape Verde',
								'Cayman Islands'           => 'Cayman Islands',
								'Central African Republic' => 'Central African Republic',
								'Chad'                     => 'Chad',
								'Chile'                    => 'Chile',
								'China'                    => 'China',
								'Christmas Island'         => 'Christmas Island',
								'Cocos (Keeling) Islands'  => 'Cocos (Keeling) Islands',
								'Colombia'                 => 'Colombia',
								'Comoros'                  => 'Comoros',
								'Congo'                    => 'Congo',
								'Congo, the Democratic Republic of the' => 'Congo, the Democratic Republic of the',
								'Cook Islands'             => 'Cook Islands',
								'Costa Rica'               => 'Costa Rica',
								"Cote d'Ivoire"            => "Cote d'Ivoire",
								'Croatia (Hrvatska)'       => 'Croatia (Hrvatska)',
								'Cuba'                     => 'Cuba',
								'Cyprus'                   => 'Cyprus',
								'Czech Republic'           => 'Czech Republic',
								'Denmark'                  => 'Denmark',
								'Djibouti'                 => 'Djibouti',
								'Dominica'                 => 'Dominica',
								'Dominican Republic'       => 'Dominican Republic',
								'East Timor'               => 'East Timor',
								'Ecuador'                  => 'Ecuador',
								'Egypt'                    => 'Egypt',
								'El Salvador'              => 'El Salvador',
								'Equatorial Guinea'        => 'Equatorial Guinea',
								'Eritrea'                  => 'Eritrea',
								'Estonia'                  => 'Estonia',
								'Ethiopia'                 => 'Ethiopia',
								'Falkland Islands (Malvinas)' => 'Falkland Islands (Malvinas)',
								'Faroe Islands'            => 'Faroe Islands',
								'Fiji'                     => 'Fiji',
								'Finland'                  => 'Finland',
								'France'                   => 'France',
								'France Metropolitan'      => 'France Metropolitan',
								'French Guiana'            => 'French Guiana',
								'French Polynesia'         => 'French Polynesia',
								'French Southern Territories' => 'French Southern Territories',
								'Gabon'                    => 'Gabon',
								'Gambia'                   => 'Gambia',
								'Georgia'                  => 'Georgia',
								'Germany'                  => 'Germany',
								'Ghana'                    => 'Ghana',
								'Gibraltar'                => 'Gibraltar',
								'Greece'                   => 'Greece',
								'Greenland'                => 'Greenland',
								'Grenada'                  => 'Grenada',
								'Guadeloupe'               => 'Guadeloupe',
								'Guam'                     => 'Guam',
								'Guatemala'                => 'Guatemala',
								'Guinea'                   => 'Guinea',
								'Guinea-Bissau'            => 'Guinea-Bissau',
								'Guyana'                   => 'Guyana',
								'Haiti'                    => 'Haiti',
								'Heard and Mc Donald Islands' => 'Heard and Mc Donald Islands',
								'Holy See (Vatican City State)' => 'Holy See (Vatican City State)',
								'Honduras'                 => 'Honduras',
								'Hong Kong'                => 'Hong Kong',
								'Hungary'                  => 'Hungary',
								'Iceland'                  => 'Iceland',
								'India'                    => 'India',
								'Indonesia'                => 'Indonesia',
								'Iran (Islamic Republic of)' => 'Iran (Islamic Republic of)',
								'Iraq'                     => 'Iraq',
								'Ireland'                  => 'Ireland',
								'Israel'                   => 'Israel',
								'Italy'                    => 'Italy',
								'Jamaica'                  => 'Jamaica',
								'Japan'                    => 'Japan',
								'Jordan'                   => 'Jordan',
								'Kazakhstan'               => 'Kazakhstan',
								'Kenya'                    => 'Kenya',
								'Kiribati'                 => 'Kiribati',
								"Korea, Democratic People's Republic of" => "Korea, Democratic People's Republic of",
								'Korea, Republic of'       => 'Korea, Republic of',
								'Kuwait'                   => 'Kuwait',
								'Kyrgyzstan'               => 'Kyrgyzstan',
								"Lao, People's Democratic Republic" => "Lao, People's Democratic Republic",
								'Latvia'                   => 'Latvia',
								'Lebanon'                  => 'Lebanon',
								'Lesotho'                  => 'Lesotho',
								'Liberia'                  => 'Liberia',
								'Libyan Arab Jamahiriya'   => 'Libyan Arab Jamahiriya',
								'Liechtenstein'            => 'Liechtenstein',
								'Lithuania'                => 'Lithuania',
								'Luxembourg'               => 'Luxembourg',
								'Macau'                    => 'Macau',
								'Macedonia, The Former Yugoslav Republic of' => 'Macedonia, The Former Yugoslav Republic of',
								'Madagascar'               => 'Madagascar',
								'Malawi'                   => 'Malawi',
								'Malaysia'                 => 'Malaysia',
								'Maldives'                 => 'Maldives',
								'Mali'                     => 'Mali',
								'Malta'                    => 'Malta',
								'Marshall Islands'         => 'Marshall Islands',
								'Martinique'               => 'Martinique',
								'Mauritania'               => 'Mauritania',
								'Mauritius'                => 'Mauritius',
								'Mayotte'                  => 'Mayotte',
								'Mexico'                   => 'Mexico',
								'Micronesia, Federated States of' => 'Micronesia, Federated States of',
								'Moldova, Republic of'     => 'Moldova, Republic of',
								'Monaco'                   => 'Monaco',
								'Mongolia'                 => 'Mongolia',
								'Montserrat'               => 'Montserrat',
								'Morocco'                  => 'Morocco',
								'Mozambique'               => 'Mozambique',
								'Myanmar'                  => 'Myanmar',
								'Namibia'                  => 'Namibia',
								'Nauru'                    => 'Nauru',
								'Nepal'                    => 'Nepal',
								'Netherlands'              => 'Netherlands',
								'Netherlands Antilles'     => 'Netherlands Antilles',
								'New Caledonia'            => 'New Caledonia',
								'New Zealand'              => 'New Zealand',
								'Nicaragua'                => 'Nicaragua',
								'Niger'                    => 'Niger',
								'Nigeria'                  => 'Nigeria',
								'Niue'                     => 'Niue',
								'Norfolk Island'           => 'Norfolk Island',
								'Northern Mariana Islands' => 'Northern Mariana Islands',
								'Norway'                   => 'Norway',
								'Oman'                     => 'Oman',
								'Pakistan'                 => 'Pakistan',
								'Palau'                    => 'Palau',
								'Panama'                   => 'Panama',
								'Papua New Guinea'         => 'Papua New Guinea',
								'Paraguay'                 => 'Paraguay',
								'Peru'                     => 'Peru',
								'Philippines'              => 'Philippines',
								'Pitcairn'                 => 'Pitcairn',
								'Poland'                   => 'Poland',
								'Portugal'                 => 'Portugal',
								'Puerto Rico'              => 'Puerto Rico',
								'Qatar'                    => 'Qatar',
								'Reunion'                  => 'Reunion',
								'Romania'                  => 'Romania',
								'Russian Federation'       => 'Russian Federation',
								'Rwanda'                   => 'Rwanda',
								'Saint Kitts and Nevis'    => 'Saint Kitts and Nevis',
								'Saint Lucia'              => 'Saint Lucia',
								'Saint Vincent and the Grenadines' => 'Saint Vincent and the Grenadines',
								'Samoa'                    => 'Samoa',
								'San Marino'               => 'San Marino',
								'Sao Tome and Principe'    => 'Sao Tome and Principe',
								'Saudi Arabia'             => 'Saudi Arabia',
								'Senegal'                  => 'Senegal',
								'Seychelles'               => 'Seychelles',
								'Sierra Leone'             => 'Sierra Leone',
								'Singapore'                => 'Singapore',
								'Slovakia (Slovak Republic)' => 'Slovakia (Slovak Republic)',
								'Slovenia'                 => 'Slovenia',
								'Solomon Islands'          => 'Solomon Islands',
								'Somalia'                  => 'Somalia',
								'South Africa'             => 'South Africa',
								'South Georgia and the South Sandwich Islands' => 'South Georgia and the South Sandwich Islands',
								'Spain'                    => 'Spain',
								'Sri Lanka'                => 'Sri Lanka',
								'St. Helena'               => 'St. Helena',
								'St. Pierre and Miquelon'  => 'St. Pierre and Miquelon',
								'Sudan'                    => 'Sudan',
								'Suriname'                 => 'Suriname',
								'Svalbard and Jan Mayen Islands' => 'Svalbard and Jan Mayen Islands',
								'Swaziland'                => 'Swaziland',
								'Sweden'                   => 'Sweden',
								'Switzerland'              => 'Switzerland',
								'Syrian Arab Republic'     => 'Syrian Arab Republic',
								'Taiwan, Province of China' => 'Taiwan, Province of China',
								'Tajikistan'               => 'Tajikistan',
								'Tanzania, United Republic of' => 'Tanzania, United Republic of',
								'Thailand'                 => 'Thailand',
								'Togo'                     => 'Togo',
								'Tokelau'                  => 'Tokelau',
								'Tonga'                    => 'Tonga',
								'Trinidad and Tobago'      => 'Trinidad and Tobago',
								'Tunisia'                  => 'Tunisia',
								'Turkey'                   => 'Turkey',
								'Turkmenistan'             => 'Turkmenistan',
								'Turks and Caicos Islands' => 'Turks and Caicos Islands',
								'Tuvalu'                   => 'Tuvalu',
								'Uganda'                   => 'Uganda',
								'Ukraine'                  => 'Ukraine',
								'United Arab Emirates'     => 'United Arab Emirates',
								'United Kingdom'           => 'United Kingdom',
								'United States'            => 'United States',
								'United States Minor Outlying Islands' => 'United States Minor Outlying Islands',
								'Uruguay'                  => 'Uruguay',
								'Uzbekistan'               => 'Uzbekistan',
								'Vanuatu'                  => 'Vanuatu',
								'Venezuela'                => 'Venezuela',
								'Vietnam'                  => 'Vietnam',
								'Virgin Islands (British)' => 'Virgin Islands (British)',
								'Virgin Islands (U.S.)'    => 'Virgin Islands (U.S.)',
								'Wallis and Futuna Islands' => 'Wallis and Futuna Islands',
								'Western Sahara'           => 'Western Sahara',
								'Yemen'                    => 'Yemen',
								'Yugoslavia'               => 'Yugoslavia',
								'Zambia'                   => 'Zambia',
								'Zimbabwe'                 => 'Zimbabwe',
							),

						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_education',
						array(
							'label'          => __( 'For education', 'buddyboss-app' ),
							'desc'           => '',
							'type'           => 'checkbox',
							'label_checkbox' => __( 'Enable', 'buddyboss-app' ),
							'value'          => '',
							'placeholder'    => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_marketing_optout',
						array(
							'label'          => __( 'Marketing Opt-Out', 'buddyboss-app' ),
							'desc'           => '',
							'type'           => 'checkbox',
							'label_checkbox' => __( 'Enable', 'buddyboss-app' ),
							'value'          => '',
							'placeholder'    => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_content_guidelines',
						array(
							'label'          => __( 'Content Guidelines', 'buddyboss-app' ),
							'desc'           => '',
							'type'           => 'checkbox',
							'label_checkbox' => __( 'Enable', 'buddyboss-app' ),
							'value'          => '',
							'placeholder'    => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_us_export_laws',
						array(
							'label'          => __( 'US Export Laws', 'buddyboss-app' ),
							'desc'           => '',
							'type'           => 'checkbox',
							'label_checkbox' => __( 'Enable', 'buddyboss-app' ),
							'value'          => '',
							'placeholder'    => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.app_changelog',
						array(
							'label'       => __( 'Changelog', 'buddyboss-app' ),
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',

						)
					)
				);

				/**
				 * Do later.
				 *
				 * @todo: create our own documentation and change the link
				 */
				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.youtube_api_key',
						array(
							'label'  => __( 'API Key for YouTube', 'buddyboss-app' ),
							'desc'   => __(
								'Log into <a href="https://console.developers.google.com/project">Google Developers</a>, go to <a href="https://console.developers.google.com/project">Google API Console</a> and create a project. Once the project is created, copy the API Key. You can find step-by-step instructions in <a href="https://www.slickremix.com/docs/get-api-key-for-youtube/">this tutorial</a>.',
								'buddyboss-app'
							),
							'type'   => 'text',
							'value'  => '',
							'filter' => function ( $value ) {
								return $value;
							},
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_publish_android_firebase',
						array(
							'type'  => 'heading',
							'value' => __( 'Firebase Settings', 'buddyboss-app' ),
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.google_services_json',
						array(
							'label'    => __( 'Google Play: Authorize JSON Key File', 'buddyboss-app' ),
							'desc'     => '',
							'type'     => '\BuddyBossApp\Admin\Fields\Upload',
							'settings' => array(
								'allowed_ext'    => array( 'json' ),
								'show_file_link' => true,
							),
							'value'    => '',
						)
					)
				);
			}
		}
	}

	/**
	 * Saves the settings
	 *
	 * @param array $settings Publish settings.
	 *
	 * @return bool
	 */
	public function save_settings( $settings ) {
		ManageApp::instance()->update_app_settings( $settings );

		return true;
	}

	/**
	 * Gets the settings
	 */
	public function get_settings() {
		$settings = ManageApp::instance()->get_app_settings();

		return $settings;
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {
		InnerPublish::instance()->interval_publish_status_check(); // Refresh build check automatic.

		$build_option   = OuterPublish::instance()->get_publish_options();
		$app_env        = 'publish';
		$settings_array = array();

		if ( ( isset( $build_option['ios'][ $app_env ]['status'] ) && 'cancelled' === $build_option['ios'][ $app_env ]['status'] ) || ( ! isset( $build_option['ios'][ $app_env ]['status'] ) ) ) {
			$settings_array['ios'] = __( 'Apple App Store', 'buddyboss-app' );
		}

		if ( ( isset( $build_option['android'][ $app_env ]['status'] ) && 'cancelled' === $build_option['android'][ $app_env ]['status'] ) || ( ! isset( $build_option['android'][ $app_env ]['status'] ) ) ) {
			$settings_array['android'] = __( 'Google Play Store', 'buddyboss-app' );
		}

		$tabs = $settings_array;

		return $tabs;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {
		$default      = 'ios';
		$setting      = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Renders the setting screen.
	 */
	public function render_screen() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			echo '<p>' . esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) . '</p>';

			return false;
		}

		$bbapp            = \BuddyBossApp\ManageApp::instance()->get_app();
		$maintenance_mode = \BuddyBossApp\ManageApp::instance()->is_appcenter_under_maintenance();

		if ( empty( $bbapp ) || empty( $bbapp['verified'] ) || is_wp_error( $maintenance_mode ) ) {
			include bbapp()->plugin_dir . 'views/settings/publish/publish-no-connection.php';

			return false;
		} elseif ( ! empty( $bbapp ) && ! empty( $bbapp['bbapp_site_type'] ) && 'secondary' === $bbapp['bbapp_site_type'] ) {
			include bbapp()->plugin_dir . '/views/settings/publish/connect-dev.php';

			return false;
		} elseif ( ! is_wp_error( $maintenance_mode ) && true === $maintenance_mode ) {
			include bbapp()->plugin_dir . '/views/settings/publish/maintenance-mode.php';

			return false;
		}

		echo '<div class="wrap wrap--model">';
		echo '<h1 class="wp-heading wp-heading--key">' . esc_html__( 'BuddyBoss Publishing Service', 'buddyboss-app' );
		echo '</h1>';

		if ( ! empty( $_GET['agree-publish-term'] ) && empty( get_option( 'bbapp_publish_term_condition' ) ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			update_option( 'bbapp_publish_term_condition', 1 );
		}

		if ( ! empty( $_GET['agree-publish-term'] ) && empty( get_option( 'bbapp_publish_term_condition' ) ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			include bbapp()->plugin_dir . 'views/settings/publish/publish-terms.php';
		} else {

			\BuddyBossApp\Admin\SetupAdmin::instance()->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab() );

			InnerPublish::instance()->render();
		}
		echo '</div>';
	}

	/**
	 * Sync Data with Appcenter.
	 */
	public function sync_data_with_appcenter() {
		$publish_options = OuterPublish::instance()->get_publish_options();
		if ( ! empty( $publish_options ) && empty( $publish_options['publish_requested'] ) ) {
			return;
		}

		$options = isset( $publish_options['publish_requested'] ) ? $publish_options['publish_requested'] : array();

		if ( ! empty( $options ) ) {
			foreach ( $options as $platform => $publish ) {

				if ( empty( $publish['publish_id'] ) ) {
					continue;
				}

				if ( isset( $publish['publish_status'] ) ) {

					$publish_id = isset( $publish['publish_id'] ) ? $publish['publish_id'] : '';

					// Skip, if publish id is empty.
					if ( empty( $publish_id ) ) {
						continue;
					}

					delete_transient( 'get_publish_status_' . $publish_id );
					delete_site_transient( 'get_publish_status_' . $publish_id );
					$publish_status = \BuddyBossApp\Admin\Publish\Publish::instance()->get_publish_status( $publish_id );

					if ( ! empty( $publish_status ) ) {
						$options[ $platform ]['publish_status'] = $publish_status;
					}
				}

				$publish_detail = \BuddyBossApp\Admin\Publish\Publish::instance()->get_publish_details( $publish_id, true );

				if ( 'ios' === $platform && ! is_wp_error( $publish_detail ) && ! empty( $publish_detail ) ) {
					if ( ! empty( $publish_detail['status'] ) && ! in_array(
						$publish_detail['status'],
						array(
							'completed',
							'cancelled',
							'connection_lost',
						),
						true
					) ) {
						$apple_id = isset( $publish_detail['apple_id'] ) ? $publish_detail['apple_id'] : '';
						\BuddyBossApp\AppStores\Apple::instance()->display_app_data( true, $apple_id );
						InnerPublish::instance()->update_data_into_appcenter( $publish_id, 'ios', $apple_id );
					}
				}

				if ( 'android' === $platform && ! is_wp_error( $publish_detail ) && ! empty( $publish_detail ) ) {
					if ( ! empty( $publish_detail['status'] ) && ! in_array(
						$publish_detail['status'],
						array(
							'completed',
							'cancelled',
							'connection_lost',
						),
						true
					) ) {
						\BuddyBossApp\AppStores\Android::instance()->get_release_data( true );
						InnerPublish::instance()->update_data_into_appcenter( $publish_id, 'android', '' );
					}
				}
			}
		}

		$publish_options['publish_requested'] = $options;

		OuterPublish::instance()->update_publish_options( $publish_options );
	}

}
