<?php
/**
 * Holds functionality to show the notification list in backend.
 *
 * @package BuddyBossApp\Admin\Notification
 */

namespace BuddyBossApp\Admin\Notification;

use BuddyBossApp\Admin\Notification;
use BuddyBossApp\Notification\AutomaticNotificationList;
use BuddyBossApp\Notification\ManualNotificationList;

/**
 * Admin notification class.
 */
class Notifications {

	/**
	 * Class instance.
	 *
	 * @var false $instance
	 */
	private static $instance;

	/**
	 * If it's current page.
	 *
	 * @var bool $is_current_page
	 */
	private $is_current_page;

	/**
	 * Messages.
	 *
	 * @var array $messages
	 */
	private $messages;

	/**
	 * Notification list.
	 *
	 * @var object
	 */
	private $notification_list;

	/**
	 * Get the instance of the class.
	 *
	 * @return Notifications
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Class construct function.
	 */
	public function __construct() {
	}

	/**
	 * Load function.
	 *
	 * @return void
	 */
	public function load() {

		add_action( 'admin_init', array( $this, 'admin_init' ), 15 );
		add_action( 'load-buddyboss-app_page_bbapp-notification', array( $this, 'get_screen_options' ) );
		add_filter( 'set-screen-option', array( $this, 'bbapp_admin_screen_options' ), 10, 3 );
	}

	/**
	 * Functions tells & sets thats if current page is one where it's will render.
	 *
	 * @param bool $set To show the page or not.
	 *
	 * @return bool
	 */
	public function will_render( $set = false ) {

		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Admin init hook.
	 *
	 * @return void
	 */
	public function admin_init() {

		if ( $this->will_render() ) {

			if ( 'manual' === Notification::instance()->get_current_sub_sub_tab() ) {
				$this->notification_list = new ManualNotificationList();
			} elseif ( 'automatic' === Notification::instance()->get_current_sub_sub_tab() ) {
				$this->notification_list = new AutomaticNotificationList();
			}
		}
	}

	/**
	 * Renders the branding screen
	 *
	 * @return bool|mixed
	 */
	public function render() {

		global $plugin_page;

		if ( ! current_user_can( 'manage_options' ) ) {
			echo '<p>' . esc_html__( "You don't have permissions to access this page.", 'buddyboss-app' ) . '</p>';

			return false;
		}

		$push_id_request = ( isset( $_REQUEST['push_id'] ) ) ? true : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$action_request  = ( ! empty( $_REQUEST['action'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['action'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( $push_id_request && 'view_push' === $action_request ) {
			return $this->notification_list->pushSingleScreen();
		}

		$form_cls    = 'bbapp-manual-notification';
		$form_method = 'post';
		if ( 'automatic' === Notification::instance()->get_current_sub_sub_tab() ) {
			$form_cls    = 'bbapp-automatic-notification';
			$form_method = 'get';
		}
		?>
		<form method="<?php echo esc_attr( $form_method ); ?>" class="bbapp-notification-form <?php echo esc_attr( $form_cls ); ?> ">
			<input type="hidden" name="page" value="<?php echo esc_attr( $plugin_page ); ?>"/>
			<?php
			$this->notification_list->prepare_items();
			if ( ! empty( $this->notification_list->_args['is_search_box'] ) ) {
				$this->notification_list->search_box( 'Search', 'search' );
			}
			$this->notification_list->display();
			?>
		</form>
		<?php
	}

	/**
	 * Output the error or update messages to render.
	 */
	public function show_messages() {
		if ( ! empty( $this->messages ) ) {
			foreach ( $this->messages as $message ) {
				printf( '<div id="message" class="%1$s"><p>%2$s</p></div>', esc_attr( $message['type'] ), esc_html( $message['message'] ) );
			}
		}
	}

	/**
	 * Register the screen options.
	 */
	public function get_screen_options() {
		$screen = get_current_screen();

		if ( ! is_object( $screen ) || 'buddyboss-app_page_bbapp-notification' !== $screen->id ) {
			return;
		}

		// Loop through all the options and add a screen option for each.
		$option = 'per_page';
		$args   = array(
			'label'   => __( 'Number of items per page:', 'buddyboss-app' ),
			'default' => 10,
			'option'  => 'push_notifications_per_page',
		);

		add_screen_option( $option, $args );

		// Loop through all the options and add a screen option for each.
		$option = 'recipients_per_page';
		$args   = array(
			'label'   => __( 'Number of recipients per page:', 'buddyboss-app' ),
			'default' => 10,
			'option'  => 'push_recipients_per_page',
		);

		add_screen_option( $option, $args );
	}

	/**
	 * Handle save/update of screen options for notification.
	 *
	 * @param string $value     Will always be false unless another plugin filters it first.
	 * @param string $option    Screen option name.
	 * @param string $new_value Screen option form value.
	 *
	 * @return string|int Option value. False to abandon update.
	 */
	public function bbapp_admin_screen_options( $value, $option, $new_value ) {
		if ( 'push_notifications_per_page' !== $option && 'push_recipients_per_page' !== $option ) {
			return $value;
		}

		// Per page.
		$new_value = (int) $new_value;
		if ( $new_value < 1 || $new_value > 999 ) {
			return $value;
		}

		return $new_value;
	}
}
