<?php
/**
 * Compose notification.
 *
 * @package BuddyBossApp\Admin\Notification\Compose
 */

namespace BuddyBossApp\Admin\Notification;

use BuddyBossApp\Notification\Push;
use BuddyBossApp\UserSegment;

/**
 * Class Compose
 */
class Compose {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Is current page.
	 *
	 * @var $is_current_page
	 */
	private $is_current_page;

	/**
	 * Admin notice messages.
	 *
	 * @var string|array $messages Message list.
	 */
	private $messages;

	/**
	 * User per page
	 *
	 * @var int $per_page_user User per page.
	 */
	private $per_page_user = 20;

	/**
	 * Get the instance of the class.
	 *
	 * @return Compose
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Compose constructor.
	 */
	public function __construct() {
	}

	/**
	 * Instance method.
	 */
	public function load() {

		add_action( 'admin_init', array( $this, 'admin_init' ) );

		/**
		 * Ajax Hooks
		 */
		add_action( 'wp_ajax_nopriv_bbapp_push_submit', array( $this, 'push_submit' ) );
		add_action( 'wp_ajax_bbapp_push_submit', array( $this, 'push_submit' ) );
		add_action( 'wp_ajax_bbapp_search_users', array( $this, 'bbapp_search_users' ) );
		add_action( 'wp_ajax_bbapp_output_selected_users', array( $this, 'output_selected_users' ) );
		add_action( 'wp_ajax_bbapp_output_selected_users_count', array( $this, 'output_selected_users_count' ) );
		add_action( 'wp_ajax_bbapp_remove_specific_user', array( $this, 'remove_specific_user' ) );
		add_action( 'wp_ajax_bbapp_add_specific_user', array( $this, 'add_specific_user' ) );

	}

	/**
	 * Functions tells & sets that's if current page is one where it's will render.
	 *
	 * @param bool $set Current page flag.
	 *
	 * @return bool
	 */
	public function will_render( $set = false ) {

		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Load Compose.
	 *
	 * @return void
	 */
	public function admin_init() {

		if ( $this->will_render() ) {

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_die( esc_html__( "You don't have permissions to make changes.", 'buddyboss-app' ) );
			}

			$this->push_compose(); // make a unique compose ID when visit the compose page.

		}
	}

	/**
	 * Renders the branding screen
	 *
	 * @return bool|mixed
	 */
	public function render() {

		if ( ! current_user_can( 'manage_options' ) ) {
			echo sprintf( '<p>%s</p>', esc_html__( 'You don\'t have permissions to access this page.', 'buddyboss-app' ) );

			return false;
		}

		include bbapp()->plugin_dir . 'views/push-notifications/new.php';

	}

	/**
	 * Entry data fields.
	 *
	 * @param array|string $field   Fields data.
	 * @param string       $default default value.
	 *
	 * @return array|string
	 */
	public function entry_data( $field, $default = '' ) {
		return isset( $_POST[ $field ] ) ? bbapp_input_clean( wp_unslash( $_POST[ $field ] ) ) : $default; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Missing
	}

	/**
	 * Renders Warnings, Success, Notices on admin screens.
	 */
	public function show_messages() {
		$settings_updated = ! empty( $_GET['settings-updated'] ) ? bbapp_input_clean( wp_unslash( $_GET['settings-updated'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		if ( ! empty( $settings_updated ) ) {
			$this->messages[] = array(
				'type'    => 'updated',
				'message' => __( 'Settings Updated!', 'buddyboss-app' ),
			);
		}

		if ( ! empty( $this->messages ) ) {
			foreach ( $this->messages as $message ) {
				if ( 'error' === $message['type'] ) {
					$message['message'] = sprintf( '<b>%1$s</b>%2$s', __( 'Push Notification Error:', 'buddyboss-app' ), $message['message'] );
				}
				echo sprintf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( $message['type'] ), esc_html( $message['message'] ) );
			}
		}
	}

	/**
	 * Look & create new compose draft for sending push notification.
	 * Main reason is to have unique id for each compose to store unique value on multiple compose screen on same user.
	 */
	public function push_compose() {

		$current_page = isset( $_GET['page'] ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		$setting      = isset( $_GET['setting'] ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		// check if user is on push compose screen.
		if ( 'bbapp-notification' === $current_page && 'new' === $setting ) {

			$push_compose_id = $this->get_compose_id();
			if ( empty( $push_compose_id ) ) {
				$uniqueid = time();
				wp_safe_redirect( admin_url( 'admin.php?page=bbapp-notification&setting=new&compose_id=' . $uniqueid ) );
				exit();
			}
		}

	}

	/**
	 * Returns push compose id.
	 *
	 * @return int
	 */
	public function get_compose_id() {
		$push_compose_id = isset( $_GET['compose_id'] ) ? (int) bbapp_input_clean( wp_unslash( $_GET['compose_id'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
		if ( empty( $push_compose_id ) ) {
			$push_compose_id = $this->entry_data( 'compose_id', 0 );
		}

		return $push_compose_id;
	}


	/**
	 * Send error when push submit.
	 *
	 * @param string $error Error.
	 * @param bool   $focus Focus.
	 */
	public function send_json_error_push_submit( $error, $focus = false ) {
		wp_send_json_error(
			array(
				'focus' => $focus,
				'error' => $error,
			)
		);
	}

	/**
	 * Manual Push Notification Process Code.
	 * This code triggers when manual push form is submitted.
	 */
	public function push_submit() {

		if ( ! isset( $_POST['_wpnonce'] ) ) {
			$this->send_json_error_push_submit( __( "You don't have permission to access this page.", 'buddyboss-app' ) );
		}

		check_admin_referer( 'manual_push_nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			$this->send_json_error_push_submit( __( "You don't have permission to access this page.", 'buddyboss-app' ) );

			return false;
		}

		$primary_text   = $this->entry_data( 'primary_text' );
		$secondary_text = $this->entry_data( 'secondary_text' );
		$sent_as        = $this->entry_data( 'send_as', 1 );
		$send_to        = $this->entry_data( 'send_to', 'all_users' );
		$link_url       = $this->entry_data( 'link_url' );

		if ( empty( $secondary_text ) ) {
			$this->send_json_error_push_submit( __( 'Message cannot be empty.', 'buddyboss-app' ), 'bbapp-message-text' );

			return;
		}

		if ( ! empty( $link_url ) && ! bbapp_is_valid_url( $link_url ) ) {
			$this->send_json_error_push_submit( __( 'Given link URL is not a valid URL format.', 'buddyboss-app' ), 'bbapp-link-url' );

			return;
		}

		$user_ids = $this->get_users_by_send_to();

		if ( empty( $user_ids['total'] ) ) {
			$this->send_json_error_push_submit( __( 'No recipients found with current send to user rules.', 'buddyboss-app' ), 'bbapp-send-to' );

			return;
		}

		$schedule_date = false;
		$schedule      = $this->entry_data( 'schedule' );

		if ( isset( $schedule ) && '1' === $schedule ) {
			if ( empty( $_POST['date_schedule'] ) ) {
				$this->send_json_error_push_submit( __( 'Please select the schedule date.', 'buddyboss-app' ), 'schedule_date_container' );
			}

			if ( empty( $_POST['date_schedule_time'] ) ) {
				$this->send_json_error_push_submit( __( 'Please select the schedule date.', 'buddyboss-app' ), 'schedule_date_container' );
			}

			$date_schedule          = $this->entry_data( 'date_schedule' );
			$date_schedule_time     = $this->entry_data( 'date_schedule_time' );
			$schedule_date          = $date_schedule . ' ' . $date_schedule_time; // this date should be WordPress local timezone.
			$date_schedule_timezone = $this->entry_data( 'date_schedule_timezone' );

			if ( ! isset( $date_schedule_timezone ) ) {
				$this->send_json_error_push_submit( __( '#1 Schedule timezone is invalid.', 'buddyboss-app' ), 'schedule_date_container' );

				return;
			}

			// Update Timezone Value if offset is given.
			if ( ! empty( $date_schedule_timezone ) && preg_match( '/^UTC[+-]/', $date_schedule_timezone ) ) {
				$offset     = preg_replace( '/UTC\+?/', '', $date_schedule_timezone );
				$offset_sec = $offset * 60 * 60;
				$tz         = timezone_name_from_abbr( '', $offset_sec, 1 );

				if ( false === $tz ) {
					$tz = timezone_name_from_abbr( '', $offset_sec, 0 );
				}

				$_POST['date_schedule_timezone'] = $tz;
			}

			try {
				$date_schedule_timezone = $this->entry_data( 'date_schedule_timezone' );
				$datetime               = date_create( $schedule_date, new \DateTimeZone( $date_schedule_timezone ) );
				if ( ! $datetime ) {
					$this->send_json_error_push_submit( __( '#2 Schedule timezone is invalid.', 'buddyboss-app' ), 'schedule_date_container' );

					return;
				}
				$datetime->setTimezone( new \DateTimeZone( 'UTC' ) );
				$utc_schedule_date = $datetime->format( 'Y-m-d H:i:s' );

			} catch ( \Exception $e ) {
				$this->send_json_error_push_submit( __( '#3 Schedule timezone is invalid.', 'buddyboss-app' ), 'schedule_date_container' );

				return;
			}

			$schedule_date = gmdate( 'Y-m-d H:i:s', strtotime( $utc_schedule_date ) );

			// Validate that if date is in future.
			if ( strtotime( $schedule_date ) < strtotime( gmdate( 'Y-m-d H:i:s' ) ) ) {
				$this->send_json_error_push_submit( __( 'Your selected date must be in the future.', 'buddyboss-app' ), 'schedule_date_container' );

				return;
			}
		}

		$segment_id = $this->get_user_segment_id();

		$target_type = 'specific_users';
		$segment     = array();
		if ( 'all_users' === $send_to ) {
			$target_type = 'all_users';
		}

		if ( 'filter_users' === $send_to ) {
			$target_type = 'filtered_users';
			$segment     = \BuddyBossApp\UserSegment::instance()->get_filter_data_from_post_var();
		}

		if ( 'specific_users' === $send_to ) {
			$target_type = 'specific_users';
			$segment_id  = $this->get_user_segment_id( 'specific' );
		}

		/**
		 * Make sure segment id is there.
		 */
		if ( 'all_users' !== $send_to && empty( $segment_id ) ) {
			$this->send_json_error_push_submit( __( 'No recipients found with current send to user rules.', 'buddyboss-app' ), 'bbapp-send-to' );

			return;
		}

		$date_schedule_timezone = $this->entry_data( 'date_schedule_timezone' );
		$notification           = array(
			'primary_text'        => sanitize_text_field( $primary_text ),
			'secondary_text'      => sanitize_textarea_field( $secondary_text ),
			'sent_as'             => $sent_as,
			'agent'               => 'manual_push_notification',
			'data'                => array(
				'segment_id' => $segment_id,
				'timezone'   => $date_schedule_timezone,
				'segment'    => $segment,
			),
			'push_data'           => array(),
			'normal_notification' => true,
			/**
			 * Target type.
			 *
			 * @todo: Have to finish a real target type support for schedule . So we will use this to know if user has to be taken live from DB or from user_ids array.
			 */
			'target_type'         => $target_type, // specific_users , all_users , filtered_users.
		);

		/**
		 * Add the Link URL into Push Data.
		 */
		if ( ! empty( $link_url ) ) {
			$notification['push_data']['link'] = $link_url;
			$notification['data']['link']      = $link_url;
		}

		if ( $schedule_date ) {
			$notification['date_schedule'] = $schedule_date;
		}

		/**
		 * Notification Filters
		 * Using these filter we can alter the compose manual push notification data.
		 */

		$notification['push_data'] = apply_filters( 'bbapp_manual_push_notification_compose_push_data', $notification['push_data'], $notification );
		$notification['data']      = apply_filters( 'bbapp_manual_push_notification_compose_data', $notification['data'], $notification );

		// Validate Data Return from Filters.
		foreach ( array( $notification['push_data'], $notification['data'] ) as $_validate_data ) {
			/**
			 * Using is_wp_error we can trigger manual error on compose screen.
			 */
			if ( is_wp_error( $_validate_data ) ) { // When it returned the wp error.
				/**
				 * Send json error for push submit.
				 *
				 * @var $get_callback_value \WP_Error
				 */
				$this->send_json_error_push_submit( $_validate_data->get_error_message() );

				return;

			} elseif ( ! is_array( $_validate_data ) || ! isset( $_validate_data ) ) { // when it's not wp error nor a valid format.
				$this->send_json_error_push_submit( __( 'Notification has not been sent because the selected notification type has an invalid value format.', 'buddyboss-app' ) );

				return;
			}
		}

		$segments = UserSegment::instance();
		if ( 'all_users' !== $target_type ) {
			$notification['data']['send_to'] = $segments->get_total_users( $segment_id, true );
		} else {
			$fetch_users                     = Push::instance()->get_all_users( 1, 1 );
			$notification['data']['send_to'] = $fetch_users['total'];
		}

		$push_id = $this->save_to_db( $notification );

		// Push has been added into db it will trigger automatic.
		if ( is_wp_error( $push_id ) || empty( $push_id ) ) {
			$this->send_json_error_push_submit( __( 'Unexpected error while sending push notification.', 'buddyboss-app' ) );

			return;
		}
		$delivery = false;
		if ( ! $schedule_date ) {
			// Start the delivery of push notification.
			$delivery = Push::instance()->do_push_delivery( $push_id );
		}

		if ( is_wp_error( $delivery ) ) {
			$this->send_json_error_push_submit( $delivery->get_error_message() );

			return;
		}

		$compose_id = $this->get_compose_id();

		if ( ! $schedule_date ) {
			wp_send_json_success( array( 'url' => admin_url( "admin.php?page=bbapp-notification&setting=new&notification_sent=1&compose_id={$compose_id}" ) ) );
		} else {
			wp_send_json_success( array( 'url' => admin_url( "admin.php?page=bbapp-notification&setting=new&notification_sent=2&compose_id={$compose_id}" ) ) );
		}

		exit();
	}

	/**
	 * Returns the admin send as users.
	 *
	 * @return array
	 */
	public function admin_users_options() {
		$options = array();
		$users   = get_users( 'orderby=nicename&role=administrator' );

		if ( ! empty( $users ) && ! is_wp_error( $users ) ) {
			foreach ( $users as $user ) {
				$options[] = array(
					'id'     => $user->ID,
					'name'   => $user->display_name,
					'avatar' => get_avatar_url( $user->ID ),
				);
			}
		}

		return $options;
	}

	/**
	 * Allow users to search & display on select2 on push screen.
	 * Ajax Handler.
	 */
	public function bbapp_search_users() {

		$term = ! empty( $_REQUEST['term'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['term'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended

		$args = array(
			'search'         => '*' . $term['term'] . '*',
			'search_columns' => array( 'user_login', 'display_name' ),
		);

		/**
		 * Suspended users exclude from users list when manual notification prepare.
		 */
		if ( function_exists( 'bp_is_active' ) && bp_is_active( 'moderation' ) ) {
			$args['exclude'] = $this->get_moderation_suspended_users();
		}

		// Get the results.
		$response   = array();
		$items      = array();
		$user_query = new \WP_User_Query( $args );
		$users      = $user_query->get_results();
		if ( ! empty( $users ) && ! is_wp_error( $users ) ) {
			foreach ( $users as $user ) {
				$user_name = bbaap_get_user_display_name( $user->ID );
				if ( function_exists( 'bp_moderation_is_user_blocked' ) && bp_moderation_is_user_blocked( $user->ID ) ) {
					$user_name = $user->display_name;
				}
				$items[] = array(
					'id'            => $user->ID,
					'name'          => $user_name,
					'avatar'        => get_avatar_url( $user->ID ),
					'avatarHtml'    => get_avatar( $user->ID, 16 ),
					'link'          => admin_url( "user-edit.php?user_id={$user->ID}" ),
					'loginName'     => $user->user_login,
					'app_installed' => false,
				);
			}
		}

		$response['items'] = $items;

		wp_send_json( $response );

	}

	/**
	 * Function to get all suspended users list.
	 *
	 * @since 1.6.8
	 * @return null|array|object|\stdClass|void
	 */
	public function get_moderation_suspended_users() {
		global $wpdb;
		$bp = buddypress();

		return $wpdb->get_col( $wpdb->prepare( "SELECT item_id FROM {$bp->moderation->table_name} WHERE item_type = %s AND user_suspended = 1", \BP_Suspend_Member::$type ) ); // phpcs:ignore
	}

	/**
	 * Returns User ID based on Send To Information Form Provided.
	 * Expects Sent to Field Information In $_POST
	 *
	 * @param int $page  Page.
	 * @param int $limit Limit.
	 *
	 * @return array
	 */
	public function get_users_by_send_to( $page = 1, $limit = 20 ) {
		$send_to = $this->entry_data( 'send_to', 'all_users' );
		if ( 'all_users' === $send_to ) {

			$users = Push::instance()->get_all_users( $page, $limit );

			return array(
				'users' => $users['users'],
				'total' => $users['total'],
			);

		} else {

			$segment_id = $this->get_user_segment_id();

			if ( 'specific_users' === $send_to ) {
				$segment_id = $this->get_user_segment_id( 'specific' );
			}

			$user_segment = \BuddyBossApp\UserSegment::instance();

			$user_ids   = $user_segment->get_users( $segment_id, $page, $limit, true );
			$user_count = $user_segment->get_total_users( $segment_id, true );

			return array(
				'users' => $user_ids,
				'total' => $user_count,
			);

		}

		return array(
			'users' => array(),
			'total' => 0,
		);

	}

	/**
	 * Return the Selected Users Based on Send To Field.
	 */
	public function get_selected_users() {

		$temp_work_key = '_abb_push_temp_users_work_' . $this->get_compose_id();
		$user_ids      = (array) get_transient( $temp_work_key );

		return $user_ids;

	}

	/**
	 * Return the ID to be used on user segment.
	 *
	 * @param string $postfix Post fix.
	 *
	 * @return string
	 */
	public function get_user_segment_id( $postfix = '' ) {
		$compose_id = $this->get_compose_id();
		if ( ! empty( $postfix ) ) {
			$postfix = '_' . $postfix;
		}

		return 'push_' . $compose_id . $postfix;
	}


	/**
	 * Output the list based on given users array on param.
	 *
	 * @param array $user_ids     User isds.
	 *
	 * @param int   $current_page Current page.
	 * @param int   $total_pages  Total pages.
	 *
	 * @return string
	 */
	private function get_user_list_markup( $user_ids, $current_page = 1, $total_pages = 1 ) {

		if ( empty( $user_ids ) ) {
			return false;
		}
		$send_to = $this->entry_data( 'send_to', 'all_users' );

		$users = get_users( array( 'include' => $user_ids ) );

		// Sometime get users returns duplicates users let's fix them.
		$_tmp = array();
		foreach ( $users as $user ) {
			$_tmp[ $user->ID ] = $user;
		}
		$users = $_tmp;
		unset( $_tmp );

		ob_start();

		echo sprintf( "<div id='bbapp_push_users' data-current_page='%s'>", esc_attr( $current_page ) );

		foreach ( $users as $user ) {
			$link      = admin_url( "user-edit.php?user_id={$user->ID}" );
			$user_name = get_user_meta( $user->ID, 'first_name', true ) . ' ' . get_user_meta( $user->ID, 'last_name', true );
			if ( empty( trim( $user_name ) ) ) {
				$user_name = $user->user_login;
			}
			?>
			<div class="username">

				<?php if ( 'all_users' !== $send_to ) : ?>
					<a href="#" title="<?php esc_attr__( 'Remove', 'buddyboss-app' ); ?>" data-user_id="<?php echo esc_attr( $user->ID ); ?>" class="bbapp-delete-user">
						<span class="dashicons dashicons-no-alt"></span>
					</a>
				<?php endif; ?>

				<span>
					<?php echo get_avatar( $user->ID, 16 ); ?>
					<a class="bbapp-username-link" href="<?php echo esc_url( $link ); ?>" target="_blank">
						<?php echo esc_html( $user_name ); ?>
					</a>
				</span>

			</div>
			<?php
		}

		echo '</div>';

		echo '<ul class="pagination">';

		$big = 999999999; // need an unlikely integer.

		echo wp_kses_post(
			paginate_links(
				array(
					'base'    => str_replace( $big, '%#%', '#' . $big ),
					'format'  => '?paged=%#%',
					'current' => max( 1, $current_page ),
					'total'   => $total_pages,
					'type'    => 'list',
				)
			)
		);

		echo '</ul>';

		return ob_get_clean();
	}


	/**
	 * Return the bbapp filters user ids list with pagination.
	 */
	public function output_selected_users() {
		$current_page = $this->entry_data( 'page', 1 );
		$current_page = max( $current_page, 1 );

		$user_ids = $this->get_users_by_send_to( $current_page, $this->per_page_user );

		if ( empty( $user_ids['total'] ) ) {
			wp_send_json_error(
				array(
					'type'    => 'error',
					'message' => __(
						'No members found.',
						'buddyboss-app'
					),
				)
			);
		}

		$max_page = ceil( $user_ids['total'] / $this->per_page_user );

		$markup = $this->get_user_list_markup( $user_ids['users'], $current_page, $max_page );

		wp_send_json_success(
			array(
				'total'    => (int) $user_ids['total'],
				'max_page' => $max_page,
				'html'     => $markup,
			)
		);

	}

	/**
	 * Return the bbapp filters user ids list with pagination.
	 */
	public function output_selected_users_count() {
		$user_ids = $this->get_users_by_send_to( 1, 1 );
		wp_send_json_success( array( 'total' => (int) $user_ids['total'] ) );
	}

	/**
	 * Add specific user id into user segment.
	 */
	public function add_specific_user() {
		$user_id = $this->entry_data( 'add_user_id', 0 );
		if ( false === (bool) $this->user_has_device_token( $user_id ) ) {
			$user = get_userdata( $user_id );
			wp_send_json_error(
				array(
					'type'    => 'error',
					'message' => sprintf( '<b>%s</b> %s', esc_html( $user->display_name ), __( 'cannot receive push notifications until they have downloaded and logged into the app.', 'buddyboss-app' ) ),
				)
			);
		}

		$segment_id = $this->get_user_segment_id( 'specific' );

		if ( empty( $segment_id ) ) {
			wp_send_json_error(
				array(
					'type'    => 'error',
					'message' => __(
						'Invalid user segment. Please refresh the page and try again.',
						'buddyboss-app'
					),
				)
			);
		}

		$user_segment = \BuddyBossApp\UserSegment::instance();
		$current_page = $this->entry_data( 'page', 1 );
		$current_page = max( $current_page, 1 );

		$add_user = $user_segment->add_user( $segment_id, $user_id );

		if ( ! $add_user ) {
			wp_send_json_error(
				array(
					'type'    => 'error',
					'message' => __( 'There was an error while deleting the user. The user might have been deleted already.', 'buddyboss-app' ),
				)
			);
		}

		$user_ids = $this->get_users_by_send_to( $current_page, $this->per_page_user );
		$max_page = ceil( $user_ids['total'] / $this->per_page_user );

		$markup = $this->get_user_list_markup( $user_ids['users'], $current_page, $max_page );

		wp_send_json_success(
			array(
				'total' => (int) $user_ids['total'],
				'html'  => $markup,
			)
		);
	}

	/**
	 * Specific user check the device token exists or not.
	 *
	 * @param int $user_id User id.
	 *
	 * @return false|string|null
	 */
	public function user_has_device_token( $user_id = 0 ) {
		global $wpdb;
		$devices_table = bbapp_get_network_table( 'bbapp_user_devices' );
		if ( ! empty( $user_id ) ) {
			$wp_user_capabilities = bbapp_get_global_db_prefix() . 'capabilities'; // this will force to collect user from correct blog.

			// Query only users who have device token registered.
			return $wpdb->get_var( $wpdb->prepare( "SELECT DISTINCT u.ID as user_id FROM {$wpdb->users} as u, {$wpdb->usermeta} as um , {$devices_table} as devices WHERE u.ID = um.user_id AND u.ID = devices.user_id AND u.user_status = '0' AND devices.user_id=%s AND (um.meta_key = %s) order by u.id ASC", $user_id, $wp_user_capabilities ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}

		return false;

	}

	/**
	 * Remove the specific user from user segment.
	 */
	public function remove_specific_user() {

		$segment_id = $this->get_user_segment_id();
		$send_to    = $this->entry_data( 'send_to', 'all_users' );
		if ( 'specific_users' === $send_to ) {
			$segment_id = $this->get_user_segment_id( 'specific' );
		}

		if ( empty( $segment_id ) ) {
			wp_send_json_error(
				array(
					'type'    => 'error',
					'message' => __(
						'Invalid user segment. Please refresh the page and try again.',
						'buddyboss-app'
					),
				)
			);
		}

		$user_segment = \BuddyBossApp\UserSegment::instance();
		$user_id      = $this->entry_data( 'remove_user_id', 0 );
		$current_page = $this->entry_data( 'page', 1 );
		$current_page = max( $current_page, 1 );

		$remove_user = $user_segment->remove_user( $segment_id, $user_id );

		if ( ! $remove_user ) {
			wp_send_json_error(
				array(
					'type'    => 'error',
					'message' => __(
						'There was an error while deleting the user. The user might have been deleted already.',
						'buddyboss-app'
					),
				)
			);
		}

		$user_ids = $this->get_users_by_send_to( $current_page, $this->per_page_user );
		$max_page = ceil( $user_ids['total'] / $this->per_page_user );

		$markup = $this->get_user_list_markup( $user_ids['users'], $current_page, $max_page );

		wp_send_json_success( array( 'html' => $markup ) );

	}

	/**
	 * Store Manual Push Notification.
	 *
	 * @param array $notification Push notification data.
	 *
	 * @return false|true|int
	 */
	public function save_to_db( $notification ) {
		global $wpdb;

		$defaults = array(
			'primary_text'             => '',
			'secondary_text'           => '',
			'user_ids'                 => array(),
			'sent_as'                  => 1,
			'agent'                    => 'event', // whether this notification was triggered manually or by any event.
			'status'                   => 'pending',
			'date_updated'             => current_time( 'mysql', 1 ),
			'date_created'             => current_time( 'mysql', 1 ),
			'blog_id'                  => get_current_blog_id(),
			'topic'                    => '',
			'target_type'              => 'specific_users', // specific_users , all , filter.
			'date_schedule'            => false,
			'data'                     => array(),
			'push_data'                => array(),
			'normal_notification'      => false, // should create notification or not.
			'normal_notification_data' => array(),
		);

		$notification = wp_parse_args( $notification, $defaults );

		if ( 'all_users' !== $notification['target_type'] ) {
			$segment_id = ( isset( $notification['data']['segment_id'] ) ) ? $notification['data']['segment_id'] : false;

			if ( empty( $segment_id ) ) {
				return false;
			}
		}

		$table_name = bbapp_get_network_table( 'bbapp_push_notifications' );

		// 0 means all users, -1 means multiple users.
		$target = ( count( $notification['user_ids'] ) === 1 ) ? reset( $notification['user_ids'] ) : - 1;

		$_data                             = $notification['data'];
		$_data['user_ids']                 = $notification['user_ids'];
		$_data['push_data']                = $notification['push_data'];
		$_data['normal_notification']      = $notification['normal_notification'];
		$_data['normal_notification_data'] = $notification['normal_notification_data'];

		$data = array(
			'primary_text'   => stripcslashes( $notification['primary_text'] ),
			'secondary_text' => stripcslashes( $notification['secondary_text'] ),
			'target'         => $target,
			'sent_as'        => $notification['sent_as'],
			'data'           => maybe_serialize( $_data ),
			'agent'          => $notification['agent'],
			'status'         => 'processing', // default status when push trigger.
			'date_updated'   => $notification['date_updated'],
			'date_created'   => $notification['date_created'],
			'blog_id'        => $notification['blog_id'],
			'target_type'    => $notification['target_type'],
		);

		if ( $notification['date_schedule'] ) {
			$data['status']        = 'pending';
			$data['is_schedule']   = 1;
			$data['date_schedule'] = $notification['date_schedule'];
			// Expire date so it shouldn't be treated after that.
			$data['date_expire'] = gmdate( 'Y-m-d H:i:s', strtotime( '+3 hours', strtotime( $notification['date_schedule'] ) ) );
		}

		$insert_push_db = $wpdb->insert( $table_name, $data ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery

		if ( ! $insert_push_db ) {
			return false;
		}

		return $wpdb->insert_id;
	}
}
