<?php
/**
 * Holds the functionality related the notification.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\Admin\Notification\Compose;
use BuddyBossApp\Admin\Notification\Notifications;
use BuddyBossApp\Notification\Notification as AppNotification;

/**
 * Notification class.
 */
class Notification {

	/**
	 * Class instance.
	 *
	 * @var bool $instance
	 */
	private static $instance;

	/**
	 * Notification option name.
	 *
	 * @var string $option_setting
	 */
	public $option_setting = 'bbapp_push_notif_options';

	/**
	 * Settings form variable.
	 *
	 * @var $settings_form
	 */
	private $settings_form;

	/**
	 * Get the instance of the class.
	 *
	 * @return Notification
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Construct function.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Load function.
	 *
	 * @return void
	 */
	public function load() {

		Notifications::instance();
		Compose::instance();

		add_action( 'admin_init', array( $this, 'load_init' ), 9 );
		add_action( 'admin_init', array( $this, 'load_service_push_admin' ) );

	}

	/**
	 * Function to load current backend page.
	 *
	 * @return void
	 */
	public function load_init() {
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'admin.php' === SetupAdmin::instance()->get_page_now() && 'bbapp-notification' === $page_get ) {
			if ( 'list' === $this->get_current_sub_tab() && bbapp_is_admin_page() ) {
				Notifications::instance()->will_render( true ); // tell the class that it will render.
			}

			if ( 'new' === $this->get_current_sub_tab() && bbapp_is_admin_page() ) {
				Compose::instance()->will_render( true ); // tell the class that it will render.
			}
		}
	}

	/**
	 * Load Settings.
	 */
	public function load_settings() {

		$this->settings_form = new FormProvider( 'push_notif_settings' );

		// holds app settings storage handle.
		$settings_provider = new SettingProvider( 'push_notif_settings' );

		// register app settings storing & receiving.
		$settings_provider->save_setting_hook( array( $this, 'save_settings' ) );
		$settings_provider->get_setting_hook( array( $this, 'get_settings' ) );

		// Link Provider In Form Provider.
		$this->settings_form->link_setting_provider( $settings_provider );

		add_action( 'admin_init', array( $this->settings_form, 'save' ) );

		/**
		 * Push Notification Settings
		 */
		$this->settings_form->add_field(
			$settings_provider->field(
				false,
				array(
					'type'    => 'table-open',
					'classes' => 'buddyboss-app-admin-card bbapp-admin-push_notification',
				)
			)
		);

		/* translators: %s: Firebase settings page link. */
		$skip_desc = ( ! bbapp_get_app_push_firebase_key() ) ? sprintf( __( 'In order to use push notifications in the app, please add your <a href="%s">Firebase Server Key</a>.', 'buddyboss-app' ), bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=firebase' ) ) : '';

		$this->settings_form->add_field(
			$settings_provider->field(
				'app_push_notification_skip_active_member',
				array(
					'type'         => 'heading',
					'value'        => esc_html__( 'Push Notifications', 'buddyboss-app' ),
					'desc'         => $skip_desc,
					'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=121838' ),
					'icon'         => 'bb-icon-bf bb-icon-bell',
				)
			)
		);

		if ( bbapp_get_app_push_firebase_key() ) {
			// Skip active member.
			$this->settings_form->add_field(
				$settings_provider->field(
					'push.skip_active_members',
					array(
						'label'          => esc_html__( 'Skip Active Members', 'buddyboss-app' ),
						'label_checkbox' => esc_html__( 'Don\'t send push notifications when members are active on a device', 'buddyboss-app' ),
						'desc'           => esc_html__( 'When a member is actively using your site on any device, they won\'t receive push notifications. Once a member is inactive, they will begin receiving push notifications after a short delay.', 'buddyboss-app' ),
						'type'           => 'checkbox',
						'value'          => false,
					)
				)
			);
		}

		$this->settings_form->add_field(
			$settings_provider->field(
				false,
				array(
					'type'    => 'table-open',
					'classes' => 'buddyboss-app-admin-card',
				),
			)
		);

		$this->settings_form->add_field(
			$settings_provider->field(
				'app_push_notification_heading',
				array(
					'type'         => 'heading',
					'value'        => esc_html__( 'Automatic Notifications', 'buddyboss-app' ),
					'desc'         => '',
					'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=121838' ),
					'icon'         => 'bb-icon-bf bb-icon-bell',
				)
			)
		);

		if ( defined( 'BP_PLATFORM_VERSION' ) && bp_is_active( 'notifications' ) && ! bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {
			$subscriptions_settings_lists = AppNotification::instance()->get_push_notifications_subs_settings();
			if ( count( $subscriptions_settings_lists ) > 2 ) {
				$this->settings_form->add_field(
					$settings_provider->field(
						'app_push_notification_bbp_active',
						array(
							'type'  => 'desc',
							'value' => sprintf(
							/* translators: %s: Notifications article link. */
								__( 'When using BuddyBoss Platform, we recommend registering all custom notifications as a BuddyBoss notification so that members can manage the notification in their notification preferences. <a href="%s" target="_blank">Learn more</a>', 'buddyboss-app' ),
								esc_url(
									bp_get_admin_url(
										add_query_arg(
											array(
												'page'    => 'bp-help',
												'article' => 125373,
											),
											'admin.php'
										)
									)
								)
							),
							'class' => 'bbapp-platform-active-notice',
						)
					)
				);
			}
		}

		if ( ! bbapp_get_app_push_firebase_key() ) {
			/* translators: %s: Firebase settings page link. */
			$desc = sprintf( __( 'In order to use push notifications in the app, please add your <a href="%s">Firebase Server Key</a>.', 'buddyboss-app' ), bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=firebase' ) );
		} else {
			/* translators: %s: Notifications page link. */
			$desc = sprintf(
			/* translators: %s: Notifications article link. */
				__( 'Configure which notifications will be sent automatically as push notifications. You can send manual push notification to your users on the <a href="%s">Push Notifications</a> page.', 'buddyboss-app' ),
				esc_url(
					bbapp_get_admin_url(
						add_query_arg(
							array(
								'page'    => 'bbapp-notification',
								'setting' => 'new',
							),
							'admin.php'
						)
					)
				)
			);
		}


		$this->settings_form->add_field(
			$settings_provider->field(
				'push_notification_bbapp_desc',
				array(
					'type'  => 'desc',
					'value' => $desc,
				)
			)
		);

		if ( bbapp_get_app_push_firebase_key() ) {
			$subscriptions_settings_lists = AppNotification::instance()->get_push_notifications_subs_settings();
			if ( count( $subscriptions_settings_lists ) > 2 ) {

				foreach ( $subscriptions_settings_lists as $subscriptions_settings_list ) {

					if ( in_array( $subscriptions_settings_list['name'], array( '', 'announcements' ), true ) ) {
						continue;
					}

					if ( ! empty( $subscriptions_settings_list['settings'] ) ) {
						$label    = $subscriptions_settings_list['label'] . __( ' Notification', 'buddyboss-app' );
						$class    = 'section_element';
						$last_key = array_key_last( $subscriptions_settings_list['settings'] );
						foreach ( $subscriptions_settings_list['settings'] as $setting_key => $subscriptions_setting ) {
							if ( $setting_key === $last_key ) {
								$class = 'section_last_element';
							}
							$this->settings_form->add_field(
								$settings_provider->field(
									$setting_key,
									array(
										'label'          => $label,
										'label_checkbox' => $subscriptions_setting['admin_label'],
										'type'           => 'checkbox',
										'class'          => $class,
										'value'          => true,
									)
								)
							);
							$label = '';
						}
					}
				}
			} else {
				$this->settings_form->add_field(
					$settings_provider->field(
						'app_push_no_notification_notice',
						array(
							'type'  => 'desc',
							'value' => sprintf( __( '<i class="bb-icon-info-circle"></i> No custom notifications registered.', 'buddyboss-app' ) ),
							'class' => 'bbapp-no-push-notice',
						)
					)
				);
			}

			if ( bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {


				$this->settings_form->add_field(
					$settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-notification-notice',
						)
					)
				);

				$this->settings_form->add_field(
					$settings_provider->field(
						'app_push_notification_custom_notice ',
						array(
							'type'  => 'desc',
							/* translators: %s: Notifications extending article link. */
							'value' => sprintf( __( 'App push notifications registered using <a href="%s" target="_blank">custom code</a> will show up in this section.', 'buddyboss-app' ), 'https://www.buddyboss.com/resources/dev-docs/app-development/extending-the-buddyboss-app-plugin/creating-automatic-push-notifications/' ),
						)
					)
				);
			}

			if ( defined( 'BP_PLATFORM_VERSION' ) && bp_is_active( 'notifications' ) && ! bbapp_is_platform_notification_preferences_legacy_mode_enabled() ) {

				if ( count( $subscriptions_settings_lists ) > 2 ) {
					$this->settings_form->add_field(
						$settings_provider->field(
							'app_push_notification_bbp_active',
							array(
								'type'  => 'desc',
								'value' => sprintf(
								/* translators: %s: Notifications article link. */
									__( 'When using BuddyBoss Platform, we recommend registering all custom notifications as a BuddyBoss notification so that members can manage the notification in their notification preferences. <a href="%s" target="_blank">Learn more</a>', 'buddyboss-app' ),
									esc_url(
										bp_get_admin_url(
											add_query_arg(
												array(
													'page' => 'bp-help',
													'article' => 125373,
												),
												'admin.php'
											)
										)
									)
								),
								'class' => 'bbapp-platform-active-notice',
							)
						)
					);
				}

				$this->settings_form->add_field(
					$settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-notification-notice',
						)
					)
				);

				$this->settings_form->add_field(
					$settings_provider->field(
						'bb_push_notification_custom_notice',
						array(
							'type'  => 'desc',
							/* translators: %s: Notifications page link. */
							'value' => sprintf( __( 'You can configure BuddyBoss Platform notifications in the <a href="%s" target="_blank">Notifications</a> settings.', 'buddyboss-app' ), admin_url( 'admin.php?page=bp-settings&tab=bp-notifications' ) ),
						)
					)
				);
			}
		}
	}

	/**
	 * Saves the settings
	 *
	 * @param array $settings Setting name.
	 *
	 * @return bool
	 */
	public function save_settings( $settings ) {
		bbapp_set_network_option( $this->option_setting, $settings );

		return true;
	}

	/**
	 * Gets the settings
	 */
	public function get_settings() {
		$settings = bbapp_get_network_option( $this->option_setting );

		return $settings;
	}

	/**
	 * Function will call _init_admin of each app selected push service on init_admin.
	 */
	public function load_service_push_admin() {
		$push_instance = bbapp_get_app_push_instance();

		if ( $push_instance ) {
			$push_instance->service_init_admin();
		}
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {

		$return = array();

		if ( bbapp_is_admin_page() ) {
			$return['list'] = __( 'Push Notifications', 'buddyboss-app' );
			$return['new']  = __( 'Send New', 'buddyboss-app' );
		}

		return $return;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {

		$default = 'list';

		foreach ( $this->sub_settings() as $k => $v ) {
			$default = $k;
			break;
		}

		$setting      = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_sub_settings() {

		$return = array();

		if ( bbapp_is_admin_page() ) {
			$return['manual']    = __( 'Manual', 'buddyboss-app' );
			$return['automatic'] = __( 'Automatic', 'buddyboss-app' );
		}

		return $return;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_sub_tab() {
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'bbapp-notification' === $page_get ) {
			$default      = 'manual';
			$setting      = ( isset( $_GET['sub-setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['sub-setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$sub_settings = $this->sub_sub_settings();

			if ( isset( $sub_settings[ $setting ] ) ) {
				return $setting;
			}

			return $default;
		}

		return null;
	}

	/**
	 * Render Setting Form.
	 */
	public function render_settings() {
		$this->settings_form->render_form();
	}

	/**
	 * Renders the setting screen.
	 */
	public function render_screen() {

		echo '<div class="wrap">';

		$this->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab() );

		if ( 'list' === $this->get_current_sub_tab() ) {
			Notifications::instance()->render();
		}

		if ( 'new' === $this->get_current_sub_tab() ) {
			Compose::instance()->render();
		}

		echo '</div>';

	}

	/**
	 * Function to display sub tabs.
	 *
	 * @param string $sub_settings Page subs settings.
	 * @param string $current_tab  Page current tab.
	 * @param string $param_key    Parameter key.
	 *
	 * @return bool
	 */
	public function render_subtabs( $sub_settings, $current_tab, $param_key = 'setting' ) {

		// only render something when there is more than 1 sub menu.
		if ( empty( $sub_settings ) || count( $sub_settings ) < 2 ) {
			return false;
		}

		$push_id_get = isset( $_GET['push_id'] ) ? true : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( $push_id_get ) {
			?>
			<a class="bbapp-back-to-notification" href="<?php echo esc_url( admin_url( 'admin.php?page=bbapp-notification' ) ); ?>">
				<?php esc_html_e( 'Back to Notifications', 'buddyboss-app' ); ?>
			</a>
			<?php
		} else {
			$page = isset( $_GET['page'] ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if ( 'new' === $current_tab ) {
				$sub_tab       = 'list';
				$tab_title     = $sub_settings['new'];
				$sub_tab_title = $sub_settings['list'];

			} else {
				$sub_tab       = 'new';
				$tab_title     = $sub_settings['list'];
				$sub_tab_title = $sub_settings['new'];
			}
			?>
			<div class="flex" style="position: relative">
				<h1 class="wp-heading-inline"><?php echo esc_html( $tab_title ); ?></h1>
				<div class="flex align-center bbapp-sub-nav">
					<a class="page-title-action button button-primary" href="
					<?php
					echo esc_url(
						add_query_arg(
							array(
								'page'     => $page,
								$param_key => $sub_tab,
							),
							bbapp_get_admin_url( 'admin.php' )
						)
					);
					?>
					"><?php echo esc_html( $sub_tab_title ); ?></a>
					<a class="page-title-action button" href="
					<?php
					echo esc_url(
						add_query_arg(
							array(
								'page'    => 'bbapp-settings',
								'setting' => 'push_notifications',
							),
							bbapp_get_admin_url( 'admin.php' )
						)
					);
					?>
							">
						<?php esc_html_e( 'View Settings', 'buddyboss-app' ); ?>
					</a>
				</div>
				<a class="button button-secondary bbapp-tutorial-btn" target="_blank" href="<?php echo esc_url( 'https://www.buddyboss.com/resources/docs/app/push-notifications/' ); ?>">
					<?php esc_html_e( 'View Tutorial', 'buddyboss-app' ); ?>
				</a>
			</div>
			<?php
		}

		$setting_get = ! empty( $_GET['setting'] ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$push_id_get = isset( $_GET['push_id'] ) ? true : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( 'new' !== $setting_get && ! $push_id_get ) {
			\BuddyBossApp\Admin\SetupAdmin::instance()->render_subtabs( $this->sub_sub_settings(), $this->get_current_sub_sub_tab(), 'sub-setting' );
		}
	}
}
