<?php
/**
 * Holds the menu related functionality.
 *
 * @package BuddyBossApp\Admin\Menus
 */

use BuddyBossApp\AppSettings;
use BuddyBossApp\Common\IconPicker;

/**
 * Function get menu type label.
 *
 * @param string $menu_type Nav menu type.
 *
 * @since 1.5.2.1
 * @return mixed
 */
function bbapp_get_menu_type_label( $menu_type = 'tabbar' ) {
	$menu_types_label = array(
		'tabbar'    => __( 'Tab Bar', 'buddyboss-app' ),
		'headerbar' => __( 'Header Bar', 'buddyboss-app' ),
		'more'      => __( 'More Menu', 'buddyboss-app' ),
	);

	return $menu_types_label[ $menu_type ];
}

/**
 * Function to get menu type.
 *
 * @param string $menu_type Default menu name.
 *
 * @since 1.5.2.1
 * @return mixed|string
 */
function bbapp_get_menu_type( $menu_type = 'tabbar' ) {
	return \BuddyBossApp\Menu::instance()->get_current_menu( $menu_type );
}


/**
 * Get code app menus.
 *
 * @since 1.5.2.1
 * @return array|bool
 */
function bbapp_get_core_appmenu() {
	/**
	 * Get all menus.
	 *
	 * @since 1.5.2.1
	 */
	$all_menus  = apply_filters( 'bbapp_get_all_menus', array() );
	$core_menus = ! empty( $all_menus['core'] ) ? $all_menus['core'] : array();

	return $core_menus;
}


/**
 * Function to get all nav menus based menu type.
 *
 * @param string $menu_type Nav menu type.
 *
 * @since 1.5.2.1
 * @return array|false|mixed|void
 *
 * @note Required for updated_30() function.
 * @todo: will remove in after some release.
 */
function bbapp_get_main_menus( $menu_type = null ) {
	_deprecated_function(
		'bbapp_get_main_menus',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->get_menus()'
	);

	$main_menus = get_db_main_menus();
	if ( ! empty( $main_menus[ $menu_type ] ) ) {
		$main_menus = $main_menus[ $menu_type ];
	} else {
		$main_menus = array();
	}

	if ( ! isset( $main_menus["default-{$menu_type}"] ) ) {
		$main_menus["default-{$menu_type}"] = bbapp_get_default_menu( $menu_type );
	}

	return $main_menus;
}

/**
 * Default nav menu.
 *
 * @param string $menu_type menu type.
 *
 * @return array
 */
function bbapp_get_default_menu( $menu_type ) {
	$menu_data = array(
		'id'            => "default-{$menu_type}",
		'label'         => __( 'Default', 'buddyboss-app' ),
		'login_state'   => 'logged-in',
		'access_groups' => array(),
		'priority'      => 99,
		'is_default'    => true,
		'menu_type'     => $menu_type,
	);

	if ( 'tabbar' === $menu_type ) {
		$menu_data['headerbar_id'] = 'default-headerbar';
	}

	if ( 'headerbar' === $menu_type ) {
		$menu_data['title_position']     = 'left';
		$menu_data['more_menu_position'] = 'tabbar';
		$menu_data['tabbar_id']          = 'default-tabbar';
	}

	return $menu_data;
}

/**
 * Function to get database nav menu.
 *
 * @since 1.5.2.1
 * @return array|false|mixed|void
 *
 * @note Required for updated_30() function.
 * @todo: will remove in after some release.
 */
function get_db_main_menus() {

	$main_menus = bbapp_get_option( 'bbapp_menus' );
	if ( empty( $main_menus ) ) {
		$main_menus = array();
	}

	return $main_menus;
}

/**
 * Function to return the app menu items.
 *
 * @param string $nav_id    Nav id.
 * @param string $menu_type Nav menu type.
 *
 * @since 1.5.2.1
 * @return array|bool
 */
function bbapp_get_menus_items( $nav_id, $menu_type = 'tabbar' ) {

	_deprecated_function(
		'bbapp_get_menus_items',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->get_menu_items()'
	);

	$args = array(
		'menu_id'   => $nav_id,
		'menu_type' => $menu_type,
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->get_menu_items( $args );
}

/**
 * Function to check if more menu can add or not.
 *
 * @since 1.5.2.1
 */
function can_more_menu_add() {

	_deprecated_function(
		'can_more_menu_add',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->can_more_menu_add()'
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->can_more_menu_add();
}

/**
 * Function to get db nav menu items.
 *
 * @param string $nav_id Nav id.
 *
 * @since 1.5.2.1
 * @return array|false|mixed|void
 *
 * @note Required for updated_30() function.
 * @todo: will remove in after some release.
 */
function get_db_menu_items( $nav_id ) {
	$menu_items = bbapp_get_option( "bbapp_menus_{$nav_id}" );

	if ( empty( $menu_items ) ) {
		$menu_items = array();
	}

	return $menu_items;
}

/**
 * Function to get single menu based on menu id.
 *
 * @param string $nav_id    Nav id.
 * @param string $menu_type Nav menu type.
 *
 * @since 1.5.2.1
 * @return false|mixed
 */
function bbapp_get_main_menu( $nav_id, $menu_type ) {

	_deprecated_function(
		'bbapp_get_main_menu',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->get_menu()'
	);

	$main_menus = bbapp_get_main_menus( $menu_type );

	if ( isset( $main_menus[ $nav_id ] ) ) {
		return $main_menus[ $nav_id ];
	}

	return false;
}

/**
 * Function to update db nav menu.
 *
 * @param array $nav_menus Nav menus.
 *
 * @since 1.5.2.1
 * @return bool
 */
function update_db_main_menus( $nav_menus = array() ) {

	_deprecated_function(
		'update_db_main_menus',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->update_menus()'
	);

	return bbapp_set_option( 'bbapp_menus', $nav_menus );
}

/**
 * Function to update db nav items.
 *
 * @param string $nav_id     Nav id.
 * @param array  $menu_items Nav items.
 *
 * @since 1.5.2.1
 * @return bool
 */
function update_db_menu_items( $nav_id, $menu_items = array() ) {
	return bbapp_set_option( "bbapp_menus_{$nav_id}", $menu_items );
}

/**
 * Function to check is nav menu exists.
 * Returns true if it is; false otherwise.
 *
 * @param string $nav_id    Nav id.
 * @param string $menu_type Nav menu type.
 *
 * @since 1.5.2.1
 * @return bool Whether the menu exists.
 */
function bbapp_is_main_menu( $nav_id, $menu_type ) {

	_deprecated_function(
		'bbapp_is_main_menu',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->is_menu()'
	);

	$menu = \BuddyBossApp\Menus\MenuManager::instance()->get_menu( $nav_id, $menu_type );

	return $menu ? true : false;
}

/**
 * Function to check logout menu id.
 *
 * @param string $menu_type Nav menu type.
 *
 * @since 1.5.2.1
 * @return bool
 */
function bbapp_get_logged_out_menu_id( $menu_type ) {

	_deprecated_function(
		'bbapp_get_logged_out_menu_id',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->get_logout_menu_id()'
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->get_logout_menu_id( $menu_type );
}

/**
 * Function to delete a navigation Menu.
 *
 * @param string $nav_id    Nav id.
 * @param string $menu_type Nav menu type.
 *
 * @since 1.5.2.1
 * @return null|bool True on success, false or WP_Error object on failure.
 */
function bbapp_delete_main_menu( $nav_id, $menu_type ) {

	_deprecated_function(
		'bbapp_delete_main_menu',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->delete_menu()'
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->delete_menu( $nav_id );
}

/**
 * Function to delete nav menu items.
 *
 * @param string $nav_id Nav id.
 *
 * @since 1.5.2.1
 * @return bool
 */
function bbapp_delete_menu_items( $nav_id ) {

	_deprecated_function(
		'bbapp_delete_menu_items',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->delete_menu_items()'
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->delete_menu_items( $nav_id );
}

/**
 * Function check is default logged out.
 *
 * @param string $nav_id    Nav id.
 * @param string $menu_type Nav menu type.
 * @param object $nav_menu  Nav menu.
 *
 * @since 1.5.2.1
 * @return bool
 */
function bbapp_is_default_logged_out( $nav_id, $menu_type, $nav_menu ) {

	_deprecated_function(
		'bbapp_is_default_logged_out',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->is_default_logged_out()'
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->is_default_logged_out( $nav_id, $menu_type, $nav_menu );
}



/**
 * Function to check access group exists in menu.
 *
 * @param int    $access_group_id group id.
 * @param string $menu_type       menu type.
 *
 * @since 1.5.2.1
 * @return false|mixed
 */
function bb_access_controls_has_group_in_menu( $access_group_id, $menu_type ) {

	_deprecated_function(
		'bb_access_controls_has_group_in_menu',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->get_menu_by_access_group()'
	);

	$nav_menus = \BuddyBossApp\Menus\MenuManager::instance()->get_menu_by_access_group( $access_group_id, $menu_type );

	if ( ! empty( $nav_menus ) ) {
		return $nav_menus;
	}

	return false;
}

/**
 * Function to get default menu items.
 *
 * @since 2.1.90
 * @return array
 */
function bbapp_headerbar_action_menu_default_items() {

	_deprecated_function(
		'bbapp_headerbar_action_menu_default_items',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->headerbar_action_menu_default_items()'
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->headerbar_action_menu_default_items();
}

/**
 * Function to check headerbar menu exists.
 *
 * @param string $tabbar_id Tabbar id.
 *
 * @since 2.1.90
 * @return bool
 */
function bbapp_headerbar_from_tabbar( $tabbar_id ) {

	_deprecated_function(
		'bbapp_headerbar_from_tabbar',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->get_headerbar_menu_from_tabbar_id()'
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->get_headerbar_menu_from_tabbar_id( $tabbar_id );
}

/**
 * Function to get tabbar from headerbar.
 *
 * @param string $headerbar_id Headerbar id.
 *
 * @since 2.1.90
 * @return bool
 */
function bbapp_tabbar_from_headerbar( $headerbar_id ) {

	_deprecated_function(
		'bbapp_tabbar_from_headerbar',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->get_tabbar_menu_from_headerbar_id()'
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->get_tabbar_menu_from_headerbar_id( $headerbar_id );
}

/**
 * Function to get more menu position.
 *
 * @param string $nav_id Nav id.
 *
 * @since 2.1.90
 * @return string
 */
function bbapp_get_more_menu_position( $nav_id ) {

	_deprecated_function(
		'bbapp_get_more_menu_position',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->get_more_menu_position()'
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->get_more_menu_position( $nav_id );
}

/**
 * Function to get headerbar title position.
 *
 * @param string $nav_id Nav id.
 *
 * @since 2.1.90
 * @return string
 */
function bbapp_get_headerbar_title_position( $nav_id ) {

	_deprecated_function(
		'bbapp_get_headerbar_title_position',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->get_title_position()'
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->get_title_position( $nav_id );
}

/**
 * Function to get headerbar post button status.
 *
 * @param string $nav_id Nav id.
 *
 * @since 2.2.20
 * @return bool
 */
function bbapp_get_headerbar_post_button_status( $nav_id ) {

	_deprecated_function(
		'bbapp_get_headerbar_post_button_status',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->get_post_input_field_status()'
	);

	return \BuddyBossApp\Menus\MenuManager::instance()->get_post_input_field_status( $nav_id );
}

/**
 * Function to check whether tabbar exists or not.
 *
 * @since 2.1.90
 * @return bool
 */
function bbapp_is_tabbar_exist() {

	_deprecated_function(
		'bbapp_is_tabbar_exist',
		'2.4.10',
		'BuddyBossApp\Menus\MenuManager::instance()->is_tabbar_menu_exists()'
	);

	$menu_id = \BuddyBossApp\Menus\MenuManager::instance()->get_default_menu_id( 'tabbar' );

	return ! empty( $menu_id );
}
