<?php
/**
 * Holds menu related functionality.
 *
 * @package BuddyBossApp\Admin\Menus
 */

namespace BuddyBossApp\Admin\Menus;

use BuddyBossApp\Common\IconPicker;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Menus\MenuManager;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

/**
 * Menus class.
 */
class Menus {

	/**
	 * Class instance.
	 *
	 * @since 1.5.2.1
	 * @var object
	 */
	private static $instance;

	/**
	 * Error messages.
	 *
	 * @var $messages
	 */
	private $messages = array();

	/**
	 * Using Singleton, see instance()
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.5.2.1
	 *
	 * @return Menus
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Add related class and hooks
	 *
	 * @since 1.5.2.1
	 */
	public function init() {
		if ( ! in_array( trailingslashit( plugin_dir_path( __FILE__ ) ) . 'functions.php', get_included_files(), true ) ) {
			require_once trailingslashit( plugin_dir_path( __FILE__ ) ) . 'functions.php';
		}

		add_action( 'admin_init', array( $this, 'admin_init' ), 15 );
		add_action( 'wp_ajax_add-appmenu-item', array( $this, 'wp_ajax_add_menu_item' ) );
		add_action( 'wp_ajax_add_more_menu', array( $this, 'add_more_menu' ) );
		add_action( 'wp_ajax_get_tabbar_menu_count', array( $this, 'get_tabbar_menu_count' ) );
		add_action( 'wp_ajax_bbapp_create_default_menus', array( $this, 'ajax_create_default_menus' ) );
		add_action( 'bbapp_render_menu_metabox', array( $this, 'render_menu_metabox' ), 10 );
		add_action( 'admin_head', array( $this, 'app_icon_render_script' ) );
	}
	/**
	 * Renders the menu screen
	 *
	 * @since 1.5.2.1
	 * @return void
	 */
	public function render() {
		$this->action_notices();
		$this->show_messages();

		include bbapp()->plugin_dir . '/views/menus/menu.php';
	}

	/**
	 * Load Menus List.
	 *
	 * @since 1.5.2.1
	 *
	 * @return void
	 */
	public function admin_init() {
		// Allowed actions: add, update, delete.
		$action    = isset( $_REQUEST['action'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['action'] ) ) : 'edit'; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$menu_type = bbapp_get_menu_type();

		switch ( $action ) {
			case 'create':
				if ( ! isset( $_REQUEST['update-nav-appmenu-nonce'] ) ) {
					break;
				}

				check_admin_referer( 'update-nav_appmenu', 'update-nav-appmenu-nonce' );

				$this->create_nav_menu( $action, $menu_type, $_POST );

				do_action( 'bbapp_menu_created', $menu_type );
				break;
			case 'update':
				if ( ! isset( $_REQUEST['update-nav-appmenu-nonce'] ) ) {
					break;
				}

				check_admin_referer( 'update-nav_appmenu', 'update-nav-appmenu-nonce' );

				// The menu id of the current menu being edited.
				$nav_id = isset( $_REQUEST['menu'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['menu'] ) ) : '0'; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

				$this->update_nav_menu( $nav_id, $action, $menu_type, $_POST );

				do_action( 'bbapp_menu_updated', $menu_type );
				break;
			case 'app_menu_setting_save':
				if ( ! isset( $_REQUEST['update-nav-appmenu-setting-nonce'] ) ) {
					break;
				}

				check_admin_referer( 'update-nav-appmenu-setting', 'update-nav-appmenu-setting-nonce' );

				// Save Menu Settings.
				$appmenu_settings = ! empty( $_REQUEST['appmenu_settings'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['appmenu_settings'] ) ) : array(); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				$this->save_menu_settings( $action, $menu_type, $appmenu_settings );

				/**
				 * Triggers after menu settings saved.
				 *
				 * @param string $menu_type Menu type.
				 */
				do_action( 'bbapp_menu_setting_saved', $menu_type );
				break;
			case 'delete':
				if ( ! isset( $_REQUEST['delete_nav'] ) ) {
					break;
				}

				// The menu id of the current menu being edited.
				$nav_id = isset( $_REQUEST['menu'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['menu'] ) ) : '0'; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

				check_admin_referer( 'delete-nav_menu-' . $nav_id, 'delete_nav' );

				$this->delete_nav_menu( $nav_id, $action, $menu_type );

				/**
				 * Triggers after menu settings deleted.
				 *
				 * @param string $menu_type Menu type.
				 */
				do_action( 'bbapp_menu_setting_deleted', $menu_type );
				break;
			case 'save-menu-priority':
				if ( ! isset( $_REQUEST['bbapp_set_menu_priority'] ) ) {
					break;
				}

				if ( isset( $_REQUEST['save_priority'] ) ) {
					check_admin_referer( "bbapp_set_menu_priority{$menu_type}_nonce", 'bbapp_set_menu_priority' );
					$nav_menu_order = ! empty( $_REQUEST['nav_menu_order'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['nav_menu_order'] ) ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
					$this->save_menu_priority( $action, $menu_type, $nav_menu_order );

					/**
					 * Triggers after menu priority changed.
					 *
					 * @param string $menu_type Menu type.
					 */
					do_action( 'bbapp_menu_priority', $menu_type );
				}
				break;
		}
	}

	/**
	 * Function to show the menu action messages.
	 *
	 * @since 1.5.2.1
	 */
	public function action_notices() {
		$messages = array();

		if ( empty( $messages ) ) {
			if ( isset( $_REQUEST['created'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$messages[] = sprintf( __( 'The menu has been created successfully.', 'buddyboss-app' ) );
			}

			if ( isset( $_REQUEST['updated'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$messages[] = sprintf( __( 'The menu has been updated successfully.', 'buddyboss-app' ) );
			}

			if ( isset( $_REQUEST['deleted'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$messages[] = sprintf( __( 'The menu has been deleted successfully.', 'buddyboss-app' ) );
			}
		}

		if ( $messages ) {
			echo wp_kses_post( sprintf( '<div id="message" class="updated notice is-dismissible"><p>%s</p></div>', implode( ' ', $messages ) ) );
		}
	}

	/**
	 * Output the error or update messages to render.
	 *
	 * @since 1.5.2.1
	 */
	public function show_messages() {
		if ( ! empty( $this->messages ) && is_array( $this->messages ) ) {
			foreach ( $this->messages as $message ) {
				if ( isset( $message['type'] ) && isset( $message['message'] ) ) {
					printf( '<div id="message" class="%1$s"><p>%2$s</p></div>', esc_attr( $message['type'] ), wp_kses_post( $message['message'] ) );
				}
			}
		}
	}

	/**
	 * Function to validate the nav menu.
	 *
	 * @param array $_data POST data.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function validate_prepare_nav_data( $_data ) {
		$result             = array();
		$result['validate'] = true;
		$result['data']     = array();
		$menu_name          = ( ! empty( $_data['menu-name'] ) ) ? bbapp_input_clean( wp_unslash( $_data['menu-name'] ) ) : '';
		$login_state        = ( ! empty( $_data['login_status'] ) ) ? bbapp_input_clean( wp_unslash( $_data['login_status'] ) ) : '0';
		$access_groups      = ! empty( $_data['access_groups'] ) ? $_data['access_groups'] : array();

		if ( empty( $menu_name ) ) {
			$result['validate'] = false;

			$this->messages[] = array(
				'type'    => 'error notice is-dismissible',
				'message' => __( 'Please enter a valid menu name.', 'buddyboss-app' ),
			);
		}

		if (
			! empty( $_data['setting'] )
			&& 'tabbar' === $_data['setting']
			&& ! empty( $_data['appmenu-item'] )
			&& is_array( $_data['appmenu-item'] )
		) {
			$headerbar_id       = MenuManager::instance()->get_headerbar_menu_from_tabbar_id(
				intval( bbapp_input_clean( wp_unslash( $_data['menu'] ) ) )
			);
			$more_menu_position = MenuManager::instance()->get_more_menu_position( $headerbar_id );

			// Check if there's already a "more" menu item in the submitted data to avoid duplicates.
			$more_item_count = 0;
			foreach ( $_data['appmenu-item'] as $item ) {
				if ( ! empty( $item['object'] ) && 'more' === $item['object'] ) {
					++$more_item_count;
				}
			}

			// Simple validation: Only allow one "More" item when position is tabbar.
			if ( 'tabbar' === $more_menu_position && $more_item_count > 1 ) {
				$result['validate'] = false;
				$this->messages[]   = array(
					'type'    => 'error notice is-dismissible',
					'message' => __( 'Only one "More" menu item is allowed in the Tab Bar.', 'buddyboss-app' ),
				);
			}
		}

		if ( ! empty( $access_groups ) ) {
			foreach ( $access_groups as $access_group ) {
				$_nav_item = MenuManager::instance()->get_menu_by_access_group( $access_group, bbapp_get_menu_type() );

				if ( ! empty( $_nav_item ) && (string) $_nav_item->id !== (string) $_data['menu'] && $_nav_item->language_code === $_data['lang'] ) {
					$result['validate'] = false;

					$menu_url = add_query_arg(
						array(
							'page'    => 'bbapp-menus',
							'setting' => bbapp_get_menu_type(),
							'menu'    => $_nav_item->id,
						),
						admin_url( 'admin.php' )
					);

					$menu_label = isset( $_nav_item->menu_name ) ? $_nav_item->menu_name : 'Menu #' . $_nav_item->id;

					$this->messages[] = array(
						'type'    => 'error notice is-dismissible',
						'message' => sprintf(
							__( "This group is already assigned to <a href='%1\$s' target='_blank'>%2\$s</a>", 'buddyboss-app' ),
							esc_url( $menu_url ),
							esc_html( $menu_label )
						),
					);
				}
			}
		}

		if ( true === $result['validate'] ) {
			$result['data'] = array(
				'menu_name' => $menu_name,
				'menu_type' => bbapp_get_menu_type(),
			);

			if ( ! empty( $access_groups ) ) {
				$result['data']['access_groups'] = array_unique( array_filter( $access_groups ) );
			}

			if ( isset( $login_state ) ) {
				if ( '0' === $login_state ) {
					$result['data']['access_groups'] = array();
				}

				$result['data']['login_state'] = $login_state;
			}
		}

		return $result;
	}

	/**
	 * Get appropriate menu priority based on menu name and login state.
	 *
	 * @param string $menu_name   Menu name.
	 * @param int    $login_state Login state (1 for logged in, 0 for logged out).
	 * @param int    $priority    Optional. Priority if provided.
	 * @param string $menu_type   Optional. Menu type (tabbar, headerbar, more).
	 *
	 * @since 2.4.10
	 * @return int Menu priority.
	 */
	private function get_appropriate_menu_priority( $menu_name, $login_state, $priority = null, $menu_type = '' ) {
		// If priority is explicitly provided, use it first.
		if ( null !== $priority ) {
			return intval( $priority );
		}

		// If no menu type is provided, try to get it from the current context or default to 'tabbar'.
		if ( empty( $menu_type ) ) {
			// Get menu type from the current request if available or default to 'tabbar'.
			$menu_type = isset( $_REQUEST['setting'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['setting'] ) ) : 'tabbar'; //phpcs:ignore
		}

		// Use the MenuManager to get a dynamic priority based on menu type, name and login state.
		return MenuManager::instance()->get_next_menu_priority(
			$menu_type,
			$menu_name,
			$login_state
		);
	}

	/**
	 * This function to create nav menu.
	 *
	 * @param string $action    current action.
	 * @param string $menu_type current menu type. (tabbar|more).
	 * @param array  $_data     POST data.
	 *
	 * @since 1.5.2.1
	 */
	public function create_nav_menu( $action, $menu_type, $_data ) {
		if ( ! empty( $action ) && 'create' === $action ) {
			$result = $this->validate_prepare_nav_data( $_data );

			if ( true === $result['validate'] ) {
				// Prepare menu data for database insertion or update.
				$menu_data = array(
					'menu_type'   => $menu_type,
					'menu_name'   => $result['data']['menu_name'],
					'login_state' => isset( $result['data']['login_state'] ) && '0' === $result['data']['login_state'] ? 0 : 1,
				);

				// Handle access groups if present and not empty.
				if ( ! empty( $result['data']['access_groups'] ) ) {
					$menu_data['access_groups'] = $result['data']['access_groups'];
				}

				// Set appropriate priority based on menu type and name.
				$menu_data['priority'] = $this->get_appropriate_menu_priority(
					$menu_data['menu_name'],
					$menu_data['login_state'],
					isset( $result['data']['priority'] ) ? $result['data']['priority'] : null,
					$menu_type
				);

				// Insert the menu into the database and get the ID.
				$nav_id = MenuManager::instance()->create_menu( $menu_type, $menu_data );

				if ( ! empty( $nav_id ) && ! is_wp_error( $nav_id ) ) {
					// Save menu items if present in the request.
					if ( isset( $_REQUEST['appmenu-item'] ) ) { //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended
						$menu_items = bbapp_input_clean( wp_unslash( $_REQUEST['appmenu-item'] ) ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended

						// Save each menu item to the database with the appropriate data.
						$order = 0;
						foreach ( $menu_items as $index => $item ) {
							$item_data = array(
								'menu_id'    => $nav_id,
								'label'      => $item['label'],
								'item_type'  => $item['type'],
								'item_slug'  => $item['object'],
								'menu_order' => $order,
								'parent_id'  => isset( $item['data']['parent'] ) && ! empty( $item['data']['parent'] ) ? $item['data']['parent'] : 0,
								'item_id'    => isset( $item['data']['id'] ) ? $item['data']['id'] : 0,
							);

							// Handle item icon if present and decode it if it's a string.
							if ( isset( $item['icon'] ) ) {
								$item_data['item_icon_data'] = json_decode( stripslashes( $item['icon'] ), true );
							}

							// Save menu item to the database.
							$item_id = MenuManager::instance()->insert_menu_item( $item_data );
						}
					}

					$redirect_to = add_query_arg(
						array(
							'page'    => 'bbapp-menus',
							'setting' => $menu_type,
							'action'  => 'edit',
							'menu'    => $nav_id,
							'lang'    => isset( $_data['lang'] ) ? $_data['lang'] : '',
							'created' => 1,
						),
						'admin.php'
					);

					wp_safe_redirect( $redirect_to );
					exit;
				} else {
					$redirect_to = add_query_arg(
						array(
							'page'    => 'bbapp-menus',
							'setting' => $menu_type,
							'lang'    => isset( $_data['lang'] ) ? $_data['lang'] : '',
						),
						'admin.php'
					);

					wp_safe_redirect( $redirect_to );
					exit;
				}
			}
		}
	}

	/**
	 * This function to update nav menu.
	 *
	 * @param string $nav_id    nav id as unique string.
	 * @param string $action    current action.
	 * @param string $menu_type current menu type. (tabbar|more).
	 * @param array  $_data     POST data.
	 *
	 * @since 1.5.2.1
	 */
	public function update_nav_menu( $nav_id, $action, $menu_type, $_data ) {
		if ( ! empty( $action ) && 'update' === $action ) {
			$result = $this->validate_prepare_nav_data( $_data );

			if ( true === $result['validate'] ) {
				// Prepare menu data for database update.
				$menu_data = array(
					'menu_name'   => $result['data']['menu_name'],
					'login_state' => isset( $result['data']['login_state'] ) && '0' === $result['data']['login_state'] ? 0 : 1,
				);

				if ( 'Logged-out' !== $menu_data['menu_name'] ) {
					$menu_data['login_state'] = 1;
				}

				// Handle access groups if present.
				if ( isset( $result['data']['access_groups'] ) ) {
					$menu_data['access_groups'] = $result['data']['access_groups'];
				}

				// Set appropriate priority based on menu type and name.
				$menu_data['priority'] = $this->get_appropriate_menu_priority(
					$menu_data['menu_name'],
					$menu_data['login_state'],
					isset( $result['data']['priority'] ) ? $result['data']['priority'] : null,
					$menu_type
				);

				// Update the menu in the database.
				$updated = MenuManager::instance()->update_menu( $nav_id, $menu_data );

				if ( false !== $updated ) {
					// Get existing menu items for this menu.
					$existing_menu_items = MenuManager::instance()->get_menu_items(
						array(
							'menu_id'  => $nav_id,
							'fields'   => 'id',
							'as_array' => true,
						)
					);

					// Create a map of existing menu items by ID for quick lookup and update.
					$existing_item_map = array();
					$existing_ids      = array();

					foreach ( $existing_menu_items as $existing_item_id ) {
						$existing_item_map[ $existing_item_id ] = $existing_item_id;
						$existing_ids[]                         = (int) $existing_item_id;
					}

					// Handle menu updates based on menu type and items.
					if ( ! empty( $_REQUEST['appmenu-item'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
						$menu_items = bbapp_input_clean( wp_unslash( $_REQUEST['appmenu-item'] ) ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Recommended

						// Use different update handlers based on menu type and items.
						switch ( $menu_type ) {
							case 'more':
								$updated_items = $this->update_more_menu_items( $nav_id, $menu_items );
								// Delete items that weren't updated or created.
								foreach ( $existing_ids as $existing_id ) {
									if ( ! in_array( $existing_id, $updated_items, true ) ) {
										MenuManager::instance()->delete_menu_item( $existing_id );
									}
								}
								break;
							case 'headerbar':
								$this->update_headerbar_menu_items( $nav_id, $menu_items, $existing_item_map );
								break;
							case 'tabbar':
							default:
								$this->update_tabbar_menu_items( $nav_id, $menu_items, $existing_item_map );
								break;
						}
					} else {
						foreach ( $existing_ids as $existing_id ) {
							MenuManager::instance()->delete_menu_item( $existing_id );
						}
					}

					$is_disabled = false;

					if ( ( isset( $result['data']['menu_name'] ) && in_array(
						$result['data']['menu_name'],
						array(
							'Default',
							'Logged-out',
							true,
						),
						true
					) ) ) {
						$is_disabled = true;
					}

					$main_menu_item_label = $result['data']['menu_name'];

					if ( $is_disabled ) {
						$main_menu_item_label .= '&nbsp;';
						$main_menu_item_label .= bbapp_get_menu_type_label( $menu_type );
					}

					$this->messages[] = array(
						'type'    => 'updated notice is-dismissible',
						/* translators: %s: Main menu label. */
						'message' => sprintf( __( '<strong>%s</strong> has been updated.', 'buddyboss-app' ), $main_menu_item_label ),
					);
				} else {
					$redirect_to = add_query_arg(
						array(
							'page'    => 'bbapp-menus',
							'setting' => $menu_type,
							'action'  => 'edit',
							'menu'    => $nav_id,
							'lang'    => isset( $_data['lang'] ) ? $_data['lang'] : '',
						),
						'admin.php'
					);

					wp_safe_redirect( $redirect_to );
					exit;
				}
			}
		}
	}

	/**
	 * Updates items in a TabBar menu
	 *
	 * @param int   $menu_id           The menu ID.
	 * @param array $menu_items        Menu items from the form.
	 * @param array $existing_item_map Map of existing menu item IDs to item data.
	 *
	 * @since 2.4.10
	 * @return array Updated item IDs
	 */
	private function update_tabbar_menu_items( $menu_id, $menu_items, $existing_item_map ) {
		$updated_items = array();
		$order         = 0;

		$header_bar_id      = MenuManager::instance()->get_headerbar_menu_from_tabbar_id( $menu_id );
		$more_menu_position = MenuManager::instance()->get_more_menu_position( $header_bar_id );

		// Separate regular items and "More" items for proper ordering and limits.
		$regular_items = array();
		$more_item     = null;

		// First pass: separate regular items from "More" items based on position.
		foreach ( $menu_items as $menu_item_id => $item ) {
			if ( 'more' === $item['object'] ) {
				// Skip "More" items if position doesn't match current menu type (tabbar).
				if ( 'tabbar' !== $more_menu_position ) {
					continue;
				}

				// Only keep the first "More" item, ignore duplicates and other "More" items.
				if ( ! $more_item ) {
					$more_item = array(
						'id'   => $menu_item_id,
						'data' => $item,
					);
				}
			} else {
				// Regular tabbar item (not "More").
				$regular_items[ $menu_item_id ] = $item;
			}
		}

		// Enforce 5-item limit based on more menu position and "More" item presence.
		if ( 'tabbar' === $more_menu_position && $more_item ) {
			// Case 1: 4 regular + 1 more = 5 total (no limit) or 5 total (with more).
			$max_regular_items = 4;
		} else {
			// Case 2: 5 regular items (no more) = 5 total (no limit) or 5 total (with more).
			$max_regular_items = 5;
		}

		// Limit regular items to the calculated maximum if necessary.
		if ( count( $regular_items ) > $max_regular_items ) {
			$regular_items = array_slice( $regular_items, 0, $max_regular_items, true );
		}

		// Process regular items first (in order).
		foreach ( $regular_items as $menu_item_id => $item ) {
			$item_data = array(
				'menu_id'    => $menu_id,
				'label'      => $item['label'],
				'item_type'  => $item['type'],
				'item_slug'  => $item['object'],
				'menu_order' => $order++,
				'parent_id'  => 0, // TabBar items have no parents or sections.
				'item_id'    => isset( $item['data']['id'] ) ? $item['data']['id'] : 0,
			);

			// Handle item icon if present and decode it if it's a string.
			if ( isset( $item['icon'] ) ) {
				if ( is_string( $item['icon'] ) ) {
					$item_data['item_icon_data'] = json_decode( stripslashes( $item['icon'] ), true );
				} else {
					$item_data['item_icon_data'] = $item['icon'];
				}
			}

			// Handle item link if present and decode it if it's a string.
			if ( isset( $item['data']['link'] ) ) {
				$item_data['item_link'] = $item['data']['link'];
			}

			// Use update_menu_item which handles both update and insert internally.
			$result = MenuManager::instance()->update_menu_item( $menu_item_id, $item_data );
			if ( $result ) {
				// For updates, result is the number of updated rows; for inserts, it's the new ID.
				$updated_items[] = $menu_item_id > 0 ? $menu_item_id : $result;
			}
		}

		// Process "More" item last (if it should be included) and update it if necessary.
		if ( $more_item ) {
			$item         = $more_item['data'];
			$menu_item_id = $more_item['id'];

			$item_data = array(
				'menu_id'    => $menu_id,
				'label'      => $item['label'],
				'item_type'  => $item['type'],
				'item_slug'  => $item['object'],
				'menu_order' => $order++, // "More" item gets the last order
				'parent_id'  => 0,
				'item_id'    => isset( $item['data']['id'] ) ? $item['data']['id'] : 0,
			);

			// Handle item icon if present and decode it if it's a string.
			if ( isset( $item['icon'] ) ) {
				if ( is_string( $item['icon'] ) ) {
					$item_data['item_icon_data'] = json_decode( stripslashes( $item['icon'] ), true );
				} else {
					$item_data['item_icon_data'] = $item['icon'];
				}

				$item_data['item_icon_data']['extra']['icon_type'] = $item['icon_type'];
			}

			// Handle item link if present and decode it if it's a string.
			if ( isset( $item['data']['link'] ) ) {
				$item_data['item_link'] = $item['data']['link'];
			}

			// Use update_menu_item which handles both update and insert internally.
			$result = MenuManager::instance()->update_menu_item( $menu_item_id, $item_data );
			if ( $result ) {
				// For updates, result is the number of updated rows; for inserts, it's the new ID.
				$updated_items[] = $menu_item_id > 0 ? $menu_item_id : $result;
			}
		}

		// Delete items that weren't updated or created.
		foreach ( $existing_item_map as $existing_id ) {
			if ( ! in_array( $existing_id, $updated_items, true ) ) {
				MenuManager::instance()->delete_menu_item( $existing_id );
			}
		}

		return $updated_items;
	}

	/**
	 * Updates items in a HeaderBar menu
	 *
	 * @param int   $menu_id           The menu ID.
	 * @param array $menu_items        Menu items from the form.
	 * @param array $existing_item_map Map of existing menu item IDs.
	 *
	 * @since 2.4.10
	 * @return array Updated item IDs
	 */
	private function update_headerbar_menu_items( $menu_id, $menu_items, $existing_item_map ) {
		$action_parent_id = 0;
		$action_children  = array();
		$final_menu_items = array();

		$is_platform_active = bbapp_is_buddyboss_platform_enabled();
		$more_menu_position = MenuManager::instance()->get_more_menu_position( $menu_id );
		$child_order        = 0; // Initialize child_order variable.

		// Calculate maximum allowed menus (limit to 3 total).
		$max_total_menus = 3;
		$reserved_slots  = 0;

		// Reserve slot for action menu if platform is active.
		if ( $is_platform_active ) {
			++$reserved_slots;
		}

		// Reserve slot for more menu if position is headerbar.
		if ( 'headerbar' === $more_menu_position ) {
			++$reserved_slots;
		}

		// Calculate available slots for other menus.
		$available_other_slots = $max_total_menus - $reserved_slots;

		// First pass: Separate menu items by type for proper ordering.
		$action_menu      = null;
		$more_menu        = null;
		$other_menus      = array();
		$other_menu_count = 0;

		foreach ( $menu_items as $menu_item_id => $item ) {
			// Skip items that only have an icon property (these are action children).
			if ( ! isset( $item['type'] ) && isset( $item['icon'] ) ) {
				continue;
			}

			// Process regular menu items only.
			if ( isset( $item['type'] ) ) {
				// Skip "More" items if position doesn't match current menu type (headerbar)
				if ( 'more' === $item['object'] && 'headerbar' !== $more_menu_position ) {
					continue;
				}

				// Skip headerbar-action if platform is not active - this is a special case for the platform.
				if ( 'headerbar-action' === $item['object'] && ! $is_platform_active ) {
					continue;
				}

				// Create item data structure.
				$item_data = array(
					'menu_id'    => $menu_id,
					'label'      => $item['label'],
					'item_type'  => $item['type'],
					'item_slug'  => $item['object'],
					'menu_order' => 0, // Will be set later based on priority.
					'parent_id'  => isset( $item['data']['parent'] ) && ! empty( $item['data']['parent'] ) ? $item['data']['parent'] : 0,
					'item_id'    => $menu_item_id,
				);

				// Handle item icon if present and decode it if it's a string.
				if ( isset( $item['icon'] ) ) {
					$item_data['item_icon_data'] = is_string( $item['icon'] ) ? json_decode( stripslashes( $item['icon'] ), true ) : $item['icon'];
				}

				if ( 'more' === $item['object'] ) {
					$item_data['item_icon_data']['extra']['icon_type'] = $item['icon_type'];
				}

				// Handle item link if present and decode it if it's a string.
				if ( isset( $item['data']['link'] ) ) {
					$item_data['item_link'] = $item['data']['link'];
				}

				// Categorize menu items by type for proper ordering.
				if ( 'headerbar-action' === $item['object'] ) {
					// Action menu - priority 1 (top).
					$action_menu      = $item_data;
					$action_parent_id = isset( $item['data']['id'] ) ? $item['data']['id'] : 'headerbar';
				} elseif ( 'more' === $item['object'] ) {
					// More menu - priority 3 (bottom).
					$more_menu = $item_data;
				} else {
					// Other menus - priority 2 (middle).
					if ( $other_menu_count >= $available_other_slots ) {
						// Skip this menu - we've reached the limit for other menus.
						continue;
					}
					$other_menus[] = $item_data;
					++$other_menu_count;
				}
			}
		}

		// Second pass: Build final menu items in priority order with correct menu_order.
		$menu_order = 0;

		// 1. Action menu first (if exists)
		if ( $action_menu ) {
			$action_menu['menu_order'] = $menu_order++;
			$final_menu_items[]        = $action_menu;
		}

		// 2. Other menus in middle
		foreach ( $other_menus as $other_menu ) {
			$other_menu['menu_order'] = $menu_order++;
			$final_menu_items[]       = $other_menu;
		}

		// 3. More menu last (if exists)
		if ( $more_menu ) {
			$more_menu['menu_order'] = $menu_order++;
			$final_menu_items[]      = $more_menu;
		}

		// Third pass: Process action children with correct parent_id (only if platform is active).
		if ( $is_platform_active ) {
			foreach ( $menu_items as $menu_item_id => $item ) {
				// Process only action children (items without type but with icon).
				if ( ! isset( $item['type'] ) && isset( $item['icon'] ) ) {
					$icon_data = is_string( $item['icon'] ) ? json_decode( stripslashes( $item['icon'] ), true ) : $item['icon'];

					$icon_label = isset( $icon_data['extra']['label'] ) ? $icon_data['extra']['label'] : 'Action Item';
					$icon_id    = isset( $icon_data['extra']['uniq_id'] ) ? $icon_data['extra']['uniq_id'] : '';
					$item_slug  = isset( $icon_data['extra']['item_slug'] ) ? $icon_data['extra']['item_slug'] : '';

					$child_item_data = array(
						'menu_id'        => $menu_id,
						'label'          => $icon_label,
						'item_type'      => 'headerbar-action-menu',
						'menu_order'     => $child_order++,
						'parent_id'      => $action_parent_id, // Now this will have the correct parent ID.
						'item_id'        => $menu_item_id,
						'item_icon_data' => $icon_data,
					);

					if ( ! empty( $item_slug ) ) {
						$child_item_data['item_slug'] = $item_slug;
					}

					$action_children[] = $child_item_data;
				}
			}
		}

		// Now actually update/insert the menu items.
		$updated_items = array();
		$menu_manager  = MenuManager::instance();

		// Collect IDs of items we want to keep for cleanup later.
		$items_to_keep = array();
		foreach ( $final_menu_items as $item_data ) {
			if ( $item_data['item_id'] ) {
				$items_to_keep[] = $item_data['item_id'];
			}
		}

		// Process regular menu items and capture the actual headerbar-action ID.
		$actual_action_parent_id = 0;
		foreach ( $final_menu_items as $item_data ) {
			if ( $item_data['item_id'] && isset( $existing_item_map[ $item_data['item_id'] ] ) ) {
				// Update existing item - update_menu_item returns the new ID when updating.
				$result = $menu_manager->update_menu_item( $item_data['item_id'], $item_data );
				if ( false !== $result ) {
					$updated_items[] = $item_data['item_id'];

					// If this is the headerbar-action item, store its actual database ID.
					if ( 'headerbar-action' === $item_data['item_slug'] ) {
						$actual_action_parent_id = $item_data['item_id'];
					}
				}
			} else {
				// Create new item - update_menu_item returns the new ID when creating.
				$new_id = $menu_manager->update_menu_item( $item_data['item_id'], $item_data );
				if ( $new_id ) {
					$updated_items[] = $new_id;

					// If this is the headerbar-action item, store its actual database ID.
					if ( 'headerbar-action' === $item_data['item_slug'] ) {
						$actual_action_parent_id = $new_id;
					}
				}
			}
		}

		// Process action children with the actual parent ID.
		if ( $actual_action_parent_id && ! empty( $action_children ) ) {
			foreach ( $action_children as $child_data ) {
				// Update the parent_id to the actual database ID.
				$child_data['parent_id'] = $actual_action_parent_id;

				// Action children are always new items - use insert_menu_item directly.
				$new_id = $menu_manager->update_menu_item( $child_data['item_id'], $child_data );
				if ( $new_id ) {
					$updated_items[] = $new_id;
				}
			}
		}

		// Delete extra menu items that are no longer needed.
		foreach ( $existing_item_map as $existing_id ) {
			if ( ! in_array( $existing_id, $items_to_keep, true ) ) {
				// This item exists in database but not in our final menu items.
				$menu_manager->delete_menu_item( $existing_id );
			}
		}

		return $updated_items;
	}

	/**
	 * Updates items in a More menu, handling sections and their child items
	 *
	 * @param int   $menu_id           The menu ID.
	 * @param array $menu_items        Menu items from the form.
	 *
	 * @since 2.4.10
	 * @return array Updated item IDs
	 */
	private function update_more_menu_items( $menu_id, $menu_items ) {
		$updated_items    = array();
		$structured_items = array();

		// First identify section items and organize non-section items as their children.
		foreach ( $menu_items as $menu_item_id => $item ) {
			// Check if this is a section item and add it to the structured items array.
			if ( isset( $item['type'] ) && 'section' === $item['type'] ) {
				// This is a section item - add it to the structured items array with an empty children array.
				$structured_items[ $menu_item_id ]             = $item;
				$structured_items[ $menu_item_id ]['children'] = array();
			} elseif ( ! empty( $item['data']['parent'] ) ) {
				$parent_id = $item['data']['parent'];

				// Check if parent exists in our structured items array.
				if ( isset( $structured_items[ $parent_id ] ) ) {
					// Add this item as a child of the section item.
					$structured_items[ $parent_id ]['children'][ $menu_item_id ] = $item;
				} else {
					// This item has a parent that's not a section, treat as a regular item with no parent.
					$structured_items[ $menu_item_id ] = $item;
				}
			} else {
				$structured_items[ $menu_item_id ] = $item;
			}
		}

		// Process structured items and save to database.
		$menu_manager   = MenuManager::instance();
		$order          = 0;
		$section_id_map = array();

		// First process all top-level items (sections and standalone items).
		foreach ( $structured_items as $menu_item_id => $item ) {
			// Prepare item data for database insertion or update.
			$item_data = array(
				'menu_id'    => $menu_id,
				'label'      => $item['label'],
				'item_type'  => $item['type'],
				'item_slug'  => $item['object'],
				'menu_order' => $order++,
				'parent_id'  => isset( $item['data']['parent'] ) ? $item['data']['parent'] : 0,
				'item_id'    => isset( $item['data']['id'] ) ? $item['data']['id'] : 0,
			);

			// Handle icon data if present and decode it if it's a string.
			if ( isset( $item['icon'] ) ) {
				if ( is_string( $item['icon'] ) ) {
					$item_data['item_icon_data'] = json_decode( stripslashes( $item['icon'] ), true );
				} else {
					$item_data['item_icon_data'] = $item['icon'];
				}
			}

			$item_data['item_data'] = isset( $item['data'] ) ? $item['data'] : array();

			// Handle link data if present and decode it if it's a string.
			if ( isset( $item['data']['link'] ) ) {
				$item_data['item_link'] = $item['data']['link'];
			}

			// Check if this is an existing item that needs updating or a new item to be created.
			$db_item_id = null;

			if ( is_int( $menu_item_id ) ) {
				// Update existing item if it exists.
				$updated = $menu_manager->update_menu_item( $menu_item_id, $item_data );
				if ( $updated ) {
					$updated_items[] = $menu_item_id;
					$db_item_id      = $menu_item_id;
				}
			} else {
				// Insert new item if it doesn't exist.
				$new_id = $menu_manager->insert_menu_item( $item_data );
				if ( $new_id ) {
					$updated_items[] = $new_id;
					$db_item_id      = $new_id;
				}
			}

			// If this is a section, store its ID for children processing.
			if ( isset( $item['type'] ) && 'section' === $item['type'] && $db_item_id ) {
				$section_id_map[ $menu_item_id ] = $db_item_id;

				$existing_children = $menu_manager->get_menu_items(
					array(
						'menu_id'   => $menu_id,
						'parent_id' => $db_item_id,
						'fields'    => 'id',
					)
				);

				$section_child_ids = array();

				// Process children of this section if they exist.
				if ( ! empty( $item['children'] ) ) {
					$child_order = 0;
					foreach ( $item['children'] as $child_id => $child_item ) {
						// Prepare child item data for database insertion or update.
						$child_data = array(
							'menu_id'    => $menu_id,
							'label'      => $child_item['label'],
							'item_type'  => $child_item['type'],
							'item_slug'  => $child_item['object'],
							'menu_order' => $child_order++,
							'parent_id'  => $db_item_id, // Use the parent section's ID for the child.
							'item_id'    => isset( $child_item['data']['id'] ) ? $child_item['data']['id'] : 0,
						);

						// Handle icon data if present and decode it if it's a string.
						if ( isset( $child_item['icon'] ) ) {
							if ( is_string( $child_item['icon'] ) ) {
								$child_data['item_icon_data'] = json_decode( stripslashes( $child_item['icon'] ), true );
							} else {
								$child_data['item_icon_data'] = $child_item['icon'];
							}
						}

						// Handle link data if present and decode it if it's a string.
						if ( isset( $child_item['data']['link'] ) ) {
							$child_data['item_link'] = $child_item['data']['link'];
						}

						// Check if this is an existing child item that needs updating or a new item to be created.
						if ( is_int( $child_id ) ) {
							// Update existing child item if it exists.
							$updated = $menu_manager->update_menu_item( $child_id, $child_data );
							if ( $updated ) {
								$updated_items[] = $child_id;
							}
							$section_child_ids[] = $child_id;
						} else {
							// Insert new child item if it doesn't exist.
							$new_child_id = $menu_manager->insert_menu_item( $child_data );
							if ( $new_child_id ) {
								$updated_items[]     = $new_child_id;
								$section_child_ids[] = $new_child_id;
							}
						}
					}

					// Delete children that exist in the database but weren't included in the updated menu items.
					if ( ! empty( $existing_children ) ) {
						foreach ( $existing_children as $child_id ) {
							if ( ! in_array( $child_id, $section_child_ids, true ) ) {
								$menu_manager->delete_menu_item( $child_id );
							}
						}
					}
				}
			}
		}

		return $updated_items;
	}

	/**
	 * This function to delete nav menu.
	 *
	 * @param string $nav_id    nav id as unique string.
	 * @param string $action    current action.
	 * @param string $menu_type current menu type. (tabbar|more).
	 *
	 * @since 1.5.2.1
	 */
	public function delete_nav_menu( $nav_id, $action, $menu_type ) {
		if ( ! empty( $action ) && 'delete' === $action ) {
			$menu = MenuManager::instance()->get_menu( $nav_id );
			if ( $menu ) {
				// Get headerbar ID BEFORE deleting the tabbar menu to avoid losing it.
				$headerbar_nav_id = null;
				if ( 'tabbar' === $menu_type ) {
					$headerbar_nav_id = MenuManager::instance()->get_headerbar_menu_from_tabbar_id( $nav_id );
				}

				$deletion = MenuManager::instance()->delete_menu( $nav_id );

				// Delete attached header bar of the tab bar.
				if ( $deletion && 'tabbar' === $menu_type && $headerbar_nav_id ) {
					MenuManager::instance()->delete_menu( $headerbar_nav_id );
				}

				if ( $deletion ) {
					$this->messages[] = array(
						'type'    => 'updated notice is-dismissible',
						'message' => __( 'The menu has been successfully deleted.', 'buddyboss-app' ),
					);
				}
			}

			wp_safe_redirect(
				add_query_arg(
					array(
						'page'    => 'bbapp-menus',
						'setting' => $menu_type,
						'deleted' => 1,
					),
					admin_url( 'admin.php' )
				)
			);
			exit;
		}
	}

	/**
	 * This function to save menu settings.
	 *
	 * @param string $action        current action.
	 * @param string $menu_type     current menu type. (tabbar|more).
	 * @param array  $menu_settings POST data as menu setting.
	 *
	 * @since 1.5.2.1
	 */
	public function save_menu_settings( $action, $menu_type, $menu_settings ) {
		if ( ! empty( $action ) && 'app_menu_setting_save' === $action ) {
			if ( ! is_array( $menu_settings ) ) {
				$menu_settings = array();
			}

			// Settings key allowed to be saved.
			$whitelist_settings_key = array();
			$updated                = false;

			if ( 'headerbar' === $menu_type ) {
				if ( ! empty( $menu_settings['headerbar'] ) ) {
					$headerbar_id = $menu_settings['headerbar'];
					$menu         = MenuManager::instance()->get_menu( $headerbar_id );

					if ( $menu ) {
						$data = ! empty( $menu->data ) ? $menu->data : array();

						// Avoid settings key conflicts for different form submissions with the same action key.
						if ( array_key_exists( 'title_position', $menu_settings ) || array_key_exists( 'more_menu_position', $menu_settings ) ) {
							$data['title_position']     = ( ! empty( $menu_settings['title_position'] ) ) ? $menu_settings['title_position'] : 'left';
							$data['more_menu_position'] = ( ! empty( $menu_settings['more_menu_position'] ) ) ? $menu_settings['more_menu_position'] : 'tabbar';
						} else {
							$data['post_input_field'] = ! empty( $menu_settings['post_input_button'] );
						}

						$more_from_menu_id = $headerbar_id;
						$move_to_menu_id   = $data['related_tabbar_id'];
						if ( 'headerbar' === $data['more_menu_position'] ) {
							$more_from_menu_id = $data['related_tabbar_id'];
							$move_to_menu_id   = $headerbar_id;
						}

						if ( ! empty( $more_from_menu_id ) && ! empty( $move_to_menu_id ) ) {
							$more_item_id = MenuManager::instance()->get_menu_item_id_by_slug( $more_from_menu_id, 'more' );

							if ( $more_item_id ) {
								// Update the menu item data with the new settings.
								MenuManager::instance()->update_menu_item(
									$more_item_id,
									array(
										'menu_id' => $move_to_menu_id,
									)
								);
							}
						}

						// Update the menu data with the new settings.
						$updated = MenuManager::instance()->update_menu( $headerbar_id, array( 'data' => $data ) );
					}
				}
			} else {
				if ( 'more' === $menu_type ) {
					$whitelist_settings_key['more_icon_style'] = 'string';
				} else {
					$whitelist_settings_key['tab_bar_icon_style'] = 'string';
					$whitelist_settings_key['tab_bar_visibility'] = 'string';
					$whitelist_settings_key['appmenu_labels']     = 'bool';
				}

				$app_settings = ManageApp::instance()->get_app_settings();

				if ( ! empty( $whitelist_settings_key ) ) {
					foreach ( $whitelist_settings_key as $setting_key => $type ) {
						$setting_value = ( isset( $menu_settings[ $setting_key ] ) ) ? $menu_settings[ $setting_key ] : '';

						// correct the type of data.
						if ( 'bool' === $type ) {
							if ( '1' === (string) $setting_value ) {
								$setting_value = true;
							} else {
								$setting_value = false;
							}
						}
						$app_settings[ "app_menu.{$setting_key}" ] = $setting_value;
					}
				}

				$updated = ManageApp::instance()->update_app_settings( $app_settings );

				unset( $app_settings );
			}

			if ( $updated ) {
				$this->messages[] = array(
					'type'    => 'updated notice is-dismissible',
					/* translators: %s: Menu label. */
					'message' => sprintf( __( '<strong>%s</strong> settings has been updated.', 'buddyboss-app' ), bbapp_get_menu_type_label( $menu_type ) ),
				);
			}
		}
	}

	/**
	 * This function save menu priority.
	 *
	 * @param string $action         current action.
	 * @param string $menu_type      current menu type. (tabbar|more).
	 * @param string $nav_menu_order comma separated string.
	 *
	 * @since 1.5.2.1
	 */
	public function save_menu_priority( $action, $menu_type, $nav_menu_order ) {
		if ( ! empty( $action ) && 'save-menu-priority' === $action ) {
			if ( $nav_menu_order ) {
				$nav_priorities = explode( ',', $nav_menu_order );

				if ( ! empty( $nav_priorities ) ) {
					// Build criteria array for priority update
					$criteria = array();

					// Get default and logged-out menu IDs
					$menu_manager       = MenuManager::instance();
					$default_menu_id    = $menu_manager->get_default_menu_id( $menu_type );
					$logged_out_menu_id = $menu_manager->get_logout_menu_id( $menu_type );

					// Process normal menus first (excluding default and logged-out)
					$index = 0;
					foreach ( $nav_priorities as $nav_id ) {
						if ( is_numeric( $nav_id ) ) {
							$nav_id = intval( $nav_id );

							// Skip default and logged-out menus, they'll be handled separately
							if ( $nav_id !== intval( $default_menu_id ) && $nav_id !== intval( $logged_out_menu_id ) ) {
								$criteria[] = array(
									'menu_id'  => $nav_id,
									'priority' => $index++,
								);
							}
						}
					}

					// Get menu objects for special menus to pass to our helper function
					$logged_out_menu = null;
					$default_menu    = null;

					if ( ! empty( $logged_out_menu_id ) ) {
						$logged_out_menu = $menu_manager->get_menu( $logged_out_menu_id );
					}

					if ( ! empty( $default_menu_id ) ) {
						$default_menu = $menu_manager->get_menu( $default_menu_id );
					}

					// Add logged-out menu with appropriate priority if it exists and is not empty.
					if ( ! empty( $logged_out_menu ) ) {
						$criteria[] = array(
							'menu_id'  => intval( $logged_out_menu_id ),
							'priority' => $this->get_appropriate_menu_priority( $logged_out_menu->menu_name, 0 ),
						);
					}

					// Add default menu with appropriate priority if it exists and is not empty.
					if ( ! empty( $default_menu ) ) {
						$criteria[] = array(
							'menu_id'  => intval( $default_menu_id ),
							'priority' => $this->get_appropriate_menu_priority( $default_menu->menu_name, 1 ),
						);
					}

					// Update priorities with the MenuManager method and pass the criteria array.
					$updated = MenuManager::instance()->update_menu_priority( $menu_type, $criteria );

					if ( $updated ) {
						$this->messages[] = array(
							'type'    => 'updated notice is-dismissible',
							/* translators: %s: Menu label. */
							'message' => sprintf( __( '<strong>%s</strong> menu priority has been updated.', 'buddyboss-app' ), bbapp_get_menu_type_label( $menu_type ) ),
						);
					}
				}
			}
		}
	}

	/**
	 * Added related js and css data.
	 *
	 * @since 1.5.2.1
	 */
	public function load_header_script() {
		foreach (
			array(
				'jquery-ui-sortable',
				'jquery-ui-draggable',
				'jquery-ui-droppable',
				'wp-lists',
				'postbox',
			) as $script
		) {
			wp_enqueue_script( $script );
		}

		$nav_appmenus_l10n = array(
			'noResultsFound' => __( 'No results found.', 'buddyboss-app' ),
			'saveAlert'      => __( 'The changes you made will be lost if you navigate away from this page.', 'buddyboss-app' ),
			'untitled'       => _x( '(no label)', 'missing menu item navigation label', 'buddyboss-app' ),
		);

		wp_localize_script( 'bbapp-script', 'bbappNavMenuL10n', $nav_appmenus_l10n );

		?>
		<style>
			/* Force visible if screen option is disabled */
			.nav-appmenu-meta .outer-border > .control-section {
				display: block !important;
			}
			/* Style for disabled button */
			.button-primary.disabled {
				opacity: 0.6;
				cursor: not-allowed;
			}
		</style>
		<?php
	}

	/**
	 * Called when new app menu is added.
	 *
	 * @since 1.5.2.1
	 */
	public function wp_ajax_add_menu_item() {
		check_ajax_referer( 'add-appmenu_item', 'menu-settings-column-nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( - 1 );
		}

		$current_menu_type = isset( $_POST['current_menu'] ) ? bbapp_input_clean( wp_unslash( $_POST['current_menu'] ) ) : 'tabbar'; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$post_menu_item    = isset( $_POST['menu-item'] ) ? bbapp_input_clean( wp_unslash( $_POST['menu-item'] ) ) : array(); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$menu_items        = array();

		foreach ( (array) $post_menu_item as $menu_item_data ) {
			// Or other menu type.
			if ( ! empty( $menu_item_data['menu-item-type'] ) ) {
				$object        = isset( $menu_item_data['menu-item-object'] ) ? $menu_item_data['menu-item-object'] : $menu_item_data['menu-item-type'];
				$deeplink_path = isset( $menu_item_data['menu-item-deeplink-path'] ) ? $menu_item_data['menu-item-deeplink-path'] : $object;

				if ( 'core' === $menu_item_data['menu-item-type'] ) {
					$menu_item_data['menu-item-object-id'] = '';
				}

				$object_item                      = array();
				$object_item['current_menu_type'] = $current_menu_type;
				$object_item['id']                = uniqid();
				$object_item['item_type']         = $menu_item_data['menu-item-type'];
				$object_item['item_slug']         = ! empty( $menu_item_data['menu-item-object'] ) ? $menu_item_data['menu-item-object'] : $menu_item_data['menu-item-type'];
				$object_item['label']             = stripslashes_deep( $menu_item_data['menu-item-title'] );
				$object_item['data']              = array();

				if ( ! isset( $menu_item_data['menu-item-object'] ) ) {
					$menu_item_data['menu-item-object'] = $menu_item_data['menu-item-type'];
				}

				if ( isset( $menu_item_data['menu-item-object'] ) && in_array( $menu_item_data['menu-item-object'], array( 'section', 'custom' ), true ) ) {
					$menu_item_data['menu-item-object-id'] = uniqid();
				}

				if ( 'custom' === $object_item['item_type'] ) {
					$object_item['url'] = $menu_item_data['menu-item-url'];
				}

				$object_item['data']['id']            = $menu_item_data['menu-item-object-id'];
				$object_item['data']['parent']        = isset( $menu_item_data['menu-item-parent-id'] ) ? $menu_item_data['menu-item-parent-id'] : '';
				$object_item['data']['link']          = ! empty( $menu_item_data['menu-item-url'] ) ? urldecode( esc_url_raw( $menu_item_data['menu-item-url'] ) ) : '#';
				$object_item['data']['deeplink_path'] = $deeplink_path;
				$object_item['section']               = ( 'section' === $menu_item_data['menu-item-object'] ) ? '1' : '0';

				$object_item['icon'] = bbapp_get_menu_icon( $object_item['item_slug'], $object_item['item_type'] );

				$menu_items[] = $object_item;
			}
		}

		foreach ( $menu_items as $menu_item ) {
			switch ( $menu_item['item_type'] ) {
				case 'section':
					$this->render_appmenu_page( $menu_item, $menu_item['data']['id'] );
					break;
				default:
					$this->render_appmenu_page( $menu_item, uniqid() );
					break;
			}
		}

		wp_die();
	}

	/**
	 * Add more menu.
	 *
	 * @since 2.1.90
	 */
	public function add_more_menu() {
		$nonce = ( ! empty( $_POST['nonce'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['nonce'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! wp_verify_nonce( $nonce, 'add-more-menu' ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce.', 'buddyboss-app' ) ) );
		}

		$more_menu      = '';
		$more_menu_item = MenuManager::instance()->get_default_more_menu();

		ob_start();
		$this->render_appmenu_page( $more_menu_item, uniqid() );
		$more_menu .= ob_get_clean();

		unset( $more_menu_item );

		wp_send_json_success( array( 'more_menu' => $more_menu ) );
	}

	/**
	 * Render app menu page.
	 *
	 * @param array  $menu_item Menu item.
	 * @param string $index     item index.
	 *
	 * @since 1.5.2.1
	 */
	public function render_appmenu_page( $menu_item, $index = '' ) {
		global $section_loop, $section_id, $menu_parent_id;

		$depth = 0;

		if ( 'section' === $menu_item['item_type'] ) {
			$section_loop = true;
			$section_id   = $menu_item['id'];
		} elseif ( isset( $section_loop ) && $section_loop && ! empty( $menu_item['data']['parent'] ) && $menu_item['data']['parent'] === $section_id ) {
			$depth                       = 1;
			$menu_parent_id              = $menu_item['data']['parent']; // fix the parent id.
			$menu_item['data']['parent'] = $section_id; // fix the parent id.
		} else {
			$section_loop                = false;
			$menu_item['data']['parent'] = '';
		}

		$screen_group = $menu_item['item_type'];

		/**
		 * Register menu metaboxs.
		 *
		 * @since 1.5.2.1
		 */
		$register_meta_boxes = apply_filters( 'bbapp_register_menu_metaboxes', array() );

		if ( 'custom' === $screen_group ) { // To added correct label.
			$screen_group = 'custom_link';
		}

		if ( 'more' === $menu_item['item_slug'] ) {
			$screen_group_label = __( 'More Tab', 'buddyboss-app' );
		} elseif ( 'post_type' === $screen_group ) {
			if ( 'page' === $menu_item['item_slug'] ) {
				$screen_group_label = __( 'WordPress Page', 'buddyboss-app' );
			} elseif ( 'app_page' === $menu_item['item_slug'] ) {
				$screen_group_label = __( 'App Page', 'buddyboss-app' );
			} else {
				$post_type_object   = get_post_type_object( $menu_item['item_slug'] );
				$screen_group_label = ! empty( $post_type_object ) ? $post_type_object->label : __( 'Custom Post Type', 'buddyboss-app' );
			}
		} elseif ( isset( $register_meta_boxes[ $screen_group ] ) ) {
			$screen_group_label = $register_meta_boxes[ $screen_group ]['title'];
		} elseif ( 'headerbar' === $screen_group ) {
			$screen_group_label = __( 'Headerbar', 'buddyboss-app' );
		} else {
			$screen_group_label = $screen_group;
		}

		$menu_section = 0;

		if ( ! empty( $menu_item['section'] ) ) {
			if ( is_array( $menu_item['section'] ) ) {
				$menu_section = 1;
			}
		}

		$menu_item_argument = array(
			'id'          => isset( $menu_item['id'] ) ? $menu_item['id'] : uniqid(),
			'original'    => isset( $menu_item['original'] ) ? $menu_item['original'] : $menu_item['label'],
			'index'       => ! empty( $index ) ? $index : 0,
			'label'       => $menu_item['label'],
			'type_label'  => $screen_group_label,
			'item_type'   => $menu_item['item_type'],
			'item_slug'   => $menu_item['item_slug'],
			'type'        => $menu_item['item_type'], // @todo: We will support item_type and remove this.
			'object'      => $menu_item['item_slug'], // @todo: We will support item_slug and remove this.
			'depth'       => $depth,
			'data'        => array(
				'id'            => ! empty( $menu_item['data']['id'] ) ? $menu_item['data']['id'] : '',
				'link'          => ! empty( $menu_item['data']['link'] ) ? urldecode( $menu_item['data']['link'] ) : '',
				'parent'        => isset( $menu_item['data']['parent'] ) ? $menu_item['data']['parent'] : $index,
				'external'      => ! empty( $menu_item['data']['open_external'] ) ? $menu_item['data']['open_external'] : '',
				'is_readonly'   => false,
				'deeplink_path' => ! empty( $menu_item['data']['deeplink_path'] ) ? $menu_item['data']['deeplink_path'] : '',
			),
			'icon'        => IconPicker::instance()->prepare_menu_icon_argument( $menu_item ),
			'has_disable' => ( isset( $menu_item['has_disable'] ) && true === $menu_item['has_disable'] ),
		);

		if ( 'more' === $menu_item['item_slug'] ) {
			if ( ! empty( $menu_item['icon']['extra']['icon_type'] ) ) {
				$icon_type = $menu_item['icon']['extra']['icon_type'];
			}

			if ( empty( $icon_type ) ) {
				$icon_type = 'avatar';
			}

			$menu_item_argument['icon_type'] = $icon_type;
		}

		if ( ! empty( $menu_item['item_type'] ) && 'headerbar' === $menu_item['item_type'] ) {
			$menu_item_argument['icon'] = array();
			$action_menus               = ( ! empty( $menu_item['child_items'] ) ) ? $menu_item['child_items'] : array();
			$icons_arr                  = array();

			if ( ! empty( $action_menus ) ) {
				foreach ( $action_menus as $action_menu ) {
					$icon_data                       = IconPicker::instance()->prepare_menu_icon_argument( $action_menu );
					$icon_data['extra']['label']     = $action_menu['icon']['extra']['label'];
					$icon_data['extra']['item_slug'] = $action_menu['item_slug'];
					$icons_arr[]                     = $icon_data;
				}
			}

			$menu_item_argument['icons'] = $icons_arr;
		}

		/**
		 * Check unlinked menu items for CPT.
		 */
		$linked = true;

		/**
		 * App menu allow post types.
		 *
		 * @param array $default Provide post type array.
		 *
		 * @since 1.5.2.1
		 */
		$allowed_post_types =
			array(
				'page',
				'post',
				'app_page',
			);

		/**
		 * Filters allowed post types for menus.
		 *
		 * @param array $allowed_post_types Allowed post types.
		 */
		$cpt_menu_type = apply_filters( 'bbapp_app_menus_allowed_post_types', $allowed_post_types );

		if ( 'post_type' === $menu_item['item_type'] && in_array( $menu_item['item_slug'], $cpt_menu_type, true ) ) {
			$post = get_post( $menu_item['data']['id'] );

			if ( empty( $post ) || is_wp_error( $post ) || 'publish' !== $post->post_status ) {
				$linked                            = false;
				$menu_item_argument['has_disable'] = true;
			} else {
				$menu_item_argument['label']               = ! empty( $menu_item['label'] ) ? $menu_item['label'] : $post->post_title;
				$menu_item_argument['data']['link']        = get_permalink( $post );
				$menu_item_argument['data']['is_readonly'] = true;
				$menu_item_argument['original']            = $post->post_title;
			}
		}

		$menu_item_argument['deeplink_url'] = $this->get_menus_deeplinking_url( $menu_item_argument );

		include bbapp()->plugin_dir . '/views/menus/sections/menu-markup.php';
	}

	/**
	 * Get deeplinking url
	 *
	 * @param array $menu Menu.
	 *
	 * @since 1.5.2.1
	 * @return string|void
	 */
	public function get_menus_deeplinking_url( $menu ) {
		switch ( $menu['item_type'] ) {
			case 'post_type':
				if ( 'app_page' === $menu['item_slug'] ) {
					$deeplinking_url = home_url( 'bbapp/page/' . $menu['data']['id'] );
				} else {
					$deeplinking_url = $menu['data']['link'];
				}
				break;
			default:
				$deeplinking_url = home_url( 'bbapp/' . $menu['data']['deeplink_path'] );
		}

		return $deeplinking_url;
	}


	/**
	 * Menu dependency added.
	 *
	 * @param bool  $status status of menu.
	 * @param array $menu   menu item.
	 *
	 * @since 1.5.2.1
	 * @return false
	 */
	public function filter_app_menu_by_dependency( $status, $menu ) {
		if ( empty( $menu ) ) {
			return $status;
		}

		$item_type = $menu['item_type'];
		$item_slug = $menu['item_slug'];

		if ( in_array( $item_type, array( 'section', 'custom', 'headerbar' ), true ) ) {
			return $status;
		}

		if ( in_array( $item_slug, array( 'more' ), true ) ) {
			return $status;
		}

		if ( 'post_type' === $item_type ) {
			$post = get_post( $menu['data']['id'] );

			if ( empty( $post ) || is_wp_error( $post ) || 'publish' !== $post->post_status ) {
				return false;
			}
		} else {
			/**
			 * Get all menus.
			 *
			 * @since 1.5.2.1
			 */
			$all_menus = apply_filters( 'bbapp_get_all_menus', array() );

			if ( ! isset( $all_menus[ $item_type ][ $item_slug ] ) ) {
				return false;
			}
		}

		/**
		 * Check menu dependency.
		 *
		 * @param bool  $status status of menu.
		 * @param array $menu   menu item.
		 *
		 * @since 2.4.10
		 *
		 * @return bool
		 */
		return apply_filters( 'bbapp_menu_item_dependency', $status, $menu );
	}

	/**
	 * Excluded menu from menu list.
	 *
	 * @param array  $prepared_items     Menu list.
	 * @param string $app_menu_type Menu type.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function exclude_menu_from_menu_list( $prepared_items, $app_menu_type ) {
		if ( ! is_array( $prepared_items ) ) {
			return array();
		}

		/**
		 * Get all menus.
		 *
		 * @param array $menus All menus.
		 *
		 * @since 1.5.2.1
		 */
		$all_menus = apply_filters( 'bbapp_get_all_menus', array() );

		if ( ! empty( $prepared_items ) ) {
			foreach ( $prepared_items as $key => $menu_item ) {
				// Always add deeplink path to menu items for deeplinking.
				$prepared_items[ $key ] = $this->bbapp_deeplink_path( $menu_item, $all_menus );

				$item_type = $menu_item['item_type'];

				if ( ! is_user_logged_in() && 'bb_logged_in' === $item_type ) {
					unset( $prepared_items[ $key ] );
				}

				if ( in_array( $item_type, array( 'custom', 'post_type' ), true ) ) {
					if ( $this->bbapp_is_not_published_post_type_menu( $menu_item['data']['id'], $item_type, bbapp_is_rest() ) ) {
						unset( $prepared_items[ $key ] );
					}
					continue;
				}

				$item_slug = $menu_item['item_slug'];

				if ( 'section' === $item_slug ) {
					$_exclude_section_menus            = $this->exclude_menu_from_menu_list( $menu_item['section'], $app_menu_type );
					$prepared_items[ $key ]['section'] = $_exclude_section_menus;
				} elseif ( 'headerbar' !== $item_type && 'headerbar-action-menu' !== $item_type && ! isset( $all_menus[ $item_type ][ $item_slug ] ) && ! ( 'core' === $item_type && 'more' === $item_slug ) ) {
					unset( $prepared_items[ $key ] );
				}
			}
		}

		return $prepared_items;
	}

	/**
	 * Check post type menu is not published so exclude from menu api.
	 *
	 * @param int    $id      Post id.
	 * @param string $type    Menu type.
	 * @param false  $is_rest is rest api or not.
	 *
	 * @since 1.5.2.1
	 * @return bool return true if provided post id is not published.
	 */
	public function bbapp_is_not_published_post_type_menu( $id, $type, $is_rest = false ) {
		$is_not_published = false;

		if ( $is_rest && 'post_type' === $type ) {
			$post = get_post( $id );

			if ( empty( $post ) || is_wp_error( $post ) || 'publish' !== $post->post_status ) {
				$is_not_published = true;
			}
		}

		return $is_not_published;
	}

	/**
	 * Create deeplinking path.
	 *
	 * @param array $app_menu  app menu.
	 * @param array $all_menus all menus.
	 *
	 * @since 1.5.2.1
	 * @return mixed
	 */
	public function bbapp_deeplink_path( $app_menu, $all_menus ) {
		/**
		 * Register menu metaboxs.
		 *
		 * @param array $register_meta_boxes Meta boxes.
		 *
		 * @since 1.5.2.1
		 */
		$register_meta_boxes = apply_filters( 'bbapp_register_menu_metaboxes', array() );

		$menu_type = isset( $app_menu['item_type'] ) ? $app_menu['item_type'] : '';

		if ( isset( $menu_type ) && ! empty( $register_meta_boxes[ $menu_type ] ) && ! empty( $register_meta_boxes[ $menu_type ]['args'] ) ) {
			$meta_boxes_args                   = $register_meta_boxes[ $menu_type ]['args'];
			$slug                              = isset( $all_menus[ $menu_type ][ $app_menu['object'] ] ) ? $all_menus[ $menu_type ][ $app_menu['object'] ]['slug'] : '';
			$app_menu['data']['deeplink_path'] = ( $slug !== $meta_boxes_args['deeplink_base'] ) ? $meta_boxes_args['deeplink_base'] . '/' . $slug : $slug;
		}

		return $app_menu;
	}

	/**
	 * Renders all menu metaboxes.
	 *
	 * @since 1.5.2.1
	 */
	public function render_menu_metabox() {
		/**
		 * Register menu metaboxs.
		 *
		 * @param array $register_meta_boxes Meta boxes.
		 *
		 * @since 1.5.2.1
		 */
		$register_meta_boxes = apply_filters( 'bbapp_register_menu_metaboxes', array() );

		$menu_type           = bbapp_get_menu_type();
		$logged_out_menu_id  = MenuManager::instance()->get_logout_menu_id( $menu_type );
		$default_menu_id     = MenuManager::instance()->get_default_menu_id( $menu_type );
		$current_nav_menu_id = isset( $_REQUEST['menu'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['menu'] ) ) : $default_menu_id; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! empty( $register_meta_boxes ) ) {
			foreach ( $register_meta_boxes as $register_meta_box ) {
				if ( isset( $register_meta_box['logged_in'] ) && $register_meta_box['logged_in'] && $logged_out_menu_id === $current_nav_menu_id ) {
					continue;
				}
				$screen   = sanitize_title_with_dashes( $register_meta_box['title'] );
				$callback = ! empty( $register_meta_box['callback'] ) ? $register_meta_box['callback'] : array(
					$this,
					'render_items_markup',
				);
				add_meta_box( "bbapp-{$screen}-metabox", $register_meta_box['title'], $callback, 'nav-menus', 'side', 'default', $register_meta_box['args'] );
			}
		}
	}

	/**
	 * Render items markup.
	 *
	 * @param object $post    post object if set.
	 * @param array  $metabox custom metabox args.
	 *
	 * @since 1.5.2.1
	 */
	public function render_items_markup( $post, $metabox ) {
		/**
		 * Get all menus.
		 *
		 * @since 1.5.2.1
		 */
		$all_menus = apply_filters( 'bbapp_get_all_menus', array() );

		if ( isset( $metabox['args']['screen_group'] ) ) {
			$screen_group  = $metabox['args']['screen_group'];
			$bbapp_menus   = isset( $all_menus[ $screen_group ] ) ? $all_menus[ $screen_group ] : array();
			$menu_desc     = '';
			$is_hide       = false;
			$deeplink_base = '';

			if ( ! empty( $metabox['args'] ) ) {
				if ( isset( $metabox['args']['desc'] ) ) {
					$menu_desc = $metabox['args']['desc'];
				}
				if ( isset( $metabox['args']['is_hide'] ) ) {
					$is_hide = $metabox['args']['is_hide'];
				}
				if ( isset( $metabox['args']['deeplink_base'] ) ) {
					$deeplink_base = $metabox['args']['deeplink_base'];
				}
			}

			include bbapp()->plugin_dir . 'views/menus/sections/markup.php';
		}
	}

	/**
	 * App icon render script.
	 *
	 * @since 1.7.0
	 */
	public function app_icon_render_script() {
		// Using PHP heredoc to avoid linter errors with JavaScript template syntax.
		$script = <<<'EOT'
		<script type="text/html" id="tmpl-buddyboss-app-render-icon">
			<div class="bbapp_icon_picker {{data.extra.uniq_id}}_container">
				<# 
				var jsonConvertedData = JSON.stringify(data);
				var defaultColor = 'undefined' !== typeof window.bb_app_preview_data ? window.bb_app_preview_data.colors.bottomTabsColor : "#000";
				var iconColor = 'default' !== data.color ? data.color : defaultColor;
				var box_style = ('box' === data.box_style && 'legacy' === data.type) ? "background:" + iconColor : '';
				var svg_fill_color = ('box' === data.box_style && 'legacy' === data.type ) ? '#fff' : iconColor;
				var foreground_color_style = '';
				
				if ('filled' === data.style && ('box' === data.box_style || 'round' === data.box_style)) {
					foreground_color_style = "background: " + data.fg_color + "; border-radius: 50%;";
				}
				#>
				<a href="javascript:void(0)" data-id="{{data.extra.uniq_id}}" class="icon-picker bbapp-icon-picker" data-type="{{data.type}}" data-icon="{{JSON.stringify(data)}}" style="{{box_style}}">
					<# if ('legacy' === data.type || 'custom' === data.type) { #>
					<img style="filter: url(#bbapp-{{data.extra.uniq_id}})" src="{{data.extra.img_url}}">
						<# if ('custom' !== data.type || 'custom' === data.type && true === data.fill_color) { #>
						<svg id="{{data.extra.uniq_id}}" style="position: absolute; height: 0; visibility: hidden; width: 0;">
							<defs>
								<filter id="bbapp-{{data.extra.uniq_id}}">
									<feFlood flood-color="{{svg_fill_color}}"></feFlood>
									<feComposite in2="SourceAlpha" operator="atop"></feComposite>
								</filter>
							</defs>
						</svg>
						<# } #>
					<# } #>
					<# if ('buddyboss' === data.type) { 
						var iconStyle = ('filled' === data.style) ? 'f' : 'l';
						var iconBoxStyle = '';

						if ('box' === data.box_style) {
							iconBoxStyle = 'b';
						} else if ('round' === data.box_style) {
							iconBoxStyle = 'r';
						}
						var iconApplyStyle = iconBoxStyle + iconStyle;
					#>
					<i class="bb-icon-{{iconApplyStyle}} bb-icon-{{data.id}}" style="color:{{svg_fill_color}};{{foreground_color_style}}"></i>
					<# } #>
				</a>
				<input type="hidden" class="bbapp_app_icon_picker_data" name="appmenu-item[{{data.extra.uniq_id}}][icon]" value="{{JSON.stringify(data)}}">
			</div>
		</script>
EOT;
		echo $script;
	}

	/**
	 * Get Tabbar Menu Item Count.
	 *
	 * @since 2.2.40
	 */
	public function get_tabbar_menu_count() {
		$nonce        = ! empty( $_POST['nonce'] ) ? bbapp_input_clean( wp_unslash( $_POST['nonce'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$headerbar_id = ! empty( $_POST['headerbar_id'] ) ? bbapp_input_clean( wp_unslash( $_POST['headerbar_id'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! wp_verify_nonce( $nonce, 'add-more-menu' ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce.', 'buddyboss-app' ) ) );
		}

		if ( empty( $headerbar_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid header bar.', 'buddyboss-app' ) ) );
		}

		$tabbar_id = MenuManager::instance()->get_tabbar_menu_from_headerbar_id( $headerbar_id );

		if ( $tabbar_id ) {
			$tabbar_items_count = MenuManager::instance()->get_menu_items_count( $tabbar_id );

			wp_send_json_success( array( 'count' => $tabbar_items_count ) );
		} else {
			wp_send_json_success( array( 'count' => 0 ) );
		}
	}

	/**
	 * AJAX handler for creating default menus.
	 *
	 * @since 2.4.10
	 * @return void
	 */
	public function ajax_create_default_menus() {
		// Check nonce for security verification.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'bbapp-create-default-menus' ) ) { //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			wp_send_json_error(
				array(
					'message' => __( 'Security check failed.', 'buddyboss-app' ),
				)
			);
		}

		// Check capabilities for the current user.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error(
				array(
					'message' => __( 'You do not have permission to perform this action.', 'buddyboss-app' ),
				)
			);
		}

		// Create the default menus using the MenuManager class.
		$menu_manager = MenuManager::instance();

		// Get parameters from the request.
		$menu_type     = isset( $_POST['menu_type'] ) ? sanitize_text_field( wp_unslash( $_POST['menu_type'] ) ) : '';
		$login_state   = isset( $_POST['login_state'] ) ? intval( $_POST['login_state'] ) : 1;
		$language_code = isset( $_POST['lang'] ) ? sanitize_text_field( wp_unslash( $_POST['lang'] ) ) : $menu_manager->current_lang_code;

		// Determine which menu types to create based on the provided menu type.
		$types_to_create      = empty( $menu_type ) ? array( 'tabbar', 'more' ) : array( $menu_type );
		$created_menus        = array();
		$created_tabbar_id    = null;
		$created_headerbar_id = null;

		// First pass: Create the menus if they don't exist.
		foreach ( $types_to_create as $type ) {
			// Create the menu if it doesn't exist for the specified login state.
			if ( ! $menu_manager->default_menu_exists( $type, $login_state, $language_code ) ) {
				// Prepare menu name based on login state.
				$menu_name = ( 0 === $login_state ) ? 'Logged-out' : 'Default';

				// Create menu data.
				$menu_data = array(
					'menu_name'     => $menu_name,
					'login_state'   => $login_state,
					'menu_type'     => $type,
					'language_code' => $language_code,
					'priority'      => $this->get_appropriate_menu_priority( $menu_name, $login_state ),
				);

				$menu_id = $menu_manager->create_menu( $type, $menu_data );

				if ( $menu_id ) {
					$created_menus[] = array(
						'type'        => $type,
						'login_state' => $login_state,
						'id'          => $menu_id,
					);

					// Store created tabbar ID for later use if needed.
					if ( 'tabbar' === $type ) {
						$created_tabbar_id = $menu_id;
						// Get the linked headerbar ID if it exists.
						$created_headerbar_id = $menu_manager->get_headerbar_menu_from_tabbar_id( $menu_id );
					}
				}
			}
		}

		// Second pass: Add default items to the headerbar if needed and not already created.
		if ( 1 === $login_state && $created_headerbar_id ) {
			// Add to a created menus list if not already there to avoid duplicates.
			$found = false;
			foreach ( $created_menus as $menu ) {
				if ( 'headerbar' === $menu['type'] && $menu['id'] === $created_headerbar_id ) {
					$found = true;
					break;
				}
			}

			if ( ! $found ) {
				$created_menus[] = array(
					'type'        => 'headerbar',
					'login_state' => $login_state,
					'id'          => $created_headerbar_id,
				);
			}
		}

		if ( empty( $created_menus ) ) {
			wp_send_json_error(
				array(
					'message' => __( 'No menus were created. Default menus may already exist.', 'buddyboss-app' ),
				)
			);
		} else {
			wp_send_json_success(
				array(
					'message'       => sprintf(
					// translators: %d is the number of created menus.
						_n(
							'%d default menu structure has been created successfully. You can now add menu items.',
							'%d default menu structures have been created successfully. You can now add menu items.',
							count( $created_menus ),
							'buddyboss-app'
						),
						count( $created_menus )
					),
					'created_menus' => $created_menus,
				)
			);
		}
	}
}
