<?php
/**
 * Holds menu related admin functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

/**
 * App Menus class.
 */
class Menus {

	/**
	 * Class instance.
	 *
	 * @since 1.5.2.1
	 * @var object
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.5.2.1
	 * @return Menus
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Using Singleton, see instance()
	 *
	 * @since 1.5.2.1
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Add related class and hooks
	 *
	 * @since 1.5.2.1
	 */
	public function load() {
		Menus\Menus::instance();
		add_action( 'admin_init', array( $this, 'admin_init' ) );
	}

	/**
	 * Load admin init actions and classes.
	 *
	 * @since 1.5.2.1
	 */
	public function admin_init() {
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'admin.php' === SetupAdmin::instance()->get_page_now() && ! empty( $page_get ) && 'bbapp-menus' === $page_get ) {
			add_filter( 'admin_body_class', array( $this, 'custom_body_class' ), 99 );

			// Load the app info so it will update the license type.
			\BuddyBossApp\ManageApp::instance()->verify_license_data();
		}
	}

	/**
	 * Return the sub settings.
	 *
	 * @since 1.5.2.1
	 * @return array
	 */
	public function sub_settings() {
		$return = array(
			'tabbar'    => __( 'Tab Bar', 'buddyboss-app' ),
			'headerbar' => __( 'Header Bar', 'buddyboss-app' ),
			'more'      => __( 'More Menu', 'buddyboss-app' ),
		);

		return $return;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @since 1.5.2.1
	 * @return string
	 */
	public function get_current_sub_tab() {
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'bbapp-menus' !== $page_get ) {
			return null;
		}

		$default = 'tabbar';
		$setting = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Renders the setting screen.
	 *
	 * @since 1.5.2.1
	 */
	public function render_screen() {
		add_action( 'admin_footer', array( $this, 'footer' ), 50 );

		SetupAdmin::instance()->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab() );

		if ( in_array( $this->get_current_sub_tab(), array( 'tabbar', 'more', 'headerbar' ), true ) ) {
			Menus\Menus::instance()->render();
		}
	}

	/**
	 * Added footer hook and related data.
	 *
	 * @since 1.5.2.1
	 */
	public function footer() {
		// Localize Icon picker.
		wp_localize_script(
			'bbapp-script',
			'iconsPicker',
			array(
				'bbapp_icons_path'       => \BuddyBossApp\Common\IconPicker::instance()->icon_picker_app_icon_url(),
				'bbapp_custom_icons_url' => \BuddyBossApp\Common\IconPicker::instance()->icon_picker_custom_icon_url(),
			)
		);
	}

	/**
	 * Set a custom css on body of the page where current render will initiate.
	 *
	 * @param string $classes Classes.
	 *
	 * @since 1.5.2.1
	 * @return string
	 */
	public function custom_body_class( $classes ) {
		if ( false === strpos( $classes, 'nav-appmenu' ) ) {
			$classes .= ' nav-appmenu bbapp-menu buddyboss-app-menu';
		}

		return $classes;
	}
}
