<?php
/**
 * Holds app management functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\AppStores\Android;
use BuddyBossApp\AppStores\Apple;
use BuddyBossApp\Helpers\BBAPP_File;
use MingYuanYun\AppStore\Exceptions\ConfigException;
use WP_Error;

/**
 * Manage app class.
 */
class ManageApp {

	/**
	 * Class instnace.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return ManageApp
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * ManageApp constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Load hooks here.
	 */
	public function load() {
		$this->hooks(); // WordPress hooks.
	}

	/**
	 * Hooks here.
	 */
	public function hooks() {
		$this->ajax_hooks( 'bbapp_app_connection', array( $this, 'ajax_manage_app' ) );
		$this->ajax_hooks( 'bbapp_site_disconnect', array( $this, 'ajax_site_disconnect' ) );
		$this->ajax_hooks( 'bbapp_secondary_site_cancel', array( $this, 'ajax_secondary_site_cancel' ) );
		$this->ajax_hooks( 'bpapp_download_keystore', array( $this, 'bpapp_download_keystore' ) );
		$this->ajax_hooks( 'bbapp_validate_namespace', array( $this, 'bbapp_validate_namespace' ) );
		$this->ajax_hooks( 'bbapp_validate_google_permissions', array( $this, 'bbapp_validate_google_permissions' ) );

		add_action( 'admin_init', array( $this, 'load_init' ) );
	}

	/**
	 * Init
	 */
	public function load_init() {
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'admin.php' === SetupAdmin::instance()->get_page_now() && 'bbapp-tools' === $page_get ) {
			$settings_get = ( ! empty( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( 'generate-keystore' === $settings_get ) {
				$settings_get = ( ! empty( $_GET['success'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['success'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

				if ( true === (bool) $settings_get ) {
					$this->keystore_file_delete();
				}
			}
		}
	}

	/**
	 * Function to check the permission;
	 */
	public function bbapp_validate_google_permissions() {
		$result   = array();
		$validate = \BuddyBossApp\AppStores\Services\Android\AndroidPublisher::instance()->operation_permission();

		if ( true === $validate ) {
			$result['message'] = __( 'All checks passed.', 'buddyboss-app' );

			wp_send_json_success( $result );
		} else {
			$result['message'] = __( 'The service account key provided does not have all the required permissions. Please resolve in your Google Developer account', 'buddyboss-app' );

			wp_send_json_error( $result );
		}
	}

	/**
	 * Delete keystore file.
	 */
	public function keystore_file_delete() {
		$upload          = wp_upload_dir();
		$keystore_dir    = $upload['basedir'] . '/bbapp/keystore/';
		$export_zip_path = trailingslashit( $keystore_dir ) . 'buddyboss-app-keystore.zip';

		if ( file_exists( $export_zip_path ) ) {
			wp_delete_file( $export_zip_path );
		}
	}

	/**
	 * Download key store record.
	 */
	public function bpapp_download_keystore() {
		$result            = array();
		$result['message'] = esc_html__( 'Something went wrong. Please try again later.', 'buddyboss-app' );
		$nonce             = ( ! empty( $_POST['keystore_nonce'] ) ) ? wp_unslash( $_POST['keystore_nonce'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$password          = filter_input( INPUT_POST, 'password', FILTER_DEFAULT );
		$alias             = ( ! empty( $_POST['alias'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['alias'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$fl_name           = ( ! empty( $_POST['fl_name'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['fl_name'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$organization      = ( ! empty( $_POST['organization'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['organization'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$organization_unit = ( ! empty( $_POST['organization_unit'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['organization_unit'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$city              = ( ! empty( $_POST['city'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['city'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$state             = ( ! empty( $_POST['state'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['state'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$country_code      = ( ! empty( $_POST['country_code'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['country_code'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$error_message     = __( 'There was an error generating your KeyStore certificate:', 'buddyboss-app' ) . '<br/>';
		$is_error          = false;

		if ( ! wp_verify_nonce( $nonce, 'bbapp_generate_keystore' ) ) {
			$error_message .= __( '- Security nonce verification failed. Please retry after refreshing your page.', 'buddyboss-app' ) . '<br/>';
			$is_error       = true;
		}

		if ( empty( $nonce ) ) {
			$error_message .= __( '- Nonce is not exists.', 'buddyboss-app' ) . '<br/>';
			$is_error       = true;
		}

		if ( empty( $password ) ) {
			$error_message .= __( '- KeyStore Password is required to generate KeyStore.', 'buddyboss-app' ) . '<br/>';
			$is_error       = true;
		}

		if ( empty( $alias ) ) {
			$error_message .= __( '- KeyStore Alias is required to generate KeyStore.', 'buddyboss-app' ) . '<br/>';
			$is_error       = true;
		}

		if ( empty( $fl_name ) ) {
			$error_message .= __( '- First and Last name is required to generate KeyStore.', 'buddyboss-app' ) . '<br/>';
			$is_error       = true;
		}

		if ( true === $is_error ) {
			$result['message'] = sprintf( '<div class="bbapp-notice-box bbapp-notice-box--error">%s</div>', $error_message );

			wp_send_json_error( $result );
		}

		$body_params = array(
			'password'           => $password,
			'storepass'          => $password,
			'alias_name'         => sanitize_title_with_dashes( $alias ),
			'firstname_lastname' => $fl_name,
		);

		if ( ! empty( $organization_unit ) ) {
			$body_params['organization_unit'] = $organization_unit;
		}

		if ( ! empty( $organization ) ) {
			$body_params['organization_name'] = $organization;
		}

		if ( ! empty( $city ) ) {
			$body_params['locality'] = $city;
		}

		if ( ! empty( $state ) ) {
			$body_params['local_state'] = $state;
		}

		if ( ! empty( $country ) ) {
			$body_params['country'] = $country;
		}

		if ( ! empty( $country_code ) ) {
			$body_params['country_code'] = $country_code;
		}

		$app = \BuddyBossApp\ManageApp::instance()->get_app();

		$do_request = bbapp_remote_post(
			'https://bbapp-cert-services.buddyboss.com/api/keystore',
			array(
				'method'  => 'POST',
				'timeout' => 45,
				'body'    => $body_params,
				'headers' => array(
					'bbapp_id'  => empty( $app ) ? false : $app['bbapp_app_id'],
					'bbapp_key' => \BuddyBossApp\ManageApp::instance()->get_auth_app_key(),
				),
			)
		);

		if ( is_wp_error( $do_request ) || empty( $do_request ) ) {
			$error_message    .= esc_html__( '- Something went wrong. Please try again later.', 'buddyboss-app' );
			$result['message'] = sprintf( '<div class="bbapp-notice-box bbapp-notice-box--error">%s</div>', $error_message );

			wp_send_json_error( $result );
		}

		$body     = wp_remote_retrieve_body( $do_request );
		$p12_file = $body;
		$body     = json_decode( $body, true );

		if ( isset( $body['status'] ) && 'error' === $body['status'] && isset( $body['message'] ) ) {
			if ( isset( $body['message'] ) && ! empty( $body['message'] ) && is_array( $body['message'] ) ) {
				foreach ( $body['message'] as $key => $message ) {
					$error_message .= '- ' . $message[0] . '<br/>';
				}
			} else {
				$error_message .= '- ' . $body['message'] . '<br/>';
			}

			$result['message'] = sprintf( '<div class="bbapp-notice-box bbapp-notice-box--error">%s</div>', $error_message );

			wp_send_json_error( $result );
		}

		$keystore_jks_file_path = '';
		$body                   = $p12_file;

		if ( ! empty( $body ) && 200 === wp_remote_retrieve_response_code( $do_request ) ) {
			/**
			 * Get/Create JSK file
			 */
			$file_name = wp_remote_retrieve_header( $do_request, 'content-disposition' );

			if ( empty( $file_name ) ) {
				$error_message    .= '- ' . $body;
				$result['message'] = sprintf( '<div class="bbapp-notice-box bbapp-notice-box--error">%s</div>', $error_message );
				wp_send_json_error( $result );
			}

			$file_name     = str_replace( 'inline; filename=', '', $file_name );
			$file_ext      = pathinfo( $file_name, PATHINFO_EXTENSION );
			$file_name     = strtotime( gmdate( 'Y-m-d H:i:s' ) ) . '.' . $file_ext;
			$upload        = wp_upload_dir();
			$file_path     = $upload['basedir'] . '/bbapp/keystore/';
			$file_uploaded = BBAPP_File::file_handler( $file_path . $file_name, $body );

			if ( ! empty( $file_uploaded ) || ! is_wp_error( $file_uploaded ) ) {
				$keystore_jks_file_path = $file_path . $file_name;
			}

			/**
			 * Create Zip for .jks and detials.
			 */
			$upload          = wp_upload_dir();
			$keystore_dir    = $upload['basedir'] . '/bbapp/keystore/';
			$keystore_url    = $upload['baseurl'] . '/bbapp/keystore/';
			$root_path       = realpath( $keystore_dir );
			$export_zip_path = trailingslashit( $keystore_dir ) . 'buddyboss-app-keystore.zip';
			$export_zip_url  = trailingslashit( $keystore_url ) . 'buddyboss-app-keystore.zip';

			// Initialize archive object.
			$zip = new \ZipArchive();

			if ( $zip->open( $export_zip_path, \ZipArchive::CREATE | \ZipArchive::OVERWRITE ) ) {
				if ( file_exists( $keystore_jks_file_path ) ) {
					// Put jks file into zip.
					$relative_path = substr( $keystore_jks_file_path, strlen( $root_path ) + 1 );
					$zip->addFile( $keystore_jks_file_path, $relative_path ); // Add file into the Zip.

					// Creating keystore data file and put the key store data.
					BBAPP_File::write_file( "{$keystore_dir}keystore_data.json", wp_json_encode( $body_params, JSON_PRETTY_PRINT ) );
					$keystore_json_file_path = "{$keystore_dir}keystore_data.json";
					$relative_path           = substr( $keystore_json_file_path, strlen( $root_path ) + 1 );

					$zip->addFile( $keystore_json_file_path, $relative_path ); // Add file into the Zip.
					$zip->close(); // // Zip archive will be created only after closing object

					if ( file_exists( $keystore_jks_file_path ) ) {
						wp_delete_file( $keystore_jks_file_path );
					}

					if ( file_exists( $keystore_json_file_path ) ) {
						wp_delete_file( $keystore_json_file_path );
					}

					wp_send_json_success(
						array(
							'name' => 'buddyboss-app-keystore.zip',
							'url'  => $export_zip_url,
						)
					);
				} else {
					$error_message    .= __( '- File not exists.', 'buddyboss-app' );
					$result['message'] = sprintf( '<div class="bbapp-notice-box bbapp-notice-box--error">%s</div>', $error_message );

					wp_send_json_error( $result );
				}
			} else {
				$error_message    .= __( '- Sorry the provided font is invalid or not supported to download.', 'buddyboss-app' );
				$result['message'] = sprintf( '<div class="bbapp-notice-box bbapp-notice-box--error">%s</div>', $error_message );

				wp_send_json_error( $result );
			}
		}
	}

	/**
	 * Function to validate namespace.
	 */
	public function bbapp_validate_namespace() {
		$result                 = array();
		$name                   = ( ! empty( $_POST['name'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['name'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$platform               = ( ! empty( $_POST['platform'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['platform'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$success_cls            = 'notice-success bbapp-notice-box bbapp-notice-box--success';
		$error_cls              = 'notice-error bbapp-notice-box bbapp-notice-box--error';
		$result['message']      = '';
		$alpha_numeric_cls      = ! preg_match( '/^[a-z0-9_.]+$/', $name ) ? $error_cls : $success_cls;
		$segment_cls            = ! strpos( $name, '.' ) ? $error_cls : $success_cls;
		$individual_segment_cls = ! preg_match( '/^[a-z][a-z0-9_]*(\.[a-z][a-z0-9_]*)+$/', $name ) ? $error_cls : $success_cls;

		$result['message'] .= sprintf( '<div class="%s">%s</div>', esc_attr( $segment_cls ), __( 'Application IDs must have at least two segments, separated by a dot', 'buddyboss-app' ) );
		$result['message'] .= sprintf( '<div class="%s">%s</div>', esc_attr( $individual_segment_cls ), __( 'Each segment in your Application ID must start with a letter', 'buddyboss-app' ) );
		$result['message'] .= sprintf( '<div class="%s">%s</div>', esc_attr( $alpha_numeric_cls ), __( 'Only lowercase letters, numbers or underscore can be used in application IDs', 'buddyboss-app' ) );

		if ( 'android' === $platform ) {
			if ( 200 === Android::instance()->validate_package_name( $name ) ) {
				$result['message'] .= sprintf( '<div class="notice-warning bbapp-notice-box bbapp-notice-box--warning">%s</div>', __( 'This Application ID is already used by a published app in the Google Play Store. You will only be able to publish a build using this Application ID if it is registered to you.', 'buddyboss-app' ) );
			} else {
				$result['message'] .= sprintf( '<div class="notice-success bbapp-notice-box bbapp-notice-box--success">%s</div>', __( 'This Application ID is not being used by a published app in the Google Play Store', 'buddyboss-app' ) );
			}
		}

		wp_send_json_success( $result );
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {
		return array( 'Manage' => __( 'Manage', 'buddyboss-app' ) );
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {
		$default      = 'manage';
		$setting      = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Renders the setting screen.
	 */
	public function render_screen() {
		\BuddyBossApp\Admin\SetupAdmin::instance()->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab() );

		if ( 'manage' === $this->get_current_sub_tab() ) {
			include bbapp()->plugin_dir . 'views/settings/manage_app.php';
		}
	}

	/**
	 * Ajax hooks helper.
	 *
	 * @param string $action Action name of ajax hook.
	 * @param string $method Method to be call.
	 */
	public function ajax_hooks( $action, $method ) {
		add_action( "wp_ajax_nopriv_$action", $method );
		add_action( "wp_ajax_$action", $method );
	}

	/**
	 * Admin hooks helper.
	 *
	 * @param string $action Action name of admin post hook.
	 * @param string $method Method to be call.
	 */
	public function admin_post_hooks( $action, $method ) {
		add_action( "admin_post_nopriv_$action", $method );
		add_action( "admin_post_$action", $method );
	}

	/**
	 * Handle Ajax Request Related to App Connect & Disconnect.
	 */
	public function ajax_manage_app() {
		if ( ! is_user_logged_in() ) {
			bbapp_ajax_error_response( 'no_allowed', __( "You don't have permission to do this action.", 'buddyboss-app' ) );
		}

		// check if current user can do manage app.
		if ( ! \BuddyBossApp\Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			bbapp_ajax_error_response( 'no_allowed', __( "You don't have permission to do this action. Action can be only performed from network.", 'buddyboss-app' ) );
		}

		$do          = ( isset( $_POST['do'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['do'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$allowed_dos = array( 'connect', 'disconnect' );

		if ( ! in_array( $do, $allowed_dos, true ) ) {
			bbapp_ajax_error_response( 'no_allowed', __( "You don't have permission to do this action.", 'buddyboss-app' ) );
		}

		// Verify Nonce.
		$nonce_post = ( ! empty( $_POST['_wpnonce'] ) ) ? wp_unslash( $_POST['_wpnonce'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! wp_verify_nonce( $nonce_post, 'bbapp_manage_app_nonce' ) ) {
			bbapp_ajax_error_response( 'security_nonce', __( 'Unable to verify security nonce, Please refresh the page and try again.', 'buddyboss-app' ) );
		}

		if ( 'connect' === $do ) {
			if ( ! isset( $_POST['bbapp_app_id'] ) || ! isset( $_POST['bbapp_app_key'] ) ) {
				bbapp_ajax_error_response( 'data_missing', __( 'Please provide BuddyBoss App ID & App Key.', 'buddyboss-app' ) );
			}

			$bbapp_app_id    = ( ! empty( $_POST['bbapp_app_id'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['bbapp_app_id'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$bbapp_app_key   = ( ! empty( $_POST['bbapp_app_key'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['bbapp_app_key'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$bbapp_site_type = ( ! empty( $_POST['bbapp_site_type'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['bbapp_site_type'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$connect         = \BuddyBossApp\ManageApp::instance()->connect_app( $bbapp_app_id, $bbapp_app_key, $bbapp_site_type );

			if ( ! $connect || is_wp_error( $connect ) ) {
				if ( is_wp_error( $connect ) ) {
					if ( 'primary_not_possible' === $connect->get_error_code() ) {
						ob_start();
						include bbapp()->plugin_dir . 'views/settings/connect/secondary-site-confirm.php';
						$content = ob_get_clean();
						bbapp_ajax_error_response( $connect->get_error_code(), $content );
					}

					/**
					 * Ajax error response.
					 *
					 * @var $connect WP_Error
					 */
					bbapp_ajax_error_response( $connect->get_error_code(), $connect->get_error_message() );
				} else {
					bbapp_ajax_error_response( 'error_connecting', __( 'Unknown error while connecting app', 'buddyboss-app' ) );
				}
			} else {
				ob_start();
				?>
				<div class="bbapp-notice-box bbapp-notice-box--success">
					<?php if ( 'secondary' === $bbapp_site_type ) : ?>
						<?php esc_html_e( 'This site is now connected as a secondary site.', 'buddyboss-app' ); ?>
					<?php else : ?>
						<?php esc_html_e( 'This site is now connected as your primary site.', 'buddyboss-app' ); ?>
					<?php endif; ?>
				</div>
				<?php
				include bbapp()->plugin_dir . 'views/settings/connect/main-connect.php';
				$content = ob_get_clean();
				bbapp_ajax_success_response( 'connected', $content );
			}
		}

		if ( 'disconnect' === $do ) {
			$app        = \BuddyBossApp\ManageApp::instance()->get_app();
			$disconnect = \BuddyBossApp\ManageApp::instance()->disconnect_app( $app['bbapp_app_id'], $app['bbapp_app_key'], $app['bbapp_site_type'] );

			if ( $disconnect ) {
				ob_start();
				?>
				<div class="bbapp-notice-box bbapp-notice-box--success">
					<?php esc_html_e( 'You have successfully disconnected this site.', 'buddyboss-app' ); ?>
				</div>
				<?php
				include bbapp()->plugin_dir . 'views/settings/connect/main-connect.php';
				$content = ob_get_clean();
				bbapp_ajax_success_response( 'disconnected', $content );
			} else {
				bbapp_ajax_error_response( 'error_disconnecting', __( 'Unknown error while attempting to disconnect the app. Please refresh the page and try again.', 'buddyboss-app' ) );
			}
		}
	}

	/**
	 * Handle Ajax Request Related to Site Disconnect.
	 */
	public function ajax_site_disconnect() {
		if ( ! is_user_logged_in() ) {
			bbapp_ajax_error_response( 'no_allowed', __( 'You don\'t have permission to perform this action.', 'buddyboss-app' ) );
		}

		// check if current user can do manage app.
		if ( ! \BuddyBossApp\Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			bbapp_ajax_error_response( 'no_allowed', __( 'You don\'t have permission to perform this action. The action can only be performed from the network.', 'buddyboss-app' ) );
		}

		// Verify Nonce.
		$nonce_post = ( ! empty( $_POST['_wpnonce'] ) ) ? wp_unslash( $_POST['_wpnonce'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! wp_verify_nonce( $nonce_post, 'bbapp_manage_app_nonce' ) ) {
			bbapp_ajax_error_response( 'security_nonce', __( 'Unable to verify security nonce, Please refresh the page and try again.', 'buddyboss-app' ) );
		}

		if ( empty( $_POST['site_url'] ) ) {
			bbapp_ajax_error_response( 'security_nonce', __( 'Please provide site URL to disconnect.', 'buddyboss-app' ) );
		}

		$app = \BuddyBossApp\ManageApp::instance()->get_app();

		if ( ! $app ) {
			bbapp_ajax_error_response( 'no_allowed', __( 'The app you are trying to perform the action on was not found.', 'buddyboss-app' ) );
		}

		$site_url   = bbapp_input_clean( wp_unslash( $_POST['site_url'] ) ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$disconnect = \BuddyBossApp\ManageApp::instance()->disconnect_app( $app['bbapp_app_id'], $app['bbapp_app_key'], 'secondary', $site_url );

		if ( $disconnect ) {
			bbapp_ajax_success_response( 'disconnected', __( 'Site successfully disconnected.', 'buddyboss-app' ) );
		} else {
			bbapp_ajax_error_response( 'error_disconnecting', __( 'Unknown error while attempting to disconnect the app. Please refresh the page and try again.', 'buddyboss-app' ) );
		}
	}

	/**
	 * Handle Ajax Request Related to Site secondary Cancel.
	 */
	public function ajax_secondary_site_cancel() {
		if ( ! is_user_logged_in() ) {
			bbapp_ajax_error_response( 'no_allowed', __( 'You don\'t have permission to perform this action.', 'buddyboss-app' ) );
		}

		// check if current user can do manage app.
		if ( ! \BuddyBossApp\Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			bbapp_ajax_error_response( 'no_allowed', __( 'You don\'t have permission to perform this action. The action can only be performed from the network.', 'buddyboss-app' ) );
		}

		$app = \BuddyBossApp\ManageApp::instance()->get_app();

		if ( ! $app ) {
			bbapp_ajax_error_response( 'no_allowed', __( 'The app you are trying to perform the action on was not found.', 'buddyboss-app' ) );
		}

		ob_start();
		include bbapp()->plugin_dir . 'views/settings/connect/main-connect.php';
		$content = ob_get_clean();
		bbapp_ajax_success_response( 'disconnected', $content );
	}

	/**
	 * Verify the app stores connections.
	 *
	 * Check CronJobs.php
	 *
	 * @since 1.4.3
	 */
	public function verify_apps_connection() {
		$is_app_center_connected = \BuddyBossApp\ManageApp::instance()->is_app_center_connected( 'primary' );

		if ( true === $is_app_center_connected ) {
			/**
			 * Check for Apple Connection
			 */
			if ( 0 === (int) $this->get_connection_failed_attempt( 'ios' ) ) { // Only check if there are no retry attempts requires.
				$issuer_id   = \BuddyBossApp\AppStores\Apple::instance()->get_issuer_id();
				$key_id      = \BuddyBossApp\AppStores\Apple::instance()->get_key_id();
				$private_key = \BuddyBossApp\AppStores\Apple::instance()->get_private_key();

				if ( ! empty( $issuer_id ) && ! empty( $key_id ) && ! empty( $private_key ) ) {
					$apple_connection = \BuddyBossApp\AppStores\Apple::instance()->is_connected( true );

					// Apple connection.
					if ( true !== $apple_connection ) {
						// run cron for 30 min to check apple connect.
						$this->set_connection_failed_attempt( 'ios' );
					} else {
						$this->set_connection_failed_attempt( 'ios', true );
					}
				}
			}

			/**
			 * Check for Android Connection.
			 */
			if ( 0 === (int) $this->get_connection_failed_attempt( 'android' ) ) { // Only check if there are no retry attempts requires.
				$file_name = \BuddyBossApp\AppStores\Android::instance()->get_private_key();

				if ( ! empty( $file_name ) ) {
					$android_connection = ( ! empty( Android::instance()->is_google_connected( true ) ) ) ? true : false;

					// Android connection.
					if ( empty( $android_connection ) || is_wp_error( $android_connection ) ) {
						// run cron for 30 min to check apple connect.
						$this->set_connection_failed_attempt( 'android' );
					} else {
						$this->set_connection_failed_attempt( 'android', true );
					}
				}
			}
		}
	}

	/**
	 * Set connection failed attempt.
	 *
	 * @param string $platform       App platform.
	 * @param bool   $remove_attempt Remove attempt count.
	 * @param string $log_msg        Log message.
	 *
	 * @since 1.4.3
	 */
	public function set_connection_failed_attempt( $platform, $remove_attempt = false, $log_msg = '' ) {
		$bbapp_connection_failed_attempt = $this->get_connection_failed_attempt( 'all' );

		if ( false === $remove_attempt ) {
			$attempt                                      = isset( $bbapp_connection_failed_attempt[ $platform ] ) ? intval( $bbapp_connection_failed_attempt[ $platform ] ) + 1 : 1;
			$bbapp_connection_failed_attempt[ $platform ] = $attempt;

			if ( $attempt <= 20 ) {
				$platform_type = ( 'ios' === $platform ) ? __( 'iOS', 'buddyboss-app' ) : __( 'Android', 'buddyboss-app' );
				\BuddyBossApp\ManageApp::instance()->bbapp_appcenter_app_log( $platform, 'connection_attempt', sprintf( '(%d) Connection attempt to %s failed.', (int) $attempt, $platform_type ) );
			}
		} else {
			$bbapp_connection_failed_attempt[ $platform ] = 0; // reset the attempts.

			if ( ! empty( $log_msg ) ) {
				/**
				 * Disconnect account based on platform.
				 */
				$this->disconnect_account( $platform, $log_msg );
			}
		}

		bbapp_set_network_option( 'bbapp_connection_failed_attempt', $bbapp_connection_failed_attempt );
	}

	/**
	 * Get iOS or Android connection attempt.
	 *
	 * @param string $platform App platform.
	 *
	 * @return int|mixed
	 * @since 1.4.3
	 */
	public function get_connection_failed_attempt( $platform = 'ios' ) {
		$bbapp_connection_failed_attempt = bbapp_get_network_option( 'bbapp_connection_failed_attempt' );
		$bbapp_connection_failed_attempt = ( ! empty( $bbapp_connection_failed_attempt ) ) ? $bbapp_connection_failed_attempt : array();

		if ( 'all' === $platform ) {
			return $bbapp_connection_failed_attempt;
		}

		if ( isset( $bbapp_connection_failed_attempt[ $platform ] ) ) {
			return $bbapp_connection_failed_attempt[ $platform ];
		}

		// Default Attempts.
		$bbapp_connection_failed_attempt[ $platform ] = 0;

		return $bbapp_connection_failed_attempt[ $platform ];
	}

	/**
	 * Reconnect Apple and Android account in every 30 min.
	 *
	 * Reattempts only happens when connection failed attempts are more then 0.
	 *
	 * Check CronJobs.php
	 *
	 * @throws ConfigException IOS connect exception message.
	 * @since 1.4.3
	 */
	public function attempt_to_reconnect_account() {
		$is_app_center_connected = \BuddyBossApp\ManageApp::instance()->is_app_center_connected( 'primary' );

		if ( true === $is_app_center_connected ) {
			/**
			 * Apple connection.
			 */
			$ios_connection_attempt = self::instance()->get_connection_failed_attempt( 'ios' );

			if ( 0 !== $ios_connection_attempt ) {
				$issuer_id   = Apple::instance()->get_issuer_id();
				$key_id      = Apple::instance()->get_key_id();
				$private_key = Apple::instance()->get_private_key();

				if ( ! empty( $issuer_id ) && ! empty( $key_id ) && ! empty( $private_key ) ) {
					$apple_connection = Apple::instance()->is_connected( true );

					if ( empty( $apple_connection ) || is_wp_error( $apple_connection ) ) {
						$this->set_connection_failed_attempt( 'ios' );
						$ios_connection_attempt ++; // add this attempt into last value.

						/**
						 * When failed attempts exceed max limit disconnect the store.
						 */
						if ( isset( $ios_connection_attempt ) && $ios_connection_attempt > 20 ) {
							$error = Apple::instance()->get_error_message();

							if ( ! empty( $error ) ) {
								$this->set_connection_failed_attempt( 'ios', true, 'Failed connection attempts exceeded disconnecting iOS.' );

							}
						}
					} else {
						$this->set_connection_failed_attempt( 'ios', true );
					}
				}
			}

			/**
			 * Android connection.
			 */
			$android_connection_attempt = self::instance()->get_connection_failed_attempt( 'android' );

			if ( 0 !== $android_connection_attempt ) {
				$android_connection = ( ! empty( Android::instance()->is_google_connected( true ) ) ) ? true : false;

				if ( empty( $android_connection ) || is_wp_error( $android_connection ) ) {
					$this->set_connection_failed_attempt( 'android' );
					$android_connection_attempt ++; // add this attempt into last value.

					/**
					 * When failed attempts exceed max limit disconnect the store.
					 */
					if ( isset( $android_connection_attempt ) && $android_connection_attempt > 20 ) {
						$file_name = Android::instance()->get_private_key();

						if ( false === $android_connection && ! empty( $file_name ) ) {
							$this->set_connection_failed_attempt( 'android', true, 'Failed connection attempts exceeded disconnecting Android.' );
						}
					}
				} else {
					$this->set_connection_failed_attempt( 'android', true );
				}
			}
		}
	}

	/**
	 * Remove android account keys from plugin.
	 *
	 * @param string $platform platform eg.ios|android.
	 * @param string $log_msg  default blank.
	 *
	 * @since 1.4.3
	 */
	public function disconnect_account( $platform, $log_msg = '' ) {
		if ( 'ios' === $platform ) {
			$upload_dir = wp_upload_dir();
			$pvt_key    = Configure::instance()->option( 'publish.ios.private_key' );

			if ( ! empty( $pvt_key ) && file_exists( $upload_dir['basedir'] . $pvt_key ) ) {
				wp_delete_file( $upload_dir['basedir'] . $pvt_key );
			}

			Configure::instance()->save_option( 'publish.ios.private_key', '' );
			Configure::instance()->save_option( 'publish.ios.issuer_id', '' );
			Configure::instance()->save_option( 'publish.ios.key_id', '' );

			if ( ! empty( $log_msg ) ) {
				// Add App Center log.
				\BuddyBossApp\ManageApp::instance()->bbapp_appcenter_app_log( 'ios', 'clear_data', $log_msg );
			}
		}

		if ( 'android' === $platform ) {
			$upload_dir = wp_upload_dir();
			$acc_key    = Configure::instance()->option( 'publish.android.account_key' );

			if ( ! empty( $acc_key ) && file_exists( $upload_dir['basedir'] . $acc_key ) ) {
				wp_delete_file( $upload_dir['basedir'] . $acc_key );
			}

			Configure::instance()->save_option( 'publish.android.account_key', '' );

			if ( ! empty( $log_msg ) ) {
				// Add App Center log.
				\BuddyBossApp\ManageApp::instance()->bbapp_appcenter_app_log( 'android', 'clear_data', $log_msg );
			}
		}
	}
}
