<?php
/**
 * Holds third party integration class functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\ManageApp;

/**
 * Integration class.
 */
class Integrations {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Settings form.
	 *
	 * @var $settings_form
	 */
	public $settings_form;

	/**
	 * Selector args.
	 *
	 * @var array[]
	 */
	public $selector_args = array(
		'left'  => array(
			'posts'       => array(),
			'total_items' => 0,
		),
		'right' => array(
			'posts'       => array(),
			'total_items' => 0,
		),
	);

	/**
	 * Item per page.
	 *
	 * @var int
	 */
	public $item_per_page = 10;

	/**
	 * Get the instance of the class.
	 *
	 * @return Settings
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Integrations constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		add_action( 'init', array( $this, 'load_init' ) );
		add_action( 'wp_ajax_bbapp_selector_pager', array( $this, 'bbapp_selector_pager_callback' ) );

		$this->selector_args['left']['args']  = array(
			'post_type'      => 'sfwd-courses',
			'posts_per_page' => $this->item_per_page,
			'paged'          => 1,
			'order'          => 'desc',
			'orderby'        => 'date',
		);
		$this->selector_args['right']['args'] = array(
			'post_type'      => 'sfwd-courses',
			'posts_per_page' => $this->item_per_page,
			'paged'          => 1,
			'order'          => 'desc',
			'orderby'        => 'date',
			'meta_query'     => array( //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
				array(
					'key'     => '_hide_in_app',
					'value'   => 'yes',
					'compare' => '=',
				),
			),
		);
	}

	/**
	 * Init method.
	 *
	 * @return false|void
	 */
	public function load_init() {
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		/**
		 * Only load on bbapp-settings page.
		 */
		if ( 'admin.php' !== SetupAdmin::instance()->get_page_now() || empty( $page_get ) || 'bbapp-integrations' !== $page_get ) {
			return false;
		}

		// If the current sub-tab is link redirect it.
		if ( bbapp_is_valid_url( $this->get_current_sub_tab() ) && ! isset( $_GET['setting'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			wp_safe_redirect( $this->get_current_sub_tab() );
			exit;
		}

		$this->settings_form = new FormProvider( 'settings' );

		// holds app settings storage handle.
		$app_settings_provider = new SettingProvider( 'app_settings' );

		// Register app settings storing & receiving.
		$app_settings_provider->save_setting_hook( array( $this, 'save_settings' ) );
		$app_settings_provider->get_setting_hook( array( $this, 'get_settings' ) );

		// Link Provider In Form Provider.
		$this->settings_form->link_setting_provider( $app_settings_provider );

		add_action( 'admin_init', array( $this->settings_form, 'save' ) );

		/**
		 * Hooks on integration settings.
		 *
		 * @param object $this                  Class object.
		 * @param object $app_settings_provider App settings provider.
		 */
		do_action( 'bbapp_integrations_settings', $this, $app_settings_provider );

		$settings = $this->get_settings();

		// Google.
		if ( 'google' === $this->get_current_sub_tab() ) {
			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'google_services_key',
					array(
						'type'         => 'heading',
						'value'        => __( 'Google Fonts', 'buddyboss-app' ),
						'desc'         => '',
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=121830' ),
						'icon'         => 'bb-icon-bf bb-icon-brand-google-fonts',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'google.font_service_key',
					array(
						'label'      => __( 'Google Fonts API Key', 'buddyboss-app' ),
						'desc'       => __( 'Go to <a href="https://developers.google.com/fonts/docs/developer_api" target="_blank">Google Font Developer API</a> to get your Google Fonts API key, to import Google fonts for typography.', 'buddyboss-app' ),
						'type'       => 'password',
						'value'      => '',
						'validation' => function ( $value ) {
							$is_verified = is_google_api_valid( $value );

							if ( ! is_wp_error( $is_verified ) && false !== $is_verified ) {
								return $value;
							}

							return $is_verified;
						},
					)
				)
			);
		}

		// LearnDash.
		if ( 'learndash' === $this->get_current_sub_tab() ) {

			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'learndash_settings',
					array(
						'type'         => 'heading',
						'value'        => __( 'LearnDash', 'buddyboss-app' ),
						'desc'         => '',
						'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=121847' ),
						'icon'         => 'bb-icon-bf bb-icon-brand-learndash',
					)
				)
			);

			if ( bbapp_is_learndash_enabled() ) {
				$this->settings_form->add_field(
					$app_settings_provider->field(
						'learndash_course_downloading',
						array(
							'label'          => __( 'Offline Courses', 'buddyboss-app' ),
							'label_checkbox' => __( 'Enable offline courses', 'buddyboss-app' ),
							'desc'           => __( 'Allow members to download LearnDash courses, so they can be accessed without an internet connection.', 'buddyboss-app' ),
							'type'           => 'checkbox',
							'value'          => false,
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'learndash_author_visible',
						array(
							'label'          => __( 'Course Author', 'buddyboss-app' ),
							'label_checkbox' => __( 'Display Author', 'buddyboss-app' ),
							'desc'           => __( 'Display the course author on courses, lessons and topics.', 'buddyboss-app' ),
							'type'           => 'checkbox',
							'value'          => false,
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'learndash_date_visible',
						array(
							'label'          => __( 'Course Date', 'buddyboss-app' ),
							'label_checkbox' => __( 'Display Date', 'buddyboss-app' ),
							'desc'           => __( 'Display the course date on courses, lessons and topics.', 'buddyboss-app' ),
							'type'           => 'checkbox',
							'value'          => false,
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'learndash_reader_app_compatibility',
						array(
							'label'          => __( 'Reader App Compatibility', 'buddyboss-app' ),
							'label_checkbox' => __( 'Enable reader app compatibility for LearnDash courses', 'buddyboss-app' ),
							'desc'           => __( 'If enabled, members will only be able to see courses that are "Open", "Free" or they are currently enrolled in. Courses set to "Buy now", "Recurring" or "Closed" will be hidden unless the member is already enrolled. In addition, pricing labels will be removed.', 'buddyboss-app' ),
							'type'           => 'checkbox',
							'value'          => false,
						)
					)
				);

				$selected_ids = $this->get_array_value( $settings, 'learndash_bbapp_hide_in_app', array() );

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'learndash_bbapp_hide_in_app',
						array(
							'label'    => __( 'Hide Specific Courses', 'buddyboss-app' ),
							'type'     => '\BuddyBossApp\Admin\Fields\SelectorSection',
							'settings' => array(
								'desc'                  => __( 'Use this option to hide specific courses in your app, unless a member has already enrolled in the course from the website. This is useful to hide courses that are sold for higher prices than supported through in-app purchases.', 'buddyboss-app' ),
								'html_id'               => 'learndash_bbapp_hide_in_app',
								'remove_element'        => 'learndash_bbapp_hide_in_app_remove',
								'post_type'             => 'sfwd-courses',
								'options_html_callback' => function ( $posts, $position = '', $total_posts = 0 ) {
									return $this->build_options_html( $posts, $position );
								},
								'selected_ids'          => $selected_ids,
								'selector_data'         => function () {
									$left_posts_query                            = new \WP_Query();
									$this->selector_args['left']['posts']        = $left_posts_query->query( $this->selector_args['left']['args'] );
									$this->selector_args['left']['total_items']  = $left_posts_query->found_posts;
									$right_posts_query                           = new \WP_Query();
									$this->selector_args['right']['posts']       = $right_posts_query->query( $this->selector_args['right']['args'] );
									$this->selector_args['right']['total_items'] = $right_posts_query->found_posts;

									return array(
										'sections'      => array(
											'left'     => array(
												'placeholder' => __( 'Search all courses', 'buddyboss-app' ),
												'position' => 'left',
												'html_slug' => 'all_courses',
												'posts'    => $this->selector_args['left']['posts'],
												'pager'    => array(
													'current_page' => 1,
													'per_page' => $this->item_per_page,
													'total_items' => $this->selector_args['left']['total_items'],
													'total_pages' => ceil( intval( $this->selector_args['left']['total_items'] ) / intval( $this->item_per_page ) ),
												),
											),
											'controls' => array(
												'position' => 'controls',
											),
											'right'    => array(
												'placeholder' => __( 'Search hide courses', 'buddyboss-app' ),
												'position' => 'right',
												'html_slug' => 'hide_courses',
												'posts'    => $this->selector_args['right']['posts'],
												'pager'    => array(
													'current_page' => 1,
													'per_page' => $this->item_per_page,
													'total_items' => $this->selector_args['right']['total_items'],
													'total_pages' => ceil( intval( $this->selector_args['right']['total_items'] ) / intval( $this->item_per_page ) ),
												),
											),
										),
										'selector_data' => array(
											'selector_action' => 'bbapp_selector_pager',
											'query_vars' => array(
												'post_type' => 'sfwd-courses',
												'paged'   => 1,
												'posts_per_page' => $this->item_per_page,
												'order'   => 'desc',
												'orderby' => 'date',
												'search_by_title' => '',
											),
										),
									);
								},
							),
						)
					)
				);
			} else {
				$this->settings_form->add_field(
					$app_settings_provider->field(
						'learndash_html',
						array(
							'type'  => 'html',
							'value' => '<p>If using <a href="https://learndash.idevaffiliate.com/111.html" target="_blank">LearnDash LMS</a> we  display your courses in the app, with the option to download and take the courses offline.</p>',
						)
					)
				);
			}
		}

		// Vimeo.
		if ( 'vimeo' === $this->get_current_sub_tab() ) {

			$this->settings_form->add_field(
				$app_settings_provider->field(
					false,
					array(
						'type'    => 'table-open',
						'classes' => 'buddyboss-app-admin-card',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'vimeo_settings',
					array(
						'type'  => 'heading',
						'value' => __( 'Vimeo', 'buddyboss-app' ),
						'desc'  => '',
						'icon'  => 'bb-icon-bf bb-icon-brand-vimeo',
					)
				)
			);

			$this->settings_form->add_field(
				$app_settings_provider->field(
					'vimeo.auth_code',
					array(
						'label'      => __( 'Vimeo Authentication Code', 'buddyboss-app' ),
						'desc'       => __( 'Go to <a href="https://developer.vimeo.com/apps/" target="_blank">Vimeo Developer</a> to get your Authentication Code, to enable downloading of videos through the Vimeo API.', 'buddyboss-app' ),
						'type'       => 'password',
						'value'      => '',
						'validation' => function ( $value ) {
							$is_verified = is_vimeo_api_valid( $value );

							if ( ! is_wp_error( $is_verified ) && false !== $is_verified ) {
								return $value;
							}

							return $is_verified;
						},
					)
				)
			);
		}
	}


	/**
	 * Saves the settings which are not app id specific.
	 *
	 * @param array $settings Settings to save.
	 *
	 * @return bool
	 */
	public function save_global_settings( $settings ) {
		ManageApp::instance()->update_settings( $settings );

		return true;
	}

	/**
	 * Gets the settings which are not app id specific.
	 */
	public function get_settings() {
		$settings = ManageApp::instance()->get_settings();

		return $settings;
	}

	/**
	 * Saves the settings which are app id specific.
	 *
	 * @param array $settings Settings to save.
	 *
	 * @return bool
	 */
	public function save_settings( $settings ) {
		// Fetch old settings.
		$old_settings = $this->get_settings();

		ManageApp::instance()->update_app_settings( $settings );
		$this->bbapp_clear_transients_for_integration_tab( $old_settings );

		return true;
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {
		$tabs = array(
			'google'    => __( 'Google Fonts', 'buddyboss-app' ),
			'learndash' => __( 'LearnDash', 'buddyboss-app' ),
			'vimeo'     => __( 'Vimeo', 'buddyboss-app' ),
		);

		/**
		 * Menus which will be available to Subsite on BuddyBossApp Multisite Mode.
		 */
		$none_super_admin_menus = array();

		if ( ! bbapp_is_super_admin_page() ) {
			foreach ( $tabs as $k => $v ) {
				if ( ! in_array( $k, $none_super_admin_menus, true ) ) {
					unset( $tabs[ $k ] );
				}
			}
		}

		/**
		 * Filters Integrations sub settings.
		 *
		 * @param array $tabs Tabs.
		 */
		return apply_filters( 'bbapp_integrations_sub_settings', $tabs );
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {
		$default = '';

		// Capture first tab which is default.
		foreach ( $this->sub_settings() as $k => $v ) {
			$default = $k;
			break;
		}

		$setting      = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Renders the setting screen.
	 */
	public function render_screen() {
		\BuddyBossApp\Admin\SetupAdmin::instance()->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab() );

		if ( isset( $this->settings_form ) && $this->settings_form ) {
			$this->settings_form->render_form();
		}
	}

	/**
	 * Function will remove transient data from db when key change.
	 *
	 * @param array $old_settings Old settings.
	 */
	public function bbapp_clear_transients_for_integration_tab( $old_settings ) {
		$old_vimeo_auth_code         = '';
		$old_google_font_service_key = '';

		if ( isset( $old_settings['vimeo.auth_code'] ) ) {
			$old_vimeo_auth_code = $old_settings['vimeo.auth_code'];
		}

		if ( isset( $old_settings['google.font_service_key'] ) ) {
			$old_google_font_service_key = $old_settings['google.font_service_key'];
		}

		// Fetch new settings for integration.
		$new_settings                = $this->get_settings();
		$new_vimeo_auth_code         = '';
		$new_google_font_service_key = '';

		if ( isset( $new_settings['vimeo.auth_code'] ) ) {
			$new_vimeo_auth_code = $new_settings['vimeo.auth_code'];
		}

		if ( isset( $new_settings['google.font_service_key'] ) ) {
			$new_google_font_service_key = $new_settings['google.font_service_key'];
		}

		// Clear transient for the vimeo after changes authentication key and save in the Integration section.
		if ( $old_vimeo_auth_code !== $new_vimeo_auth_code ) {
			bbapp_delete_transients( 'bbapp_vimeo_vid_cache' );
		}

		// Clear transient for the google font after changes google key and save in the Integration section.
		if ( $old_google_font_service_key !== $new_google_font_service_key ) {
			bbapp_delete_transients( '_bbapp_google_typography_fonts' );
		}
	}

	/**
	 * Handler function for AJAX pager.
	 */
	public function bbapp_selector_pager_callback() {
		$reply_data = array( 'status' => false );
		$nonce_post = ( ! empty( $_POST['nonce'] ) ) ? wp_unslash( $_POST['nonce'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( wp_verify_nonce( $nonce_post, 'bbapp-selector-' . get_current_user_id() ) ) {
			if ( ( isset( $_POST['query_data'] ) ) && ( ! empty( $_POST['query_data'] ) ) ) {
				if ( ( isset( $_POST['query_data']['query_vars'] ) ) && ( ! empty( $_POST['query_data']['query_vars'] ) ) ) {
					$args = map_deep( $_POST['query_data']['query_vars'], 'sanitize_text_field' ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash

					if ( ( isset( $args['include'] ) ) && ( ! empty( $args['include'] ) ) ) {
						if ( learndash_is_valid_JSON( wp_unslash( $args['include'] ) ) ) {
							$args['include'] = (array) json_decode( wp_unslash( $args['include'] ) );
						}
					}

					if ( ( isset( $args['exclude'] ) ) && ( ! empty( $args['exclude'] ) ) ) {
						if ( learndash_is_valid_JSON( wp_unslash( $args['exclude'] ) ) ) {
							$args['exclude'] = (array) json_decode( wp_unslash( $args['exclude'] ) );
						}
					}

					if ( ( isset( $_POST['query_data']['selected_ids'] ) ) && ( ! empty( $_POST['query_data']['selected_ids'] ) ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized,WordPress.Security.ValidatedSanitizedInput.MissingUnslash
						$args['selected_ids'] = (array) json_decode( wp_unslash( $_POST['query_data']['selected_ids'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized,WordPress.Security.ValidatedSanitizedInput.MissingUnslash
					}

					// Set our reference flag so other functions know we are running pager.
					$args['is_pager'] = true;

					if ( ( isset( $_POST['query_data']['position'] ) ) && ( ! empty( $_POST['query_data']['position'] ) ) ) {
						if ( ( isset( $_POST['query_data']['query_vars']['search_by_title'] ) ) && ( ! empty( $_POST['query_data']['query_vars']['search_by_title'] ) ) ) {
							$reply_data = $this->load_search_ajax( $_POST['query_data']['query_vars'], bbapp_input_clean( wp_unslash( $_POST['query_data']['position'] ) ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized,WordPress.Security.ValidatedSanitizedInput.MissingUnslash
						} else {
							$reply_data = $this->load_pager_ajax( $_POST['query_data']['query_vars'], bbapp_input_clean( wp_unslash( $_POST['query_data']['position'] ) ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized,WordPress.Security.ValidatedSanitizedInput.MissingUnslash
						}
					}
				}
			}
		}

		if ( ! empty( $reply_data ) ) {
			echo wp_json_encode( $reply_data );
		}

		wp_die(); // this is required to terminate immediately and return a proper response.
	}

	/**
	 * Load selector section search AJAX.
	 *
	 * @param array  $query_vars Query vars.
	 * @param string $position   Value for 'left' or 'right' position.
	 *
	 * @return array
	 */
	public function load_search_ajax( $query_vars, $position = '' ) {
		$args        = $query_vars;
		$args['s']   = $args['search_by_title'];
		$posts       = array();
		$total_items = 0;

		if ( ! empty( $args ) ) {
			add_filter( 'posts_search', array( $this, 'search_filter_by_title' ), 10, 2 );
			$posts_query = new \WP_Query();
			$posts       = $posts_query->query( $args );
			$total_items = $posts_query->found_posts;
			remove_filter( 'posts_search', array( $this, 'search_filter_by_title' ), 10 );
		}
		$reply_data['pager']        = array(
			'current_page' => $args['paged'],
			'per_page'     => $args['posts_per_page'],
			'total_items'  => $total_items,
			'total_pages'  => ceil( intval( $total_items ) / intval( $args['posts_per_page'] ) ),
		);
		$reply_data['html_options'] = $this->build_options_html( $posts, $position );

		return $reply_data;
	}

	/**
	 * Load selector section search AJAX.
	 *
	 * @param array  $query_vars Query vars.
	 * @param string $position   Value for 'left' or 'right' position.
	 *
	 * @return array
	 */
	public function load_pager_ajax( $query_vars, $position = '' ) {
		$args        = $query_vars;
		$posts       = array();
		$total_items = 0;

		if ( ! empty( $args ) ) {
			add_filter( 'posts_search', array( $this, 'search_filter_by_title' ), 10, 2 );
			$posts_query = new \WP_Query();
			$posts       = $posts_query->query( $args );
			$total_items = $posts_query->found_posts;
			remove_filter( 'posts_search', array( $this, 'search_filter_by_title' ), 10 );
		}
		$reply_data['pager']        = array(
			'current_page' => $args['paged'],
			'per_page'     => $args['posts_per_page'],
			'total_items'  => $total_items,
			'total_pages'  => ceil( intval( $total_items ) / intval( $args['posts_per_page'] ) ),
		);
		$reply_data['html_options'] = $this->build_options_html( $posts, $position );

		return $reply_data;
	}

	/**
	 * Get build option html.
	 *
	 * @param array  $posts    Array of posts.
	 * @param string $position Position.
	 *
	 * @return string
	 */
	public function build_options_html( $posts, $position ) {
		$options_html = '';

		if ( ! empty( $posts ) ) {
			foreach ( $posts as $post ) {

				$disabled_class = '';
				$disabled_state = '';
				$_hide_in_app   = get_post_meta( $post->ID, '_hide_in_app', true );

				if ( 'yes' === $_hide_in_app ) {
					if ( 'left' === $position ) {
						$disabled_class = 'bbapp-selector-item-disabled';
						$disabled_state = ' disabled="disabled" ';
					}
				}

				$options_html .= '<option class="bbapp-selector-item ' . esc_attr( $disabled_class ) . '" value="' . esc_attr( $post->ID ) . '" ' . $disabled_state . ' data-value="' . esc_attr( $post->ID ) . '"> ' . esc_attr( $post->post_title ) . ' </option>';
			}
		}

		return $options_html;
	}

	/**
	 * Search filter by Title.
	 *
	 * @param string    $search   Search pattern.
	 * @param \WP_Query $wp_query WP_Query object.
	 *
	 * @return string
	 */
	public function search_filter_by_title( $search, \WP_Query $wp_query ) {
		if ( ! empty( $search ) && ! empty( $wp_query->query_vars['search_by_title'] ) ) {
			global $wpdb;

			$q      = $wp_query->query_vars;
			$n      = ! empty( $q['exact'] ) ? '' : '%';
			$search = array();

			foreach ( (array) $q['search_by_title'] as $term ) {
				$search[] = $wpdb->prepare( "$wpdb->posts.post_title LIKE %s", $n . $term . $n );
			}

			if ( ! is_user_logged_in() ) {
				$search[] = "$wpdb->posts.post_password = ''";
			}

			$search = ' AND ' . implode( ' AND ', $search );
		}

		return $search;
	}

	/**
	 * Function to get value based on key.
	 *
	 * @param array      $array   Array to get value.
	 * @param int|string $key     Key to get value.
	 * @param string     $default Default value to return.
	 *
	 * @return string
	 */
	public function get_array_value( $array, $key, $default = '' ) {
		if ( isset( $array[ $key ] ) ) {
			return $array[ $key ];
		} else {
			return $default;
		}
	}
}
