<?php
/**
 * Holds functionality to update IAP orders.
 *
 * @package BuddyBossApp\Admin\InAppPurchases
 */

namespace BuddyBossApp\Admin\InAppPurchases;

/**
 * Upgrade helper class.
 */
class UpgradeHelper {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * UpgradeHelper constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return UpgradeHelper
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Magic methods for old methods.
	 *
	 * @param string             $name      Method name.
	 * @param string|mixed|array $arguments Method arguments.
	 *
	 * @since 1.7.5
	 *
	 * @return mixed|void
	 */
	public static function __callStatic( $name, $arguments ) {
		// Compatibility Support for CamelCase.
		$compatibility_func = array(
			'upgradeProductsToNewSchema' => 'upgrade_products_to_new_schema',
			'insertOrUpdateProduct'      => 'insert_or_update_product',
			'upgradeOrdersToNewSchema'   => 'upgrade_orders_to_new_schema',
			'insertOrUpdateOrder'        => 'insert_or_update_order',
		);

		foreach ( $compatibility_func as $old => $new ) {
			if ( $name === $old ) {
				_deprecated_function( esc_html( $old ), esc_html( bbapp()->plugin_version ), esc_html( $new ) );

				return call_user_func_array( array( self::instance(), $new ), $arguments );
			}
		}
	}

	/**
	 * Load additional hooks such as : admin_init
	 * NOTE : Method is similar to concept of init
	 *
	 * @return void {void}
	 */
	public function load() {
		// All actions/filters below.
	}

	/**
	 * Get old IAP-products data, to be refactored
	 *
	 * @return void
	 */
	public static function upgrade_products_to_new_schema() {
		global $wpdb;

		$global_prefix = \bbapp_iap()->get_global_dbprefix();
		$rows          = $wpdb->get_results( "SELECT * FROM {$global_prefix}bbapp_iap_products_OLD" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( IAP_LOG ) {
			$total_rows = count( $rows );

			echo 'Total rows need to be imported : ' . esc_html( $total_rows );

			foreach ( $rows as $row ) {
				echo sprintf( '<br/>$1%s$2%s$3%s$4%s', esc_html__( 'Attempt for ID:', 'buddyboss-app' ), esc_html( $row->id ), esc_html__( '    Status:', 'buddyboss-app' ), esc_html( self::insert_or_update_product( $row ) ) );
			}
		}
	}

	/**
	 * Insert/Update old IAP-products data into new IAP-products-table
	 *
	 * @param object $row Order row.
	 *
	 * @return null|bool|int|\mysqli_result|resource
	 */
	public static function insert_or_update_product( $row ) {
		global $wpdb;

		$table_name        = \bbapp_iap()->get_global_dbprefix() . 'bbapp_iap_products';
		$item_ids          = null;
		$device_platform   = null;
		$integration_types = null;
		$date_created      = null;
		$date_updated      = null;

		if ( isset( $row->item_id ) ) {
			$item_ids = maybe_serialize( $row->item_id );
		}

		if ( isset( $row->iap_type ) ) {
			// NOTE : Old DB column has single value(and NOT serialized). Eg : apple or google.
			$iap_type = $row->iap_type;

			if ( 'apple' === $iap_type ) {
				$device_platform = 'ios';
			} elseif ( 'google' === $iap_type ) {
				$device_platform = 'android';
			}
		}

		if ( isset( $row->item_type ) ) {
			$integration_types = maybe_serialize( $row->item_type );
		}

		if ( isset( $row->date_created ) ) {
			$date_created = $row->date_created;
			// NOTE : Keeping date_updated same as date_created since old IAP didn't had date_updated column.
			$date_updated = $date_created;
		}

		// NOTE : we don't use this column at-all currently.
		$secondary_id = 0;

		// TODO : Below need to be calculated since old IAP didn't had this fields.
		$buddyboss_app_product_id = - 11;
		$exists                   = $wpdb->get_results( $wpdb->prepare( "SELECT count(*) as total FROM {$table_name} WHERE id=%d", $row->id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( isset( $exists[0]->total ) && 0 === (int) $exists[0]->total ) {

			$args_data = array(
				'id'                => $row->id,
				'blog_id'           => $row->blog_id,
				'user_id'           => $row->user_id,
				'bbapp_product_id'  => $buddyboss_app_product_id,
				'device_platform'   => $device_platform,
				'order_status'      => $row->order_status,
				'store_product_id'  => $row->storeProductId, // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				'integration_types' => $integration_types,
				'item_ids'          => $item_ids,
				'is_recurring'      => $row->is_recurring,
				'secondary_id'      => $secondary_id,
				'date_created'      => $date_created,
				'date_updated'      => $date_updated,
				'expire_at'         => $row->expire_at,
				'user_email'        => $row->user_email,
			);

			$status = $wpdb->insert( $table_name, $args_data ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery

		} else {
			$args_data = array(
				'blog_id'           => $row->blog_id,
				'user_id'           => $row->user_id,
				'bbapp_product_id'  => $row->order_status,
				'device_platform'   => $device_platform,
				'order_status'      => $row->order_status,
				'store_product_id'  => $row->order_status,
				'integration_types' => $integration_types,
				'item_ids'          => $item_ids,
				'is_recurring'      => $row->is_recurring,
				'secondary_id'      => $secondary_id,
				'date_created'      => $date_created,
				'date_updated'      => $date_updated,
				'expire_at'         => $row->expire_at,
				'user_email'        => $row->user_email,
			);
			$status    = $wpdb->update( $table_name, $args_data, array( 'id' => $row->id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		}

		return $status;
	}

	/**
	 * Get old data, to be refactored
	 *
	 * @return void
	 */
	public static function upgrade_orders_to_new_schema() {
		global $wpdb;

		$rows       = $wpdb->get_results( "SELECT * FROM {$wpdb->prefix}appb_iap_orders_OLD" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$total_rows = count( $rows );

		echo 'Total rows need to be imported : ' . esc_html( $total_rows );

		foreach ( $rows as $row ) {
			echo sprintf( '<br/>$1%s$2%s$3%s$4%s', esc_html__( 'Attempt for ID:', 'buddyboss-app' ), esc_html( $row->id ), esc_html__( '    Status:', 'buddyboss-app' ), esc_html( self::insert_or_update_order( $row ) ) );
		}
	}

	/**
	 * Insert/Update into NEW
	 *
	 * @param object $row order row.
	 *
	 * @return null|bool|int|\mysqli_result|resource
	 */
	public static function insert_or_update_order( $row ) {
		global $wpdb;

		$item_ids          = null;
		$device_platform   = null;
		$integration_types = null;

		if ( isset( $row->item_id ) ) {
			$item_ids = maybe_serialize( $row->item_id );
		}

		if ( isset( $row->iap_type ) ) {
			// NOTE : Old DB column has single value(and un-serialized). Eg : apple or google.
			$iap_type = $row->iap_type;

			if ( 'apple' === $iap_type ) {
				$device_platform = 'ios';
			} elseif ( 'google' === $iap_type ) {
				$device_platform = 'android';
			}
		}

		if ( isset( $row->item_type ) ) {
			$integration_types = maybe_serialize( $row->item_type );
		}

		$date_created = null;
		$date_updated = null;

		if ( isset( $row->date_created ) ) {
			$date_created = $row->date_created;
			// NOTE : Keeping date_updated same as date_created since old IAP didn't had date_updated column.
			$date_updated = $date_created;
		}

		// NOTE : we don't use this column at-all currently.
		$secondary_id = 0;

		// TODO : Below need to be calculated since old IAP didn't had this fields.
		$buddyboss_app_product_id = - 11;
		$exists                   = $wpdb->get_results( $wpdb->prepare( "SELECT count(*) as total FROM {$wpdb->prefix}bbapp_iap_orders WHERE id=%d", $row->id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( isset( $exists[0]->total ) && (int) 0 === $exists[0]->total ) {

			$args_data = array(
				'id'                => $row->id,
				'blog_id'           => $row->blog_id,
				'user_id'           => $row->user_id,
				'bbapp_product_id'  => $buddyboss_app_product_id,
				'device_platform'   => $device_platform,
				'order_status'      => $row->order_status,
				'store_product_id'  => $row->storeProductId, // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				'integration_types' => $integration_types,
				'item_ids'          => $item_ids,
				'is_recurring'      => $row->is_recurring,
				'secondary_id'      => $secondary_id,
				'date_created'      => $date_created,
				'date_updated'      => $date_updated,
				'expire_at'         => $row->expire_at,
				'user_email'        => $row->user_email,
			);

			$status = $wpdb->insert( "{$wpdb->prefix}bbapp_iap_orders", $args_data ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery

		} else {
			$args_data = array(
				'blog_id'           => $row->blog_id,
				'user_id'           => $row->user_id,
				'bbapp_product_id'  => $row->order_status,
				'device_platform'   => $device_platform,
				'order_status'      => $row->order_status,
				'store_product_id'  => $row->order_status,
				'integration_types' => $integration_types,
				'item_ids'          => $item_ids,
				'is_recurring'      => $row->is_recurring,
				'secondary_id'      => $secondary_id,
				'date_created'      => $date_created,
				'date_updated'      => $date_updated,
				'expire_at'         => $row->expire_at,
				'user_email'        => $row->user_email,
			);
			$status    = $wpdb->update( "{$wpdb->prefix}bbapp_iap_orders", $args_data, array( 'id' => $row->id ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		}

		return $status;
	}
}
