<?php
/**
 * Holds functionality subscription group listing.
 *
 * @package BuddyBossApp\Admin\InAppPurchases
 */

namespace BuddyBossApp\Admin\InAppPurchases;

use BuddyBossApp\InAppPurchases\SubscriptionGroupTaxonomy;

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Subscription group list class.
 */
class SubscriptionGroupsList extends \WP_List_Table {
	/**
	 * The type of view currently being displayed.
	 *
	 * E.g. "All"...
	 *
	 * @since BuddyPress 1.7.0
	 * @var string
	 */
	public $view = 'all';

	/**
	 * Taxonomy name.
	 *
	 * @var string $taxonomy
	 */
	private $taxonomy;

	/**
	 * Function callback arguments.
	 *
	 * @var $callback_args
	 */
	public $callback_args;

	/**
	 * Class constructor
	 *
	 * @param array $args Constructor arguments.
	 */
	public function __construct( $args = array() ) {
		global $post_type, $taxonomy, $action, $tax;

		parent::__construct(
			array(
				'plural'   => 'tags',
				'singular' => 'tag',
				'screen'   => isset( $args['screen'] ) ? $args['screen'] : 'app-subscription-group',
			)
		);

		$this->taxonomy = SubscriptionGroupTaxonomy::$taxonomy;
		$post_type      = 'post'; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		$taxonomy       = $this->taxonomy; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited

		if ( empty( $taxonomy ) ) {
			$taxonomy = 'post_tag'; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		}

		if ( ! taxonomy_exists( $taxonomy ) ) {
			wp_die( esc_html__( 'Invalid taxonomy.', 'buddyboss-app' ) );
		}

		$tax = get_taxonomy( $taxonomy ); // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited

		// @todo Still needed? Maybe just the show_ui part.
		if ( empty( $post_type ) || ! in_array( $post_type, get_post_types( array( 'show_ui' => true ) ), true ) ) {
			$post_type = 'post'; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		}

	}

	/**
	 * Check whether current user have permissions.
	 *
	 * @return bool
	 */
	public function ajax_user_can() {
		return current_user_can( get_taxonomy( $this->taxonomy )->cap->manage_terms );
	}

	/**
	 * Prepare items.
	 */
	public function prepare_items() {
		$tags_per_page         = $this->get_items_per_page( 'edit_' . $this->taxonomy . '_per_page' );
		$this->_column_headers = array(
			$this->get_columns(),
			array(),
			$this->get_sortable_columns(),
		); // Init column headers.
		$search                = ! empty( $_REQUEST['s'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['s'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$orderby_request       = ! empty( $_REQUEST['orderby'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['orderby'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$order_request         = ! empty( $_REQUEST['order'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['order'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		$args = array(
			'search' => $search,
			'page'   => $this->get_pagenum(),
			'number' => $tags_per_page,
		);

		if ( ! empty( $orderby_request ) ) {
			$args['orderby'] = $orderby_request;
		}

		if ( ! empty( $order_request ) ) {
			$args['order'] = $order_request;
		}

		$this->callback_args = $args;

		$this->set_pagination_args(
			array(
				'total_items' => wp_count_terms(
					array(
						'taxonomy' => $this->taxonomy,
						'search'   => $search,
					)
				),
				'per_page'    => $tags_per_page,
			)
		);
	}

	/**
	 * Check if has items.
	 *
	 * @return bool
	 */
	public function has_items() {
		// @todo Populate $this->items in prepare_items().
		return true;
	}

	/**
	 * No items found method.
	 */
	public function no_items() {
		echo esc_html( get_taxonomy( $this->taxonomy )->labels->not_found );
	}

	/**
	 * Get bulk actions.
	 *
	 * @return array
	 */
	protected function get_bulk_actions() {
		$actions = array();

		if ( current_user_can( get_taxonomy( $this->taxonomy )->cap->delete_terms ) ) {
			$actions['delete'] = __( 'Delete', 'buddyboss-app' );
		}

		return $actions;
	}

	/**
	 * Get current bulk action.
	 *
	 * @return string
	 */
	public function current_action() {
		$action_request  = ( ! empty( $_REQUEST['action'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['action'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$action2_request = ( ! empty( $_REQUEST['action2'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['action2'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( isset( $_REQUEST['delete_tags'] ) && ( 'delete' === $action_request || 'delete' === $action2_request ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			return 'bulk-delete';
		}

		return parent::current_action();
	}

	/**
	 * Get list columns.
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns = array(
			'cb'   => '<input type="checkbox" />',
			'name' => _x( 'Name', 'term name', 'buddyboss-app' ),
			'slug' => __( 'Slug', 'buddyboss-app' ),
		);

		return $columns;
	}

	/**
	 * Function to get sortable columns.
	 *
	 * @return array
	 */
	protected function get_sortable_columns() {
		return array(
			'name' => 'name',
			'slug' => 'slug',
		);
	}

	/**
	 * Placeholder disaplay method.
	 */
	public function display_rows_or_placeholder() {
		$taxonomy = $this->taxonomy;

		$args = wp_parse_args(
			$this->callback_args,
			array(
				'taxonomy'   => $taxonomy,
				'page'       => 1,
				'number'     => 20,
				'search'     => '',
				'hide_empty' => 0,
			)
		);

		$page           = $args['page'];
		$number         = $args['number']; // Set variable because $args['number'] can be subsequently overridden.
		$offset         = ( $page - 1 ) * $number;
		$args['offset'] = $offset;
		$count          = 0; // Convert it to table rows.

		if ( is_taxonomy_hierarchical( $taxonomy ) && ! isset( $args['orderby'] ) ) {
			// We'll need the full set of terms then.
			$args['number'] = 0;
			$args['offset'] = $args['number'];
		}

		$terms = get_terms( $args );

		if ( empty( $terms ) || ! is_array( $terms ) ) {
			echo '<tr class="no-items"><td class="colspanchange" colspan="' . esc_attr( $this->get_column_count() ) . '">';
			$this->no_items();
			echo '</td></tr>';

			return;
		}

		if ( is_taxonomy_hierarchical( $this->taxonomy ) && ! isset( $args['orderby'] ) ) {
			if ( ! empty( $args['search'] ) ) {// Ignore children on searches.
				$children = array();
			} else {
				$children = _get_term_hierarchy( $this->taxonomy );
			}

			/*
			 * Some funky recursion to get the job done (paging & parents mainly) is contained within.
			 * Skip it for non-hierarchical taxonomies for performance sake.
			 */
			$this->rows( $this->taxonomy, $terms, $children, $offset, $number, $count );
		} else {
			foreach ( $terms as $term ) {
				$this->single_row( $term );
			}
		}
	}

	/**
	 * Internal function for row.
	 *
	 * @param string       $taxonomy Taxonomy name.
	 * @param object       $terms    Terms.
	 * @param object|array $children Term children objects.
	 * @param int          $start    Starting point.
	 * @param int          $per_page Number of items per page.
	 * @param int          $count    Item count.
	 * @param int          $parent   Parent id.
	 * @param int          $level    Level.
	 */
	private function rows( $taxonomy, $terms, &$children, $start, $per_page, &$count, $parent = 0, $level = 0 ) {
		$end            = $start + $per_page;
		$search_request = ( ! empty( $_REQUEST['s'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['s'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		foreach ( $terms as $key => $term ) {
			if ( $count >= $end ) {
				break;
			}

			if ( $term->parent !== $parent && empty( $search_request ) ) {
				continue;
			}

			// If the page starts in a subtree, print the parents.
			if ( $count === $start && $term->parent > 0 && empty( $search_request ) ) {
				$my_parents = array();
				$parent_ids = array();
				$p          = $term->parent;
				while ( $p ) {
					$my_parent    = get_term( $p, $taxonomy );
					$my_parents[] = $my_parent;
					$p            = $my_parent->parent;
					if ( in_array( $p, $parent_ids, true ) ) { // Prevent parent loops.
						break;
					}
					$parent_ids[] = $p;
				}
				unset( $parent_ids );

				$num_parents = count( $my_parents );
				$my_parent   = array_pop( $my_parents );
				while ( $my_parent ) {
					echo "\t";
					$this->single_row( $my_parent, $level - $num_parents );
					$num_parents --;
				}
			}

			if ( $count >= $start ) {
				echo "\t";
				$this->single_row( $term, $level );
			}

			++ $count;

			unset( $terms[ $key ] );

			if ( isset( $children[ $term->term_id ] ) && empty( $search_request ) ) {
				$this->rows( $taxonomy, $terms, $children, $start, $per_page, $count, $term->term_id, $level + 1 );
			}
		}
	}

	/**
	 * Single row.
	 *
	 * @param \WP_Term $tag   Term object.
	 * @param int      $level Level.
	 *
	 * @global string  $taxonomy
	 */
	public function single_row( $tag, $level = 0 ) {
		$taxonomy = $this->taxonomy;
		$tag      = sanitize_term( $tag, $taxonomy );

		$this->level = $level;

		if ( $tag->parent ) {
			$count = count( get_ancestors( $tag->term_id, $taxonomy, 'taxonomy' ) );
			$level = 'level-' . $count;
		} else {
			$level = 'level-0';
		}

		echo sprintf( '<tr id="tag-%s" class="%s">', esc_html( $tag->term_id ), esc_html( $level ) );
		$this->single_row_columns( $tag );
		echo '</tr>';
	}

	/**
	 * Checkbox columns.
	 *
	 * @param \WP_Term $tag Term object.
	 *
	 * @return string
	 */
	public function column_cb( $tag ) {
		if ( current_user_can( 'delete_term', $tag->term_id ) ) {
			/* translators: %s: Taxonomy term name. */
			return sprintf( '<label class="screen-reader-text" for="cb-select-%1$s">%2$s</label><input type="checkbox" name="delete_tags[]" value="%1$s" id="cb-select-%1$s" />', $tag->term_id, sprintf( __( 'Select %s', 'buddyboss-app' ), $tag->name ) );
		}

		return '&nbsp;';
	}

	/**
	 * Columns name.
	 *
	 * @param \WP_Term $tag Term object.
	 *
	 * @return string
	 */
	public function column_name( $tag ) {
		$taxonomy = $this->taxonomy;
		$pad      = str_repeat( '&#8212; ', max( 0, $this->level ) );

		/**
		 * Filters display of the term name in the terms list table.
		 *
		 * The default output may include padding due to the term's
		 * current level in the term hierarchy.
		 *
		 * @param string   $pad_tag_name The term name, padded if not top-level.
		 * @param \WP_Term $tag          Term object.
		 *
		 * @since 2.5.0
		 *
		 * @see   WP_Terms_List_Table::column_name()
		 */
		$name               = apply_filters( 'term_name', $pad . ' ' . $tag->name, $tag );
		$qe_data            = get_term( $tag->term_id, $taxonomy, OBJECT, 'edit' );
		$server_request_uri = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$uri                = wp_doing_ajax() ? wp_get_referer() : $server_request_uri;
		$page_request       = ( ! empty( $_REQUEST['page'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$args               = array(
			'tag_ID'  => $tag->term_id,
			'page'    => $page_request,
			'action'  => 'edit',
			'setting' => $taxonomy,
		);
		$edit_link          = add_query_arg( $args, bbapp_get_admin_url( 'admin.php' ) );

		if ( $edit_link ) {
			$edit_link = add_query_arg( 'wp_http_referer', rawurlencode( wp_unslash( $uri ) ), $edit_link );
			/* translators: %s: Taxonomy term name. */
			$name = sprintf( '<a class="row-title" href="%s" aria-label="%s">%s</a>', esc_url( $edit_link ), esc_attr( sprintf( __( '&#8220;%s&#8221; (Edit)', 'buddyboss-app' ), $tag->name ) ), $name );
		}

		$out = sprintf(
			'<strong>%s</strong><br />',
			$name
		);

		$out .= '<div class="hidden" id="inline_' . $qe_data->term_id . '">';
		$out .= '<div class="name">' . $qe_data->name . '</div>';

		// This filter is documented in wp-admin/edit-tag-form.php.
		$out .= '<div class="slug">' . apply_filters( 'editable_slug', $qe_data->slug, $qe_data ) . '</div>';
		$out .= '<div class="parent">' . $qe_data->parent . '</div></div>';

		return $out;
	}

	/**
	 * Gets the name of the default primary column.
	 *
	 * @since 4.3.0
	 * @return string Name of the default primary column, in this case, 'name'.
	 */
	protected function get_default_primary_column_name() {
		return 'name';
	}

	/**
	 * Generates and displays row action links.
	 *
	 * @param \WP_Term $tag         Tag being acted upon.
	 * @param string   $column_name Current column name.
	 * @param string   $primary     Primary column name.
	 *
	 * @since 4.3.0
	 * @return string Row actions output for terms, or an empty string
	 *                if the current column is not the primary column.
	 */
	protected function handle_row_actions( $tag, $column_name, $primary ) {
		if ( $primary !== $column_name ) {
			return '';
		}

		$actions      = array();
		$page_request = ( ! empty( $_REQUEST['page'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( current_user_can( 'edit_term', $tag->term_id ) ) {
			$args     = array(
				'tag_ID'  => $tag->term_id,
				'page'    => ! empty( $page_request ) ? $page_request : 'bbapp-iap',
				'action'  => 'edit',
				'setting' => $this->taxonomy,
			);
			$location = add_query_arg( $args, bbapp_get_admin_url( 'admin.php' ) );
			/* translators: %s: Taxonomy term name. */
			$actions['edit'] = sprintf( '<a href="%s" aria-label="%s">%s</a>', esc_url( $location ), esc_attr( sprintf( __( 'Edit &#8220;%s&#8221;', 'buddyboss-app' ), $tag->name ) ), esc_html__( 'Edit', 'buddyboss-app' ) );
		}

		if ( current_user_can( 'delete_term', $tag->term_id ) ) {
			$args     = array(
				'tag_ID'   => $tag->term_id,
				'page'     => $page_request,
				'action'   => 'delete',
				'setting'  => $this->taxonomy,
				'taxonomy' => $this->taxonomy,
			);
			$location = add_query_arg( $args, bbapp_get_admin_url( 'admin.php' ) );
			/* translators: %s: Taxonomy term name. */
			$actions['delete'] = sprintf( '<a href="%s" class="delete-tag aria-button-if-js" aria-label="%s">%s</a>', wp_nonce_url( $location, 'delete-tag_' . $tag->term_id ), esc_attr( sprintf( __( 'Delete &#8220;%s&#8221;', 'buddyboss-app' ), $tag->name ) ), esc_html__( 'Delete', 'buddyboss-app' ) );
		}

		return $this->row_actions( $actions );
	}

	/**
	 * Columns slug.
	 *
	 * @param \WP_Term $tag Term object.
	 *
	 * @return string
	 */
	public function column_slug( $tag ) {
		/** This filter is documented in wp-admin/edit-tag-form.php */
		return apply_filters( 'editable_slug', $tag->slug, $tag );
	}


	/**
	 * Column default data.
	 *
	 * @param \WP_Term $tag         Term object.
	 * @param string   $column_name Name of the column.
	 *
	 * @return string
	 */
	public function column_default( $tag, $column_name ) {
		/**
		 * Filters the displayed columns in the terms list table.
		 *
		 * The dynamic portion of the hook name, `$this->taxonomy`,
		 * refers to the slug of the current taxonomy.
		 *
		 * @param string $string      Blank string.
		 * @param string $column_name Name of the column.
		 * @param int    $term_id     Term ID.
		 *
		 * @since 2.8.0
		 */
		return apply_filters( "manage_{$this->taxonomy}_custom_column", '', $column_name, $tag->term_id );
	}
}
