<?php
/**
 * Holds subscription group related helping functions.
 *
 * @package BuddyBossApp\Admin\InAppPurchases
 */

namespace BuddyBossApp\Admin\InAppPurchases;

use BuddyBossApp\InAppPurchases\SubscriptionGroupTaxonomy;

/**
 * Subscription group helper class.
 */
class SubscriptionGroupsHelpers {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * Subscription group taxonomy.
	 *
	 * @var $taxonomy
	 */
	private $taxonomy;

	/**
	 * Subscription group list.
	 *
	 * @var SubscriptionGroupsList $subscription_groups_list
	 */
	private $subscription_groups_list;


	/**
	 * SubscriptionGroupsHelpers constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return object
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Show the errors & updates
	 */
	public function show_messages() {
		/** Also used by the Edit Tag  form */
		require_once ABSPATH . 'wp-admin/includes/edit-tag-messages.php';

		$class = ( isset( $_REQUEST['error'] ) ) ? 'error' : 'updated'; //phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( $message ) : ?>
			<div id="message" class="<?php echo esc_attr( $class ); ?> notice is-dismissible"><p><?php echo esc_html( $message ); ?></p></div>
			<?php
			$referer                = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$_SERVER['REQUEST_URI'] = remove_query_arg( array( 'message', 'error' ), $referer );
		endif;
	}

	/**
	 * Initialize actions/filters
	 *
	 * @return void
	 */
	public function init() {
		$this->taxonomy = SubscriptionGroupTaxonomy::$taxonomy;
		// NOTE : can't use admin_post hook since it don't exists on multi-site.
		add_action( 'admin_init', array( $this, 'action_subscription_group' ) );
		add_action( 'wp_ajax_create-term', array( $this, 'create_subscription_group' ) );
	}

	/**
	 * Ajax create subscription group.
	 */
	public function create_subscription_group() {
		check_ajax_referer( 'create-term', '_wpnonce_create-term' );

		$taxonomy      = ! empty( $_POST['taxonomy'] ) ? bbapp_input_clean( wp_unslash( $_POST['taxonomy'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$tag_name_post = ! empty( $_POST['tag-name'] ) ? bbapp_input_clean( wp_unslash( $_POST['tag-name'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$tax           = get_taxonomy( $taxonomy );

		if ( ! current_user_can( $tax->cap->edit_terms ) ) {
			wp_die( - 1 );
		}

		$x = new \WP_Ajax_Response();

		$tag = wp_insert_term( $tag_name_post, $taxonomy, $_POST );

		if ( $tag && ! is_wp_error( $tag ) ) {
			$tag = get_term( $tag['term_id'], $taxonomy );
		}

		if ( ! $tag || is_wp_error( $tag ) ) {
			$message = esc_html__( 'An error has occurred. Please reload the page and try again.', 'buddyboss-app' );

			if ( is_wp_error( $tag ) && $tag->get_error_message() ) {
				$message = $tag->get_error_message();
			}

			$x->add(
				array(
					'what' => 'taxonomy',
					'data' => new \WP_Error( 'error', $message ),
				)
			);
			$x->send();
		}

		$this->subscription_groups_list = new SubscriptionGroupsList();

		$level     = 0;
		$noparents = '';

		if ( is_taxonomy_hierarchical( $taxonomy ) ) {
			$level = count( get_ancestors( $tag->term_id, $taxonomy, 'taxonomy' ) );
			ob_start();
			$this->subscription_groups_list->single_row( $tag, $level );
			$noparents = ob_get_clean();
		}

		ob_start();
		$this->subscription_groups_list->single_row( $tag );
		$parents = ob_get_clean();

		$x->add(
			array(
				'what'         => 'taxonomy',
				'supplemental' => compact( 'parents', 'noparents' ),
			)
		);

		$x->add(
			array(
				'what'         => 'term',
				'position'     => $level,
				'supplemental' => (array) $tag,
			)
		);

		$x->send();
	}

	/**
	 * Trigger all actions.
	 */
	public function action_subscription_group() {
		if ( ( isset( $_GET['page'] ) && 'bbapp-iap' === $_GET['page'] ) && isset( $_GET['setting'] ) && $this->taxonomy === $_GET['setting'] ) {
			$taxonomy                       = $this->taxonomy;
			$tax                            = get_taxonomy( $taxonomy );
			$this->subscription_groups_list = new SubscriptionGroupsList();
			$pagenum                        = $this->subscription_groups_list->get_pagenum();
			$location                       = false;
			$referer                        = wp_get_referer();

			if ( ! $referer ) { // For POST requests.
				$referer = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			}

			$referer = remove_query_arg(
				array(
					'_wp_http_referer',
					'_wpnonce',
					'error',
					'message',
					'paged',
				),
				$referer
			);

			switch ( $this->subscription_groups_list->current_action() ) {
				case 'create-term':
					check_admin_referer( 'create-term', '_wpnonce_create-term' );

					if ( ! current_user_can( $tax->cap->edit_terms ) ) {
						wp_die( sprintf( '<h1>%1$s</h1><p>%2$s</p>', esc_html__( 'You need a higher level of permission.', 'buddyboss-app' ), esc_html__( 'Sorry, you are not allowed to create terms in this taxonomy.', 'buddyboss-app' ) ), 403 );
					}

					$tag_name_post = ( ! empty( $_POST['tag-name'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['tag-name'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
					$ret           = wp_insert_term( $tag_name_post, $taxonomy, $_POST );

					if ( $ret && ! is_wp_error( $ret ) ) {
						$location = add_query_arg( 'message', 1, $referer );
					} else {
						$location = add_query_arg(
							array(
								'error'   => true,
								'message' => 4,
							),
							$referer
						);
					}
					break;
				case 'delete':
					if ( ! isset( $_REQUEST['tag_ID'] ) ) {
						break;
					}

					$tag_id = (int) $_REQUEST['tag_ID'];

					check_admin_referer( 'delete-tag_' . $tag_id );

					if ( ! current_user_can( 'delete_term', $tag_id ) ) {
						wp_die( sprintf( '<h1>%1$s</h1><p>%3$s</p>', esc_html__( 'You need a higher level of permission.', 'buddyboss-app' ), esc_html__( 'Sorry, you are not allowed to delete this item.', 'buddyboss-app' ) ), 403 );
					}

					wp_delete_term( $tag_id, $taxonomy );

					$location = add_query_arg( 'message', 2, $referer );

					// When deleting a term, prevent the action from redirecting back to a term that no longer exists.
					$location = remove_query_arg( array( 'tag_ID', 'action' ), $location );

					break;

				case 'bulk-delete':
					if ( ! $this->subscription_groups_list->current_action() || ! isset( $_REQUEST['delete_tags'] ) ) {
						break;
					}
					check_admin_referer( 'bulk-tags' );

					if ( ! current_user_can( $tax->cap->delete_terms ) ) {
						wp_die( sprintf( '<h1>%1$s</h1><p>%2$s</p>', esc_html__( 'You need a higher level of permission.', 'buddyboss-app' ), esc_html__( 'Sorry, you are not allowed to delete these items.', 'buddyboss-app' ) ), 403 );
					}

					$tags = (array) map_deep( wp_unslash( $_REQUEST['delete_tags'] ), 'sanitize_text_field' );
					foreach ( $tags as $tag_id ) {
						wp_delete_term( $tag_id, $taxonomy );
					}

					$location = add_query_arg( 'message', 6, $referer );
					break;
				case 'editedtag':
					$tag_id = ( ! empty( $_POST['tag_ID'] ) ) ? bbapp_input_clean( wp_unslash( (int) $_POST['tag_ID'] ) ) : 0;

					check_admin_referer( 'update-tag_' . $tag_id );

					if ( ! current_user_can( 'edit_term', $tag_id ) ) {
						wp_die( sprintf( '<h1>%1$s</h1><p>%2$s</p>', esc_html__( 'You need a higher level of permission.', 'buddyboss-app' ), esc_html__( 'Sorry, you are not allowed to edit this item.', 'buddyboss-app' ) ), 403 );
					}

					$tag = get_term( $tag_id, $taxonomy );

					if ( ! $tag ) {
						wp_die( esc_html__( 'You attempted to edit an item that doesn&#8217;t exist. Perhaps it was deleted?', 'buddyboss-app' ) );
					}

					$ret = wp_update_term( $tag_id, $taxonomy, $_POST );

					if ( $ret && ! is_wp_error( $ret ) ) {
						$original_referer = wp_get_original_referer();

						if ( false !== $original_referer ) {
							$referer = $original_referer;
						}

						$referer  = remove_query_arg( array( 'tag_ID', 'action' ), $referer );
						$location = add_query_arg( 'message', 3, $referer );
					} else {
						$location = add_query_arg(
							array(
								'error'   => true,
								'message' => 5,
							),
							$referer
						);
					}
					break;
				default:
					if ( ! $this->subscription_groups_list->current_action() || ! isset( $_REQUEST['delete_tags'] ) ) {
						break;
					}

					check_admin_referer( 'bulk-tags' );

					$screen = get_current_screen()->id;
					$tags   = (array) map_deep( wp_unslash( $_REQUEST['delete_tags'] ), 'sanitize_text_field' );

					/** This action is documented in wp-admin/edit.php */
					$location = apply_filters( "handle_bulk_actions-{$screen}", $location, $this->subscription_groups_list->current_action(), $tags ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
					break;
			}

			if ( ! $location && ! empty( $_REQUEST['_wp_http_referer'] ) ) {
				$request_uri = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

				$location = remove_query_arg(
					array(
						'_wp_http_referer',
						'_wpnonce',
					),
					$request_uri
				);
			}

			if ( $location ) {
				if ( $pagenum > 1 ) {
					$location = add_query_arg( 'paged', $pagenum, $location ); // $pagenum takes care of $total_pages.
				}

				/**
				 * Filters the taxonomy redirect destination URL.
				 *
				 * @param string $location The destination URL.
				 *
				 * @since 4.6.0
				 */
				wp_safe_redirect( $location );
				exit;
			}
		}
	}

	/**
	 * Renders the branding screen
	 *
	 * @return void
	 */
	public function render() {
		$this->show_messages();
		$action = isset( $_REQUEST['action'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['action'] ) ) : 'lists'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		wp_enqueue_script( 'admin-tags' );

		switch ( $action ) {
			case 'edit':
				include bbapp()->plugin_dir . '/views/iap/admin/subscription-groups/edit.php';
				break;
			default:
				$this->subscription_groups_list = new SubscriptionGroupsList();
				$this->subscription_groups_list->prepare_items();
				include bbapp()->plugin_dir . '/views/iap/admin/subscription-groups/lists.php';
				break;
		}
	}
}
