<?php
/**
 * Holds functionality to show the product list.
 *
 * @package BuddyBossApp\Admin\InAppPurchases
 */

namespace BuddyBossApp\Admin\InAppPurchases;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\InAppPurchases\SubscriptionGroupTaxonomy;

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Product list table view.
 */
class ProductList extends \WP_List_Table {
	/**
	 * The type of view currently being displayed.
	 *
	 * E.g. "All", "Pending", "Approved", "Spam"...
	 *
	 * @since BuddyPress 1.7.0
	 * @var string $view
	 */
	public $view = 'all';

	/**
	 * Group counts for each group type.
	 *
	 * @since BuddyPress 1.7.0
	 * @var array $iap_product_counts
	 */
	public $iap_product_counts = array();

	/**
	 * Class constructor
	 */
	public function __construct() {
		parent::__construct(
			array(
				'screen'   => 'products',
				'singular' => __( 'IAP Product', 'buddyboss-app' ), // singular name of the listed records.
				'plural'   => __( 'IAP Products', 'buddyboss-app' ), // plural name of the listed records.
				'ajax'     => false, // should this table support ajax?
			)
		);
	}

	/**
	 * Get products from database.
	 *
	 * @param int    $per_page    Items per page.
	 * @param int    $page_number Current page number.
	 * @param string $view        Tab view.
	 *
	 * @return mixed
	 */
	public static function get_products( $per_page = 5, $page_number = 1, $view = '' ) {
		$args = array();

		if ( isset( $_REQUEST['orderby'] ) && ! empty( $_REQUEST['orderby'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$args['orderby'] = sanitize_text_field( wp_unslash( $_REQUEST['orderby'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		}

		if ( isset( $_REQUEST['order'] ) && ! empty( $_REQUEST['order'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$args['order'] = sanitize_text_field( wp_unslash( $_REQUEST['order'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		}

		if ( ! empty( $view ) && 'all' !== $view ) {
			$args['status'] = $view;
		}

		$args['per_page']     = $per_page;
		$args['current_page'] = $page_number;
		$results              = bbapp_iap_get_products( $args );

		// convert to array.
		foreach ( $results as $k => $v ) {
			$results[ $k ] = (array) $v;
		}

		return $results;
	}

	/**
	 * Sync all products from apple and google.
	 *
	 * @param string $which 'top' or 'bottom'.
	 *
	 * @since BuddyBoss App 1.1.6
	 */
	public function extra_tablenav( $which ) {
		// Bail on bottom table nav.
		if ( 'bottom' === $which ) {
			return;
		}
		?>

		<button class="sync_product_button button" data-platform="all">
			<div class="bbapp_loading" style="display:none;"></div>
			<span class="sync_product_button_lable"><?php echo esc_html__( 'Sync Products', 'buddyboss-app' ); ?></span>
		</button>
		<?php
	}

	/**
	 * Returns the count of records in the database.
	 *
	 * @param string $status eg.trash.
	 *
	 * @return null|string
	 */
	public static function record_count( $status = '' ) {
		return bbapp_iap_get_products_count( $status );
	}

	/**
	 * Return the column available to this table list.
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns        = array();
		$product_status = ( ! empty( $_GET['product_status'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['product_status'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'trash' !== $product_status ) {
			$columns['reorder'] = '';
		}

		$columns['cb']                 = '<input type="checkbox" />';
		$columns['name']               = __( 'Product Name', 'buddyboss-app' );
		$columns['ios_product']        = __( 'iOS Product', 'buddyboss-app' );
		$columns['android_product']    = __( 'Android Product', 'buddyboss-app' );
		$columns['products_screen']    = __( 'Products Screen', 'buddyboss-app' );
		$columns['subscription-group'] = __( 'Subscription Group', 'buddyboss-app' );
		$columns['date_created']       = __( 'Date Created', 'buddyboss-app' );
		$columns['purchases']          = __( 'Purchases', 'buddyboss-app' );

		if ( bbapp_is_connected_as_secondary_site() ) {
			unset( $columns['ios_product'] );
			unset( $columns['android_product'] );
			unset( $columns['purchases'] );
		}
		return $columns;
	}

	/**
	 * Gets the name of the primary column.
	 *
	 * @return string The name of the primary column.
	 */
	public function get_primary_column_name() {
		return 'name';
	}

	/**
	 * No items.
	 */
	public function no_items() {
		esc_html_e( 'No products found.', 'buddyboss-app' );
	}

	/**
	 * Columns default data.
	 *
	 * @param array  $item        IAP product item array.
	 * @param string $column_name Column name.
	 *
	 * @return void
	 */
	public function column_default( $item, $column_name ) {
		$store_data    = maybe_unserialize( $item['store_data'] );
		$misc_settings = maybe_unserialize( $item['misc_settings'] );

		// NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_type or misc_settings.
		if ( bbapp()->is_network_activated() ) {
			$misc_settings = $misc_settings[ get_current_blog_id() ];
		}

		switch ( $column_name ) {
			case 'name':
				echo wp_kses_post( "<a class='row-title' href=admin.php?page=bbapp-iap&setting=products&action=edit&id=" . $item['id'] . '>' . $item['name'] . '</a>' );
				break;
			case 'ios_product':
				$store_product_detail = $this->iap_store_product_detail( $store_data, 'ios' );

				echo ! empty( $store_product_detail ) ? wp_kses_post( $store_product_detail ) : esc_html__( 'N/A', 'buddyboss-app' );
				break;
			case 'android_product':
				$store_product_detail = $this->iap_store_product_detail( $store_data, 'android' );

				echo ! empty( $store_product_detail ) ? wp_kses_post( $store_product_detail ) : esc_html__( 'N/A', 'buddyboss-app' );
				break;
			case 'date_created':
				$date_format = get_option( 'date_format' );
				$date_format = ! empty( $date_format ) ? $date_format : 'F j, Y';
				$time_format = get_option( 'time_format' );
				$time_format = ! empty( $time_format ) ? $time_format : 'g:i a';

				echo sprintf( '%1$s %2$s %3$s', esc_html( get_date_from_gmt( $item['date_created'], $date_format ) ), esc_html__( 'at', 'buddyboss-app' ), esc_html( get_date_from_gmt( $item['date_created'], $time_format ) ) );
				break;
			case 'products_screen':
				$product_available = 'No';

				if ( isset( $misc_settings['global_subscription'] ) ) {
					$product_available = $misc_settings['global_subscription'] ? 'Yes' : 'No';
				}

				echo esc_html( $product_available );
				break;
			case 'subscription-group':
				if ( ! empty( $item['iap_group'] ) ) {
					$product_subscription_group = SubscriptionGroupTaxonomy::get_group_name( $item['iap_group'] );
				}
				echo ! empty( $product_subscription_group ) ? esc_html( $product_subscription_group ) : esc_html__( 'N/A', 'buddyboss-app' );
				break;
			case 'id':
				echo esc_html( $item['id'] );
				break;
			case 'purchases':
				$this->bbapp_purchases_order_data( $item['id'] );
				break;
			default:
				echo 'N/A';
				break;
		}
	}

	/**
	 * IAP Store product detail
	 *
	 * @param array  $store_data      Product store data.
	 * @param string $device_platform Device platform.
	 *
	 * @return string
	 */
	public function iap_store_product_detail( $store_data, $device_platform ) {
		$store_product = '';

		if ( empty( $store_data ) ) {
			return $store_product;
		}

		$product_type = isset( $store_data['bbapp_product_type'] ) ? $store_data['bbapp_product_type'] : '';

		if ( 'free' === $product_type ) {
			$store_product .= __( 'Free', 'buddyboss-app' );
		} else {
			$enabled_iap_device_platform = $store_data['device_platforms'];
			$store_product_id            = ! empty( $store_data['store_product_ids'][ $device_platform ] ) ? $store_data['store_product_ids'][ $device_platform ] : '';
			$store_product_type          = ! empty( $store_data['store_product_types'][ $device_platform ] ) ? $store_data['store_product_types'][ $device_platform ] : '';

			if ( ! is_array( $enabled_iap_device_platform ) || ! in_array( $device_platform, $enabled_iap_device_platform, true ) ) {
				return __( 'Not set', 'buddyboss-app' );
			}

			// Check Product exists in store product.
			$store_products = bbapp_iap_get_store_products( $device_platform );
			if ( ! in_array( $store_product_id, array_column( $store_products, 'id' ), true ) ) {
				$store_product_id = '';
			}

			if ( ! empty( $store_product_id ) ) {
				$store_product           .= $store_product_id . '<br />';
				$store_product_type_label = Helpers::get_store_product_info( $store_product_type, $device_platform );

				if ( ! empty( $store_product_type_label ) ) {
					$store_product .= ' (' . $store_product_type_label . ') ';
				}

				$store_product_status = bbapp_iap_get_store_product( $store_product_id, $device_platform, 'status' );
				$product_status       = Helpers::get_store_product_status_info( $store_product_status, $device_platform );
				/* translators: %s: Product status. */
				$store_product .= sprintf( __( '<br /><span class="iap_product_status"><b>Status:</b> %s</div>', 'buddyboss-app' ), ucwords( $product_status ) );
			} else {
				$store_product .= __( 'Not set', 'buddyboss-app' );
			}
		}

		return $store_product;
	}

	/**
	 * Function to get sortable columns.
	 *
	 * @return array
	 */
	protected function get_sortable_columns() {
		return array(
			'name'         => array( 'name', true ),
			'date_created' => array( 'date_created', true ),
		);
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 */
	public function prepare_items() {
		// Init column headers.
		$this->_column_headers = array( $this->get_columns(), array(), array() );

		// Process bulk action.
		$this->process_bulk_action();

		$product_status = ( ! empty( $_GET['product_status'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['product_status'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		// Set the current view.
		if ( in_array( $product_status, array( 'trash', 'disabled' ), true ) ) {
			$this->view = $product_status;
		}

		$per_page                 = $this->get_items_per_page( 'iap_product_per_page', 10 );
		$current_page             = $this->get_pagenum();
		$total_items              = self::get_product_status_count(); // Get group type counts for display in the filter tabs.
		$this->iap_product_counts = array(); // Get order type counts for display in the filter tabs.

		foreach ( $total_items as $view => $total_item ) {
			$this->iap_product_counts[ $view ] = $total_item;
		}

		$this->set_pagination_args(
			array(
				'total_items' => $this->iap_product_counts[ $this->view ],
				// WE have to calculate the total number of items.
				'per_page'    => $per_page,
				// WE have to determine how many items to show on a page.
			)
		);

		$this->items = self::get_products( $per_page, $current_page, $this->view );
	}

	/**
	 * Get an array containing ids for each product status count.
	 *
	 * A bit of a kludge workaround for some issues
	 *
	 * @since BuddyPress 1.7.0
	 * @return array
	 */
	public static function get_product_status_count() {
		$status             = array();
		$status['all']      = self::record_count();
		$status['trash']    = self::record_count( 'trash' );
		$status['disabled'] = self::record_count( 'disabled' );

		return $status;
	}

	/**
	 * Checkbox for bulk items.
	 *
	 * @param array $item Product item array.
	 *
	 * @return string|void
	 */
	public function column_reorder( $item ) {
		return '<span class="iap-reorder-product"><i class="bb-icon-l bb-icon-bars"></i></span>';
	}

	/**
	 * Checkbox for bulk items.
	 *
	 * @param array $item Product item array.
	 *
	 * @return string|void
	 */
	public function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" id="cb-select-%s" name="iap_product_ids[]" value="%s" />',
			$item['id'],
			$item['id']
		);
	}

	/**
	 * Bulk action.
	 *
	 * @return array
	 */
	public function get_bulk_actions() {
		$actions        = array();
		$product_status = ( ! empty( $_GET['product_status'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['product_status'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'trash' === $product_status ) {
			$actions['bulk-untrash'] = esc_html__( 'Restore', 'buddyboss-app' );
			$actions['bulk-delete']  = esc_html__( 'Delete permanently', 'buddyboss-app' );
		} else {
			$actions['bulk-trash'] = esc_html__( 'Move to Trash', 'buddyboss-app' );
		}

		return $actions;
	}

	/**
	 * Bulk action process.
	 *
	 * @return bool
	 */
	public function process_bulk_action() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( "You don't have permission to access this page.", 'buddyboss-app' );
		}

		$request_uri = ( ! empty( $_SERVER['REQUEST_URI'] ) ) ? bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		// Build redirection URL.
		$redirect_to = remove_query_arg(
			array(
				'action',
				'action2',
				'iap_product',
				'iap_product_ids',
				'deleted',
				'error',
				'updated',
				'success_new',
				'error_new',
				'success_modified',
				'error_modified',
				'_wpnonce',
			),
			$request_uri
		);

		$action = $this->current_action();

		switch ( $action ) {
			case 'delete':
			case 'bulk-delete':
				if ( ! empty( $_REQUEST['iap_product_ids'] ) ) {
					$nonce_key = ( 'bulk-delete' === $action ) ? 'bulk-' . sanitize_key( $this->_args['plural'] ) : 'cc_delete_iap_product';

					check_admin_referer( $nonce_key );

					$iap_product_ids = wp_parse_id_list( map_deep( wp_unslash( $_REQUEST['iap_product_ids'] ), 'sanitize_text_field' ) );
					$count           = 0;

					foreach ( $iap_product_ids as $iap_product_id ) {
						if ( $this->delete_iap_product( $iap_product_id ) ) {
							$count ++;
						}
					}

					$redirect_to = add_query_arg(
						array(
							'deleted' => $count,
							'ids'     => $iap_product_ids,
						),
						$redirect_to
					);

					wp_safe_redirect( $redirect_to );
					exit();
				}
				break;
			case 'trash':
			case 'bulk-trash':
				if ( ! empty( $_REQUEST['iap_product_ids'] ) ) {
					$nonce_key = ( 'bulk-trash' === $action ) ? 'bulk-' . sanitize_key( $this->_args['plural'] ) : 'cc_trash_iap_product';

					check_admin_referer( $nonce_key );

					$iap_product_ids = wp_parse_id_list( map_deep( wp_unslash( $_REQUEST['iap_product_ids'] ), 'sanitize_text_field' ) );
					$count           = 0;

					foreach ( $iap_product_ids as $iap_product_id ) {
						if ( $this->trash_iap_product( $iap_product_id ) ) {
							$count ++;
						}
					}

					/**
					 * Action after trash iap product.
					 *
					 * @param array $iap_product_ids  Product ids.
					 *
					 * @since 1.6.7
					 */
					do_action( 'bbapp_after_trashed_iap_product', $iap_product_ids );

					$redirect_to = add_query_arg(
						array(
							'trashed' => $count,
							'ids'     => $iap_product_ids,
						),
						$redirect_to
					);

					wp_safe_redirect( $redirect_to );
					exit();
				}
				break;
			case 'untrash':
			case 'bulk-untrash':
				if ( ! empty( $_REQUEST['iap_product_ids'] ) ) {
					$nonce_key = ( 'bulk-untrash' === $action ) ? 'bulk-' . sanitize_key( $this->_args['plural'] ) : 'cc_untrash_iap_product';

					check_admin_referer( $nonce_key );

					if ( ! is_array( $_REQUEST['iap_product_ids'] ) ) {
						$_REQUEST['iap_product_ids'] = explode( ',', sanitize_text_field( wp_unslash( $_REQUEST['iap_product_ids'] ) ) );
					}

					$iap_product_ids = wp_parse_id_list( map_deep( wp_unslash( $_REQUEST['iap_product_ids'] ), 'sanitize_text_field' ) );
					$count           = 0;

					foreach ( $iap_product_ids as $iap_product_id ) {
						if ( $this->untrash_iap_product( $iap_product_id ) ) {
							$count ++;
						}
					}

					/**
					 * Action after trash iap product.
					 *
					 * @param array $iap_product_ids  Product ids.
					 *
					 * @since 1.6.7
					 */
					do_action( 'bbapp_after_untrashed_iap_product', $iap_product_ids );

					$redirect_to = add_query_arg(
						array(
							'untrashed' => $count,
							'ids'       => $iap_product_ids,
						),
						$redirect_to
					);

					wp_safe_redirect( $redirect_to );
					exit();
				}
				break;
		}

		return true;
	}

	/**
	 * Delete IAP product.
	 *
	 * @param int $id Product id to delete.
	 *
	 * @return false
	 */
	public function delete_iap_product( $id ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		return ProductHelper::instance()->delete_iap_product( $id );
	}

	/**
	 * Trash IAP product.
	 *
	 * @param int $id Product id to trash.
	 *
	 * @return false
	 */
	public function trash_iap_product( $id ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		return ProductHelper::instance()->update_iap_product_status( $id, 'trash' );
	}

	/**
	 * Trash IAP product.
	 *
	 * @param int $id Product id to untrash.
	 *
	 * @return false
	 */
	public function untrash_iap_product( $id ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		return ProductHelper::instance()->update_iap_product_status( $id );
	}


	/**
	 * Get the list of views available on this table (e.g. "all", "public").
	 */
	public function get_views() {
		$url_base = bbapp_get_admin_url( 'admin.php?page=bbapp-iap&setting=products' );
		?>
		<h2 class="screen-reader-text"><?php esc_html_e( 'Filter IAP Products list', 'buddyboss-app' ); ?></h2>

		<ul class="subsubsub">
			<li class="all">
				<a href="<?php echo esc_url( $url_base ); ?>" class="<?php echo ( 'all' === $this->view ) ? 'current' : ''; ?>">
					<?php
					printf( '%1$s <span class="count">(%2$s)</span>', esc_html__( 'All', 'buddyboss-app' ), esc_html( $this->iap_product_counts['all'] ) );
					?>
				</a>
				|
			</li>
			<li class="trashed">
				<a href="<?php echo esc_url( add_query_arg( 'product_status', 'trash', $url_base ) ); ?>" class="<?php echo ( 'trash' === $this->view ) ? 'current' : ''; ?>">
					<?php
					/* translators: %s: Trashed item count. */
					printf( wp_kses_post( _n( 'Trash <span class="count">(%s)</span>', 'Trash <span class="count">(%s)</span>', esc_html( number_format_i18n( $this->iap_product_counts['trash'] ) ), 'buddyboss-app' ) ), esc_html( number_format_i18n( $this->iap_product_counts['trash'] ) ) );
					?>
				</a>
			</li>
			<li class="disabled">
				<a href="<?php echo esc_url( add_query_arg( 'product_status', 'disabled', $url_base ) ); ?>" class="<?php echo ( 'disabled' === $this->view ) ? esc_attr( 'current' ) : ''; ?>">
					<?php
					/* translators: %1$s: Product count singular, %2$s: Product count plural. */
					printf( wp_kses_post( _n( '%1$s <span class="count">(%2$s)</span>', '%1$s <span class="count">(%2$s)</span>', esc_html( $this->iap_product_counts['disabled'] ), 'buddyboss-app' ) ), esc_html__( 'Disabled', 'buddyboss-app' ), esc_html( number_format_i18n( $this->iap_product_counts['disabled'] ) ) );
					?>
				</a>
			</li>
		</ul>
		<?php
	}

	/**
	 * Column data function.
	 *
	 * @param array $item IAP item array.
	 */
	public function column_name( $item ) {
		$actions        = array();
		$page_request   = ( ! empty( $_REQUEST['page'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$base_url       = bbapp_get_admin_url( 'admin.php?page=' . esc_attr( $page_request ) . '&amp;setting=products&amp;iap_product_ids=' . $item['id'] ); // Build actions URLs.
		$product_status = ( ! empty( $_GET['product_status'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['product_status'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$base_url       = add_query_arg( 'product_status', $product_status, $base_url );

		if ( 'trash' === $product_status ) {
			$restore_url = wp_nonce_url( $base_url . '&amp;action=untrash', 'cc_untrash_iap_product' );
			$delete_url  = wp_nonce_url( $base_url . '&amp;action=delete', 'cc_delete_iap_product' );
			/* translators: %s: Post title. */
			$actions['untrash'] = sprintf( '<a href="%s" aria-label="%s">%s</a>', $restore_url, esc_attr( sprintf( __( 'Restore &#8220;%s&#8221; from the Trash', 'buddyboss-app' ), $item['name'] ) ), __( 'Restore', 'buddyboss-app' ) );
			/* translators: %s: Post title. */
			$actions['delete'] = sprintf( '<a href="%s" class="submitdelete" onclick="return showNotice.warn();" aria-label="%s">%s</a>', $delete_url, esc_attr( sprintf( __( 'Delete &#8220;%s&#8221; permanently', 'buddyboss-app' ), $item['name'] ) ), __( 'Delete Permanently', 'buddyboss-app' ) );

			return sprintf( '%1$s %2$s', $item['name'], $this->row_actions( $actions ) );
		}

		$trash_url = wp_nonce_url( $base_url . '&amp;action=trash', 'cc_trash_iap_product' );
		/* translators: %s: Post title. */
		$actions['edit'] = sprintf( '<a href="%s" aria-label="%s">%s</a>', sprintf( 'admin.php?page=%s&setting=products&action=%s&id=%s', $page_request, 'edit', $item['id'] ), esc_attr( sprintf( __( 'Edit &#8220;%s&#8221;', 'buddyboss-app' ), $item['name'] ) ), __( 'Edit', 'buddyboss-app' ) );
		/* translators: %s: Post title. */
		$actions['trash'] = sprintf( '<a href="%s" aria-label="%s">%s</a>', $trash_url, esc_attr( sprintf( __( 'Move &#8220;%s&#8221; to the Trash', 'buddyboss-app' ), $item['name'] ) ), __( 'Trash', 'buddyboss-app' ) );

		// NOTE : Delete operation is only permitted at main(default) site,
		// network level don't have product creation screen).
		if ( ! is_main_site() ) {
			unset( $actions['trash'] );
			unset( $actions['delete'] );
		}

		$disabled_html = '';

		if ( 'disabled' === $item['status'] ) {
			$disabled_html = sprintf( '<span class="iap-status iap-status-disabled">%1$s</span>', esc_html__( 'Disabled', 'buddyboss-app' ) );
		}

		$edit_product_url = esc_url( "admin.php?page=bbapp-iap&setting=products&action=edit&id={$item['id']}" );

		return sprintf( '<div class="iap-title"><a class="row-title" href="%1$s">%2$s</a> %3$s</div> %4$s', esc_url( $edit_product_url ), esc_html( $item['name'] ), wp_kses_post( $disabled_html ), wp_kses_post( $this->row_actions( $actions ) ) );
	}

	/**
	 * Single row function.
	 *
	 * @param array $item  iap product array.
	 * @param int   $level Level.
	 */
	public function single_row( $item, $level = 0 ) {
		$classes = '';

		if ( $item['status'] ) {
			$classes = 'iap-product status-' . $item['status'];
		}
		?>
		<tr id="iap-product-<?php echo esc_attr( $item['id'] ); ?>" class="<?php esc_attr( $classes ); ?>">
			<?php $this->single_row_columns( $item ); ?>
		</tr>
		<?php
	}

	/**
	 * Return count of all orders status based on particular product id.
	 *
	 * @param int $purchase_product_id Product id.
	 */
	public function bbapp_purchases_order_data( $purchase_product_id ) {
		$order_list_obj   = new OrderList();
		$all_order_status = $order_list_obj::get_orders_type_ids( $purchase_product_id );

		// Added expired order status with count in above array.
		// We have not added in above function because this is displaying count in table row in IAP screen for filtering purpose.
		$all_order_status['expired'] = $order_list_obj::record_count( $purchase_product_id, 'expired' );

		if ( ! empty( $all_order_status ) ) {
			$order_link = admin_url( 'admin.php?page=bbapp-iap&setting=orders' );
			$order_link = add_query_arg( 'iap_product_id', $purchase_product_id, $order_link );
			unset( $all_order_status['all'] );
			?>
			<ul>
				<?php
				foreach ( $all_order_status as $order_status => $order_count ) {
					if ( ! empty( $order_count ) ) {
						?>
						<li>
							<strong><?php echo esc_html( ucfirst( $order_status ) ); ?></strong>: <?php echo esc_html( $order_count ); ?>
						</li>
						<?php
					}
				}
				?>
				<a href="<?php echo esc_url( $order_link ); ?>"><?php esc_html_e( 'View Purchases', 'buddyboss-app' ); ?></a>
			</ul>
			<?php
		}
	}
}
