<?php
/**
 * Holds IAP related helper functions.
 *
 * @package BuddyBossApp\Admin\InAppPurchases
 */

namespace BuddyBossApp\Admin\InAppPurchases;

use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\InAppPurchases\Orders;
use BuddyBossApp\InAppPurchases\SubscriptionGroupTaxonomy;
use BuddyBossApp\Helpers\BBAPP_File;
use BuddyBossApp\AppSettings;

/**
 * IAP product helping functions class.
 */
class ProductHelper {

	/**
	 * Class instance.
	 *
	 * @var null $instance
	 */
	private static $instance = null;

	/**
	 * Current page to load screen.
	 *
	 * @var $is_current_page
	 */
	private $is_current_page;

	/**
	 * IAP product list.
	 *
	 * @var ProductList
	 */
	private $products_list;

	/**
	 * ProductHelper constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Get the instance of the class.
	 *
	 * @return ProductHelper
	 */
	public static function instance() {

		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Magic methods for old methods.
	 *
	 * @param string             $name      Method name.
	 * @param string|mixed|array $arguments Method arguments.
	 *
	 * @since 1.7.5
	 *
	 * @return mixed|void
	 */
	public static function __callStatic( $name, $arguments ) {
		// Compatibility Support for CamelCase.
		$compatibility_func = array(
			'getActiveOrder'        => 'get_active_order',
			'hasActiveOrder'        => 'has_active_order',
			'getGroupProductsIds'   => 'get_group_products_ids',
			'getGroupActiveOrderId' => 'get_group_active_order_id',
			'getIapProduct'         => 'get_iap_product',
			'getIAProductStatus'    => 'get_iap_product_status',
		);

		foreach ( $compatibility_func as $old => $new ) {
			if ( $name === $old ) {
				_deprecated_function( esc_html( $old ), esc_html( bbapp()->plugin_version ), esc_html( $new ) );

				return call_user_func_array( array( self::instance(), $new ), $arguments );
			}
		}
	}

	/**
	 * Functions to check/verify if there is order for this product
	 *
	 * @param object $iap_product IAP product data.
	 * @param int    $user_id     User id.
	 *
	 * @return bool|array
	 */
	public static function get_active_order( $iap_product, $user_id = 0 ) {
		global $wpdb;

		if ( is_array( $iap_product ) ) {
			$iap_product = (object) $iap_product;
		}

		if ( empty( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		$table = Orders::instance()->table_name;

		/**
		 * We are checking below things :
		 * order_status is one of 'subscribed', 'completed'
		 * matching bbapp_product_id
		 * matching user_id
		 */
		$result = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table} WHERE bbapp_product_id = %s AND user_id = %d AND order_status IN ('subscribed', 'completed')", $iap_product->id, $user_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! empty( $result ) ) {
			return $result;
		}

		return false;
	}

	/**
	 * Functions to check/verify if there is order for this product
	 *
	 * @param object $iap_product IAP product id.
	 * @param int    $user_id     User id.
	 *
	 * @return bool
	 */
	public static function has_active_order( $iap_product, $user_id = 0 ) {
		global $wpdb;

		if ( is_array( $iap_product ) ) {
			$iap_product = (object) $iap_product;
		}

		if ( empty( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		$table = Orders::instance()->table_name;

		/**
		 * We are checking below things :
		 * order_status is one of 'subscribed', 'completed'
		 * matching bbapp_product_id
		 * matching user_id
		 */
		$count = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$table} WHERE bbapp_product_id = %s AND user_id = %d AND order_status IN ('subscribed', 'completed')", $iap_product->id, $user_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( $count > 0 ) {
			return true;
		}

		$misc_settings       = Helpers::bbapp_iap_product_mics_setting( $iap_product );
		$integration_data    = Helpers::bbapp_iap_product_integration_data( $iap_product );
		$integration_type    = ( isset( $misc_settings['integration_type'] ) ? $misc_settings['integration_type'] : '' );
		$integrated_item_ids = Helpers::bbapp_iap_product_integration_ids( $integration_type, $integration_data );

		/**
		 * Filter to check given $integrated_item_ids has access by user or not.
		 */
		return apply_filters( 'bbapp_iap_is_has_access', false, $integrated_item_ids, $integration_type, $user_id );
	}

	/**
	 * Function to get product id of active order in given group
	 *
	 * @param int $group_id Group Id.
	 * @param int $user_id  User Id.
	 *
	 * @return bool
	 */
	public static function get_group_active_order_product_id( $group_id, $user_id = 0 ) {
		if ( empty( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		if ( empty( $group_id ) || ! term_exists( (int) $group_id, SubscriptionGroupTaxonomy::$taxonomy ) ) {
			return 0;
		}

		$products = self::get_group_products_ids( (int) $group_id, 'all' );

		if ( empty( $products ) ) {
			return 0;
		}

		$retval = 0;

		foreach ( $products as $product ) {
			$status = self::has_active_order( $product, $user_id );

			if ( true === $status ) {
				$retval = $product->id;
				break;
			}
		}

		return (int) $retval;
	}

	/**
	 * Function to get active order in given group
	 *
	 * @param int $group_id Group Id.
	 * @param int $user_id   User Id.
	 *
	 * @return bool
	 */
	public static function get_group_active_order_id( $group_id, $user_id ) {
		global $wpdb;

		$product_ids = self::get_group_products_ids( $group_id );

		if ( empty( $product_ids ) ) {
			return false;
		}

		$product_ids = implode( ',', $product_ids );
		$table       = Orders::instance()->table_name;

		/**
		 * We are checking below things :
		 * order_status is one of 'subscribed', 'completed'
		 * matching bbapp_product_ids
		 * matching user_id
		 */

		return (int) $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$table} WHERE bbapp_product_id IN ({$product_ids}) AND user_id = %s AND order_status IN ('subscribed', 'completed') limit 1", $user_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Function to get active order in given group except given order id.
	 *
	 * @param int $group_id  Group Id.
	 * @param int $user_id   User Id.
	 * @param int $order_id  Order Id.
	 *
	 * @since [BBBAPPVERSION]
	 * 
	 * @return bool
	 */
	public static function get_group_active_order_id_except( $group_id, $user_id, $order_id ) {
		global $wpdb;

		$product_ids = self::get_group_products_ids( $group_id );

		if ( empty( $product_ids ) ) {
			return false;
		}

		$product_ids = implode( ',', $product_ids );
		$table       = Orders::instance()->table_name;

		/**
		 * We are checking below things :
		 * order_status is one of 'subscribed', 'completed'
		 * matching bbapp_product_ids
		 * matching user_id
		 * except given order id
		 */
		return (int) $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$table} WHERE bbapp_product_id IN ({$product_ids}) AND user_id = %s AND order_status IN ('subscribed', 'completed') AND id != %d limit 1", $user_id, $order_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Initialize actions/filters
	 *
	 * @return void
	 */
	public function init() {
		add_action( 'admin_init', array( $this, 'admin_init' ) );

		// NOTE : can't use admin_post hook since it doesn't exist on multi-site.
		add_action( 'admin_init', array( $this, 'store' ) );
		add_action( 'admin_init', array( $this, 'update' ) );

		// IAP Product sorting.
		add_action( 'wp_ajax_update-iap-products-order', array( $this, 'bbapp_update_iap_products_ui_sorting' ) );
		add_action( 'wp_ajax_bbapp_get_lms_dropdown', array( $this, 'bbapp_get_lms_dropdown' ) );

		// IAP screen options.
		add_action( 'load-buddyboss-app_page_bbapp-iap', array( $this, 'get_screen_options' ) );
		add_filter( 'set-screen-option', array( $this, 'bbapp_admin_screen_options' ), 10, 3 );

		// Notice for App registration products enabled.
		add_action( 'admin_notices', array( $this, 'bbapp_no_registration_product' ), 99 );

		// Background Job.
		add_action( 'bbapp_queue_task_verify_iap_products', array( $this, 'verify_miss_configured_bg_job' ), 10, 1 );
		add_action( 'bbapp_connect_app', array( $this, 'iap_product_run_job' ), 10, 1 );
		if ( ! bbapp_is_connected_as_secondary_site() ) {
			add_action( 'bbapp_disconnect_app', array( $this, 'iap_product_run_job' ), 10, 1 );
			add_action( 'bbapp_after_trashed_iap_product', array( $this, 'iap_product_run_job' ), 10, 1 );
			add_action( 'bbapp_after_untrashed_iap_product', array( $this, 'iap_product_run_job' ), 10, 1 );
			add_action( 'bbapp_after_iap_created_product', array( $this, 'iap_save_product_run_job' ), 10, 2 );
			add_action( 'bbapp_after_iap_updated_product', array( $this, 'iap_save_product_run_job' ), 10, 2 );
		}
	}

	/**
	 * Function to add notice when there is no registration products
	 *
	 * @since 1.4.4
	 */
	public function bbapp_no_registration_product() {
		if ( bbapp_is_active( 'iap' ) && AppSettings::instance()->get_setting_value( 'iap.purchase_before_register' ) && method_exists( 'BuddyBossApp\Admin\InAppPurchases\Helpers', 'get_products' ) && 0 >= count( Helpers::get_products( array( 'registration_products' => true ) ) ) ) {
			printf( '<div class="notice notice-error is-dismissible"><p>%1$s <a href="%2$s">%3$s</a>.</p></div>', esc_html__( 'To add products to registration in your app, please enable the Registration field in at least one of your', 'buddyboss-app' ), esc_url( admin_url( 'admin.php?page=bbapp-iap&setting=products' ) ), esc_html__( 'products', 'buddyboss-app' ) );
		}
	}

	/**
	 * Store a newly created BuddyBossApp-Product in DB
	 *
	 * @return void
	 */
	public function store() {
		if ( isset( $_REQUEST['iap_save_wpnonce'] ) ) {
			$nonce_request = ( ! empty( $_REQUEST['iap_save_wpnonce'] ) ) ? sanitize_text_field( wp_unslash( $_REQUEST['iap_save_wpnonce'] ) ) : '';

			// Verify the nonce here.
			if ( wp_verify_nonce( $nonce_request, 'save_iap_product' ) ) {
				$name = isset( $_REQUEST['name'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['name'] ) ) : null;

				if ( empty( $name ) ) {
					wp_die( esc_html__( 'Missing Product Name Param.', 'buddyboss-app' ), esc_html__( 'IAP Product', 'buddyboss-app' ) );
				}

				$tagline                              = isset( $_REQUEST['tagline'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['tagline'] ) ) : null;
				$description                          = isset( $_REQUEST['description'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['description'] ) ) : null;
				$bbapp_product_type                   = isset( $_REQUEST['bbapp_product_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['bbapp_product_type'] ) ) : 'free';
				$misc_settings                        = array();
				$misc_settings['integration_type']    = isset( $_REQUEST['integration_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['integration_type'] ) ) : null;
				$misc_settings['integration_type']    = 'membership' === $misc_settings['integration_type'] ? ( isset( $_REQUEST['integration_sub_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['integration_sub_type'] ) ) : null ) : $misc_settings['integration_type'];
				$misc_settings['benefits']            = isset( $_REQUEST['benefits'] ) ? map_deep( wp_unslash( $_REQUEST['benefits'] ), 'sanitize_text_field' ) : array();
				$misc_settings['global_subscription'] = isset( $_REQUEST['global_subscription'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['global_subscription'] ) ) : 0;
				$group_id                             = isset( $_REQUEST[ SubscriptionGroupTaxonomy::$taxonomy ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ SubscriptionGroupTaxonomy::$taxonomy ] ) ) : '';

				// Course access data.
				$is_not_lms_integration         = ( ( 'learndash' !== $misc_settings['integration_type'] && bbapp_is_learndash_enabled() ) || ( 'tutor-lms-course' !== $misc_settings['integration_type'] && bbapp_is_tutor_lms_enabled() ) || bbapp_is_memberpress_lms_enabled() );
				$misc_settings['course_access'] = ( ! empty( $_REQUEST['course_access'] ) && ! empty( $_REQUEST['integration_type'] ) && 'membership' === $_REQUEST['integration_type'] && $is_not_lms_integration ) ? 1 : 0;

				$device_platforms    = isset( $_REQUEST['bbapp_store_enabled'] ) ? map_deep( wp_unslash( $_REQUEST['bbapp_store_enabled'] ), 'sanitize_text_field' ) : null;
				$store_product_ids   = array();
				$store_product_types = array();
				$store_data          = array();
				$integration_data    = array();

				if ( ! empty( $device_platforms ) ) {
					// Going through device.
					foreach ( (array) $device_platforms as $device_platform ) {
						$iap_device_platform_id           = "iap-$device_platform-id";
						$integration_device_platform_type = "bbapp-iap-integration-$device_platform-product_types";

						$store_product_ids[ $device_platform ]   = isset( $_REQUEST[ $iap_device_platform_id ] ) ? trim( sanitize_text_field( wp_unslash( $_REQUEST[ $iap_device_platform_id ] ) ) ) : null;
						$store_product_types[ $device_platform ] = isset( $_REQUEST[ $integration_device_platform_type ] ) ? trim( sanitize_text_field( wp_unslash( $_REQUEST[ $integration_device_platform_type ] ) ) ) : null;
					}
				}

				/**
				 * Check selected integration enabled & integration data [ unlock item ] selected then store it.
				 */
				$iap_integrations = Controller::instance()->get_integrations();
				if ( isset( $iap_integrations[ $misc_settings['integration_type'] ] ) ) {
					if ( $iap_integrations[ $misc_settings['integration_type'] ]['enabled'] && isset( $_REQUEST[ $misc_settings['integration_type'] ] ) ) {
						$integration_data[ $misc_settings['integration_type'] ] = map_deep( wp_unslash( $_REQUEST[ $misc_settings['integration_type'] ] ), 'sanitize_text_field' );
					}
				}

				// IAP feature image update in misc setting.
				$store_data['bbapp_product_type']  = $bbapp_product_type;
				$store_data['device_platforms']    = $device_platforms;
				$store_data['store_product_types'] = $store_product_types;
				$store_data['store_product_ids']   = $store_product_ids;

				// NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_type or misc_settings.
				if ( bbapp()->is_network_activated() ) {
					$integration_data[ get_current_blog_id() ] = $integration_data;
					$misc_settings[ get_current_blog_id() ]    = $misc_settings;
				}

				// NOTE : Insert into DB.
				$create_product = bbapp_iap_create_product(
					array(
						'name'             => trim( $name ),
						'tagline'          => trim( $tagline ),
						'description'      => trim( $description ),
						'misc_settings'    => maybe_serialize( $misc_settings ),
						'store_data'       => maybe_serialize( $store_data ),
						'integration_data' => maybe_serialize( $integration_data ),
						'iap_group'        => $group_id,
						'status'           => 'published',
					)
				);

				if ( is_wp_error( $create_product ) ) {
					wp_die( sprintf( '%1$s %2$s', esc_html__( 'An error occurred while creating the product:', 'buddyboss-app' ), esc_html( $create_product->get_error_message() ) ) );
				}

				$lms_show_on = ! empty( $_REQUEST['lms_show_on'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['lms_show_on'] ) ) : '';

				if ( ! empty( $lms_show_on ) && 1 === $misc_settings['course_access'] ) {
					bbapp_iap_update_meta( $create_product['id'], 'course_access_type', $lms_show_on );
				}

				if ( ! empty( $_REQUEST['lms_courses_selection'] ) && 1 === $misc_settings['course_access'] ) {
					$new_key_type          = $this->bbapp_get_lms_meta_key_type( $lms_show_on );
					$lms_courses_selection = map_deep( wp_unslash( $_REQUEST['lms_courses_selection'] ), 'sanitize_text_field' );
					$lms_selection         = array_unique( $lms_courses_selection );

					// Add new meta based on the selection.
					foreach ( $lms_selection as $item ) {
						bbapp_iap_add_meta( $create_product['id'], $new_key_type, $item );
					}
				}

				// IAP feature image store in update meta.
				if ( isset( $_REQUEST['iap_product_image_name_delete'] ) ) {
					$this->bbapp_delete_featured_image( sanitize_text_field( wp_unslash( $_REQUEST['iap_product_image_name_delete'] ) ) );

					if ( isset( $_FILES ) ) {
						$this->bbapp_process_featured_image( $create_product['id'], $_FILES, sanitize_text_field( wp_unslash( $_REQUEST['iap_product_image_name_delete'] ) ) );
					}
				}

				// IAP product registration store in update meta.
				bbapp_iap_update_meta( $create_product['id'], 'product_visibility_registration', ! empty( $_REQUEST['product_visibility_registration'] ) ? 1 : 0 );

				$redirect_to = add_query_arg(
					array(
						'page'    => 'bbapp-iap',
						'setting' => 'products',
						'action'  => 'edit',
						'id'      => $create_product['id'],
						'created' => 1,
					),
					'admin.php'
				);

				wp_safe_redirect( $redirect_to );
				exit;
			}
		}
	}

	/**
	 * Update a BuddyBossApp-Product in DB.
	 *
	 * @return void
	 */
	public function update() {
		if ( isset( $_REQUEST['iap_update_wpnonce'] ) ) {
			$nonce_request = ( ! empty( $_REQUEST['iap_update_wpnonce'] ) ) ? sanitize_text_field( wp_unslash( $_REQUEST['iap_update_wpnonce'] ) ) : '';
			// Verify the nonce here.
			if ( wp_verify_nonce( $nonce_request, 'update_iap_product' ) ) {
				$id   = ! empty( $_REQUEST['id'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['id'] ) ) : 0;
				$name = isset( $_REQUEST['name'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['name'] ) ) : null;

				if ( empty( $name ) ) {
					wp_die( esc_html__( 'Required fields:  product name, and description.', 'buddyboss-app' ) );
				}

				$tagline                              = isset( $_REQUEST['tagline'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['tagline'] ) ) : null;
				$description                          = isset( $_REQUEST['description'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['description'] ) ) : null;
				$bbapp_product_type                   = isset( $_REQUEST['bbapp_product_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['bbapp_product_type'] ) ) : 'free';
				$misc_settings                        = array();
				$misc_settings['integration_type']    = isset( $_REQUEST['integration_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['integration_type'] ) ) : null;
				$misc_settings['integration_type']    = 'membership' === $misc_settings['integration_type'] ? ( isset( $_REQUEST['integration_sub_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['integration_sub_type'] ) ) : null ) : $misc_settings['integration_type'];
				$misc_settings['benefits']            = isset( $_REQUEST['benefits'] ) ? map_deep( wp_unslash( $_REQUEST['benefits'] ), 'sanitize_text_field' ) : array();
				$misc_settings['global_subscription'] = isset( $_REQUEST['global_subscription'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['global_subscription'] ) ) : 0;
				$group_id                             = isset( $_REQUEST[ SubscriptionGroupTaxonomy::$taxonomy ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ SubscriptionGroupTaxonomy::$taxonomy ] ) ) : '';

				// Course access data.
				$is_not_lms_integration         = ( ( 'learndash' !== $misc_settings['integration_type'] && bbapp_is_learndash_enabled() ) || ( 'tutor-lms-course' !== $misc_settings['integration_type'] && bbapp_is_tutor_lms_enabled() ) || bbapp_is_memberpress_lms_enabled() );
				$misc_settings['course_access'] = ( ! empty( $_REQUEST['course_access'] ) && ! empty( $_REQUEST['integration_type'] ) && 'membership' === $_REQUEST['integration_type'] && $is_not_lms_integration ) ? 1 : 0;

				if ( ! empty( $_REQUEST['lms_show_on'] ) && 1 === $misc_settings['course_access'] ) {
					$old_course_type = bbapp_iap_get_meta( $id, 'course_access_type', true );

					if ( ! empty( $old_course_type ) ) {
						$old_key_type = $this->bbapp_get_lms_meta_key_type( $old_course_type );

						if ( ! empty( $old_key_type ) ) {
							$old_meta = bbapp_iap_get_meta( $id, $old_key_type, false );

							if ( ! empty( $old_meta ) ) {
								foreach ( $old_meta as $old_item ) {
									bbapp_iap_delete_meta( $id, $old_key_type, $old_item );
								}
							}
						}
					}

					$lms_show_on = ! empty( $_REQUEST['lms_show_on'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['lms_show_on'] ) ) : '';

					bbapp_iap_update_meta( $id, 'course_access_type', $lms_show_on );

					$new_key_type = $this->bbapp_get_lms_meta_key_type( $lms_show_on );

					if ( ! empty( $_REQUEST['lms_courses_selection'] ) ) {
						$lms_courses_selection = map_deep( wp_unslash( $_REQUEST['lms_courses_selection'] ), 'sanitize_text_field' );
						$lms_selection         = array_unique( $lms_courses_selection );
						// Add new meta based on the selection.
						foreach ( $lms_selection as $item ) {
							bbapp_iap_add_meta( $id, $new_key_type, $item );
						}
					}
				} else {
					/**
					 * Filters meta items to be deleted.
					 *
					 * @param array $meta_items Meta items to be deleted.
					 */
					$iap_meta_delete = apply_filters(
						'iap_course_delete_meta',
						array(
							'course_access_cat_id',
							'course_access_tag_id',
							'course_access_id',
							'course_access_type',
						)
					);

					if ( ! empty( $iap_meta_delete ) ) {
						foreach ( $iap_meta_delete as $iap_meta_key ) {
							bbapp_iap_delete_meta( $id, $iap_meta_key, '', true );
						}
					}
				}

				$device_platforms    = isset( $_REQUEST['bbapp_store_enabled'] ) ? map_deep( wp_unslash( $_REQUEST['bbapp_store_enabled'] ), 'sanitize_text_field' ) : null;
				$store_product_ids   = array();
				$store_product_types = array();
				$store_data          = array();
				$integration_data    = array();

				if ( ! empty( $device_platforms ) ) {
					// Going through device.
					foreach ( (array) $device_platforms as $device_platform ) {
						$iap_device_platform_id           = "iap-$device_platform-id";
						$integration_device_platform_type = "bbapp-iap-integration-$device_platform-product_types";

						$store_product_ids[ $device_platform ]   = isset( $_REQUEST[ $iap_device_platform_id ] ) ? trim( sanitize_text_field( wp_unslash( $_REQUEST[ $iap_device_platform_id ] ) ) ) : null;
						$store_product_types[ $device_platform ] = isset( $_REQUEST[ $integration_device_platform_type ] ) ? trim( sanitize_text_field( wp_unslash( $_REQUEST[ $integration_device_platform_type ] ) ) ) : null;
					}
				}

				// Check selected integration enabled & integration data [ unlock item ] selected then store it.
				$iap_integrations = Controller::instance()->get_integrations();
				if ( isset( $iap_integrations[ $misc_settings['integration_type'] ] ) ) {
					if ( $iap_integrations[ $misc_settings['integration_type'] ]['enabled'] && isset( $_REQUEST[ $misc_settings['integration_type'] ] ) ) {
						$integration_data[ $misc_settings['integration_type'] ] = map_deep( wp_unslash( $_REQUEST[ $misc_settings['integration_type'] ] ), 'sanitize_text_field' );
					}
				}

				$store_data['bbapp_product_type']  = $bbapp_product_type;
				$store_data['device_platforms']    = $device_platforms;
				$store_data['store_product_types'] = $store_product_types;
				$store_data['store_product_ids']   = $store_product_ids;

				// NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_type or misc_settings.
				if ( bbapp()->is_network_activated() ) {
					$buddyboss_app_product = bbapp_iap_get_product( $id );
					// NOTE : For misc_settings we need to append data NOT override.
					$current_misc_settings                          = maybe_unserialize( $buddyboss_app_product['misc_settings'] );
					$current_misc_settings[ get_current_blog_id() ] = $misc_settings;
					$misc_settings                                  = $current_misc_settings;
					// NOTE : For integration_data we need to append data NOT override.
					$current_integration_sata                          = maybe_unserialize( $buddyboss_app_product['integration_data'] );
					$current_integration_sata[ get_current_blog_id() ] = $integration_data;
					$integration_data                                  = $current_integration_sata;
				}

				// NOTE : Update it into DB.
				$updated_data['name']             = trim( $name );
				$updated_data['tagline']          = trim( $tagline );
				$updated_data['description']      = trim( $description );
				$updated_data['misc_settings']    = maybe_serialize( $misc_settings );
				$updated_data['store_data']       = maybe_serialize( $store_data );
				$updated_data['integration_data'] = maybe_serialize( $integration_data );
				$updated_data['date_updated']     = current_time( 'mysql', 1 );
				$updated_data['iap_group']        = $group_id;
				$updated_data['status']           = 'published';

				// NOTE : We only allow edits at NetworkLevel NOT at subSite(s) for MultiSite WordPress.
				if ( bbapp()->is_network_activated() && ! is_network_admin() ) {
					unset( $updated_data['name'] );
					unset( $updated_data['tagline'] );
					unset( $updated_data['description'] );
					unset( $updated_data['store_data'] );
					unset( $updated_data['name'] );
				}

				// NOTE : Update.
				$update_product = bbapp_iap_update_product( $id, $updated_data );

				// IAP feature image store in update meta.
				if ( isset( $_REQUEST['iap_product_image_name_delete'] ) ) {
					$this->bbapp_delete_featured_image( sanitize_text_field( wp_unslash( $_REQUEST['iap_product_image_name_delete'] ) ) );

					if ( isset( $_FILES ) ) {
						$this->bbapp_process_featured_image( $id, $_FILES, sanitize_text_field( wp_unslash( $_REQUEST['iap_product_image_name_delete'] ) ) );
					}
				}

				// IAP product registration store in update meta.
				bbapp_iap_update_meta( $id, 'product_visibility_registration', ! empty( $_REQUEST['product_visibility_registration'] ) ? 1 : 0 );

				if ( is_wp_error( $update_product ) ) {
					/* translators: %s: Update product error. */
					wp_die( sprintf( '%1$s %2$s', esc_html__( 'An error occurred while updating the product:', 'buddyboss-app' ), esc_html( $update_product->get_error_message() ) ) );
				}

				$redirect_to = add_query_arg(
					array(
						'page'    => 'bbapp-iap',
						'setting' => 'products',
						'action'  => 'edit',
						'id'      => $id,
						'updated' => 1,
					),
					'admin.php'
				);

				wp_safe_redirect( $redirect_to );
				exit;
			}
		}
	}

	/**
	 * Function to delete IAP featured image.
	 *
	 * @param string $img_name image name.
	 *
	 * @since 1.4.4
	 *
	 * @return bool
	 */
	public function bbapp_delete_featured_image( $img_name ) {
		$delete = false;

		if ( ! $img_name ) {
			return $delete;
		}

		$out_location = get_iap_upload_dir();
		$file_path    = $out_location . '/' . $img_name;

		if ( file_exists( $file_path ) && is_file( $file_path ) ) {
			if ( unlink( $file_path ) ) {
				$delete = true;
			}
		}

		return $delete;
	}

	/**
	 * Function to process IAP featured image.
	 *
	 * @param int    $iap_id                        IAP product ID.
	 * @param array  $files                         Uploaded files.
	 * @param string $iap_product_image_name_delete product featured image name.
	 *
	 * @since 1.4.4
	 *
	 * @return bool
	 */
	public function bbapp_process_featured_image( $iap_id, $files, $iap_product_image_name_delete ) {
		$updated = false;

		if ( isset( $files ) && ! empty( $iap_id ) ) {
			$field_name = 'iap_product_image';
			if ( ( isset( $files[ $field_name ]['name'] ) && ! empty( $files[ $field_name ]['name'] ) ) || ! empty( $iap_product_image_name_delete ) ) {
				$file_data = $this->bbapp_iap_uploaded_feature_image( $files, $field_name, $iap_id );
				bbapp_iap_update_meta( $iap_id, 'iap_product_image', $file_data );
				$updated = true;
			}
		}

		return $updated;
	}

	/**
	 * Function to get the meta key type.
	 *
	 * @param string $selection LMS show on selection.
	 *
	 * @since 1.4.0
	 *
	 * @return string
	 */
	public function bbapp_get_lms_meta_key_type( $selection ) {
		$key_type = '';

		if ( 'specific-category-courses' === $selection ) {
			$key_type = 'course_access_cat_id';
		} elseif ( 'specific-tag-courses' === $selection ) {
			$key_type = 'course_access_tag_id';
		} elseif ( 'specific-course' === $selection ) {
			$key_type = 'course_access_id';
		}

		return $key_type;
	}

	/**
	 * Load Product List.
	 *
	 * @return void
	 */
	public function admin_init() {
		if ( $this->will_render() ) {
			$this->products_list = new ProductList();
			$this->products_list->prepare_items();
		}
	}

	/**
	 * Functions tells & sets if current page is one where it will render.
	 *
	 * @param bool $set Whether to render screen.
	 *
	 * @return bool
	 */
	public function will_render( $set = false ) {
		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Renders the branding screen
	 */
	public function render() {
		$action           = isset( $_REQUEST['action'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['action'] ) ) : 'lists'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$device_platforms = bbapp_iap_get_types();

		$this->iap_screen_notice();

		switch ( $action ) {
			case 'create':
				// Used in template file.
				$iap_integrations   = Controller::instance()->get_integrations();
				$bbapp_product_type = 'free';
				$is_edit            = false;

				include bbapp()->plugin_dir . '/views/iap/admin/product/edit.php';
				break;
			case 'edit':
				$id = isset( $_REQUEST['id'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['id'] ) ) : false; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

				if ( $id ) {
					// Used in template file.
					$iap_integrations  = Controller::instance()->get_integrations();
					$product           = $this->get_iap_product( $id );
					$misc_settings     = maybe_unserialize( $product->misc_settings );
					$bbapp_iap_link_to = maybe_unserialize( $product->integration_data );

					// NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_type or misc_settings.
					if ( bbapp()->is_network_activated() ) {
						$misc_settings     = $misc_settings[ get_current_blog_id() ];
						$bbapp_iap_link_to = $bbapp_iap_link_to[ get_current_blog_id() ];
					}

					$integration_type       = $misc_settings['integration_type'];
					$benefits               = $misc_settings['benefits'];
					$is_global_subscription = $misc_settings['global_subscription'];
					$course_access          = ( ! empty( $misc_settings['course_access'] ) ) ? $misc_settings['course_access'] : 0;
					$course_show_on_type    = bbapp_iap_get_meta( $id, 'course_access_type', true );
					$course_show_on         = ( ! empty( $course_show_on_type ) ) ? $course_show_on_type : '';
					$key_type               = $this->bbapp_get_lms_meta_key_type( $course_show_on );
					$lms_selection_type     = bbapp_iap_get_meta( $id, $key_type, false );
					$lms_selection          = ( ! empty( $lms_selection_type ) ) ? $lms_selection_type : '';
					$store_data             = maybe_unserialize( $product->store_data );
					$enabled_iap_types      = $store_data['device_platforms'];
					$bbapp_product_type     = isset( $store_data['bbapp_product_type'] ) && ! empty( $store_data['bbapp_product_type'] ) ? $store_data['bbapp_product_type'] : 'free';
					$store_product_types    = $store_data['store_product_types'];
					$store_product_ids      = $store_data['store_product_ids'];
					$is_edit                = true;

					// LearnDash.
					$course_access  = ( ! empty( $misc_settings['course_access'] ) ) ? $misc_settings['course_access'] : 0;
					$course_show_on = ( ! empty( bbapp_iap_get_meta( $id, 'course_access_type' ) ) ) ? bbapp_iap_get_meta( $id, 'course_access_type' ) : '';
					$key_type       = $this->bbapp_get_lms_meta_key_type( $course_show_on );
					$ld_selection   = ( ! empty( bbapp_iap_get_meta( $id, $key_type, false ) ) ) ? bbapp_iap_get_meta( $id, $key_type, false ) : '';

					include bbapp()->plugin_dir . '/views/iap/admin/product/edit.php';
				}
				break;
			case 'upgrade':
				$upgrade_helper = UpgradeHelper::instance();

				$upgrade_helper::upgrade_products_to_new_schema();
				break;
			default:
				$this->products_list = new ProductList();
				$this->products_list->prepare_items();
				include bbapp()->plugin_dir . '/views/iap/admin/product/lists.php';
				break;
		}
	}

	/**
	 * IAP Screen notice.
	 */
	public function iap_screen_notice() {
		$bulk_counts   = array(
			'deleted'   => isset( $_REQUEST['deleted'] ) ? absint( $_REQUEST['deleted'] ) : 0, //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			'trashed'   => isset( $_REQUEST['trashed'] ) ? absint( $_REQUEST['trashed'] ) : 0, //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			'untrashed' => isset( $_REQUEST['untrashed'] ) ? absint( $_REQUEST['untrashed'] ) : 0, //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		);
		$bulk_messages = array(
			/* translators: %s: Number of IAP Products. */
			'deleted'   => _n( '%s product permanently deleted.', '%s products permanently deleted.', $bulk_counts['deleted'], 'buddyboss-app' ),
			/* translators: %s: Number of pages. */
			'trashed'   => _n( '%s product moved to the Trash.', '%s products moved to the Trash.', $bulk_counts['trashed'], 'buddyboss-app' ),
			/* translators: %s: Number of pages. */
			'untrashed' => _n( '%s product restored from the Trash.', '%s products restored from the Trash.', $bulk_counts['untrashed'], 'buddyboss-app' ),
		);
		$bulk_counts   = array_filter( $bulk_counts );
		$messages      = array();

		foreach ( $bulk_counts as $message => $count ) {
			if ( isset( $bulk_messages[ $message ] ) ) {
				$messages[] = sprintf( $bulk_messages[ $message ], number_format_i18n( $count ) );
			}

			$ids          = ( ! empty( $_REQUEST['ids'] ) ) ? array_map( 'absint', $_REQUEST['ids'] ) : array(); //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$page_request = ( ! empty( $_REQUEST['page'] ) ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( 'trashed' === $message && isset( $ids ) ) {
				$base_url   = bbapp_get_admin_url( 'admin.php?page=' . esc_attr( $page_request ) . '&amp;setting=products&amp;iap_product_ids=' . implode( ',', $ids ) );
				$messages[] = '<a href="' . esc_url( wp_nonce_url( $base_url . '&amp;action=untrash', 'cc_untrash_iap_product' ) ) . '">' . __( 'Undo', 'buddyboss-app' ) . '</a>';
			}

			if ( 'untrashed' === $message && isset( $ids ) ) {
				if ( 1 === count( $ids ) ) {
					$messages[] = sprintf( '<a href="%1$s">%2$s</a>', esc_url( bbapp_get_admin_url( sprintf( 'admin.php?page=%s&setting=products&action=%s&id=%s', $page_request, 'edit', $ids[0] ) ) ), esc_html( 'Edit' ) );
				}
			}
		}

		if ( empty( $messages ) ) {
			if ( isset( $_REQUEST['created'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$messages[] = sprintf( __( 'The product has been created successfully.', 'buddyboss-app' ) );
			}

			if ( isset( $_REQUEST['updated'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$messages[] = sprintf( __( 'The product has been updated successfully.', 'buddyboss-app' ) );
			}
		}

		if ( $messages ) {
			echo sprintf( '<div id="message" class="updated notice is-dismissible"><p>%s</p></div>', wp_kses_post( implode( ' ', $messages ) ) );
		}
	}

	/**
	 * Get IAP product from DB
	 *
	 * @param int $id IAP product id.
	 *
	 * @return object Single product
	 */
	public function get_iap_product( $id ) {
		global $wpdb;

		$global_prefix = \bbapp_iap()->get_global_dbprefix();

		return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$global_prefix}bbapp_iap_products WHERE id=%d", $id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Get IAP product status.
	 *
	 * @param int $id IAP product id.
	 *
	 * @since 1.5.2.1
	 * @return null|string
	 */
	public function get_iap_product_status( $id ) {
		global $wpdb;

		$global_prefix = \bbapp_iap()->get_global_dbprefix();

		return $wpdb->get_var( $wpdb->prepare( "SELECT status FROM {$global_prefix}bbapp_iap_products WHERE id=%d", $id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Delete iap product.
	 *
	 * @param int $iap_product_id IAP product id.
	 *
	 * @return bool|int
	 */
	public function delete_iap_product( $iap_product_id ) {
		global $wpdb;

		$iap_product = $this->get_iap_product( $iap_product_id );

		if ( empty( $iap_product ) ) {
			return false;
		}

		$table_name = \bbapp_iap()->get_global_dbprefix() . 'bbapp_iap_products';
		$iap_delete = $wpdb->delete( $table_name, array( 'id' => $iap_product_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

		if ( $iap_delete ) {
			bbapp_iap_delete_meta( $iap_product_id );
		}

		return $iap_delete;
	}

	/**
	 * Iap product Update status.
	 *
	 * @param int    $iap_product_id IAP product id.
	 * @param string $status         Status to update.
	 *
	 * @return bool|int
	 */
	public function update_iap_product_status( $iap_product_id, $status = 'published' ) {
		global $wpdb;

		$iap_product = $this->get_iap_product( $iap_product_id );

		if ( empty( $iap_product ) ) {
			return false;
		}

		$table_name = \bbapp_iap()->get_global_dbprefix() . 'bbapp_iap_products';

		return $wpdb->update( $table_name, array( 'status' => $status ), array( 'id' => $iap_product_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	}

	/**
	 * Get IAP Products Ids by group from DB.
	 *
	 * @param int    $group_id Group ID.
	 * @param string $column   Column name to fetch.
	 *
	 * @return array
	 */
	public static function get_group_products_ids( $group_id, $column = 'id' ) {
		global $wpdb;

		$global_prefix = \bbapp_iap()->get_global_dbprefix();

		if ( 'all' === $column ) {
			return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$global_prefix}bbapp_iap_products WHERE iap_group=%d", $group_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		} else {
			return $wpdb->get_col( $wpdb->prepare( "SELECT {$column} FROM {$global_prefix}bbapp_iap_products WHERE iap_group=%d", $group_id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
	}

	/**
	 * Update IAP Product order.
	 */
	public function bbapp_update_iap_products_ui_sorting() {
		global $wpdb, $userdata;

		set_time_limit( 600 );

		$screen = ( ! empty( $_POST['screen'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['screen'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$paged  = filter_var( wp_unslash( $_POST['paged'] ), FILTER_SANITIZE_NUMBER_INT ); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated
		$nonce  = ( ! empty( $_POST['sort_nonce'] ) ) ? wp_unslash( $_POST['sort_nonce'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		// verify the nonce.
		if ( ! wp_verify_nonce( $nonce, 'bbapp_sort_nonce_' . $userdata->ID ) ) {
			die();
		}

		parse_str( bbapp_input_clean( $_POST['order'] ), $data ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.InputNotValidated, WordPress.Security.ValidatedSanitizedInput.MissingUnslash

		if ( ! is_array( $data ) || count( $data ) < 1 ) {
			die();
		}

		// retrieve a list of all objects.
		$table_name = \bbapp_iap()->get_global_dbprefix() . 'bbapp_iap_products';
		$results    = $wpdb->get_results( "SELECT ID FROM {$table_name} WHERE status IN ('published') ORDER BY menu_order, date_created DESC" ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( ! is_array( $results ) || count( $results ) < 1 ) {
			die();
		}

		// Create the list of ID's.
		$objects_ids = array();

		foreach ( $results as $result ) {
			$objects_ids[] = (int) $result->ID;
		}

		$objects_per_page = get_user_meta( $userdata->ID, $screen . '_per_page', true );
		if ( empty( $objects_per_page ) ) {
			$objects_per_page = 20; // default pagination as screen option.
		}

		$edit_start_at = $paged * $objects_per_page - $objects_per_page;
		$index         = 0;

		for ( $i = $edit_start_at; $i < ( $edit_start_at + $objects_per_page ); $i++ ) {
			if ( ! isset( $objects_ids[ $i ] ) || ! isset( $data['iap-product'][ $index ] ) ) {
				break;
			}

			$objects_ids[ $i ] = (int) $data['iap-product'][ $index ];

			++$index;
		}

		// Update the menu_order within database.
		foreach ( $objects_ids as $menu_order => $id ) {
			$data = array(
				'menu_order' => $menu_order,
			);

			$wpdb->update( $table_name, $data, array( 'id' => $id ) ); //phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		}

		/**
		 * Triggers after IAP product sorting order update.
		 *
		 * @param array $objects_ids IAP product ids.
		 *
		 * @since [BBAPPVERSSION]
		 */
		do_action( 'bbapp_after_update_iap_products_ui_sorting', $objects_ids );
		wp_die();
	}

	/**
	 * Function to get the LearnDash dropdown based on the show on selection.
	 *
	 * @since 1.4.0
	 */
	public function bbapp_get_lms_dropdown() {
		$result         = array();
		$result['html'] = '';
		$nonce          = ( ! empty( $_POST['lms_nonce'] ) ) ? wp_unslash( $_POST['lms_nonce'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'lms-dropdown-nonce' ) || ! current_user_can( 'manage_options' ) ) {
			$result['message'] = '<p class="bbapp-notice-box bbapp-notice-box--error">' . esc_html__( 'Security check failed', 'buddyboss-app' ) . '</p>';
			wp_send_json_error( $result );
		}

		$show_on = ( ! empty( $_POST['show_on'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['show_on'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( empty( $show_on ) ) {
			$result['message'] = '<p class="bbapp-notice-box bbapp-notice-box--error">' . esc_html__( 'Please select option to show the LearnDash courses', 'buddyboss-app' ) . '</p>';
			wp_send_json_error( $result );
		}

		$result['message'] = esc_html__( 'Results found', 'buddyboss-app' );
		$result['html']    = $this->bbapp_get_lms_dropdown_html( $show_on );

		wp_send_json_success( $result );
	}


	/**
	 * Function to get the show on dropdown results.
	 *
	 * @param string $selection  Show on section.
	 * @param array  $pre_filled array for save items.
	 *
	 * @since 1.4.0
	 *
	 * @return string
	 */
	public function bbapp_get_lms_dropdown_html( $selection, $pre_filled = array() ) {
		$pre_filled = ! empty( $pre_filled ) ? array_map( 'intval', $pre_filled ) : array();

		$lms_dropdown_data = $this->get_lms_dropdown_data( $selection );
		$selection_lists   = $lms_dropdown_data['selection_lists'];
		$place_holder      = $lms_dropdown_data['place_holder'];
		$btn_label         = $lms_dropdown_data['btn_label'];

		$dropdown_html = '';
		$unique        = 1;

		ob_start();
		?>
		<div class="bbapp-selection-main-wrapper">
			<div class="bbapp-selection-inr-wrapper">
				<?php
				if ( ! empty( $pre_filled ) ) {
					foreach ( $pre_filled as $single_item ) {
						?>
						<div class="bbapp-selection-cmn">
							<select id="bbapp-show-on-selection-<?php echo esc_attr( $unique ); ?>" class="bbapp-show-on-selection"
									name="lms_courses_selection[]"
									data-placeholder="<?php echo esc_html( $place_holder ); ?>"
									data-select-key="<?php echo esc_attr( $unique ); ?>" style="width:100%">
								<?php
								if ( ! empty( $selection_lists ) ) {
									foreach ( $selection_lists as $key => $selection_list ) {
										?>
										<option value="<?php echo esc_attr( $key ); ?>" <?php echo ( $single_item === $key ) ? 'selected=selected' : ''; ?>><?php echo esc_html( $selection_list ); ?></option>
										<?php
									}
								}
								?>
							</select>
							<?php
							if ( 1 !== $unique ) {
								?>
								<a href="javascript:void(0);" class="bbapp-remove-clone">
									<i class="dashicons dashicons-no-alt"></i>
								</a>
								<?php
							}
							?>
						</div>
						<?php
						$unique ++;
					}
				} else {
					?>
					<div class="bbapp-selection-cmn">
						<select id="bbapp-show-on-selection-<?php echo esc_attr( $unique ); ?>" class="bbapp-show-on-selection"
								name="lms_courses_selection[]" data-placeholder="<?php echo esc_html( $place_holder ); ?>"
								data-select-key="<?php echo esc_attr( $unique ); ?>" style="width:100%">
							<option value=""><?php echo esc_html( $place_holder ); ?></option>
							<?php
							if ( ! empty( $selection_lists ) ) {
								foreach ( $selection_lists as $key => $selection_list ) {
									?>
									<option value="<?php echo esc_attr( $key ); ?>"><?php echo esc_html( $selection_list ); ?></option>
									<?php
								}
							}
							?>
						</select>
					</div>
					<?php
				}
				?>
			</div>
			<a href="javascript:void(0);" class="button bbapp-add-selection">
				<?php
				echo esc_html( '+ ' . $btn_label );
				?>
			</a>
		</div>
		<?php
		$dropdown_html .= ob_get_clean();

		return $dropdown_html;
	}

	/**
	 * Function to get the LMS dropdown based on the show on selection.
	 *
	 * @param string $selection LMS show on selection.
	 *
	 * @since [BBAPPVERSSION]
	 *
	 * @return array
	 */
	public function get_lms_dropdown_data( $selection ) {
		$selection_lists = array();
		$place_holder    = '';
		$btn_label       = '';

		if ( bbapp_is_memberpress_lms_enabled() ) {
			if ( 'specific-category-courses' === $selection ) {
				$selection_lists = get_terms(
					'mpcs-course-categories',
					array(
						'hide_empty' => false,
						'fields'     => 'id=>name',
					)
				);
				$place_holder    = esc_html__( 'Please select Categories', 'buddyboss-app' );
				$btn_label       = esc_html__( 'Add Category', 'buddyboss-app' );
			} elseif ( 'specific-tag-courses' === $selection ) {
				$selection_lists = get_terms(
					'mpcs-course-tags',
					array(
						'hide_empty' => false,
						'fields'     => 'id=>name',
					)
				);
				$place_holder    = esc_html__( 'Please select Tags', 'buddyboss-app' );
				$btn_label       = esc_html__( 'Add Tag', 'buddyboss-app' );
			} else {
				$lms_courses  = new \WP_Query(
					array(
						'post_type'       => 'mpcs-course',
						'posts_per_page ' => - 1,
						'nopaging'        => true,
					)
				);
				$place_holder = esc_html__( 'Please select Courses', 'buddyboss-app' );
				$btn_label    = esc_html__( 'Add Course', 'buddyboss-app' );

				if ( ! empty( $lms_courses->posts ) ) {
					$selection_lists = array();
					foreach ( $lms_courses->posts as $ld_course ) {
						$selection_lists[ $ld_course->ID ] = $ld_course->post_title;
					}
				}
			}
		}

		if ( bbapp_is_learndash_enabled() ) {
			if ( 'specific-category-courses' === $selection ) {
				$selection_lists = get_terms(
					'ld_course_category',
					array(
						'hide_empty' => false,
						'fields'     => 'id=>name',
					)
				);
				$place_holder    = esc_html__( 'Please select Categories', 'buddyboss-app' );
				$btn_label       = esc_html__( 'Add Category', 'buddyboss-app' );
			} elseif ( 'specific-tag-courses' === $selection ) {
				$selection_lists = get_terms(
					'ld_course_tag',
					array(
						'hide_empty' => false,
						'fields'     => 'id=>name',
					)
				);
				$place_holder    = esc_html__( 'Please select Tags', 'buddyboss-app' );
				$btn_label       = esc_html__( 'Add Tag', 'buddyboss-app' );
			} else {
				$lms_courses  = new \WP_Query(
					array(
						'post_type'       => 'sfwd-courses',
						'posts_per_page ' => - 1,
						'nopaging'        => true,
					)
				);
				$place_holder = esc_html__( 'Please select Courses', 'buddyboss-app' );
				$btn_label    = esc_html__( 'Add Course', 'buddyboss-app' );

				if ( ! empty( $lms_courses->posts ) ) {
					$selection_lists = array();
					foreach ( $lms_courses->posts as $ld_course ) {
						$selection_lists[ $ld_course->ID ] = $ld_course->post_title;
					}
				}
			}
		}

		if ( bbapp_is_tutor_lms_enabled() ) {
			if ( 'specific-category-courses' === $selection ) {
				$selection_lists = get_terms(
					array(
						'taxonomy'   => 'course-category',
						'hide_empty' => false,
						'fields'     => 'id=>name',
					)
				);
				$place_holder    = esc_html__( 'Please select Categories', 'buddyboss-app' );
				$btn_label       = esc_html__( 'Add Category', 'buddyboss-app' );
			} elseif ( 'specific-tag-courses' === $selection ) {
				$selection_lists = get_terms(
					array(
						'taxonomy'   => 'course-tag',
						'hide_empty' => false,
						'fields'     => 'id=>name',
					)
				);
				$place_holder    = esc_html__( 'Please select Tags', 'buddyboss-app' );
				$btn_label       = esc_html__( 'Add Tag', 'buddyboss-app' );
			} else {
				$tutor_courses = new \WP_Query(
					array(
						'post_type'       => 'courses',
						'posts_per_page ' => - 1,
						'nopaging'        => true,
					)
				);
				$place_holder  = esc_html__( 'Please select Courses', 'buddyboss-app' );
				$btn_label     = esc_html__( 'Add Course', 'buddyboss-app' );

				if ( ! empty( $tutor_courses->posts ) ) {
					$selection_lists = array();
					foreach ( $tutor_courses->posts as $ld_course ) {
						$selection_lists[ $ld_course->ID ] = $ld_course->post_title;
					}
				}
			}
		}

		$data['selection_lists'] = $selection_lists;
		$data['place_holder']    = $place_holder;
		$data['btn_label']       = $btn_label;

		return $data;
	}

	/**
	 * Register the screen options.
	 */
	public function get_screen_options() {
		$screen = get_current_screen();

		if ( ! is_object( $screen ) || 'buddyboss-app_page_bbapp-iap' !== $screen->id ) {
			return;
		}

		$setting_get = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'products' !== $setting_get ) {
			return;
		}

		// Loop through all the options and add a screen option for each.
		add_screen_option(
			'per_page',
			array(
				'label'   => __( 'Number of items per page:', 'buddyboss-app' ),
				'default' => 20,
				'option'  => 'iap_product_per_page',
			)
		);
	}

	/**
	 * Handle save/update of screen options for iap product.
	 *
	 * @param string $value     Will always be false unless another plugin filters it first.
	 * @param string $option    Screen option name.
	 * @param string $new_value Screen option form value.
	 *
	 * @return string|int Option value. False to abandon update.
	 */
	public function bbapp_admin_screen_options( $value, $option, $new_value ) {
		if ( 'iap_product_per_page' !== $option ) {
			return $value;
		}

		// Per page.
		$new_value = (int) $new_value;

		if ( $new_value < 1 || $new_value > 999 ) {
			return $value;
		}

		return $new_value;
	}

	/**
	 * Function will return image name and hash.
	 *
	 * @param array  $files_data Files data.
	 * @param string $field_name Name of file.
	 * @param int    $product_id Product ID.
	 *
	 * @since 1.4.4
	 *
	 * @return array $file_datas
	 */
	public function bbapp_iap_uploaded_feature_image( $files_data, $field_name, $product_id ) {
		$file_hash    = '';
		$file_name    = '';
		$out_location = get_iap_upload_dir();
		$file_data    = $files_data[ $field_name ];
		$file_data    = BBAPP_File::read_file( $file_data['tmp_name'] );
		if ( ! empty( $file_data ) ) {
			$file_hash = time();
			$im        = function_exists( 'imagecreatefromstring' ) ? imagecreatefromstring( $file_data ) : '';

			if ( $im ) {
				$width             = imagesx( $im ); // Actual width of image.
				$height            = imagesy( $im ); // Actual height of image.
				$explode_file_name = explode( '.', $files_data[ $field_name ]['name'] );
				$extension         = end( $explode_file_name );
				$file_name         = $product_id . '.' . $extension; // New image name as product id.

				// Save the image.
				if ( 'image/png' === $files_data[ $field_name ]['type'] ) {
					$new = imagecreatetruecolor( $width, $height );

					imagecolortransparent( $new, imagecolorallocatealpha( $new, 0, 0, 0, 127 ) );
					imagealphablending( $new, false );
					imagesavealpha( $new, true );
					imagecopyresampled( $new, $im, 0, 0, 0, 0, $width, $height, $width, $height );

					$file_path = $out_location . '/' . $file_name;

					if ( file_exists( $file_path ) ) {
						wp_delete_file( $file_path );
					}

					imagepng( $new, $file_path );
				}

				if ( 'image/jpg' === $files_data[ $field_name ]['type'] || 'image/jpeg' === $files_data[ $field_name ]['type'] ) {
					// in-case image source is png and background is transparent turn it into white.
					$new   = imagecreatetruecolor( $width, $height );
					$white = imagecolorallocate( $new, 255, 255, 255 );

					imagefilledrectangle( $new, 0, 0, $width, $height, $white );
					imagecopy( $new, $im, 0, 0, 0, 0, $width, $height );

					$file_path = $out_location . '/' . $file_name;

					if ( file_exists( $file_path ) ) {
						wp_delete_file( $file_path );
					}

					imagejpeg( $new, $file_path );
				}

				imagedestroy( $im );
			}
		}
		$file_datas = array();
		if ( ! empty( $file_name ) && ! empty( $file_hash ) ) {
			$file_datas = array( $file_name => $file_hash );
		}

		return $file_datas;
	}

	/**
	 * Job to verify the IAP products configurations. If required information is not found inside any specific product, change the product status to disable. and add a notice to
	 * the site admin.
	 *
	 * @param array|object $task Job task.
	 *
	 * @since 1.6.7
	 */
	public function verify_miss_configured_bg_job( $task ) {
		$task_data          = maybe_unserialize( $task->data );
		$current_page       = $task_data['current_page'];
		$is_miss_configured = ( 1 === (int) $current_page ) ? false : get_option( 'bbapp_show_missconfigured_iap_product_notice', false );
		$results            = $this->get_all_products( $current_page );
		$is_secondary_site  = bbapp_is_connected_as_secondary_site();
		if ( ! empty( $results ) ) {
			foreach ( $results as $result ) {
				$store_data = maybe_unserialize( $result->store_data );
				if ( 'paid' === $store_data['bbapp_product_type'] ) { // Check if it's paid product.
					if ( true !== $is_secondary_site ) { // Check if it's not secondary site connected.
						if ( ( isset( $store_data['device_platforms']['ios'] ) && empty( $store_data['store_product_ids']['ios'] ) ) || ( isset( $store_data['device_platforms']['android'] ) && empty( $store_data['store_product_ids']['android'] ) ) ) {
							$this->update_iap_product_status( $result->id, 'disabled' );
							$is_miss_configured = true;
						}
					} else {
						// If secondary site connected and configuration is correct then product should be published.
						if ( isset( $store_data['device_platforms']['ios'] ) ) {
							$this->update_iap_product_status( $result->id );
						}
					}
				}
			}
			$current_page ++;
			Helpers::instance()->bbapp_start_iap_product_job( $current_page );
			update_option( 'bbapp_show_missconfigured_iap_product_notice', $is_miss_configured );
		}
	}

	/**
	 * Get all product for process the job.
	 *
	 * @param int $page     Page number.
	 * @param int $per_page Item per page.
	 *
	 * @since 1.6.7
	 *
	 * @return null|array|object|\stdClass[]
	 */
	public function get_all_products( $page = 1, $per_page = 10 ) {
		global $wpdb;
		$offset        = ( $page - 1 ) * $per_page;
		$global_prefix = \bbapp_iap()->get_global_dbprefix();

		return $wpdb->get_results( "SELECT * FROM {$global_prefix}bbapp_iap_products AS iap WHERE iap.status != 'trashed'  ORDER BY iap.menu_order ASC LIMIT {$offset}, {$per_page}" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Run the job when item move to trashed.
	 *
	 * @param int|array $iap_product_ids IAP product ids.
	 *
	 * @since 1.6.7
	 */
	public function iap_product_run_job( $iap_product_ids ) {
		Helpers::instance()->bbapp_start_iap_product_job();
	}

	/**
	 * Run the job when item move to trashed.
	 *
	 * @param int|string   $product_id IAP product ids.
	 * @param array|object $data       IAP product data.
	 *
	 * @since 1.6.7
	 */
	public function iap_save_product_run_job( $product_id, $data ) {
		Helpers::instance()->bbapp_start_iap_product_job();
	}
}
