<?php
/**
 * Order list table.
 *
 * @package BuddyBossApp\Admin\InAppPurchases
 */

namespace BuddyBossApp\Admin\InAppPurchases;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

use BuddyBossApp\InAppPurchases\Orders;

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Order list table class.
 */
class OrderList extends \WP_List_Table {

	/**
	 * The type of view currently being displayed.
	 *
	 * E.g. "All", "Pending", "Approved", "Spam"...
	 *
	 * @since BuddyPress 1.7.0
	 * @var string
	 */
	public $view = 'all';

	/**
	 * Order counts for each order type.
	 *
	 * @since BuddyPress 1.7.0
	 */
	public $iap_purchases_counts;

	/** Class constructor */
	public function __construct() {

		parent::__construct(
			array(
				'screen'   => 'order-list',
				'singular' => __( 'IAP Order', 'buddyboss-app' ), // singular name of the listed records.
				'plural'   => __( 'IAP Orders', 'buddyboss-app' ), // plural name of the listed records.
				'ajax'     => false, // should this table support ajax?
			)
		);
	}

	/**
	 * Get orders from database.
	 *
	 * @param int    $per_page Per page.
	 * @param int    $page_number Limit.
	 * @param string $view View.
	 * @param int    $iap_product Product id.
	 *
	 * @return mixed
	 */
	public static function get_orders( $per_page = 5, $page_number = 1, $view = '', $iap_product = 0, $filter_by = array() ) {

		$args = array();
		if ( isset( $_REQUEST['orderby'] ) && ! empty( $_REQUEST['orderby'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$args['orderby'] = bbapp_input_clean( wp_unslash( $_REQUEST['orderby'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}
		if ( isset( $_REQUEST['order'] ) && ! empty( $_REQUEST['order'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$args['order'] = bbapp_input_clean( wp_unslash( $_REQUEST['order'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}

		$args['per_page']     = $per_page;
		$args['current_page'] = $page_number;

		$where_args = array();

		// NOTE : For multi-site, we are displaying order of it's own site.
		if ( bbapp()->is_network_activated() ) {
			$where_args['blog_id'] = get_current_blog_id();
		}

		if ( ! empty( $view ) && 'all' !== $view ) {
			$where_args['order_status'] = $view;
		}

		if ( ! empty( $iap_product ) ) {
			$where_args['bbapp_product_id'] = $iap_product;
		}

        if( ! empty( $filter_by['search_by_email'] ) ) {
            $where_args['user_email'] = $filter_by['search_by_email'];
        }

		$results = Orders::instance()->get_orders( $where_args, $args );

		if ( ! is_array( $results ) ) {
			$results = array();
		}

		// Convert to array.
		foreach ( $results as $k => $v ) {
			$results[ $k ] = (array) $v;
		}

		return $results;

	}

	/**
	 * Returns the count of records in the database.
	 *
	 * @param int    $iap_product  IAP product.
	 * @param string $order_status Order status.
	 *
	 * @return null|string
	 */
	public static function record_count( $iap_product, $order_status = '' ) {
		return Orders::instance()->get_total_orders_count( $order_status, $iap_product );
	}

	/**
	 * Return the column available to this table list.
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns = array(
			'cb'               => '<input type="checkbox" />',
			'id'               => __( 'Purchase ID', 'buddyboss-app' ),
			'user'             => __( 'Customer', 'buddyboss-app' ),
			'status'           => __( 'Status', 'buddyboss-app' ),
			'device_platform'  => __( 'Platform ', 'buddyboss-app' ),
			'iap_product'      => __( 'Product ', 'buddyboss-app' ),
			'integration'      => __( 'Integration', 'buddyboss-app' ),
			'date_created'     => __( 'Order Created', 'buddyboss-app' ),
			'transaction_date' => __( 'Transaction Date', 'buddyboss-app' ),
		);
		if ( bbapp_is_connected_as_secondary_site() ) {
			unset( $columns['iap_product'] );
		}

		return $columns;

	}

	/**
	 * No items found message.
	 */
	public function no_items() {
		esc_html_e( 'No purchases found.', 'buddyboss-app' );
	}

	/**
	 * Default columns
	 *
	 * @param object $item        Column item.
	 * @param string $column_name Column name.
	 *
	 * @return string|void
	 */
	public function column_default( $item, $column_name ) {

		$item_ids = maybe_unserialize( $item['item_ids'] ); // db keys.
		$items    = Helpers::parse_item_ids( $item_ids );

		// NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_types or item_ids.
		if ( bbapp()->is_network_activated() ) {
			$item_ids = $item_ids[ get_current_blog_id() ];
			$items    = Helpers::parse_item_ids( $item_ids );
		}

		$items_readable = implode( '<br/>', $items );

		switch ( $column_name ) {
			case 'id':
				echo esc_html( $item['id'] );
				break;
			case 'user':
				echo wp_kses_post( OrderHelper::instance()->bbapp_get_iap_user_url( $item['user_id'], $item['user_email'] ) );
				break;
			case 'status':
				echo sprintf( '<div class="order-status %1$s"><span>%2$s</span></div>', esc_attr( strtolower( $item['order_status'] ) ), esc_html( ucfirst( $item['order_status'] ) ) );
				break;
			case 'device_platform':
				$iap = bbapp_iap()->iap[ $item['device_platform'] ];
				echo esc_html( $iap->get_label() );
				break;
			case 'iap_product':
				$iap_product = bbapp_iap_get_product( $item['bbapp_product_id'] );
				if ( ! empty( $iap_product ) ) {
					echo sprintf( '<a href="%1$s" target="_blank">%2$s</a>', esc_url( bbapp_get_admin_url( "admin.php?page=bbapp-iap&setting=products&action=edit&id={$item['bbapp_product_id']}" ) ), esc_html( $iap_product['name'] ) );
				} else {
					echo esc_html__( 'Deleted', 'buddyboss-app' );
				}
				break;
			case 'integration':
				echo wp_kses_post( $items_readable );
				break;
			case 'date_created':
				$date_format = get_option( 'date_format' );
				$date_format = ! empty( $date_format ) ? $date_format : __( 'F j, Y', 'buddyboss-app' );
				$time_format = get_option( 'time_format' );
				$time_format = ! empty( $time_format ) ? $time_format : __( 'g:i a', 'buddyboss-app' );

				echo sprintf(
					'%1$s %2$s %3$s',
					esc_html( get_date_from_gmt( $item['date_created'], $date_format ) ),
					esc_html__( 'at', 'buddyboss-app' ),
					esc_html( get_date_from_gmt( $item['date_created'], $time_format ) )
				);
				break;
			case 'transaction_date':
				$date_format      = get_option( 'date_format' );
				$date_format      = ! empty( $date_format ) ? $date_format : __( 'F j, Y', 'buddyboss-app' );
				$time_format      = get_option( 'time_format' );
				$time_format      = ! empty( $time_format ) ? $time_format : __( 'g:i a', 'buddyboss-app' );
				$transaction_date = ( 'ios' === $item['device_platform'] ) ? OrderHelper::instance()->bbapp_trim_ios_date( Orders::instance()->get_meta( $item['id'], '_transaction_date' ) ) : Orders::instance()->get_meta( $item['id'], '_transaction_date' );
				if ( ! empty( $transaction_date ) ) {
					echo sprintf(
						'%1$s %2$s %3$s',
						esc_html( get_date_from_gmt( $transaction_date, $date_format ) ),
						esc_html__( 'at', 'buddyboss-app' ),
						esc_html( get_date_from_gmt( $transaction_date, $time_format ) )
					);
				} else {
					echo 'N/A';
				}
				break;
			default:
				echo 'N/A';
				break;
		}

	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 */
	public function prepare_items() {

		/**
		 * Init column headers.
		 */
		$this->_column_headers = array( $this->get_columns(), array(), $this->get_sortable_columns() );

		/** Process bulk action */
		$this->process_bulk_action();

		$order_status = isset( $_GET['order_status'] ) ? sanitize_text_field( wp_unslash( $_GET['order_status'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$search_by_email = isset( $_GET['search_by_email'] ) ? sanitize_email( wp_unslash( $_GET['search_by_email'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		// Set the current view.
		if ( in_array(
			$order_status,
			array(
				'subscribed',
				'completed',
				'pending',
				'cancelled',
				'failed',
			),
			true
		) ) {
			$this->view = $order_status;
		}

		$iap_product = $this->bbapp_requested_purchased_product_id();

		$per_page     = $this->get_items_per_page( 'iap_order_per_page', 10 );
		$current_page = $this->get_pagenum();
		$total_items  = self::get_orders_type_ids();
		// Get order type counts for display in the filter tabs.
		$this->iap_purchases_counts = array();
		foreach ( $total_items as $view => $total_item ) {
			$this->iap_purchases_counts[ $view ] = $total_item;
		}

		$this->items = self::get_orders( $per_page, $current_page, $this->view, $iap_product, array( 'search_by_email' => $search_by_email ) );
		if ( ! empty( $iap_product ) ) {
			$current_view = 'all' === $this->view ? '' : $this->view; // If status is all then pass blank for query.
			$total_count  = self::record_count( $iap_product, $current_view );
		} else {
			$total_count = $this->iap_purchases_counts[ $this->view ];
		}
		$this->set_pagination_args(
			array(
				'total_items' => $total_count, // WE have to calculate the total number of items.
				'per_page'    => $per_page, // WE have to determine how many items to show on a page.
			)
		);
	}

	/**
	 * Get an array containing ids for each order type.
	 *
	 * A bit of a kludge workaround for some issues
	 *
	 * @param int|string $iap_product IAP product.
	 *
	 * @return array
	 * @since BuddyPress 1.7.0
	 */
	public static function get_orders_type_ids( $iap_product = '' ) {

		$ids = array();

		$ids['all']        = self::record_count( $iap_product, '' );
		$ids['subscribed'] = self::record_count( $iap_product, 'subscribed' );
		$ids['completed']  = self::record_count( $iap_product, 'completed' );
		$ids['pending']    = self::record_count( $iap_product, 'pending' );
		$ids['cancelled']  = self::record_count( $iap_product, 'cancelled' );
		$ids['failed']     = self::record_count( $iap_product, 'failed' );

		return $ids;
	}

	/**
	 * Checkbox for bulk items.
	 *
	 * @param object|array $item Order list item.
	 *
	 * @return string|void
	 */
	public function column_cb( $item ) {
		if ( in_array(
			$item['order_status'],
			array(
				'subscribed',
				'completed',
			),
			true
		) && current_user_can( 'manage_options' ) ) {

			return sprintf(
				'<input type="checkbox" name="iap_purchases_ids[]" value="%s" />',
				$item['id']
			);
		}

		return '';
	}

	/**
	 * Bulk action process.
	 *
	 * @return bool
	 */
	public function process_bulk_action() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( "You don't have permission to access this page.", 'buddyboss-app' );
		}
		// Build redirection URL.
		$redirect_to = remove_query_arg(
			array(
				'action',
				'action2',
				'cc_revoke_iap_purchases',
				'revoked',
				'revoke_order',
				'iap_purchases_ids',
				'error',
				'updated',
				'success_new',
				'error_new',
				'success_modified',
				'_wpnonce',
				'error_modified',
			),
			bbapp_input_clean( wp_unslash( $_SERVER['REQUEST_URI'] ) ) // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.InputNotValidated
		);

		$action = $this->current_action();

		if ( 'do_revoke' === $action && ! empty( $_POST['revoke_iap_purchases'] ) ) {
			check_admin_referer( 'cc_revoke_iap_purchases' );
			$iap_purchases_ids = wp_parse_id_list( wp_unslash( $_POST['revoke_iap_purchases'] ) );

			$count = 0;
			foreach ( $iap_purchases_ids as $iap_purchases_id ) {

				$current_user_id = get_current_user_id();
				$success         = Orders::instance()->cancel_order( $iap_purchases_id, esc_html__( 'Order has been cancelled manually by user #', 'buddyboss-app' ) . esc_html( $current_user_id ) . '.' );

				if ( is_wp_error( $success ) ) {
					wp_die( wp_kses_post( $success->get_error_message() ) );
				} else {
					// Store cancel detail on meta.
					Orders::instance()->update_meta( $iap_purchases_id, 'order_cancelled_manually', 1 );
					Orders::instance()->update_meta( $iap_purchases_id, 'order_cancelled_user_id', get_current_user_id() );
					$count ++;
				}
			}
			$redirect_to = add_query_arg( 'revoked', $count, $redirect_to );
			wp_safe_redirect( $redirect_to );
		} elseif ( ! empty( $_GET['_wp_http_referer'] ) ) {
			wp_redirect( remove_query_arg( array( '_wp_http_referer', '_wpnonce' ), stripslashes( $_SERVER['REQUEST_URI'] ) ) );
			exit;
		}

		return true;
	}

	/**
	 * Bulk action.
	 *
	 * @return array
	 */
	public function get_bulk_actions() {
		return array(
			'bulk-revoke-order' => __( 'Revoke Purchase', 'buddyboss-app' ),
		);
	}

	/**
	 * Get the list of views available on this table (e.g. "all", "public").
	 */
	public function get_views() {
		$base_url = add_query_arg( array(
			'page'    => 'bbapp-iap',
			'setting' => 'orders'
		), admin_url( 'admin.php' ) );
		$iap_product_id = $this->bbapp_requested_purchased_product_id();

		$search_by_email = isset( $_GET['search_by_email'] ) ? sanitize_email( wp_unslash( $_GET['search_by_email'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

        if ( ! empty( $search_by_email ) ) {
			$base_url = add_query_arg( array( 'search_by_email' => $search_by_email ), $base_url );
		}
		if ( ! empty( $iap_product_id ) ) {
			$base_url = add_query_arg( array( 'iap_product_id' => $iap_product_id ), $base_url );
		}
		?>

		<h2 class="screen-reader-text">
			<?php
			/* translators: accessibility text */
			esc_html_e( 'Filter IAP Orders list', 'buddyboss-app' );
			?>
		</h2>

		<?php
		$iap_tabs['all'] = array(
			'url'   => remove_query_arg( array( 'order_status', 'iap_product_id' ), $base_url ),
			'class' => ( 'all' === $this->view ) ? 'current' : '',
			'count' => sprintf( '%1$s <span class="count">(%2$s)</span>', esc_html__( 'All', 'buddyboss-app' ), number_format_i18n( $this->iap_purchases_counts['all'] ) ),
		);
		if ( ! empty( $this->iap_purchases_counts['subscribed'] ) ) {
			$iap_tabs['subscribed'] = array(
				'url' => add_query_arg( array( 'order_status' => 'subscribed' ), $base_url ),
				'class' => ( 'subscribed' === $this->view ) ? 'current' : '',
				'count' => sprintf( '%1$s <span class="count">(%2$s)</span>', esc_html__( 'Subscribed', 'buddyboss-app' ), number_format_i18n( $this->iap_purchases_counts['subscribed'] ) ),
			);
		}
		if ( ! empty( $this->iap_purchases_counts['completed'] ) ) {
			$iap_tabs['completed'] = array(
				'url' => add_query_arg( array( 'order_status' => 'completed' ), $base_url ),
				'class' => ( 'completed' === $this->view ) ? 'current' : '',
				'count' => sprintf( '%1$s <span class="count">(%2$s)</span>', esc_html__( 'Completed', 'buddyboss-app' ), number_format_i18n( $this->iap_purchases_counts['completed'] ) ),
			);
		}
		if ( ! empty( $this->iap_purchases_counts['pending'] ) ) {
			$iap_tabs['pending'] = array(
				'url'   => add_query_arg( array( 'order_status' => 'pending' ), $base_url ),
				'class' => ( 'pending' === $this->view ) ? 'current' : '',
				'count' => sprintf( '%1$s <span class="count">(%2$s)</span>', esc_html__( 'Pending', 'buddyboss-app' ), number_format_i18n( $this->iap_purchases_counts['pending'] ) ),
			);
		}
		if ( ! empty( $this->iap_purchases_counts['cancelled'] ) ) {
			$iap_tabs['cancelled'] = array(
				'url'   => add_query_arg( array( 'order_status' => 'cancelled' ), $base_url ),
				'class' => ( 'cancelled' === $this->view ) ? 'current' : '',
				'count' => sprintf( '%1$s <span class="count">(%2$s)</span>', esc_html__( 'Cancelled', 'buddyboss-app' ), number_format_i18n( $this->iap_purchases_counts['cancelled'] ) ),
			);
		}
		if ( ! empty( $this->iap_purchases_counts['failed'] ) ) {
			$iap_tabs['failed'] = array(
				'url'   => add_query_arg( array( 'order_status' => 'failed' ), $base_url ),
				'class' => ( 'failed' === $this->view ) ? 'current' : '',
				'count' => sprintf( '%1$s <span class="count">(%2$s)</span>', esc_html__( 'Failed', 'buddyboss-app' ), number_format_i18n( $this->iap_purchases_counts['failed'] ) ),
			);
		}

		foreach ( $iap_tabs as $key => $iap_tab ) {
			$action           = sprintf( "<a href='%s' class='%s'>%s</a>", esc_url( $iap_tab['url'] ), esc_attr( $iap_tab['class'] ), $iap_tab['count'] );
			$iap_tabs[ $key ] = "\t<li class='$key'>$action";
		}
		?>
		<ul class="subsubsub">
			<?php echo wp_kses_post( implode( " |</li>\n", $iap_tabs ) ) . "</li>\n"; ?>
		</ul>
		<?php
	}


	/**
	 * Column id.
	 *
	 * @param array $item Order item.
	 *
	 * @return string
	 */
	public function column_id( $item ) {

		$request_page = ! empty( $_REQUEST['page'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		$actions = array(
			'view' => sprintf( '<span class="view"><a href="admin.php?page=%s&action=view_order&order_id=%s">View</a></span>', $request_page, $item['id'] ),
		);

		if (
			in_array(
				$item['order_status'],
				array(
					'subscribed',
					'completed',
				),
				true
			)
			&& current_user_can( 'manage_options' )
		) {

			$base_url     = bbapp_get_admin_url( 'admin.php?page=' . esc_attr( $request_page ) . '&amp;iap_purchases_ids=' . $item['id'] );
			$complete_url = wp_nonce_url( $base_url . '&amp;action=revoke_order', 'cc_revoke_iap_purchases' );

			$actions['cancel'] = sprintf( '<span class="trash"><a href="%s">Revoke</a></span>', $complete_url );
		}
		$test_mode_label = '';

		if ( true === (bool) Orders::instance()->get_meta( $item['id'], '_transaction_data_test_mode' ) ) {
			$test_mode_label = __( ' (Test App)', 'buddyboss-app' );
		}

		return sprintf( '<a class="row-title" href="admin.php?page=%1$s&action=view_order&order_id=%2$s">#%2$s%3$s %4$s', esc_attr( $request_page ), esc_html( $item['id'] ), esc_html( $test_mode_label ), wp_kses_post( $this->row_actions( $actions ) ) );
	}

	/**
	 * Added filter here.
	 *
	 * @param string $which Where we need to show table nav.
	 */
	protected function extra_tablenav( $which ) {
		global $wpdb;
		$table_name = \bbapp_iap()->get_global_dbprefix() . 'bbapp_iap_products';
		if ( 'top' === $which ) {
			$iap_product = $this->bbapp_requested_purchased_product_id();
			$results     = $wpdb->get_results( "SELECT id, name FROM {$table_name} WHERE status IN ('published') ORDER BY menu_order, date_created DESC", ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			if ( $results ) {
				?>
				<div class="alignleft actions">
					<label class="screen-reader-text" for="filter-by-iap-products"><?php echo esc_html__( 'All Products', 'buddyboss-app' ); ?></label>
					<select name="iap_product_id" id="filter-by-iap-products" class="bbap-filter-product">
						<option value=""><?php echo esc_html__( 'All Products', 'buddyboss-app' ); ?></option>
						<?php
						foreach ( $results as $result ) {
							?>
							<option value="<?php echo esc_attr( $result['id'] ); ?>" <?php echo selected( (int) $iap_product, (int) $result['id'], false ); ?>><?php echo esc_html( $result['name'] ); ?></option>
							<?php
						}
						?>
					</select>
					<input type="button" name="iap_filter_action" id="iap_filter_action" class="button action" value="<?php esc_html_e( 'Filter', 'buddyboss-app' ); ?>"/>

                    <!-- Search by email -->
                    <input type="hidden" name="page" value="<?php echo esc_attr( $_REQUEST['page'] ); ?>">
                    <input type="hidden" name="setting" value="<?php echo ! empty( $_REQUEST['setting'] ) ? esc_attr( $_REQUEST['setting'] ) : ''; ?>">
                    <input type="search" id="search-by-email" name="search_by_email" placeholder="Search by Email" value="<?php echo isset( $_REQUEST['search_by_email'] ) ? esc_attr( $_REQUEST['search_by_email'] ) : ''; ?>">
                    <input type="button" id="iap_purchases_search" class="button action" value="Go">
				</div>
				<?php
			}
		}
	}

	/**
	 * Return product id from the request.
	 *
	 * @return mixed|string
	 */
	public function bbapp_requested_purchased_product_id() {
		return isset( $_REQUEST['iap_product_id'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['iap_product_id'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	}

}
