<?php
/**
 * Handl all order helper functions.
 *
 * @package BuddyBossApp\Admin\InAppPurchases
 */

namespace BuddyBossApp\Admin\InAppPurchases;

use BuddyBossApp\InAppPurchases\Orders;

/**
 * Class OrderHelper
 */
class OrderHelper {

	/**
	 * Class instance.
	 *
	 * @var null
	 */
	private static $instance = null;

	/**
	 * Check is current page.
	 *
	 * @var $is_current_page
	 */
	private $is_current_page;

	/**
	 * Message data used in list html page.
	 *
	 * @var array $messages
	 */
	private $messages = array();

	/**
	 * Order list.
	 *
	 * @var $orders_list
	 */
	private $orders_list;

	/**
	 * Class constructor.
	 */
	public function __construct() {}

	/**
	 * Get the instance of the class.
	 *
	 * @return OrderHelper
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			$class_name     = __CLASS__;
			self::$instance = new $class_name();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Magic methods for old methods.
	 *
	 * @param string             $name      Method name.
	 * @param string|mixed|array $arguments Method arguments.
	 *
	 * @since 1.7.5
	 *
	 * @return mixed|void
	 */
	public static function __callStatic( $name, $arguments ) {
		// Compatibility Support for CamelCase.
		$compatibility_func = array(
			'orderSingleScreen' => 'order_single_screen',
		);

		foreach ( $compatibility_func as $old => $new ) {
			if ( $name === $old ) {
				_deprecated_function( esc_html( $old ), esc_html( bbapp()->plugin_version ), esc_html( $new ) );

				return call_user_func_array( array( self::instance(), $new ), $arguments );
			}
		}
	}

	/**
	 * Initialize actions/filters
	 *
	 * @return void
	 */
	public function init() {
		add_action( 'admin_init', array( $this, 'admin_init' ), 15 );
		// IAP screen options.
		add_action( 'load-buddyboss-app_page_bbapp-iap', array( $this, 'get_screen_options' ) );
		add_filter( 'set-screen-option', array( $this, 'bbapp_admin_screen_options' ), 10, 3 );
	}

	/**
	 * Functions tells & sets if current page is one where it's will render.
	 *
	 * @param bool $set Set value for current page.
	 *
	 * @return bool
	 */
	public function will_render( $set = false ) {

		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Load Order List.
	 *
	 * @return void
	 */
	public function admin_init() {

		if ( $this->will_render() ) {

			$this->orders_list = new OrderList();
			$this->orders_list->prepare_items();
		}

	}

	/**
	 * Renders screen based on action variable in url.
	 */
	public function render() {
		$action = isset( $_REQUEST['action'] ) ? bbapp_input_clean( wp_unslash( $_REQUEST['action'] ) ) : 'lists'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		switch ( $action ) {
			case 'upgrade':
				UpgradeHelper::upgrade_orders_to_new_schema();
				break;
			case 'revoke_order':
			case 'bulk-revoke-order':
				if ( isset( $_REQUEST['iap_purchases_ids'] ) && ! empty( $_REQUEST['iap_purchases_ids'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
					include bbapp()->plugin_dir . '/views/iap/admin/order/revoke-order.php';
				} else {
					include bbapp()->plugin_dir . '/views/iap/admin/order/lists.php';
				}
				break;
			default:
				include bbapp()->plugin_dir . '/views/iap/admin/order/lists.php';
				break;
		}
	}

	/**
	 * Renders the Single Screen of InAppPurchase Order.
	 *
	 * @return void
	 */
	public function order_single_screen() {
		$order_id = ! empty( $_REQUEST['order_id'] ) ? (int) $_REQUEST['order_id'] : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$order    = Orders::instance()->get_by_id( $order_id );

		if ( empty( $order_id ) || is_wp_error( $order ) ) {
			include bbapp()->plugin_dir . '/views/iap/admin/order/not-found.php';
		} else {

			$date_format  = get_option( 'date_format' ) . ' @ ' . get_option( 'time_format' );
			$iap          = bbapp_iap()->iap[ $order->device_platform ];
			$device_label = $iap->get_label();

			$buddybossapp_product = bbapp_iap_get_product( $order->bbapp_product_id );
			$integration_types    = maybe_unserialize( $order->integration_types );
			$item_ids             = maybe_unserialize( $order->item_ids );
			$items                = Helpers::parse_item_ids( $item_ids );
			$items_readable       = implode( '<br/>', $items );

			// NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_types or item_ids.
			if ( bbapp()->is_network_activated() ) {
				$item_ids       = $item_ids[ get_current_blog_id() ];
				$items          = Helpers::parse_item_ids( $item_ids );
				$items_readable = implode( '<br/>', $items );
			}

			$integration_labels = array();
			foreach ( $integration_types as $integration_type ) {
				$integration_labels[] = Helpers::get_integration_label( $integration_type );
			}

			// if multisite we need to get integration types from current blog site.
			if ( bbapp()->is_network_activated() ) {
				$integration_types = $integration_types[ get_current_blog_id() ];
			}
			$integration_readable = implode( '<br/>', $integration_labels );
			$user_data            = get_userdata( $order->user_id );
			$name                 = __( 'Deleted', 'buddyboss-app' );

			if ( function_exists( 'bp_core_get_user_displayname' ) ) {
				$name = bp_core_get_user_displayname( $order->user_id );
			} elseif ( ! empty( $user_data ) ) {
				$name = $user_data->display_name;
			}
			$user_details['id']           = $order->user_id;
			$user_details['display_name'] = $name;

			include bbapp()->plugin_dir . '/views/iap/admin/order/single.php';
		}

	}

	/**
	 * Get IAP user url.
	 *
	 * @param int         $user_id    User id.
	 * @param bool|string $user_email User email.
	 *
	 * @return bool|string|void
	 */
	public function bbapp_get_iap_user_url( $user_id = 0, $user_email = false ) {
		if ( empty( $user_id ) ) {
			if ( $user_email && is_email( $user_email ) ) {
				$url = admin_url( "users.php?s={$user_email}" );

				return "<a href='" . esc_url( $url ) . "' target='_blank'>{$user_email}</a>";
			}

			return __( 'No user found.', 'buddyboss-app' );
		}
		$user_data = get_userdata( $user_id );
		$name      = __( 'Deleted', 'buddyboss-app' );
		$url       = admin_url( "user-edit.php?user_id={$user_id}" );
		if ( function_exists( 'bp_core_get_user_displayname' ) ) {
			$name = bp_core_get_user_displayname( $user_id );
		} elseif ( ! empty( $user_data ) ) {
			$name = $user_data->display_name;
		}

		return ! empty( $url ) ? "<a href='" . esc_url( $url ) . "' target='_blank'>{$name}</a>" : $name;
	}


	/**
	 * Register the screen options.
	 *
	 * @since 1.4.4
	 */
	public function get_screen_options() {
		$screen = get_current_screen();

		if ( ! is_object( $screen ) || 'buddyboss-app_page_bbapp-iap' !== $screen->id ) {
			return;
		}

		if ( isset( $_GET['setting'] ) && 'orders' !== bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			return;
		}

		// Loop through all the options and add a screen option for each.
		add_screen_option(
			'per_page',
			array(
				'label'   => __( 'Number of items per page:', 'buddyboss-app' ),
				'default' => 20,
				'option'  => 'iap_order_per_page',
			)
		);
	}

	/**
	 * Handle save/update of screen options for iap order.
	 *
	 * @param string $value     Will always be false unless another plugin filters it first.
	 * @param string $option    Screen option name.
	 * @param string $new_value Screen option form value.
	 *
	 * @return string|int Option value. False to abandon update.
	 * @since 1.4.4
	 */
	public function bbapp_admin_screen_options( $value, $option, $new_value ) {
		if ( 'iap_order_per_page' !== $option ) {
			return $value;
		}

		// Per page.
		$new_value = (int) $new_value;
		if ( $new_value < 1 || $new_value > 999 ) {
			return $value;
		}

		return $new_value;
	}

	/**
	 * Function to trim IOS date.
	 *
	 * @param string $date IOS date.
	 *
	 * @since 1.4.4
	 *
	 * @return false|mixed|string
	 */
	public function bbapp_trim_ios_date( $date ) {

		if ( ! $date ) {
			return false;
		}

		$date_exploded = explode( ' ', $date );
		$ios_date      = '';

		if ( isset( $date_exploded[0] ) ) {
			$ios_date .= $date_exploded[0];
		}
		if ( isset( $date_exploded[1] ) ) {
			$ios_date .= ' ' . $date_exploded[1];
		}

		return $ios_date;
	}
}
