<?php
/**
 * In app purchases helper class.
 *
 * @package BuddyBossApp\Admin\InAppPurchases
 */

namespace BuddyBossApp\Admin\InAppPurchases;

use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\AppSettings;
use BuddyBossApp\Jobs;

/**
 * Class Helpers
 */
class Helpers {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Class constructor
	 */
	public function __construct() {}

	/**
	 * Get the instance of the class.
	 *
	 * @return Helpers
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Magic methods for old methods.
	 *
	 * @param string             $name      Method name.
	 * @param string|mixed|array $arguments Method arguments.
	 *
	 * @since 1.7.5
	 *
	 * @return mixed|void
	 */
	public static function __callStatic( $name, $arguments ) {
		// Compatibility Support for CamelCase.
		$compatibility_func = array(
			'getProducts'                  => 'get_products',
			'platformStoreProductTypeslug' => 'platform_store_product_typeslug',
			'getIntegrationItems'          => 'get_integration_items',
			'parseItemIds'                 => 'parse_item_ids',
			'getIntegrationInstance'       => 'get_integration_instance',
			'getIntegrationLabel'          => 'get_integration_label',
			'getIntegrationSlug'           => 'get_integration_slug',
			'getIntegrationItemLabel'      => 'get_integration_item_label',
			'getProductById'               => 'get_product_by_id',
			'searchIntegration'            => 'search_integration',
			'isRecurringType'              => 'is_recurring_type',
			'getStoreProductInfo'          => 'get_store_product_info',
			'platformStoreProductTypes'    => 'platform_store_product_types',
		);

		foreach ( $compatibility_func as $old => $new ) {
			if ( $name === $old ) {
				_deprecated_function( esc_html( $old ), esc_html( bbapp()->plugin_version ), esc_html( $new ) );

				return call_user_func_array( array( self::instance(), $new ), $arguments );
			}
		}
	}

	/**
	 * Load additional hooks such as : admin_init
	 * NOTE : Method is similar to concept of init
	 *
	 * @return void
	 */
	public function load() {}

	/**
	 * Parse(split) custom format(id:title) to get id and attach it to associated integration(learndash-course, memberpress or woo-membership)
	 *
	 * @param array|string $item_ids Item ids.
	 *
	 * @return array {array} items items
	 */
	public static function parse_item_ids( $item_ids ) {
		$items = array();
		foreach ( $item_ids as $integration_type => $identifiers ) {
			$readable_item_ids = array();

			foreach ( maybe_unserialize( $identifiers ) as $identifier ) {
				$split = explode( ':', $identifier );
				$id    = $split[0];

				/**
				 * Filter to get Edit permalink for item id as it different for every 3rd party plugin.
				 *
				 * @param string     $link             Item permalink.
				 * @param int|string $id               Item id.
				 * @param string     $integration_type Integration type.
				 */
				$link = apply_filters( 'bbapp_iap_integration_item_id_permalink', '#', $id, $integration_type );

				$readable_item_ids[] = "<a href='$link' target='_blank'>$id</a>";
			}
			$integration_item_label = self::get_integration_item_label( $integration_type );
			$items[]                = "$integration_item_label: " . implode( ', ', $readable_item_ids );
		}

		return $items;

	}

	/**
	 * Helper function to get integration-instance via integration-slug
	 *
	 * @param string $slug Integration slug.
	 *
	 * @return  string integration-slug
	 */
	public static function get_integration_instance( $slug ) {

		$integrations = Controller::instance()->get_integrations();
		if ( isset( $integrations[ $slug ] ) && isset( $integrations[ $slug ]['class'] ) ) {
			return $integrations[ $slug ]['class']::instance();
		}

		return false;
	}

	/**
	 * Helper function to get integration-label via integration-type
	 *
	 * @param string $type Integration type.
	 *
	 * @return  string integration-label
	 */
	public static function get_integration_label( $type ) {

		$label        = $type;
		$integrations = Controller::instance()->get_integrations();
		foreach ( $integrations as $slug => $details ) {
			if ( $type === $details['type'] ) {
				$label = $details['label'];
				break;
			}
		}

		return $label;
	}

	/**
	 * Helper function to get integration-slug via integration-type
	 *
	 * @param string $type Integration type.
	 *
	 * @return  string integration-label
	 */
	public static function get_integration_slug( $type ) {

		$slug         = $type;
		$integrations = Controller::instance()->get_integrations();
		foreach ( $integrations as $i_slug => $details ) {
			if ( $type === $details['type'] ) {
				$slug = $i_slug;
				break;
			}
		}

		return $slug;
	}

	/**
	 * Helper function to get integration items via integration-slug
	 *
	 * @param string $slug Integration slug.
	 *
	 * @return  array integration-slug
	 */
	public static function get_integration_items( $slug ) {
		global $wpdb;

		/**
		 * Allow to update list of integration items for linking by 3rd party integration for admin screen.
		 *
		 * @param array  $options Linking options.
		 * @param string $slug    Integration slug.
		 */
		$results = apply_filters( 'bbapp_ajax_iap_linking_options', array(), $slug );

		if ( empty( $results ) ) {
			$query   = "SELECT CONCAT(posts.ID, \":\" , posts.post_title, \"\") as 'id', CONCAT(posts.post_title, \" (#\" , posts.ID, \")\")  as 'text' FROM $wpdb->posts posts WHERE posts.post_type = \"$slug\"  AND posts.post_status = 'publish' ";
			$results = $wpdb->get_results( $query, ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		}

		return $results;
	}

	/**
	 * Helper function to get integration-item-label via integration-type
	 *
	 * @param string $type Integration type.
	 *
	 * @return  string integration-label
	 */
	public static function get_integration_item_label( $type ) {

		$label        = $type;
		$integrations = Controller::instance()->get_integrations();
		foreach ( $integrations as $slug => $details ) {
			if ( $type === $details['type'] && ! empty( $details['class'] ) ) {
				$class = $details['class'];
				$label = $class::instance()->get_item_label();
				break;
			}
		}

		return $label;
	}

	/**
	 * BuddyBossApp Product(for InAppPurchase) by id
	 *
	 * @param int $id Product id.
	 *
	 * @return array {object}
	 */
	public static function get_product_by_id( $id ) {

		global $wpdb;

		$global_prefix = \bbapp_iap()->get_global_dbprefix();
		$result        = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$global_prefix}bbapp_iap_products WHERE id = %d AND status = 'published'", $id ), OBJECT ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		$product = array();

		if ( ! empty( $result ) ) {

			$store_data          = maybe_unserialize( $result->store_data );
			$misc_settings       = self::bbapp_iap_product_mics_setting( $result );
			$integration_data    = self::bbapp_iap_product_integration_data( $result );
			$integration_type    = isset( $misc_settings['integration_type'] ) ? $misc_settings['integration_type'] : '';
			$integrated_item_ids = self::bbapp_iap_product_integration_ids( $integration_type, $integration_data );

			$ios                = self::bbapp_iap_ios_product_info( $store_data );
			$android            = self::bbapp_iap_android_product_info( $store_data );
			$bbapp_product_type = isset( $store_data['bbapp_product_type'] ) ? $store_data['bbapp_product_type'] : 'free';

			// Check Product is configure properly or not. If not it should not return in api response.
			if (
				( isset( $store_data['device_platforms'] ) && in_array( 'ios', $store_data['device_platforms'], true ) && ( empty( $ios['status'] ) || ( 'free' !== $bbapp_product_type && empty( $ios['store_product_id'] ) ) ) ) ||
				( isset( $store_data['device_platforms'] ) && in_array( 'android', $store_data['device_platforms'], true ) && ( empty( $android['status'] ) || ( 'free' !== $bbapp_product_type && empty( $android['store_product_id'] ) ) ) ) ||
				empty( $integrated_item_ids )
			) {
				return $product;
			}

			if ( is_user_logged_in() ) {
				// NOTE : Get user_id and check into bbapp_orders if this bbapp.
				$has_access = ProductHelper::has_active_order( $result, get_current_user_id() );

				// Check Any order exist for same group product.
				$group_active_product = 0;
				if ( ! empty( $result->iap_group ) ) {
					$group_active_product = ProductHelper::get_group_active_order_product_id( $result->iap_group, get_current_user_id() );
				}
			} else {
				$has_access           = false;
				$group_active_product = 0;
			}
			if ( false === self::instance()->is_enabled_integration( $integration_type ) ) {
				return $product;
			}
			$product = array(
				'product_id'           => (int) $result->id,
				'product_name'         => $result->name,
				'product_tagline'      => $result->tagline,
				'product_desc'         => $result->description,
				'benefits'             => $misc_settings['benefits'],
				'global_subscription'  => $misc_settings['global_subscription'] ? true : false,
				'bbapp_product_type'   => $bbapp_product_type,
				'ios'                  => $ios,
				'android'              => $android,
				'integration_type'     => $misc_settings['integration_type'],
				'integrated_item_ids'  => $integrated_item_ids,
				'has_access'           => $has_access,
				'group_active_product' => $group_active_product,
				'sort_order'           => (int) $result->menu_order,
			);
		}

		return $product;

	}

	/**
	 * BuddyBossApp Products(for InAppPurchase)
	 *
	 * @param array $args Product argument.
	 *
	 * @return array {array}
	 */
	public static function get_products( $args = array() ) {
		global $wpdb;

		$global_prefix = \bbapp_iap()->get_global_dbprefix();
		$where         = array();

		$query   = "SELECT * FROM {$global_prefix}bbapp_iap_products AS iap";
		$where[] = "iap.status = 'published'";

		if ( ! empty( $args['registration_products'] ) ) {
			if ( AppSettings::instance()->get_setting_value( 'iap.purchase_before_register' ) ) {
				$query  .= " INNER JOIN {$global_prefix}bbapp_iap_productmeta AS iapm ON (iap.id = iapm.iap_id )";
				$where[] = "iapm.meta_key='product_visibility_registration'";
				$where[] = 'iapm.meta_value=1';
			} else {
				// If registration product argument passed but, setting is not enabled from the backend.
				return array();
			}
		}

		if ( isset( $args['include'] ) && ! empty( $args['include'] ) ) {
			$where[] = 'id IN (' . implode( ',', $args['include'] ) . ')';
		}

		$where_conditions = implode( ' AND ', $where );
		$query           .= ' WHERE ' . $where_conditions;

		if ( ! empty( $args['orderby'] ) ) {
			$query .= ' ORDER BY iap.menu_order,iap.' . esc_sql( $args['orderby'] );
			$query .= ! empty( $args['order'] ) ? ' ' . esc_sql( $args['order'] ) : ' ASC';
		} else {
			$query .= ' ORDER BY iap.menu_order ASC';
		}

		$products = array();
		$results  = $wpdb->get_results( $query, OBJECT ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

		foreach ( $results as $result ) {

			$store_data       = maybe_unserialize( $result->store_data );
			$misc_settings    = self::bbapp_iap_product_mics_setting( $result );
			$integration_data = self::bbapp_iap_product_integration_data( $result );

			if ( false === self::instance()->is_enabled_integration( $misc_settings['integration_type'] ) ) {
				continue;
			}

			/**
			 * If global_subscription filter active and product isn't global_subscription then filter it.
			 */
			if ( isset( $args['global_subscription'] ) && ! empty( $args['global_subscription'] ) && ! $misc_settings['global_subscription'] ) {
				continue;
			}

			$integration_type    = isset( $misc_settings['integration_type'] ) ? $misc_settings['integration_type'] : '';
			$integrated_item_ids = self::bbapp_iap_product_integration_ids( $integration_type, $integration_data );

			$ios                = self::bbapp_iap_ios_product_info( $store_data );
			$android            = self::bbapp_iap_android_product_info( $store_data );
			$bbapp_product_type = isset( $store_data['bbapp_product_type'] ) ? $store_data['bbapp_product_type'] : 'free';

			if ( ! bbapp_is_connected_as_secondary_site() ) {
				// Check Product is configure properly or not. If not it should not return in api response.
				if (
				( isset( $store_data['device_platforms'] ) && in_array( 'ios', $store_data['device_platforms'], true ) && ( empty( $ios['status'] ) || ( 'free' !== $bbapp_product_type && empty( $ios['store_product_id'] ) ) ) ) ||
				( isset( $store_data['device_platforms'] ) && in_array( 'android', $store_data['device_platforms'], true ) && ( empty( $android['status'] ) || ( 'free' !== $bbapp_product_type && empty( $android['store_product_id'] ) ) ) ) ||
				empty( $integrated_item_ids )
				) {
					continue;
				}
			}

			if ( is_user_logged_in() ) {
				// NOTE : Get user_id and check into bbapp_orders if this bbapp.
				$has_access = ProductHelper::has_active_order( $result, get_current_user_id() );

				// Check Any order exist for same group product.
				$group_active_product = 0;
				if ( ! empty( $result->iap_group ) ) {
					$group_active_product = ProductHelper::get_group_active_order_product_id( $result->iap_group, get_current_user_id() );
				}
			} else {
				$has_access           = false;
				$group_active_product = 0;
			}
			$iap_product_image               = isset( $result->id ) ? get_iap_feature_image( $result->id ) : '';
			$purchase_before_register        = bbapp_is_active( 'iap' ) ? AppSettings::instance()->get_setting_value( 'iap.purchase_before_register' ) : false;
			$product_visibility_registration = $purchase_before_register && isset( $result->id ) ? bbapp_iap_get_meta( $result->id, 'product_visibility_registration', true ) : '';

			/**
			 *  Filter to check given result is skip or not.
			 *
			  * @param bool   $flag   skip flag.
			  * @param object $result product object.
			  * @since 1.8.50
			 */
			if ( apply_filters( 'bbapp_is_product_skip', false, $result ) ) {
				continue;
			}

			$products[] = array(
				'product_id'           => (int) $result->id,
				'product_name'         => $result->name,
				'product_tagline'      => $result->tagline,
				'product_desc'         => $result->description,
				'benefits'             => $misc_settings['benefits'],
				'global_subscription'  => $misc_settings['global_subscription'] ? true : false,
				'bbapp_product_type'   => $bbapp_product_type,
				'ios'                  => $ios,
				'android'              => $android,
				'integration_type'     => $misc_settings['integration_type'],
				'integrated_item_ids'  => $integrated_item_ids,
				'has_access'           => $has_access,
				'group_active_product' => $group_active_product,
				'sort_order'           => (int) $result->menu_order,
				'product_image'        => $iap_product_image,
				'product_registration' => (bool) $product_visibility_registration,
			);
		}

		return $products;
	}

	/**
	 * Function to check if there are platform specific products are available or not.
	 *
	 * @param string $device_platform device platform like ios, android.
	 *
	 * @since 1.4.4
	 *
	 * @return bool
	 */
	public static function bbapp_device_platform_products_available( $device_platform ) {
		$available = false;

		if ( empty( $device_platform ) || ! in_array( $device_platform, array( 'ios', 'android' ), true ) ) {
			return $available;
		}

		$products = self::get_products( array( 'registration_products' => true ) );

		if ( ! empty( $products ) ) {
			foreach ( $products as $product ) {
				if ( isset( $product[ $device_platform ] ) && ( 'free' === $product['bbapp_product_type'] || ( ! empty( $product[ $device_platform ]['store_product_id'] ) && ! empty( $product[ $device_platform ]['store_product_status'] ) ) ) ) {
					$available = true;
					break;
				}
			}
		}

		return $available;
	}

	/**
	 * Search IntegrationAbstract for itemId[ Example: Course ID ]
	 *
	 * @param null   $for_item_id      : Item id. it can be course_id or membership id or anything which connected with membership plugin.
	 * @param string $integration_type : Integration type.
	 * @param array  $args             : Seach integration argument.
	 *
	 * @return array {array}
	 */
	public static function search_integration( $for_item_id = null, $integration_type = '', $args = array() ) {
		global $wpdb;

		$integrated_products = array();

		$global_prefix = \bbapp_iap()->get_global_dbprefix();
		$query         = "SELECT * FROM {$global_prefix}bbapp_iap_products WHERE status = 'published'";

		if ( ! empty( $integration_type ) ) {
			$integration_slug     = self::get_integration_slug( $integration_type );
			$integration_type_str = sprintf( 's:%d:"%s";s:%d:"%s";', strlen( 'integration_type' ), 'integration_type', strlen( $integration_slug ), $integration_slug );
			$query               .= ' AND misc_settings like ';
			$query               .= "'%$integration_type_str%'";
		}

		if ( ! empty( $args['orderby'] ) ) {
			$query .= ' ORDER BY menu_order,' . esc_sql( $args['orderby'] );
			$query .= ! empty( $args['order'] ) ? ' ' . esc_sql( $args['order'] ) : ' ASC';
		} else {
			$query .= ' ORDER BY menu_order ASC';
		}

		$buddybossapp_products = $wpdb->get_results( $query, OBJECT ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

		/**
		 * We need to check through all all bbapp product of $integrationType to find items which is direct/indirect connected and give access for given $itemID
		 * For 3rd party integration, to identify connection might be different so it's will be custom code.
		 * We can do that code in while extending support using hooks
		 */
		foreach ( $buddybossapp_products as $buddybossapp_product ) {

			$misc_settings    = maybe_unserialize( $buddybossapp_product->misc_settings );
			$integration_data = maybe_unserialize( $buddybossapp_product->integration_data );

			if ( bbapp()->is_network_activated() ) {
				$misc_settings    = $misc_settings[ get_current_blog_id() ];
				$integration_data = $integration_data[ get_current_blog_id() ];
			}

			$integration_slug = $misc_settings['integration_type'];

			$integration_data = isset( $integration_data[ $integration_slug ] ) ? $integration_data[ $integration_slug ] : array();

			foreach ( $integration_data as $value ) {

				// NOTE : Sample storage is like => "id:label".
				$split               = explode( ':', $value );
				$integration_item_id = $split[0];

				/**
				 * Filter to check given $forItemId is connect directly/indirectly with given $integration_item_id
				 *
				 * @param bool       $available           If the purchase is available.
				 * @param int        $for_item_id         If the purchase available for the item.
				 * @param int|string $integration_item_id Integration item id.
				 * @param string     $integration_slug    Integration slug.
				 */
				$found_configured = apply_filters( 'bbapp_iap_is_purchase_available', false, $for_item_id, $integration_item_id, $integration_slug );

				if ( null === $for_item_id || $found_configured ) {
					$store_data          = maybe_unserialize( $buddybossapp_product->store_data );
					$misc_settings       = self::bbapp_iap_product_mics_setting( $buddybossapp_product );
					$integration_data    = self::bbapp_iap_product_integration_data( $buddybossapp_product );
					$integration_type    = ( isset( $misc_settings['integration_type'] ) ? $misc_settings['integration_type'] : '' );
					$integrated_item_ids = self::bbapp_iap_product_integration_ids( $integration_type, $integration_data );

					$ios                = self::bbapp_iap_ios_product_info( $store_data );
					$android            = self::bbapp_iap_android_product_info( $store_data );
					$bbapp_product_type = isset( $store_data['bbapp_product_type'] ) ? $store_data['bbapp_product_type'] : 'free';

					// Check Product is configure properly or not. If not it should not return in api response.
					if (
						( isset( $store_data['device_platforms'] ) && in_array( 'ios', $store_data['device_platforms'], true ) && ( empty( $ios['status'] ) || ( 'free' !== $bbapp_product_type && empty( $ios['store_product_id'] ) ) ) ) ||
						( isset( $store_data['device_platforms'] ) && in_array( 'android', $store_data['device_platforms'], true ) && ( empty( $android['status'] ) || ( 'free' !== $bbapp_product_type && empty( $android['store_product_id'] ) ) ) ) ||
						empty( $integrated_item_ids )
					) {
						continue;
					}

					if ( false === self::instance()->is_enabled_integration( $integration_type ) ) {
						continue;
					}

					if ( is_user_logged_in() ) {
						// NOTE : Get user_id and check into bbapp_orders if this bbapp.
						$has_access = ProductHelper::has_active_order( $buddybossapp_product, get_current_user_id() );

						// Check Any order exist for same group product.
						$group_active_product = 0;
						if ( ! empty( $buddybossapp_product->iap_group ) ) {
							$group_active_product = ProductHelper::get_group_active_order_product_id( $buddybossapp_product->iap_group, get_current_user_id() );
						}
					} else {
						$has_access           = false;
						$group_active_product = 0;
					}

					$integrated_products[] = array(
						'product_id'           => (int) $buddybossapp_product->id,
						'product_name'         => $buddybossapp_product->name,
						'product_tagline'      => $buddybossapp_product->tagline,
						'product_desc'         => $buddybossapp_product->description,
						'benefits'             => $misc_settings['benefits'],
						'global_subscription'  => $misc_settings['global_subscription'] ? true : false,
						'bbapp_product_type'   => $bbapp_product_type,
						'ios'                  => $ios,
						'android'              => $android,
						'has_access'           => $has_access,
						'group_active_product' => $group_active_product,
						'sort_order'           => (int) $buddybossapp_product->menu_order,
					);

				}
			}
		}

		/**
		 * Filter to add courses to the integrated products.
		 *
		 * @param array  $integrated_products Array of integrated products.
		 * @param int    $for_item_id         Membership id.
		 * @param string $integration_type    Integration type.
		 *
		 * @since 1.4.0
		 */
		return apply_filters( 'bbapp_iap_integrated_products', $integrated_products, $for_item_id, $integration_type );
	}

	/**
	 * NOTE : Below should match based on what's define while initializing integration
	 *
	 * @param string $store_product_type Store product type.
	 *
	 * @return bool
	 */
	public static function is_recurring_type( $store_product_type ) {

		$is_recurring = false;

		// Recurring ones: ios => auto_renewable, non_auto_renewable; android =>  auto_renewable.
		$recurring_types = array( 'auto_renewable', 'non_auto_renewable' );

		if ( in_array( $store_product_type, $recurring_types, true ) ) {
			$is_recurring = true;
		}

		return $is_recurring;
	}

	/**
	 * Return info(Store Product) via Type (Store Product Type) Default it'll return lable
	 *
	 * @param string $store_product_type : Store Product Type for ios or android. Eg : auto_renewable, non_auto_renewable.
	 * @param string $device_platform    : Platform like ios,android.
	 * @param string $column             : Column name.
	 *
	 * @return string | null
	 */
	public static function get_store_product_info( $store_product_type, $device_platform = 'ios', $column = 'label' ) {

		$store_product_types = self::platform_store_product_types( $device_platform );

		// Needle, haystack.
		if ( in_array( $store_product_type, array_keys( $store_product_types ), true ) ) {

			return $store_product_types[ $store_product_type ][ $column ];
		}

		return null;
	}

	/**
	 * Convert store product type into local slug
	 *
	 * @param string $store_platform_type Store Platform type.
	 *
	 * @return string
	 */
	public static function platform_store_product_typeslug( $store_platform_type ) {
		$slug = '';

		switch ( $store_platform_type ) {
			case 'managedUser':
			case 'CONSUMABLE':
				$slug = 'consumable';
				break;
			case 'NON_CONSUMABLE':
				$slug = 'non_consumable';
				break;
			case 'subscription':
			case 'AUTOMATICALLY_RENEWABLE_SUBSCRIPTION':
				$slug = 'auto_renewable';
				break;
			case 'NON_RENEWING_SUBSCRIPTION':
				$slug = 'non_auto_renewable';
				break;
		}

		return $slug;
	}

	/**
	 * Return Store Product Types via platformType
	 *
	 * @param string $platform_type : device OS. Eg : ios, android.
	 *
	 * @return array $types : Array with label, recurring and available values
	 */
	public static function platform_store_product_types( $platform_type ) {

		switch ( $platform_type ) {
			case 'ios':
				$types = array(
					'consumable'         => array(
						'label'       => __( 'Consumable', 'buddyboss-app' ),
						'description' => __( "This product type is used once, after which it becomes depleted and must be purchased again. There isn't a use case for this in BuddyBoss App, so you shouldn’t use this product type.", 'buddyboss-app' ),
						'recurring'   => false,
						'available'   => false,
					),
					'non_consumable'     => array(
						'label'       => __( 'Non-Consumable', 'buddyboss-app' ),
						'description' => __( 'This product type is purchased once and does not expire. Example: Membership or course that is purchased once and never expires.', 'buddyboss-app' ),
						'recurring'   => false,
						'available'   => true,
					),
					'auto_renewable'     => array(
						'label'       => __( 'Auto-Renewable Subscription', 'buddyboss-app' ),
						'description' => __( 'This product type allows users to purchase content for a set period, renewing automatically unless cancelled by the user. Example: Monthly subscription for a membership or course.', 'buddyboss-app' ),
						'recurring'   => true,
						'available'   => true,
					),
					'non_auto_renewable' => array(
						'label'       => __( 'Non-Renewing Subscription', 'buddyboss-app' ),
						'description' => __( 'This product type allows users to purchase content for a limited duration, which will not renew automatically. Example: Membership or course that will expire and not renew automatically.', 'buddyboss-app' ),
						'recurring'   => false,
						'available'   => true,
					),
				);

				break;
			case 'android':
				$types = array(
					// NOTE : It is same product type as non-consumable (managed product) we have to do consume call on app.
					'consumable'     => array(
						'label'       => __( 'In-App Product', 'buddyboss-app' ),
						'description' => __( 'This product type is purchased once and does not expire. Example: Membership or course that is purchased once and never expires.', 'buddyboss-app' ),
						'recurring'   => false,
						'available'   => false,
					),
					// NOTE: Subscriptions are in-app content or services that are billed to users on a recurring basis.
					'auto_renewable' => array(
						'label'       => __( 'Subscription', 'buddyboss-app' ),
						'description' => __( 'This product type allows users to purchase content for a set period, renewing automatically unless cancelled by the user. Example: Monthly subscription for a membership or course.', 'buddyboss-app' ),
						'recurring'   => true,
						'available'   => true,
					),
				);

				break;

			default:
				$types = array();
				break;
		}

		return $types;

	}

	/**
	 * Return Store product status information
	 *
	 * @param string $status          : Store Product status for ios or android. Eg : active,inactive,approve etc.
	 * @param string $device_platform : Platform.
	 * @param string $column          : Column name.
	 *
	 * @return string | null
	 */
	public static function get_store_product_status_info( $status, $device_platform = 'ios', $column = 'label' ) {

		$store_product_status = self::platform_store_product_status( $device_platform, $status );

		// Needle, haystack.
		if ( in_array( $column, array_keys( $store_product_status ), true ) ) {
			return $store_product_status[ $column ];
		}

		return null;
	}

	/**
	 * Return Store Product Status via Platform
	 *
	 * @param string $platform : device OS. Eg : ios, android.
	 * @param string $status Store product status.
	 *
	 * @return array $status : Array with label, recurring and available values
	 */
	public static function platform_store_product_status( $platform, $status ) {
		switch ( $platform ) {
			case 'ios':
				switch ( strtoupper( $status ) ) {
					case 'APPROVED':
						$store_product_status = array(
							'label'       => __( 'Approved', 'buddyboss-app' ),
							'description' => __( 'This product has been approved by Apple and can be purchased in your release app. You can test the product from your test app, but you can only see real pricing, terms and purchase options in your release app.', 'buddyboss-app' ),
						);
						break;
					default:
						$store_product_status = array(
							'label'       => __( 'Not Approved', 'buddyboss-app' ),
							'description' => __( 'This product has not yet been approved by Apple. Until it has been submitted and approved, it will only be visible to admins in your release app, and cannot yet be purchased through Apple. You can still test the product from your test app, but you can only see real pricing and terms in your release app.', 'buddyboss-app' ),
						);
						break;
				}
				break;
			case 'android':
				switch ( strtoupper( $status ) ) {
					case 'ACTIVE':
						$store_product_status = array(
							'label'       => __( 'Active', 'buddyboss-app' ),
							'description' => __( 'This product is marked as active in the Google Play Console.', 'buddyboss-app' ),
						);
						break;
					default:
						$store_product_status = array(
							'label'       => __( 'Inactive', 'buddyboss-app' ),
							'description' => __( 'This product is marked as inactive in the Google Play Console. Until it is active, it will not be visible in your app.', 'buddyboss-app' ),
						);
						break;
				}
				break;
			default:
				$store_product_status = array();
				break;
		}

		return $store_product_status;

	}

	/**
	 * Get store product id by valid status.
	 *
	 * @param string|int $store_product_id Store product id.
	 * @param string     $product_type     Product type.
	 * @param string     $device_platform  Device platform like ios, android.
	 *
	 * @return string
	 */
	public function bbapp_iap_get_store_product_id_by_valid_status( $store_product_id, $product_type, $device_platform = 'ios' ) {

		if ( 'free' === $product_type ) {
			return 'ios' === $device_platform ? 'approved' : 'active';
		} else {
			$store_product_status = bbapp_iap_get_store_product( $store_product_id, $device_platform, 'status' );
			if ( 'ios' === $device_platform ) {
				if ( 'APPROVED' === strtoupper( $store_product_status ) ) {
					return 'approved';
				} else {
					return 'not-approved';
				}
			} else {
				if ( 'ACTIVE' === strtoupper( $store_product_status ) ) {
					return 'active';
				} else {
					return 'inactive';
				}
			}
		}
	}

	/**
	 * Return IAP product mics setting from product.
	 *
	 * @param object $ipa_app_product IAP product object.
	 *
	 * @return array|mixed|string
	 */
	public static function bbapp_iap_product_mics_setting( $ipa_app_product ) {

		$misc_settings = ( isset( $ipa_app_product->misc_settings ) ? maybe_unserialize( $ipa_app_product->misc_settings ) : array() );

		// NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_type or misc_settings.
		if ( bbapp()->is_network_activated() ) {
			$misc_settings = ( array_key_exists( get_current_blog_id(), $misc_settings ) ? $misc_settings[ get_current_blog_id() ] : array() );
		}

		return $misc_settings;
	}

	/**
	 * Return IAP product Integration from product.
	 *
	 * @param object $ipa_app_product IAP product object.
	 *
	 * @return array|mixed|string
	 */
	public static function bbapp_iap_product_integration_data( $ipa_app_product ) {

		$integration_data = ( isset( $ipa_app_product->integration_data ) ? maybe_unserialize( $ipa_app_product->integration_data ) : array() );

		// NOTE : For multi-site, we are storing data as [:blog-id][some_key_index]. Eg : integration_type or misc_settings.
		if ( bbapp()->is_network_activated() ) {
			$integration_data = ( array_key_exists( get_current_blog_id(), $integration_data ) ? $integration_data[ get_current_blog_id() ] : array() );
		}

		return $integration_data;
	}

	/**
	 * Return IAP product Integration product ids.
	 *
	 * @param string $integration_type IAP product object.
	 * @param object $integration_data IAP product object.
	 *
	 * @return array|mixed|string
	 */
	public static function bbapp_iap_product_integration_ids( $integration_type, $integration_data ) {

		$integrated_item_ids = array();
		if ( ! empty( $integration_type ) && ! empty( $integration_data ) && is_array( $integration_data ) && array_key_exists( $integration_type, $integration_data ) ) {
			$item_ids = $integration_data[ $integration_type ];
			if ( is_array( $item_ids ) && ! empty( $item_ids ) ) {
				foreach ( $item_ids as $item_identifier ) {
					$split = explode( ':', $item_identifier );
					if ( isset( $split[0] ) ) {
						$integrated_item_ids[] = $split[0];
					}
				}
			}
		}

		return $integrated_item_ids;
	}

	/**
	 * Return the IOS information.
	 *
	 * @param array $store_data IAP product store data.
	 *
	 * @return array
	 */
	public static function bbapp_iap_ios_product_info( $store_data ) {
		$ios_data = array(
			'store_product_id'      => '',
			'store_product_type'    => '',
			'store_product_status'  => '',
			'subscription_duration' => '',
			'recurring'             => '',
			'status'                => false,
		);

		if ( empty( $store_data ) ) {
			return $ios_data;
		}

		$bbapp_product_type                = isset( $store_data['bbapp_product_type'] ) ? $store_data['bbapp_product_type'] : 'free';
		$ios_data['store_product_id']      = isset( $store_data['store_product_ids']['ios'] ) ? $store_data['store_product_ids']['ios'] : null;
		$ios_data['store_product_type']    = isset( $store_data['store_product_types']['ios'] ) ? $store_data['store_product_types']['ios'] : '';
		$ios_data['store_product_status']  = isset( $ios_data['store_product_id'] ) ? self::instance()->bbapp_iap_get_store_product_id_by_valid_status( $ios_data['store_product_id'], $bbapp_product_type, 'ios' ) : null;
		$ios_data['subscription_duration'] = isset( $store_data['subscription_duration']['subscription_duration']['ios'] ) ? $store_data['store_product_types']['ios'] : null;
		$ios_data['recurring']             = self::is_recurring_type( $ios_data['store_product_type'] );

		// Check Product exists in store product.
		$store_products = bbapp_iap_get_store_products( 'ios' );
		if ( false === array_search( $ios_data['store_product_id'], array_column( $store_products, 'id' ), true ) ) {
			$ios_data['store_product_id']   = '';
			$ios_data['store_product_type'] = '';
		}

		if ( is_array( $store_data['device_platforms'] ) && in_array( 'ios', $store_data['device_platforms'], true ) && ( 'free' === $bbapp_product_type || ! empty( $ios_data['store_product_id'] ) ) ) {
			$ios_data['status'] = true;
		}

		if ( bbapp_is_connected_as_secondary_site() ) {
			$ios_data['status'] = true;
		}

		return $ios_data;
	}

	/**
	 * Return the Android information.
	 *
	 * @param array $store_data IAP product store data.
	 *
	 * @return array
	 */
	public static function bbapp_iap_android_product_info( $store_data ) {
		$android_data = array(
			'store_product_id'      => '',
			'store_product_type'    => '',
			'store_product_status'  => '',
			'subscription_duration' => '',
			'recurring'             => '',
			'status'                => false,
		);

		if ( empty( $store_data ) ) {
			return $android_data;
		}

		$bbapp_product_type = isset( $store_data['bbapp_product_type'] ) ? $store_data['bbapp_product_type'] : 'free';

		// Attach - Additional android Information.
		$android_data['store_product_id']     = isset( $store_data['store_product_ids']['android'] ) ? $store_data['store_product_ids']['android'] : null;
		$android_data['store_product_type']   = isset( $store_data['store_product_types']['android'] ) ? $store_data['store_product_types']['android'] : '';
		$android_data['store_product_status'] = isset( $android_data['store_product_id'] ) ? self::instance()->bbapp_iap_get_store_product_id_by_valid_status( $android_data['store_product_id'], $bbapp_product_type, 'android' ) : null;
		$android_data['recurring']            = self::is_recurring_type( $android_data['store_product_type'] );

		// Check Product exists in store product.
		$store_products = bbapp_iap_get_store_products( 'android' );
		if ( false === array_search( $android_data['store_product_id'], array_column( $store_products, 'id' ), true ) ) {
			$ios_data['store_product_id']   = '';
			$ios_data['store_product_type'] = '';
		}

		if ( is_array( $store_data['device_platforms'] ) && in_array( 'android', $store_data['device_platforms'], true ) && ( 'free' === $bbapp_product_type || ! empty( $android_data['store_product_id'] ) ) ) {
			$android_data['status'] = true;
		}

		if ( bbapp_is_connected_as_secondary_site() ) {
			$android_data['status'] = true;
		}

		return $android_data;
	}

	/**
	 * Check integration is enabled or not.
	 *
	 * @param null $integration_type Integration type.
	 *
	 * @since 1.5.5
	 * @return bool
	 */
	private function is_enabled_integration( $integration_type = null ) {
		if ( empty( $integration_type ) ) {
			return false;
		}

		$iap_integrations = Controller::instance()->get_enabled_integrations();

		// Skip the IAP product if the product membership plugin is not activated.
		return in_array( $integration_type, array_keys( $iap_integrations ), true );
	}

	/**
	 * Function to start miss configured IAP product to verify.
	 *
	 * @param int $current_page Current page number.
	 *
	 * @since 1.6.7
	 * @return bool
	 */
	public function bbapp_start_iap_product_job( $current_page = 1 ) {
		// Add the queue job to run on background.
		$bbapp_queue = Jobs::instance();
		$bbapp_queue->add(
			'verify_iap_products',
			array(
				'current_page' => $current_page,
			),
			1
		); // Job type allows only 20 char.

		return $bbapp_queue->start();
	}

}
