<?php
/**
 * In App Purcahses class.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\Admin\InAppPurchases\Helpers;
use BuddyBossApp\Admin\InAppPurchases\OrderHelper;
use BuddyBossApp\Admin\InAppPurchases\ProductHelper;
use BuddyBossApp\Admin\InAppPurchases\SubscriptionGroupsHelpers;
use BuddyBossApp\AppStores\Apple;
use BuddyBossApp\InAppPurchases\Controller;
use BuddyBossApp\InAppPurchases\SubscriptionGroupTaxonomy;
use BuddyBossApp\ManageApp;

/**
 * Class InAppPurchases
 */
class InAppPurchases {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Settings form.
	 *
	 * @var $settings_form
	 */
	private $settings_form;

	/**
	 * App settings provider.
	 *
	 * @var $app_settings_provider
	 */
	private $app_settings_provider;

	/**
	 * Get the instance of the class.
	 *
	 * @return InAppPurchases
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->init();
		}

		return self::$instance;
	}

	/**
	 * Class constructor
	 */
	public function __construct() {}

	/**
	 * Load additional hooks such as : admin_init
	 * NOTE : Method is similar to concept of init
	 *
	 * @return void
	 */
	public function init() {

		Helpers::instance();
		OrderHelper::instance();
		ProductHelper::instance();
		SubscriptionGroupsHelpers::instance();

		add_action( 'init', array( $this, 'load_init' ) );

		add_action( 'admin_notices', array( $this, 'bbapp_server_notification_notice' ), 99 );
		add_action( 'admin_footer', array( $this, 'bbapp_notification_script' ) );
		add_action( 'wp_ajax_bbapp_check_ios_server_notification_status', array( $this, 'bbapp_check_ios_server_notification_status' ) );
	}

	/**
	 * Load In-app puchases orders, products and subscription groups.
	 *
	 * @return void
	 */
	public function load_init() {

		$page     = isset( $_GET['page'] ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$page_now = SetupAdmin::instance()->get_page_now();

		if ( 'admin.php' === $page_now && 'bbapp-iap' === $page ) {

			$current_sub_tab = $this->get_current_sub_tab();
			switch ( $current_sub_tab ) {

				case 'orders':
					OrderHelper::instance()->will_render( true );
					break;
				case 'products':
					ProductHelper::instance()->will_render( true );
					break;
				case 'app-subscription-group':
					SubscriptionGroupsHelpers::instance();
					break;
				default:
					echo esc_html__( 'Tab not found', 'buddyboss-app' );
					break;
			}
		}

	}

	/**
	 * Function to add admin notice if server notification url is not configured.
	 *
	 * @since 1.4.2
	 */
	public function bbapp_server_notification_notice() {

		$app_data      = \BuddyBossApp\AppStores\Apple::instance()->fetch_app_data();
		$s2s_url       = add_query_arg( array( 'ios_s2s_notification' => 1 ), get_home_url() );
		$s2s_url_slash = add_query_arg( array( 'ios_s2s_notification' => 1 ), get_home_url( null, '/' ) );

		if ( bbapp_is_active( 'iap' ) && Apple::instance()->is_connected() && ( empty( $app_data['server_notification']['production']['url'] ) || ( $s2s_url !== $app_data['server_notification']['production']['url'] && $s2s_url_slash !== $app_data['server_notification']['production']['url'] ) ) && ! bbapp_is_connected_as_secondary_site() ) { ?>
			<div class="notice notice-error is-dismissible">
				<div class="bbapp-s2s-notif-msg">
					<p>
						<?php echo sprintf( '%1$s <a href="https://www.buddyboss.com/resources/docs/app/in-app-purchases/ios-in-app-products/#enter-url-for-app-store-server-notifications" target="_blank"><strong>%2$s</strong></a> %3$s', esc_html__( 'To receive accurate information about in-app purchases made on iOS devices, please', 'buddyboss-app' ), esc_html__( 'setup your App Store Notifications URL', 'buddyboss-app' ), esc_html__( 'in App Store Connect.', 'buddyboss-app' ) ); ?>
					</p>
					<p>
						<strong class="bbapp-s2s-nc"><?php esc_html_e( 'Not Configured.', 'buddyboss-app' ); ?></strong>
						<a href="javascript:void(0);" class="bbapp-check-server-notification-status-chk-btn">
							<?php esc_html_e( 'Check Again', 'buddyboss-app' ); ?>
						</a>
					</p>
					<div class="bbapp-server-notification-status">
						<p class="bbapp_loading dark" style="display:none"></p>
					</div>
				</div>
			</div>
			<?php
		}
	}

	/**
	 * Function to add the admin bottom script
	 *
	 * @since 1.4.2
	 */
	public function bbapp_notification_script() {
		?>
		<script>
            jQuery( document ).ready( function( $ ) {
                $( document ).on( 'click', '.bbapp-check-server-notification-status-chk-btn', function() {
                    const curObj = $( this );
                    const mainWrp = curObj.parent().closest( '.notice' );
                    const inrWrp = curObj.parent().closest( '.notice' ).find( '.bbapp-s2s-notif-msg' );
                    const loader = curObj.parent().closest( '.notice' ).find( '.bbapp_loading' );
                    const msgDiv = curObj.parent().closest( '.notice' ).find( '.bbapp-server-notification-status' );
                    const dismissObj = curObj.parent().closest( '.notice' ).find( '.notice-dismiss' );

                    msgDiv.find( '.bbapp-err-msg' ).remove();
                    curObj.hide();
                    loader.css( 'display', 'block' );

                    const data = {
                        action: 'bbapp_check_ios_server_notification_status',
                    };

                    $.post( ajaxurl, data, function( response ) {
                        if ( true === response.success ) {
                            inrWrp.empty();
                            mainWrp.addClass( 'notice-success' );
                            mainWrp.removeClass( 'notice-error' );
                            inrWrp.append( '<p class="bbapp-success-msg">' + response.data.message + '</p>' );

                            setTimeout( function() {
                                dismissObj.trigger( 'click' );
                            }, 2500 );
                        } else {
                            curObj.show();
                            curObj.parent().find( '.bbapp-s2s-nc' ).show();
                            loader.css( 'display', 'none' );
                            msgDiv.append( '<p class="bbapp-err-msg">' + response.data.message + '</p>' );
                        }
                    } );
                } );
            } );
        </script>
		<?php
	}

	/**
	 * Function to check the IOS server notification URL status
	 *
	 * @since 1.4.2
	 */
	public function bbapp_check_ios_server_notification_status() {
		$server_notification            = array();
		$server_notification['message'] = __( 'Please make sure you are connected to your Apple Developer Account', 'buddyboss-app' );

		if ( bbapp_is_active( 'iap' ) && Apple::instance()->is_connected() ) {
			$app_data      = \BuddyBossApp\AppStores\Apple::instance()->fetch_app_data( 0, true );
			$s2s_url       = add_query_arg( array( 'ios_s2s_notification' => 1 ), get_home_url() );
			$s2s_url_slash = add_query_arg( array( 'ios_s2s_notification' => 1 ), get_home_url( null, '/' ) );

			if ( ! empty( $app_data['server_notification']['production']['url'] ) && ! empty( $app_data['server_notification']['production']['version'] ) && ( $s2s_url === $app_data['server_notification']['production']['url'] || $s2s_url_slash === $app_data['server_notification']['production']['url'] ) ) {
				$server_notification['message'] = sprintf( __( 'Your App <strong>Store Server Notifications</strong> have been setup.', 'buddyboss-app' ) );
				wp_send_json_success( $server_notification );
			} else {
				$server_notification['message'] = __( 'App Store Server Notifications URL is not configured', 'buddyboss-app' );
				wp_send_json_error( $server_notification );
			}
		}
		wp_send_json_error( $server_notification );
	}

	/**
	 * Load Settings.
	 */
	public function load_settings() {

		$this->settings_form = new FormProvider( 'settings' );

		// holds app settings storage handle.
		$this->app_settings_provider = new SettingProvider( 'app_settings' );

		// register app settings storing & receiving.
		$this->app_settings_provider->save_setting_hook( array( $this, 'save_settings' ) );
		$this->app_settings_provider->get_setting_hook( array( $this, 'get_settings' ) );

		// Link Provider In Form Provider.
		$this->settings_form->link_setting_provider( $this->app_settings_provider );

		add_action( 'admin_init', array( $this->settings_form, 'save' ) );

		/**
		 * In-App Purchase Settings
		 */
		$this->settings_form->add_field(
			$this->app_settings_provider->field(
				false,
				array(
					'type'    => 'table-open',
					'classes' => 'buddyboss-app-admin-card bbapp-admin-in_app_purchase',
				)
			)
		);

		$this->settings_form->add_field(
			$this->app_settings_provider->field(
				'app_in_app_purchase_heading',
				array(
					'type'         => 'heading',
					'value'        => __( 'In-App Purchases', 'buddyboss-app' ),
					'desc'         => '',
					'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=122177' ),
					'icon'         => 'bb-icon-bf bb-icon-credit-card',
				)
			)
		);

		if ( ! bbapp()->is_network_activated() ) {

			$this->settings_form->add_field(
				$this->app_settings_provider->field(
					'iap.lock_app_notice',
					array(
						'label' => __( 'Require Purchase', 'buddyboss-app' ),
						'value' => sprintf( '<p class="bbapp-notice-box bbapp-notice-box--warning"><a href="%1$s">%2$s</a> %3$s <a href="%4$s" target="_blank">%5$s</a>.</p>', esc_url( bb_access_controls_enable_link() ), __( 'Access Controls', 'buddyboss-app' ), esc_html__( 'will now manage which purchases are required to unlock your app’s content.', 'buddyboss-app' ), esc_url( admin_url( 'admin.php?page=bbapp-help&article=125680' ) ), esc_html__( 'View tutorial', 'buddyboss-app' ) ),
						'type'  => 'html',
					)
				)
			);

			if ( ! bbapp_is_active( 'access_controls' ) ) {
				// Lock App if user has not purchased a product.
				$this->settings_form->add_field(
					$this->app_settings_provider->field(
						'iap.lock_app',
						array(
							'label'          => '',
							'label_checkbox' => sprintf( '%1$s <a href="%2$s">%3$s</a> %4$s', __( 'Require members to purchase an', 'buddyboss-app' ), esc_url( admin_url( 'admin.php?page=bbapp-iap&setting=products' ) ), __( 'in-app product', 'buddyboss-app' ), __( 'before viewing app content.', 'buddyboss-app' ) ),
							'desc'           => __( 'When members log into the app, display the Products Screen and require a purchase before they can view the rest of your app content. If disabled, the Products Screen can still be added to your Tab Bar for selling products.', 'buddyboss-app' ),
							'type'           => 'checkbox',
						)
					)
				);

				$product_list_html = $this->get_iap_global_product_lists();

				if ( ! empty( $product_list_html ) ) {
					$field_html  = '<p class="description">' . __( 'The following memberships and courses provide instant access to the app as they are connected to an active in-app purchase. Members with access to these will not be required to purchase a product before viewing the app\'s content.', 'buddyboss-app' ) . '</p>';
					$field_html .= $product_list_html;

					$this->settings_form->add_field(
						$this->app_settings_provider->field(
							'iap.lock_app_html',
							array(
								'label' => '&nbsp;',
								'value' => '<div class="iap-desc">' . $field_html . '</div>',
								'type'  => 'html',
								'class' => function () {
									return empty( Configure::instance()->option( 'iap.lock_app' ) ) ? 'iap-products hidden' : 'iap-products';
								},
							)
						)
					);
				}
			}

			// Purchase while registering.
			$this->settings_form->add_field(
				$this->app_settings_provider->field(
					'iap.purchase_before_register',
					array(
						'label'          => __( 'Add to Registration', 'buddyboss-app' ),
						'label_checkbox' => sprintf( '%1$s <a href="%2$s">%3$s</a> %4$s.', __( 'Require members to purchase an', 'buddyboss-app' ), esc_url( admin_url( 'admin.php?page=bbapp-iap&setting=products' ) ), __( 'in-app product', 'buddyboss-app' ), __( 'while registering for an account.', 'buddyboss-app' ) ),
						'desc'           => sprintf( '%1$s <a href="%2$s">%3$s</a>.', __( 'When registering a new account, display products during registration process, and require a purchase before registration can be completed. You can manage which products show during registration when editing your', 'buddyboss-app' ), esc_url( admin_url( 'admin.php?page=bbapp-iap&setting=products' ) ), __( 'products', 'buddyboss-app' ) ),
						'type'           => 'checkbox',
					)
				)
			);

			// Purchase while registering and show product.
			$this->settings_form->add_field(
				$this->app_settings_provider->field(
					'iap.register.show_products',
					array(
						'label'       => '',
						'pre_label'   => __( 'Show Products', 'buddyboss-app' ),
						'desc'        => '',
						'type'        => 'select',
						'options'     => array(
							'before_registration' => 'Before Registration',
							'on_registration'     => 'On Registration',
							'after_registration'  => 'After Registration',
						),
						'placeholder' => '',
						'class'       => function () {
							if ( ! $this->option( 'iap.purchase_before_register' ) ) {
								return 'hidden';
							}

							return '';
						},
					)
				)
			);

		}

		// Terms of Service.
		$this->settings_form->add_field(
			$this->app_settings_provider->field(
				'iap.terms',
				array(
					'label' => __( 'Terms of Service', 'buddyboss-app' ),
					'type'  => '\BuddyBossApp\Admin\Fields\WPPage',
					'desc'  => sprintf( '%1$s <a href="%2$s">%3$s</a> %4$s', __( 'Select a', 'buddyboss-app' ), esc_url( bbapp_get_admin_url( 'edit.php?post_type=page' ) ), __( 'WordPress Page', 'buddyboss-app' ), __( 'to display as your Terms of Service when purchasing a product. This is required by Apple and Google for selling in-app purchases.', 'buddyboss-app' ) ),
				)
			)
		);

		// Privacy Policy.
		$this->settings_form->add_field(
			$this->app_settings_provider->field(
				'iap.policy',
				array(
					'label' => __( 'Privacy Policy', 'buddyboss-app' ),
					'type'  => '\BuddyBossApp\Admin\Fields\WPPage',
					'desc'  => sprintf( '%1$s <a href="%2$s">%3$s</a> %4$s', __( 'Select a', 'buddyboss-app' ), esc_url( bbapp_get_admin_url( 'edit.php?post_type=page' ) ), __( 'WordPress Page', 'buddyboss-app' ), __( 'to display as your Privacy Policy when purchasing a product. This is required by Apple and Google for selling in-app purchases.', 'buddyboss-app' ) ),
				)
			)
		);

		if ( ! bbapp_is_connected_as_secondary_site() ) {
			// iOS Review Version.
			$builds = $this->get_build_review_versions( 'ios' );
			if ( ! empty( $builds ) ) {
				$this->settings_form->add_field(
					$this->app_settings_provider->field(
						'iap.review_version.ios',
						array(
							'label'       => __( 'iOS Review Version', 'buddyboss-app' ),
							'desc'        => __( 'You can select a specific iOS Release App build that products will show to everyone in, even when they are not approved. This will ensure they are visible to Apple\'s review team while they are reviewing the latest version of your app.', 'buddyboss-app' ),
							'type'        => 'select',
							'options'     => $this->get_build_review_versions( 'ios' ),
							'placeholder' => '',
						)
					)
				);
			} else {
				$this->settings_form->add_field(
					$this->app_settings_provider->field(
						'iap.review_version.ios',
						array(
							'label' => __( 'iOS Review Version', 'buddyboss-app' ),
							'type'  => 'html',
							'value' => sprintf( '<div class="bbapp-notice-box bbapp-notice-box--warning">%1$s <a href="%2$s">%3$s</a></div>', __( 'No builds available. please', 'buddyboss-app' ), esc_url( admin_url( 'admin.php?page=bbapp-build&setting=request-build' ) ), __( 'generate a new iOS Release App build', 'buddyboss-app' ) ),
							'class' => '',
						)
					)
				);
			}
		}

		/**
		 * Products Screen Settings
		 */
		$this->settings_form->add_field(
			$this->app_settings_provider->field(
				false,
				array(
					'type'    => 'table-open',
					'classes' => 'buddyboss-app-admin-card bbapp-admin-pricing_modal',
				)
			)
		);

		$this->settings_form->add_field(
			$this->app_settings_provider->field(
				'app_pricing_screen_heading',
				array(
					'type'  => 'heading',
					'value' => __( 'Products Screen', 'buddyboss-app' ),
					'desc'  => '',
					'icon'  => 'bb-icon-bf bb-icon-box',
				)
			)
		);

		// Subscription Message.
		$this->settings_form->add_field(
			$this->app_settings_provider->field(
				'iap.subscribe_message',
				array(
					'label' => __( 'Sales Message', 'buddyboss-app' ),
					'desc'  => sprintf( '%1$s <a href="%2$s">%3$s</a>.', __( 'Add a message at the top of the Products Screen to help sell your', 'buddyboss-app' ), esc_url( admin_url( 'admin.php?page=bbapp-iap&setting=products' ) ), __( 'products.', 'buddyboss-app' ) ),
					'type'  => 'textarea',
				)
			)
		);

	}

	/**
	 * Saves the settings which are app id specific.
	 *
	 * @param array $settings Settings data.
	 *
	 * @return bool
	 */
	public function save_settings( $settings ) {
		ManageApp::instance()->update_app_settings( $settings );
		return true;
	}

	/**
	 * Gets the settings which are app id specific.
	 */
	public function get_settings() {
		$settings = ManageApp::instance()->get_app_settings();
		return $settings;

	}

	/**
	 * Get settings - option value.
	 *
	 * @param string $field_name Field name.
	 * @param false  $default    Default value.
	 *
	 * @return false
	 */
	public function option( $field_name, $default = false ) {
		if ( ! empty( $this->app_settings_provider ) ) {
			return $this->app_settings_provider->get_value( $field_name, $default, true );
		}

		return false;
	}


	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {

		$tabs = array(
			'orders'                             => __( 'Purchases', 'buddyboss-app' ),
			'products'                           => __( 'Products', 'buddyboss-app' ),
			SubscriptionGroupTaxonomy::$taxonomy => __( 'Subscription Groups', 'buddyboss-app' ),
		);

		// NOTE : In MultiSite Network, don't show "orders" tab on network admin page(s). Does not make sense.
		if ( ! bbapp_is_admin_page() ) {
			unset( $tabs['orders'] );
		}

		$bbapp_setting = ! empty( $_GET['setting'] ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$bbapp_action  = ! empty( $_GET['action'] ) ? bbapp_input_clean( wp_unslash( $_GET['action'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'products' === $bbapp_setting && in_array(
			$bbapp_action,
			array(
				'create',
				'edit',
			),
			true
		)
		) {
			unset( $tabs['orders'] );
			unset( $tabs[ SubscriptionGroupTaxonomy::$taxonomy ] );
		}
		return $tabs;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {

		$default      = 'orders';
		$setting      = isset( $_GET['setting'] ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		// NOTE : In MultiSite Network, default can't be "orders" on network admin page(s). Don't exists.
		if ( ! bbapp_is_admin_page() ) {
			$default = 'products';
		}

		return $default;
	}

	/**
	 * Return the current action page from active screen.
	 *
	 * @return string
	 */
	public function get_current_action() {

		$default      = 'orders';
		$setting      = isset( $_GET['action'] ) ? bbapp_input_clean( wp_unslash( $_GET['action'] ) ) : $default; // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Render Setting Form.
	 */
	public function render_settings() {
		$this->settings_form->render_form();

		?>
		<script>
            jQuery( document ).ready( function() {
                jQuery( document ).on( 'change', '#bbapp_field_app_settings_iap__lock_app', function() {
                    if ( jQuery( this ).prop( 'checked' ) === true ) {
                        jQuery( '.iap-products' ).removeClass( 'hidden' );
                    } else {
                        jQuery( '.iap-products' ).addClass( 'hidden' );
                    }
                } );

                jQuery( document ).on( 'change', '#bbapp_field_app_settings_iap__purchase_before_register', function() {
                    if ( jQuery( this ).prop( 'checked' ) === true ) {
                        jQuery( '.bbapp_field_wrapper_app_settings_iap__register__show_products' ).removeClass( 'hidden' );
                    } else {
                        jQuery( '.bbapp_field_wrapper_app_settings_iap__register__show_products' ).addClass( 'hidden' );
                    }
                } );
            } );
        </script>
		<?php
	}

	/**
	 * Renders the setting screen.
	 */
	public function render_screen() {
		SetupAdmin::instance()->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab() );

		if ( 'orders' === $this->get_current_sub_tab() ) {
			OrderHelper::instance()->render();
		}

		if ( 'products' === $this->get_current_sub_tab() ) {
			ProductHelper::instance()->render();
		}

		if ( $this->get_current_sub_tab() === SubscriptionGroupTaxonomy::$taxonomy ) {
			SubscriptionGroupsHelpers::instance()->render();
		}
	}

	/**
	 * Get public iap products list as html.
	 *
	 * @return string
	 */
	public function get_iap_global_product_lists() {
		$product_args   = array( 'global_subscription' => true );
		$bbapp_products = Helpers::get_products( $product_args );

		if ( empty( $bbapp_products ) ) {
			return '';
		}

		$products = array();

		foreach ( $bbapp_products as $iap_product ) {
			if ( ! empty( $iap_product['integration_type'] ) && ! array_key_exists( $iap_product['integration_type'], $products ) ) {
				$products[ $iap_product['integration_type'] ] = array();
			}

			if ( ! empty( $iap_product['integration_type'] ) && ! empty( $iap_product['integrated_item_ids'] ) ) {
				$products[ $iap_product['integration_type'] ] = array_unique( array_merge( $products[ $iap_product['integration_type'] ], $iap_product['integrated_item_ids'] ) );
			}
		}

		$iap_integrations = Controller::instance()->get_enabled_integrations();

		$output = '';
		if ( ! empty( $products ) ) {
			ob_start();
			foreach ( $products as $integration => $product_ids ) {
				if ( ! empty( $product_ids ) && array_key_exists( $integration, $iap_integrations ) ) {
					$iap_type_products = Helpers::get_integration_items( $integration );
					$iap_type_products = array_map(
						function ( $val ) {
							$id = explode( ':', $val['id'] );

							return array(
								'id'   => $id[0],
								'text' => ( ! empty( $id[1] ) ? $id[1] : $val['text'] ),
							);
						},
						$iap_type_products
					);
					$iap_type_products = array_column( $iap_type_products, 'text', 'id' );
					echo '<li>';
						echo '<span class="integration-label">' . esc_html( $iap_integrations[ $integration ]['label'] ) . '</span>';
						echo '<ul class="sub-product-list">';
					foreach ( $product_ids as $pid ) {
						if ( array_key_exists( $pid, $iap_type_products ) ) {
							$link = apply_filters( 'bbapp_iap_integration_item_id_permalink', '#', $pid, $iap_integrations[ $integration ]['type'] );
							echo '<li><a href="' . esc_url( $link ) . '">' . esc_html( $iap_type_products[ $pid ] ) . '</a></li>';
						}
					}
						echo '</ul>';
					echo '</li>';
				}
			}
			$output = ob_get_clean();
			if ( ! empty( $output ) ) {
				ob_start();
				echo '<ul>';
				echo wp_kses_post( $output );
				echo '</ul>';
				$output = ob_get_clean();
			}
		}
		return $output;
	}

	/**
	 * Get Builds Review Version.
	 *
	 * @param string $platform    Platform like ios,android.
	 * @param bool   $purge_cache Purge cache.
	 *
	 * @return array
	 */
	public function get_build_review_versions( $platform = 'ios', $purge_cache = false ) {
		$builds_options = array();

		if ( SetupAdmin::instance()->is_current_app_connected() ) {
			$builds = \BuddyBossApp\Build::instance()->get_app_build_versions( $platform, 'live', 'completed', $purge_cache );

			if ( ! is_wp_error( $builds ) && ! empty( $builds ) ) {
				$builds_options[] = __( ' -- Select Build -- ', 'buddyboss-app' );
				foreach ( (array) $builds[ $platform ]['live'] as $build ) {
					$builds_options[ $build['build_version'] ] = $build['build_version'];
				}
			}
		}

		return $builds_options;
	}

}
