<?php
/**
 * Holds gutenberg extend functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

/**
 * GUtenberg blocks base class.
 */
abstract class GutenbergBlockAbstract {

	/**
	 * Block namespace.
	 *
	 * @var $namespace
	 */
	private $namespace;

	/**
	 * Blocks title.
	 *
	 * @var $title
	 */
	private $title;

	/**
	 * Block description.
	 *
	 * @var $description
	 */
	private $description;

	/**
	 * Icon.
	 *
	 * @var $icon
	 */
	private $icon;

	/**
	 * Keywords.
	 *
	 * @var $keywords
	 */
	private $keywords;

	/**
	 * Attributes.
	 *
	 * @var $attributes
	 */
	private $attributes;

	/**
	 * Blocks preview.
	 *
	 * @var $preview
	 */
	private $preview;

	/**
	 * Filters/hooks here.
	 */
	public function init() {
		add_action( 'admin_print_footer_scripts', array( $this, 'print_js' ) );
		add_filter( 'bbapp_disallowed_block_for_other_cpt', array( $this, 'disallowed_block_for_other_cpt' ), 10, 1 );
		add_filter(
			'bbapp_allowed_custom_block_for_app_content',
			array(
				$this,
				'allowed_custom_block_for_app_content',
			),
			10,
			1
		);
		add_filter( 'bbapp_custom_block_data', array( $this, 'update_custom_block_data' ), 10, 3 );
		add_filter( 'bbapp_app_page_app_blocks', array( $this, 'bbapp_app_page_app_blocks' ), 10, 3 );
	}

	/**
	 * Set block name as disallowed block for the other CPT like page, post, except app_page
	 *
	 * @param array $disallowed_blocks Array of not allowed blocks.
	 *
	 * @return array
	 */
	public function disallowed_block_for_other_cpt( $disallowed_blocks ) {

		if ( ! empty( $this->namespace ) ) {
			$disallowed_blocks[] = $this->namespace;
		}

		return $disallowed_blocks;
	}

	/**
	 * Set block name as allowed block for the app pages and app editors.
	 *
	 * @param array $allowed_blocks Array of allowed blocks.
	 *
	 * @return array
	 */
	public function allowed_custom_block_for_app_content( $allowed_blocks ) {

		if ( ! empty( $this->namespace ) ) {
			$allowed_blocks[] = $this->namespace;
		}

		return $allowed_blocks;
	}

	/**
	 * Set namespace for block
	 *
	 * @param string $namespace Namespace.
	 */
	public function set_namespace( $namespace ) {
		$this->namespace = $namespace;
	}

	/**
	 * Set title for block
	 *
	 * @param string $title Title.
	 */
	public function set_title( $title ) {
		$this->title = $title;
	}

	/**
	 * Set description for block.
	 *
	 * @param string $description Description.
	 */
	public function set_description( $description ) {
		$this->description = $description;
	}

	/**
	 * Set icon for block.
	 *
	 * @param array $icon Icon data.
	 */
	public function set_icon( $icon ) {
		$this->icon = $icon;
	}

	/**
	 * Set keywords for block.
	 *
	 * @param array $keywords Keywords.
	 */
	public function set_keywords( $keywords ) {
		$this->keywords = $keywords;
	}

	/**
	 * Set attributes for block.
	 *
	 * @param array $attributes Attributes.
	 */
	public function set_attributes( $attributes ) {
		$this->attributes = $attributes;
	}

	/**
	 * Set preview for block.
	 *
	 * @param string $preview Block preview.
	 */
	public function set_preview( $preview ) {
		$this->preview = $preview;
	}

	/**
	 * Get block config data.
	 *
	 * @return mixed|string|void
	 */
	private function get_config() {
		$config = array(
			'namespace'   => $this->namespace,
			'title'       => $this->title,
			'description' => $this->description,
			'icon'        => $this->icon,
			'keywords'    => $this->keywords,
			'attributes'  => $this->attributes,
			'preview'     => $this->preview,
		);

		return wp_json_encode( $config );
	}

	/**
	 * Print blog config.
	 */
	public function print_js() {
		$mobile_svg = bbapp()->plugin_url . 'assets/img/mobile.svg';
		$image_path = bbapp()->plugin_url . 'assets/img/img-path.svg';

		echo '<style>.bbapp_supported_blocks .block-editor-inserter__panel-content .editor-block-list-item-' . esc_attr( str_replace( '/', '-', $this->namespace ) ) . ':before{
		background: url(' . esc_url( $mobile_svg ) . ") no-repeat; background-size: contain; content: ''; position: absolute; width: 12px; height: 19px; top: 4px; right: 7px;}
	    .bbapp_supported_blocks .block-editor-inserter__panel-content .editor-block-list-item-" . esc_attr( str_replace( '/', '-', $this->namespace ) ) . '{
	    background: url(' . esc_url( $image_path ) . ') no-repeat top right; border-radius: 4px; border: 1px solid rgba(0, 0, 0, 0.08); margin: 0 5px 10px; width: calc(100% - 10px);}</style>';

		$config = $this->get_config();

		echo '<script type="text/javascript">';
		echo "if (typeof window.gutenberg_blocks === 'undefined') {";
		echo 'window.gutenberg_blocks = [];';
		echo '}';
		echo "window.gutenberg_blocks['" . esc_attr( $this->namespace ) . "'] = " . $config . ';'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo '</script>';
	}

	/**
	 * Return data for this block.
	 *
	 * @param array  $app_page_data App page editor data.
	 * @param array  $block_data Block data.
	 * @param string $render Rendered data.
	 *
	 * @return mixed
	 */
	public function update_custom_block_data( $app_page_data, $block_data, $render ) {
		// Return if not match block namespace.
		if ( $this->namespace !== $block_data['blockName'] ) {
			return $app_page_data;
		}

		$attributes = $block_data['attrs'];

		if ( isset( $attributes['monochrome_settings'] ) ) {
			unset( $attributes['monochrome_settings'] );
		}

		$data = $attributes;

		// Add defaults attr data when not exists.
		foreach ( $this->attributes as $attr ) {
			if ( ! isset( $data[ $attr['name'] ] ) ) {
				$default               = isset( $attr['default'] ) ? $attr['default'] : '';
				$data[ $attr['name'] ] = $default;
			}
		}

		$data = $this->get_data( $data );

		$app_page_data['data']   = $data;
		$app_page_data['result'] = $this->get_results( $attributes );

		return $app_page_data;
	}

	/**
	 * Should be used to trim unwanted data or modify them.
	 * Function can be abstract. but it's not required.
	 * Important - Don't change this function to abstract else it will become required.
	 *
	 * @param array $data Block data.
	 *
	 * @return mixed
	 */
	public function get_data( $data ) {
		return $data;
	}

	/**
	 * To be used for adding extra result information.
	 *
	 * @param array $attributes Attributes.
	 *
	 * @return mixed
	 */
	abstract public function get_results( $attributes );

	/**
	 * Set block name as allowed block for the CPT app_page.
	 *
	 * @param array $allowed_block_types Allowed block types.
	 * @param int   $post_id             Post id.
	 * @param int   $default_app_page    Default app page id.
	 *
	 * @return array
	 */
	public function bbapp_app_page_app_blocks( $allowed_block_types, $post_id, $default_app_page ) {
		$allowed_block_types[] = $this->namespace;

		return $allowed_block_types;
	}
}
