<?php
/**
 * Holds form provider functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

/**
 * Form provider class.
 */
class FormProvider {

	/**
	 * Fields.
	 *
	 * @var array $fields
	 */
	private $fields = array();

	/**
	 * Settings providers.
	 *
	 * @var array $setting_providers
	 */
	private $setting_providers = array();

	/**
	 * Provider name.
	 *
	 * @var string $provider_name
	 */
	private $provider_name = '';

	/**
	 * Notices/warning messages.
	 *
	 * @var array $messages
	 */
	private $messages = array();

	/**
	 * Field errors.
	 *
	 * @var array $field_errors
	 */
	private $field_errors = array();

	/**
	 * FormProvider constructor.
	 *
	 * @param string $provider_name Provider name.
	 */
	public function __construct( $provider_name ) {
		$this->provider_name = $provider_name;
		add_action( 'admin_notices', array( $this, 'show_messages' ), 99 );
	}

	/**
	 * Default types.
	 *
	 * @return array
	 */
	public function default_types() {
		return array(
			'text',
			'password',
			'checkbox',
			'multi-checkbox',
			'select',
			'multiple-image-picker',
			'image-picker',
			'textarea',
			'heading',
			'submit',
			'button',
			'radio',
			'tabs',
			'table-open',
		);
	}

	/**
	 * Link the Setting Provider to Form Provider.
	 *
	 * @param SettingProvider $provider Setting provider.
	 */
	public function link_setting_provider( SettingProvider $provider ) {
		$this->setting_providers[ $provider->provider_name() ] = $provider;
	}

	/**
	 * $field data provided from Setting_Provider::field
	 *
	 * @uses SettingProvider::field()
	 *
	 * @param array $field Field data.
	 *
	 * @return string
	 */
	public function add_field( $field ) {
		$key                  = $this->get_field_key( $field['name'], $field['provider'] );
		$this->fields[ $key ] = $field;

		return $key;
	}

	/**
	 * Make & return the field key based on it's properties.
	 *
	 * @param string $name     File name.
	 * @param string $provider Provide name.
	 *
	 * @return string
	 */
	public function get_field_key( $name, $provider ) {
		if ( ! $name ) {
			$name = uniqid() . time();
		}

		return "{$provider}_{$name}";
	}

	/**
	 * Nonce method.
	 *
	 * @return string
	 */
	public function get_nonce_name() {
		return 'bbapp_settings_' . $this->provider_name;
	}

	/**
	 * Renders the message on screen if generated by form provider.
	 */
	public function show_messages() {
		if ( ! empty( $this->messages ) ) {
			foreach ( $this->messages as $message ) {
				printf( '<div id="message" class="%1$s"><p>%2$s</p></div>', esc_attr( $message['type'] ), wp_kses_post( $message['message'] ) );
			}
		}
	}

	/**
	 * Close table method.
	 */
	public function close_table() {
		?>
		</tbody>
		</table>
		</div>
		<?php
	}

	/**
	 * Open table method.
	 *
	 * @param string $classes Open table class.
	 */
	public function open_table( $classes ) {
		?>
		<div class="<?php echo esc_attr( $classes ); ?>">
		<table class="form-table">
		<tbody>
		<?php
	}

	/**
	 * Renders the inputs.
	 *
	 * @param bool $save_button Save button.
	 */
	public function render_form( $save_button = true ) {
		$default_types = $this->default_types();
		$unique_id     = wp_unique_id();
		?>
		<form id="update-settings-<?php echo esc_attr( $unique_id ); ?>" method="post" class="bbapp-update-settings" enctype="multipart/form-data">
			<?php
			wp_nonce_field( $this->get_nonce_name(), $this->get_nonce_name(), false );

			$i                = 0;
			$should_end_table = false;

			foreach ( $this->fields as $option_key => $option ) {
				$i ++;

				$defaults           = array(
					'class'     => '',
					'name'      => '',
					'type'      => '',
					'label'     => '',
					'value'     => '',
					'disabled'  => false,
					'purge_url' => '',
				);
				$field              = wp_parse_args( $option, $defaults );
				$field['type']      = is_callable( $field['type'] ) && ! is_string( $field['type'] ) ? $field['type']() : $field['type'];
				$field['name_ori']  = $field['name'];
				$field['value_ori'] = $field['value'];

				if ( isset( $field['disabled'] ) ) {
					$field['disabled'] = is_callable( $field['disabled'] ) && ! is_string( $field['disabled'] ) ? $field['disabled']() : $field['disabled'];
				}

				if ( isset( $field['desc'] ) ) {
					$field['desc'] = is_callable( $field['desc'] ) && ! is_string( $field['desc'] ) ? $field['desc']() : $field['desc'];
				}

				/**
				 * Settings provider class variable.
				 *
				 * @var $provider SettingProvider
				 */
				$provider       = $this->setting_providers[ $option['provider'] ];
				$field['value'] = $provider->get_value( $field['name_ori'], $option['value'] );

				if ( isset( $option['override_value'] ) ) {
					$field['value'] = $option['override_value'];
				}

				$validation_error = false;

				if ( isset( $this->field_errors[ $provider->provider_name() ] ) && isset( $this->field_errors[ $provider->provider_name() ][ $field['name'] ] ) ) {
					$validation_error = $this->field_errors[ $provider->provider_name() ][ $field['name'] ];
					$field['value']   = '';
				}

				// Change the key name based on provider.
				$field['name'] = $provider->provider_name() . "[{$field["name"]}]";
				$field['id']   = ( $option_key );
				// Strip dots from ids.
				$field['id'] = str_replace( '.', '__', $field['id'] );
				$field['id'] = str_replace( '.', '__', $field['id'] );

				if ( ! isset( $field['placeholder'] ) ) {
					$field['placeholder'] = false;
				}

				if ( ( 1 === (int) $i && 'table-open' !== $field['type'] ) || 'table-open' === $field['type'] ) {
					if ( 1 === (int) $i ) {
						$should_end_table = true;
					}

					$table_classes = '';

					if ( 'table-open' === $field['type'] && isset( $field['classes'] ) ) {
						$table_classes = $field['classes'];
					}

					if ( 1 !== (int) $i ) {
						$this->close_table();
					}

					$this->open_table( $table_classes );

					if ( 'table-open' === $field['type'] ) {
						continue;
					}
				}

				$depends = '';

				if ( isset( $field['depends'] ) ) {
					$depends = wp_json_encode( $field['depends'] );
				}

				$is_input = true;

				if ( in_array(
					$field['type'],
					array(
						'subheading',
						'subheading_secondary',
						'heading',
						'html',
						'desc',
						'submit',
						'button',
					),
					true
				) ) {
					$is_input       = false;
					$field['value'] = $field['value_ori'];
				}
				?>

				<?php /* Input used to track which fields are being updated */ ?>

				<?php if ( ! empty( $field['name_ori'] ) && $is_input ) : ?>
					<input name="_settings[<?php echo esc_attr( $option['provider'] ); ?>][]" value="<?php echo esc_attr( $field['name_ori'] ); ?>" type="hidden"/>
				<?php endif; ?>

				<tr class="bbapp_field_wrapper bbapp_field_wrapper_<?php echo esc_attr( $field['id'] ); ?> <?php echo esc_attr( is_callable( $field['class'] ) && ! is_string( $field['class'] ) ? $field['class']() : $field['class'] ); ?>"
					data-id="<?php echo esc_attr( $field['id'] ); ?>" data-depend="<?php echo esc_attr( $depends ); ?>">
					<?php
					if ( 'heading' === $field['type'] || 'subheading' === $field['type'] ) {
						?>
						<td class="bbapp-form-html bbapp-form-html--heading" scope="row" colspan="2">
							<?php
							$include_file = bbapp()->plugin_dir . "views/settings/fields/{$field["type"]}.php";
							include $include_file;
							?>

							<?php if ( isset( $field['desc'] ) && ! empty( $field['desc'] ) ) : ?>
								<p>
									<?php echo wp_kses_post( $field['desc'] ); ?>
								</p>
							<?php endif; ?>
						</td>
						<?php
					} elseif ( 'subheading_secondary' === $field['type'] ) {
						?>
						<th scope="row"></th>
						<td class="bbapp-form-html" scope="row">
							<?php
							$include_file = bbapp()->plugin_dir . "views/settings/fields/{$field["type"]}.php";
							include $include_file;
							?>
							<?php if ( isset( $field['desc'] ) && ! empty( $field['desc'] ) ) : ?>
								<p>
									<?php echo wp_kses_post( $field['desc'] ); ?>
								</p>
							<?php endif; ?>
						</td>
						<?php
					} elseif ( 'desc' === $field['type'] ) {
						?>
						<td class="bbapp-form-html" scope="row" colspan="2">
							<p>
								<?php echo wp_kses_post( $field['value'] ); ?>
							</p>
						</td>
						<?php
					} elseif ( 'submit' === $field['type'] ) {
						?>
						<td class="bbapp-form-html" scope="row" colspan="2">
							<div class="submit">
								<div class="major-publishing-actions wp-clearfix">
									<div class="publishing-action">
										<button type="submit" name="save_settings"
												id="<?php echo esc_attr( $field['id'] ); ?>"
												class="button button-primary bbapp-save-setting-btn" <?php echo ( true === $field['disabled'] ) ? 'disabled' : ''; ?>>
											<div class="bbapp_loading dark" style="display:none"></div>
											<?php echo is_callable( $field['value'] ) && ! is_string( $field['value'] ) ? esc_html( $field['value']() ) : ( ! empty( $field['value'] ) ? esc_html( $field['value'] ) : esc_html__( 'Save Changes', 'buddyboss-app' ) ); ?>
										</button>
									</div>
								</div>
							</div>
						</td>
						<?php
					} elseif ( 'button' === $field['type'] ) {
						?>
						<td class="bbapp-form-html" scope="row" colspan="2">
							<div class="submit">
								<div class="wp-clearfix">
									<button type="button" name="save_settings" id="<?php echo esc_attr( $field['id'] ); ?>" class="button button-primary bbapp-save-setting-btn" <?php echo ( true === $field['disabled'] ) ? 'disabled' : ''; ?>>
										<div class="bbapp_loading dark" style="display:none"></div>
										<?php echo is_callable( $field['value'] ) && ! is_string( $field['value'] ) ? esc_html( $field['value']() ) : ( ! empty( $field['value'] ) ? esc_html( $field['value'] ) : esc_html__( 'Save Changes', 'buddyboss-app' ) ); ?>
									</button>
								</div>
							</div>
						</td>
						<?php
					} elseif ( 'html' === $field['type'] ) {
						$class   = ( isset( $field['class_td'] ) && ! is_callable( $field['class_td'] ) && is_string( $field['class_td'] ) ? $field['class_td'] : 'bbapp-form-html' );
						$style   = ( isset( $field['style'] ) ? $field['style'] : 'width: 250px;' );
						$colspan = ( isset( $field['colspan'] ) ? $field['colspan'] : '2' );

						if ( ! empty( $field['label'] ) ) {
							?>
							<th scope="row" style="<?php echo ! empty( $style ) ? esc_attr( $style ) : ''; ?>">
								<label><?php echo esc_html( $field['label'] ); ?></label></th>
							<?php
						}
						?>
						<td class="<?php echo esc_attr( $class ); ?>" scope="row" <?php echo( ! empty( $colspan ) ? 'colspan="' . esc_attr( $colspan ) . '"' : '' ); ?>>
							<?php
							$include_file = bbapp()->plugin_dir . "views/settings/fields/{$field["type"]}.php";
							include $include_file;
							?>

							<?php if ( isset( $field['desc'] ) && ! empty( $field['desc'] ) ) : ?>
								<p class="description bbapp-admin-description">
									<?php echo wp_kses_post( $field['desc'] ); ?>
								</p>
							<?php endif; ?>
						</td>
						<?php
					} else {
						?>
						<?php if ( empty( $field['no_label'] ) ) { ?>
							<th scope="row"><label><?php echo ! empty( $field['label'] ) ? wp_kses_post( $field['label'] ) : ''; ?></label></th>
						<?php } ?>
						<td <?php echo( ! empty( $field['no_label'] ) ? 'colspan="2"' : '' ); ?>
								class="<?php echo( ! empty( $field['no_label'] ) ? 'bbapp-form-html' : '' ); ?> <?php echo ( isset( $field['show_file_link'] ) && true === $field['show_file_link'] && ! empty( $field['value'] ) ) ? 'bbapp-form__has-preview' : ''; ?>">
							<?php if ( $validation_error ) { ?>
								<div class="bbapp-notice-box bbapp-notice-box--error"><?php echo esc_html( $validation_error ); ?></div>
								<?php
							}

							if ( isset( $field['pre_label'] ) ) {
								?>
								<label for="bbapp_field_<?php echo esc_attr( $field['id'] ); ?>" class="bbapp-field-pre-label <?php echo esc_attr( $field['id'] ); ?>">
									<?php echo wp_kses_post( $field['pre_label'] ); ?>
								</label>
								<?php
							}

							if ( in_array( $field['type'], $default_types, true ) ) {
								$include_file = bbapp()->plugin_dir . "views/settings/fields/{$field["type"]}.php";
								include $include_file;
							} else {
								$class_exists = class_exists( $field['type'] );

								if ( $class_exists ) {
									/**
									 * Field abstract class instance.
									 *
									 * @var $instance FieldAbstract Field abstract class.
									 */
									$instance = new $field['type']();
									$instance->set_value( $field['value'] );
									$instance->set_settings( $field['settings'] );
									$instance->set_field_name( $field['name'] );
									$instance->field_render();
								} else {
									esc_html_e( 'Field type not found.', 'buddyboss-app' );
								}
							}

							if ( isset( $field['post_label'] ) ) {
								?>
								<label for="bbapp_field_<?php echo esc_attr( $field['id'] ); ?>" class="bbapp-field-post-label <?php echo esc_attr( $field['id'] ); ?>">
									<?php echo wp_kses_post( $field['post_label'] ); ?>
								</label>
								<?php
							}
							?>
							<?php if ( ( isset( $field['desc'] ) && ! empty( $field['desc'] ) ) || ! empty( $field['purge_url'] ) ) : ?>
								<p class="description bbapp-admin-description">
									<?php
									echo ! empty( $field['desc'] ) ? wp_kses_post( $field['desc'] ) : '';
									if ( ! empty( $field['purge_url'] ) ) {
										echo '<a class="purge-btn ' . ( empty( $field['value'] ) ? 'disabled' : '' ) . '" href="' . esc_url( $field['purge_url'] ) . '">' . esc_html__( 'Purge Cache', 'buddyboss-app' ) . '</a>';
									}
									?>
								</p>
							<?php endif; ?>

							<?php if ( isset( $field['notice'] ) && ! empty( $field['notice'] ) ) : ?>
								<p class="bbapp_notice">
									<?php echo wp_kses_post( $field['notice'] ); ?>
								</p>
							<?php endif; ?>
						</td>
					<?php } ?>
				</tr>
				<?php
			} // end foreach

			if ( $should_end_table ) {
				$this->close_table();
			}

			if ( true === $save_button ) {
				?>
				<div class="submit">
					<div class="major-publishing-actions wp-clearfix">
						<div class="publishing-action">
							<button type="submit" name="save_settings" id="save_settings"
									class="button button-primary menu-save bbapp-save-setting-btn">
								<div class="bbapp_loading dark" style="display:none"></div> <?php esc_html_e( 'Save Changes', 'buddyboss-app' ); ?>
							</button>
						</div>
						<!-- END .publishing-action -->
					</div><!-- END .major-publishing-actions -->
				</div>
			<?php } ?>
		</form>
        <script>
            jQuery( document ).ready( function () {
                jQuery( document ).on( 'click', '.bbapp_wp_media_delete', function(e) {
                    e.preventDefault();

                    const curObj = jQuery( this );

                    curObj.parent().find('img').attr('src', '');
                    curObj.parent().find('input[type=text]').attr('value', '');
		            curObj.parent().find('.bbapp_wp_media_picker').removeClass( 'selected' );
		            curObj.parent().find('.bbapp-preview-wrp').removeClass( 'has-preview' );
                    curObj.addClass('bbapp-hide' );
                } );
            } );
            /**
             * Fields Depends Logics.
             */
            setTimeout( function() {
                /**
                 * jump to field
                 */
                if ( window.location.href.indexOf( '#goto' ) !== -1 ) {
                    jQuery( 'html, body' ).animate( { scrollTop: jQuery( '.bbapp_field_wrapper_' + window.location.href.split( '#goto-' )[ 1 ] ).offset().top - 100 }, 500, function() {
                        jQuery( '.bbapp_field_wrapper_' + window.location.href.split( '#goto-' )[ 1 ] ).addClass( 'bbapp_highlight' ); // highlight the field.
                    } );
                }
            }, 2000 );

            /**
             * BuddyBossApp Color Picker.
             */
            jQuery( '.bbapp_wp_media_picker' ).click( function( e ) {
                e.preventDefault();

                // eslint-disable-next-line prefer-const
                let imageFrame;

                //If this has already been initialized, just open it
                if ( imageFrame ) {
                    imageFrame.open();
                }

                // Define image_frame as wp.media object
                imageFrame = wp.media( {
                    title: '<?php esc_html_e( 'Select Icon', 'buddyboss-app' );?>',
                    multiple: false,
                    library: {
                        type: 'image',
                    },
                } );

                /**
                 *THE KEY BUSINESS
                 *When multiple images are selected, get the multiple attachment objects
                 *and convert them into a usable array of attachments
                 */
                imageFrame.on( 'select', function () {
                    const selection = imageFrame.state().get( 'selection' );

                    if ( selection.length > 0 ) {
                        jQuery( e.currentTarget ).addClass( 'selected' );
                        jQuery( e.currentTarget ).parent().find( '.bbapp-preview-wrp' ).addClass( 'has-preview' );
                    }
                } );

                imageFrame.on( 'close', function() {
                    const selection = imageFrame.state().get( 'selection' );

                    if ( selection.length > 0 ) {
                        jQuery( e.currentTarget ).prev().attr( 'value', selection.toJSON()[ 0 ].url );
                        jQuery( e.currentTarget ).parent().find('img').attr( 'src', selection.toJSON()[ 0 ].url );
                        jQuery( e.currentTarget ).next('.bbapp_wp_media_delete').removeClass('bbapp-hide');
                    }
                } );

                imageFrame.open();
            } );

            jQuery( '.bbapp_multiple_wp_media_picker' ).click( function( e ) {
                e.preventDefault();

                // eslint-disable-next-line prefer-const
                let imageFrame;

                //If this has already been initialized, just open it
                if ( imageFrame ) {
                    imageFrame.open();
                }

                // Define image_frame as wp.media object
                imageFrame = wp.media( {
                    title: '<?php esc_html_e( 'Select Icon', 'buddyboss-app' );?>',
                    multiple: true,
                    library: {
                        type: 'image',
                    },
                } );

                /**
                 *THE KEY BUSINESS
                 *When multiple images are selected, get the multiple attachment objects
                 *and convert them into a usable array of attachments
                 */
                imageFrame.on( 'select', function() {
                    imageFrame.state().get( 'selection' ).map(

                        function( attachment ) {
                            attachment.toJSON();
                            return attachment;
                        } );

                    //Loop through the array and do things with each attachment.

                    let i;

                    for ( i = 0; i < attachments.length; ++i ) {
                        //Sample function 2: add hidden input for each image.
                        jQuery( '.bbapp-meida-placeholder' ).last().append(
                            '<input id="bbapp-meida-image-input' + attachments[ i ].id + '" type="hidden" name="' + jQuery( e.currentTarget ).next().find( 'input' ).attr( 'name' ) + '"  value="' + attachments[ i ].id + '">',
                        );

                        //Sample function 1: add image preview.
                        jQuery( '.bbapp-meida-placeholder' ).last().append( '<div class="bbapp-meida-image-preview"><img width="50" height="50" src="' + attachments[ i ].attributes.url + '" ><span class="delete-image" >X</span></div>' );
                    }
                } );

                imageFrame.on( 'close', function() {

                } );

                imageFrame.open();
            } );

            jQuery( document ).on( 'click', '.delete-image', function() {
                jQuery( this ).parent().prev().remove();
                jQuery( this ).parent().remove();
            } );
            function bbappGetUrlParameter( sParam ) {
                let sParameterName,
                    i;

                const sPageURL = window.location.search.substring( 1 );
                const sURLVariables = sPageURL.split( '&' );

                for ( i = 0; i < sURLVariables.length; i++ ) {
                    sParameterName = sURLVariables[i].split( '=' );

                    if ( sParameterName[ 0 ] === sParam ) {
                        return sParameterName[ 1 ] === undefined ? true : decodeURIComponent( sParameterName[ 1 ] );
                    }
                }
            }

            var currentPage = bbappGetUrlParameter( 'page' );

            if ( 'undefined' === typeof currentPage ) {
                currentPage = 'bbapp-components';
            }

            if ( 'bbapp-configure' !== currentPage ) {
                jQuery( '.bbapp-update-settings' ).submit( function() {
                    jQuery( 'body .bbapp-save-setting-btn' ).prop( 'disabled', true ).find( '.bbapp_loading' ).fadeIn();
                } );
            }

            /**
             * Upload Field Type JS Code.
             */
            jQuery( 'a.bbapp-upload-field-upload-btn' ).click( function( e ) {
                e.preventDefault();

                jQuery( this ).hide();
                jQuery( this ).prev().hide();
                jQuery( this ).next().show();
            } );

            jQuery( 'a.bbapp-upload-field-delete-btn' ).click( function( e ) {
                e.preventDefault();

                jQuery( this ).closest( 'td' ).find( '.delete-field' ).val( '1' );
                jQuery( this ).closest( 'td' ).find( '.bbapp-uploaded-file-name' ).html( '' );
                jQuery( this ).closest( 'td' ).find( '.show_preview' ).hide();
                jQuery( this ).hide();
            } );

            // Change registration forms description in Sttings -> General.
            if( jQuery( '.bbapp_field_app_global_settings_app_auth__signup_form' ).length ) {
                const selectedVal = jQuery( '.bbapp_field_app_global_settings_app_auth__signup_form' ).val();

                setRegistrationFromDescription( selectedVal );

                jQuery( '.bbapp_field_app_global_settings_app_auth__signup_form' ).change( function( e ) {
                    e.preventDefault();

                    const currentVal = jQuery( this ).val();

                    setRegistrationFromDescription( currentVal );
                } );

                function setRegistrationFromDescription( val ) {
                    jQuery( '.registration_description' ).hide();
                    jQuery( '#' + val ).show();
                }
            }

            // Configure firebase tabs.
            if ( jQuery( '.bbapp-admin-tabs' ).length ) {
                loadAdminTabs();

                jQuery( '.bbapp-admin-tabs li' ).click( function() {
                    jQuery( '.bbapp-admin-tabs li' ).removeClass( 'selected' );

                    jQuery( '.bbapp-admin-tabs li' ).each( function() {
                        jQuery( '.bbapp_field_wrapper_' + jQuery( this ).attr( 'id' ).replace( '.', '__' ) ).addClass( 'hidden' );
                    } );

                    jQuery( this ).addClass( 'selected' );

                    loadAdminTabs();
                } );

                function loadAdminTabs() {
                    const selectedTab = jQuery( '.bbapp-admin-tabs' ).find( 'li.selected' ).attr( 'id' );
                    jQuery( '.bbapp_field_wrapper_' + selectedTab.replace( '.', '__' ) ).removeClass( 'hidden' );
                }
            }
        </script>
		<?php
	}

	/**
	 * Triggers Save Settings.
	 */
	public function save() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( - 1 );
		}

		$nonce_name = $this->get_nonce_name();

		if ( ! isset( $_POST[ $nonce_name ] ) ) {
			return;
		}

		if ( ! check_admin_referer( $nonce_name, $nonce_name ) ) {
			$this->messages[] = array(
				'type'    => 'error',
				'message' => __( 'Something unexpected happened. Please try again later.', 'buddyboss-app' ),
			);
		}

		/**
		 * Setting provider class variable.
		 *
		 * @var $provider SettingProvider
		 */
		foreach ( $this->setting_providers as $provider_name => $provider ) {
			$_data = ( isset( $_POST[ $provider_name ] ) ) ? $_POST[ $provider_name ] : array(); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
			$_data = stripslashes_deep( $_data );

			if ( ! is_array( $_data ) ) {
				$_data = array();
			}

			$_fields = ( isset( $_POST['_settings'] ) && isset( $_POST['_settings'][ $provider_name ] ) ) ? bbapp_input_clean( wp_unslash( $_POST['_settings'][ $provider_name ] ) ) : array(); //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( ! is_array( $_fields ) ) {
				$_fields = array();
			}

			$settings = $provider->get_settings(); // get the previous saved values.

			foreach ( $_fields as $_field ) {
				if ( false === $_field || 'false' === $_field ) {
					continue;
				}

				$key   = $this->get_field_key( $_field, $provider_name );
				$value = ( isset( $_data[ $_field ] ) ) ? $_data[ $_field ] : null;
				$field = ( isset( $this->fields[ $key ] ) ) ? $this->fields[ $key ] : null;

				if ( is_null( $field ) ) {
					continue;
				}

				$field['value'] = $provider->get_value( $field['name'], $field['value'] );
				$field['type']  = is_callable( $field['type'] ) && ! is_string( $field['type'] ) ? $field['type']() : $field['type'];

				if ( 'checkbox' === $field['type'] ) {
					if ( is_null( $value ) ) {
						$value = false;
					}

					if ( ! is_int( $value ) && ! is_bool( $value ) ) {
						$value = (int) $value; // Force int if string given.
					}
				}

				if ( 'multiple-image-picker' === $field['type'] ) {
					$value = array_filter( $value );
				}

				/**
				 * Trim the value to max length if max length is provided.
				 */
				if ( isset( $field['maxlength'] ) && 'select' !== $field['type'] ) {
					$value = ( function_exists( 'mb_substr' ) ) ? mb_substr( $value, 0, intval( $field['maxlength'] ), 'UTF-8' ) : bbapp_utf8_substr( $value, 0, intval( $field['maxlength'] ) );
				}

				if ( ! empty( $value ) && isset( $field['filter'] ) ) {
					$value = $field['filter']( $value );
				}

				/**
				 * Validation.
				 */
				if ( ! empty( $value ) && isset( $field['validation'] ) ) {
					$validation_val = $field['validation']( $value, $field['value'] );

					if ( is_wp_error( $validation_val ) ) {
						/**
						 * Error class.
						 *
						 * @var $validation_val \WP_Error
						 */
						$this->field_errors[ $provider_name ][ $field['name'] ] = $validation_val->get_error_message();
						// if validation have error skip this field.
						continue;
					} else {
						$value = $validation_val;
					}
				}

				// if it's a class based custom fields.
				if ( false !== strpos( $field['type'], 'BuddyBossApp' ) ) {
					if ( class_exists( $field['type'] ) ) {
						/**
						 * Class variable.
						 **
						 * Field abstract class instance.
						 *
						 * @var $instance FieldAbstract Field abstract class.
						 */
						$instance = new $field['type']();
						$instance->set_settings( $field['settings'] );
						$instance->set_value( $field['value'] );
						$instance->set_field_name( "{$provider_name}[{$field["name"]}]" );
						$value = $instance->field_save( $value );

					}
				}

				$settings[ $_field ] = ( is_string( $value ) ? trim( $value ) : $value );
			}

			$provider->save_settings( $settings );
		}

		global $success_message;

		if ( empty( $success_message ) ) {
			$success_message  = true;
			$this->messages[] = array(
				'type'    => 'updated',
				'message' => __( 'Settings saved.', 'buddyboss-app' ),
			);
		}
	}
}
