<?php
/**
 * Holds select dropdown related functionality.
 *
 * @package BuddyBossApp\Admin\Fields
 */

namespace BuddyBossApp\Admin\Fields;

use BuddyBossApp\Admin\FieldAbstract;
use BuddyBossApp\Helpers\BBAPP_File;
use function GuzzleHttp\Psr7\str;

/**
 * Class Upload
 *
 * Available settings.
 * allowed_ext = array();
 * upload_dir = ''; default 'bbapp-upload'
 *
 * @package BuddyBossApp\Admin\Fields
 */
class Upload extends FieldAbstract {
	/**
	 * Settings id.
	 *
	 * @var string $setting_id Settings id.
	 */
	public $setting_id = '';

	/**
	 * Upload constructor.
	 */
	public function __construct() {
		$this->setting_id = uniqid();
	}

	/**
	 * Get select dropdown settings.
	 *
	 * @return array
	 */
	public function get_settings() {
		$settings = parent::get_settings();

		if ( ! is_array( $settings ) ) {
			$settings = array();
		}

		return $settings;
	}

	/**
	 * Render select dropdown field.
	 */
	public function render() {
		echo '<div class="bbapp-upload-field">';

		$field_name      = $this->get_field_name();
		$value           = $this->get_value();
		$upload_btn_text = esc_html__( 'Upload File', 'buddyboss-app' );

		if ( ! empty( $value ) ) {
			$upload_btn_text = esc_html__( 'Replace File', 'buddyboss-app' );

			// Paste after trimming hash value.
			echo '<span class="bbapp-uploaded-file-name">' . esc_html( preg_replace( '/(.*?)-(.*)/', '$2', basename( $value ) ) ) . '</span>';
		}

		$settings = $this->get_settings();

		if ( ! empty( $settings['upload_btn_text'] ) && ! isset( $settings['show_file_link'] ) ) {
			$upload_btn_text = $settings['upload_btn_text'];
		}

		if ( ! isset( $settings['allowed_ext'] ) ) {
			$settings['allowed_ext'] = array();
		}

		$upload_field_name = 'upload_' . uniqid() . wp_rand( 99, 9999 ); // unique id.

		$extentions = implode( ', ', $settings['allowed_ext'] );

		if ( isset( $settings['show_file_link'] ) && true === $settings['show_file_link'] && ! empty( $value ) ) {
			$upload_dir = wp_upload_dir();
			$url        = $upload_dir['baseurl'];
			$file_url   = $url . $value;
			$file_name  = trim( substr( $value, strrpos( $value, '/' ) + 1 ) );

			printf( '<p class="show_preview"><span class="bbapp-notice-box bbapp-notice-box--info"><a href="%1$s" target="_blank" class="show_preview__name">%2$s</a><a href="#" class="bbapp-upload-field-delete-btn" title="%3$s"><span class="dashicons dashicons-no"></span><span></a></p>', esc_url( $file_url ), esc_html( $file_name ), esc_html__( 'Delete', 'buddyboss-app' ) );
		}
		?>
		<a href="#" class="bbapp-upload-field-upload-btn button"><?php echo esc_html( $upload_btn_text ); ?></a>
		<input style="display:none" name="<?php echo esc_attr( "{$upload_field_name}" ); ?>" type='file'/>
		<input name="<?php echo esc_attr( "{$field_name}" ); ?>[value]" type='hidden' value="<?php echo esc_attr( $value ); ?>"/>
		<input name="<?php echo esc_attr( "{$field_name}" ); ?>[delete]" type='hidden' value="0" class="delete-field"/>
		<input name="<?php echo esc_attr( "{$field_name}" ); ?>[field_name]" type='hidden' value="<?php echo esc_attr( $upload_field_name ); ?>"/>
		<p style="font-size:12px">
			<?php
			/* translators: %s: Allowed extensions. */
			printf( esc_html__( 'Allowed file extension(s): %s', 'buddyboss-app' ), esc_html( $extentions ) );
			?>
		</p>
		<?php

		echo '</div>';
	}

	/**
	 * Filtering the data and everything will go there :).
	 *
	 * @param array $post_value Upload field post value.
	 *
	 * @return mixed
	 */
	public function save( $post_value ) {
$field_name_slug = str_replace( '-', '_', sanitize_title( str_replace( '[', '_', $this->get_field_name() ) ) );

		// Create key from the setting name and setup into global to check the extension invalid.
		$field_name_slug_invalid_file             = $field_name_slug . '_invalid_file';
		$GLOBALS[ $field_name_slug_invalid_file ] = false;
		$upload_name                 = $post_value['field_name'];
		$settings                    = $this->get_settings();

		if ( ! isset( $settings['allowed_ext'] ) ) {
			$settings['allowed_ext'] = array();
		}

		$old_value = $this->get_value();

		// When request to delete file.
		if ( isset( $post_value['delete'] ) && '1' === (string) $post_value['delete'] ) {
			$old_file   = $old_value;
			$upload_dir = wp_upload_dir();

			if ( ! empty( $old_file ) && file_exists( $upload_dir['basedir'] . '/' . $old_file ) ) {
				wp_delete_file( $upload_dir['basedir'] . '/' . $old_file );
			}

if ( empty( $upload['tmp_name'] ) ) {			return '';
		}}

		// New File Update.
		if ( ! empty( $_FILES[ $upload_name ]['tmp_name'] ) ) {
			$upload     = map_deep( $_FILES[ $upload_name ], 'sanitize_text_field' ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
			$upload_dir = wp_upload_dir();
			$dir        = 'bbapp/uploads';

			if ( isset( $settings['upload_dir'] ) ) {
				$dir = $settings['upload_dir'];
			}

			$dir = "{$upload_dir['basedir']}/{$dir}";

			BBAPP_File::create_dir( $dir );

			// Securing the upload directory from indexing.
			if ( ! file_exists( $dir . '/index.html' ) ) {
				touch( $dir . '/index.html' );
			}

			$salt = bbapp_generate_password( 30 );
			$hash = hash( 'sha1', microtime() . uniqid() . wp_rand() . $salt );

			$file_name = $hash . '-' . $upload['name'];

			// Delete the old file.
			$old_file = $old_value;

			if ( ! empty( $old_file ) && file_exists( $upload_dir['basedir'] . '/' . $old_file ) ) {
				wp_delete_file( $upload_dir['basedir'] . '/' . $old_file );
			}

			$upload_file = $dir . '/' . $file_name;

			// File Extension Validation.
			$ext = pathinfo( $file_name, PATHINFO_EXTENSION );

			if ( ! in_array( $ext, $settings['allowed_ext'], true ) ) {
				$GLOBALS[ $field_name_slug_invalid_file ] = true; // Setup global variable to true when the invalid file extension.
				return $old_value;
			}

			if ( ! empty( $settings['validation'] ) && is_callable( $settings['validation'] ) ) {
				$validation = $settings['validation']( $settings, $upload, $this );
				if ( is_wp_error( $validation ) ) {
					$GLOBALS[ $field_name_slug . '_validation' ] = $validation;

					return '';
				}
			}

			if ( move_uploaded_file( $upload['tmp_name'], $upload_file ) ) {

				// Keep only relative path to upload dir.
				$upload_file = str_replace( $upload_dir['basedir'], '', $upload_file );

				return $upload_file;
			}
		}

		return $old_value; // return old value if new was nt uploaded.
	}
}
