<?php
/**
 * Holds configuration related functionality for IOS and Android.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\Admin\Build\Build as InnerBuild;
use BuddyBossApp\Admin\Build\iOSHelper;
use BuddyBossApp\AppStores\Android;
use BuddyBossApp\AppStores\Apple;
use BuddyBossApp\Helpers\BBAPP_File;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Notification\Push;
use BuddyBossApp\Permissions;
use MingYuanYun\AppStore\Exceptions\ConfigException;

/**
 * Class Configure
 *
 * @package BuddyBossApp\Admin
 * @todo    : Update all configure related settings where they used. example setting_apple_iap_secret_key -> publish.ios.private_key
 */
class Configure {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Settings form.
	 *
	 * @var $settings_form
	 */
	private $settings_form;

	/**
	 * Settings form provider.
	 *
	 * @var $settings_provider
	 */
	private $settings_provider;

	/**
	 * IOS assets link.
	 *
	 * @var string $bbapp_ios_assetlinks
	 */
	private $bbapp_ios_assetlinks;

	/**
	 * The iOS assets alt.
	 *
	 * @var string $bbapp_ios_assetlinks_alt
	 */
	private $bbapp_ios_assetlinks_alt;

	/**
	 * Android assets link.
	 *
	 * @var string $bbapp_android_assetlinks
	 */
	private $bbapp_android_assetlinks;

	/**
	 * Form provider for Android.
	 *
	 * @var $settings_form_android_connect
	 */
	private $settings_form_android_connect;

	/**
	 * Form provider for IOS.
	 *
	 * @var $settings_form_ios_connect
	 */
	private $settings_form_ios_connect;

	/**
	 * Form provider for IOS configuration.
	 *
	 * @var $settings_form_ios_configuration
	 */
	private $settings_form_ios_configuration;

	/**
	 * Get the instance of the class.
	 *
	 * @return Configure
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Configure construct.
	 */
	public function __construct() {
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		add_action( 'init', array( $this, 'load_init' ), 9 ); // make the setting available before most of code uses it.
		add_action( 'wp_ajax_save_ios_devices', array( $this, 'bbapp_save_ios_devices' ) );
		add_action( 'wp_ajax_nopriv_save_ios_devices', array( $this, 'bbapp_save_ios_devices' ) );

		add_action( 'wp_ajax_update_ios_qr_code', array( $this, 'bbapp_update_ios_qr_code' ) );
		add_action( 'wp_ajax_nopriv_update_ios_qr_code', array( $this, 'bbapp_update_ios_qr_code' ) );
		add_action( 'wp_ajax_refresh_ios_devices', array( $this, 'bbapp_refresh_ios_devices' ) );
		add_action( 'wp_ajax_nopriv_refresh_ios_devices', array( $this, 'bbapp_refresh_ios_devices' ) );

		add_action( 'wp_ajax_bbapp_configure_branch_refresh', array( $this, 'bbapp_configure_branch_refresh' ) );
		add_action( 'wp_ajax_disconnect_apple_account', array( $this, 'bbapp_disconnect_apple_account' ) );
		add_action( 'wp_ajax_disconnect_android_account', array( $this, 'bbapp_disconnect_android_account' ) );
		add_action(
			'wp_ajax_bbapp_configure_custom_repo_disconnect',
			array(
				$this,
				'bbapp_configure_custom_repo_disconnect',
			)
		);
		add_action( 'admin_notices', array( $this, 'bbapp_google_firebase_v1_notice' ), 99 );
		add_filter( 'send_notification_send_push', array( $this, 'maintenance_mode_disable_push_notifications' ), 99, 2 );
	}

	/**
	 * Init function.
	 *
	 * @throws ConfigException Apple configuration exception error message.
	 */
	public function load_init() {
		InnerBuild::instance(); // Initiate the instance.

		$this->settings_provider = new SettingProvider( 'app_settings' ); // holds app settings storage handle.
		$this->settings_provider->get_setting_hook(
			array(
				$this,
				'get_settings',
			)
		); // register app settings storing & receiving.

		$build_link = bbapp_get_super_admin_url( 'admin.php?page=bbapp-build&setting=request-build' );
		$page_get   = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'admin.php' === SetupAdmin::instance()->get_page_now() && 'bbapp-configure' === $page_get ) {
			// Purge Cache.
			if ( isset( $_POST ) && ! empty( $_POST ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Missing
				// delete the release provisioning profiles so it get trigger.
				bbapp_set_network_option( '_bbapp_automatic_provisioning_profiles_triggered', '0', false );
				bbapp_set_network_option( '_bbapp_automatic_cert_triggered', '0', false );
			}

			$this->settings_provider->save_setting_hook( array( $this, 'save_settings' ) );

			// General settings form provide.
			$this->settings_form = new FormProvider( 'settings' );
			$this->settings_form->link_setting_provider( $this->settings_provider ); // Link Provider In Form Provider.
			add_action( 'admin_init', array( $this->settings_form, 'save' ), 10 );

			// iOS Connect settings form provide.
			$this->settings_form_ios_connect = new FormProvider( 'settings_ios_connect' );
			$this->settings_form_ios_connect->link_setting_provider( $this->settings_provider ); // Link Provider In Form Provider.
			add_action( 'admin_init', array( $this->settings_form_ios_connect, 'save' ) );

			// iOS App bundle ids settings form provide.
			$this->settings_form_ios_configuration = new FormProvider( 'settings_ios_configuration' );
			$this->settings_form_ios_configuration->link_setting_provider( $this->settings_provider ); // Link Provider In Form Provider.
			add_action( 'admin_init', array( $this->settings_form_ios_configuration, 'save' ) );

			// iOS App bundle ids settings form provide.
			$this->settings_form_android_connect = new FormProvider( 'settings_android_connect' );
			$this->settings_form_android_connect->link_setting_provider( $this->settings_provider ); // Link Provider In Form Provider.
			add_action( 'admin_init', array( $this->settings_form_android_connect, 'save' ) );

			// holds app settings storage handle.
			$app_settings_provider = new SettingProvider( 'app_settings' );

			// register app settings storing & receiving.
			$app_settings_provider->save_setting_hook( array( $this, 'save_settings' ) );
			$app_settings_provider->get_setting_hook( array( $this, 'get_settings' ) );

			/**
			 * General.
			 */
			if ( 'configure' === $this->get_current_sub_tab() || 'all-builds' === $this->get_current_sub_tab() ) {
				InnerBuild::instance()->will_render( true );
			}

			/**
			 * General Related Settings.
			 */
			if ( 'general' === $this->get_current_sub_tab() ) {
				$this->settings_form->add_field(
					$this->settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_publish_general_heading',
						array(
							'type'         => 'heading',
							'value'        => __( 'General Configuration', 'buddyboss-app' ),
							'desc'         => '',
							'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=122292' ) ),
							'icon'         => 'bb-icon-bf bb-icon-cog',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.general.appname',
						array(
							'label'       => __( 'App Icon Name', 'buddyboss-app' ),
							'desc'        => __( 'Your app name is the name used next to your app icon on your member\'s device. App names are limited to 30 characters, but only 12 characters will show under your icon on the home screen of your member\'s devices.', 'buddyboss-app' ) . '<div class="bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require new iOS and Android builds.', 'buddyboss-app' ) ) . '<p></div>',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
							'maxlength'   => 30,
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_publish_general_min_app_version_subheading',
						array(
							'type'  => 'subheading',
							'value' => __( 'Minimum Build Version', 'buddyboss-app' ),
							'desc'  => '',
							'class' => 'configure-subheadline',
						)
					)
				);

				$ios_android_builds = \BuddyBossApp\Build::instance()->get_app_build_versions(
					array(
						'ios',
						'android',
					)
				);

				if ( ! is_wp_error( $ios_android_builds ) && ( empty( $ios_android_builds['ios']['live'] ) || empty( $this->option( 'publish.ios.namespace' ) ) ) ) {
					$this->save_option( 'publish.general.ios.min_app', '' );
				}

				$ios_minimum_app_versions = array(
					'label' => __( 'iOS', 'buddyboss-app' ),
					'type'  => ( ( SetupAdmin::instance()->is_current_app_connected() && ! is_wp_error( $ios_android_builds ) && ! empty( $ios_android_builds['ios'] ) && ! empty( $ios_android_builds['ios']['live'] ) ) ? 'select' : 'html' ),
					'value' => '',
				);

				if ( 'select' === $ios_minimum_app_versions['type'] ) {
					$ios_array_options[''] = __( '-- Select Minimum Version --', 'buddyboss-app' );

					if ( ! empty( $ios_android_builds['ios']['live'] ) ) {
						foreach ( $ios_android_builds['ios']['live'] as $build ) {
							$ios_array_options[ $build['build_version_code'] ] = $build['build_version'] . ' (' . $build['build_version_code'] . ')';
						}
					}

					$ios_minimum_app_versions['options'] = $ios_array_options;
				}

				if ( 'html' === $ios_minimum_app_versions['type'] ) {
					$ios_minimum_app_versions['value'] = __( 'You must have generated at least one iOS Release App build to set a minimum version.', 'buddyboss-app' );
				}

				// min_version_code.ios.
				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.general.ios.min_app',
						$ios_minimum_app_versions
					)
				);

				if ( ! is_wp_error( $ios_android_builds ) && ( empty( $ios_android_builds['android']['live'] ) || empty( $this->option( 'publish.android.namespace' ) ) ) ) {
					$this->save_option( 'publish.general.android.min_app', '' );
				}

				$android_minimum_app_versions = array(
					'label' => __( 'Android', 'buddyboss-app' ),
					'desc'  => __( 'If your members try to use an app build previous to the one specified here, they will be prompted to update your app before they can use it again.', 'buddyboss-app' ),
					'type'  => ( ( SetupAdmin::instance()->is_current_app_connected() && ! is_wp_error( $ios_android_builds ) && ! empty( $ios_android_builds['android'] ) && ! empty( $ios_android_builds['android']['live'] ) && ! empty( $this->option( 'publish.android.namespace' ) ) ) ? 'select' : 'html' ),
				);

				if ( 'select' === $android_minimum_app_versions['type'] ) {
					$android_array_options[''] = __( '-- Select Minimum Version --', 'buddyboss-app' );

					if ( ! is_wp_error( $ios_android_builds ) && ! empty( $ios_android_builds['android']['live'] ) ) {
						foreach ( $ios_android_builds['android']['live'] as $build ) {
							$android_array_options[ $build['build_version_code'] ] = $build['build_version'] . ' (' . $build['build_version_code'] . ')';
						}
					}

					$android_minimum_app_versions['options'] = $android_array_options;
				}

				if ( 'html' === $android_minimum_app_versions['type'] ) {
					$android_minimum_app_versions['value'] = __( 'You must have generated at least one Android Release App build to set a minimum version.', 'buddyboss-app' );
				}

				// min_version_code.android.
				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.general.android.min_app',
						$android_minimum_app_versions
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_maintenance_heading',
						array(
							'type'         => 'heading',
							'value'        => __( 'Maintenance Mode', 'buddyboss-app' ),
							'desc'         => '',
							'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=126797' ) ),
							'icon'         => 'bb-icon-bf bb-icon-tools',
						)
					)
				);

				// Enable Maintenance mode.
				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_maintenance_mode',
						array(
							'label'          => __( 'Maintenance Mode', 'buddyboss-app' ),
							'label_checkbox' => __( 'Enable in app maintenance mode', 'buddyboss-app' ),
							'desc'           => esc_html__( 'When enabled, all app access will be restricted and a maintenance screen will be displayed.', 'buddyboss-app' ),
							'type'           => 'checkbox',
						)
					)
				);

				if ( ! empty( $this->has_custom_developer_access() ) ) {

					$this->settings_form->add_field(
						$this->settings_provider->field(
							false,
							array(
								'type'    => 'table-open',
								'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
							)
						)
					);

					$this->settings_form->add_field(
						$this->settings_provider->field(
							'app_publish_general_custom_development',
							array(
								'type'         => 'heading',
								'value'        => __( 'Development', 'buddyboss-app' ),
								'desc'         => '',
								'tutorial_url' => admin_url( 'admin.php?page=bbapp-help&article=123889' ),
								'icon'         => 'bb-icon-bf bb-icon-code',
							)
						)
					);

					$this->settings_form->add_field(
						$this->settings_provider->field(
							'app_publish_general_core_development_subheading',
							array(
								'type'  => 'subheading',
								'value' => __( 'Core App Branches', 'buddyboss-app' ),
								'desc'  => '<p>' . __( 'Select which branches of the core app to build your app from.', 'buddyboss-app' ) . '</p><p>' . __( 'The <strong>Production</strong> branch is the current stable release and the default branch. The <strong>Pre-Release</strong> branch includes all the latest stable changes that are not yet live. The <strong>Development</strong> branch has non-reviewed ongoing development work that may not be stable yet, so we recommend it should only be used by developers.', 'buddyboss-app' ) . '</p>',
								'class' => 'configure-subheadline',
							)
						)
					);

					$this->settings_form->add_field(
						$this->settings_provider->field(
							'publish.general.core.branch',
							array(
								'label'   => __( 'Release App', 'buddyboss-app' ),
								'type'    => 'select',
								'options' => $this->get_core_development_branch( false, 'publish.general.core.branch' ),
								'value'   => '',
								'desc'    => '',
							)
						)
					);

					$this->settings_form->add_field(
						$this->settings_provider->field(
							'publish.general.core.dev_branch',
							array(
								'label'   => __( 'Test App', 'buddyboss-app' ),
								'type'    => 'select',
								'options' => $this->get_core_development_branch( false, 'publish.general.core.dev_branch' ),
								'value'   => $this->get_core_access_branches(),
								'desc'    => '</p><div class="bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require new iOS and Android builds.', 'buddyboss-app' ) ) . '<p></div>',
							)
						)
					);

					$this->settings_form->add_field(
						$this->settings_provider->field(
							'app_publish_general_custom_development_subheading',
							array(
								'type'  => 'subheading',
								'value' => __( 'Custom Git Branches', 'buddyboss-app' ),
								'desc'  => __( 'Select branches from your custom Git repo to automatically merge into your app builds.', 'buddyboss-app' ),
								'class' => 'configure-subheadline',
							)
						)
					);

					if ( empty( $this->get_custom_development_branch() ) ) {
						$this->settings_form->add_field(
							$this->settings_provider->field(
								'app_publish_general_custom_dev_field',
								array(
									'label' => __( 'Git Repository', 'buddyboss-app' ),
									'type'  => 'html',
									'value' => sprintf( '<p>%1$s <a href="%2$s" target="_blank">%3$s</a> %4$s</p>', esc_html__( 'Add your Git repo in your', 'buddyboss-app' ), $this->user_account_link(), esc_html__( 'BuddyBoss account area', 'buddyboss-app' ), esc_html__( 'to enable merging custom code from a Git branch into your Release App and Test App builds.', 'buddyboss-app' ) ),
									'desc'  => '',
								)
							)
						);

						$this->settings_form->add_field(
							$this->settings_provider->field(
								'publish.general.dev.information',
								array(
									'label' => __( 'Git Connection', 'buddyboss-app' ),
									'type'  => 'html',
									'value' => '<p><button class="button button-secondary custom-dev-refresh-data"><div class="bbapp_loading" style="display:none;"></div>' . __( 'Refresh Data', 'buddyboss-app' ) . '</button> &nbsp; </p><div class="bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require new iOS and Android builds.', 'buddyboss-app' ) ) . '<p></div>',
								)
							)
						);

					} else {

						$this->settings_form->add_field(
							$this->settings_provider->field(
								'publish.general.branch',
								array(
									'label'   => __( 'Release App', 'buddyboss-app' ),
									'type'    => 'select',
									'options' => $this->get_custom_development_branch(),
									'value'   => '',
									'desc'    => '',
								)
							)
						);

						$this->settings_form->add_field(
							$this->settings_provider->field(
								'publish.general.dev.branch',
								array(
									'label'   => __( 'Test App', 'buddyboss-app' ),
									'type'    => 'select',
									'options' => $this->get_custom_development_branch(),
									'value'   => '',
									'desc'    => sprintf(
									/* translators: 1. Setting up your Git repo. 2. Deploying Custom Code. */
										__( 'Select a Git branch to be automatically merged into your Release App and Test App builds. Make sure to view the tutorials for <a href="%1$s" target="_blank">setting up your Git repo</a>, and for <a href="%2$s" target="_blank">deploying custom code</a> without having to create new builds.', 'buddyboss-app' ),
										'https://www.buddyboss.com/resources/dev-docs/app-development/app-development-environment/setting-up-your-git-repo/',
										'https://www.buddyboss.com/resources/dev-docs/app-development/app-development-environment/deploying-custom-code/'
									),
								)
							)
						);

						$this->settings_form->add_field(
							$this->settings_provider->field(
								'publish.general.dev.information',
								array(
									'label' => __( 'Git Connection', 'buddyboss-app' ),
									'type'  => 'html',
									'value' => '<p>' . ( ! empty( $this->user_account_link() ) ? '<a class="button button-secondary custom-dev-git-setting" href="' . $this->user_account_link() . '" target="_blank">' . __( 'Git Settings', 'buddyboss-app' ) . '</a> &nbsp; ' : '' ) . '<button class="button button-secondary custom-dev-refresh-data"><div class="bbapp_loading" style="display:none;"></div>' . __( 'Refresh Data', 'buddyboss-app' ) . '</button> &nbsp; <button class="button button-secondary custom-dev-disconnect-data"><div class="bbapp_loading" style="display:none;"></div>' . __( 'Disconnect', 'buddyboss-app' ) . '</button></p><div class="bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require new iOS and Android builds.', 'buddyboss-app' ) ) . '<p></div>',
								)
							)
						);
					}
				}
			}

			/**
			 * IOS Related Settings.
			 */
			if ( $this->get_current_sub_tab() === 'ios' ) {
				if ( ! ManageApp::instance()->is_current_secondary_site() ) {
					$this->settings_form_ios_connect->add_field(
						$this->settings_provider->field(
							false,
							array(
								'type'    => 'table-open',
								'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
							)
						)
					);

					$this->settings_form_ios_connect->add_field(
						$this->settings_provider->field(
							'app_config_ios_account_settings',
							array(
								'type'         => 'heading',
								'value'        => function () {
									$value = __( 'Connect Your Apple Developer Account', 'buddyboss-app' );
									if ( Apple::instance()->is_connected() ) {
										return $value . sprintf( "<span class='account-connect-status connected'>%s</span>", __( 'Connected', 'buddyboss-app' ) );
									}

									return $value . sprintf( "<span  class='account-connect-status not-connected'>%s</span>", __( 'Not Connected', 'buddyboss-app' ) );
								},
								'desc'         => __( 'Please connect your Apple Developer account to enable using iOS in-app products and publishing the iOS version of your app via our publishing service. Once connected, you\'ll be able to automatically generate signing certificates and provisioning profiles.', 'buddyboss-app' ),
								'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=122274' ) ),
								'icon'         => 'bb-icon-bf bb-icon-brand-apple',
							)
						)
					);

					$this->settings_form_ios_connect->add_field(
						$this->settings_provider->field(
							'app_publish_ios_connection_status',
							array(
								'type'  => 'html',
								'value' => function () {
									if ( Apple::instance()->is_connected() ) {
										$key_identifier = Configure::instance()->option( 'publish.ios.key_id' );

										return '<div class="bbapp-account-connected bbapp-notice-box">' . __( 'Connected with Key ID: ', 'buddyboss-app' ) . $key_identifier . '</div>';
									}

									return '';
								},
								'desc'  => '',
								'class' => function () {
									if ( ! Apple::instance()->is_connected() && empty( Apple::instance()->get_error_message() ) ) {
										return 'hidden';
									}

									return '';
								},
							)
						)
					);

					// setting_apple_iap_secret_key.
					$this->settings_form_ios_connect->add_field(
						$this->settings_provider->field(
							'publish.ios.private_key',
							array(
								'label'    => __( 'Apple API Key', 'buddyboss-app' ),
								'type'     => '\BuddyBossApp\Admin\Fields\Upload',
								'value'    => '',
								'desc'     => __( 'To connect your account you will need to generate a new API key. Log into your <a href="https://itunesconnect.apple.com/" target="_blank">App Store Connect</a> account and click Users and Access. In the Keys tab, generate a new API key and upload it here.', 'buddyboss-app' ),
								'settings' => array(
									'allowed_ext'    => array( 'p8' ),
									'upload_dir'     => 'bbapp/uploads',
									'show_file_link' => true,
								),
								'class'    => function () {
									if ( Apple::instance()->is_connected() ) {
										return 'hidden';
									}

									return 'bbapp_is_required';
								},
							)
						)
					);

					// setting_apple_iap_issuer_id.
					$this->settings_form_ios_connect->add_field(
						$this->settings_provider->field(
							'publish.ios.issuer_id',
							array(
								'label' => __( 'Issuer ID', 'buddyboss-app' ),
								'type'  => 'password',
								'value' => '',
								'desc'  => __( 'Enter the <strong>Issuer ID</strong> for your API key.', 'buddyboss-app' ),
								'class' => function () {
									if ( Apple::instance()->is_connected() ) {
										return 'hidden';
									}

									return 'bbapp_is_required';
								},
							)
						)
					);

					// setting_apple_iap_key_identifier.
					$this->settings_form_ios_connect->add_field(
						$this->settings_provider->field(
							'publish.ios.key_id',
							array(
								'label' => __( 'Key ID', 'buddyboss-app' ),
								'type'  => 'password',
								'value' => '',
								'desc'  => __( 'Enter the <strong>Key ID</strong> for your API key.', 'buddyboss-app' ),
								'class' => function () {
									if ( Apple::instance()->is_connected() ) {
										return 'hidden';
									}

									return 'bbapp_is_required';
								},
							)
						)
					);

					$this->settings_form_ios_connect->add_field(
						$this->settings_provider->field(
							'app_publish_ios_connection_submit',
							array(
								'type'  => function () {
									if ( Apple::instance()->is_connected() ) {
										return 'button';
									}

									return 'submit';
								},
								'value' => function () {
									if ( Apple::instance()->is_connected() ) {
										return __( 'Disconnect Account', 'buddyboss-app' );
									}

									return __( 'Connect Account', 'buddyboss-app' );
								},
								'desc'  => '',
								'class' => function () {
									if ( Apple::instance()->is_connected() ) {
										return 'bbapp_apple_connected';
									} else {
										return 'bbapp_apple_disconnected';
									}
								},
							)
						)
					);

					$this->settings_form_ios_connect->add_field(
						$this->settings_provider->field(
							'app_publish_ios_connection_errors',
							array(
								'type'  => 'html',
								'value' => function () {
									global $app_settings_publish_ios_private_key_invalid_file;

									if ( true === $app_settings_publish_ios_private_key_invalid_file ) {
										return '<div class="bbapp-account-not-connected bbapp-notice-box bbapp-notice-box--error">' . __( 'Please enter a valid API key (.p8 file) generated from App Store Connect.', 'buddyboss-app' ) . '</div>';
									}

									$error = Apple::instance()->get_error_message();

									if ( ! empty( $error ) ) {
										Apple::instance()->apple_connect_error = false;

										return '<div class="bbapp-account-not-connected bbapp-notice-box bbapp-notice-box--error">' . $error . '</div>';
									}

									return '';
								},
								'desc'  => '',
								'class' => function () {
									global $app_settings_publish_ios_private_key_invalid_file;
									if ( ! Apple::instance()->is_connected() && empty( Apple::instance()->get_error_message() ) && false === $app_settings_publish_ios_private_key_invalid_file ) {
										return 'hidden';
									}

									return '';
								},
							)
						)
					);

					// IOS Bundle IDs.
					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							false,
							array(
								'type'    => 'table-open',
								'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_live_settings',
							array(
								'type'         => 'heading',
								'value'        => __( 'App Configuration', 'buddyboss-app' ),
								'desc'         => '',
								'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=122248' ) ),
								'icon'         => 'bb-icon-bf bb-icon-cog',
							)
						)
					);

					$html_signing_certificate = '<p>' . __( 'To generate iOS builds of your BuddyBoss App, you need bundle IDs, signing certificates and provisioning profiles registered in your Apple Developer account.', 'buddyboss-app' ) . '</p>' .
												'<p>' . __( 'You can generate signing certificates and provisioning profiles automatically, as long as you have:', 'buddyboss-app' ) . '</p>' .
												'<ul class="list-style">' .
												'<li>' . __( 'Connected your Apple Developer Account', 'buddyboss-app' ) . '</li>' .
												'<li>' . __( 'Entered the Bundle IDs created in your Apple Developer account', 'buddyboss-app' ) . '</li>' .
												'</ul>' .
												'<p>' . __( 'Otherwise, please create your own signing certificates and provisioning profiles in your Apple Developer account and upload them here.', 'buddyboss-app' ) . '</p>';

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_app_configuration_description',
							array(
								'type'  => 'html',
								'value' => $html_signing_certificate,
								'desc'  => '',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_bundle_ids_subheadline',
							array(
								'type'  => 'subheading',
								'value' => __( 'Bundle IDs', 'buddyboss-app' ),
								'desc'  => '',
								'class' => 'configure-subheadline',
							)
						)
					);

					$apple_is_connected_html = '';
					if ( Apple::instance()->is_connected() ) {
						$apple_is_connected_html = sprintf( '<p><button type="button" class="button button-secondary refresh-bundle-ids"><div class="bbapp_loading" style="display:none;"></div>%1$s</button></p>', __( 'Refresh Bundle IDs', 'buddyboss-app' ) );
					}

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.namespace_warn',
							array(
								'type'  => 'html',
								'value' => sprintf( '<p>%1$s <a href="https://developer.apple.com/account/resources/identifiers/list" target="_blank">%2$s</a> %3$s</p>', __( 'To assign bundle IDs to your release and test apps, please create a bundle ID for each in your', 'buddyboss-app' ), __( 'Apple Developer', 'buddyboss-app' ), __( 'account.', 'buddyboss-app' ) . $apple_is_connected_html ),
								'class' => function () {
									$bundle_ids = ( Apple::instance()->is_connected() ? Apple::instance()->get_registered_bundles() : array() );

									if ( ! Apple::instance()->is_connected() ) {
										return 'hidden';
									}

									if ( Apple::instance()->is_connected() && ! empty( $bundle_ids ) ) {
										return 'hidden';
									}

									return '';
								},
							)
						)
					);

					$ios_namespace_arg = array(
						'label' => __( 'Release App Bundle ID', 'buddyboss-app' ),
						'desc'  => function () {
							global $bbapp_ios_namespace_err;

							$desc = '';

							if ( true === $bbapp_ios_namespace_err ) {
								$desc .= sprintf( '<div class="bbapp-notice-box bbapp-notice-box--error bbapp-plain-box--error">%s</div>', esc_html__( 'Invalid Bundle ID. You can only use alphanumeric characters, "." and "-"', 'buddyboss-app' ) );
							}

							return $desc . __( 'Enter the <strong>Bundle ID</strong> for your release app.', 'buddyboss-app' );
						},
						'type'  => function () {
							return ( Apple::instance()->is_connected() ? 'select' : 'text' );
						},
						'value'        => '',
						'placeholder'  => '',
						'select_class' => 'bbapp-select2-cmn-wrp',
						'class'        => function () {
							$bundle_ids = ( Apple::instance()->is_connected() ? Apple::instance()->get_registered_bundles() : array() );

							if ( Apple::instance()->is_connected() && empty( $bundle_ids ) ) {
								return 'hidden';
							}

							return '';
						},
						'options'      => function () {
							$bundle_ids = ( Apple::instance()->is_connected() ? Apple::instance()->get_registered_bundles() : array() );
							$options[]  = __( '-- Select Bundle ID --', 'buddyboss-app' );

							if ( ! empty( $bundle_ids ) ) {
								foreach ( $bundle_ids as $bundle_id ) {
									$options[ $bundle_id ] = $bundle_id;
								}
							}

							return $options;
						},
					);

					$ios_dev_namespace_arg = array(
						'label'        => __( 'Test App Bundle ID', 'buddyboss-app' ),
						'desc'         => function () {
							global $bbapp_ios_dev_namespace_err;

							$build_link              = bbapp_get_super_admin_url( 'admin.php?page=bbapp-build&setting=request-build' );
							$apple_is_connected_html = '';
							if ( Apple::instance()->is_connected() ) {
								$apple_is_connected_html = sprintf( '<p><button type="button" class="button button-secondary refresh-bundle-ids"><div class="bbapp_loading" style="display:none;"></div>%1$s</button></p>', __( 'Refresh Bundle IDs', 'buddyboss-app' ) );
							}
							$new_ios_build = sprintf( '<div class="bbapp_notice bbapp_notice--context bbapp_notice--static"><p><a href="%1$s">%2$s</p></div>', esc_url( $build_link ), __( 'Changes will require a new iOS build.', 'buddyboss-app' ) );
							if ( true === $bbapp_ios_dev_namespace_err ) {
								return sprintf( '<div class="bbapp-plain-notice bbapp-plain-box--error">%s</div>', esc_html__( 'Invalid Bundle ID. You can only use alphanumeric characters, "." and "-"', 'buddyboss-app' ) ) . sprintf( '%1$s', __( 'Enter the <strong>Bundle ID</strong> for your test app.', 'buddyboss-app' ) ) . $apple_is_connected_html . $new_ios_build;
							}

							return sprintf( '%1$s', __( 'Enter the <strong>Bundle ID</strong> for your test app.', 'buddyboss-app' ) ) . $apple_is_connected_html . $new_ios_build;
						},
						'type'         => function () {
							return ( Apple::instance()->is_connected() ? 'select' : 'text' );
						},
						'value'        => '',
						'placeholder'  => '',
						'select_class' => 'bbapp-select2-cmn-wrp',
						'class'        => function () {
							$bundle_ids = ( Apple::instance()->is_connected() ? Apple::instance()->get_registered_bundles() : array() );

							if ( Apple::instance()->is_connected() && empty( $bundle_ids ) ) {
								return 'hidden';
							}

							return '';
						},
						'options'      => function () {
							$bundle_ids = ( Apple::instance()->is_connected() ? Apple::instance()->get_registered_bundles() : array() );
							$options[]  = __( '-- Select Bundle ID --', 'buddyboss-app' );
							if ( ! empty( $bundle_ids ) ) {
								foreach ( $bundle_ids as $bundle_id ) {
									$options[ $bundle_id ] = $bundle_id;
								}
							}

							return $options;
						},
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.namespace',
							$ios_namespace_arg
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.dev.namespace',
							$ios_dev_namespace_arg
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.namespace_error',
							array(
								'type'  => 'html',
								'value' => '<p class="bbapp-notice-box bbapp-notice-box--error">' . __( 'You must select different Bundle IDs for your release and test apps.', 'buddyboss-app' ) . '</p>',
								'class' => 'hidden validate-bundle-ids',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_signing_certificates_subheadline',
							array(
								'type'  => 'subheading',
								'value' => __( 'Signing Certificates', 'buddyboss-app' ),
								'desc'  => '',
								'class' => 'configure-subheadline',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.signing_certificates_automatic_html',
							array(
								'label' => __( 'Generate Certificates', 'buddyboss-app' ),
								'type'  => 'html',
								'value' => function () {
									if ( ! Apple::instance()->is_connected() ) {
										$this->save_option( 'publish.ios.signing_certificates_automatic', '' );
									}

									return sprintf( '<p class="description" style="margin: 0"><a href="#update-settings-1" class="bbapp-smooth-link">%1$s</a> %2$s</p>', esc_html__( 'Connect your Apple Developer Account', 'buddyboss-app' ), esc_html__( 'to generate signing certificates automatically.', 'buddyboss-app' ) );
								},
								'class' => function () {
									if ( Apple::instance()->is_connected() ) {
										return 'hidden';
									}

									return '';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.signing_certificates_automatic',
							array(
								'label'          => __( 'Generate Certificates', 'buddyboss-app' ),
								'type'           => 'checkbox',
								'label_checkbox' => __( 'Automatically generate signing certificates', 'buddyboss-app' ),
								'desc'           => __( 'If enabled, signing certificates will be generated automatically and saved to your Apple Developer account. You must be connected to your Apple Developer account to enable this option.', 'buddyboss-app' ),
								'class'          => function () {
									if ( ! Apple::instance()->is_connected() ) {
										return 'hidden';
									}

									return '';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_signing_certificates_release_app',
							array(
								'type'  => 'subheading_secondary',
								'value' => __( 'Release App Signing Certificate', 'buddyboss-app' ),
								'desc'  => '',
								'class' => 'configure-subheadline-secondary',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.signing_certificate_warn',
							array(
								'label' => __( 'Signing Certificate', 'buddyboss-app' ),
								'type'  => 'html',
								'value' => function () {
									$certificate_option = $this->option( 'publish.ios.signing_certificates_automatic' );
									$bundle_id          = $this->option( 'publish.ios.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) && empty( $bundle_id ) ) {
										return '<p class="bbapp-notice-box bbapp-notice-box--warning">' . __( 'Please select and save a Bundle ID for your release app.', 'buddyboss-app' ) . '</p>';
									}

									$auto_generate = $this->option( 'publish.ios.signing_certificates_automatic' );
									$cert_error    = bbapp_get_network_option( '_bbapp_automatic_release_cert_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $auto_generate ) && ! empty( $cert_error ) && empty( $this->option( 'publish.ios.signing_certificate' ) ) ) {
										return sprintf( '<p class="bbapp-notice-box bbapp-notice-box--error">%1$s <br/>%2$s</p>', esc_html__( 'There was a problem generating your release app signing certificate:', 'buddyboss-app' ), $cert_error );
									}

									return '';
								},
								'class' => function () {
									$certificate_option = $this->option( 'publish.ios.signing_certificates_automatic' );
									$bundle_id          = $this->option( 'publish.ios.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) && empty( $bundle_id ) ) {
										return 'bbapp-label-secondary';
									}

									$auto_generate = $this->option( 'publish.ios.signing_certificates_automatic' );
									$cert_error    = bbapp_get_network_option( '_bbapp_automatic_release_cert_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $auto_generate ) && ! empty( $cert_error ) && empty( $this->option( 'publish.ios.signing_certificate' ) ) ) {
										return 'bbapp-label-secondary';
									}

									return 'hidden';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.signing_certificate',
							array(
								'label'    => __( 'Signing Certificate', 'buddyboss-app' ),
								'desc'     => __( 'Upload the <strong>iOS Distribution</strong> certificate for your release app.', 'buddyboss-app' ),
								'type'     => '\BuddyBossApp\Admin\Fields\Upload',
								'settings' => array(
									'allowed_ext'    => array( 'p12' ),
									'show_file_link' => true,
								),
								'value'    => '',
								'class'    => function () {
									$certificate_option = $this->option( 'publish.ios.signing_certificates_automatic' );
									$bundle_id          = $this->option( 'publish.ios.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) && empty( $bundle_id ) ) {
										return 'hidden';
									}

									$auto_generate = $this->option( 'publish.ios.signing_certificates_automatic' );
									$cert_error    = bbapp_get_network_option( '_bbapp_automatic_release_cert_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $auto_generate ) && ! empty( $cert_error ) && empty( $this->option( 'publish.ios.signing_certificate' ) ) ) {
										return 'hidden';
									}

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) ) {
										return 'apple-connected bbapp-label-secondary';
									}

									return 'bbapp-label-secondary';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.signing_certificate_id',
							array(
								'label' => __( 'Signing Certificate ID', 'buddyboss-app' ),
								'desc'  => __( 'Enter the Certificate ID for the signing certificate entered above. This can be found in the URL of the page when viewing the certificate in your Apple Developer account.', 'buddyboss-app' ),
								'type'  => 'text',
								'value' => '',
								'class' => function () {
									$certificate_option = $this->option( 'publish.ios.signing_certificates_automatic' );
									$bundle_id          = $this->option( 'publish.ios.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) && empty( $bundle_id ) ) {
										return 'hidden';
									}

									$auto_generate = $this->option( 'publish.ios.signing_certificates_automatic' );
									$cert_error    = bbapp_get_network_option( '_bbapp_automatic_release_cert_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $auto_generate ) && ! empty( $cert_error ) && empty( $this->option( 'publish.ios.signing_certificate' ) ) ) {
										return 'hidden';
									}

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) ) {
										return 'apple-connected bbapp-label-secondary';
									}

									return 'bbapp-label-secondary';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.signing_certificate_password',
							array(
								'label' => __( 'Signing Certificate Password', 'buddyboss-app' ),
								'desc'  => __( 'If you uploaded a signing certificate which uses a password, please enter it here.', 'buddyboss-app' ),
								'type'  => 'password',
								'value' => '',
								'class' => function () {
									$certificate_option = $this->option( 'publish.ios.signing_certificates_automatic' );
									$bundle_id          = $this->option( 'publish.ios.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) && empty( $bundle_id ) ) {
										return 'hidden';
									}

									$auto_generate = $this->option( 'publish.ios.signing_certificates_automatic' );
									$cert_error    = bbapp_get_network_option( '_bbapp_automatic_release_cert_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $auto_generate ) && ! empty( $cert_error ) && empty( $this->option( 'publish.ios.signing_certificate' ) ) ) {
										return 'hidden';
									}

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) ) {
										return 'apple-connected bbapp-label-secondary';
									}

									return 'bbapp-label-secondary';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_signing_certificates_test_app',
							array(
								'type'  => 'subheading_secondary',
								'value' => __( 'Test App Signing Certificate', 'buddyboss-app' ),
								'desc'  => '',
								'class' => 'configure-subheadline-secondary',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.dev.signing_certificate_warn',
							array(
								'label' => __( 'Signing Certificate', 'buddyboss-app' ),
								'type'  => 'html',
								'value' => function () {
									$certificate_option = $this->option( 'publish.ios.signing_certificates_automatic' );
									$bundle_id          = $this->option( 'publish.ios.dev.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) && empty( $bundle_id ) ) {
										return '<p class="bbapp-notice-box bbapp-notice-box--warning">' . __( 'Please select and save a Bundle ID for your test app.', 'buddyboss-app' ) . '</p>';
									}

									$auto_generate = $this->option( 'publish.ios.signing_certificates_automatic' );
									$cert_error    = bbapp_get_network_option( '_bbapp_automatic_dev_cert_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $auto_generate ) && ! empty( $cert_error ) && empty( $this->option( 'publish.ios.dev.signing_certificate' ) ) ) {
										return sprintf( '<p class="bbapp-notice-box bbapp-notice-box--error">%1$s <br/>%2$s</p>', esc_html__( 'There was a problem generating your test app signing certificate::', 'buddyboss-app' ), $cert_error );
									}

									return '';
								},
								'class' => function () {
									$certificate_option = $this->option( 'publish.ios.signing_certificates_automatic' );
									$bundle_id          = $this->option( 'publish.ios.dev.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) && empty( $bundle_id ) ) {
										return 'bbapp-label-secondary';
									}

									$auto_generate = $this->option( 'publish.ios.signing_certificates_automatic' );
									$cert_error    = bbapp_get_network_option( '_bbapp_automatic_dev_cert_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $auto_generate ) && ! empty( $cert_error ) && empty( $this->option( 'publish.ios.dev.signing_certificate' ) ) ) {
										return 'bbapp-label-secondary';
									}

									return 'hidden';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.dev.signing_certificate',
							array(
								'label'    => __( 'Signing Certificate', 'buddyboss-app' ),
								'desc'     => __( 'Upload the <strong>iOS App Development</strong> certificate for your test app.', 'buddyboss-app' ),
								'type'     => '\BuddyBossApp\Admin\Fields\Upload',
								'settings' => array(
									'allowed_ext'    => array( 'p12' ),
									'show_file_link' => true,
								),
								'value'    => '',
								'class'    => function () {
									$certificate_option = $this->option( 'publish.ios.signing_certificates_automatic' );
									$bundle_id          = $this->option( 'publish.ios.dev.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) && empty( $bundle_id ) ) {
										return 'hidden';
									}

									$auto_generate = $this->option( 'publish.ios.signing_certificates_automatic' );
									$cert_error    = bbapp_get_network_option( '_bbapp_automatic_dev_cert_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $auto_generate ) && ! empty( $cert_error ) && empty( $this->option( 'publish.ios.dev.signing_certificate' ) ) ) {
										return 'hidden';
									}

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) ) {
										return 'apple-connected bbapp-label-secondary';
									}

									return 'bbapp-label-secondary';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.dev.signing_certificate_password',
							array(
								'label' => __( 'Signing Certificate Password', 'buddyboss-app' ),
								'desc'  => __( 'If you upload a .p12 file which uses a password, please enter it here.', 'buddyboss-app' ),
								'type'  => 'password',
								'value' => '',
								'class' => function () {
									$certificate_option = $this->option( 'publish.ios.signing_certificates_automatic' );
									$bundle_id          = $this->option( 'publish.ios.dev.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) && empty( $bundle_id ) ) {
										return 'hidden';
									}

									$auto_generate = $this->option( 'publish.ios.signing_certificates_automatic' );
									$cert_error    = bbapp_get_network_option( '_bbapp_automatic_dev_cert_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $auto_generate ) && ! empty( $cert_error ) && empty( $this->option( 'publish.ios.dev.signing_certificate' ) ) ) {
										return 'hidden';
									}

									if ( Apple::instance()->is_connected() && ! empty( $certificate_option ) ) {
										return 'apple-connected bbapp-label-secondary';
									}

									return 'bbapp-label-secondary';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.dev.signing_certificate_build_button_html',
							array(
								'type'  => 'html',
								'value' => '<div class="bbapp-build-button bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require a new iOS build.', 'buddyboss-app' ) ) . '</p></div>',
								'label' => '&nbsp;',
								'class' => '',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_provisioning_profiles_subheadline',
							array(
								'type'  => 'subheading',
								'value' => __( 'Provisioning Profiles', 'buddyboss-app' ),
								'desc'  => '',
								'class' => 'configure-subheadline',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.provisioning_profile_option_html',
							array(
								'label' => __( 'Generate Profiles', 'buddyboss-app' ),
								'type'  => 'html',
								'value' => function () {
									if ( ! Apple::instance()->is_connected() ) {
										$this->save_option( 'publish.ios.provisioning_profile_automatic', '' );
									}

									return sprintf( '<p class="description" style="margin: 0"><a href="#update-settings-1" class="bbapp-smooth-link">%1$s</a> %2$s</p>', esc_html__( 'Connect your Apple Developer Account', 'buddyboss-app' ), esc_html__( 'to generate provisioning profiles automatically.', 'buddyboss-app' ) );
								},
								'class' => function () {
									if ( Apple::instance()->is_connected() ) {
										return 'hidden';
									}

									return '';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.provisioning_profile_automatic',
							array(
								'label'          => __( 'Generate Profiles', 'buddyboss-app' ),
								'type'           => 'checkbox',
								'label_checkbox' => __( 'Automatically generate provisioning profiles ', 'buddyboss-app' ),
								'desc'           => __( 'If enabled, provisioning profiles will be generated automatically and saved to your Apple Developer account. You must be connected to your Apple Developer account to enable this option.', 'buddyboss-app' ),
								'class'          => function () {
									if ( ! Apple::instance()->is_connected() ) {
										return 'hidden';
									}

									return '';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.automatic_html_generate_new_profile_btn',
							array(
								'label' => ' ',
								'type'  => 'html',
								'value' => function () {
									if ( Apple::instance()->is_connected() ) {
										$settings       = ManageApp::instance()->get_app_settings();
										$configure_type = ( isset( $settings['publish.ios.provisioning_profile_automatic'] ) ) ? $settings['publish.ios.provisioning_profile_automatic'] : false;

										if ( true === (bool) $configure_type ) {
											return '<p><button type="button" class="button button-secondary bbapp-generate-new-profiles" data-nonce="' . wp_create_nonce( 'bbapp_create_new_profile' ) . '"><div class="bbapp_loading" style="display:none;"></div>' . __( 'Generate New Profiles', 'buddyboss-app' ) . '</button></p>';
										}
									}
								},
								'class' => function () {
									if ( ! Apple::instance()->is_connected() ) {
										return 'hidden';
									}

									return '';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.provisioning_profile_warn',
							array(
								'label' => __( 'Release App Provisional Profile', 'buddyboss-app' ),
								'type'  => 'html',
								'value' => function () {
									$generate_option = $this->option( 'publish.ios.provisioning_profile_automatic' );
									$bundle_id       = $this->option( 'publish.ios.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $bundle_id ) ) {
										return '<p class="bbapp-notice-box bbapp-notice-box--warning">' . __( 'Please select and save a Bundle ID for your release app.', 'buddyboss-app' ) . '</p>';
									}

									$cert_option = $this->option( 'publish.ios.signing_certificate' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $cert_option ) ) {
										return '<p class="bbapp-notice-box bbapp-notice-box--warning">' . __( 'Please upload or generate a signing certificate for your release app.', 'buddyboss-app' ) . '</p>';
									}

									// Automatic Provisioning Profile Generating Error.
									$pp_error = bbapp_get_network_option( '_bbapp_automatic_release_provisioning_profiles_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $pp_error ) && empty( $this->option( 'publish.ios.provisioning_profile' ) ) ) {
										return sprintf( '<p class="bbapp-notice-box bbapp-notice-box--error">%1$s <br/>%2$s</p>', esc_html__( 'There was a problem generating your release app provisioning profile:', 'buddyboss-app' ), $pp_error );
									}

									return '';
								},
								'class' => function () {
									$generate_option = $this->option( 'publish.ios.provisioning_profile_automatic' );
									$bundle_id       = $this->option( 'publish.ios.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $bundle_id ) ) {
										return '';
									}

									$cert_option = $this->option( 'publish.ios.signing_certificate' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $cert_option ) ) {
										return '';
									}

									// Automatic Provisioning Profile Generating Error.
									$pp_error = bbapp_get_network_option( '_bbapp_automatic_release_provisioning_profiles_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $pp_error ) && empty( $this->option( 'publish.ios.provisioning_profile' ) ) ) {
										return '';
									}

									return 'hidden';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.provisioning_profile',
							array(
								'label'    => __( 'Release App Provisional Profile', 'buddyboss-app' ),
								'desc'     => __( 'Upload an <strong>App Store</strong> provisioning profile created for your release app.', 'buddyboss-app' ),
								'type'     => '\BuddyBossApp\Admin\Fields\Upload',
								'settings' => array(
									'allowed_ext'    => array( 'mobileprovision' ),
									'show_file_link' => true,
								),
								'value'    => '',
								'class'    => function () {
									$generate_option = $this->option( 'publish.ios.provisioning_profile_automatic' );
									$bundle_id       = $this->option( 'publish.ios.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $bundle_id ) ) {
										return 'hidden';
									}

									$cert_option = $this->option( 'publish.ios.signing_certificate' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $cert_option ) ) {
										return 'hidden';
									}

									$pp_error = bbapp_get_network_option( '_bbapp_automatic_release_provisioning_profiles_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $pp_error ) && empty( $this->option( 'publish.ios.provisioning_profile' ) ) ) {
										return 'hidden';
									}

									$provisioning_option = $this->option( 'publish.ios.provisioning_profile_automatic' );

									if ( Apple::instance()->is_connected() && ! empty( $provisioning_option ) ) {
										return 'apple-connected';
									}

									return '';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.dev.provisioning_profile_warn',
							array(
								'label' => __( 'Test App Provisional Profile', 'buddyboss-app' ),
								'type'  => 'html',
								'value' => function () {
									$generate_option = $this->option( 'publish.ios.provisioning_profile_automatic' );
									$bundle_id       = $this->option( 'publish.ios.dev.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $bundle_id ) ) {
										return '<p class="bbapp-notice-box bbapp-notice-box--warning">' . __( 'Please select and save a Bundle ID for your test app.', 'buddyboss-app' ) . '</p>';
									}

									$cert_option = $this->option( 'publish.ios.dev.signing_certificate' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $cert_option ) ) {
										return '<p class="bbapp-notice-box bbapp-notice-box--warning">' . __( 'Please upload or generate a signing certificate for your test app.', 'buddyboss-app' ) . '</p>';
									}

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( Apple::instance()->get_local_devices() ) ) {
										return sprintf( '<p class="bbapp-notice-box bbapp-notice-box--warning" style="display: block;">%1$s <a href="%2$s" style="color:#8B6300;">%3$s</a> %4$s</p>', __( 'Please', 'buddyboss-app' ), esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=ios_devices' ) ), __( 'register an iOS Device', 'buddyboss-app' ), __( 'to generate a provisioning profile for your test app.', 'buddyboss-app' ) );
									}

									// Automatic Provisioning Profile Generating Error.
									$pp_error = bbapp_get_network_option( '_bbapp_automatic_dev_provisioning_profiles_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $pp_error ) && empty( $this->option( 'publish.ios.dev.provisioning_profile' ) ) ) {
										return sprintf( '<p class="bbapp-notice-box bbapp-notice-box--error">%1$s <br/>%2$s</p>', esc_html__( 'There was a problem generating your test app provisioning profile:', 'buddyboss-app' ), $pp_error );
									}

									return '';
								},
								'class' => function () {
									$generate_option = $this->option( 'publish.ios.provisioning_profile_automatic' );
									$bundle_id       = $this->option( 'publish.ios.dev.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $bundle_id ) ) {
										return '';
									}

									$cert_option = $this->option( 'publish.ios.dev.signing_certificate' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $cert_option ) ) {
										return '';
									}

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( Apple::instance()->get_local_devices() ) ) {
										return '';
									}

									// Automatic Provisioning Profile Generating Error.
									$pp_error = bbapp_get_network_option( '_bbapp_automatic_dev_provisioning_profiles_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $pp_error ) && empty( $this->option( 'publish.ios.dev.provisioning_profile' ) ) ) {
										return '';
									}

									return 'hidden';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.dev.provisioning_profile',
							array(
								'label'    => __( 'Test App Provisional Profile', 'buddyboss-app' ),
								'desc'     => sprintf( __( 'Upload an <strong>iOS App Developer</strong> provisioning profile for your test app.', 'buddyboss-app' ) ),
								'type'     => '\BuddyBossApp\Admin\Fields\Upload',
								'settings' => array(
									'allowed_ext'    => array( 'mobileprovision' ),
									'show_file_link' => true,
								),
								'value'    => '',
								'class'    => function () {
									$generate_option = $this->option( 'publish.ios.provisioning_profile_automatic' );
									$bundle_id       = $this->option( 'publish.ios.dev.namespace' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $bundle_id ) ) {
										return 'hidden';
									}

									$cert_option = $this->option( 'publish.ios.dev.signing_certificate' );

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( $cert_option ) ) {
										return 'hidden';
									}

									if ( Apple::instance()->is_connected() && ! empty( $generate_option ) && empty( Apple::instance()->get_local_devices() ) ) {
										return 'hidden';
									}

									// Automatic Provisioning Profile Generating Error.
									$pp_error = bbapp_get_network_option( '_bbapp_automatic_dev_provisioning_profiles_error_msg' );

									if ( Apple::instance()->is_connected() && ! empty( $pp_error ) && empty( $this->option( 'publish.ios.dev.provisioning_profile' ) ) ) {
										return 'hidden';
									}

									$provisioning_option = $this->option( 'publish.ios.provisioning_profile_automatic' );

									if ( Apple::instance()->is_connected() && ! empty( $provisioning_option ) ) {
										return 'apple-connected';
									}

									return '';
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.dev.provisioning_profile_build_button_html',
							array(
								'type'  => 'html',
								'value' => '<div class="bbapp-build-button bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require a new iOS build.', 'buddyboss-app' ) ) . '</p></div>',
								'label' => '&nbsp;',
								'class' => '',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_save_app_configuration',
							array(
								'type'  => 'submit',
								'value' => __( 'Save Changes', 'buddyboss-app' ),
								'desc'  => '',
							)
						)
					);

					// IOS App Information.
					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							false,
							array(
								'type'    => 'table-open',
								'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_app_information',
							array(
								'type'  => 'heading',
								'value' => __( 'App Information', 'buddyboss-app' ),
								'desc'  => '',
								'icon'  => 'bb-icon-bf bb-icon-clipboard',
							)
						)
					);

					// ios.store_app_id.
					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.store_app_id',
							array(
								'label' => __( 'Apple App ID', 'buddyboss-app' ),
								'type'  => function () {
									return ( Apple::instance()->is_connected() && ! empty( Apple::instance()->get_apple_release_bundle_store_app_id() ) ) ? 'html' : 'password';
								},
								'value' => function () {
									return ( Apple::instance()->is_connected() ) ? Apple::instance()->get_apple_release_bundle_store_app_id() : '';
								},
								'desc'  => sprintf(
								/* translators: 1. Smart banner screen link. 2. Rate app screen. */
									__( 'Enter the App ID that was assigned to your release app by Apple. This will be used to connect the <a href="%1$s">Smart Banner</a> and <a href="%2$s">Rate this App</a> features to your listing in the Apple App Store, after the app is published.', 'buddyboss-app' ),
									bbapp_get_admin_url( 'admin.php?page=bbapp-settings&setting=smart_banner' ),
									bbapp_get_admin_url( 'admin.php?page=bbapp-settings&setting=feedback#app_settings_app_rating_heading' )
								),
							)
						)
					);

					// setting_apple_iap_secret.
					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.shared_secret',
							array(
								'label' => __( 'App Specific Shared Secret', 'buddyboss-app' ),
								'desc'  => __( 'To use in-app products on your iOS app, please enter the app-specific Shared Secret for your release app after connecting to your Apple Developer account.', 'buddyboss-app' ),
								'type'  => 'password',
								'value' => '',
							)
						)
					);

					if ( bbapp_is_active( 'iap' ) ) {
						$apple_app_id             = ( Apple::instance()->is_connected() ) ? Apple::instance()->get_apple_release_bundle_store_app_id() : '';
						$server_notification_link = ( ! empty( $apple_app_id ) ) ? "https://appstoreconnect.apple.com/apps/{$apple_app_id}/appstore/info" : 'https://appstoreconnect.apple.com';

						// setting_apple_iap_server_notification.
						$this->settings_form_ios_configuration->add_field(
							$this->settings_provider->field(
								'publish.ios.server_notification_url',
								array(
									'label' => __( 'App Store Server Notifications', 'buddyboss-app' ),
									'desc'  => sprintf( '%1$s <a href="%2$s" target="_blank">%3$s</a> %4$s', esc_html__( 'App Store Server Notifications provide information about key events related to your in-app purchases, such as changes to a subscription\'s status or the refund of an in-app purchase. In order to receive these notifications from the App store. you must provide a URL in', 'buddyboss-app' ), $server_notification_link, esc_html__( 'App Store Connect', 'buddyboss-app' ), esc_html__( 'that links to your server', 'buddyboss-app' ) ),
									'type'  => 'html',
									'value' => function () {
										$copy_html = '';
										ob_start();
										?>
										<div class="bbapp-server-notifications-main-wrp">
											<div class="bbapp-server-notifcation-inr-wrp">
												<div class="bbapp-sub-elements">
												<span class="txt" id="bbapp_ios_server_notifcation_url">
													<?php echo esc_url( add_query_arg( array( 'ios_s2s_notification' => 1 ), get_home_url() ) ); ?>
												</span>
												</div>
												<div class="bbapp-sub-tooltip">
													<button type="button"
															onClick="window.bbappCore.core.bbapp_copy_paste_url('bbapp_ios_server_notifcation_url')">
													<span class="bbapp-tooltip-text" id="bbapp-span-tooltip-bbapp_ios_server_notifcation_url"></span>
														<i class="bb-icon-f bb-icon-clipboard"></i>
													</button>
												</div>
											</div>
										</div>
										<?php
										$copy_html .= ob_get_clean();

										return $copy_html;
									},
								)
							)
						);
					}

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_save_app_information',
							array(
								'type'  => 'submit',
								'value' => __( 'Save Changes', 'buddyboss-app' ),
								'desc'  => '',
							)
						)
					);

					/**
					 * IOS Upload Configuration.
					 */
					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							false,
							array(
								'type'    => 'table-open',
								'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_upload_configuration',
							array(
								'type'         => 'heading',
								'value'        => __( 'Upload Configuration', 'buddyboss-app' ),
								'desc'         => '',
								'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=124747' ) ),
								'icon'         => 'bb-icon-bf bb-icon-upload',
							)
						)
					);

					$users   = Apple::instance()->get_acccount_users( false, array( 'APP_MANAGER', 'ADMIN' ) );
					$options = array( '' => __( 'Select User', 'buddyboss-app' ) );

					if ( isset( $users['data'] ) && is_array( $users['data'] ) ) {
						foreach ( $users['data'] as $user ) {
							$options[ $user['attributes']['username'] ] = "{$user['attributes']['firstName']} ( {$user['attributes']['username']} )";
						}
					}

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.app_specific_password_account_id',
							array(
								'label'   => __( 'Apple Account ID', 'buddyboss-app' ),
								'desc'    => __( 'Select the Apple account to use when uploading builds to App Store Connect. Only accounts with the Admin or App Manager role in App Store Connect can be used.', 'buddyboss-app' ),
								'type'    => 'select',
								'value'   => '',
								'options' => $options,
							)
						)
					);

					// ios.app_specific_password.
					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.app_specific_password',
							array(
								'label' => __( 'App-Specific Password', 'buddyboss-app' ),
								'desc'  => sprintf( '%1$s <a href="https://appleid.apple.com/account/manage" target="_blank">Apple ID account page</a>. %2$s', esc_html__( 'Please provide an app-specific password for the Apple account selected above. These can be generated on the', 'buddyboss-app' ), esc_html__( 'If you change or reset your primary Apple ID password, Apple will automatically revoke your app-specific passwords to protect the security of your account.', 'buddyboss-app' ) ),
								'type'  => 'password',
								'value' => '',
							)
						)
					);

					/**
					 * iOS App Permissions
					 */
					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							false,
							array(
								'type'    => 'table-open',
								'classes' => 'buddyboss-app-admin-card bbapp-admin-ios-app-permissions bbapp-admin-publish',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_permissions_configuration',
							array(
								'type'         => 'heading',
								'value'        => __( 'Permissions', 'buddyboss-app' ),
								'desc'         => '',
								'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=#' ) ), // Todo: Add tutorial link.
								'icon'         => 'bb-icon-bf bb-icon-lock-open',
							)
						)
					);

					// Enable Android App media playback permission.
					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.permissions.play_music',
							array(
								'label'          => __( 'Play Music <div class="bbapp-dev-only">Custom development use only</div>', 'buddyboss-app' ),
								'label_checkbox' => __( 'Allow your iOS app to play audio in the background', 'buddyboss-app' ),
								'desc'           => __(
									'When enabled, background media playback will be allowed in your app, ensuring continued audio playback while running in the background.',
									'buddyboss-app'
								),
								'type'           => 'checkbox',
								'disabled'       => ! self::instance()->has_custom_developer_access(),
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.permissions.permission_build_button_html',
							array(
								'type'  => 'html',
								'value' => '<div class="bbapp-build-button bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf(
										'<a href="%1$s">%2$s</a>',
										esc_url( $build_link ),
										esc_html__( 'Changes will require a new iOS build.', 'buddyboss-app' )
									) . '</p></div>',
								'label' => '&nbsp;',
								'class' => '',
							)
						)
					);

					/**
					 * IOS Universal Links Configuration.
					 */
					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							false,
							array(
								'type'    => 'table-open',
								'classes' => 'buddyboss-app-admin-card bbapp-admin-publish bbapp-admin-ios-universal-links',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_universal_links_configuration',
							array(
								'type'         => 'heading',
								'value'        => __( 'Universal Links', 'buddyboss-app' ),
								'desc'         => '',
								'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=125472' ) ),
								'icon'         => 'bb-icon-bf bb-icon-link',
							)
						)
					);

					// Enable Universal Links.
					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.universal_link',
							array(
								'label'          => __( 'Enable Universal Links', 'buddyboss-app' ),
								'label_checkbox' => __( 'Open links using your Site URL in the BuddyBoss App.', 'buddyboss-app' ),
								'desc'           => sprintf( '%1$s (<a href="%2$s">%3$s</a>) %4$s', esc_html__( 'When enabled, all links using your Site URL', 'buddyboss-app' ), get_home_url(), get_home_url(), esc_html__( 'will open in your app, as long as the app is installed on the member\'s iOS device.', 'buddyboss-app' ) ),
								'type'           => 'checkbox',
							)
						)
					);

					$this->bbapp_ios_assetlinks = home_url( 'apple-app-site-association' );

					$this->bbapp_ios_assetlinks_alt = home_url( 'apple-app-site-association' );                 if ( function_exists( 'bbapp_is_valid_link' ) && ( ! bbapp_is_valid_link( $this->bbapp_ios_assetlinks ) && ! bbapp_is_valid_link( $this->bbapp_ios_assetlinks_alt ) ) ) {
						$this->settings_form_ios_configuration->add_field(
							$this->settings_provider->field(
								'app_ios_universal_links_connection_status_message_sub',
								array(
									'label' => '&nbsp',
									'type'  => 'html',
									'value' => function () {
										return sprintf( '<div class="bbapp-notice-box bbapp-notice-box--warning"><div class="data-update-ajax"><b>%1$s</b><p>%2$s <a targe="_blank" href="%3$s" >%4$s</a> %5$s <a href="%6$s" >%7$s</a> %8$s <a href="%9$s" >%10$s</a></p><p class="bbapp-universal-links">%11$s</p></div>', esc_html__( 'Associated Domains Not Accessible', 'buddyboss-app' ), esc_html__( 'Apple needs access to the following URLs in order to read the', 'buddyboss-app' ), 'https://developer.apple.com/documentation/xcode/supporting-associated-domains', esc_html__( 'associated domains', 'buddyboss-app' ), esc_html__( 'for your iOS App:', 'buddyboss-app' ), esc_url( $this->bbapp_ios_assetlinks ), esc_url( $this->bbapp_ios_assetlinks ), esc_html__( 'OR', 'buddyboss-app' ), esc_url( $this->bbapp_ios_assetlinks_alt ), esc_url( $this->bbapp_ios_assetlinks_alt ), esc_html__( 'To ensure Universal Links work as intended, please contact your host to enable access to these URLs. ', 'buddyboss-app' ) );
									},
									'desc'  => '',
									'class' => function () {
										$class        = 'universal_external_links';
										$get_settings = $this->get_settings();
										if ( empty( $get_settings ) || empty( $get_settings['publish.ios.universal_link'] ) ) {
											$class .= ' hidden';
										}

										return $class;
									},
								)
							)
						);
					}

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.dev.universal_link_build_button_html',
							array(
								'type'  => 'html',
								'value' => '<div class="bbapp-build-button bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require a new iOS build.', 'buddyboss-app' ) ) . '</p></div>',
								'label' => '&nbsp;',
								'class' => '',
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'publish.ios.exclude_links',
							array(
								'label'       => __( 'Exclude Links', 'buddyboss-app' ),
								'desc'        => __( 'Enter URLs or URI fragments (e.g. /groups/) to exclude from opening in your app. Enter one URL or URI per line', 'buddyboss-app' ),
								'type'        => 'textarea',
								'placeholder' => __( 'Add text', 'buddyboss-app' ),
								'value'       => '',
								'filter'      => function ( $value ) {
									return $value;
								},
								'class'       => function () {
									$class        = 'universal_external_links';
									$get_settings = $this->get_settings();
									if ( empty( $get_settings ) || empty( $get_settings['publish.ios.universal_link'] ) ) {
										$class .= ' hidden';
									}

									return $class;
								},
							)
						)
					);

					$this->settings_form_ios_configuration->add_field(
						$this->settings_provider->field(
							'app_publish_ios_save_upload_configuration',
							array(
								'type'  => 'submit',
								'value' => __( 'Save Changes', 'buddyboss-app' ),
								'desc'  => '',
							)
						)
					);
				}
			}

			/**
			 * IOS Registered iOS Devices Settings.
			 */
			if ( $this->get_current_sub_tab() === 'ios_devices' ) {
				if ( ! ManageApp::instance()->is_current_secondary_site() ) {
					// Registered iOS Devices.
					$this->settings_form->add_field(
						$this->settings_provider->field(
							false,
							array(
								'type'    => 'table-open',
								'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
							)
						)
					);

					$this->settings_form->add_field(
						$this->settings_provider->field(
							'app_publish_ios_devices',
							array(
								'type'         => 'heading',
								'value'        => __( 'Registered iOS Devices', 'buddyboss-app' ),
								'id'           => 'bbapp_dev_settings',
								'desc'         => __( 'Use this tool to register your device in your Apple Developer account. You\'ll only be able to install iOS Test App builds on your device if your device is registered. Without registering your device, you\'ll have to use Apple\'s TestFlight to download and test your app before publishing.', 'buddyboss-app' ),
								'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=122270' ) ),
								'icon'         => 'bb-icon-bf bb-icon-mobile',
							)
						)
					);

					$this->settings_form->add_field(
						$this->settings_provider->field(
							'publish.ios.devices',
							array(
								'no_label' => true,
								'type'     => '\BuddyBossApp\Admin\Fields\DevicesiOS',
								'value'    => '',
								'desc'     => '<div class="bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require a new iOS build.', 'buddyboss-app' ) ) . '<p></div>',
							)
						)
					);

				}
			}

			/***
			 * Android Settings.
			 */
			if ( 'android' === $this->get_current_sub_tab() ) {
				// Connect to Android.
				$this->settings_form_android_connect->add_field(
					$this->settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form_android_connect->add_field(
					$this->settings_provider->field(
						'app_config_android_account_settings',
						array(
							'type'         => 'heading',
							'value'        => function () {
								$account_connection = ( ! empty( Android::instance()->is_google_connected() ) ) ? true : false;
								$value              = __( 'Connect Your Google Developer Account', 'buddyboss-app' );

								if ( true === $account_connection ) {
									return $value .= sprintf( "<span class='account-connect-status connected'>%s</span>", __( 'Connected', 'buddyboss-app' ) );
								}

								return $value .= sprintf( "<span  class='account-connect-status not-connected'>%s</span>", __( 'Not Connected', 'buddyboss-app' ) );
							},
							'desc'         => __( 'Please connect your Google Developer Account to automate configuration of your Android settings and to enable publishing the Android version of your app via our publishing service. You must also connect your account to sell in-app products on Android devices.', 'buddyboss-app' ),
							'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=122276' ) ),
							'icon'         => 'bb-icon-bf bb-icon-brand-android',
						)
					)
				);

				$this->settings_form_android_connect->add_field(
					$this->settings_provider->field(
						'app_publish_android_connection_status',
						array(
							'type'  => 'html',
							'value' => function () {
								if ( ! empty( Android::instance()->is_google_connected() ) ) {
									$key_identifier = Configure::instance()->option( 'publish.android.account_key' );

									return '<div class="bbapp-account-connected bbapp-notice-box">' . __( 'Connected with Service Key: ', 'buddyboss-app' ) . basename( $key_identifier ) . '</div>';
								}

								return '';
							},
							'desc'  => '',
							'class' => function () {
								if ( empty( Android::instance()->is_google_connected() ) ) {
									return 'hidden';
								}

								return '';
							},
						)
					)
				);

				// google_service_account_json.
				$this->settings_form_android_connect->add_field(
					$this->settings_provider->field(
						'publish.android.account_key',
						array(
							'label'    => __( 'Service Account Key', 'buddyboss-app' ),
							'type'     => '\BuddyBossApp\Admin\Fields\Upload',
							'value'    => '',
							'settings' => array(
								'allowed_ext'    => array( 'json' ),
								'upload_dir'     => 'bbapp/uploads',
								'show_file_link' => true,
							),
							'desc'     => __( 'Log into your <a href="https://console.cloud.google.com/" target="_blank">Google Cloud Platform</a> console to generate a service account key as a JSON file, and then upload the JSON file here. For step-by-step instructions follow <a href="https://cloud.google.com/iam/docs/creating-managing-service-account-keys" target="_blank">this tutorial</a>.', 'buddyboss-app' ),
							'class'    => function () {
								if ( ! empty( Android::instance()->is_google_connected() ) ) {
									return 'hidden';
								}

								return 'bbapp_is_required';
							},
						)
					)
				);

				$this->settings_form_android_connect->add_field(
					$this->settings_provider->field(
						'app_publish_android_connection_submit',
						array(
							'type'     => function () {
								if ( ! empty( Android::instance()->is_google_connected() ) ) {
									return 'button';
								}

								return 'submit';
							},
							'value'    => function () {
								if ( ! empty( Android::instance()->is_google_connected() ) ) {
									return __( 'Disconnect Account', 'buddyboss-app' );
								} else {
									return __( 'Connect Account', 'buddyboss-app' );
								}
							},
							'desc'     => '',
							'class'    => function () {
								if ( ! empty( Android::instance()->is_google_connected() ) ) {
									return 'bbapp_android_connected';
								} else {
									return 'bbapp_android_disconnected';
								}
							},
							'disabled' => function () {
								return ( Android::instance()->get_private_key() || Android::instance()->is_google_connected() ) ? false : true;
							},
						)
					)
				);

				$this->settings_form_android_connect->add_field(
					$this->settings_provider->field(
						'app_publish_android_connection_status_message',
						array(
							'type'  => 'html',
							'value' => function () {
								global $app_settings_publish_android_account_key_invalid_file;

								$error_messages = Android::instance()->get_google_connection_error();

								if ( true === $app_settings_publish_android_account_key_invalid_file ) {
									return sprintf( '<div class="bbapp-notice-box bbapp-notice-box--error"><span class="bbapp-err-label">%s</span></div>', esc_html__( 'Please enter a valid Service Account Key (.json file) generated from your Google Developer account.', 'buddyboss-app' ) );
								}

								if ( ! empty( $error_messages ) ) {
									return sprintf( '<div class="bbapp-notice-box bbapp-notice-box--error"><span class="bbapp-err-label">%s</span><br/>%s</div>', esc_html__( 'There was a problem connecting to your Google Developer account: ', 'buddyboss-app' ), $error_messages );
								}

								return '';
							},
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_publish_android_live_settings',
						array(
							'type'         => 'heading',
							'value'        => __( 'App Configuration', 'buddyboss-app' ),
							'desc'         => '',
							'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=122254' ) ),
							'icon'         => 'bb-icon-bf bb-icon-cog',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_publish_android_app_information',
						array(
							'type'  => 'subheading',
							'value' => __( 'App Information', 'buddyboss-app' ),
							'id'    => 'bbapp_dev_settings',
							'desc'  => '',
							'class' => 'configure-subheadline',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.namespace',
						array(
							'label'       => __( 'Application ID', 'buddyboss-app' ),
							'desc'        => esc_html__( 'Select an Application ID to use in Android builds of your app.', 'buddyboss-app' ) . '<br/><br/>' . esc_html__( 'Google requires a unique Application ID for every app on their store, so it is recommended to use a reverse-domain naming convention such as: com.yourdomainname.yourappname', 'buddyboss-app' ) . '<br/>',
							'type'        => 'text',
							'value'       => '',
							'placeholder' => 'com.yourdomainname.yourappname',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish_android_dev_namespace',
						array(
							'label' => ' ',
							'type'  => 'html',
							'class' => function () {
								if ( empty( $this->option( 'publish.android.namespace' ) ) && empty( $this->option( 'publish.android.dev.namespace' ) ) ) {
									return 'hidden';
								}

								return '';
							},
							'value' => function () {
								$android_dev_namespace_id = '';

								if ( ! empty( $this->option( 'publish.android.dev.namespace' ) ) ) {
									$android_dev_namespace_id = $this->option( 'publish.android.dev.namespace' );
								} else {
									if ( ! empty( $this->option( 'publish.android.namespace' ) ) ) {
										$android_dev_namespace_id = $this->option( 'publish.android.namespace' ) . '.test';
										$this->save_option( 'publish.android.dev.namespace', $android_dev_namespace_id );
									}
								}

								return sprintf( '<p>%1$s <strong class="bbapp-android-dev-namespace-id" data-namespace-id="com.domain.app.test">%2$s</strong></p>', esc_html__( 'Test App Application ID:', 'buddyboss-app' ), $android_dev_namespace_id );
							},
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_publish_android_required_build',
						array(
							'label' => ' ',
							'type'  => 'html',
							'value' => '',
							'desc'  => '<div class="bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require a new Android build.', 'buddyboss-app' ) ) . '</p></div><br/>',
						)
					)
				);

				$iap_link = bbapp_is_active( 'iap' ) ? bbapp_get_admin_url( 'admin.php?page=bbapp-iap' ) : '#';

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_publish_android_registered_id_description',
						array(
							'label' => __( 'Register Application ID', 'buddyboss-app' ),
							'type'  => 'html',
							'value' => '',
							'desc'  => esc_html__( 'After creating an app in your Google Developer Account, you need to upload a build for it to register your Application ID in the Google Play Store. Once your app has a registered Application ID, Google will only allow you to submit new builds to the app with the same Application ID.', 'buddyboss-app' ) . '<br/><br/>' . esc_html__( 'You must have registered you Application ID in your Google Developer account in order to:', 'buddyboss-app' ) .
									sprintf(
										'<ul><li>%s <a href="%s">%s</a> %s</li><li>%s</li><li>%s <a href="%s">%s</a> %s <a href="%s">%s</a> %s</li></ul>',
										esc_html__( 'Sync in-app purchases created in your Google Developer account with those created on the', 'buddyboss-app' ),
										$iap_link,
										esc_html__( 'In-App Purchases', 'buddyboss-app' ),
										esc_html__( 'page', 'buddyboss-app' ),
										esc_html__( 'Publish and Android version of your app using the BuddyBoss Publishing Service', 'buddyboss-app' ),
										esc_html__( 'Link the', 'buddyboss-app' ),
										bbapp_get_admin_url( 'admin.php?page=bbapp-settings&setting=feedback#app_settings_app_rating_heading' ),
										esc_html__( 'Rate This App', 'buddyboss-app' ),
										esc_html__( 'and', 'buddyboss-app' ),
										bbapp_get_admin_url( 'admin.php?page=bbapp-settings&setting=smart_banner#app_settings_app_rating_heading' ),
										esc_html__( 'Smart Banner', 'buddyboss-app' ),
										esc_html__( 'features to your app in the Google Play Store after it is published', 'buddyboss-app' )
									),
							'class' => function () {
								if ( empty( $this->option( 'publish.android.namespace' ) ) || empty( $this->option( 'publish.android.account_key' ) ) ) {
									return 'hidden';
								}

								return '';
							},
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.namespace.registered',
						array(
							'label'          => ' ',
							'label_checkbox' => __( 'I\'ve registered this Application ID in Google Developer Account', 'buddyboss-app' ),
							'type'           => 'checkbox',
							'value'          => '',
							'class'          => function () {
								if ( empty( $this->option( 'publish.android.namespace' ) ) || empty( $this->option( 'publish.android.account_key' ) ) ) {
									return 'hidden';
								}

								return '';
							},
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_publish_android_connection_status_message_sub',
						array(
							'label' => ' ',
							'type'  => 'html',
							'value' => function () {
								$msgs = '';

								if ( ! empty( $this->option( 'publish.android.namespace' ) ) && 1 === $this->option( 'publish.android.namespace.registered' ) && ! empty( $this->option( 'publish.android.account_key' ) ) ) {
									$account_connection = Android::instance()->is_connected( true );

									if ( false === $account_connection || is_wp_error( $account_connection ) ) {
										$error_msgs = Android::instance()->get_connection_error_messages();
										$msgs       = '';
										$msgs_inr   = '';

										if ( ! empty( $error_msgs ) ) {
											foreach ( $error_msgs as $error_msg ) {
												$msgs_inr .= '<p>' . $error_msg['message'] . '</p>';
											}
										}

										$msgs .= '<div class="bbapp-notice-box bbapp-notice-box--warning"><div class="data-update-ajax"><p>' . esc_html__( 'Please ensure you\'ve uploaded a release app build to your app in the Google Developer Play Console. The build must have been generated using the Application ID entered above.', 'buddyboss-app' ) . '</p><p>' . esc_html__( 'If you\'ve already done so, make sure the Service Account Key you are connected with has the following permissions for your app:', 'buddyboss-app' ) . '</p><ul class="list-style"><li>' . esc_html__( 'View app information and download bulk reports', 'buddyboss-app' ) . '</li><li>' . esc_html__( 'View financial data, orders, and cancellation survey responses', 'buddyboss-app' ) . '</li><li>' . esc_html__( 'Create, edit and delete draft apps', 'buddyboss-app' ) . '</li><li>' . esc_html__( 'Release to production, exclude devices, and use Play App Signing', 'buddyboss-app' ) . '</li><li>' . esc_html__( 'Release apps to testing tracks', 'buddyboss-app' ) . '</li><li>' . esc_html__( 'Manage testing tracks and edit tester lists', 'buddyboss-app' ) . '</li></ul></div></div><br/>';
									}
								}

								return $msgs;
							},
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_publish_android_live_sub_headline',
						array(
							'type'  => 'subheading',
							'value' => __( 'KeyStore Information', 'buddyboss-app' ),
							'desc'  => '',
							'class' => 'configure-subheadline',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.import_keystore',
						array(
							'label'    => __( 'Import KeyStore', 'buddyboss-app' ),
							'desc'     => sprintf(
							/* translators: Generate KeyStore tool admin url. */
								__( 'Import a KeyStore created using our <a href="%s">Generate KeyStore</a> tool.', 'buddyboss-app' ),
								bbapp_get_admin_url( 'admin.php?page=bbapp-tools&setting=generate-keystore' )
							),
							'type'     => '\BuddyBossApp\Admin\Fields\Upload',
							'settings' => array(
								'allowed_ext'     => array( 'zip' ),
								'upload_btn_text' => __( 'Import', 'buddyboss-app' ),
								'upload_dir'      => 'bbapp-import',
							),
							'value'    => '',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.keystore',
						array(
							'label'    => __( 'KeyStore', 'buddyboss-app' ),
							'desc'     => __( 'Provide your Android app signing key.', 'buddyboss-app' ),
							'type'     => '\BuddyBossApp\Admin\Fields\Upload',
							'settings' => array(
								'allowed_ext'    => array( 'jks' ),
								'show_file_link' => true,
							),
							'value'    => '',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.keystore_pwd',
						array(
							'label'       => __( 'KeyStore Password', 'buddyboss-app' ),
							'desc'        => __( 'Enter a password for your keystore.', 'buddyboss-app' ),
							'type'        => 'password',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.keystore_alias',
						array(
							'label'       => __( 'KeyStore Alias', 'buddyboss-app' ),
							'desc'        => __( 'Enter an identifying name for your key.', 'buddyboss-app' ),
							'type'        => 'text',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.keystore_key_pwd',
						array(
							'label'       => __( 'KeyStore Key Password', 'buddyboss-app' ),
							'desc'        => __( 'Enter a password for your key.', 'buddyboss-app' ) . '<div class="bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require a new Android build.', 'buddyboss-app' ) ) . '<p></div>',
							'type'        => 'password',
							'value'       => '',
							'placeholder' => '',
						)
					)
				);

				/**
				 * Android App Permissions
				 */
				$this->settings_form->add_field(
					$this->settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-android-app-permissions bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_publish_android_permissions_configuration',
						array(
							'type'         => 'heading',
							'value'        => __( 'Permissions', 'buddyboss-app' ),
							'desc'         => '',
							'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=#' ) ),
							'icon'         => 'bb-icon-bf bb-icon-lock-open',
						)
					)
				);

				// Enable Android App recording permission.
				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.permissions.audio_recording',
						array(
							'label'          => __( 'Audio Recording', 'buddyboss-app' ),
							'label_checkbox' => __( 'Allow your app to record audio', 'buddyboss-app' ),
							'desc'           => __( 'When enabled audio recording will be allowed in your app.', 'buddyboss-app' ),
							'type'           => 'checkbox',
						)
					)
				);

				// Enable Android App media playback permission.
				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.permissions.foreground_media_playback',
						array(
							'label'          => __( 'Foreground Service <div class="bbapp-dev-only">Custom development use only</div>', 'buddyboss-app' ),
							'label_checkbox' => __( 'Allow your app to play media in the background', 'buddyboss-app' ),
							'desc'           => __( 'When enabled, background media playback will be allowed in your app, ensuring continuous audio or video playback even when the app is minimized.', 'buddyboss-app' ),
							'type'           => 'checkbox',
							'disabled'       => ! Configure::instance()->has_custom_developer_access(),
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.permissions.permission_build_button_html',
						array(
							'type'  => 'html',
							'value' => '<div class="bbapp-build-button bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require a new Android build.', 'buddyboss-app' ) ) . '</p></div>',
							'label' => '&nbsp;',
							'class' => '',
						)
					)
				);

				/**
				 * Android App Links Configuration
				 */
				$this->settings_form->add_field(
					$this->settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card bbapp-admin-android-app-links bbapp-admin-publish',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_publish_android_universal_links_configuration',
						array(
							'type'         => 'heading',
							'value'        => __( 'Android App Links', 'buddyboss-app' ),
							'desc'         => '',
							'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=125475' ) ),
							'icon'         => 'bb-icon-bf bb-icon-link',
						)
					)
				);

				// Enable Android App Links.
				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.universal_link',
						array(
							'label'          => __( 'Enable Android App Links', 'buddyboss-app' ),
							'label_checkbox' => __( 'Open links using your Site URL in the BuddyBoss App.', 'buddyboss-app' ),
							'desc'           => sprintf( '%1$s (<a href="%2$s">%3$s</a>) %4$s', esc_html__( 'When enabled, all links using your Site URL', 'buddyboss-app' ), get_home_url(), get_home_url(), esc_html__( 'will open in your app, as long as the app is installed on the member\'s Android device.', 'buddyboss-app' ) ),
							'type'           => 'checkbox',
						)
					)
				);

				$this->bbapp_android_assetlinks = home_url( 'assetlinks.json' );

				if ( function_exists( 'bbapp_is_valid_link' ) && ! bbapp_is_valid_link( $this->bbapp_android_assetlinks ) ) {
					$this->settings_form->add_field(
						$this->settings_provider->field(
							'app_android_universal_links_connection_status_message_sub',
							array(
								'label' => '&nbsp',
								'type'  => 'html',
								'value' => function () {
									return '<div class="bbapp-notice-box bbapp-notice-box--warning"><div class="data-update-ajax"><b>' . esc_html__( 'Digital Asset Links Not Accessible', 'buddyboss-app' ) . '</b><p>' . sprintf( 'Google needs access to the following URL in order to read the <a targe="_blank" href="%s" >digital asset links</a> for your android App: <a href="%s" >%s</a>', 'https://developers.google.com/digital-asset-links/v1/getting-started', $this->bbapp_android_assetlinks, $this->bbapp_android_assetlinks, 'buddyboss-app' ) . '</p><p class="bbapp-universal-links">' . esc_html__( 'To ensure Android App Links work as intended, please contact your host to enable access to this URL. ', 'buddyboss-app' ) . '</div>';
								},
								'desc'  => '',
								'class' => function () {
									$class        = 'universal_links_to_open';
									$get_settings = $this->get_settings();
									if ( empty( $get_settings ) || empty( $get_settings['publish.android.universal_link'] ) ) {
										$class .= ' hidden';
									}

									return $class;
								},
							)
						)
					);
				}

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.links_to_open',
						array(
							'label'     => __( 'Links to Open', 'buddyboss-app' ),
							'type'      => 'select',
							'options'   => array(
								'all'      => __( 'all', 'buddyboss-app' ),
								'specific' => __( 'specific', 'buddyboss-app' ),
							),
							'value'     => '',
							'pre_text'  => __( 'Open', 'buddyboss-app' ),
							'post_text' => __( 'Links in the BuddyBoss App', 'buddyboss-app' ),
							'desc'      => __( 'You can select for all or specific links using your Site URL to open in your app. Android does not support the functionality to exclude specific links.', 'buddyboss-app' ),
							'class'     => function () {
								$class        = 'universal_links_to_open';
								$get_settings = $this->get_settings();
								if ( empty( $get_settings ) || empty( $get_settings['publish.android.universal_link'] ) ) {
									$class .= ' hidden';
								}

								return $class;
							},
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.exclude_links',
						array(
							'label'       => '',
							'desc'        => __( 'Enter URLs or URI fragments (e.g. /groups/) to exclude from opening in your app. Enter one URL or URI per line', 'buddyboss-app' ),
							'type'        => 'textarea',
							'placeholder' => __( 'Add text', 'buddyboss-app' ),
							'value'       => '',
							'class'       => function () {
								$class        = 'universal_external_links';
								$get_settings = $this->get_settings();
								if ( empty( $get_settings ) || empty( $get_settings['publish.android.universal_link'] ) || empty( $get_settings['publish.android.links_to_open'] ) || 'all' === $get_settings['publish.android.links_to_open'] ) {
									$class .= ' hidden';
								}

								return $class;
							},
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.google_play_signing_certificate',
						array(
							'label'       => __( 'Google Play Signing Certificate', 'buddyboss-app' ),
							'type'        => 'text',
							'value'       => '',
							'placeholder' => __( 'Add text', 'buddyboss-app' ),
							'desc'        => sprintf( '%1$s <a target="_blank" href="https://support.google.com/googleplay/android-developer/answer/9842756?hl=en-GB">Google Play app signing</a>, %2$s <b>%3$s</b> %4$s <a target="_blank" href="https://play.google.com/console/developers">Google Play Developer Console</a> %5$s', esc_html__( 'If you are using', 'buddyboss-app' ), esc_html__( 'you need to enter the', 'buddyboss-app' ), esc_html__( 'SHA-256 certificate fingerprint', 'buddyboss-app' ), esc_html__( 'provided by Google so that the app can auto-verify your Android App Links. This can be found by navigating to your app in the', 'buddyboss-app' ), esc_html__( 'and going to Release > App integrity > App signing.', 'buddyboss-app' ) ),
							'class'       => function () {
								$class        = 'universal_links_to_open';
								$get_settings = $this->get_settings();
								if ( empty( $get_settings ) || empty( $get_settings['publish.android.universal_link'] ) ) {
									$class .= ' hidden';
								}

								return $class;
							},
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'publish.android.dev.universal_link_build_button_html',
						array(
							'type'  => 'html',
							'value' => '<div class="bbapp-build-button bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require a new Android build.', 'buddyboss-app' ) ) . '</p></div>',
							'label' => '&nbsp;',
							'class' => '',
						)
					)
				);

				$this->settings_form->add_field(
					$this->settings_provider->field(
						'app_publish_android_app_configuration_submit',
						array(
							'type'  => 'submit',
							'value' => __( 'Save Changes', 'buddyboss-app' ),
							'desc'  => '',
						)
					)
				);
			}

			/***
			 * Firebase Settings.
			 */
			if ( 'firebase' === $this->get_current_sub_tab() ) {
				$this->settings_form->add_field(
					$app_settings_provider->field(
						false,
						array(
							'type'    => 'table-open',
							'classes' => 'buddyboss-app-admin-card',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'firebase_heading',
						array(
							'type'         => 'heading',
							'value'        => esc_html__( 'Google Firebase', 'buddyboss-app' ),
							'desc'         => '',
							'tutorial_url' => esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-help&article=122280' ) ),
							'icon'         => 'bb-icon-bf bb-icon-brand-google-firebase',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'firebase_tabs_message',
						array(
							'label' => '',
							'type'  => 'html',
							'value' => function () {

								return '<div class="bbapp-notice-box bbapp-notice-box--warning"><p>' . esc_html__( 'Google Firebase is retiring their legacy Cloud Messaging API. For the time being, push notifications can still be sent through the legacy API, but we recommend connecting to their new API as soon as possible.', 'buddyboss-app' ) . '</p></div>';

							},
							'desc'  => '',
							'class' => function () {
								$class        = 'firebase_tabs_message';
								if ( empty( Configure::instance()->option( 'push.firebase_server_key' ) ) || ! empty( Configure::instance()->option( 'push.firebase_admin_key' ) ) ) {
									$class .= ' hidden';
								}

								return $class;
							},
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'firebase_tabs',
						array(
							'type'        => 'tabs',
							'no_label'    => true,
							'tabs'        => array(
								'push.firebase_admin_key'  => esc_html__( 'Firebase Cloud Messaging API (V1)', 'buddyboss-app' ),
								'push.firebase_server_key' => esc_html__( 'Cloud Messaging API (Legacy)', 'buddyboss-app' ),
							),
							'default_tab' => 'push.firebase_admin_key',
							'desc'        => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'push.firebase_admin_key_errors',
						array(
							'type'  => 'html',
							'value' => function () {
								global $app_settings_push_firebase_admin_key_invalid_file,$firebase_admin_key_invalid_file;

								if ( true === $app_settings_push_firebase_admin_key_invalid_file ) {
									return '<div class="bbapp-firebase-admin-key-error bbapp-notice-box bbapp-notice-box--error">' . __( 'There was a problem with the private key provided: File extension is invalid.', 'buddyboss-app' ) . '</div>';
								}

								if ( true === $firebase_admin_key_invalid_file ) {
									return '<div class="bbapp-firebase-admin-key-error bbapp-notice-box bbapp-notice-box--error">' . __( 'There was a problem with the private key provided: File is invalid.', 'buddyboss-app' ) . '</div>';
								}

								return '';
							},
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'push.firebase_admin_key',
						array(
							'label'    => esc_html__( 'Firebase Admin Key', 'buddyboss-app' ),
							'desc'     => sprintf( '%1$s <a href="https://console.firebase.google.com/" target="_blank">%2$s</a>.', esc_html__( 'To enable sending push notifications, please provide a private key for accessing the Firebase server APIs in your', 'buddyboss-app' ), esc_html__( 'Google Firebase Console', 'buddyboss-app' ) ),
							'type'     => '\BuddyBossApp\Admin\Fields\Upload',
							'settings' => array(
								'allowed_ext'    => array( 'json' ),
								'show_file_link' => true,
							),
							'value'    => '',
							'class'    => 'hidden',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'push.firebase_server_key',
						array(
							'label' => esc_html__( 'Google Firebase Server Key ', 'buddyboss-app' ),
							'desc'  => __( 'If you are newly integrating messaging into your app, use the latest Firebase Cloud Messaging API (V1). If you are an existing user of Cloud Messaging API (Legacy), please migrate to the latest Firebase Cloud Messaging API (V1).', 'buddyboss-app' ),
							'type'  => 'password',
							'value' => false,
							'class' => 'hidden',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_integration_ios_configuration_subheadline',
						array(
							'type'  => 'subheading',
							'value' => __( 'iOS Configuration', 'buddyboss-app' ),
							'desc'  => '',
							'class' => 'configure-subheadline',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.firebase_plist_errors',
						array(
							'type'  => 'html',
							'value' => function () {
								global $app_settings_publish_ios_firebase_plist_invalid_file, $app_settings_publish_ios_firebase_plist_validation, $app_settings_publish_ios_dev_firebase_plist_validation;

								$upload_dir         = wp_upload_dir();
								$publish_bundle_id  = self::option( 'publish.ios.namespace' );
								$firebase_bundle_id = get_option( 'bbapp_ios_firebase_plist_bundle_id', $publish_bundle_id );

								// Check if the bundle id is different from the one in the publish settings.
								if ( ! empty( $publish_bundle_id ) && $firebase_bundle_id !== $publish_bundle_id ) {

									$ios_firebase_plist = self::option( 'publish.ios.firebase_plist' );
									$file_path          = ! empty( $ios_firebase_plist ) ? $upload_dir['basedir'] . $ios_firebase_plist : '';
									$plist_validation   = self::validate_ios_firebase_plist( 'ios_firebase_plist', $file_path );

									if ( is_wp_error( $plist_validation ) ) {
										return '<div class="bbapp-ios-firebase-plist-error bbapp-notice-box bbapp-notice-box--error">' . esc_html( $plist_validation->get_error_message() ) . '</div>';
									}
								}

								$publish_dev_bundle_id  = self::option( 'publish.ios.dev.namespace' );
								$firebase_dev_bundle_id = get_option( 'bbapp_ios_dev_firebase_plist_bundle_id', $publish_dev_bundle_id );

								// Check if the bundle id is changed.
								if ( ! empty( $publish_dev_bundle_id ) && $firebase_dev_bundle_id !== $publish_dev_bundle_id ) {

									$ios_dev_firebase_plist = self::option( 'publish.ios.dev.firebase_plist' );
									$file_path              = ! empty( $ios_dev_firebase_plist ) ? $upload_dir['basedir'] . $ios_dev_firebase_plist : '';
									$plist_validation       = self::validate_ios_firebase_plist( 'ios_dev_firebase_plist', $file_path );

									if ( is_wp_error( $plist_validation ) ) {
										return '<div class="bbapp-ios-dev-firebase-plist-error bbapp-notice-box bbapp-notice-box--error">' . esc_html( $plist_validation->get_error_message() ) . '</div>';
									}
								}

								if ( true === $app_settings_publish_ios_firebase_plist_invalid_file ) {
									return '<div class="bbapp-ios-firebase-plist-error bbapp-notice-box bbapp-notice-box--error">' . __( 'There was a problem with the private key provided: File extension is invalid.', 'buddyboss-app' ) . '</div>';
								}

								if ( is_wp_error( $app_settings_publish_ios_firebase_plist_validation ) ) {
									return '<div class="bbapp-ios-firebase-plist-error bbapp-notice-box bbapp-notice-box--error">' . esc_html( $app_settings_publish_ios_firebase_plist_validation->get_error_message() ) . '</div>';
								}

								if ( is_wp_error( $app_settings_publish_ios_dev_firebase_plist_validation ) ) {
									return '<div class="bbapp-ios-firebase-plist-error bbapp-notice-box bbapp-notice-box--error">' . esc_html( $app_settings_publish_ios_dev_firebase_plist_validation->get_error_message() ) . '</div>';
								}

								return '';
							},
							'desc'  => '',
						)
					)
				);

				$build_link = bbapp_get_admin_url( 'admin.php?page=bbapp-build&setting=request-build' );

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.firebase_plist',
						array(
							'label'    => __( 'iOS Release App Configuration File', 'buddyboss-app' ),
							'desc'     => sprintf( __( 'Go to your <a href="https://console.firebase.google.com" target="_blank">Google Firebase Console</a> to download the GoogleService-Info.plist file for your iOS release app.', 'buddyboss-app' ) ),
							'type'     => '\BuddyBossApp\Admin\Fields\Upload',
							'settings' => array(
								'allowed_ext'    => array( 'plist' ),
								'show_file_link' => true,
								'validate_name'  => 'ios_firebase_plist',
								'validation'     => array( $this, 'callback_validate_ios_firebase_plist' ),
							),
							'value'    => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.ios.dev.firebase_plist',
						array(
							'label'    => __( 'iOS Test App Configuration File', 'buddyboss-app' ),
							'desc'     => sprintf( '%1$s <a href="https://console.firebase.google.com" target="_blank">Google Firebase Console</a> %2$s', esc_html__( 'Go to your', 'buddyboss-app' ), esc_html__( 'to download the GoogleService-Info.plist file for your iOS test app.', 'buddyboss-app' ) ) . '<div class="bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require a new iOS build.', 'buddyboss-app' ) ) . '<p></div>',
							'type'     => '\BuddyBossApp\Admin\Fields\Upload',
							'settings' => array(
								'allowed_ext'    => array( 'plist' ),
								'show_file_link' => true,
								'validate_name'  => 'ios_dev_firebase_plist',
								'validation'     => array( $this, 'callback_validate_ios_firebase_plist' ),
							),
							'value'    => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'app_integration_android_configuration_subheadline',
						array(
							'type'  => 'subheading',
							'value' => __( 'Android Configuration', 'buddyboss-app' ),
							'desc'  => '',
							'class' => 'configure-subheadline',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.firebase_json_errors',
						array(
							'type'  => 'html',
							'value' => function () {
								global $app_settings_publish_android_firebase_json_invalid_file, $app_settings_publish_android_firebase_json_validation, $app_settings_publish_android_dev_firebase_json_validation;

								$upload_dir         = wp_upload_dir();
								$firebase_bundle_id = get_option( 'bbapp_android_firebase_json_bundle_id' );
								$publish_bundle_id  = self::option( 'publish.android.namespace' );

								// Check if the bundle id is changed.
								if ( ! empty( $publish_bundle_id ) && $firebase_bundle_id !== $publish_bundle_id ) {

									$android_firebase_json = self::option( 'publish.android.firebase_json' );
									$file_path             = ! empty( $android_firebase_json ) ? $upload_dir['basedir'] . $android_firebase_json : '';
									$json_validation       = self::validate_android_firebase_json( 'android_firebase_json', $file_path );

									if ( is_wp_error( $json_validation ) ) {
										return '<div class="bbapp-android-firebase-json-error bbapp-notice-box bbapp-notice-box--error">' . esc_html( $json_validation->get_error_message() ) . '</div>';
									}
								}

								$firebase_dev_bundle_id = get_option( 'bbapp_android_dev_firebase_json_bundle_id' );
								$publish_dev_bundle_id  = self::option( 'publish.android.dev.namespace' );

								// Check if the bundle id is changed.
								if ( ! empty( $publish_dev_bundle_id ) && $firebase_dev_bundle_id !== $publish_dev_bundle_id ) {

									$android_dev_firebase_json = self::option( 'publish.android.dev.firebase_json' );
									$file_path                 = ! empty( $android_dev_firebase_json ) ? $upload_dir['basedir'] . $android_dev_firebase_json : '';
									$json_validation           = self::validate_android_firebase_json( 'android_dev_firebase_json', $file_path );

									if ( is_wp_error( $json_validation ) ) {
										return '<div class="bbapp-android-dev-firebase-json-error bbapp-notice-box bbapp-notice-box--error">' . esc_html( $json_validation->get_error_message() ) . '</div>';
									}
								}

								if ( true === $app_settings_publish_android_firebase_json_invalid_file ) {
									return '<div class="bbapp-android-firebase-json-error bbapp-notice-box bbapp-notice-box--error">' . __( 'There was a problem with the private key provided: File extension is invalid.', 'buddyboss-app' ) . '</div>';
								}

								if ( is_wp_error( $app_settings_publish_android_firebase_json_validation ) ) {
									return '<div class="bbapp-android-firebase-json-error bbapp-notice-box bbapp-notice-box--error">' . esc_html( $app_settings_publish_android_firebase_json_validation->get_error_message() ) . '</div>';
								}

								if ( is_wp_error( $app_settings_publish_android_dev_firebase_json_validation ) ) {
									return '<div class="bbapp-android-firebase-json-error bbapp-notice-box bbapp-notice-box--error">' . esc_html( $app_settings_publish_android_dev_firebase_json_validation->get_error_message() ) . '</div>';
								}

								return '';
							},
							'desc'  => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.firebase_json',
						array(
							'label'    => __( 'Android Release App Configuration File', 'buddyboss-app' ),
							'desc'     => sprintf( __( 'Go to your <a href="https://console.firebase.google.com" target="_blank">Google Firebase Console</a> to download the GoogleService-Info.json file for your Android release app.', 'buddyboss-app' ) ),
							'type'     => '\BuddyBossApp\Admin\Fields\Upload',
							'settings' => array(
								'allowed_ext'    => array( 'json' ),
								'show_file_link' => true,
								'validate_name'  => 'android_firebase_json',
								'validation'     => array( $this, 'callback_validate_android_firebase_json' ),
							),
							'value'    => '',
						)
					)
				);

				$this->settings_form->add_field(
					$app_settings_provider->field(
						'publish.android.dev.firebase_json',
						array(
							'label'    => __( 'Android Test App Configuration File', 'buddyboss-app' ),
							'desc'     => sprintf( '%1$s <a href="https://console.firebase.google.com" target="_blank">Google Firebase Console</a> %2$s', esc_html__( 'Go to your', 'buddyboss-app' ), esc_html__( 'to download the GoogleService-Info.json file for your Android test app.', 'buddyboss-app' ) ) . '<div class="bbapp_notice bbapp_notice--context bbapp_notice--static"><p>' . sprintf( '<a href="%1$s">%2$s</a>', esc_url( $build_link ), esc_html__( 'Changes will require a new Android build.', 'buddyboss-app' ) ) . '<p></div>',
							'type'     => '\BuddyBossApp\Admin\Fields\Upload',
							'settings' => array(
								'allowed_ext'    => array( 'json' ),
								'show_file_link' => true,
								'validate_name'  => 'android_dev_firebase_json',
								'validation'     => array( $this, 'callback_validate_android_firebase_json' ),
							),
							'value'    => '',
						)
					)
				);
			}
		}
	}

	/**
	 * Ajax function to Registered the iOS devices.
	 */
	public function bbapp_save_ios_devices() {
		if ( ! Permissions::instance()->can_manage_app() ) {
			wp_send_json_error(
				array(
					'message' => __( 'You don\'t have permissions to do this.', 'buddyboss-app' ),
				)
			);
		}

		$name               = ( isset( $_POST['device_name'] ) ? bbapp_input_clean( wp_unslash( $_POST['device_name'] ) ) : '' ); //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$udid               = ( isset( $_POST['device_udid'] ) ? bbapp_input_clean( wp_unslash( $_POST['device_udid'] ) ) : '' ); //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$create_new_profile = false;

		if ( ! empty( $name ) && ! empty( $udid ) ) {
			$registered = Apple::instance()->register_development_device( $name, $udid );

			if ( $registered ) {
				$create_new_profile = true;
			}
		}

		// Create new provisioning profile.
		if ( $create_new_profile ) {
			Apple::instance()->update_provisioning_profile( true );
			wp_send_json_success(
				array(
					'device_name' => $name,
					'device_udid' => $udid,
				)
			);
		}

		wp_send_json_error(
			array(
				'message' => __( 'There has been a problem. Please try again.', 'buddyboss-app' ),
			)
		);

		die();
	}

	/**
	 * Saves the settings
	 *
	 * @param array $settings Settings array.
	 *
	 * @return bool
	 */
	public function save_settings( $settings ) {
		if ( isset( $settings['publish.android.import_keystore'] ) && ! empty( $settings['publish.android.import_keystore'] ) ) {
			$upload     = wp_upload_dir();
			$file_path  = $upload['basedir'] . $settings['publish.android.import_keystore'];
			$file_name  = basename( $file_path, '.zip' );
			$dir        = Tools\ImportExport::instance()->get_import_folder( false ) . '/' . $file_name;
			$zip        = new \ZipArchive();
			$zip_result = $zip->open( $file_path );

			if ( true === (bool) $zip_result ) {
				if ( ! file_exists( $dir ) ) {
					BBAPP_File::create_dir( $dir );
				}

				$zip->extractTo( $dir );
				$zip->close();
			}

			$files = scandir( $dir, true );
			$jks   = preg_grep( '~\.(jks)$~', $files );
			$json  = preg_grep( '~\.(json)$~', $files );

			if ( ! class_exists( '\WP_Filesystem_Direct' ) ) {
				require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';
				require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php';
			}

			$wp_files_system = new \WP_Filesystem_Direct( new \stdClass() );

			if ( ! empty( $jks ) ) {
				$jks_file    = $dir . '/' . current( $jks );
				$destination = '/bbapp/uploads/' . $file_name . '.jks';
				$copy        = $wp_files_system->copy( $jks_file, $upload['basedir'] . $destination, true, 0644 );

				if ( ! empty( $copy ) && ! is_wp_error( $copy ) ) {
					$settings['publish.android.keystore'] = $destination;
				}
			}

			if ( ! empty( $json ) ) {
				$json_file = $dir . '/' . current( $json );
				$json_data = $wp_files_system->get_contents( $json_file );

				if ( ! empty( $json_data ) && ! is_wp_error( $json_data ) ) {
					$field = json_decode( $json_data, true );
					if ( ! empty( $field['password'] ) ) {
						$settings['publish.android.keystore_pwd'] = $field['password'];
					}
					if ( ! empty( $field['alias_name'] ) ) {
						$settings['publish.android.keystore_alias'] = $field['alias_name'];
					}

					if ( ! empty( $field['storepass'] ) ) {
						$settings['publish.android.keystore_key_pwd'] = $field['storepass'];
					}
				}
			}

			$settings['publish.android.import_keystore'] = '';
			$wp_files_system->rmdir( $dir, true );
			$wp_files_system->delete( $file_path );
		}

		if ( isset( $settings['publish.android.namespace'] ) && ! empty( $settings['publish.android.namespace'] ) ) {
			$settings['publish.android.dev.namespace'] = $settings['publish.android.namespace'] . '.test';
		} else {
			$settings['publish.android.dev.namespace'] = '';
		}

		$regex = '/^[A-Za-z0-9._-]+$/';
		global $bbapp_ios_namespace_err;

		if ( ! empty( $settings['publish.ios.namespace'] ) && ! preg_match( $regex, $settings['publish.ios.namespace'] ) ) {
			$bbapp_ios_namespace_err           = true;
			$settings['publish.ios.namespace'] = '';
		}

		global $bbapp_ios_dev_namespace_err;

		if ( ! empty( $settings['publish.ios.dev.namespace'] ) && ! preg_match( $regex, $settings['publish.ios.dev.namespace'] ) ) {
			$bbapp_ios_dev_namespace_err           = true;
			$settings['publish.ios.dev.namespace'] = '';
		}

		// Don't update the core branch if it isn't exist in development branch from appcenter.
		$get_app_settings = ManageApp::instance()->get_app_settings();

		if ( ! empty( $this->get_core_development_branch() ) && ! in_array( $settings['publish.general.core.branch'], array_keys( $this->get_core_development_branch() ), true ) ) {
			$settings['publish.general.core.branch'] = $get_app_settings['publish.general.core.branch'];
		}

		if ( ! empty( $this->get_core_development_branch() ) && ! in_array( $settings['publish.general.core.dev_branch'], array_keys( $this->get_core_development_branch() ), true ) ) {
			$settings['publish.general.core.dev_branch'] = $get_app_settings['publish.general.core.dev_branch'];
		}
		$update_android_sha256cert = false;

		// If Android application ID is changed than regenerate android sha256 certificate.
		if ( ( isset( $settings['publish.android.namespace'] ) && $settings['publish.android.namespace'] !== $get_app_settings['publish.android.namespace'] ) || ( isset( $settings['publish.android.universal_link'] ) && $settings['publish.android.universal_link'] !== $get_app_settings['publish.android.universal_link'] ) ) {
			$update_android_sha256cert = true;

		}

		/**
		 * Sync the core branches information with buddyboss appcenter.
		 */
		if ( isset( $settings['push.firebase_admin_key'] ) && ! empty( $settings['push.firebase_admin_key'] ) ) {

			$upload_dir         = wp_upload_dir();
			$firebase_admin_key = $settings['push.firebase_admin_key'];
			if ( ! empty( $firebase_admin_key ) && file_exists( $upload_dir['basedir'] . $firebase_admin_key ) ) {
				$file_data     = BBAPP_File::read_file( $upload_dir['basedir'] . $firebase_admin_key );
				$file_data     = json_decode( $file_data, true );
				$required_keys = array(
					'type',
					'project_id',
					'private_key_id',
					'private_key',
					'client_email',
					'client_id',
					'auth_uri',
					'token_uri',
				);
				foreach ( $required_keys as $required_key ) {
					if ( ! isset( $file_data[ $required_key ] ) ) {
						global $firebase_admin_key_invalid_file;
						$firebase_admin_key_invalid_file     = true;
						$settings['push.firebase_admin_key'] = '';
					}
				}

				if ( ! empty( $settings['push.firebase_admin_key'] ) && ! empty( $file_data['project_id'] ) ) {
					bbapp_set_option( 'bbapp_firebease_project_id', $file_data['project_id'], null, get_current_network_id() );
				} else {
					bbapp_delete_option( 'bbapp_firebease_project_id', get_current_network_id() );
				}
			}
		}

		ManageApp::instance()->update_app_settings( $settings );

		/**
		 * Sync the custom branches information with buddyboss appcenter.
		 */
		// If Android application ID is changed then regenerate android sha256 certificate.
		if ( $update_android_sha256cert && function_exists( 'bbapp_generate_android_sha256cert' ) ) {
			bbapp_generate_android_sha256cert();
		}

		$args = array();

		if ( isset( $settings['publish.general.branch'] ) && ! empty( $settings['publish.general.branch'] ) ) {
			$args['custom_live_ref'] = $settings['publish.general.branch'];
		}

		if ( isset( $settings['publish.general.dev.branch'] ) && ! empty( $settings['publish.general.dev.branch'] ) ) {
			$args['custom_dev_ref'] = $settings['publish.general.dev.branch'];
		}

		// Sync the core branches information with buddyboss appcenter.
		if ( isset( $settings['publish.general.core.branch'] ) && ! empty( $settings['publish.general.core.branch'] ) ) {
			$args['live_ref'] = $settings['publish.general.core.branch'];
		}

		if ( isset( $settings['publish.general.core.dev_branch'] ) && ! empty( $settings['publish.general.core.dev_branch'] ) ) {
			$args['dev_ref'] = $settings['publish.general.core.dev_branch'];
		}

		$args['bbapp_id']                  = ManageApp::instance()->get_app_id();
		$android_connection                = \BuddyBossApp\AppStores\Android::instance()->is_google_connected();
		$apple_connection                  = \BuddyBossApp\AppStores\Apple::instance()->is_connected();
		$args['bbapp_key']                 = ManageApp::instance()->get_auth_app_key();
		$args['bbapp_is_google_connected'] = ( is_wp_error( $android_connection ) ? false : (bool) $android_connection );
		$args['bbapp_is_apple_connected']  = ( is_wp_error( $apple_connection ) ? false : (bool) $apple_connection );

		$this->update_custom_development_branch( $args );

		return true;
	}

	/**
	 * Gets the settings
	 */
	public function get_settings() {
		$settings = ManageApp::instance()->get_app_settings();

		return $settings;
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {
		$tabs = array(
			'general'     => __( 'General', 'buddyboss-app' ),
			'ios'         => __( 'iOS Settings', 'buddyboss-app' ),
			'ios_devices' => __( 'iOS Devices', 'buddyboss-app' ),
			'android'     => __( 'Android Settings', 'buddyboss-app' ),
			'firebase'    => __( 'Google Firebase', 'buddyboss-app' ),
		);

		return $tabs;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {
		$default      = 'general';
		$setting      = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Renders the setting screen.
	 */
	public function render_screen() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			echo '<p>' . esc_html__( "You don't have permission to access this page.", 'buddyboss-app' ) . '</p>';

			return false;
		}

		$bbapp = ManageApp::instance()->get_app();

		if ( ! empty( $bbapp ) && ! empty( $bbapp['verified'] ) && ! empty( $bbapp['bbapp_site_type'] ) && 'primary' === $bbapp['bbapp_site_type'] ) {
			\BuddyBossApp\Admin\SetupAdmin::instance()->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab() );

			$save_button = true;

			if ( $this->get_current_sub_tab() === 'ios' && isset( $this->settings_form_ios_connect ) && $this->settings_form_ios_connect ) {
				$this->settings_form_ios_connect->render_form( false );
				$save_button = false;
			}

			if ( $this->get_current_sub_tab() === 'ios' && isset( $this->settings_form_ios_configuration ) && $this->settings_form_ios_configuration ) {
				$this->settings_form_ios_configuration->render_form( false );
				$save_button = false;
			}

			if ( $this->get_current_sub_tab() === 'ios_devices' ) {
				$save_button = false;
			}

			if ( $this->get_current_sub_tab() === 'android' && isset( $this->settings_form_android_connect ) && $this->settings_form_android_connect ) {
				$this->settings_form_android_connect->render_form( false );
				$save_button = false;
			}

			if ( isset( $this->settings_form ) && $this->settings_form ) {
				$this->settings_form->render_form( $save_button );
				?>
				<script>
                    function enableIosButtons() {
                        let ext = '';
                        const parent = jQuery( '.buddyboss-app-admin-card' );
                        const FileVal = parent.find( '*[data-id="app_settings_publish__ios__private_key"] input[type="file"]' ).val() || parent.find( '*[data-id="app_settings_publish__ios__private_key"] .show_preview:visible' ).find( '.show_preview__name' ).attr( 'href' );
                        const issuerId = parent.find( 'input[name="app_settings[publish.ios.issuer_id]"]' ).val();
                        const keyId = parent.find( 'input[name="app_settings[publish.ios.key_id]"]' ).val();

                        if ( FileVal !== '' && typeof FileVal !== 'undefined' ) {
                            ext = FileVal.split( '.' ).pop();
                        }

                        if ( '' !== FileVal && issuerId !== '' && keyId !== '' && 'p8' === ext ) {
                            parent.find( '.bbapp_apple_disconnected .bbapp-save-setting-btn' ).prop( 'disabled', false );
                        } else {
                            parent.find( '.bbapp_apple_disconnected .bbapp-save-setting-btn' ).prop( 'disabled', true );
                        }
                    }

                    jQuery( document ).ready( function() {
                        jQuery( '.buddyboss-app-admin-card' ).find( '*[data-id="app_settings_publish__ios__private_key"] .bbapp-upload-field-delete-btn' ).on( 'click', function() {
                            setTimeout( function() {
                                enableIosButtons();
                            }, 400 );
                        } );

                        // when generate new profile button clicked.
                        jQuery( '.bbapp-generate-new-profiles' ).click( function( e ) {
                            e.preventDefault();

							//var confirm = window.confirm ( "<?php esc_html_e( 'Are you sure you want to regenerate provisional profile ?', 'buddyboss-app' ); ?>" );

							if ( ! confirm ) {
								return false;
							}

							const _this = jQuery( this );
                            _this.find( '.bbapp_loading' ).show();
                            _this.prop( 'disabled', true );
                            const post = jQuery.post( ajaxurl, {
                                nonce: _this.data( 'nonce' ),
                                action: 'bbapp_regenerate_provisional_profile',
                            } );

                            post.done( function( data ) {
                                if ( data === '0' || data === '' ) {
                                    // eslint-disable-next-line no-alert
									alert ( "<?php esc_attr_e( 'There is an unknown error while regenerating provisional profile', 'buddyboss-app' ); ?>");
                                } else {
                                    if ( data.success ) {
                                        window.location.reload();
                                    }

                                    // eslint-disable-next-line no-alert
                                    alert( data.data );
                                }
                            } );

                            post.fail( function() {
                                // eslint-disable-next-line no-alert
                                alert("<?php esc_attr_e( 'There is an unknown error while regenerating provisional profile', 'buddyboss-app' ); ?>");
                            } );

                            post.always( function() {
                                _this.find( '.bbapp_loading' ).hide();
                                _this.prop( 'disabled', false );
                            } );
                        } );

                        // Event to trigger on load.
                        let currentRequest = null;

                        jQuery( '.bbapp_field_app_settings_publish__android__namespace' ).on( 'input', function() {
                            jQuery( '#bbapp_field_app_settings_publish__android__namespace__registered' ).prop( 'checked', false );

                            const curObj = jQuery( this );
                            const value = jQuery( this ).val();
                            const bbappAndroidDevNamespaceSelector = jQuery( '.bbapp-android-dev-namespace-id' );
                            const appSettingsPublishAndroidDevNamespace = jQuery( '.bbapp_field_wrapper_app_settings_publish_android_dev_namespace' );

                            if ( '' === value ) {
                                appSettingsPublishAndroidDevNamespace.addClass( 'hidden' );
                                bbappAndroidDevNamespaceSelector.text( bbappAndroidDevNamespaceSelector.attr( 'data-namespace-id' ) );
                            } else {
                                appSettingsPublishAndroidDevNamespace.removeClass( 'hidden' );
                                bbappAndroidDevNamespaceSelector.text( value + '.test' );
                            }

                            curObj.parent().find( '.bbapp_loading' ).remove();
                            curObj.parent().find( '.bbapp-namespace-validation-msgs' ).remove();
                            curObj.parent().find( '.notice-success' ).remove();

                            if ( ! value ) {
                                if ( currentRequest !== null ) {
                                    currentRequest.abort();
                                }

                                return false;
                            }

                            curObj.parent().find( '.bbapp-admin-description' ).after( '<div class="bbapp_loading dark"></div>' );

                            const data = {
                                action: 'bbapp_validate_namespace',
                                name: value,
                            };

                            if ( curObj.hasClass ( 'bbapp_field_app_settings_publish__android__namespace' ) ) {
                                data.platform = 'android';
                            } else {
                                data.platform = 'ios';
                            }

                            currentRequest = jQuery.ajax( {
                                type: 'POST',
                                url: ajaxurl,
                                data,
                                beforeSend() {
                                    if ( currentRequest !== null ) {
                                        currentRequest.abort();
                                    }
                                },
                                success( response ) {
                                    curObj.parent().find( '.bbapp-admin-description' ).after( '<div class="bbapp-namespace-validation-msgs">' + response.data.message + '</div>' );
                                    curObj.parent().find( '.bbapp_loading' ).remove();
                                },
                            } );
                        } );

                        jQuery ( '*[data-id="app_settings_publish__android__account_key"] input[type="file"]' ).change ( function() {
                            const FileVal = jQuery ( this ).val ();
                            const ext = FileVal.split ( '.' ).pop ();
                            const ConnectBtnEl = jQuery ( '.bbapp_field_wrapper_app_settings_app_publish_android_connection_submit button.bbapp-save-setting-btn' );

                            if ( ''===FileVal || 'json'!==ext ) {
                                ConnectBtnEl.prop ( 'disabled', true );
                            } else {
                                ConnectBtnEl.prop ( 'disabled', false );
                            }
                        } );

                        jQuery( '*[data-id="app_settings_publish__ios__private_key"] input[type="file"]' ).change( function() {
                            enableIosButtons();
                        } );

						jQuery( 'input[name="app_settings[publish.ios.issuer_id]"], input[name="app_settings[publish.ios.key_id]"]' ).keyup( function () {
							enableIosButtons();
						} );

                        enableIosButtons ();

                        jQuery ( document ).on ( 'click', '.refresh-bundle-ids', function( e ) {
                            e.preventDefault ();

                            jQuery ( '.refresh-bundle-ids' ).find ( '.bbapp_loading' ).fadeIn ();
                            jQuery ( this ).prop ( 'disabled', true );

                            jQuery.ajax( {
                                url: bbappAjax.ajaxurl,
                                type: 'POST',
                                data: {
                                    action: 'refresh_bundle_ids',
                                },
                            } ).done( function() {
                                location.reload();
                            } ).fail( function() {
                                location.reload();
                            } );
                        } );

                        jQuery( document ).on( 'change', 'input[name="app_settings[publish.ios.universal_link]"]', function() {
                            if ( jQuery( this ).prop( 'checked' ) ) {
                                jQuery( '.universal_external_links' ).removeClass( 'hidden' );
                            } else {
                                jQuery( '.universal_external_links' ).addClass( 'hidden' );
                            }
                        } );

                        jQuery( document ).on( 'change', 'input[name="app_settings[publish.android.universal_link]"]', function() {
                            if ( jQuery( this ).prop( 'checked' ) ) {
                                jQuery( '.universal_links_to_open' ).removeClass( 'hidden' );
                                jQuery( 'select[name="app_settings[publish.android.links_to_open]"]' ).trigger( 'change' );
                            } else {
                                jQuery( '.universal_links_to_open' ).addClass( 'hidden' );
                                jQuery( '.universal_external_links' ).addClass( 'hidden' );
                            }
                        } );

                        // Add event listener for input change event
                        jQuery( '#app_settings_publish__ios__app_specific_password' ).on( 'input', function () {
                            const curObj = jQuery( this );
                            const saveBtn = jQuery( '#update-settings-2 .bbapp-save-setting-btn, #update-settings-3 .bbapp-save-setting-btn' );
                            const inputVal = curObj.val();
                            const errorContainer = curObj.parent().find( '.bbapp-invalid-app-specific-psw' );
                            const regexPattern = /^([a-z]{4}-){3}[a-z]{4}$/;
                            const bbappAppSpecificPasswordSuccessMsg = '<?php echo __( 'Please enter valid app-specific password. i.e. <strong>abcd-efgh-igkl-mnop</strong>', 'buddyboss-app' ); ?>';

                            // Remove existing error message
                            errorContainer.remove();

                            // Enable the save button by default
                            saveBtn.prop( 'disabled', false );

                            // Check if input value is undefined or empty
                            if( typeof inputVal === 'undefined' || !inputVal.trim() ) {
                                return;
                            }

                            // Validate the input value against the regex pattern
                            if( !regexPattern.test( inputVal ) ) {
                                // If validation fails, disable the save button and display an error message
                                saveBtn.prop( 'disabled', true );
                                curObj.parent().append( '<div class="bbapp-invalid-app-specific-psw">' + bbappAppSpecificPasswordSuccessMsg + '</div>' );
                            }
                        } );

                        jQuery( document ).on( 'change', 'select[name="app_settings[publish.android.links_to_open]"]', function() {
                            if ( 'specific' === jQuery( this ).val() ) {
                                jQuery( '.universal_external_links' ).removeClass( 'hidden' );
                            } else {
                                jQuery( '.universal_external_links' ).addClass( 'hidden' );
                            }
                        } );
                    } );
                </script>
				<?php
			}
		} elseif ( ! empty( $bbapp ) && ( ! empty( $bbapp['verified'] ) && ! empty( $bbapp['bbapp_site_type'] ) && 'secondary' === $bbapp['bbapp_site_type'] ) ) {
			$primary_url       = \BuddyBossApp\ManageApp::instance()->get_primary_site_url();
			$primary_admin_url = \BuddyBossApp\ManageApp::instance()->get_primary_site_admin_url();
			?>
			<div class="bbapp-build-container">
				<div class="bbapp-maintenance-block text-left">
					<div class="bbapp-dev-connect flex">
						<div class="bbapp-left-part">
							<p><?php esc_attr_e( 'Your app can only be configured on your primary site: ', 'buddyboss-app' ); ?>
								<br/>
								<?php
								if ( ! empty( $primary_url ) ) {
									?>
								<a class="link-primary-site" href="<?php echo esc_url( $primary_url ); ?>">
									<?php echo esc_url( $primary_url ); ?>
								</a>
							</p>
							<?php } ?>
						</div>
						<?php if ( ! empty( $primary_admin_url ) ) { ?>
							<div class="bbapp-right-part">
								<p>
									<a class="button button-secondary" target="_blank" href="<?php echo esc_url( trailingslashit( $primary_admin_url ) . 'wp-admin/admin.php?page=bbapp-configure' ); ?>">
										<?php esc_html_e( 'Configure App', 'buddyboss-app' ); ?>
									</a>
								</p>
							</div>
						<?php } ?>
					</div>
				</div>
			</div>
			<?php
		} else {
			?>
			<div class="bbapp-build-container">
				<div class="bbapp-maintenance-block text-left">
					<p><?php esc_attr_e( 'Please connect your app to BuddyBoss before configuring your app.', 'buddyboss-app' ); ?></p>
					<p><a class="button button-primary" href="<?php echo esc_attr( bbapp_get_super_admin_url( 'admin.php?page=bbapp-connect' ) ); ?>">
							<?php esc_html_e( 'Connect App', 'buddyboss-app' ); ?>
						</a>
					</p>
				</div>
			</div>
			<?php
		}

		?>
		<script>
            jQuery( document ).ready( function() {
                /**
                 * Run this function when the disconnect button is clicked.
                 */
                jQuery( '.bbapp_apple_connected' ).find( 'button' ).click( function( e ) {
                    e.preventDefault();

                    jQuery( this ).prop( 'disabled', true );

                    bbappRunAjax( 'disconnect_apple_account' );
                } );

                /**
                 * Run this function when the disconnect button is clicked.
                 */
                jQuery( '.bbapp_android_connected' ).find( 'button' ).click( function( e ) {
                    e.preventDefault();

                    jQuery( this ).prop( 'disabled', true );

                    bbappRunAjax( 'disconnect_android_account' );
                } );

                /**
                 * Call the ajax function based on action name.
                 *
                 * @param {string} actionName The name of the action to call.
                 */
                function bbappRunAjax( actionName ) {
                    jQuery.ajax( {
                        url: bbappAjax.ajaxurl,
                        type: 'POST',
                        data: {
                            action: actionName,
                        },
                    } ).done( function() {
                        location.reload();
                    } ).fail( function() {
                        location.reload();
                    } );
                }
            } );
        </script>
		<?php
	}

	/**
	 * Get settings - option value.
	 *
	 * @param string $field_name Field name.
	 * @param false  $default    Default value.
	 *
	 * @return false
	 */
	public function option( $field_name, $default = false ) {
		if ( ! empty( $this->settings_provider ) ) {
			return $this->settings_provider->get_value( $field_name, $default, true );
		}

		return false;
	}

	/**
	 * Function to save the option.
	 *
	 * @param string $field_name Field name.
	 * @param bool   $default    Default value to save.
	 *
	 * @return null|array
	 */
	public function save_option( $field_name, $default = false ) {
		$settings = $this->settings_provider->get_settings();

		if ( ! empty( $field_name ) ) {
			$settings[ $field_name ] = $default;
		}

		return ManageApp::instance()->update_app_settings( $settings );
	}

	/**
	 * Update QR code for device.
	 */
	public function bbapp_update_ios_qr_code() {
		if ( ! Permissions::instance()->can_manage_app() ) {
			wp_send_json_error(
				array(
					'message' => __( 'You don\'t have permissions to do this.', 'buddyboss-app' ),
				)
			);
		}

		$name       = ( isset( $_POST['device_name'] ) ? bbapp_input_clean( wp_unslash( $_POST['device_name'] ) ) : '' ); //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$device_url = \BuddyBossApp\Builds\Installer::instance()->get_device_register_url();

		if ( ! empty( $name ) ) {
			$device_url = add_query_arg(
				array(
					'device_owner_name' => $name,
				),
				$device_url
			);
		}

		$response = array(
			'url' => ( \BuddyBossApp\Library\Composer::instance()->qr_code_instance()->qr_code() )->render( $device_url ),
		);

		wp_send_json_success( $response );
		die();
	}

	/**
	 * Refresh iOS devices.
	 *
	 * @todo : Add Noune Support
	 */
	public function bbapp_refresh_ios_devices() {
		if ( ! Permissions::instance()->can_manage_app() ) {
			return false;
		}

		$devices = Apple::instance()->get_local_devices( true );
		$data    = array();

		if ( ! empty( $devices ) && ! is_wp_error( $devices ) ) {
			$devices = array_reverse( $devices );
			foreach ( (array) $devices as $device ) {
				$data[] = array(
					'device_name' => $device['name'],
					'device_udid' => $device['udid'],
				);
			}
		}

		wp_send_json_success( $data );
	}

	/**
	 * Check user has access for custom development or not.
	 *
	 * @return bool
	 */
	public function has_custom_developer_access() {
		$app_info = \BuddyBossApp\ManageApp::instance()->get_app_info();

		if ( ! is_wp_error( $app_info ) && ! empty( $app_info ) && ! empty( $app_info['has_developer_access'] ) ) {
			return true;
		}

		return false;
	}

	/**
	 * User's my account link.
	 *
	 * @return string
	 */
	public function user_account_link() {
		$app_info = \BuddyBossApp\ManageApp::instance()->get_app_info();

		if ( ! is_wp_error( $app_info ) && ! empty( $app_info ) && ! empty( $app_info['user_account_link'] ) ) {
			return esc_url( $app_info['user_account_link'] );
		}

		return '';
	}

	/**
	 * Get Custom Development Branches.
	 *
	 * @param bool $purge_cache Purge cache.
	 *
	 * @return array
	 */
	public function get_custom_development_branch( $purge_cache = false ) {
		$app_info = \BuddyBossApp\ManageApp::instance()->get_app_info( $purge_cache );
		$branches = array();

		if ( ! is_wp_error( $app_info ) && ! empty( $app_info ) && isset( $app_info['dev_access_data'] ) && ! empty( $app_info['has_developer_access'] ) ) {
			$access_data = $app_info['dev_access_data'];

			if ( isset( $access_data->branch ) ) {
				$branches[] = __( ' -- Select Branch -- ', 'buddyboss-app' );

				if ( ! empty( $access_data->branch ) ) {
					foreach ( (array) $access_data->branch as $branch_name ) {
						$branches[ $branch_name ] = $branch_name;
					}
				}
			}
		}

		return $branches;
	}

	/**
	 * Update branch data on the appcenter.
	 *
	 * @param array $args Array of arguments.
	 */
	public function update_custom_development_branch( $args = array() ) {
		$api_url = \BuddyBossApp\ClientCommon::instance()->get_center_api_url( 'v1', 'update-app-info' );

		bbapp_remote_post(
			$api_url,
			array(
				'method'  => 'POST',
				'timeout' => 15,
				'body'    => $args,
			)
		);
	}

	/**
	 * Ajax call to refresh the custom development data.
	 */
	public function bbapp_configure_branch_refresh() {
		// Fetch all app information again.
		\BuddyBossApp\ManageApp::instance()->get_app_info( true );
		wp_send_json_success();
	}

	/**
	 * Ajax call to disconnect apple account
	 *
	 * @since 1.4.3
	 */
	public function bbapp_disconnect_apple_account() {
		if ( ! Permissions::instance()->can_manage_app() ) {
			return false;
		}

		// Remove iOS connection keys from plugin.
		\BuddyBossApp\Admin\ManageApp::instance()->disconnect_account( 'ios', 'User disconnected iOS account manually.' );

		return true;
	}

	/**
	 * Ajax call to disconnect android account
	 *
	 * @since 1.4.3
	 */
	public function bbapp_disconnect_android_account() {
		if ( ! Permissions::instance()->can_manage_app() ) {
			return false;
		}

		// Remove Android account keys from plugin.
		\BuddyBossApp\Admin\ManageApp::instance()->disconnect_account( 'android', 'User disconnected Android account manually.' );

		return true;
	}

	/**
	 * Ajax call to disconnect the custom development data.
	 *
	 * @since 1.4.4
	 */
	public function bbapp_configure_custom_repo_disconnect() {
		$this->disconnect_custom_development_branch( true );

		wp_send_json_success();
	}

	/**
	 * Get Custom Development Branches.
	 *
	 * @param bool $purge_cache Purge cache.
	 *
	 * @return bool|\WP_Error
	 * @since 1.4.4
	 */
	public function disconnect_custom_development_branch( $purge_cache = false ) {
		$app_development_access = \BuddyBossApp\ManageApp::instance()->disconnect_app_development_access();

		if ( ! is_wp_error( $app_development_access ) && ! empty( $app_development_access ) ) {
			if ( true === $purge_cache ) {
				// Transients  should be clear.
				$transient_key = bbapp_get_app_info_transient_key();

				delete_transient( $transient_key );
				delete_site_transient( $transient_key );
			}

			return $app_development_access;
		}

		return false;
	}

	/**
	 * Get Core Development Branches.
	 *
	 * @param bool   $purge_cache Purge cache.
	 * @param string $field_name  Field name.
	 *
	 * @return array
	 */
	public function get_core_development_branch( $purge_cache = false, $field_name = '' ) {

		$app_info = \BuddyBossApp\ManageApp::instance()->get_app_info( $purge_cache );
		$branches = array();

		if ( ! is_wp_error( $app_info ) && ! empty( $app_info ) && isset( $app_info['developer_access_allowed_branches'] ) && ! empty( $app_info['has_developer_access'] ) ) {
			$access_data = $app_info['developer_access_allowed_branches'];

			if ( isset( $access_data ) ) {
				if ( ! empty( $access_data ) ) {
					$settings        = $this->get_settings();
					$selected_branch = ! empty( $settings[ $field_name ] ) ? $settings[ $field_name ] : '';

					/**
					 * Get hidden branch lists.
					 *
					 * @since 1.5.2.1
					 *
					 * @param array Default hidden branch lists.
					 */
					$hidden_branches = apply_filters( 'bbapp_hidden_core_app_branches', array( 'release-blockli' ) );

					foreach ( (array) $access_data as $branch_name => $label ) {
						if ( ! empty( $branch_name ) ) {
							if ( $selected_branch !== $branch_name && in_array( $branch_name, $hidden_branches, true ) ) {
								continue;
							}
							$branches[ $branch_name ] = $label;
						}
					}
				}
			}
		}

		return $branches;
	}

	/**
	 * Google firebase v1 notice.
	 *
	 * @since 1.7.6
	 */
	public function bbapp_google_firebase_v1_notice() {
		if ( ! empty( self::instance()->option( 'push.firebase_server_key' ) ) && empty( self::instance()->option( 'push.firebase_admin_key' ) ) ) {
			?>
			<div class="notice notice-error is-dismissible">
				<p>
					<?php printf( '%1$s <a href="%2$s">%3$s</a> %4$s', esc_html__( 'To improve your push notification experience, please ', 'buddyboss-app' ), esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=firebase' ) ), esc_html__( 'update your Cloud Messaging API key', 'buddyboss-app' ), esc_html__( 'to the latest version.', 'buddyboss-app' ) ); ?>
				</p>
			</div>
			<?php
		}

		if ( self::instance()->option( 'app_maintenance_mode' ) ) {
			?>
			<div class="notice notice-warning is-dismissible">
				<p>
					<?php printf( '%1$s <strong><a href="%2$s">%3$s</a></strong> %4$s', esc_html__( 'Maintenance Mode is currently enabled for your BuddyBoss App and all push notifications will be disabled. When you are ready, go to', 'buddyboss-app' ), esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-configure' ) ), esc_html__( 'Configure', 'buddyboss-app' ), esc_html__( 'to turn off this option', 'buddyboss-app' ) ); ?>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Disable the push notifications when maintenance mode is enabled.
	 *
	 * @param bool  $flag         Send push notification.
	 * @param array $notification Notification data.
	 *
	 * @since 1.8.50
	 *
	 * @return bool
	 */
	public function maintenance_mode_disable_push_notifications( $flag, $notification ) {
		if ( ManageApp::instance()->get_app_setting( 'app_maintenance_mode' ) && ( ! empty( $notification['push_notification_id'] ) || ! empty( $notification['data']['notification_id'] ) ) ) {
			$flag            = false;
			$notification_id = ! empty( $notification['push_notification_id'] ) ? $notification['push_notification_id'] : $notification['data']['notification_id'];

			Push::instance()->change_notification_status( $notification_id, 'failed' );
		}

		return $flag;
	}

	/**
	 * Get Core Access Branches.
	 *
	 * @param bool $purge_cache Purge cache.
	 *
	 * @return array|string
	 */
	public function get_core_access_branches( $purge_cache = false ) {

		$app_info = \BuddyBossApp\ManageApp::instance()->get_app_info( $purge_cache );
		$branches = '';
		if ( ! is_wp_error( $app_info ) && ! empty( $app_info ) && isset( $app_info['developer_access_allowed_branches'] ) && ! empty( $app_info['has_developer_access'] ) ) {
			$access_data = $app_info['developer_access_allowed_branches'];

			if ( isset( $access_data ) ) {
				if ( ! empty( $access_data ) ) {
					foreach ( (array) $access_data as $branch_name => $label ) {
						if ( ! empty( $branch_name ) && 'master' === $branch_name ) {
							$branches = $branch_name;
						}
					}
				}
			}
		}

		return $branches;
	}

	/**
	 * Validate iOS Firebase Plist file callback.
	 *
	 * @param array  $settings     Settings.
	 * @param array  $upload       Upload file data.
	 * @param object $field_object Field object.
	 *
	 * @since 1.8.30
	 *
	 * @return false|\WP_Error
	 */
	public function callback_validate_ios_firebase_plist( $settings, $upload, $field_object ) {
		$ios_firebase_configuration = false;
		if ( ! empty( $settings['validate_name'] ) && in_array( $settings['validate_name'], array( 'ios_firebase_plist', 'ios_dev_firebase_plist' ), true ) ) {
			$validate_name              = ! empty( $settings['validate_name'] ) ? $settings['validate_name'] : '';
			$file_path                  = ! empty( $upload['tmp_name'] ) ? $upload['tmp_name'] : '';
			$ios_firebase_configuration = $this->validate_ios_firebase_plist( $validate_name, $file_path );
		}

		return $ios_firebase_configuration;
	}

	/**
	 * Validate iOS Firebase Plist file.
	 *
	 * @param string $validate_name Validate name.
	 * @param string $file_path     File path.
	 *
	 * @since 1.8.30
	 *
	 * @return false|\WP_Error
	 */
	public function validate_ios_firebase_plist( $validate_name, $file_path ) {
		$ios_firebase_configuration = false;
		if ( ! empty( $validate_name ) && ! empty( $file_path ) && ( class_exists( 'BuddyBossAppScoping\PlistParser' ) || class_exists( 'PlistParser' ) ) ) {
			$plist_parser = \BuddyBossApp\Library\Composer::instance()->plist_parser_instance()->plist_parser();
			$result       = $plist_parser->searchKeyInPlist( $file_path, 'BUNDLE_ID' );
			$bundle_id    = 'ios_firebase_plist' === $validate_name ? self::option( 'publish.ios.namespace' ) : self::option( 'publish.ios.dev.namespace' );
			if ( trim( $bundle_id ) !== trim( $result ) ) {
				$ios_firebase_configuration = new \WP_Error( 'ios_firebase_plist_validate_failed', __( 'Your Firebase Registered App does not match the Apple Bundle ID', 'buddyboss-app' ) );
			} else {
				update_option( "bbapp_{$validate_name}_bundle_id", $bundle_id );
			}
		}

		return $ios_firebase_configuration;
	}

	/**
	 * Validate android Firebase json file callback.
	 *
	 * @param array  $settings     Settings.
	 * @param array  $upload       Upload file data.
	 * @param object $field_object Field object.
	 *
	 * @since 1.8.30
	 *
	 * @return false|\WP_Error
	 */
	public function callback_validate_android_firebase_json( $settings, $upload, $field_object ) {
		$android_firebase_configuration = false;
		if ( ! empty( $settings['validate_name'] ) && in_array( $settings['validate_name'], array( 'android_firebase_json', 'android_dev_firebase_json' ), true ) ) {
			$validate_name                  = ! empty( $settings['validate_name'] ) ? $settings['validate_name'] : '';
			$file_path                      = ! empty( $upload['tmp_name'] ) ? $upload['tmp_name'] : '';
			$android_firebase_configuration = $this->validate_android_firebase_json( $validate_name, $file_path );
		}

		return $android_firebase_configuration;
	}

	/**
	 * Validate android Firebase json file.
	 *
	 * @param string $validate_name Validate name.
	 * @param string $file_path     File path.
	 *
	 * @since 1.8.30
	 *
	 * @return false|\WP_Error
	 */
	public function validate_android_firebase_json( $validate_name, $file_path ) {
		$android_firebase_configuration = false;
		if ( ! empty( $validate_name ) && ! empty( $file_path ) ) {
			if ( ! class_exists( '\WP_Filesystem_Direct' ) ) {
				require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';
				require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php';
			}

			$wp_files_system = new \WP_Filesystem_Direct( new \stdClass() );
			$json_data       = $wp_files_system->get_contents( $file_path );
			$data            = json_decode( $json_data, true );
			$bundle_id       = 'android_firebase_json' === $validate_name ? self::option( 'publish.android.namespace' ) : self::option( 'publish.android.dev.namespace' );
			$client_info     = ( ! empty( $data['client'] ) ? array_column( $data['client'], 'client_info' ) : array() );
			$client_info     = ( ! empty( $client_info ) ? array_column( $client_info, 'android_client_info' ) : array() );
			$client_info     = ( ! empty( $client_info ) ? array_column( $client_info, 'package_name' ) : array() );

			if ( empty( $data ) || empty( $client_info ) || ! in_array( $bundle_id, $client_info, true ) ) {
				$android_firebase_configuration = new \WP_Error( 'android_firebase_json_validate_failed', __( 'Your Firebase Registered App does not match the Android Package Name', 'buddyboss-app' ) );
			} else {
				update_option( "bbapp_{$validate_name}_bundle_id", $bundle_id );
			}
		}

		return $android_firebase_configuration;
	}



}
