<?php
/**
 * Holds components related functionality.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\Permissions;

/**
 * Components class.
 */
class Components {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return Settings
	 */
	public static function instance() {

		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Components constructor.
	 */
	public function __construct() {
		// Using Singleton, see instance().
	}

	/**
	 * Filters/hookss here.
	 */
	public function load() {
		add_action( 'init', array( $this, 'load_init' ) );
	}

	/**
	 * Init method.
	 *
	 * @return false|void
	 */
	public function load_init() {
		/**
		 * Only load on bbapp-settings page.
		 */
		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'admin.php' !== SetupAdmin::instance()->get_page_now() || 'bbapp-components' !== $page_get ) {
			return false;
		}

		/**
		 * Deactivate Component.
		 */
		$do_action_get = ( ! empty( $_GET['do_action'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['do_action'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'deactivate' === $do_action_get && check_admin_referer( 'bbapp-admin-component-activation' ) ) {
			// Permission check.
			if ( ! \BuddyBossApp\Permissions::instance()->can_manage_app() ) {
				wp_die( 'You don\'t have permission to access this page.', 'buddyboss-app' );
			}

			$component_id = ( ! empty( $_GET['bbapp_component'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['bbapp_component'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$update       = \BuddyBossApp\Components::instance()->deactivate_components( array( $component_id ) );
			$all_get      = ( isset( $_GET['all'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['all'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( $update ) {
				$link = bbapp_get_super_admin_url(
					add_query_arg(
						array(
							'page'               => 'bbapp-components',
							'action'             => $all_get,
							'component_disabled' => true,
						),
						'admin.php'
					)
				);

				wp_safe_redirect( $link );
				exit;
			}
		}

		/**
		 * Activate Component
		 */
		$do_action_get = ( ! empty( $_GET['do_action'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['do_action'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'activate' === $do_action_get && check_admin_referer( 'bbapp-admin-component-activation' ) ) {
			$component_id = ( ! empty( $_GET['bbapp_component'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['bbapp_component'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$update       = \BuddyBossApp\Components::instance()->activate_components( array( $component_id ) );
			$all_get      = ( isset( $_GET['all'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['all'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( $update ) {
				$link = bbapp_get_super_admin_url(
					add_query_arg(
						array(
							'page'              => 'bbapp-components',
							'action'            => $all_get,
							'component_enabled' => true,
						),
						'admin.php'
					)
				);

				wp_safe_redirect( $link );
				exit;
			}
		}

		/**
		 * Bulk Activate Deactivate Component.
		 */
		$nonce_post = ( ! empty( $_POST['bbapp-admin-component-bulk'] ) ) ? wp_unslash( $_POST['bbapp-admin-component-bulk'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( wp_verify_nonce( $nonce_post, 'bbapp-admin-component-bulk' ) ) {
			$all_components    = \BuddyBossApp\Components::instance()->get_components();
			$action_components = array();
			$post_components   = isset( $_POST['bbapp_components'] ) && is_array( $_POST['bbapp_components'] ) ? map_deep( $_POST['bbapp_components'], 'sanitize_text_field' ) : array(); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
			foreach ( $post_components as $component_id => $v ) {
				if ( isset( $all_components[ $component_id ] ) && ! $all_components[ $component_id ]['required'] ) {
					$action_components[] = $component_id;
				}
			}

			$action  = isset( $_POST['action1'] ) ? bbapp_input_clean( wp_unslash( $_POST['action1'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$action2 = ( ! empty( $_POST['action2'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['action2'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( ! empty( $action2 ) ) {
				$action = $action2;
			}

			$update = false;

			if ( 'active' === $action ) {
				$update = \BuddyBossApp\Components::instance()->activate_components( $action_components );
			}

			if ( 'inactive' === $action ) {
				$update = \BuddyBossApp\Components::instance()->deactivate_components( $action_components );
			}

			$all_get = ( isset( $_GET['all'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['all'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( $update ) {
				$link = bbapp_get_super_admin_url(
					add_query_arg(
						array(
							'page'           => 'bbapp-components',
							'action'         => $all_get,
							'component_bulk' => true,
						),
						'admin.php'
					)
				);

				wp_safe_redirect( $link );
				exit;
			}
		}
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {
		$all_components      = \BuddyBossApp\Components::instance()->get_components();
		$active_components   = \BuddyBossApp\Components::instance()->get_active_components();
		$inactive_components = \BuddyBossApp\Components::instance()->get_inactive_components();
		$required_components = \BuddyBossApp\Components::instance()->get_required_components();
		$tabs                = array(
			'all'      => sprintf(
				'%s %s',
				__( 'All', 'buddyboss-app' ),
				'<span class="count">(' . count(
					array_filter(
						$all_components,
						array(
							$this,
							'remove_hide_components',
						)
					)
				) . ')</span>'
			),
			'active'   => sprintf(
				'%s %s',
				__( 'Active', 'buddyboss-app' ),
				'<span class="count">(' . count(
					array_filter(
						$active_components,
						array(
							$this,
							'remove_hide_components_by_id',
						),
						ARRAY_FILTER_USE_KEY
					)
				) . ')</span>'
			),
			'inactive' => sprintf(
				'%s %s',
				__( 'Inactive', 'buddyboss-app' ),
				'<span class="count">(' . count(
					array_filter(
						$inactive_components,
						array(
							$this,
							'remove_hide_components_by_id',
						),
						ARRAY_FILTER_USE_KEY
					)
				) . ')</span>'
			),
			'required' => sprintf(
				'%s %s',
				__( 'Required', 'buddyboss-app' ),
				'<span class="count">(' . count(
					array_filter(
						$required_components,
						array(
							$this,
							'remove_hide_components_by_id',
						),
						ARRAY_FILTER_USE_KEY
					)
				) . ')</span>'
			),
		);

		/**
		 * Menus which will be available to Subsite on BuddyBossApp Multisite Mode.
		 */
		$none_super_admin_menus = array();

		if ( ! bbapp_is_super_admin_page() ) {
			foreach ( $tabs as $k => $v ) {
				if ( ! in_array( $k, $none_super_admin_menus, true ) ) {
					unset( $tabs[ $k ] );
				}
			}
		}

		return $tabs;
	}

	/**
	 * Skip counting for hide components.
	 *
	 * @since 1.6.1
	 *
	 * @param array $component BB App components.
	 *
	 * @return bool
	 */
	public function remove_hide_components( $component ) {
		return ! empty( $component['hide_from_list'] ) && true === $component['hide_from_list'] ? false : true;
	}

	/**
	 * Skip counting for hide components by component id.
	 *
	 * @since 1.6.1
	 *
	 * @param string $component_id BB App component id.
	 *
	 * @return bool
	 */
	public function remove_hide_components_by_id( $component_id ) {
		$all_components = \BuddyBossApp\Components::instance()->get_components();
		$component      = ! empty( $all_components[ $component_id ] ) && is_array( $all_components[ $component_id ] ) ? $all_components[ $component_id ] : array();
		$visible        = $this->remove_hide_components( $component );

		return $visible;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {
		$default = '';

		// Capture first tab which is default.
		foreach ( $this->sub_settings() as $k => $v ) {
			$default = $k;
			break;
		}

		$setting      = ( isset( $_GET['action'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['action'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Return Components.
	 *
	 * @return mixed|void
	 */
	public function get_components() {
		$all_components      = \BuddyBossApp\Components::instance()->get_components();
		$active_components   = \BuddyBossApp\Components::instance()->get_active_components();
		$inactive_components = \BuddyBossApp\Components::instance()->get_inactive_components();
		$action_get          = ( ! empty( $_GET['action'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['action'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( in_array( $action_get, array( 'all', 'active', 'inactive', 'required' ), true ) ) {
			foreach ( $all_components as $component_id => $component ) {
				if ( 'active' === $action_get ) {
					if ( ! isset( $active_components[ $component_id ] ) || ! $active_components[ $component_id ] ) {
						if ( ! $component['required'] ) {
							unset( $all_components[ $component_id ] );
						}
					}
				}

				if ( 'inactive' === $action_get ) {
					if ( ! isset( $inactive_components[ $component_id ] ) || isset( $active_components[ $component_id ] ) || $component['required'] ) {
						unset( $all_components[ $component_id ] );
					}
				}

				if ( 'required' === $action_get ) {
					if ( ! $component['required'] ) {
						unset( $all_components[ $component_id ] );
					}
				}
			}
		}

		return $all_components;
	}

	/**
	 * Renders the setting screen.
	 */
	public function render_screen() {
		if ( ! Permissions::instance()->can_manage_app() ) {
			printf( '<p>%1$s</p>', esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) );

			return false;
		}

		\BuddyBossApp\Admin\SetupAdmin::instance()->render_subtabs( $this->sub_settings(), $this->get_current_sub_tab(), 'action' );

		include bbapp()->plugin_dir . 'views/settings/components.php';

	}

}
