<?php
/**
 * Holds build list functionality.
 *
 * @package BuddyBossApp\Admin\BuildUpload
 */

namespace BuddyBossApp\Admin\BuildUpload;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

if ( ! class_exists( '\WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Build list class.
 */
class UploadList extends \WP_List_Table {
	/**
	 * The type of view currently being displayed.
	 *
	 * E.g. "All", "Pending", "Approved", "Spam"...
	 *
	 * @since BuddyPress 1.7.0
	 * @var string
	 */
	public $view = 'all';

	/** Class constructor */
	public function __construct() {
		parent::__construct(
			array(
				'screen'   => 'upload-list',
				'singular' => __( 'Build Upload', 'buddyboss-app' ), // Singular name of the listed records.
				'plural'   => __( 'Build Uploads', 'buddyboss-app' ), // Plural name of the listed records.
				'ajax'     => false, // should this table support ajax?
			)
		);
	}

	/**
	 * Return the column available to this table list.
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns                 = array();
		$columns['id']           = __( 'Upload ID', 'buddyboss-app' );
		$columns['build_id']     = __( 'Build ID', 'buddyboss-app' );
		$columns['app_ver']      = __( 'Build Version', 'buddyboss-app' );
		$columns['core_app_ver'] = __( 'Core App Version', 'buddyboss-app' );
		$columns['requester']    = __( 'Requester', 'buddyboss-app' );
		$columns['status']       = __( 'Status', 'buddyboss-app' );
		$columns['actions']      = __( 'Actions', 'buddyboss-app' );

		return $columns;
	}

	/**
	 * Gets the name of the primary column.
	 *
	 * @return string The name of the primary column.
	 */
	public function get_primary_column_name() {
		return 'id';
	}

	/**
	 * No items.
	 */
	public function no_items() {
		esc_html_e( 'No uploads found.', 'buddyboss-app' );
	}

	/**
	 * Function to get column data.
	 *
	 * @param object $item        Current item.
	 * @param string $column_name Column name.
	 *
	 * @return string|void
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'id':
				$upload_id = isset( $item['id'] ) ? $item['id'] : 0;

				echo esc_html( "#{$upload_id}" );
				break;
			case 'build_id':
				$build_id = isset( $item['build_id'] ) ? $item['build_id'] : 0;

				echo esc_html( "#{$build_id}" );
				break;
			case 'app_ver':
				$build_app_version = isset( $item['app_version'] ) ? $item['app_version'] : '';

				if ( empty( $build_app_version ) ) {
					$build_app_version = __( 'N/A', 'buddyboss-app' );
				}

				echo esc_html( "{$build_app_version}" );
				break;
			case 'core_app_ver':
				$build_app_version = isset( $item['app_core_version'] ) ? $item['app_core_version'] : '';

				if ( empty( $build_app_version ) ) {
					$build_app_version = __( 'N/A', 'buddyboss-app' );
				}

				echo esc_html( "{$build_app_version}" );
				break;
			case 'requester':
				$user = get_user_by( 'email', $item['requester_email'] );

				if ( ! $user ) {
					$user = $item['requester_email'];

					if ( is_email( $user ) ) {
						$val = "{$user}";
					} else {
						$val = 'N/A';
					}
				} else {
					$user_link = admin_url( "user-edit.php?user_id={$user->ID}" );
					$val       = "<a href='{$user_link}'>{$user->display_name}</a>";
				}

				echo wp_kses_post( $val );
				break;
			case 'status':
				$labels        = Upload::instance()->get_statuses_labels();
				$status        = ( isset( $labels[ $item['status'] ] ) ) ? $labels[ $item['status'] ] : ucfirst( $item['status'] );
				$date_modified = $item['date_modified'];
				$date_format   = get_option( 'date_format' );
				$date_format   = ! empty( $date_format ) ? $date_format : 'F j, Y';
				$time_format   = get_option( 'time_format' );
				$time_format   = ! empty( $time_format ) ? $time_format : 'g:i a';
				$date_modified = get_gmt_from_date( gmdate( 'Y-m-d H:i:s', $date_modified ), $date_format . ' ' . $time_format );

				echo wp_kses_post( "{$status} <span class='bbapp-date-modified'>{$date_modified}</span>" );
				break;
			case 'actions':
				$actions = array();

				if ( isset( $item['status'] ) && ( 'running' === $item['status'] || 'new' === $item['status'] ) ) {
					ob_start();
					?>
					<a href="#" class="bbapp-action-btn" data-upload-id="<?php echo esc_attr( $item['id'] ); ?>" data-action="status_check">
						<div class="bbapp_loading" style="display: none"></div>
						<?php esc_html_e( 'Recheck Status', 'buddyboss-app' ); ?>
					</a>
					<?php
					$actions[] = ob_get_contents();
					ob_end_clean();
				}

				/**
				 * Failed & Cancelled Status Handle.
				 */
				if ( isset( $item['status'] ) && ( 'cancelled' === $item['status'] || 'failed' === $item['status'] ) ) {
					if ( ! isset( $item['error_message'] ) ) {
						$item['error_message'] = array();
					}

					$errors = array();

					if ( isset( $item['status'] ) && ( 'cancelled' === $item['status'] && ! empty( $item['cancel_reason'] ) ) ) {
						$errors[] = $item['cancel_reason'];
					}

					if ( isset( $item['status'] ) && ( 'failed' === $item['status'] && ! empty( $item['error_message'] ) ) ) {
						foreach ( $item['error_message'] as $err ) {
							$msg      = isset( $err['code'] ) ? sprintf( '%1$s #%2$s ', esc_html__( 'Error', 'buddyboss-app' ), esc_html( $err['code'] ) ) : '';
							$msg     .= isset( $err['message'] ) ? $err['message'] : '';
							$errors[] = $msg;
						}
					}

					if ( empty( $errors ) ) {
						$errors[] = __( 'Unknown Error', 'buddyboss-app' );
					}

					ob_start();
					?>
					<a href="#" class="bbapp-upload-cancel-reason-btn" onclick="jQuery(this).next().toggle();">
						<?php esc_html_e( 'View Errors﻿', 'buddyboss-app' ); ?>
					</a>
					<div class="bbapp-lite-dialog" style="display: none">
						<div class="overlay"></div>
						<div class="bbapp-dialog  bbapp-dialog--error">
							<h2><?php echo sprintf( '%1$s #%2$s', esc_html__( 'Build Upload', 'buddyboss-app' ), esc_html( $item['id'] ) ); ?></h2>
							<p><?php esc_html_e( 'We were unable to complete this build upload request for the following reason(s):', 'buddyboss-app' ); ?></p>
							<?php foreach ( $errors as $error ) : ?>
								<p class="bbapp-notice-box"><?php echo esc_html( $error ); ?></p>
							<?php endforeach; ?>
							<a href="#" class="close" onclick="jQuery(this).next().toggle();">Close</a>
						</div>
					</div>
					<?php
					$actions[] = ob_get_contents();
					ob_end_clean();
				}

				/**
				 * New Status Handle.
				 */
				if ( isset( $item['status'] ) && 'new' === $item['status'] ) {
					ob_start();
					?>
					<a href="#" class="bbapp-action-btn" data-upload-id="<?php echo esc_attr( $item['id'] ); ?>" data-action="cancel">
						<div class="bbapp_loading" style="display: none"></div>
						<?php esc_html_e( 'Cancel', 'buddyboss-app' ); ?>
					</a>
					<?php
					$actions[] = ob_get_contents();
					ob_end_clean();
				}

				if ( isset( $item['status'] ) && 'completed' === $item['status'] ) {
					ob_start();
					$apple_app_id = $item['apple_app_id'];
					$apple_link   = "https://appstoreconnect.apple.com/apps/{$apple_app_id}/testflight/ios";

					if ( ! empty( $apple_app_id ) ) :
						?>

						<a href="<?php echo esc_attr( $apple_link ); ?>" target="_blank">
							<?php esc_html_e( 'View in App Store Connect', 'buddyboss-app' ); ?>
						</a>

						<?php
					endif;

					$actions[] = ob_get_contents();
					ob_end_clean();
				}
				$allowed_html                 = wp_kses_allowed_html( 'post' );
				$allowed_html['a']['onclick'] = true;
				$allowed_html['span']         = array(
					'style' => array(),
				);
				echo wp_kses( implode( '<span style="color:gray">|</span>', $actions ), $allowed_html );
				break;
			default:
				echo 'N/A';
				break;
		}
	}

	/**
	 * Function to get sortable column.
	 *
	 * @return array
	 */
	protected function get_sortable_columns() {
		return array();
	}

	/**
	 * Handles data query and filter, sorting, and pagination.
	 */
	public function prepare_items() {
		/**
		 * Init column headers.
		 */
		$this->_column_headers = array( $this->get_columns(), array(), array() );
		$per_page              = $this->get_items_per_page( 'bbapp_build_uploads_per_page', 10 );
		$current_page          = $this->get_pagenum();
		$status                = '';
		$total_items           = 0;
		$results               = Upload::instance()->get_uploads( $status, $current_page, $per_page );

		if ( ! is_wp_error( $results ) ) {
			$total_items = (int) $results['headers']['total'];
			$this->items = $results['data'];
		}

		$this->set_pagination_args(
			array(
				'total_items' => $total_items,
				'per_page'    => $per_page,
			)
		);
	}

	/**
	 * Checkbox for bulk items.
	 *
	 * @param object $item Current item.
	 *
	 * @return string|void
	 */
	public function column_reorder( $item ) {
		return '<span class="iap-reorder-product"><i class="bb-icon-l bb-icon-bars"></i></span>';
	}

	/**
	 * Checkbox for bulk items.
	 *
	 * @param object $item Current item.
	 *
	 * @return string|void
	 */
	public function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" id="cb-select-%s" name="iap_product_ids[]" value="%s" />',
			$item['id'],
			$item['id']
		);
	}

	/**
	 * Script/styles here.
	 */
	public function display() {
		parent::display();
		?>
		<script>
			jQuery( document ).ready( function() {
				jQuery( '.bbapp-action-btn' ).click( function() {
					const _this = jQuery( this );

					if ( _this.data( 'loading' ) === '1' ) {
						return false;
					}

					_this.data( 'loading', '1' ).find( '.bbapp_loading' ).show();

					const post = jQuery.post( ajaxurl, {
						action: 'bbapp_build_upload_actions',
						sub_action: _this.data( 'action' ),
						upload_id: _this.data( 'upload-id' ),
						'bbapp_upload_build_actions': "<?php echo esc_attr( wp_create_nonce( 'bbapp_upload_build_actions' ) ); ?>"
					} );

					post.done( function( data ) {
						if ( typeof data.success !== 'undefined' ) {
							if ( data.success && data.data.reload ) {
								window.location.reload();
							} else {
								// eslint-disable-next-line no-alert
								alert( data.data );
							}
						}
					} );

					post.fail( function() {
						// eslint-disable-next-line no-alert
						alert('<?php esc_attr_e( 'There is error while calling action', 'buddyboss-app' ); ?>');
					} );

					post.always( function() {
						_this.data( 'loading', '0' ).find( '.bbapp_loading' ).hide();
					} );
				} );
			} );
		</script>
		<?php
	}
}
