<?php
/**
 * Hold build upload functionality.
 *
 * @package BuddyBossApp\Admin\BuildUpload
 */

namespace BuddyBossApp\Admin\BuildUpload;

use BuddyBossApp\Admin\BuildUpload;
use BuddyBossApp\AppStores\Apple;
use BuddyBossApp\ClientCommon;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Permissions;
use MingYuanYun\AppStore\Exceptions\ConfigException;
use WP_Error;

/**
 * Build upload class.
 */
class Upload {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * If it's current page to load screen.
	 *
	 * @var bool $is_current_page
	 */
	private $is_current_page = false;

	/**
	 * Build list.
	 *
	 * @var $list
	 */
	private $list;

	/**
	 * Build upload error message.
	 *
	 * @var bool $upload_error
	 */
	private $upload_error = false;

	/**
	 * Build upload success.
	 *
	 * @var bool $build_upload_success
	 */
	private $build_upload_success = false;

	/**
	 * Current build error message.
	 *
	 * @var bool $current_build_error
	 */
	private $current_build_error = false;

	/**
	 * If build is available.
	 *
	 * @var bool $build_not_available
	 */
	private $build_not_available = false;

	/**
	 * Get the instance of the class.
	 *
	 * @return Upload
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Build constructor.
	 */
	public function __construct() {
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		add_action(
			'bbapp_queue_task_check_upload_status',
			array(
				$this,
				'background_job_check_status',
			)
		); // Background Job.
		add_action( 'wp_ajax_bbapp_build_upload_actions', array( $this, 'upload_actions' ) );
		add_action( 'admin_init', array( $this, 'init' ) );
		add_action( 'admin_init', array( $this, 'process_submit' ) );
	}

	/**
	 * On this class init.
	 */
	public function init() {
		if ( $this->will_render() ) {
			$current_tab = BuildUpload::instance()->get_current_sub_tab();

			// When it's a all upload screen.
			if ( 'all-uploads' === $current_tab ) {
				$this->list = new UploadList();
				$this->list->prepare_items();
			}

			// New build upload.
			if ( in_array( $current_tab, array( 'request-ios', 'request-android' ), true ) ) {
				$platform = str_replace( 'request-', '', $current_tab );
				$this->request_upload_checklist( $platform );
			}
		}
	}

	/**
	 * Functions tells & sets that's if current page is one where it's will render.
	 *
	 * @param bool $set Set to load screen.
	 *
	 * @return bool
	 */
	public function will_render( $set = false ) {
		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Build Upload Ajax Actions.
	 */
	public function upload_actions() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			wp_send_json_error( __( "You don't have permission to perform this action.", 'buddyboss-app' ) );

			return false;
		}

		$nonce_post = ( ! empty( $_POST['bbapp_upload_build_actions'] ) ) ? sanitize_text_field( wp_unslash( $_POST['bbapp_upload_build_actions'] ) ) : '';

		if ( wp_verify_nonce( $nonce_post, 'bbapp_upload_build_actions' ) ) {
			$sub_action = ( ! empty( $_POST['sub_action'] ) ) ? sanitize_text_field( wp_unslash( $_POST['sub_action'] ) ) : '';
			$upload_id  = (int) filter_input( INPUT_POST, 'upload_id', FILTER_SANITIZE_NUMBER_INT );

			/**
			 * Status Check Action.
			 */
			if ( 'status_check' === $sub_action ) {
				$this->clear_cache();
				wp_send_json_success( array( 'reload' => true ) );
			}

			/**
			 * Cancel Upload Build.
			 */
			if ( 'cancel' === $sub_action ) {
				$cancel = $this->cancel_build_upload( $upload_id );
				if ( $cancel ) {
					wp_send_json_success( array( 'reload' => true ) );
				} else {
					wp_send_json_error( __( 'There was a problem cancelling your upload request.', 'buddyboss-app' ) );
				}
			}
		}
	}

	/**
	 * Return the each statuses labels
	 *
	 * @return array
	 */
	public function get_statuses_labels() {
		$labels = array(
			'new'       => __( 'New', 'buddyboss-app' ),
			'failed'    => __( 'Failed', 'buddyboss-app' ),
			'running'   => __( 'In Progress', 'buddyboss-app' ),
			'completed' => __( 'Uploaded', 'buddyboss-app' ),
			'cancelled' => __( 'Cancelled', 'buddyboss-app' ),
		);

		return $labels;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {
		$default      = 'request-ios';
		$setting      = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {
		$settings_array['request-ios']     = __( 'App Store Connect', 'buddyboss-app' );
		$settings_array['request-android'] = __( 'Google Play Console', 'buddyboss-app' );
		$tabs                              = $settings_array;

		return $tabs;
	}

	/**
	 * Background queue task for checking build status.
	 * 1. BuddyBossApp Server Pings WordPress to do build status check.
	 * 2. Plugin adds a background job to attempt it at background.
	 * 3. This function get triggered once job gets triggered.
	 */
	public function background_job_check_status() {
		$this->clear_cache(); // clear all api cache.
	}

	/**
	 * Cancel the build upload if the status is new.
	 *
	 * @param string $upload_id Upload id.
	 *
	 * @return bool
	 */
	public function cancel_build_upload( $upload_id ) {
		$app     = ManageApp::instance()->get_app();
		$api_url = ClientCommon::instance()->get_center_api_url( 'v2' );

		// Do the api call.
		$do_request = bbapp_remote_post(
			$api_url,
			array(
				'method'  => 'POST',
				'timeout' => 15,
				'body'    => array(
					'action'    => 'upload-cancel',
					'upload_id' => $upload_id,
					'bbapp_id'  => $app['bbapp_app_id'],
					'bbapp_key' => ManageApp::instance()->get_auth_app_key(),
				),
			)
		);

		if ( is_wp_error( $do_request ) || empty( $do_request ) ) {
			return false;
		}

		$response = isset( $do_request['body'] ) ? $do_request['body'] : '';
		$response = json_decode( $response, true );

		if ( ! isset( $response['success'] ) && $response['success'] ) {
			return false;
		}

		$this->clear_cache();

		return true;
	}

	/**
	 * Check list before upload build request.
	 * This returns the array of incomplete actions.
	 *
	 * @param string $platform Platform name.
	 *
	 * @return array|mixed
	 * @throws ConfigException IOs store exception message.
	 */
	public function request_upload_checklist( $platform ) {
		static $request_upload_checklist;

		if ( isset( $request_upload_checklist[ $platform ] ) ) {
			return $request_upload_checklist[ $platform ];
		}

		$build_id = ( ! empty( $_GET['build_id'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['build_id'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		/**
		 * ----------------------------------------------------------------
		 * iOS CheckLists.
		 */
		if ( 'ios' === $platform ) {
			/**
			 * 1. Check if connected to IOS store.
			 */
			if ( ! Apple::instance()->is_connected() ) {
				$request_upload_checklist[ $platform ] = array(
					'desc'       => __(
						'To upload a build to App Store Connect, please connect your Apple Developer account.',
						'buddyboss-app'
					),
					'button_txt' => __( 'Connect Now', 'buddyboss-app' ),
					'link'       => bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=ios#goto-app_settings_app_config_ios_account_settings' ),
					'new_window' => false,
				);

				return $request_upload_checklist[ $platform ];
			}

			/**
			 * 2. Check if user selected release bundle ID
			 */
			$release_bundle_id = Apple::instance()->get_release_bundle_id();

			if ( empty( $release_bundle_id ) ) {
				$link                                  = bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=ios#goto-app_settings_publish__ios__namespace' );
				$request_upload_checklist[ $platform ] = array(
					/* translators: %1$s: IOS settings link. */
					'desc'       => sprintf( __( 'To continue, please choose your iOS Release App Bundle ID on the iOS <a href="%1$s">Settings page</a>.', 'buddyboss-app' ), $link ),
					'button_txt' => __( 'Choose Bundle ID', 'buddyboss-app' ),
					'link'       => $link,
					'new_window' => false,
				);

				return $request_upload_checklist[ $platform ];
			}

			/**
			 * 3. Check store app ID.
			 */
			$apple_store_id = Apple::instance()->get_apple_release_bundle_store_app_id();

			if ( empty( $apple_store_id ) ) {
				$link                                  = bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=ios#goto-app_settings_publish__ios__store_app_id' );
				$link2                                 = bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=ios#goto-app_settings_publish__ios__namespace' );
				$request_upload_checklist[ $platform ] = array(
					/* translators: %1$s: IOS settings link. */
					'desc'       => sprintf( __( 'To continue, please create an app in <a href="https://itunesconnect.apple.com/login" target="_blank">App Store Connect</a> that uses the iOS Release App Bundle ID set on the <a href="%1$s">Settings page</a>.', 'buddyboss-app' ), $link2 ),
					'button_txt' => __( 'Create in App Store Connect', 'buddyboss-app' ),
					'link'       => $link,
					'new_window' => true,
				);

				return $request_upload_checklist[ $platform ];
			}

			/**
			 * 4. Check App Specific Password Account ID
			 */
			$ios_app_specific_password_account_id = Apple::instance()->get_ios_app_specific_password_account_id();

			if ( empty( $ios_app_specific_password_account_id ) ) {
				$link                                  = bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=ios#goto-app_settings_publish__ios__app_specific_password_account_id' );
				$request_upload_checklist[ $platform ] = array(
					/* translators: %1$s: IOs settings link. */
					'desc'       => sprintf( __( 'To upload builds to App Store Connect, you must select an Apple Account ID. Select it on the iOS <a href="%1$s">Settings page</a>.', 'buddyboss-app' ), $link ),
					'button_txt' => __( 'Select Apple Account ID', 'buddyboss-app' ),
					'link'       => $link,
					'new_window' => false,
				);

				return $request_upload_checklist[ $platform ];
			}

			/**
			 * 5. Check App Specific Password
			 */
			$ios_app_specific_password = Apple::instance()->get_ios_app_specific_password();

			if ( empty( $ios_app_specific_password ) ) {
				$link                                  = bbapp_get_super_admin_url( 'admin.php?page=bbapp-configure&setting=ios#goto-app_settings_publish__ios__app_specific_password' );
				$request_upload_checklist[ $platform ] = array(
					/* translators: %s: IOS settings link. */
					'desc'       => sprintf( __( 'To upload builds to App Store Connect, you must generate and enter an <b>App-Specific Password</b> for your Apple account. Once generated, enter it on the iOS <a href="%1$s">Settings page</a>.', 'buddyboss-app' ), $link ),
					'button_txt' => __( 'Enter Password', 'buddyboss-app' ),
					'link'       => $link,
					'new_window' => false,
				);

				return $request_upload_checklist[ $platform ];
			}

			/**
			 * 6. Check if builds are present.
			 */
			$builds = $this->get_builds( $platform );

			if ( empty( $builds ) && empty( $build_id ) ) {
				$this->build_not_available = true;
			}

			/**
			 * 7. Validate the provided build ID is already uploaded to AppStore using BuddyBoss.
			 */
			$current_build = false;

			if ( ! empty( $build_id ) ) {
				$builds = $this->get_builds( $platform, false );

				foreach ( $builds as $build ) {
					if ( $build['id'] === $build_id ) {
						$current_build = $build;

						if ( 'not' !== $build['is_uploaded'] ) {
							$request_upload_checklist[ $platform ] = array(
								/* translators: %s: Build version. */
								'desc'       => sprintf( __( 'You have already uploaded Build %s to App Store Connect.', 'buddyboss-app' ), "#{$build["id"]} {$build["build_version"]} ({$build["build_version_code"]})" ),
								'button_txt' => __( 'View in App Store Connect', 'buddyboss-app' ),
								'link'       => 'https://itunesconnect.apple.com/',
								'new_window' => true,
							);

							return $request_upload_checklist[ $platform ];
						}
					}
				}

				/**
				 * 7. Validate the provided build ID is already uploaded to AppStore without using BuddyBoss.
				 */
				if ( $current_build ) {
					$is_version_uploaded_on_ios_store = $this->is_build_version_uploaded_to_ios_store( $current_build['build_version'] );

					if ( $is_version_uploaded_on_ios_store ) {
						/* translators: %s: Build version  */
						$this->current_build_error = sprintf( __( 'You already have a build using the version %s in App Store Connect. Please select a different build.', 'buddyboss-app' ), "{$current_build['build_version']} ({$current_build['build_version_code']})" );
					}
				}

				/**
				 * 8. Build can be upload.
				 *
				 * Here we loop thru a build list til the last uploaded build.
				 * If we found current build before the last uploaded when then it's upload-able else it's not.
				 */
				if ( $current_build ) {
					$builds      = $this->get_builds( $platform, false );
					$found_build = false;

					foreach ( $builds as $build ) {
						if ( $build['id'] === $current_build['id'] ) {
							$found_build = true;
						}

						if ( 'not' !== $build['is_uploaded'] ) {
							break;
						}
					}

					if ( ! $found_build ) {
						/* translators: %s: build version. */
						$this->current_build_error = sprintf( __( 'You already have a build using the version %s or higher in App Store Connect. Please select a different build.', 'buddyboss-app' ), "{$current_build['build_version']} ({$current_build['build_version_code']})" );
					}
				}
			}
		}

		/**
		 * ----------------------------------------------------------------
		 * Android CheckLists.
		 */
		if ( 'android' === $platform ) {
			/**
			 * 1.  Check if builds are presents.
			 */
			$builds = $this->get_builds( $platform );

			if ( empty( $builds ) ) {
				$link                                  = bbapp_get_super_admin_url( 'admin.php?page=bbapp-build&setting=request-build' );
				$request_upload_checklist[ $platform ] = array(
					'desc'       => __(
						'Please generate an Android Release Apps to upload to the Google Play Store. Test App builds cannot be uploaded.',
						'buddyboss-app'
					),
					'button_txt' => __( 'Request Build', 'buddyboss-app' ),
					'link'       => $link,
					'new_window' => false,
				);

				return $request_upload_checklist[ $platform ];
			}
		}

		$request_upload_checklist[ $platform ] = true;

		return $request_upload_checklist[ $platform ];
	}

	/**
	 * Check if version is already uploaded to ios store or not.
	 *
	 * @param string $version      Build version.
	 *
	 * @return bool
	 * @throws ConfigException IOS store exception message.
	 */
	public function is_build_version_uploaded_to_ios_store( $version ) {
		static $cached;

		$cache_key = maybe_serialize( func_get_args() );

		if ( isset( $cached[ $cache_key ] ) ) {
			return $cached[ $cache_key ];
		}

		$app_id = Apple::instance()->get_apple_release_bundle_store_app_id();

		if ( ! $app_id ) {
			$cached[ $cache_key ] = false;

			return $cached[ $cache_key ];
		}

		$get_versions = Apple::instance()->get_app_store_versions_for_an_app( $app_id );
		$versions     = array();

		if ( isset( $get_versions['data'] ) && is_array( $get_versions['data'] ) ) {

			foreach ( $get_versions['data'] as $ver ) {
				$versions[] = $ver['attributes']['versionString'];
			}
		}

		if ( in_array( $version, $versions, true ) ) {
			$cached[ $cache_key ] = true;

			return $cached[ $cache_key ];
		}

		$cached[ $cache_key ] = false;

		return $cached[ $cache_key ];
	}

	/**
	 * Return the list of builds.
	 *
	 * @param string $platform                   Platform name.
	 * @param bool   $filter                     Build filter.
	 * @param bool   $exclude_uploaded_on_stores Exclude uploaded builds.
	 *
	 * @return array|bool
	 * @throws ConfigException IOS store exception message.
	 */
	public function get_builds( $platform, $filter = true, $exclude_uploaded_on_stores = false ) {
		static $cached;

		$cache_key = maybe_serialize( func_get_args() );

		if ( isset( $cached[ $cache_key ] ) ) {
			return $cached[ $cache_key ];
		}

		$builds = \BuddyBossApp\Build::instance()->get_app_builds( $platform, 'live', false, 'completed', 1, 99, true );

		if ( is_wp_error( $builds ) || ( is_array( $builds ) && empty( $builds['data'] ) ) ) {
			$cached[ $cache_key ] = false;

			return $cached[ $cache_key ];
		}

		$builds = $builds['data'];

		if ( $filter ) {
			/**
			 * We will keep version only till last uploaded one.
			 * Because we cannot upload a earlier version from the one is already uploaded.
			 */
			$versions_to_keep = array();

			foreach ( $builds as $build ) {
				if ( 'not' === $build['is_uploaded'] ) {
					$include = true;
					if ( $exclude_uploaded_on_stores ) {
						$include = ! $this->is_build_version_uploaded_to_ios_store( $build['build_version'] );
					}
					if ( $include ) {
						$versions_to_keep[] = $build;
					}
				} else {
					break;
				}
			}
		} else {
			$versions_to_keep = $builds;
		}

		$cached[ $cache_key ] = $versions_to_keep;

		return $cached[ $cache_key ];
	}

	/**
	 * Render Uploads
	 *
	 * @return false|void
	 */
	public function render_uploads() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			echo '<p>' . esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) . '</p>';

			return false;
		}

		include bbapp()->plugin_dir . 'views/settings/build-upload/uploads.php';
	}

	/**
	 * Render Request Upload
	 *
	 * @return bool
	 */
	public function render_request_upload() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			echo '<p>' . esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) . '</p>';

			return false;
		}

		if ( 'request-ios' === $this->get_current_sub_tab() ) {
			if ( true === ManageApp::instance()->is_app_disabled() ) {
				include bbapp()->plugin_dir . 'views/settings/build-upload/request-ios-app-is-disabled.php';
			} else {
				include bbapp()->plugin_dir . 'views/settings/build-upload/request-ios.php';
			}
		}

		if ( 'request-android' === $this->get_current_sub_tab() ) {
			if ( true === ManageApp::instance()->is_app_disabled() ) {
				include bbapp()->plugin_dir . 'views/settings/build-upload/request-android-app-is-disabled.php';
			} else {
				include bbapp()->plugin_dir . 'views/settings/build-upload/request-android.php';
			}
		}
	}

	/**
	 * Get the ios app build uploads from API.
	 *
	 * @param string  $status         Filter by build status.
	 * @param int     $page           Number of page.
	 * @param int     $per_page       Number of item per page.
	 * @param boolean $override_cache Overwrite cache.
	 *
	 * @return array | WP_Error
	 */
	public function get_uploads( $status = '', $page = 1, $per_page = 12, $override_cache = false ) {
		if ( ! ManageApp::instance()->is_app_center_connected() ) {
			return new WP_Error( 'app_not_connected', __( 'The app is not connected to the app center.', 'buddyboss-app' ) );
		}

		if ( empty( $per_page ) ) {
			$per_page = 12;
		}

		$body = array(
			'bbapp_id'  => ManageApp::instance()->get_app_id(),
			'bbapp_key' => ManageApp::instance()->get_auth_app_key(),
		);

		if ( ! empty( $page ) ) {
			$body['page'] = $page;
		}

		if ( ! empty( $per_page ) ) {
			$body['per_page'] = $per_page;
		}

		if ( ! empty( $status ) ) {
			$body['status'] = $status;
		}

		$cache_flush_key = get_option( 'bbapp_all_build_uploads_cache_flush_key' ); // This will help forcefully kill all group cache.
		$cache_key       = 'bbapp_all_build_uploads_' . md5( maybe_serialize( $body ) . $cache_flush_key );
		$cached          = get_site_transient( $cache_key );

		if ( false !== $cached && ! $override_cache ) {
			$retval = $cached;
		} else {
			$api_url = ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-uploads' );

			$request = bbapp_remote_get(
				$api_url,
				array(
					'timeout' => 15,
					'body'    => $body,
				)
			);

			$response         = wp_remote_retrieve_body( $request );
			$response         = json_decode( $response, true );
			$response_code    = wp_remote_retrieve_response_code( $request );
			$response_message = wp_remote_retrieve_response_message( $request );

			if ( 'OK' === $response_message && 200 === $response_code ) {
				$data['data']                  = $response;
				$data['headers']['total']      = wp_remote_retrieve_header( $request, 'Total' );
				$data['headers']['totalpages'] = wp_remote_retrieve_header( $request, 'TotalPages' );
				$retval                        = $data;

				set_site_transient( $cache_key, $retval, 10 * 60 ); // cache it for 10 min.
			} elseif ( 400 !== $response_code ) {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response_message ) ),
					$response_message,
					array( 'status' => $response_code )
				);
			} else {
				$retval = new WP_Error(
					str_replace( ' ', '_', strtolower( $response['code'] ) ),
					$response['message'],
					array( 'status' => $response_code )
				);
			}
		}

		return $retval;
	}

	/**
	 * Helper function to clear all build cache.
	 */
	public function clear_cache() {
		update_option( 'bbapp_all_build_uploads_cache_flush_key', uniqid() );
	}

	/**
	 * Submit build request.
	 *
	 * @return false|void
	 * @throws ConfigException IOS store exception message.
	 */
	public function process_submit() {
		if ( ! $this->will_render() ) {
			return false;
		}

		$nonce_post = ( ! empty( $_POST['bbapp_build_upload_request'] ) ) ? wp_unslash( $_POST['bbapp_build_upload_request'] ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( wp_verify_nonce( $nonce_post, 'bbapp_build_upload_request' ) ) {
			// Permission Check.
			if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
				wp_send_json_error( __( 'You don\'t have permission to access this page.', 'buddyboss-app' ) );

				return false;
			}

			$build_id         = ( ! empty( $_POST['build_id'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['build_id'] ) ) : ''; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$app              = ManageApp::instance()->get_app();
			$api_url          = ClientCommon::instance()->get_center_api_url( 'v2' );
			$user             = get_userdata( get_current_user_id() );
			$apple_account_id = Apple::instance()->get_ios_app_specific_password_account_id();
			$apple_app_id     = Apple::instance()->get_apple_release_bundle_store_app_id();

			if ( empty( $apple_account_id ) && ! is_email( $apple_account_id ) ) {
				$this->upload_error = __( 'There was a problem connecting with the Apple ID and App-Specific Shared Password provided.', 'buddyboss-app' );

				return false;
			}

			if ( empty( $apple_app_id ) ) {
				$this->upload_error = __( 'There was a problem connecting to your app inside of App Store Connect.', 'buddyboss-app' );

				return false;
			}

			$body = array(
				'action'            => 'upload-request',
				'build_id'          => $build_id,
				'bbapp_id'          => $app['bbapp_app_id'],
				'bbapp_key'         => ManageApp::instance()->get_auth_app_key(),
				'app_specific_pass' => Apple::instance()->get_ios_app_specific_password(),
				'requester_email'   => $user->user_email,
				'apple_username'    => $apple_account_id,
				'apple_app_id'      => $apple_app_id,
			);

			// Do the api call.
			$do_request = bbapp_remote_post(
				$api_url,
				array(
					'method'  => 'POST',
					'timeout' => 15,
					'body'    => $body,
				)
			);

			if ( is_wp_error( $do_request ) ) {
				return false;
			}

			$body     = wp_remote_retrieve_body( $do_request );
			$response = json_decode( $body, true );

			if ( isset( $response['code'] ) ) {
				$this->upload_error = $response['message'];

				return false;
			}

			if ( isset( $response['success'] ) && $response['success'] ) {
				// Reset cache.
				$this->clear_cache();
				$this->build_upload_success = $response['upload_id'];
			} else {
				$this->upload_error = __( 'There is an unknown error while sending build upload request to BuddyBoss server.', 'buddyboss-app' );

				return false;
			}
		}
	}

	/**
	 * Get build link.
	 *
	 * @param string $extra_param Extra parameter.
	 *
	 * @return string
	 */
	public function get_link( $extra_param ) {
		return bbapp_get_super_admin_url( "admin.php?page=bbapp-build-upload&setting=all-uploads&{$extra_param}" );
	}
}

