<?php
/**
 * Holds build upload functionality for admin.
 *
 * @package BuddyBossApp\Admin
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\Admin\BuildUpload\Upload;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Permissions;

/**
 * Build upload class.
 */
class BuildUpload {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return BuildUpload
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * BuildUpload constructor.
	 */
	public function __construct() {
	}

	/**
	 * Filters/hooks here.
	 */
	public function load() {
		add_action( 'init', array( $this, 'load_init' ) );
	}

	/**
	 * Init method.
	 */
	public function load_init() {
		Upload::instance();

		$page_get = ( ! empty( $_GET['page'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['page'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'admin.php' === SetupAdmin::instance()->get_page_now() && 'bbapp-build-upload' === $page_get ) {
			/**
			 * When All Uploads Page Will Render.
			 */
			if ( 'all-uploads' === $this->get_current_sub_tab() ) {
				Upload::instance()->will_render( true );
			}

			/**
			 * When New Upload Page Will Render.
			 */
			if ( 'request-ios' === $this->get_current_sub_tab() || 'request-android' === $this->get_current_sub_tab() ) {
				Upload::instance()->will_render( true );
			}
		}
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {
		$tabs = array(
			'all-uploads'     => __( 'All Uploads', 'buddyboss-app' ),
			'request-ios'     => __( 'Request Build', 'buddyboss-app' ),
			'request-android' => __( 'Request Build', 'buddyboss-app' ),
		);

		return $tabs;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {
		$default      = 'all-uploads';
		$setting      = ( isset( $_GET['setting'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['setting'] ) ) : $default; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Renders the setting screen.
	 */
	public function render_screen() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			echo '<p>' . esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) . '</p>';

			return false;
		}

		// Builder Server under Maintenance Mode.
		$bbapp = ManageApp::instance()->get_app();

		if ( ! empty( $bbapp ) && ! empty( $bbapp['verified'] ) && ! empty( $bbapp['bbapp_site_type'] ) && 'primary' === $bbapp['bbapp_site_type'] ) {
			$maintenance_mode = ManageApp::instance()->is_appcenter_under_maintenance();

			if ( is_wp_error( $maintenance_mode ) ) {
				include bbapp()->plugin_dir . '/views/settings/build/no-connect.php';
			} elseif ( ! is_wp_error( $maintenance_mode ) && true === $maintenance_mode ) {
				include bbapp()->plugin_dir . '/views/settings/build/maintenance-mode.php';
			} else {
				if ( $this->get_current_sub_tab() === 'all-uploads' ) {
					Upload::instance()->render_uploads();
				} elseif ( 'request-ios' === $this->get_current_sub_tab() || 'request-android' === $this->get_current_sub_tab() ) {
					Upload::instance()->render_request_upload();
				}
			}
		} elseif ( ! empty( $bbapp ) && ( ! empty( $bbapp['verified'] ) && ! empty( $bbapp['bbapp_site_type'] ) && 'secondary' === $bbapp['bbapp_site_type'] ) ) {
			include bbapp()->plugin_dir . '/views/settings/build/build-connect-dev.php';
		} else {
			include bbapp()->plugin_dir . '/views/settings/build/build-no-connection.php';
		}
	}
}
