<?php
/**
 * Holds IOS helper class functionality for builds.
 *
 * @package BuddyBossApp\Admin\Build
 */

namespace BuddyBossApp\Admin\Build;

use BuddyBossApp\Admin\Configure;
use BuddyBossApp\AppStores\Apple;
use BuddyBossApp\ManageApp;

/**
 * IOS helper class for build.
 */
class iOSHelper { // phpcs:ignore PEAR.NamingConventions.ValidClassName.StartWithCapital, Generic.Classes.OpeningBraceSameLine.ContentAfterBrace

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return iOSHelper
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * IOSHelper constructor.
	 */
	public function __construct() {
	}

	/**
	 * Filters/hook here.
	 */
	public function load() {
	}

	/**
	 * Function to get the value base on specific key from array.
	 *
	 * @param array      $array   Array to get the value.
	 * @param string|int $key     Key to get the value.
	 * @param string     $default Default data.
	 *
	 * @return string
	 */
	public function get_array_value( $array, $key, $default = '' ) {
		if ( isset( $array[ $key ] ) ) {
			return $array[ $key ];
		} else {
			return $default;
		}
	}

	/**
	 * Returns the review information related to iOS
	 *
	 * @param string $env Platform.
	 *
	 * @return array
	 */
	public function get_review_information( $env ) {
		$review_info      = array();
		$settings         = \BuddyBossApp\ManageApp::instance()->get_app_settings( true );
		$ios_edit_link    = bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=ios' );
		$app_version      = \BuddyBossApp\Build::instance()->get_app_version( 'ios' );
		$app_version_code = \BuddyBossApp\Build::instance()->get_app_version_code( 'ios' );

		$review_info['app_version_ios']      = array(
			'label'     => __( 'Build Version', 'buddyboss-app' ),
			'value'     => $app_version,
			'validated' => ( ! empty( $app_version ) ),
			'edit_link' => bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=general' ),
		);
		$review_info['app_version_code_ios'] = array(
			'label'     => __( 'Build Version Code', 'buddyboss-app' ),
			'value'     => $app_version_code,
			'validated' => ( ! empty( $app_version_code ) ),
		);
		$value                               = $this->get_array_value( $settings, 'publish.ios.namespace', '' );
		$review_info['bundleid']             = array(
			'label'        => __( 'Bundle ID', 'buddyboss-app' ),
			'value'        => $value,
			'validated'    => ( ! empty( $value ) ),
			'edit_link'    => $ios_edit_link . '#app_settings_app_publish_ios_live_settings',
			'label_prefix' => __( 'Release App', 'buddyboss-app' ),
		);

		$upload_dir = wp_upload_dir();

		if ( 'live' === $env ) {
			$value                         = ( ! empty( $this->get_array_value( $settings, 'publish.ios.firebase_plist', '' ) ) ? $upload_dir['baseurl'] . $this->get_array_value( $settings, 'publish.ios.firebase_plist', '' ) : '' );
			$value_path                    = ( ! empty( $this->get_array_value( $settings, 'publish.ios.firebase_plist', '' ) ) ? $upload_dir['basedir'] . $this->get_array_value( $settings, 'publish.ios.firebase_plist', '' ) : '' );
			$firebase_edit_link            = bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=firebase' );
			$review_info['firebase_plist'] = array(
				'label'       => __( 'Google Firebase - iOS Release App Configuration File', 'buddyboss-app' ),
				'value'       => ( ! empty( $value ) ? add_query_arg( 'cache', uniqid(), $value ) : '' ),
				'validated'   => ( is_file( $value_path ) ),
				'edit_link'   => $firebase_edit_link,
				'description' => __( 'Upload the .plist file created for your iOS release app in Google Firebase.', 'buddyboss-app' ),
			);

			if ( ! empty( $value ) ) {
				$review_info['files'][ $value ] = $value_path;
			}

			if ( isset( $review_info['firebase_plist']['validated'] ) && true === (bool) $review_info['firebase_plist']['validated'] && ! empty( $value_path ) && file_exists( $value_path ) && ( class_exists( 'BuddyBossAppScoping\PlistParser' ) || class_exists( 'PlistParser' ) ) ) {
				$plist_parser = \BuddyBossApp\Library\Composer::instance()->plist_parser_instance()->plist_parser();
				$result       = $plist_parser->searchKeyInPlist( $value_path, 'BUNDLE_ID' );
				$bundle_id    = $this->get_array_value( $settings, 'publish.ios.namespace', '' );

				if ( trim( $bundle_id ) !== trim( $result ) ) {
					$review_info['firebase_plist_invalid'] = array(
						'label'       => __( 'Google Firebase - iOS Release App Bundle ID Not Found', 'buddyboss-app' ),
						'value'       => false,
						'validated'   => false,
						'edit_link'   => $firebase_edit_link,
						'description' => __( 'Upload an iOS Release App Configuration File (.plist) that contains your iOS Release App Bundle ID.', 'buddyboss-app' ),
					);
				}
			}

			$value                               = ( ! empty( $this->get_array_value( $settings, 'publish.ios.provisioning_profile', '' ) ) ? $upload_dir['baseurl'] . $this->get_array_value( $settings, 'publish.ios.provisioning_profile', '' ) : '' );
			$value_path                          = ( ! empty( $this->get_array_value( $settings, 'publish.ios.provisioning_profile', '' ) ) ? $upload_dir['basedir'] . $this->get_array_value( $settings, 'publish.ios.provisioning_profile', '' ) : '' );
			$review_info['provisioning_profile'] = array(
				'label'        => __( 'Provisioning Profile', 'buddyboss-app' ),
				'value'        => ( ! empty( $value ) ? add_query_arg( 'cache', uniqid(), $value ) : '' ),
				'validated'    => ( is_file( $value_path ) ),
				'edit_link'    => $ios_edit_link . '#app_settings_app_publish_ios_live_settings',
				'label_prefix' => __( 'Release App', 'buddyboss-app' ),
			);

			if ( ! empty( $value ) ) {
				$review_info['files'][ $value ] = $value_path;
			}

			$value                              = ( ! empty( $this->get_array_value( $settings, 'publish.ios.signing_certificate', '' ) ) ? $upload_dir['baseurl'] . $this->get_array_value( $settings, 'publish.ios.signing_certificate', '' ) : '' );
			$value_path                         = ( ! empty( $this->get_array_value( $settings, 'publish.ios.signing_certificate', '' ) ) ? $upload_dir['basedir'] . $this->get_array_value( $settings, 'publish.ios.signing_certificate', '' ) : '' );
			$review_info['signing_certificate'] = array(
				'label'        => __( 'Signing Certificate', 'buddyboss-app' ),
				'value'        => ( ! empty( $value ) ? add_query_arg( 'cache', uniqid(), $value ) : '' ),
				'validated'    => ( is_file( $value_path ) ),
				'edit_link'    => $ios_edit_link . '#app_settings_app_publish_ios_live_settings',
				'label_prefix' => __( 'Release App', 'buddyboss-app' ),
			);

			if ( ! empty( $value ) ) {
				$review_info['files'][ $value ] = $value_path;
			}

			// Validate signing certificate expiration.
			if ( isset( $review_info['signing_certificate']['validated'] ) && true === (bool) $review_info['signing_certificate']['validated'] && ! empty( $value_path ) && file_exists( $value_path ) ) {
				$cert_password = Apple::instance()->get_signing_certificate_password();
				$cert_data     = file_get_contents( $value_path ); //phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
				$cert_info     = array();

				if ( openssl_pkcs12_read( $cert_data, $cert_info, $cert_password ) ) {
					if ( isset( $cert_info['cert'] ) ) {
						$cert_details = openssl_x509_parse( $cert_info['cert'] );

						if ( isset( $cert_details['validTo_time_t'] ) ) {
							$expiration_date = $cert_details['validTo_time_t'];
							$current_time    = time();

							if ( $expiration_date < $current_time ) {
								$review_info['signing_certificate_expired'] = array(
									'label'        => __( 'Signing Certificate Expired', 'buddyboss-app' ),
									'value'        => false,
									'validated'    => false,
									'edit_link'    => $ios_edit_link . '#app_settings_app_publish_ios_live_settings',
									'label_prefix' => __( 'Release App', 'buddyboss-app' ),
									'description'  => sprintf(
									/* translators: %s: expiration date */
										__( 'The signing certificate expired on %s. Please upload a valid certificate.', 'buddyboss-app' ),
										gmdate( 'F j, Y', $expiration_date )
									),
								);
							}
						}
					}
				}
			}

			$value                                       = Apple::instance()->get_signing_certificate_password();
			$review_info['signing_certificate_password'] = array(
				'label'        => __( 'Signing Certificate Password', 'buddyboss-app' ),
				'value'        => $value,
				'validated'    => true,
				'edit_link'    => $ios_edit_link . '#app_settings_app_publish_ios_live_settings',
				'label_prefix' => __( 'Release App', 'buddyboss-app' ),
			);

		}

		if ( 'dev' === $env ) {
			$value                         = ( ! empty( $this->get_array_value( $settings, 'publish.ios.dev.firebase_plist', '' ) ) ? $upload_dir['baseurl'] . $this->get_array_value( $settings, 'publish.ios.dev.firebase_plist', '' ) : '' );
			$value_path                    = ( ! empty( $this->get_array_value( $settings, 'publish.ios.dev.firebase_plist', '' ) ) ? $upload_dir['basedir'] . $this->get_array_value( $settings, 'publish.ios.dev.firebase_plist', '' ) : '' );
			$firebase_edit_link            = bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=firebase' );
			$review_info['firebase_plist'] = array(
				'label'       => __( 'Google Firebase - iOS Test App Configuration File', 'buddyboss-app' ),
				'value'       => ( ! empty( $value ) ? add_query_arg( 'cache', uniqid(), $value ) : '' ),
				'validated'   => ( is_file( $value_path ) ),
				'edit_link'   => $firebase_edit_link,
				'description' => __( 'Upload the .plist file created for your iOS test app in Google Firebase.', 'buddyboss-app' ),
			);

			if ( ! empty( $value ) ) {
				$review_info['files'][ $value ] = $value_path;
			}

			if ( isset( $review_info['firebase_plist']['validated'] ) && true === (bool) $review_info['firebase_plist']['validated'] && ! empty( $value_path ) && file_exists( $value_path ) && ( class_exists( 'BuddyBossAppScoping\PlistParser' ) || class_exists( 'PlistParser' ) ) ) {
				$plist_parser = \BuddyBossApp\Library\Composer::instance()->plist_parser_instance()->plist_parser();
				$result       = $plist_parser->searchKeyInPlist( $value_path, 'BUNDLE_ID' );
				$bundle_id    = $this->get_array_value( $settings, 'publish.ios.dev.namespace', '' );
				if ( trim( $bundle_id ) !== trim( $result ) ) {
					$review_info['firebase_plist_invalid'] = array(
						'label'       => __( 'Google Firebase - iOS Test App Bundle ID Not Found', 'buddyboss-app' ),
						'value'       => false,
						'validated'   => false,
						'edit_link'   => $firebase_edit_link,
						'description' => __( 'Upload an iOS Test App Configuration File (.plist) that contains your iOS Test App Bundle ID.', 'buddyboss-app' ),
					);
				}
			}

			$value                   = $this->get_array_value( $settings, 'publish.ios.dev.namespace', '' );
			$review_info['bundleid'] = array(
				'label'        => __( 'Bundle ID', 'buddyboss-app' ),
				'value'        => $value,
				'validated'    => ( ! empty( $value ) ),
				'edit_link'    => $ios_edit_link . '#app_settings_app_publish_ios_dev_settings',
				'label_prefix' => __( 'Test App', 'buddyboss-app' ),
			);

			$value                               = ( ! empty( $this->get_array_value( $settings, 'publish.ios.dev.provisioning_profile', '' ) ) ? $upload_dir['baseurl'] . $this->get_array_value( $settings, 'publish.ios.dev.provisioning_profile', '' ) : '' );
			$value_path                          = ( ! empty( $this->get_array_value( $settings, 'publish.ios.dev.provisioning_profile', '' ) ) ? $upload_dir['basedir'] . $this->get_array_value( $settings, 'publish.ios.dev.provisioning_profile', '' ) : '' );
			$review_info['provisioning_profile'] = array(
				'label'        => __( 'Provisioning Profile', 'buddyboss-app' ),
				'value'        => ( ! empty( $value ) ? add_query_arg( 'cache', uniqid(), $value ) : '' ),
				'validated'    => ( is_file( $value_path ) ),
				'edit_link'    => $ios_edit_link . '#app_settings_app_publish_ios_dev_settings',
				'label_prefix' => __( 'Test App', 'buddyboss-app' ),
			);

			if ( ! empty( $value ) ) {
				$review_info['files'][ $value ] = $value_path;
			}

			$value                              = ( ! empty( $this->get_array_value( $settings, 'publish.ios.dev.signing_certificate', '' ) ) ? $upload_dir['baseurl'] . $this->get_array_value( $settings, 'publish.ios.dev.signing_certificate', '' ) : '' );
			$value_path                         = ( ! empty( $this->get_array_value( $settings, 'publish.ios.dev.signing_certificate', '' ) ) ? $upload_dir['basedir'] . $this->get_array_value( $settings, 'publish.ios.dev.signing_certificate', '' ) : '' );
			$review_info['signing_certificate'] = array(
				'label'        => __( 'Signing Certificate', 'buddyboss-app' ),
				'value'        => ( ! empty( $value ) ? add_query_arg( 'cache', uniqid(), $value ) : '' ),
				'validated'    => ( is_file( $value_path ) ),
				'edit_link'    => $ios_edit_link . '#app_settings_app_publish_ios_dev_settings',
				'label_prefix' => __( 'Test App', 'buddyboss-app' ),
			);

			if ( ! empty( $value ) ) {
				$review_info['files'][ $value ] = $value_path;
			}

			// Validate signing certificate expiration.
			if ( isset( $review_info['signing_certificate']['validated'] ) && true === (bool) $review_info['signing_certificate']['validated'] && ! empty( $value_path ) && file_exists( $value_path ) ) {
				$cert_password = Apple::instance()->get_dev_signing_certificate_password();
				$cert_data     = file_get_contents( $value_path ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
				$cert_info     = array();

				if ( openssl_pkcs12_read( $cert_data, $cert_info, $cert_password ) ) {
					if ( isset( $cert_info['cert'] ) ) {
						$cert_details = openssl_x509_parse( $cert_info['cert'] );

						if ( isset( $cert_details['validTo_time_t'] ) ) {
							$expiration_date = $cert_details['validTo_time_t'];
							$current_time    = time();

							if ( $expiration_date < $current_time ) {
								$review_info['signing_certificate_expired'] = array(
									'label'        => __( 'Signing Certificate Expired', 'buddyboss-app' ),
									'value'        => false,
									'validated'    => false,
									'edit_link'    => $ios_edit_link . '#app_settings_app_publish_ios_dev_settings',
									'label_prefix' => __( 'Test App', 'buddyboss-app' ),
									'description'  => sprintf(
									/* translators: %s: expiration date */
										__( 'The signing certificate expired on %s. Please upload a valid certificate.', 'buddyboss-app' ),
										gmdate( 'F j, Y', $expiration_date )
									),
								);
							}
						}
					}
				}
			}

			$value = Apple::instance()->get_dev_signing_certificate_password();

			$review_info['signing_certificate_password'] = array(
				'label'        => __( 'Signing Certificate Password', 'buddyboss-app' ),
				'value'        => $value,
				'validated'    => true,
				'edit_link'    => $ios_edit_link . '#app_settings_app_publish_ios_dev_settings',
				'label_prefix' => __( 'Test App', 'buddyboss-app' ),
			);

			// Permissions.
			$permissions = array();

			if ( ! empty( ManageApp::instance()->get_app_setting( 'publish.ios.permissions.play_music' ) ) && Configure::instance()->has_custom_developer_access() ) {
				$permissions[] = 'PLAY_MUSIC';
			}

			$review_info['permissions'] = array(
				'label'             => esc_html__( 'Permissions', 'buddyboss-app' ),
				'value'             => ( ! empty( $permissions ) ? $permissions : esc_html__( 'N/A', 'buddyboss-app' ) ),
				'validated'         => true,
				'is_multi_checkbox' => true,
			);
		}

		return $review_info;
	}

	/**
	 * Returns the review information related to iOS
	 *
	 * @param string $env Platform.
	 *
	 * @return array
	 */
	public function get_review_information_v1( $env ) {
		$review_info      = array();
		$settings         = \BuddyBossApp\ManageApp::instance()->get_app_settings( true );
		$ios_edit_link    = bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=ios' );
		$app_version      = \BuddyBossApp\Build::instance()->get_app_version( 'ios' );
		$app_version_code = \BuddyBossApp\Build::instance()->get_app_version_code( 'ios' );

		$review_info['app_version_ios']      = array(
			'label'     => __( 'Build Version', 'buddyboss-app' ),
			'value'     => $app_version,
			'validated' => ( ! empty( $app_version ) ),
			'edit_link' => bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=general' ),
		);
		$review_info['app_version_code_ios'] = array(
			'label'     => __( 'Build Version Code', 'buddyboss-app' ),
			'value'     => $app_version_code,
			'validated' => ( ! empty( $app_version_code ) ),
		);
		$value                               = $this->get_array_value( $settings, 'publish.ios.namespace', '' );
		$review_info['bundleid']             = array(
			'label'        => __( 'Bundle ID', 'buddyboss-app' ),
			'value'        => $value,
			'validated'    => ( ! empty( $value ) ),
			'edit_link'    => $ios_edit_link . '#app_settings_app_publish_ios_live_settings',
			'label_prefix' => __( 'Release App', 'buddyboss-app' ),
		);

		return $review_info;
	}

	/**
	 * Returns the build information to iOS
	 *
	 * @param array $build Array of values.
	 *
	 * @return array
	 */
	public function get_build_information( $build ) {
		$review_info = array();

		if ( empty( $build ) ) {
			return array();
		}

		$review_info['app_version_ios']      = array(
			'label' => __( 'Build Version', 'buddyboss-app' ),
			'value' => ( isset( $build['app_version'] ) ? $build['app_version'] : '' ),
		);
		$review_info['app_version_code_ios'] = array(
			'label' => __( 'Build Version Code', 'buddyboss-app' ),
			'value' => ( isset( $build['app_version_code'] ) ? $build['app_version_code'] : '' ),
		);
		$review_info['bundleid']             = array(
			'label' => __( 'Bundle ID', 'buddyboss-app' ),
			'value' => ( isset( $build['bundleid'] ) ? $build['bundleid'] : '' ),
		);
		$upload_dir                          = wp_upload_dir();

		if ( ! empty( $build['app_icon'] ) ) {
			$value                                     = \BuddyBossApp\Admin\Appearance\Branding::instance()->get_upload_file_info( $build['app_icon'] );
			$review_info['files'][ $value['fileurl'] ] = $value['filepath'];
			$review_info['app_icon']                   = array(
				'label'    => __( 'iOS App Icon', 'buddyboss-app' ),
				'value'    => $value['fileurl'],
				'is_image' => true,
			);
		}

		$value   = '0';
		$preview = __( 'No', 'buddyboss-app' );

		if ( isset( $build['in_app_purchases'] ) ) {
			$value   = '1';
			$preview = __( 'Yes', 'buddyboss-app' );
		}

		$info['common']['in_app_purchases'] = array(
			'label'   => __( 'In-App Purchase Enabled', 'buddyboss-app' ),
			'value'   => $value,
			'preview' => $preview,
		);

		if ( ! empty( $build['signing_certificate'] ) ) {
			$value                              = ( ! empty( $build['signing_certificate'] ) ? $upload_dir['baseurl'] . '/bbapp/upload/' . $build['signing_certificate'] : '' );
			$value_path                         = ( ! empty( $build['signing_certificate'] ) ? $upload_dir['basedir'] . '/bbapp/upload/' . $build['signing_certificate'] : '' );
			$review_info['signing_certificate'] = array(
				'label' => __( 'Signing Certificate', 'buddyboss-app' ),
				'value' => $value,
			);
			$review_info['files'][ $value ]     = $value_path;
		}

		$review_info['signing_certificate_password'] = array(
			'label' => __( 'Signing Certificate Password', 'buddyboss-app' ),
			'value' => ( isset( $build['signing_certificate_password'] ) ? $build['signing_certificate_password'] : '' ),
		);

		if ( ! empty( $build['provisioning_profile'] ) ) {
			$value                               = ( ! empty( $build['provisioning_profile'] ) ? $upload_dir['baseurl'] . '/bbapp/upload/' . $build['provisioning_profile'] : '' );
			$value_path                          = ( ! empty( $build['provisioning_profile'] ) ? $upload_dir['basedir'] . '/bbapp/upload/' . $build['provisioning_profile'] : '' );
			$review_info['provisioning_profile'] = array(
				'label' => __( 'Provisioning Profile', 'buddyboss-app' ),
				'value' => $value,
			);
			$review_info['files'][ $value ]      = $value_path;
		}

		$value                          = ( ! empty( $build['firebase_plist'] ) ? $upload_dir['baseurl'] . '/bbapp/upload/' . $build['firebase_plist'] : '' );
		$value_path                     = ( ! empty( $build['firebase_plist'] ) ? $upload_dir['basedir'] . '/bbapp/upload/' . $build['firebase_plist'] : '' );
		$review_info['firebase_plist']  = array(
			'label'       => __( 'Firebase Info.plist file', 'buddyboss-app' ),
			'value'       => $value,
			'description' => __( 'iOS Firebase Settings', 'buddyboss-app' ),
		);
		$review_info['files'][ $value ] = $value_path;

		return $review_info;
	}
}
