<?php
/**
 * Holds build related functionality for the admin side.
 *
 * @package BuddyBossApp\Admin\Build
 */

namespace BuddyBossApp\Admin\Build;

use BuddyBossApp\Admin\Appearance\Branding;
use BuddyBossApp\Admin\Configure;
use BuddyBossApp\AppLanguages;
use BuddyBossApp\AppSettings;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Permissions;
use BuddyBossApp\Typography;

/**
 * Build class for admin.
 */
class Build {

	/**
	 * Class instance.
	 *
	 * @var Build $instance
	 */
	private static $instance;

	/**
	 * Check if current page to load HTML.
	 *
	 * @var bool $is_current_page
	 */
	private $is_current_page = false;

	/**
	 * Get the instance of the class.
	 *
	 * @return Build
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Build constructor.
	 */
	public function __construct() {
	}

	/**
	 * Filters/Hooks here.
	 */
	public function load() {
		add_action( 'admin_init', array( $this, 'redirects' ) );
		add_action( 'wp_ajax_bbapp_publish_submit', array( $this, 'process_submit' ) );
		add_action( 'wp_ajax_nopriv_bbapp_publish_submit', array( $this, 'process_submit' ) );
		add_action( 'wp_ajax_bbapp_build_status_actions', array( $this, 'build_status_actions' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );
		add_action( 'wp_ajax_bbapp_dismiss_notice', array( $this, 'bbapp_core_admin_notice_dismiss_callback' ) );
		add_action( 'wp_ajax_bbapp_update_build_permissions', array( $this, 'update_build_permissions' ) );
		add_action( 'wp_ajax_nopriv_bbapp_update_build_permissions', array( $this, 'update_build_permissions' ) );
	}

	/**
	 * Redirects.
	 */
	public function redirects() {
		if ( $this->will_render() ) {
			if ( 'review' === $this->get_current_step() ) {
				$app_platform = isset( $_GET['app_platform'] ) ? bbapp_input_clean( wp_unslash( $_GET['app_platform'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				$app_env      = isset( $_GET['app_env'] ) ? bbapp_input_clean( wp_unslash( $_GET['app_env'] ) ) : false; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

				/**
				 * Try to fetch the latest last versions
				 */
				\BuddyBossApp\Build::instance()->update_last_versions( $app_env );

				$build_option = \BuddyBossApp\Build::instance()->get_build_options();

				// if the build is in pending redirect to status page.
				if ( isset( $build_option[ $app_platform ][ $app_env ]['status'] ) && 'pending' === $build_option[ $app_platform ][ $app_env ]['status'] ) {
					$link = $this->get_link( "step=status&app_platform={$app_platform}&app_env={$app_env}" );
					wp_safe_redirect( $link );
					exit;
				}
			}
		}
	}

	/**
	 * Functions tells & sets that's if current page is one where it's will render.
	 *
	 * @param bool $set True if screen needs to load.
	 *
	 * @return bool
	 */
	public function will_render( $set = false ) {
		if ( $set ) {
			$this->is_current_page = true;
		}

		return $this->is_current_page;
	}

	/**
	 * Return the each statuses details.
	 *
	 * @return array|bool
	 */
	public function get_statuses_details() {
		$statuses = array(
			'new'       => array(
				'icon'  => 'icon-pending',
				'title' => __( 'Build Queued', 'buddyboss-app' ),
				'desc'  => __( 'Your request has been submitted and is waiting to be processed. The build process should begin within 10 minutes.', 'buddyboss-app' ),
			),
			'failed'    => array(
				'icon'  => 'icon-cancelled',
				'title' => __( 'Build Failed', 'buddyboss-app' ),
				'desc'  => __( 'The built attempt failed. BuddyBoss engineers will look into it shortly.', 'buddyboss-app' ),
			),
			'running'   => array(
				'icon'  => 'icon-pending',
				'title' => __( 'Build Processing', 'buddyboss-app' ),
				'desc'  => __( 'Your build is currently being created. Please allow up to one hour for this process to complete.', 'buddyboss-app' ),
			),
			'completed' => array(
				'icon'  => 'icon-completed',
				'title' => __( 'Your build is ready', 'buddyboss-app' ),
				'desc'  => __( 'Congrats, your build is ready! You can view it in your Build History.', 'buddyboss-app' ),
			),
			'cancelled' => array(
				'icon'  => 'icon-cancelled',
				'title' => __( 'Build Cancelled', 'buddyboss-app' ),
				'desc'  => __( 'The build review was cancelled. You can contact BuddyBoss customer support to get help, or try requesting a new build.', 'buddyboss-app' ),
			),
		);

		return $statuses;
	}

	/**
	 * Return the review information which will be sent.
	 *
	 * @param array $platforms Array of platforms.
	 * @param bool  $env       Environment.
	 *
	 * @return array
	 */
	public function get_review_information( $platforms = array(), $env = false ) {
		if ( ! in_array( $env, array( 'live', 'dev' ), true ) ) {
			return array();
		}

		if ( ! in_array( 'android', $platforms, true ) && ! in_array( 'ios', $platforms, true ) ) {
			return array();
		}

		$info = array(
			'common'  => array(),
			'ios'     => array(),
			'android' => array(),
		);

		if ( in_array( 'android', $platforms, true ) ) {
			$info['android'] = array_merge( $info['android'], AndroidHelper::instance()->get_review_information( $env ) );
		}

		if ( in_array( 'ios', $platforms, true ) ) {
			$info['ios'] = array_merge( $info['ios'], iOSHelper::instance()->get_review_information( $env ) );
		}

		$settings = ManageApp::instance()->get_app_settings();

		// Fetch App Name.
		$value                              = $this->get_array_value( $settings, 'publish.general.appname', '' );
		$androidos_edit_link                = bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=general' );
		$info['common']['appname']          = array(
			'label'     => __( 'App Icon Name', 'buddyboss-app' ),
			'value'     => $value,
			'validated' => ( ! empty( $value ) ),
			'edit_link' => $androidos_edit_link,
		);
		$info['common']['build_type']       = array(
			'label'     => __( 'Build Type', 'buddyboss-app' ),
			'value'     => $env,
			'preview'   => 'dev' === $env ? __( 'Test App', 'buddyboss-app' ) : __( 'Release App', 'buddyboss-app' ),
			'validated' => ( ! empty( $env ) ),
		);
		$info['common']['primary_site_url'] = array(
			'label'     => __( 'Primary Site URL', 'buddyboss-app' ),
			'value'     => home_url( '/' ),
			'validated' => true,
		);

		// Added support for the custom development branch.
		$branches = \BuddyBossApp\Admin\Configure::instance()->get_custom_development_branch();

		if ( ! empty( $branches ) ) {
			if ( 'dev' === $env ) {
				$branch_name = $this->get_array_value( $settings, 'publish.general.dev.branch', '' );
			} else {
				$branch_name = $this->get_array_value( $settings, 'publish.general.branch', '' );
			}

			$info['common']['repo_branch'] = array(
				'label'     => __( 'Custom Development Branch', 'buddyboss-app' ),
				'value'     => $branch_name,
				'validated' => ( ! empty( $branch_name ) ),
				'edit_link' => bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=general' ),
			);
		}

		$info['common']['header'] = array(
			'label'     => __( 'Launch Screen', 'buddyboss-app' ),
			'value'     => '',
			'validated' => true,
		);

		// Splash Screen.
		$value                               = Branding::instance()->get_upload_file_info( 'splash_screen_img.png' );
		$splash_screen_edit_link             = bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=branding&screen=splash' );
		$info['common']['splash_screen_img'] = array(
			'label'     => __( 'Launch Screen Image', 'buddyboss-app' ),
			'value'     => ( isset( $value['fileurl'] ) ? add_query_arg( 'cache', uniqid(), $value['fileurl'] ) : '' ),
			'validated' => ( ! empty( $value ) ),
			'edit_link' => $splash_screen_edit_link,
			'is_image'  => true,
		);

		if ( ! empty( $value['fileurl'] ) ) {
			$info['files'][ $value['fileurl'] ] = ! empty( $value['filepath'] ) ? $value['filepath'] : '';
		}

		// Splash Screen Spinner Checkbox.
		$value                                        = \BuddyBossApp\Branding::instance()->get_options();
		$splash_screen_spinner_edit_link              = bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=branding&screen=splash' );
		$info['common']['show_splash_screen_spinner'] = array(
			'label'     => __( 'Show Launch Screen Spinner', 'buddyboss-app' ),
			'value'     => isset( $value['splash_show_spinner'] ) && 1 === (int) $value['splash_show_spinner'] ? '1' : '0',
			'preview'   => isset( $value['splash_show_spinner'] ) && 1 === (int) $value['splash_show_spinner'] ? __( 'Yes', 'buddyboss-app' ) : __( 'No', 'buddyboss-app' ),
			'validated' => true,
			'edit_link' => $splash_screen_spinner_edit_link,
		);

		// Splash Screen Spinner Color.
		$value                                  = \BuddyBossApp\Branding::instance()->get_options();
		$splash_screen_spinner_edit_link        = bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=branding&screen=splash' );
		$info['common']['splash_spinner_color'] = array(
			'label'     => __( 'Launch Screen Spinner Color', 'buddyboss-app' ),
			'value'     => isset( $value['splash_spinner_color'] ) ? $value['splash_spinner_color'] : '#FFFFFF',
			'preview'   => '',
			'validated' => true,
			'edit_link' => $splash_screen_spinner_edit_link,
		);

		if ( isset( $info['common']['splash_spinner_color']['value'] ) ) {
			$info['common']['splash_spinner_color']['preview'] = "<div class='bb-launch-screen' style='width:38px;height:38px;display:inline-block;background:{$info['common']['splash_spinner_color']['value']};border:1px dashed gray; border-radius:100%'></div>";
		}

		// App Icon iOS.
		$value = '';
		$link  = '';
		if ( in_array( 'android', $platforms, true ) ) {
			$value                       = Branding::instance()->get_upload_file_info( 'app_icon_android.png' );
			$link                        = bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=branding&screen=appicon' );
			$info['android']['app_icon'] = array(
				'label'     => __( 'Android App Icon', 'buddyboss-app' ),
				'value'     => ( isset( $value['fileurl'] ) ? add_query_arg( 'cache', uniqid(), $value['fileurl'] ) : '' ),
				'validated' => ( ! empty( $value ) ),
				'edit_link' => $link,
				'is_image'  => true,
			);
			if ( ! empty( $value['fileurl'] ) ) {
				$info['android']['files'][ $value['fileurl'] ] = $value['filepath'];
			}
		}

		if ( in_array( 'ios', $platforms, true ) ) {
			$value                   = Branding::instance()->get_upload_file_info( 'app_icon_ios.png' );
			$link                    = bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=branding&screen=appicon' );
			$info['ios']['app_icon'] = array(
				'label'     => __( 'iOS App Icon', 'buddyboss-app' ),
				'value'     => ( isset( $value['fileurl'] ) ? add_query_arg( 'cache', uniqid(), $value['fileurl'] ) : '' ),
				'validated' => ( ! empty( $value ) ),
				'edit_link' => $link,
				'is_image'  => true,
			);
			if ( ! empty( $value['fileurl'] ) ) {
				$info['ios']['files'][ $value['fileurl'] ] = ( ! empty( $value['filepath'] ) ) ? $value['filepath'] : '';
			}
		}

		// Installed Fonts.
		$installed_fonts = Typography::instance()->get_installed_fonts( false );
		$value           = array();

		foreach ( $installed_fonts as $font_name => $font_info ) {
			foreach ( $font_info['fonts_family'] as $variant_name => $font_data ) {
				$font['url']                   = trailingslashit( Typography::instance()->get_fonts_upload_url() ) . $font_data['base_name'];
				$font['name']                  = $font_data['name'];
				$font['prev_name']             = $font_name;
				$font['ext']                   = $font_data['ext'];
				$value[]                       = $font;
				$info['files'][ $font['url'] ] = trailingslashit( Typography::instance()->get_fonts_upload_dir() ) . $font_data['base_name'];
			}
		}

		$preview = sprintf(
			'<p>%1$s %2$s</p>',
			count( array_unique( array_column( $value, 'prev_name' ) ) ),
			esc_html__( 'font(s) will be bundled with the app.', 'buddyboss-app' )
		);

		$preview .= "<ul class='bbapp-fonts-preview'>";

		$font_prev = array();

		foreach ( $value as $v ) {
			$font_prev[] = "<li><a target='_blank' href='" . bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&screen=font-title' ) . "'>{$v['prev_name']}</a></li>";
		}

		$preview .= implode( '', array_unique( $font_prev ) );
		$preview .= '</ul>';

		$font_edit_link          = bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography' );
		$info['common']['fonts'] = array(
			'label'     => __( 'Fonts', 'buddyboss-app' ),
			'value'     => $value,
			'preview'   => $preview,
			'validated' => true,
			'edit_link' => $font_edit_link,
		);

		// In-App Purchase.
		$value   = '0';
		$preview = __( 'No', 'buddyboss-app' );

		if ( bbapp_is_active( 'iap' ) ) {
			$value   = '1';
			$preview = __( 'Yes', 'buddyboss-app' );
		}

		$edit_link                          = bbapp_get_super_admin_url( 'admin.php?page=bbapp-settings&setting=inapp_purchases' );
		$info['common']['in_app_purchases'] = array(
			'label'     => __( 'In-App Purchase Enabled', 'buddyboss-app' ),
			'value'     => $value,
			'preview'   => $preview,
			'validated' => true,
			'edit_link' => $edit_link,
		);

		// Language json file.
		$languages_translations = AppLanguages::instance()->get_languages_translations();

		$value_path                      = $languages_translations['dir'];
		$value                           = $languages_translations['url'];
		$info['common']['language_json'] = array(
			'label'       => __( 'App Translations File', 'buddyboss-app' ),
			'value'       => ( ! empty( $value ) ? add_query_arg( 'cache', uniqid(), $value ) : '' ),
			'validated'   => ( is_file( $value_path ) ),
			'description' => __( 'The .json file Generating for app Translations.', 'buddyboss-app' ),
		);

		// Add WordPress default language.
		$info['common']['wp_default_language'] = array(
			'label'       => __( 'WordPress Default Language', 'buddyboss-app' ),
			'value'       => isset( $languages_translations['wp_default_language'] ) ? $languages_translations['wp_default_language'] : 'en',
			'validated'   => true,
			'description' => __( 'The default language of the WordPress installation.', 'buddyboss-app' ),
			'preview'     => AppLanguages::instance()->get_language_name( $languages_translations['wp_default_language'] ),
		);

		/* translators: %s: Supported languages count. */
		$preview = '<p>' . sprintf( __( '%s language(s) will be bundled with the app.', 'buddyboss-app' ), count( $languages_translations['supported_languages'] ) ) . '</p>';

		$preview      .= "<ul class='bbapp-languages-preview'>";
		$language_prev = array();

		foreach ( $languages_translations['supported_languages'] as $language ) {
			$language_prev[] = "<li data-language-code='{$language['code']}'>{$language['name']}</li>";
		}

		$preview .= implode( '', array_unique( $language_prev ) );
		$preview .= '</ul>';

		// Add formatted supported languages.
		if ( ! empty( $languages_translations['supported_languages'] ) ) {
			$info['common']['supported_languages'] = array(
				'label'       => __( 'Supported Languages', 'buddyboss-app' ),
				'value'       => $languages_translations['supported_languages'],
				'validated'   => true,
				'description' => __( 'Languages supported by the app.', 'buddyboss-app' ),
				'preview'     => $preview,
			);
		}

		// Is Universal Link is enabled Android.
		if ( in_array( 'android', $platforms, true ) ) {
			$info['common']['universal_link'] = array(
				'label'     => __( 'Enable Android App Links', 'buddyboss-app' ),
				'value'     => isset( $settings['publish.android.universal_link'] ) && 1 === (int) $settings['publish.android.universal_link'] ? 1 : 0,
				'preview'   => isset( $settings['publish.android.universal_link'] ) && 1 === (int) $settings['publish.android.universal_link'] ? __(
					'Yes',
					'buddyboss-app'
				) : __( 'No', 'buddyboss-app' ),
				'validated' => true,
			);
			$info['common']['links_to_open']  = array(
				'label'     => __( 'Links to Open', 'buddyboss-app' ),
				'value'     => isset( $settings['publish.android.links_to_open'] ) ? $settings['publish.android.links_to_open'] : '',
				'preview'   => isset( $settings['publish.android.links_to_open'] ) ? ucfirst( $settings['publish.android.links_to_open'] ) : '',
				'validated' => true,
			);
			$exclude_links                    = ! empty( $settings['publish.android.exclude_links'] ) ? $settings['publish.android.exclude_links'] : '';

			if ( ! empty( $exclude_links ) && isset( $settings['publish.android.links_to_open'] ) && 'specific' === $settings['publish.android.links_to_open'] ) {
				$exclude_links = explode( "\r\n", $exclude_links );
				$links_preview = '';

				foreach ( $exclude_links as $links ) {
					$links_preview .= '<p>' . $links . '</p>';
				}

				$info['common']['exclude_links'] = array(
					'label'     => __( 'Universal Link Enabled For', 'buddyboss-app' ),
					'value'     => ! empty( $exclude_links ) ? $exclude_links : '',
					'preview'   => ! empty( $links_preview ) ? $links_preview : '',
					'validated' => true,
				);
			}
		}

		// Is Universal Link is enabled iOS.
		if ( in_array( 'ios', $platforms, true ) ) {
			$info['common']['universal_link'] = array(
				'label'     => __( 'Enable Universal Links', 'buddyboss-app' ),
				'value'     => isset( $settings['publish.ios.universal_link'] ) && 1 === (int) $settings['publish.ios.universal_link'] ? 1 : 0,
				'preview'   => isset( $settings['publish.ios.universal_link'] ) && 1 === (int) $settings['publish.ios.universal_link'] ? __( 'Yes', 'buddyboss-app' ) : __(
					'No',
					'buddyboss-app'
				),
				'validated' => true,
			);
		}

		// Clear Text Traffic HTTP request.
		$settings                            = \BuddyBossApp\Admin\Settings::instance()->get_global_settings();
		$info['common']['cleartext_traffic'] = array(
			'label'     => __( 'HTTP Requests', 'buddyboss-app' ),
			'value'     => isset( $settings['cleartext_traffic.enabled'] ) && 1 === (int) $settings['cleartext_traffic.enabled'] ? '1' : '0',
			'preview'   => isset( $settings['cleartext_traffic.enabled'] ) && 1 === (int) $settings['cleartext_traffic.enabled'] ? __( 'Yes', 'buddyboss-app' ) : __(
				'No',
				'buddyboss-app'
			),
			'validated' => true,
		);

		// Add the review information for LMS.
		$info['common']['is_learndash_app']    = array(
			'label'     => __( 'Is Learndash app?', 'buddyboss-app' ),
			'value'     => bbapp_is_learndash_enabled(),
			'preview'   => bbapp_is_learndash_enabled() ? __( 'Yes', 'buddyboss-app' ) : __( 'No', 'buddyboss-app' ),
			'validated' => true,
		);
		$info['common']['is_tutor_app']        = array(
			'label'     => __( 'Is Tutor app?', 'buddyboss-app' ),
			'value'     => bbapp_is_tutor_lms_enabled(),
			'preview'   => bbapp_is_tutor_lms_enabled() ? __( 'Yes', 'buddyboss-app' ) : __( 'No', 'buddyboss-app' ),
			'validated' => true,
		);
		$info['common']['is_memberpress_app']  = array(
			'label'     => __( 'Is Memberpress app?', 'buddyboss-app' ),
			'value'     => bbapp_is_memberpress_lms_enabled(),
			'preview'   => bbapp_is_memberpress_lms_enabled() ? __( 'Yes', 'buddyboss-app' ) : __( 'No', 'buddyboss-app' ),
			'validated' => true,
		);
		$info['common']['is_platform_enabled'] = array(
			'label'     => __( 'Is Platform enabled?', 'buddyboss-app' ),
			'value'     => bbapp_is_buddyboss_platform_enabled(),
			'preview'   => bbapp_is_buddyboss_platform_enabled() ? __( 'Yes', 'buddyboss-app' ) : __( 'No', 'buddyboss-app' ),
			'validated' => true,
		);

		$info = apply_filters( 'bbapp_build_get_review_information', $info, $platforms, $env );

		return $info;
	}

	/**
	 * Build review information new.
	 *
	 * @param array  $platforms Array of platforms.
	 * @param string $env Build environment.
	 *
	 * @since 2.4.10
	 * @return array|mixed|null
	 */
	public function get_build_review_information( $platforms = array(), $env = false ) {
		if ( ! in_array( $env, array( 'live', 'dev' ), true ) ) {
			return array();
		}

		if ( ! in_array( 'android', $platforms, true ) && ! in_array( 'ios', $platforms, true ) ) {
			return array();
		}

		$settings     = ManageApp::instance()->get_app_settings();
		$app_name     = $this->get_array_value( $settings, 'publish.general.appname', '' );
		$build_type   = ( 'dev' === $env ) ? __( 'Test App', 'buddyboss-app' ) : __( 'Release App', 'buddyboss-app' );
		$push_enabled = ( bbapp_is_active( 'push_notification' ) ) ? __( 'Enabled', 'buddyboss-app' ) : __( 'Disabled', 'buddyboss-app' );
		$iap_enabled  = ( bbapp_is_active( 'iap' ) ) ? __( 'Enabled', 'buddyboss-app' ) : __( 'Disabled', 'buddyboss-app' );
		$info         = array();

		// Get the comprehensive review information.
		$review_info = $this->get_review_information( $platforms, $env );

		// Common fields for both platforms.
		$common_fields = array(
			'app_name'          => array(
				'label'     => __( 'App Name', 'buddyboss-app' ),
				'value'     => $app_name,
				'validated' => true,
				'edit_link' => bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=general' ),
			),
			'build_type'        => array(
				'label'     => __( 'Build Type', 'buddyboss-app' ),
				'value'     => $build_type,
				'validated' => true,
				'edit_link' => bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=general' ),
			),
			'push_notification' => array(
				'label'     => __( 'Push Notification', 'buddyboss-app' ),
				'value'     => $push_enabled,
				'validated' => true,
			),
			'iap'               => array(
				'label'     => __( 'In-App Purchases', 'buddyboss-app' ),
				'value'     => $iap_enabled,
				'validated' => true,
			),
		);

		// Platform specific configurations for app icons.
		$platform_configs = array(
			'ios'     => array(
				'icon_label' => __( 'iOS App Icon', 'buddyboss-app' ),
				'icon_file'  => 'app_icon_ios.png',
			),
			'android' => array(
				'icon_label' => __( 'Android App Icon', 'buddyboss-app' ),
				'icon_file'  => 'app_icon_android.png',
			),
		);

		foreach ( $platforms as $platform ) {
			if ( ! isset( $platform_configs[ $platform ] ) ) {
				continue;
			}

			$config            = $platform_configs[ $platform ];
			$info[ $platform ] = array_merge(
				$common_fields,
				array(
					'app_icon'         => array(
						'label'     => $config['icon_label'],
						'value'     => isset( $review_info[ $platform ]['app_icon']['value'] ) ? $review_info[ $platform ]['app_icon']['value'] : '',
						'validated' => isset( $review_info[ $platform ]['app_icon']['validated'] ) ? $review_info[ $platform ]['app_icon']['validated'] : false,
						'edit_link' => bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=branding&screen=appicon' ),
						'is_image'  => true,
					),
					'app_version'      => array(
						'label'     => __( 'Build Version', 'buddyboss-app' ),
						'value'     => \BuddyBossApp\Build::instance()->get_app_version( $platform ),
						'validated' => ! empty( \BuddyBossApp\Build::instance()->get_app_version( $platform ) ),
						'edit_link' => bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=general' ),
					),
					'app_version_code' => array(
						'label'     => __( 'Build Version Code', 'buddyboss-app' ),
						'value'     => \BuddyBossApp\Build::instance()->get_app_version_code( $platform ),
						'validated' => ! empty( \BuddyBossApp\Build::instance()->get_app_version_code( $platform ) ),
					),
				)
			);

			// Add files if they exist for the platform.
			if ( ! empty( $review_info[ $platform ]['files'] ) ) {
				$info[ $platform ]['files'] = $review_info[ $platform ]['files'];
			}
		}

		/**
		 * Filters build review information.
		 *
		 * @param array $info Build review information.
		 * @param array $platforms Platforms.
		 * @param string $env Build environment.
		 *
		 * @since 2.4.10
		 */
		return apply_filters( 'bbapp_build_review_information', $info, $platforms, $env );
	}

	/**
	 * Return summary of valid & invalid review info.
	 *
	 * @param array $review_info Build review information.
	 *
	 * @return array
	 */
	public function get_review_info_summary( $review_info ) {
		$return = array(
			'valid'   => array(),
			'invalid' => array(),
		);

		foreach ( $review_info as $k => $info ) {
			if ( 'files' === $k ) {
				continue;
			}

			if ( is_array( $info ) ) {
				if ( ! empty( $info ) ) {
					foreach ( $info as $key => $info_child ) {
						if ( 'files' === $key ) {
							continue;
						}

						if ( isset( $info_child['validated'] ) && $info_child['validated'] ) {
							$return['valid'][] = $key;
						} else {
							$return['invalid'][] = $key;
						}
					}
				}
			} elseif ( isset( $info['validated'] ) && $info['validated'] ) {
					$return['valid'][] = $k;
			} else {
				$return['invalid'][] = $k;
			}
		}

		return $return;
	}

	/**
	 * Renders the branding screen
	 *
	 * @return false|void
	 */
	public function render_request_build() {
		if ( ! Permissions::instance()->can_manage_app( get_Current_user_id() ) ) {
			printf( '<p>%1$s</p>', esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) );

			return false;
		}

		include bbapp()->plugin_dir . 'views/settings/build/request-build.php';
	}

	/**
	 * Render builds
	 *
	 * @return false|void
	 */
	public function render_builds() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			printf( '<p>%1$s</p>', esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) );

			return false;
		}

		include bbapp()->plugin_dir . 'views/settings/build/builds.php';
	}

	/**
	 * Output the nonce input for publish screen.
	 */
	public function publish_nonce_field() {
		wp_nonce_field(
			'bbapp_build_request_form_nonce',
			'bbapp_build_request_form_nonce'
		);
	}

	/**
	 * Submit Build Request.
	 */
	public function process_submit() {
		// Check permissions.
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			wp_send_json_error(
				array(
					'code'    => 'permission_denied',
					'message' => esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ),
				)
			);
		}

		// Verify nonce.
		$nonce = ! empty( $_POST['bbapp_build_request_form_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['bbapp_build_request_form_nonce'] ) ) : '';

		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'bbapp_build_request_form_nonce' ) ) {
			wp_send_json_error(
				array(
					'code'    => 'invalid_nonce',
					'message' => esc_html__( 'Error while verifying security nonce. Reload the page and try again.', 'buddyboss-app' ),
				)
			);
		}

		// Validate and sanitize inputs.
		$app_platforms = ! empty( $_POST['app_platform'] ) ? sanitize_text_field( wp_unslash( $_POST['app_platform'] ) ) : '';
		$app_env       = ! empty( $_POST['app_env'] ) ? sanitize_text_field( wp_unslash( $_POST['app_env'] ) ) : '';

		// Validate required fields.
		if ( empty( $app_platforms ) || empty( $app_env ) ) {
			wp_send_json_error(
				array(
					'code'    => 'missing_required_fields',
					'message' => esc_html__( 'App platform and environment are required.', 'buddyboss-app' ),
				)
			);
		}

		// Validate environment.
		if ( ! in_array( $app_env, array( 'dev', 'live' ), true ) ) {
			wp_send_json_error(
				array(
					'code'    => 'invalid_environment',
					'message' => esc_html__( 'Invalid Environment. Please try again.', 'buddyboss-app' ),
				)
			);
		}

		// Check if primary site.
		if ( ! ManageApp::instance()->is_current_primary_site() ) {
			wp_send_json_error(
				array(
					'code'    => 'not_primary_site',
					'message' => esc_html__( 'You are only allowed to send build request from primary site.', 'buddyboss-app' ),
				)
			);
		}

		// Update language selection.
		$app_langs = ! empty( $_POST['languages'] ) ? array_map( 'sanitize_text_field', (array) $_POST['languages'] ) : array(); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		AppSettings::instance()->update_setting_value( 'build_selected_languages', maybe_serialize( $app_langs ) );

		// Get app versions.
		$app_version = array(
			'ios'     => ! empty( $_POST['app_version_ios'] ) ? sanitize_text_field( wp_unslash( $_POST['app_version_ios'] ) ) : '',
			'android' => ! empty( $_POST['app_version_android'] ) ? sanitize_text_field( wp_unslash( $_POST['app_version_android'] ) ) : '',
		);

		$app_version_code = array(
			'ios'     => ! empty( $_POST['app_version_code_ios'] ) ? sanitize_text_field( wp_unslash( $_POST['app_version_code_ios'] ) ) : '',
			'android' => ! empty( $_POST['app_version_code_android'] ) ? sanitize_text_field( wp_unslash( $_POST['app_version_code_android'] ) ) : '',
		);

		$app_platforms = explode( ',', $app_platforms );

		// Lock build for each platform before processing.
		foreach ( $app_platforms as $platform ) {
			if ( ! \BuddyBossApp\Build::instance()->lock_build_for_platform( $platform ) ) {
				// If lock couldn't be acquired, unlock any previously locked platforms and return error.
				foreach ( $app_platforms as $locked_platform ) {
					\BuddyBossApp\Build::instance()->unlock_build_for_platform( $locked_platform );
				}

				$app_platform = ( 'ios' === $platform ) ? esc_html__( 'iOS', 'buddyboss-app' ) : esc_html__( 'Android', 'buddyboss-app' );

				wp_send_json_error(
					array(
						'code'    => 'build_locked_for_platform',
						'message' => sprintf(
							'%s "%s" %s',
							esc_html__( 'A build request for', 'buddyboss-app' ),
							$app_platform,
							esc_html__( 'is already being processed. Please wait and try again.', 'buddyboss-app' )
						),
					)
				);
			}
		}

		$review_information = $this->get_review_information( $app_platforms, $app_env );
		$review_summary     = $this->get_review_info_summary( $review_information );

		// Validate review information.
		if ( ! empty( $review_summary['invalid'] ) ) {
			// Unlock all platforms before returning error.
			foreach ( $app_platforms as $platform ) {
				\BuddyBossApp\Build::instance()->unlock_build_for_platform( $platform );
			}

			wp_send_json_error(
				array(
					'code'    => 'invalid_review_info',
					'message' => esc_html__( 'Please provide the required fields in order to send request', 'buddyboss-app' ),
				)
			);
		}

		// Validate build status and versions.
		foreach ( $app_platforms as $platform ) {
			if ( \BuddyBossApp\Build::instance()->is_build_in_process_for_platform( $platform ) ) {
				// Unlock all platforms before returning error.
				foreach ( $app_platforms as $locked_platform ) {
					\BuddyBossApp\Build::instance()->unlock_build_for_platform( $locked_platform );
				}

				$app_platform = ( 'ios' === $platform ) ? esc_html__( 'iOS', 'buddyboss-app' ) : esc_html__( 'Android', 'buddyboss-app' );

				wp_send_json_error(
					array(
						'code'    => 'build_in_progress_for_platform',
						'message' => sprintf(
							'%s "%s" %s',
							esc_html__( 'Request for ', 'buddyboss-app' ),
							$app_platform,
							esc_html__( ' app is already in progress. Please try again later.', 'buddyboss-app' )
						),
					)
				);
			}

			if ( isset( $app_version[ $platform ] ) && ! \BuddyBossApp\Build::instance()->is_valid_app_version( $app_version[ $platform ] ) ) {
				// Unlock all platforms before returning error.
				foreach ( $app_platforms as $locked_platform ) {
					\BuddyBossApp\Build::instance()->unlock_build_for_platform( $locked_platform );
				}

				wp_send_json_error(
					array(
						'code'    => 'invalid_version',
						'message' => sprintf(
							esc_html__( 'Provided app version is invalid for %s. Version can only contain digits and periods.', 'buddyboss-app' ),
							$platform
						),
					)
				);
			}

			if ( isset( $app_version_code[ $platform ] ) && ! is_numeric( $app_version_code[ $platform ] ) ) {
				// Unlock all platforms before returning error.
				foreach ( $app_platforms as $locked_platform ) {
					\BuddyBossApp\Build::instance()->unlock_build_for_platform( $locked_platform );
				}

				wp_send_json_error(
					array(
						'code'    => 'invalid_version_code',
						'message' => sprintf(
							esc_html__( 'App version code for %s can be numeric only.', 'buddyboss-app' ),
							$platform
						),
					)
				);
			}
		}

		// Generate build JSON.
		$build_json = $this->generate_build_json( $app_platforms, $app_env, $app_version, $app_version_code, $review_information );

		// Process build request.
		$response = $this->process_build_request( $build_json, $app_platforms, $app_env );

		// Handle response.
		if ( isset( $response['error'] ) ) {
			// Unlock all platforms before returning error.
			foreach ( $app_platforms as $platform ) {
				\BuddyBossApp\Build::instance()->unlock_build_for_platform( $platform );
			}

			wp_send_json_error( $response['error'] );
		}

		// Update build options and clear the cache.
		\BuddyBossApp\Build::instance()->clear_app_builds_cache();
		\BuddyBossApp\Build::instance()->update_build_options( $response['build_option'] );

		// Unlock all platforms after successful build request.
		foreach ( $app_platforms as $platform ) {
			\BuddyBossApp\Build::instance()->unlock_build_for_platform( $platform );
		}

		// Disable trigger build notice.
		AppSettings::instance()->update_setting_value( 'trigger_build_notice', false );

		// Send a success response.
		wp_send_json_success(
			array(
				'data'     => $response['build_option'],
				'redirect' => bbapp_get_super_admin_url(
					'admin.php?page=bbapp-build&setting=request-build&step=status&app_platform=' .
					implode( ',', $app_platforms ) . "&app_env={$app_env}"
				),
			)
		);
	}

	/**
	 * Generate build JSON for the request.
	 *
	 * @param array  $platforms         Platform types.
	 * @param string $app_env           Environment.
	 * @param array  $app_version       App versions.
	 * @param array  $app_version_code  App version codes.
	 * @param array  $review_information Review information.
	 *
	 * @since 2.4.10
	 * @return array
	 */
	private function generate_build_json( $platforms, $app_env, $app_version, $app_version_code, $review_information ) {
		$build_json = array(
			'ios'     => array(),
			'android' => array(),
		);

		// Generate the build JSON.
		foreach ( $platforms as $platform ) {
			$build_json[ $platform ] = array(
				'platform'         => $platform,
				'env'              => $app_env,
				'site_url'         => get_home_url(),
				'app_version'      => $app_version[ $platform ],
				'app_version_code' => $app_version_code[ $platform ],
			);
		}

		/**
		 * Append the review information to the build json.
		 * Review_information possible index are.
		 * android.
		 * ios.
		 * common.
		 * files.
		 */
		foreach ( $review_information as $rindex => $rvalue ) {
			if ( 'header' === $rindex ) {
				continue;
			}

			if ( 'files' === $rindex ) {
				$build_json[ $rindex ] = $rvalue;
				continue;
			}

			if ( is_array( $rvalue ) ) {
				if ( ! empty( $rvalue ) ) {
					foreach ( $rvalue as $info_name => $info ) {
						if ( 'header' === $info_name ) {
							continue;
						}

						if ( 'files' === $info_name ) {
							$build_json[ $rindex ][ $info_name ] = $info;
						} else {
							$build_json[ $rindex ][ $info_name ] = ( isset( $info['value'] ) ? $info['value'] : '' );
						}
					}
				}
			} else {
				$build_json[ $rindex ] = $rvalue['value'];
			}
		}

		// Merge the values from common rindex into platforms indexes & than unset common.
		if ( ! empty( $build_json['common'] ) ) {
			$common = $build_json['common'];

			if ( ! empty( $common ) ) {
				foreach ( $common as $info_name => $val ) {
					if ( in_array( $info_name, array( 'build_type', 'primary_site_url', 'header' ), true ) ) {
						continue;
					}

					foreach ( $platforms as $platform ) {
						$build_json[ $platform ][ $info_name ] = $val;
					}
				}
			}

			unset( $build_json['common'] );
		}

		// Merge values from files rindex into platforms indexes & then unset files.
		if ( isset( $build_json['files'] ) ) {
			if ( ! empty( $build_json['files'] ) ) {
				foreach ( $platforms as $platform ) {
					$platform_files                   = ! empty( $build_json[ $platform ]['files'] ) ? $build_json[ $platform ]['files'] : array();
					$build_json[ $platform ]['files'] = array_merge( $platform_files, $build_json['files'] );
				}
			}

			unset( $build_json['files'] );
		}

		return $build_json;
	}

	/**
	 * Process build request and handle responses.
	 *
	 * @param array  $build_json Build JSON data.
	 * @param array  $platforms  Platform types.
	 * @param string $app_env    Environment.
	 *
	 * @since 2.4.10
	 * @return array
	 */
	private function process_build_request( $build_json, $platforms, $app_env ) {
		$build_option = \BuddyBossApp\Build::instance()->get_build_options();

		if ( ! isset( $build_option['build_requested'] ) ) {
			$build_option['build_requested'] = array();
		}

		\BuddyBossApp\Build::instance()->clear_all_build_transients();

		foreach ( $platforms as $platform ) {
			if ( empty( $build_option['build_requested'][ $platform ]['build_id'] ) ) {
				unset( $build_option['build_requested'][ $platform ] );
			}

			$build_json_platform = wp_json_encode( $build_json[ $platform ], JSON_PRETTY_PRINT );

			$platform_response = \BuddyBossApp\Build::instance()->send_build_request(
				$build_json_platform,
				$build_json[ $platform ]['files']
			);

			if ( ! isset( $platform_response['build_id'] ) && isset( $platform_response['message'] ) ) {
				if ( 'maintenance_mode_enabled' === $platform_response['code'] ) {
					ManageApp::instance()->is_appcenter_under_maintenance( true );

					return array(
						'error' => array(
							'code'    => 'maintenance_mode',
							'message' => sprintf(
								esc_html__( 'The build request cannot be sent because the BuddyBoss server is under maintenance mode. Please check back later.', 'buddyboss-app' ),
								( 'ios' === $platform ? 'iOS' : 'Android' )
							),
						),
					);
				}

				$build_option['build_requested'][ $platform ] = array(
					'success'       => false,
					'build_id'      => null,
					'error_code'    => $platform_response['code'],
					'error_message' => $platform_response['message'],
				);
			} else {
				$build_option['build_requested'][ $platform ] = array(
					'success'       => true,
					'build_id'      => $platform_response['build_id'],
					'error_code'    => null,
					'error_message' => null,
					'build_status'  => \BuddyBossApp\Build::instance()->get_build_status( $platform_response['build_id'] ),
				);
			}

			$build_option['build_requested'][ $platform ]['type']         = $app_env;
			$build_option['build_requested'][ $platform ]['request_time'] = current_time( 'mysql', 1 );
			$build_option['build_requested'][ $platform ]['key']          = isset( $platform_response['data']['key'] ) ? $platform_response['data']['key'] : null;
		}

		return array(
			'build_option' => $build_option,
		);
	}

	/**
	 * Do force check build status for executing on build status page to get early results.
	 * The main status function only refresh cache in 10min. so here we override it by 1 min.
	 *
	 * @param int $build_id Build id.
	 *
	 * @return mixed|string
	 */
	public function get_build_status( $build_id ) {
		$transient_key = 'bbapp_last_build_status_interval_check';

		if ( 1 !== (int) get_transient( $transient_key ) ) {
			set_transient( $transient_key, '1', MINUTE_IN_SECONDS * 5 );

			return \BuddyBossApp\Build::instance()->get_build_status( $build_id, false );
		} else {
			return \BuddyBossApp\Build::instance()->get_build_status( $build_id );
		}
	}

	/**
	 * **Check Action**
	 * Force check the build status on bbapp server.
	 * Sometime client server can miss the update notify
	 * From bbapp server on that case this can be useful.
	 * **Cancel Action**
	 * This will allow to cancel build if it's in new status.
	 *
	 * @return bool
	 */
	public function build_status_actions() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			wp_send_json_error( __( "You don't have permission to access this page.", 'buddyboss-app' ) );

			return false;
		}

		$nonce = ( ! empty( $_POST['bbapp_build_actions'] ) ) ? sanitize_text_field( wp_unslash( $_POST['bbapp_build_actions'] ) ) : '';

		if ( wp_verify_nonce( $nonce, 'bbapp_build_actions' ) ) {
			$sub_action = isset( $_POST['sub_action'] ) ? bbapp_input_clean( wp_unslash( $_POST['sub_action'] ) ) : false; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$build_id   = (int) isset( $_POST['build_id'] ) ? bbapp_input_clean( wp_unslash( $_POST['build_id'] ) ) : false; //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			if ( 'status_check' === $sub_action ) {
				$this->update_build_status();

				wp_send_json_success( array( 'reload' => true ) );
			}

			if ( 'cancel' === $sub_action ) {
				\BuddyBossApp\Build::instance()->cancel_build( $build_id );
				$this->update_build_status();

				wp_send_json_success( array( 'reload' => true ) );
			}
		} else {
			wp_send_json_error( __( 'Error while validating security. Please refresh the page and try again.', 'buddyboss-app' ) );
		}
	}

	/**
	 * Return the current step.
	 *
	 * @return string
	 */
	public function get_current_step() {
		$step     = 'scope';
		$step_get = ( ! empty( $_GET['step'] ) ) ? bbapp_input_clean( wp_unslash( $_GET['step'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( ! empty( $step_get ) && in_array( $step_get, array( 'review', 'status' ), true ) ) {
			$step = $step_get;
		}

		$app_platforms = ( ! empty( $_POST['app_platform'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['app_platform'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$app_env       = ( ! empty( $_POST['app_env'] ) ) ? bbapp_input_clean( wp_unslash( $_POST['app_env'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( \BuddyBossApp\Build::instance()->is_build_in_process( $app_platforms, $app_env ) ) {
			$step = 'scope';
		}

		return $step;
	}

	/**
	 * Function to get link for super admin.
	 *
	 * @param string $extra_param Extra URL query string.
	 *
	 * @return string
	 */
	public function get_link( $extra_param ) {
		return bbapp_get_super_admin_url( "admin.php?page=bbapp-build&setting=build&{$extra_param}" );
	}

	/**
	 * Function to get the value from the array.
	 *
	 * @param array  $array   Array to get the value.
	 * @param string $key     Key to get the value.
	 * @param string $default Default data.
	 *
	 * @return string
	 */
	public function get_array_value( $array, $key, $default = '' ) {
		if ( isset( $array[ $key ] ) ) {
			return $array[ $key ];
		} else {
			return $default;
		}
	}

	/**
	 * Update Build status.
	 */
	public function update_build_status() {
		$build_option    = \BuddyBossApp\Build::instance()->get_build_options();
		$build_requested = isset( $build_option['build_requested'] ) ? $build_option['build_requested'] : array();

		foreach ( array( 'android', 'ios' ) as $platform ) {
			if ( isset( $build_requested[ $platform ] ) ) {
				if ( $build_requested[ $platform ]['success'] && $build_requested[ $platform ]['build_id'] ) {
					$build_option['build_requested'][ $platform ]['build_status'] = \BuddyBossApp\Build::instance()->get_build_status(
						$build_requested[ $platform ]['build_id'],
						false
					);
				} else {
					unset( $build_option['build_requested'][ $platform ] );
				}
			}
		}

		\BuddyBossApp\Build::instance()->clear_app_builds_cache();
		\BuddyBossApp\Build::instance()->update_build_options( $build_option );
	}

	/**
	 * Get Build details.
	 *
	 * @param int  $build_id       Build ID.
	 * @param bool $override_cache Override Cache.
	 *
	 * @return array|mixed
	 */
	public function get_build_details( $build_id, $override_cache = false ) {
		if ( empty( $build_id ) ) {
			return array();
		}

		$cache_key = 'bbapp_build_detail_' . $build_id;
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && empty( $override_cache ) ) {
			return $cached;
		} else {
			$args['bbapp_id']  = ManageApp::instance()->get_app_id();
			$args['bbapp_key'] = ManageApp::instance()->get_auth_app_key();
			$args['build_id']  = $build_id;
			$api_url           = \BuddyBossApp\ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/get-build-details' );
			$request           = bbapp_remote_get(
				$api_url,
				array(
					'timeout' => 15,
					'body'    => $args,
				)
			);
			$response          = wp_remote_retrieve_body( $request );
			$response          = json_decode( $response, true );
			$response_code     = wp_remote_retrieve_response_code( $request );
			$response_message  = wp_remote_retrieve_response_message( $request );

			if ( 'OK' === $response_message && 200 === $response_code && ! empty( $response ) ) {
				if ( isset( $response['id'] ) ) {
					set_site_transient( $cache_key, $response, 24 * HOUR_IN_SECONDS );

					return $response;
				}
			}
		}

		return array();
	}

	/**
	 * Get Builds information.
	 *
	 * @param array $build Array of Builds.
	 *
	 * @return array|mixed|void
	 */
	public function get_build_information( $build ) {
		if ( empty( $build ) ) {
			return array();
		}

		$info = array(
			'common'  => array(),
			'ios'     => array(),
			'android' => array(),
		);

		if ( array_key_exists( 'android', $build ) ) {
			$info['android'] = array_merge( $info['android'], AndroidHelper::instance()->get_build_information( $build['android'] ) );
		}

		if ( array_key_exists( 'ios', $build ) ) {
			$info['ios'] = array_merge( $info['ios'], iOSHelper::instance()->get_build_information( $build['ios'] ) );
		}

		$info['common']['build_info'] = array(
			'label'     => __( 'Build Information', 'buddyboss-app' ),
			'value'     => '',
			'validated' => true,
			'type'      => 'header',
		);

		// Fetch App-name.
		$info['common']['appname'] = array(
			'label' => __( 'App Icon Name', 'buddyboss-app' ),
			'value' => ( isset( $build['app_name'] ) ? $build['app_name'] : '' ),
		);

		$info['common']['primary_site_url'] = array(
			'label' => __( 'Primary Site URL', 'buddyboss-app' ),
			'value' => ( ! empty( $build['primary_url'] ) ? $build['primary_url'] : ManageApp::instance()->get_primary_site_url() ),
		);

		$info['common']['build_assets'] = array(
			'label'     => __( 'Build Assets', 'buddyboss-app' ),
			'value'     => '',
			'validated' => true,
			'type'      => 'header',
		);

		// Splash Screen.
		$value                               = Branding::instance()->get_upload_file_info( $build['splash_screen'] );
		$info['common']['splash_screen_img'] = array(
			'label'    => __( 'Launch Screen Image', 'buddyboss-app' ),
			'value'    => $value['fileurl'],
			'is_image' => true,
		);

		// Splash Screen Spinner Checkbox.
		$info['common']['show_splash_screen_spinner'] = array(
			'label'   => __( 'Show Launch Screen Spinner', 'buddyboss-app' ),
			'value'   => isset( $build['show_splash_screen_spinner'] ) && 1 === (int) $build['show_splash_screen_spinner'] ? '1' : '0',
			'preview' => isset( $build['show_splash_screen_spinner'] ) && 1 === (int) $build['show_splash_screen_spinner'] ? __( 'Yes', 'buddyboss-app' ) : __(
				'No',
				'buddyboss-app'
			),
		);

		// Splash Screen Spinner Color.
		$info['common']['splash_screen_spinner_color'] = array(
			'label'   => __( 'Launch Screen Spinner Color', 'buddyboss-app' ),
			'value'   => isset( $build['splash_screen_spinner_color'] ) ? $build['splash_screen_spinner_color'] : '#FFFFFF',
			'preview' => '',
		);

		if ( isset( $info['common']['splash_screen_spinner_color']['value'] ) && isset( $info['common']['splash_spinner_color']['value'] ) ) {
			$info['common']['splash_screen_spinner_color']['preview'] = "<div class='bb-launch-screen' style='width:38px;height:38px;display:inline-block;background:{$info['common']['splash_spinner_color']['value']};border:1px dashed gray; border-radius:100%'></div>";
		}

		// Installed Fonts.
		$installed_fonts = $build['fonts'];
		$value           = array();

		if ( ! empty( $installed_fonts ) ) {
			foreach ( $installed_fonts as $variant_name => $font_data ) {
				$font['url']  = trailingslashit( Typography::instance()->get_fonts_upload_url() ) . $font_data['url'];
				$font['name'] = $font_data['name'];
				$value[]      = $font;
			}
		}

		/* translators: %s: Fonts count. */
		$preview = '<p>' . sprintf( __( '%s font(s) will be bundled with the app.', 'buddyboss-app' ), count( array_unique( array_column( $value, 'name' ) ) ) ) . '</p>';

		$preview  .= "<ul class='bbapp-fonts-preview'>";
		$font_prev = array();

		foreach ( $value as $v ) {
			$font_prev[] = "<li><a target='_blank' href='" . bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&setting=typography&screen=font-title' ) . "'>{$v['name']}</a></li>";
		}

		$preview .= implode( '', array_unique( $font_prev ) );
		$preview .= '</ul>';

		$info['common']['fonts'] = array(
			'label'   => __( 'Fonts', 'buddyboss-app' ),
			'value'   => $value,
			'preview' => $preview,
		);

		/**
		 * Filters publish review information.
		 *
		 * @type array $info  Build information.
		 * @type array $build Build information data.
		 */
		$info = apply_filters( 'bbapp_build_get_publish_review_information', $info, $build );

		return $info;
	}

	/**
	 * Get build screen notice.
	 *
	 * @param bool $override_cache Get cached results.
	 *
	 * @return array|mixed
	 */
	public function get_appcenter_settings( $override_cache = false ) {
		$cache_key = 'bbapp_appcenter_settings';
		$cached    = get_site_transient( $cache_key );

		if ( false !== $cached && empty( $override_cache ) ) {
			return $cached;
		} else {
			$args['bbapp_id']  = ManageApp::instance()->get_app_id();
			$args['bbapp_key'] = ManageApp::instance()->get_auth_app_key();
			$api_url           = \BuddyBossApp\ClientCommon::instance()->get_center_api_url( 'v1', 'api-get/settings' );
			$request           = bbapp_remote_get(
				$api_url,
				array(
					'timeout' => 15,
					'body'    => $args,
				)
			);
			$response          = wp_remote_retrieve_body( $request );
			$response          = json_decode( $response, true );
			$response_code     = wp_remote_retrieve_response_code( $request );
			$response_message  = wp_remote_retrieve_response_message( $request );

			if ( 'OK' === $response_message && 200 === $response_code && ! empty( $response ) ) {
				set_site_transient( $cache_key, $response, 24 * HOUR_IN_SECONDS );

				return $response;
			}
		}

		return array();
	}

	/**
	 * About BuddyBoss Publishing Service Popup Screen.
	 * This is shown when BuddyBoss App iOS,Android first release build will create.
	 */
	public function about_buddyboss_publishing_service_screen() {
		include bbapp()->plugin_dir . 'views/admin/buddyboss-publishing-service.php';
	}

	/**
	 * Show admin notice for buddyboss publishing service.
	 */
	public function admin_notices() {
		$bbapp_get_latest_release_app = $this->bbapp_get_latest_release_app(
			array(
				'ios',
				'android',
			),
			array( 'completed', 'new' )
		);
		$ios_is_first_release_app     = false;
		$ios_is_new_release_app       = false;
		$android_is_first_release_app = false;
		$android_is_new_release_app   = false;

		if (
			! empty( $bbapp_get_latest_release_app ) &&
			( empty( $bbapp_get_latest_release_app['code'] ) || 'invalid_app' !== $bbapp_get_latest_release_app['code'] ) &&
			is_array( $bbapp_get_latest_release_app )
		) {
			foreach ( $bbapp_get_latest_release_app as $build ) {
				if ( is_array( $build ) && isset( $build['status'] ) && isset( $build['type'] ) ) {
					if ( 'completed' === $build['status'] && 'ios' === $build['type'] ) {
						$ios_is_first_release_app = true;
					}

					if ( 'new' === $build['status'] && 'ios' === $build['type'] ) {
						$ios_is_new_release_app = true;
					}

					if ( 'completed' === $build['status'] && 'android' === $build['type'] ) {
						$android_is_first_release_app = true;
					}

					if ( 'new' === $build['status'] && 'android' === $build['type'] ) {
						$android_is_new_release_app = true;
					}
				}
			}
		}

		// IOS success notice.
		if ( ( false === $ios_is_first_release_app && false !== $ios_is_new_release_app ) && false === \BuddyBossApp\Admin\Publish\Publish::instance()->is_publishing_services_already_requested( 'ios' ) ) {
			$notice_id = 'ios_publishing_services';

			if ( ! $this->get_stored_notice( 'bbapp-dismissed-notice-' . $notice_id ) ) {
				$msg                   = sprintf(
					'%1$s <a href="%2$s">%3$s</a>%4$s',
					esc_html__( 'Congratulations for generating your first iOS release app! Now use the', 'buddyboss-app' ),
					esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-publish&setting=ios' ) ),
					esc_html__( 'BuddyBoss Publishing Service', 'buddyboss-app' ),
					esc_html__( 'to publish to the Apple App Store.', 'buddyboss-app' )
				);
				$allowed_html          = wp_kses_allowed_html( 'post' );
				$allowed_html['input'] = array(
					'type'    => array(),
					'name'    => array(),
					'value'   => array(),
					'checked' => array(),
					'id'      => array(),
				);

				printf(
					'<div id="bbapp-site-connect-notice" class="%1$s" data-noticeid="%2$s"><p>%3$s</p>%4$s</div>',
					esc_attr( 'notice notice-success is-dismissible bbapp-is-dismissible' ),
					esc_attr( $notice_id ),
					wp_kses_post( $msg ),
					wp_kses( wp_nonce_field( "bbapp-dismissible-notice-$notice_id", "bbapp-dismissible-nonce-$notice_id", true, false ), $allowed_html )
				);
			}
		}

		// Android success notice.
		if ( ( false === $android_is_first_release_app && false !== $android_is_new_release_app ) && false === \BuddyBossApp\Admin\Publish\Publish::instance()->is_publishing_services_already_requested( 'android' ) ) {
			$notice_id = 'android_publishing_services';

			if ( ! $this->get_stored_notice( 'bbapp-dismissed-notice-' . $notice_id ) ) {
				$msg                   = sprintf(
					'%1$s <a href="%2$s">%3$s</a>%4$s',
					esc_html__( 'Congratulations for generating your first Android release app! Now use the', 'buddyboss-app' ),
					esc_url( bbapp_get_admin_url( 'admin.php?page=bbapp-publish&setting=android' ) ),
					esc_html__( 'BuddyBoss Publishing Service', 'buddyboss-app' ),
					esc_html__( 'to publish to the Google Play Store.', 'buddyboss-app' )
				);
				$allowed_html          = wp_kses_allowed_html( 'post' );
				$allowed_html['input'] = array(
					'type'    => array(),
					'name'    => array(),
					'value'   => array(),
					'checked' => array(),
					'id'      => array(),
				);
				printf(
					'<div id="bbapp-site-connect-notice" class="%1$s" data-noticeid="%2$s"><p>%3$s</p>%4$s</div>',
					esc_attr( 'notice notice-success is-dismissible bbapp-is-dismissible' ),
					esc_attr( $notice_id ),
					wp_kses_post( $msg ),
					wp_kses( wp_nonce_field( "bbapp-dismissible-notice-$notice_id", "bbapp-dismissible-nonce-$notice_id", true, false ), $allowed_html )
				);
			}
		}

		if ( AppSettings::instance()->get_setting_value( 'trigger_build_notice' ) ) {
			$notice = sprintf(
				__( 'Your license type has recently changed. <a href="%1$s">Request a new build</a> to update your app.', 'buddyboss-app' ),
				bbapp_get_admin_url( 'admin.php?page=bbapp-build&setting=request-build' )
			);

			printf( '<div id="bbapp-license-type-notice" class="notice notice-success bbapp-is-dismissible">%1$s</div>', wpautop( $notice ) );
		}

		if ( AppSettings::instance()->get_setting_value( 'trigger_site_disconnect_notice' ) ) {
			$notice = esc_html__(
				'Your site has now been disconnected. Your app will remain live for another 48 hours to allow you time to reconnect, if needed. After that, your app will also disconnect.',
				'buddyboss-app'
			);

			printf( '<div id="bbapp-disconnect-notice" class="notice notice-error">%1$s</div>', wpautop( $notice ) );
		}

		$notice_id = 'configure_data_reset';

		if ( $this->get_stored_notice( 'bbapp-dismissed-notice-' . $notice_id ) ) {
			return;
		}

		$notice                = sprintf(
			'%1$s <a href="%2$s">%3$s</a> %4$s',
			esc_html__(
				'You can now build your iOS and Android apps. To ensure your builds are successful, we have reset any existing iOS or Android app configuration settings to ensure the information provided is compatible with our build process. Please configure your apps in the',
				'buddyboss-app'
			),
			esc_url( admin_url( 'admin.php?page=bbapp-configure' ) ),
			esc_html__( 'Configure section', 'buddyboss-app' ),
			esc_html__( 'to start generating builds.', 'buddyboss-app' )
		);
		$allowed_html          = wp_kses_allowed_html( 'post' );
		$allowed_html['input'] = array(
			'type'    => array(),
			'name'    => array(),
			'value'   => array(),
			'checked' => array(),
			'id'      => array(),
		);
		printf(
			'<div class="notice notice-info is-dismissible bbapp-is-dismissible" data-noticeid="%1$s">%2$s%3$s</div>',
			esc_attr( $notice_id ),
			wp_kses_post( wpautop( $notice ) ),
			wp_kses( wp_nonce_field( "bbapp-dismissible-notice-$notice_id", "bbapp-dismissible-nonce-$notice_id", true, false ), $allowed_html )
		);
	}

	/**
	 * Catch and process an admin notice dismissal.
	 *
	 * @since BuddyBoss App 1.0.4
	 */
	public function bbapp_core_admin_notice_dismiss_callback() {
		if ( ! current_user_can( 'install_plugins' ) ) {
			wp_send_json_error();
		}

		if ( empty( $_POST['nonce'] ) || empty( $_POST['notice_id'] ) ) {
			wp_send_json_error();
		}

		$notice_id  = ( ! empty( $_POST['notice_id'] ) ) ? sanitize_text_field( wp_unslash( $_POST['notice_id'] ) ) : '';
		$nonce_post = ( ! empty( $_POST['nonce'] ) ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';

		if ( ! wp_verify_nonce( $nonce_post, 'bbapp-dismissible-notice-' . $notice_id ) ) {
			wp_send_json_error();
		}

		$notice['key']   = "bbapp-dismissed-notice-$notice_id";
		$notice['value'] = 1;

		$this->update_stored_notices( $notice );

		wp_send_json_success();
	}

	/**
	 * Update the platform-specific build permissions.
	 *
	 * @since 2.4.10
	 *
	 * @return void
	 */
	public function update_build_permissions() {
		// Verify nonce.
		$nonce = isset( $_POST['bbapp_permissions_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['bbapp_permissions_nonce'] ) ) : '';

		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'bbapp-build-permissions' ) ) {
			wp_send_json_error(
				array(
					'message' => esc_html__( 'Security check failed.', 'buddyboss-app' ),
				)
			);
		}

		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error(
				array(
					'message' => esc_html__( 'You do not have permission to perform this action.', 'buddyboss-app' ),
				)
			);
		}

		// Get the platform (android or ios).
		$platform = isset( $_POST['platform'] ) ? sanitize_text_field( wp_unslash( $_POST['platform'] ) ) : 'android';

		// Sanitize and validate permissions array.
		$raw_permissions = isset( $_POST['permissions'] ) ? (array) wp_unslash( $_POST['permissions'] ) : array(); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

		if ( 'android' === $platform ) {
			// Process Android permissions.
			$audio_recording     = ! empty( $raw_permissions['audio_recording'] ) && 'true' === $raw_permissions['audio_recording'];
			$foreground_services = ! empty( $raw_permissions['foreground_services'] ) && 'true' === $raw_permissions['foreground_services'];

			// Check if user has developer access for foreground services.
			if ( $foreground_services && ! Configure::instance()->has_custom_developer_access() ) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'You do not have permission to enable foreground services.', 'buddyboss-app' ),
					)
				);
			}

			// Update Android-specific settings.
			AppSettings::instance()->update_setting_value( 'publish.android.permissions.audio_recording', $audio_recording );
			AppSettings::instance()->update_setting_value( 'publish.android.permissions.foreground_media_playback', $foreground_services );
		} elseif ( 'ios' === $platform ) {
			// Process iOS permissions.
			$play_music = ! empty( $raw_permissions['play_music'] ) && 'true' === $raw_permissions['play_music'];

			// Update iOS-specific settings.
			AppSettings::instance()->update_setting_value( 'publish.ios.permissions.play_music', $play_music );
		}

		wp_send_json_success(
			array(
				'message' => esc_html__( 'Permissions updated successfully.', 'buddyboss-app' ),
			)
		);
	}

	/**
	 * Get last app with status.
	 *
	 * @param array $platform       Platform type ios or andriod or both.
	 * @param array $status         Build status.
	 * @param false $override_cache Override the cache.
	 *
	 * @since 1.5.7
	 *
	 * @return false|mixed
	 */
	public function is_last_release_app( $platform, $status = array( 'completed' ), $override_cache = false ) {
		static $cached;

		$cache_key = maybe_serialize( func_get_args() );

		if ( isset( $cached[ $cache_key ] ) ) {
			return $cached[ $cache_key ];
		}

		$builds = \BuddyBossApp\Build::instance()->get_app_builds( $platform, 'live', false, $status, 1, 1, $override_cache );

		if ( is_wp_error( $builds ) || ( is_array( $builds ) && empty( $builds['data'] ) ) ) {
			$cached[ $cache_key ] = false;

			return $cached[ $cache_key ];
		}

		if ( ! empty( $builds['data'] ) ) {
			$builds = $builds['data'];
		}

		$cached[ $cache_key ] = $builds;

		return $cached[ $cache_key ];
	}

	/**
	 * Get the latest app with status.
	 *
	 * @param array $platform       Platform type ios or andriod or both.
	 * @param array $status         Build status.
	 * @param false $override_cache Override the cache.
	 *
	 * @since 1.5.7
	 *
	 * @return false|mixed
	 */
	public function bbapp_get_latest_release_app( $platform, $status = array( 'completed' ), $override_cache = false ) {
		static $cached;

		$cache_key = maybe_serialize( func_get_args() );

		if ( isset( $cached[ $cache_key ] ) ) {
			return $cached[ $cache_key ];
		}

		$builds = \BuddyBossApp\Build::instance()->get_app_lateset_builds( $platform, 'live', $status, $override_cache );

		if ( is_wp_error( $builds ) || ( is_array( $builds ) && empty( $builds['data'] ) ) ) {
			$cached[ $cache_key ] = false;

			return $cached[ $cache_key ];
		}

		$builds               = ( ! empty( $builds['data'] ) ) ? $builds['data'] : false;
		$cached[ $cache_key ] = $builds;

		return $cached[ $cache_key ];
	}

	/**
	 * Get the dismissed notices.
	 *
	 * @since 1.8.80
	 * @return false|mixed|null
	 */
	public function get_stored_notices() {
		return get_option( 'bbapp-dismissed-notices', array() );
	}

	/**
	 * Get the stored notice.
	 *
	 * @param string $notice_key Notice key.
	 *
	 * @since 1.8.80
	 *
	 * @return false|mixed
	 */
	public function get_stored_notice( $notice_key ) {
		$dismissed_notices = $this->get_stored_notices();

		return isset( $dismissed_notices[ $notice_key ] ) ? $dismissed_notices[ $notice_key ] : false;
	}

	/**
	 * Update the dismissed notices.
	 *
	 * @param array $notice Notice data.
	 *
	 * @since 1.8.80
	 * @return bool
	 */
	public function update_stored_notices( $notice ) {
		if ( ! isset( $notice['key'] ) && ! isset( $notice['value'] ) ) {
			return false;
		}

		$dismissed_notices                   = $this->get_stored_notices();
		$dismissed_notices[ $notice['key'] ] = $notice['value'];

		return update_option( 'bbapp-dismissed-notices', $dismissed_notices );
	}

	/**
	 * Get the permission label.
	 *
	 * @param string $permission Permission.
	 *
	 * @return string
	 */
	public function get_permission_label( $permission ) {
		switch ( $permission ) {
			case 'RECORD_AUDIO':
				$label = __( 'Audio Recording', 'buddyboss-app' );
				break;
			case 'MODIFY_AUDIO_SETTINGS':
				$label = __( 'Modify Audio Settings', 'buddyboss-app' );
				break;
			case 'FOREGROUND_SERVICE_MEDIA_PLAYBACK':
				$label = __( 'Foreground Media Playback', 'buddyboss-app' );
				break;
			case 'PLAY_MUSIC':
				$label = __( 'Play Music', 'buddyboss-app' );
				break;
			default:
				$label = '';
		}

		return $label;
	}
}
