<?php
/**
 * Holds helper function for Android builds.
 *
 * @package BuddyBossApp\Admin\Build
 */

namespace BuddyBossApp\Admin\Build;

use BuddyBossApp\Admin\Appearance\Branding;
use BuddyBossApp\Admin\Configure;
use BuddyBossApp\ManageApp;

/**
 * Android build helper.
 */
class AndroidHelper {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Get the instance of the class.
	 *
	 * @return AndroidHelper
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * AndroidHelper constructor.
	 */
	public function __construct() {
	}

	/**
	 * Hooks/filter here.
	 */
	public function load() {
	}

	/**
	 * Function to get array value.
	 *
	 * @param array  $array   Array to get value.
	 * @param string $key     Key to get value.
	 * @param string $default Default data.
	 *
	 * @return string
	 */
	public function get_array_value( $array, $key, $default = '' ) {
		if ( isset( $array[ $key ] ) ) {
			return $array[ $key ];
		} else {
			return $default;
		}
	}

	/**
	 * Returns the review information related to Android
	 *
	 * @param string $env Environment.
	 *
	 * @return array
	 */
	public function get_review_information( $env ) {
		$settings            = ManageApp::instance()->get_app_settings( true );
		$review_info         = array();
		$app_version         = \BuddyBossApp\Build::instance()->get_app_version( 'android' );
		$app_version_code    = \BuddyBossApp\Build::instance()->get_app_version_code( 'android' );
		$androidos_edit_link = bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=android' );

		$review_info['app_version_android']      = array(
			'label'     => esc_html__( 'Build Version', 'buddyboss-app' ),
			'value'     => $app_version,
			'validated' => ( ! empty( $app_version ) ),
			'edit_link' => bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=general' ),
		);
		$review_info['app_version_code_android'] = array(
			'label'     => esc_html__( 'Build Version Code', 'buddyboss-app' ),
			'value'     => $app_version_code,
			'validated' => ( ! empty( $app_version_code ) ),
		);

		$value                   = $this->get_array_value( $settings, 'publish.android.namespace', '' );
		$review_info['bundleid'] = array(
			'label'     => esc_html__( 'Application ID', 'buddyboss-app' ),
			'value'     => $value,
			'validated' => ( ! empty( $value ) ),
			'edit_link' => $androidos_edit_link,
		);

		$upload_dir = wp_upload_dir();

		if ( 'live' === $env ) {
			$value                        = ( ! empty( $this->get_array_value( $settings, 'publish.android.firebase_json', '' ) ) ? $upload_dir['baseurl'] . $this->get_array_value( $settings, 'publish.android.firebase_json', '' ) : '' );
			$value_path                   = ( ! empty( $this->get_array_value( $settings, 'publish.android.firebase_json', '' ) ) ? $upload_dir['basedir'] . $this->get_array_value( $settings, 'publish.android.firebase_json', '' ) : '' );
			$firebase_edit_link           = bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=firebase' );
			$review_info['firebase_json'] = array(
				'label'       => esc_html__( 'Google Firebase - Android Release App Configuration File', 'buddyboss-app' ),
				'value'       => ( ! empty( $value ) ? add_query_arg( 'cache', uniqid(), $value ) : '' ),
				'validated'   => ( is_file( $value_path ) ),
				'edit_link'   => $firebase_edit_link,
				'description' => esc_html__( 'Upload the .json file created for your Android release app in Google Firebase.', 'buddyboss-app' ),
			);

			if ( ! empty( $value ) ) {
				$review_info['files'][ $value ] = $value_path;
			}

			if ( isset( $review_info['firebase_json']['validated'] ) && true === (bool) $review_info['firebase_json']['validated'] && ! empty( $value_path ) && file_exists( $value_path ) ) {
				if ( ! class_exists( '\WP_Filesystem_Direct' ) ) {
					require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';
					require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php';
				}

				$wp_files_system = new \WP_Filesystem_Direct( new \stdClass() );
				$json_data       = $wp_files_system->get_contents( $value_path );
				$data            = json_decode( $json_data, true );
				$bundle_id       = $this->get_array_value( $settings, 'publish.android.namespace', '' );
				$client_info     = ( ! empty( $data['client'] ) ? array_column( $data['client'], 'client_info' ) : array() );
				$client_info     = ( ! empty( $client_info ) ? array_column( $client_info, 'android_client_info' ) : array() );
				$client_info     = ( ! empty( $client_info ) ? array_column( $client_info, 'package_name' ) : array() );

				if ( empty( $data ) || empty( $client_info ) || ! in_array( $bundle_id, $client_info, true ) ) {
					$review_info['firebase_plist_invalid'] = array(
						'label'       => esc_html__( 'Google Firebase - Application ID Not Found', 'buddyboss-app' ),
						'value'       => false,
						'validated'   => false,
						'edit_link'   => $firebase_edit_link,
						'description' => esc_html__( 'Upload an Android Release App Configuration File (.json) that contains your Application ID.', 'buddyboss-app' ),
					);
				}
			}

			$value                   = ( ! empty( $this->get_array_value( $settings, 'publish.android.keystore', '' ) ) ? $upload_dir['baseurl'] . $this->get_array_value( $settings, 'publish.android.keystore', '' ) : '' );
			$value_path              = ( ! empty( $this->get_array_value( $settings, 'publish.android.keystore', '' ) ) ? $upload_dir['basedir'] . $this->get_array_value( $settings, 'publish.android.keystore', '' ) : '' );
			$review_info['keystore'] = array(
				'label'     => esc_html__( 'KeyStore', 'buddyboss-app' ),
				'value'     => ( ! empty( $value ) ? add_query_arg( 'cache', uniqid(), $value ) : '' ),
				'validated' => ( is_file( $value_path ) ),
				'edit_link' => $androidos_edit_link . '#app_settings_app_publish_android_live_settings',
			);

			if ( ! empty( $value ) ) {
				$review_info['files'][ $value ] = $value_path;
			}

			$value                            = $this->get_array_value( $settings, 'publish.android.keystore_pwd', '' );
			$review_info['keystore_password'] = array(
				'label'     => esc_html__( 'KeyStore Password', 'buddyboss-app' ),
				'value'     => $value,
				'validated' => ( ! empty( $value ) ),
				'edit_link' => $androidos_edit_link . '#app_settings_app_publish_android_live_settings',
			);

			$value                         = $this->get_array_value( $settings, 'publish.android.keystore_alias', '' );
			$review_info['keystore_alias'] = array(
				'label'     => esc_html__( 'KeyStore Alias', 'buddyboss-app' ),
				'value'     => $value,
				'validated' => ( ! empty( $value ) ),
				'edit_link' => $androidos_edit_link . '#app_settings_app_publish_android_live_settings',
			);

			$value                           = $this->get_array_value( $settings, 'publish.android.keystore_key_pwd', '' );
			$review_info['keystore_key_pwd'] = array(
				'label'     => esc_html__( 'KeyStore Key Password', 'buddyboss-app' ),
				'value'     => $value,
				'validated' => ( ! empty( $value ) ),
				'edit_link' => $androidos_edit_link . '#app_settings_app_publish_android_live_settings',
			);
		}

		if ( 'dev' === $env ) {
			$value                        = ( ! empty( $this->get_array_value( $settings, 'publish.android.dev.firebase_json', '' ) ) ? $upload_dir['baseurl'] . $this->get_array_value( $settings, 'publish.android.dev.firebase_json', '' ) : '' );
			$value_path                   = ( ! empty( $this->get_array_value( $settings, 'publish.android.dev.firebase_json', '' ) ) ? $upload_dir['basedir'] . $this->get_array_value( $settings, 'publish.android.dev.firebase_json', '' ) : '' );
			$firebase_edit_link           = bbapp_get_admin_url( 'admin.php?page=bbapp-configure&setting=firebase' );
			$review_info['firebase_json'] = array(
				'label'       => esc_html__( 'Google Firebase - Android Test App Configuration File', 'buddyboss-app' ),
				'value'       => ( ! empty( $value ) ? add_query_arg( 'cache', uniqid(), $value ) : '' ),
				'validated'   => ( is_file( $value_path ) ),
				'edit_link'   => $firebase_edit_link,
				'description' => esc_html__( 'Upload the .json file created for your Android test app in Google Firebase.', 'buddyboss-app' ),
			);

			if ( ! empty( $value ) ) {
				$review_info['files'][ $value ] = $value_path;
			}

			if ( isset( $review_info['firebase_json']['validated'] ) && true === (bool) $review_info['firebase_json']['validated'] && ! empty( $value_path ) && file_exists( $value_path ) ) {
				if ( ! class_exists( '\WP_Filesystem_Direct' ) ) {
					require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';
					require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php';
				}

				$wp_files_system = new \WP_Filesystem_Direct( new \stdClass() );
				$json_data       = $wp_files_system->get_contents( $value_path );
				$data            = json_decode( $json_data, true );
				$bundle_id       = $this->get_array_value( $settings, 'publish.android.dev.namespace', '' );
				$client_info     = ( ! empty( $data['client'] ) ? array_column( $data['client'], 'client_info' ) : array() );
				$client_info     = ( ! empty( $client_info ) ? array_column( $client_info, 'android_client_info' ) : array() );
				$client_info     = ( ! empty( $client_info ) ? array_column( $client_info, 'package_name' ) : array() );

				if ( empty( $data ) || empty( $client_info ) || ! in_array( $bundle_id, $client_info, true ) ) {
					$review_info['firebase_plist_invalid'] = array(
						'label'       => esc_html__( 'Google Firebase - Application ID Not Found', 'buddyboss-app' ),
						'value'       => false,
						'validated'   => false,
						'edit_link'   => $firebase_edit_link,
						'description' => esc_html__( 'Upload an Android Test App Configuration File (.json) that contains your Application ID.', 'buddyboss-app' ),
					);
				}
			}

			$value                   = $this->get_array_value( $settings, 'publish.android.dev.namespace', '' );
			$review_info['bundleid'] = array(
				'label'     => esc_html__( 'Application ID', 'buddyboss-app' ),
				'value'     => $value,
				'validated' => ( ! empty( $value ) ),
				'edit_link' => $androidos_edit_link,
			);

			$value                   = ( ! empty( $this->get_array_value( $settings, 'publish.android.keystore', '' ) ) ? $upload_dir['baseurl'] . $this->get_array_value( $settings, 'publish.android.keystore', '' ) : '' );
			$value_path              = ( ! empty( $this->get_array_value( $settings, 'publish.android.keystore', '' ) ) ? $upload_dir['basedir'] . $this->get_array_value( $settings, 'publish.android.keystore', '' ) : '' );
			$review_info['keystore'] = array(
				'label'     => esc_html__( 'KeyStore', 'buddyboss-app' ),
				'value'     => ( ! empty( $value ) ? add_query_arg( 'cache', uniqid(), $value ) : '' ),
				'validated' => ( is_file( $value_path ) ),
				'edit_link' => $androidos_edit_link . '#app_settings_app_publish_android_dev_settings',
			);

			if ( ! empty( $value ) ) {
				$review_info['files'][ $value ] = $value_path;
			}

			$value                            = $this->get_array_value( $settings, 'publish.android.keystore_pwd', '' );
			$review_info['keystore_password'] = array(
				'label'     => __( 'KeyStore Password', 'buddyboss-app' ),
				'value'     => $value,
				'validated' => ( ! empty( $value ) ),
				'edit_link' => $androidos_edit_link . '#app_settings_app_publish_android_dev_settings',
			);

			$value                         = $this->get_array_value( $settings, 'publish.android.keystore_alias', '' );
			$review_info['keystore_alias'] = array(
				'label'     => esc_html__( 'KeyStore Alias', 'buddyboss-app' ),
				'value'     => $value,
				'validated' => ( ! empty( $value ) ),
				'edit_link' => $androidos_edit_link . '#app_settings_app_publish_android_dev_settings',
			);

			$value                           = $this->get_array_value( $settings, 'publish.android.keystore_key_pwd', '' );
			$review_info['keystore_key_pwd'] = array(
				'label'     => esc_html__( 'KeyStore Key Password', 'buddyboss-app' ),
				'value'     => $value,
				'validated' => ( ! empty( $value ) ),
				'edit_link' => $androidos_edit_link . '#app_settings_app_publish_android_dev_settings',
			);
		}

		// Android Adaptive Icon.
		$value                                   = Branding::instance()->get_upload_file_info( 'app_icon_android_adaptive_bg.png' );
		$link                                    = bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&tab=app_icons' );
		$review_info['android_adaptive_icon_bg'] = array(
			'label'     => esc_html__( 'Adaptive Icon Background', 'buddyboss-app' ),
			'value'     => ( ! empty( $value['fileurl'] ) ? add_query_arg( 'cache', uniqid(), $value['fileurl'] ) : '' ),
			'validated' => ( ! empty( $value ) ),
			'edit_link' => $link,
			'is_image'  => true,
		);

		if ( $review_info['android_adaptive_icon_bg']['validated'] ) {
			$review_info['files'][ $value['fileurl'] ] = $value['filepath'];
		}

		$value                                   = Branding::instance()->get_upload_file_info( 'app_icon_android_adaptive_fg.png' );
		$link                                    = bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&tab=app_icons' );
		$review_info['android_adaptive_icon_fg'] = array(
			'label'     => esc_html__( 'Adaptive Icon Foreground', 'buddyboss-app' ),
			'value'     => ( ! empty( $value['fileurl'] ) ? add_query_arg( 'cache', uniqid(), $value['fileurl'] ) : '' ),
			'validated' => ( ! empty( $value ) ),
			'edit_link' => $link,
			'is_image'  => true,
		);

		if ( $review_info['android_adaptive_icon_fg']['validated'] ) {
			$review_info['files'][ $value['fileurl'] ] = $value['filepath'];
		}

		// When one of them is not available then remove both of them from list.
		if ( ! $review_info['android_adaptive_icon_bg']['validated'] || ! $review_info['android_adaptive_icon_fg']['validated'] ) {
			unset( $review_info['android_adaptive_icon_bg'], $review_info['android_adaptive_icon_fg'] );
		}

		// Android Notification Icon.
		$value = Branding::instance()->get_upload_file_info( 'app_icon_android_notification.png' );

		if ( ! empty( $value ) ) {
			$link                                     = bbapp_get_admin_url( 'admin.php?page=bbapp-appearance&tab=app_icons' );
			$review_info['android_notification_icon'] = array(
				'label'     => esc_html__( 'Android Notification Icon', 'buddyboss-app' ),
				'value'     => ( ! empty( $value['fileurl'] ) ? add_query_arg( 'cache', uniqid(), $value['fileurl'] ) : '' ),
				'validated' => ( ! empty( $value ) ),
				'edit_link' => $link,
				'is_image'  => true,
			);

			if ( $review_info['android_notification_icon']['validated'] ) {
				$review_info['files'][ $value['fileurl'] ] = $value['filepath'];
			}
		}

		// Permissions.
		$permissions = array();

		if ( ! empty( ManageApp::instance()->get_app_setting( 'publish.android.permissions.audio_recording' ) ) ) {
			$permissions[] = 'RECORD_AUDIO';
			$permissions[] = 'MODIFY_AUDIO_SETTINGS';
		}

		if ( ! empty( ManageApp::instance()->get_app_setting( 'publish.android.permissions.foreground_media_playback' ) ) && Configure::instance()->has_custom_developer_access() ) {
			$permissions[] = 'FOREGROUND_SERVICE_MEDIA_PLAYBACK';
		}

		$review_info['permissions'] = array(
			'label'             => esc_html__( 'Permissions', 'buddyboss-app' ),
			'value'             => ( ! empty( $permissions ) ? $permissions : esc_html__( 'N/A', 'buddyboss-app' ) ),
			'validated'         => true,
			'is_multi_checkbox' => true,
		);

		return $review_info;
	}

	/**
	 * Returns the build information to Android
	 *
	 * @param array $build Array of values.
	 *
	 * @return array
	 */
	public function get_build_information( $build ) {
		$review_info = array();

		if ( empty( $build ) ) {
			return array();
		}

		$review_info['app_version_android']      = array(
			'label' => esc_html__( 'Build Version', 'buddyboss-app' ),
			'value' => $build['app_version'],
		);
		$review_info['app_version_code_android'] = array(
			'label' => esc_html__( 'Build Version Code', 'buddyboss-app' ),
			'value' => $build['app_version_code'],
		);
		$review_info['bundleid']                 = array(
			'label' => esc_html__( 'Application ID', 'buddyboss-app' ),
			'value' => $build['application_id'],
		);
		$value                                   = '0';
		$preview                                 = esc_html__( 'No', 'buddyboss-app' );

		if ( isset( $build['in_app_purchases'] ) ) {
			$value   = '1';
			$preview = esc_html__( 'Yes', 'buddyboss-app' );
		}

		$info['common']['in_app_purchases'] = array(
			'label'   => esc_html__( 'In-App Purchase Enabled', 'buddyboss-app' ),
			'value'   => $value,
			'preview' => $preview,
		);
		$upload_dir                         = wp_upload_dir();

		if ( ! empty( $build['app_icon'] ) ) {
			$value                                     = Branding::instance()->get_upload_file_info( $build['app_icon'] );
			$review_info['app_icon']                   = array(
				'label'    => esc_html__( 'Android App Icon', 'buddyboss-app' ),
				'value'    => $value['fileurl'],
				'is_image' => true,
			);
			$review_info['files'][ $value['fileurl'] ] = $value['filepath'];
		}

		if ( ! empty( $build['android_adaptive_icon_fg'] ) ) {
			$value                                     = Branding::instance()->get_upload_file_info( $build['android_adaptive_icon_fg'] );
			$review_info['android_adaptive_icon_fg']   = array(
				'label'    => esc_html__( 'Adaptive Icon Foreground', 'buddyboss-app' ),
				'value'    => $value['fileurl'],
				'is_image' => true,
			);
			$review_info['files'][ $value['fileurl'] ] = $value['filepath'];
		}

		if ( ! empty( $build['android_adaptive_icon_bg'] ) ) {
			/**
			 * Android Adaptive Icon
			 */
			$value                                     = Branding::instance()->get_upload_file_info( $build['android_adaptive_icon_bg'] );
			$review_info['android_adaptive_icon_bg']   = array(
				'label'    => esc_html__( 'Adaptive Icon Background', 'buddyboss-app' ),
				'value'    => $value['fileurl'],
				'is_image' => true,
			);
			$review_info['files'][ $value['fileurl'] ] = $value['filepath'];
		}

		if ( ! empty( $build['android_notification_icon'] ) ) {
			// Android Notification Icon.
			$value = Branding::instance()->get_upload_file_info( $build['android_notification_icon'] );
			if ( ! empty( $value ) ) {
				$review_info['android_notification_icon']  = array(
					'label'    => esc_html__( 'Android Notification Icon', 'buddyboss-app' ),
					'value'    => $value['fileurl'],
					'is_image' => true,
				);
				$review_info['files'][ $value['fileurl'] ] = $value['filepath'];
			}
		}

		$value                            = ( ! empty( $build['keystore'] ) ? $upload_dir['baseurl'] . '/bbapp/uploads/' . $build['keystore'] : '' );
		$value_path                       = ( ! empty( $build['keystore'] ) ? $upload_dir['basedir'] . '/bbapp/uploads/' . $build['keystore'] : '' );
		$review_info['keystore']          = array(
			'label' => esc_html__( 'KeyStore', 'buddyboss-app' ),
			'value' => $value,
		);
		$review_info['files'][ $value ]   = $value_path;
		$review_info['keystore_alias']    = array(
			'label' => esc_html__( 'KeyStore Alias', 'buddyboss-app' ),
			'value' => $build['keystore_alias'],
		);
		$review_info['keystore_password'] = array(
			'label' => esc_html__( 'KeyStore Password', 'buddyboss-app' ),
			'value' => $build['keystore_password'],
		);
		$review_info['keystore_key_pwd']  = array(
			'label' => esc_html__( 'KeyStore Key Password', 'buddyboss-app' ),
			'value' => $build['keystore_key_pwd'],
		);
		$value                            = ( ! empty( $build['firebase_json'] ) ? $upload_dir['baseurl'] . '/bbapp/uploads/' . $build['firebase_json'] : '' );
		$value_path                       = ( ! empty( $build['firebase_json'] ) ? $upload_dir['basedir'] . '/bbapp/uploads/' . $build['firebase_json'] : '' );
		$review_info['firebase_json']     = array(
			'label'       => esc_html__( 'Firebase Json file', 'buddyboss-app' ),
			'value'       => $value,
			'description' => esc_html__( 'Android Firebase Settings', 'buddyboss-app' ),
		);
		$review_info['files'][ $value ]   = $value_path;

		return $review_info;
	}
}
