<?php
/**
 * Holds build admin functionality.
 *
 * @package BuddyBossApp\Admin\Build
 */

namespace BuddyBossApp\Admin;

use BuddyBossApp\Admin\Build\Build as InnerBuild;
use BuddyBossApp\ManageApp;
use BuddyBossApp\Permissions;

/**
 * Build class for admin.
 */
class Build {

	/**
	 * Class instance.
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Setting form.
	 *
	 * @var $settings_form
	 */
	private $settings_form;

	/**
	 * Get the instance of the class.
	 *
	 * @return Build
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			$class          = __CLASS__;
			self::$instance = new $class();
			self::$instance->load();
		}

		return self::$instance;
	}

	/**
	 * Build constructor.
	 */
	public function __construct() {
	}

	/**
	 * Hooks here.
	 */
	public function load() {
		add_action( 'init', array( $this, 'load_init' ) );
	}

	/**
	 * Init method.
	 */
	public function load_init() {
		InnerBuild::instance();

		$page_get   = ( ! empty( $_GET['page'] ) ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$action_get = ( ! empty( $_GET['action'] ) ) ? sanitize_text_field( wp_unslash( $_GET['action'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( 'admin.php' === SetupAdmin::instance()->get_page_now() && 'bbapp-build' === $page_get ) {
			if ( 'bbapp_min_version_build_toggle' === $action_get ) {
				$this->action_min_version_code_build_toggle();
			}

			$this->settings_form = new FormProvider( 'settings' );
			// holds app settings storage handle.
			$app_settings_provider = new SettingProvider( 'app_settings' );
			// register app settings storing & receiving.
			$app_settings_provider->save_setting_hook( array( $this, 'save_settings' ) );
			$app_settings_provider->get_setting_hook( array( $this, 'get_settings' ) );
			// Link Provider In Form Provider.
			$this->settings_form->link_setting_provider( $app_settings_provider );

			add_action( 'admin_init', array( $this->settings_form, 'save' ) );

			// When All Builds Page Will Render.
			if ( 'build' === $this->get_current_sub_tab() || 'all-builds' === $this->get_current_sub_tab() ) {
				InnerBuild::instance()->will_render( true );
			}
		}
	}

	/**
	 * Saves the settings
	 *
	 * @param array $settings Array of options.
	 *
	 * @return bool
	 */
	public function save_settings( $settings ) {
		ManageApp::instance()->update_app_settings( $settings );

		return true;
	}

	/**
	 * Gets the settings
	 */
	public function get_settings() {
		$settings = ManageApp::instance()->get_app_settings();

		return $settings;
	}

	/**
	 * Return the sub settings.
	 *
	 * @return array
	 */
	public function sub_settings() {
		$tabs = array(
			'all-builds'      => esc_html__( 'All Builds', 'buddyboss-app' ),
			'request-build'   => esc_html__( 'Request Build', 'buddyboss-app' ),
			'all-uploads'     => esc_html__( 'All Uploads', 'buddyboss-app' ),
			'request-ios'     => esc_html__( 'Request Build', 'buddyboss-app' ),
			'request-android' => esc_html__( 'Request Build', 'buddyboss-app' ),
		);

		return $tabs;
	}

	/**
	 * Return the current setting page from active screen.
	 *
	 * @return string
	 */
	public function get_current_sub_tab() {
		$default      = 'all-builds';
		$setting      = ( isset( $_GET['setting'] ) ) ? sanitize_text_field( wp_unslash( $_GET['setting'] ) ) : $default; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$sub_settings = $this->sub_settings();

		if ( isset( $sub_settings[ $setting ] ) ) {
			return $setting;
		}

		return $default;
	}

	/**
	 * Renders the setting screen.
	 */
	public function render_screen() {
		if ( ! Permissions::instance()->can_manage_app( get_current_user_id() ) ) {
			echo '<p>' . esc_html__( 'You don\'t have permission to access this page.', 'buddyboss-app' ) . '</p>';

			return false;
		}

		// Builder Server under Maintenance Mode.
		$bbapp = ManageApp::instance()->get_app();

		if ( ! empty( $bbapp ) && ! empty( $bbapp['verified'] ) && ! empty( $bbapp['bbapp_site_type'] ) && 'primary' === $bbapp['bbapp_site_type'] ) {
			$maintenance_mode = ManageApp::instance()->is_appcenter_under_maintenance();

			if ( is_wp_error( $maintenance_mode ) ) {
				include bbapp()->plugin_dir . '/views/settings/build/no-connect.php';
			} elseif ( ! is_wp_error( $maintenance_mode ) && true === $maintenance_mode ) {
				include bbapp()->plugin_dir . '/views/settings/build/maintenance-mode.php';
			} else {
				if ( $this->get_current_sub_tab() === 'request-build' ) {
					InnerBuild::instance()->render_request_build();
				} elseif ( $this->get_current_sub_tab() === 'all-builds' ) {
					InnerBuild::instance()->render_builds();
				} else {
					if ( isset( $this->settings_form ) && $this->settings_form ) {
						$this->settings_form->render_form();
					}
				}
			}
		} elseif (
			! empty( $bbapp ) &&
			(
				! empty( $bbapp['verified'] ) &&
				! empty( $bbapp['bbapp_site_type'] ) &&
				'secondary' === $bbapp['bbapp_site_type']
			)
		) {
			include bbapp()->plugin_dir . '/views/settings/build/build-connect-dev.php';
		} else {
			include bbapp()->plugin_dir . '/views/settings/build/build-no-connection.php';
		}
	}

	/**
	 * Min Version Update Action Handler.
	 */
	public function action_min_version_code_build_toggle() {
		// Verify Nonce.
		if ( isset( $_GET['bbapp_min_version_build_toggle_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['bbapp_min_version_build_toggle_nonce'] ) ), 'bbapp_min_version_build_toggle_nonce' ) ) {
			$platform           = isset( $_GET['platform'] ) ? sanitize_text_field( wp_unslash( $_GET['platform'] ) ) : false;
			$build_version_code = isset( $_GET['build_version_code'] ) ? sanitize_text_field( wp_unslash( $_GET['build_version_code'] ) ) : 0;

			if ( in_array( $platform, array( 'ios', 'android' ), true ) && $build_version_code ) {
				$min_code_support_version = \BuddyBossApp\Build::instance()->get_build_min_version_code_support( $platform );

				if ( $min_code_support_version === $build_version_code ) {
					$build_version_code = 0;
				}
				\BuddyBossApp\Build::instance()->update_build_min_version_code_support( $platform, $build_version_code );
			}
		}

	}
}
